/**
 * \file test/test_unimod.cpp
 * \date 06/04/2025
 * \author Olivier Langella
 * \brief test UNIMOD OBO terms
 */

/*******************************************************************************
 * Copyright (c) 2025 Olivier Langella
 *<Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/

// ./tests/catch2-only-tests [unimod] -s

#include <catch2/catch_test_macros.hpp>
#include <catch2/catch_approx.hpp>
#include <pappsomspp/core/obo/filterobopsimodmap.h>
#include <pappsomspp/core/obo/obounimod.h>
#include <pappsomspp/core/obo/obopsimod.h>
#include <pappsomspp/core/exception/exceptionnotpossible.h>


#include "common.h"
#include "tests/tests-config.h"
/// home/langella/developpement/git/i2masschroq/src/input/sage/sagetsvhandler.cpp@457,
/// SageTsvHandler::parsePeptide(): "[+42.0106]-M[MOD:00719]QNDAGEFVDLYVPR"

TEST_CASE("Test OBO UNIMOD", "[unimod]")
{
  // Set the debugging message formatting pattern.
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));

  SECTION("..:: Test OBO UNIMOD ::..", "[unimod]")
  {
    pappso::FilterOboPsiModMap obo_unimod_map;
    pappso::OboUnimod unimod_parser(obo_unimod_map);


    pappso::FilterOboPsiModMap obo_psimod_map;
    pappso::OboPsiMod psimod_parser(obo_psimod_map);

    // MOD:00394 == UNIMOD:1
    pappso::OboPsiModTerm uterm = obo_unimod_map.getOboPsiModTermWithAccession("UNIMOD:1");
    REQUIRE(uterm.getAccession().toStdString() == "UNIMOD:1");
    REQUIRE(uterm.m_name.toStdString() == "Acetyl");
    REQUIRE(uterm.m_diffMono == Catch::Approx(42.010565));
    REQUIRE(uterm.m_diffFormula.toStdString() == "H(2) C(2) O");
    REQUIRE(uterm.m_definition.toStdString() == "Acetylation.");


    pappso::OboPsiModTerm pterm = obo_psimod_map.getOboPsiModTermWithAccession("MOD:00394");
    REQUIRE(pterm.getAccession().toStdString() == "MOD:00394");
    REQUIRE(pterm.m_name.toStdString() == "monoacetylated residue");
    REQUIRE(pterm.m_diffMono == Catch::Approx(uterm.m_diffMono));
    REQUIRE(pterm.m_diffFormula.toStdString() == "C 2 H 2 O 1");
    REQUIRE(
      pterm.m_definition.toStdString() ==
      "A protein modification that effectively replaces one hydrogen atom with one acetyl group.");


    // Unimod:188 MOD:00544
    uterm = obo_unimod_map.getOboPsiModTermWithAccession("UNIMOD:188");
    REQUIRE(uterm.getAccession().toStdString() == "UNIMOD:188");
    REQUIRE(uterm.m_name.toStdString() == "Label:13C(6)");
    REQUIRE(uterm.m_diffMono == Catch::Approx(6.020129));
    REQUIRE(uterm.m_diffFormula.toStdString() == "C(-6) 13C(6)");
    REQUIRE(uterm.m_definition.toStdString() == "13C(6) Silac label.");


    pterm = obo_psimod_map.getOboPsiModTermWithAccession("MOD:00544");
    REQUIRE(pterm.getAccession().toStdString() == "MOD:00544");
    REQUIRE(pterm.m_name.toStdString() == "6x(13)C labeled residue");
    REQUIRE(pterm.m_diffFormula.toStdString() == "(12)C -6 (13)C 6");
    REQUIRE(pterm.m_diffMono == Catch::Approx(uterm.m_diffMono));
    REQUIRE(pterm.m_definition.toStdString() ==
            "A protein modification that effectively converts a residue containing common isotopes "
            "to a 6x(13)C labeled residue.");
  }
}
