#!/usr/bin/env perl

use strict;
use Test::More tests => 37992;
use FindBin qw($Bin);
use lib "$Bin/lib";
use MemcachedTest;


my $server = new_memcached('-m 128');
my $sock = $server->sock;

subtest 'close if no get found in 2k' => sub {
    my $ns = $server->new_sock;

    my $spaces = ' ' x 2048;
    print $ns "get $spaces manyspaces";
    sleep 1;
    print $ns "\r\n";
    is(scalar <$ns>, "END\r\n", "long ascii get was fine");

    print $ns "incr $spaces manyspaces 1";
    is(scalar <$ns>, undef, "long ascii incr was not fine");
};

# set foo (and should get it)
print $sock "set foo 0 0 6\r\nfooval\r\n";
is(scalar <$sock>, "STORED\r\n", "stored foo");
mem_get_is($sock, "foo", "fooval");

# add bar (and should get it)
print $sock "add bar 0 0 6\r\nbarval\r\n";
is(scalar <$sock>, "STORED\r\n", "stored barval");
mem_get_is($sock, "bar", "barval");

# add foo (but shouldn't get new value)
print $sock "add foo 0 0 5\r\nfoov2\r\n";
is(scalar <$sock>, "NOT_STORED\r\n", "not stored");
mem_get_is($sock, "foo", "fooval");

# replace bar (should work)
print $sock "replace bar 0 0 6\r\nbarva2\r\n";
is(scalar <$sock>, "STORED\r\n", "replaced barval 2");

# replace notexist (shouldn't work)
print $sock "replace notexist 0 0 6\r\nbarva2\r\n";
is(scalar <$sock>, "NOT_STORED\r\n", "didn't replace notexist");

# delete foo.
print $sock "delete foo\r\n";
is(scalar <$sock>, "DELETED\r\n", "deleted foo");

# delete foo again.  not found this time.
print $sock "delete foo\r\n";
is(scalar <$sock>, "NOT_FOUND\r\n", "deleted foo, but not found");

# add moo
#
print $sock "add moo 0 0 6\r\nmooval\r\n";
is(scalar <$sock>, "STORED\r\n", "stored barval");
mem_get_is($sock, "moo", "mooval");

# check-and-set (cas) failure case, try to set value with incorrect cas unique val
print $sock "cas moo 0 0 6 0\r\nMOOVAL\r\n";
is(scalar <$sock>, "EXISTS\r\n", "check and set with invalid id");

# test "gets", grab unique ID
print $sock "gets moo\r\n";
# VALUE moo 0 6 3084947704
#
my @retvals = split(/ /, scalar <$sock>);
my $data = scalar <$sock>; # grab data
my $dot  = scalar <$sock>; # grab dot on line by itself
is($retvals[0], "VALUE", "get value using 'gets'");
my $unique_id = $retvals[4];
# clean off \r\n
$unique_id =~ s/\r\n$//;
ok($unique_id =~ /^\d+$/, "unique ID '$unique_id' is an integer");
# now test that we can store moo with the correct unique id
print $sock "cas moo 0 0 6 $unique_id\r\nMOOVAL\r\n";
is(scalar <$sock>, "STORED\r\n");
mem_get_is($sock, "moo", "MOOVAL");

# pipeline is okay
print $sock "set foo 0 0 6\r\nfooval\r\ndelete foo\r\nset foo 0 0 6\r\nfooval\r\ndelete foo\r\n";
is(scalar <$sock>, "STORED\r\n",  "pipeline set");
is(scalar <$sock>, "DELETED\r\n", "pipeline delete");
is(scalar <$sock>, "STORED\r\n",  "pipeline set");
is(scalar <$sock>, "DELETED\r\n", "pipeline delete");


# Test sets up to a large size around 1MB.
# Everything up to 1MB - 1k should succeed, everything 1MB +1k should fail.

my $len = 1024;
while ($len < 1024*1028) {
    my $val = "B"x$len;
    if ($len > (1024*1024)) {
        # Ensure causing a memory overflow doesn't leave stale data.
        print $sock "set foo_$len 0 0 3\r\nMOO\r\n";
        is(scalar <$sock>, "STORED\r\n");
        print $sock "set foo_$len 0 0 $len\r\n$val\r\n";
        is(scalar <$sock>, "SERVER_ERROR object too large for cache\r\n", "failed to store size $len");
        mem_get_is($sock, "foo_$len");
    } else {
        print $sock "set foo_$len 0 0 $len\r\n$val\r\n";
        is(scalar <$sock>, "STORED\r\n", "stored size $len");
    }
    $len += 2048;
}

{
    # Test large ASCII multigets.
    print $sock "set foobarbaz 0 0 2\r\nhi\r\n";
    is(scalar <$sock>, "STORED\r\n",  "base foo stored");

    my $len = 1024 * 128;
    my $klist = '';
    my $kcount = 0;
    for (my $x = 0; $x < $len; $x += 7) {
        $klist .= "foobarbaz ";
        $kcount++;
    }
    print $sock "get $klist\r\n";

    while ($kcount--) {
        is(scalar <$sock>, "VALUE foobarbaz 0 2\r\n", "foo returned");
        is(scalar <$sock>, "hi\r\n", "foo 'hi' returned");
    }
    is(scalar <$sock>, "END\r\n", "foo END seen");
}

