//    SPDX-License-Identifier: LGPL-3.0-or-later
/*
   This file is part of FlexiBLAS, a BLAS/LAPACK interface wrapper library.
   Copyright (C) 2013-2025 Martin Koehler

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 3 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

*/

#include <stdlib.h>
#include <stdint.h>
#include <complex.h>

#include "flexiblas_config.h"
#include "flexiblas_fortran_char_len.h"
#include "flexiblas_fortran_mangle.h"

#include "profile_hook.h"
#include "cscutils/table.h"
#include "flexiblas_backend.h"
#include "flexiblas_real_lapack_3_9_1_wodprc.h"



/* Wrapper for cbbcsd */
void hook_cbbcsd(char *jobu1, char *jobu2, char *jobv1t, char *jobv2t, char *trans, blasint *m, blasint *p, blasint *q, float *theta, float *phi, lapack_complex_float *u1, blasint *ldu1, lapack_complex_float *u2, blasint *ldu2, lapack_complex_float *v1t, blasint *ldv1t, lapack_complex_float *v2t, blasint *ldv2t, float *b11d, float *b11e, float *b12d, float *b12e, float *b21d, float *b21e, float *b22d, float *b22e, float *rwork, blasint *lrwork, blasint *info, flexiblas_fortran_charlen_t jobu1_len, flexiblas_fortran_charlen_t jobu2_len, flexiblas_fortran_charlen_t jobv1t_len, flexiblas_fortran_charlen_t jobv2t_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cbbcsd((void *) jobu1, (void *) jobu2, (void *) jobv1t, (void *) jobv2t, (void *) trans, (void *) m, (void *) p, (void *) q, (void *) theta, (void *) phi, (void *) u1, (void *) ldu1, (void *) u2, (void *) ldu2, (void *) v1t, (void *) ldv1t, (void *) v2t, (void *) ldv2t, (void *) b11d, (void *) b11e, (void *) b12d, (void *) b12e, (void *) b21d, (void *) b21e, (void *) b22d, (void *) b22e, (void *) rwork, (void *) lrwork, (void *) info, jobu1_len, jobu2_len, jobv1t_len, jobv2t_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cbbcsd.timings[0] += (helpTimeStop - helpTime);
    data->cbbcsd.calls[0]++;
    
    return;
}


/* Wrapper for cbdsqr */
void hook_cbdsqr(char *uplo, blasint *n, blasint *ncvt, blasint *nru, blasint *ncc, float *d, float *e, lapack_complex_float *vt, blasint *ldvt, lapack_complex_float *u, blasint *ldu, lapack_complex_float *c, blasint *ldc, float *rwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cbdsqr((void *) uplo, (void *) n, (void *) ncvt, (void *) nru, (void *) ncc, (void *) d, (void *) e, (void *) vt, (void *) ldvt, (void *) u, (void *) ldu, (void *) c, (void *) ldc, (void *) rwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cbdsqr.timings[0] += (helpTimeStop - helpTime);
    data->cbdsqr.calls[0]++;
    
    return;
}


/* Wrapper for cgbbrd */
void hook_cgbbrd(char *vect, blasint *m, blasint *n, blasint *ncc, blasint *kl, blasint *ku, lapack_complex_float *ab, blasint *ldab, float *d, float *e, lapack_complex_float *q, blasint *ldq, lapack_complex_float *pt, blasint *ldpt, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t vect_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgbbrd((void *) vect, (void *) m, (void *) n, (void *) ncc, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) d, (void *) e, (void *) q, (void *) ldq, (void *) pt, (void *) ldpt, (void *) c, (void *) ldc, (void *) work, (void *) rwork, (void *) info, vect_len);

    helpTimeStop = flexiblas_wtime();

    data->cgbbrd.timings[0] += (helpTimeStop - helpTime);
    data->cgbbrd.calls[0]++;
    
    return;
}


/* Wrapper for cgbcon */
void hook_cgbcon(char *norm, blasint *n, blasint *kl, blasint *ku, lapack_complex_float *ab, blasint *ldab, blasint *ipiv, float *anorm, float *rcond, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgbcon((void *) norm, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) rwork, (void *) info, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->cgbcon.timings[0] += (helpTimeStop - helpTime);
    data->cgbcon.calls[0]++;
    
    return;
}


/* Wrapper for cgbequ */
void hook_cgbequ(blasint *m, blasint *n, blasint *kl, blasint *ku, lapack_complex_float *ab, blasint *ldab, float *r, float *c, float *rowcnd, float *colcnd, float *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgbequ((void *) m, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgbequ.timings[0] += (helpTimeStop - helpTime);
    data->cgbequ.calls[0]++;
    
    return;
}


/* Wrapper for cgbequb */
void hook_cgbequb(blasint *m, blasint *n, blasint *kl, blasint *ku, lapack_complex_float *ab, blasint *ldab, float *r, float *c, float *rowcnd, float *colcnd, float *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgbequb((void *) m, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgbequb.timings[0] += (helpTimeStop - helpTime);
    data->cgbequb.calls[0]++;
    
    return;
}


/* Wrapper for cgbrfs */
void hook_cgbrfs(char *trans, blasint *n, blasint *kl, blasint *ku, blasint *nrhs, lapack_complex_float *ab, blasint *ldab, lapack_complex_float *afb, blasint *ldafb, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgbrfs((void *) trans, (void *) n, (void *) kl, (void *) ku, (void *) nrhs, (void *) ab, (void *) ldab, (void *) afb, (void *) ldafb, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cgbrfs.timings[0] += (helpTimeStop - helpTime);
    data->cgbrfs.calls[0]++;
    
    return;
}


/* Wrapper for cgbsv */
void hook_cgbsv(blasint *n, blasint *kl, blasint *ku, blasint *nrhs, lapack_complex_float *ab, blasint *ldab, blasint *ipiv, lapack_complex_float *b, blasint *ldb, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgbsv((void *) n, (void *) kl, (void *) ku, (void *) nrhs, (void *) ab, (void *) ldab, (void *) ipiv, (void *) b, (void *) ldb, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgbsv.timings[0] += (helpTimeStop - helpTime);
    data->cgbsv.calls[0]++;
    
    return;
}


/* Wrapper for cgbsvx */
void hook_cgbsvx(char *fact, char *trans, blasint *n, blasint *kl, blasint *ku, blasint *nrhs, lapack_complex_float *ab, blasint *ldab, lapack_complex_float *afb, blasint *ldafb, blasint *ipiv, char *equed, float *r, float *c, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgbsvx((void *) fact, (void *) trans, (void *) n, (void *) kl, (void *) ku, (void *) nrhs, (void *) ab, (void *) ldab, (void *) afb, (void *) ldafb, (void *) ipiv, (void *) equed, (void *) r, (void *) c, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, fact_len, trans_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->cgbsvx.timings[0] += (helpTimeStop - helpTime);
    data->cgbsvx.calls[0]++;
    
    return;
}


/* Wrapper for cgbtf2 */
void hook_cgbtf2(blasint *m, blasint *n, blasint *kl, blasint *ku, lapack_complex_float *ab, blasint *ldab, blasint *ipiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgbtf2((void *) m, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) ipiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgbtf2.timings[0] += (helpTimeStop - helpTime);
    data->cgbtf2.calls[0]++;
    
    return;
}


/* Wrapper for cgbtrf */
void hook_cgbtrf(blasint *m, blasint *n, blasint *kl, blasint *ku, lapack_complex_float *ab, blasint *ldab, blasint *ipiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgbtrf((void *) m, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) ipiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgbtrf.timings[0] += (helpTimeStop - helpTime);
    data->cgbtrf.calls[0]++;
    
    return;
}


/* Wrapper for cgbtrs */
void hook_cgbtrs(char *trans, blasint *n, blasint *kl, blasint *ku, blasint *nrhs, lapack_complex_float *ab, blasint *ldab, blasint *ipiv, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgbtrs((void *) trans, (void *) n, (void *) kl, (void *) ku, (void *) nrhs, (void *) ab, (void *) ldab, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cgbtrs.timings[0] += (helpTimeStop - helpTime);
    data->cgbtrs.calls[0]++;
    
    return;
}


/* Wrapper for cgebak */
void hook_cgebak(char *job, char *side, blasint *n, blasint *ilo, blasint *ihi, float *scale, blasint *m, lapack_complex_float *v, blasint *ldv, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t side_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgebak((void *) job, (void *) side, (void *) n, (void *) ilo, (void *) ihi, (void *) scale, (void *) m, (void *) v, (void *) ldv, (void *) info, job_len, side_len);

    helpTimeStop = flexiblas_wtime();

    data->cgebak.timings[0] += (helpTimeStop - helpTime);
    data->cgebak.calls[0]++;
    
    return;
}


/* Wrapper for cgebal */
void hook_cgebal(char *job, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ilo, blasint *ihi, float *scale, blasint *info, flexiblas_fortran_charlen_t job_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgebal((void *) job, (void *) n, (void *) a, (void *) lda, (void *) ilo, (void *) ihi, (void *) scale, (void *) info, job_len);

    helpTimeStop = flexiblas_wtime();

    data->cgebal.timings[0] += (helpTimeStop - helpTime);
    data->cgebal.calls[0]++;
    
    return;
}


/* Wrapper for cgebd2 */
void hook_cgebd2(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, float *d, float *e, lapack_complex_float *tauq, lapack_complex_float *taup, lapack_complex_float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgebd2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) e, (void *) tauq, (void *) taup, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgebd2.timings[0] += (helpTimeStop - helpTime);
    data->cgebd2.calls[0]++;
    
    return;
}


/* Wrapper for cgebrd */
void hook_cgebrd(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, float *d, float *e, lapack_complex_float *tauq, lapack_complex_float *taup, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgebrd((void *) m, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) e, (void *) tauq, (void *) taup, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgebrd.timings[0] += (helpTimeStop - helpTime);
    data->cgebrd.calls[0]++;
    
    return;
}


/* Wrapper for cgecon */
void hook_cgecon(char *norm, blasint *n, lapack_complex_float *a, blasint *lda, float *anorm, float *rcond, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgecon((void *) norm, (void *) n, (void *) a, (void *) lda, (void *) anorm, (void *) rcond, (void *) work, (void *) rwork, (void *) info, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->cgecon.timings[0] += (helpTimeStop - helpTime);
    data->cgecon.calls[0]++;
    
    return;
}


/* Wrapper for cgeequ */
void hook_cgeequ(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, float *r, float *c, float *rowcnd, float *colcnd, float *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgeequ((void *) m, (void *) n, (void *) a, (void *) lda, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeequ.timings[0] += (helpTimeStop - helpTime);
    data->cgeequ.calls[0]++;
    
    return;
}


/* Wrapper for cgeequb */
void hook_cgeequb(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, float *r, float *c, float *rowcnd, float *colcnd, float *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgeequb((void *) m, (void *) n, (void *) a, (void *) lda, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeequb.timings[0] += (helpTimeStop - helpTime);
    data->cgeequb.calls[0]++;
    
    return;
}


/* Wrapper for cgees */
void hook_cgees(char *jobvs, char *sort, blaslogical *select, blasint *n, lapack_complex_float *a, blasint *lda, blasint *sdim, lapack_complex_float *w, lapack_complex_float *vs, blasint *ldvs, lapack_complex_float *work, blasint *lwork, float *rwork, blaslogical *bwork, blasint *info, flexiblas_fortran_charlen_t jobvs_len, flexiblas_fortran_charlen_t sort_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgees((void *) jobvs, (void *) sort, (void *) select, (void *) n, (void *) a, (void *) lda, (void *) sdim, (void *) w, (void *) vs, (void *) ldvs, (void *) work, (void *) lwork, (void *) rwork, (void *) bwork, (void *) info, jobvs_len, sort_len);

    helpTimeStop = flexiblas_wtime();

    data->cgees.timings[0] += (helpTimeStop - helpTime);
    data->cgees.calls[0]++;
    
    return;
}


/* Wrapper for cgeesx */
void hook_cgeesx(char *jobvs, char *sort, blaslogical *select, char *sense, blasint *n, lapack_complex_float *a, blasint *lda, blasint *sdim, lapack_complex_float *w, lapack_complex_float *vs, blasint *ldvs, float *rconde, float *rcondv, lapack_complex_float *work, blasint *lwork, float *rwork, blaslogical *bwork, blasint *info, flexiblas_fortran_charlen_t jobvs_len, flexiblas_fortran_charlen_t sort_len, flexiblas_fortran_charlen_t sense_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgeesx((void *) jobvs, (void *) sort, (void *) select, (void *) sense, (void *) n, (void *) a, (void *) lda, (void *) sdim, (void *) w, (void *) vs, (void *) ldvs, (void *) rconde, (void *) rcondv, (void *) work, (void *) lwork, (void *) rwork, (void *) bwork, (void *) info, jobvs_len, sort_len, sense_len);

    helpTimeStop = flexiblas_wtime();

    data->cgeesx.timings[0] += (helpTimeStop - helpTime);
    data->cgeesx.calls[0]++;
    
    return;
}


/* Wrapper for cgeev */
void hook_cgeev(char *jobvl, char *jobvr, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *w, lapack_complex_float *vl, blasint *ldvl, lapack_complex_float *vr, blasint *ldvr, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *info, flexiblas_fortran_charlen_t jobvl_len, flexiblas_fortran_charlen_t jobvr_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgeev((void *) jobvl, (void *) jobvr, (void *) n, (void *) a, (void *) lda, (void *) w, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) work, (void *) lwork, (void *) rwork, (void *) info, jobvl_len, jobvr_len);

    helpTimeStop = flexiblas_wtime();

    data->cgeev.timings[0] += (helpTimeStop - helpTime);
    data->cgeev.calls[0]++;
    
    return;
}


/* Wrapper for cgeevx */
void hook_cgeevx(char *balanc, char *jobvl, char *jobvr, char *sense, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *w, lapack_complex_float *vl, blasint *ldvl, lapack_complex_float *vr, blasint *ldvr, blasint *ilo, blasint *ihi, float *scale, float *abnrm, float *rconde, float *rcondv, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *info, flexiblas_fortran_charlen_t balanc_len, flexiblas_fortran_charlen_t jobvl_len, flexiblas_fortran_charlen_t jobvr_len, flexiblas_fortran_charlen_t sense_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgeevx((void *) balanc, (void *) jobvl, (void *) jobvr, (void *) sense, (void *) n, (void *) a, (void *) lda, (void *) w, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) ilo, (void *) ihi, (void *) scale, (void *) abnrm, (void *) rconde, (void *) rcondv, (void *) work, (void *) lwork, (void *) rwork, (void *) info, balanc_len, jobvl_len, jobvr_len, sense_len);

    helpTimeStop = flexiblas_wtime();

    data->cgeevx.timings[0] += (helpTimeStop - helpTime);
    data->cgeevx.calls[0]++;
    
    return;
}


/* Wrapper for cgehd2 */
void hook_cgehd2(blasint *n, blasint *ilo, blasint *ihi, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgehd2((void *) n, (void *) ilo, (void *) ihi, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgehd2.timings[0] += (helpTimeStop - helpTime);
    data->cgehd2.calls[0]++;
    
    return;
}


/* Wrapper for cgehrd */
void hook_cgehrd(blasint *n, blasint *ilo, blasint *ihi, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgehrd((void *) n, (void *) ilo, (void *) ihi, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgehrd.timings[0] += (helpTimeStop - helpTime);
    data->cgehrd.calls[0]++;
    
    return;
}


/* Wrapper for cgejsv */
void hook_cgejsv(char *joba, char *jobu, char *jobv, char *jobr, char *jobt, char *jobp, blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, float *sva, lapack_complex_float *u, blasint *ldu, lapack_complex_float *v, blasint *ldv, lapack_complex_float *cwork, blasint *lwork, float *rwork, blasint *lrwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t joba_len, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobv_len, flexiblas_fortran_charlen_t jobr_len, flexiblas_fortran_charlen_t jobt_len, flexiblas_fortran_charlen_t jobp_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgejsv((void *) joba, (void *) jobu, (void *) jobv, (void *) jobr, (void *) jobt, (void *) jobp, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) sva, (void *) u, (void *) ldu, (void *) v, (void *) ldv, (void *) cwork, (void *) lwork, (void *) rwork, (void *) lrwork, (void *) iwork, (void *) info, joba_len, jobu_len, jobv_len, jobr_len, jobt_len, jobp_len);

    helpTimeStop = flexiblas_wtime();

    data->cgejsv.timings[0] += (helpTimeStop - helpTime);
    data->cgejsv.calls[0]++;
    
    return;
}


/* Wrapper for cgelq */
void hook_cgelq(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *t, blasint *tsize, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgelq((void *) m, (void *) n, (void *) a, (void *) lda, (void *) t, (void *) tsize, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgelq.timings[0] += (helpTimeStop - helpTime);
    data->cgelq.calls[0]++;
    
    return;
}


/* Wrapper for cgelq2 */
void hook_cgelq2(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgelq2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgelq2.timings[0] += (helpTimeStop - helpTime);
    data->cgelq2.calls[0]++;
    
    return;
}


/* Wrapper for cgelqf */
void hook_cgelqf(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgelqf((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgelqf.timings[0] += (helpTimeStop - helpTime);
    data->cgelqf.calls[0]++;
    
    return;
}


/* Wrapper for cgelqt */
void hook_cgelqt(blasint *m, blasint *n, blasint *mb, lapack_complex_float *a, blasint *lda, lapack_complex_float *t, blasint *ldt, lapack_complex_float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgelqt((void *) m, (void *) n, (void *) mb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgelqt.timings[0] += (helpTimeStop - helpTime);
    data->cgelqt.calls[0]++;
    
    return;
}


/* Wrapper for cgelqt3 */
void hook_cgelqt3(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *t, blasint *ldt, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgelqt3((void *) m, (void *) n, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgelqt3.timings[0] += (helpTimeStop - helpTime);
    data->cgelqt3.calls[0]++;
    
    return;
}


/* Wrapper for cgels */
void hook_cgels(char *trans, blasint *m, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgels((void *) trans, (void *) m, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cgels.timings[0] += (helpTimeStop - helpTime);
    data->cgels.calls[0]++;
    
    return;
}


/* Wrapper for cgelsd */
void hook_cgelsd(blasint *m, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, float *s, float *rcond, blasint *rank, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgelsd((void *) m, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) s, (void *) rcond, (void *) rank, (void *) work, (void *) lwork, (void *) rwork, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgelsd.timings[0] += (helpTimeStop - helpTime);
    data->cgelsd.calls[0]++;
    
    return;
}


/* Wrapper for cgelss */
void hook_cgelss(blasint *m, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, float *s, float *rcond, blasint *rank, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgelss((void *) m, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) s, (void *) rcond, (void *) rank, (void *) work, (void *) lwork, (void *) rwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgelss.timings[0] += (helpTimeStop - helpTime);
    data->cgelss.calls[0]++;
    
    return;
}


/* Wrapper for cgelsy */
void hook_cgelsy(blasint *m, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, blasint *jpvt, float *rcond, blasint *rank, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgelsy((void *) m, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) jpvt, (void *) rcond, (void *) rank, (void *) work, (void *) lwork, (void *) rwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgelsy.timings[0] += (helpTimeStop - helpTime);
    data->cgelsy.calls[0]++;
    
    return;
}


/* Wrapper for cgemlq */
void hook_cgemlq(char *side, char *trans, blasint *m, blasint *n, blasint *k, lapack_complex_float *a, blasint *lda, lapack_complex_float *t, blasint *tsize, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgemlq((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) t, (void *) tsize, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cgemlq.timings[0] += (helpTimeStop - helpTime);
    data->cgemlq.calls[0]++;
    
    return;
}


/* Wrapper for cgemlqt */
void hook_cgemlqt(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *mb, lapack_complex_float *v, blasint *ldv, lapack_complex_float *t, blasint *ldt, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgemlqt((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) mb, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cgemlqt.timings[0] += (helpTimeStop - helpTime);
    data->cgemlqt.calls[0]++;
    
    return;
}


/* Wrapper for cgemqr */
void hook_cgemqr(char *side, char *trans, blasint *m, blasint *n, blasint *k, lapack_complex_float *a, blasint *lda, lapack_complex_float *t, blasint *tsize, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgemqr((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) t, (void *) tsize, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cgemqr.timings[0] += (helpTimeStop - helpTime);
    data->cgemqr.calls[0]++;
    
    return;
}


/* Wrapper for cgemqrt */
void hook_cgemqrt(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *nb, lapack_complex_float *v, blasint *ldv, lapack_complex_float *t, blasint *ldt, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgemqrt((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) nb, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cgemqrt.timings[0] += (helpTimeStop - helpTime);
    data->cgemqrt.calls[0]++;
    
    return;
}


/* Wrapper for cgeql2 */
void hook_cgeql2(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgeql2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeql2.timings[0] += (helpTimeStop - helpTime);
    data->cgeql2.calls[0]++;
    
    return;
}


/* Wrapper for cgeqlf */
void hook_cgeqlf(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgeqlf((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeqlf.timings[0] += (helpTimeStop - helpTime);
    data->cgeqlf.calls[0]++;
    
    return;
}


/* Wrapper for cgeqp3 */
void hook_cgeqp3(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, blasint *jpvt, lapack_complex_float *tau, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgeqp3((void *) m, (void *) n, (void *) a, (void *) lda, (void *) jpvt, (void *) tau, (void *) work, (void *) lwork, (void *) rwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeqp3.timings[0] += (helpTimeStop - helpTime);
    data->cgeqp3.calls[0]++;
    
    return;
}


/* Wrapper for cgeqr */
void hook_cgeqr(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *t, blasint *tsize, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgeqr((void *) m, (void *) n, (void *) a, (void *) lda, (void *) t, (void *) tsize, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeqr.timings[0] += (helpTimeStop - helpTime);
    data->cgeqr.calls[0]++;
    
    return;
}


/* Wrapper for cgeqr2 */
void hook_cgeqr2(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgeqr2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeqr2.timings[0] += (helpTimeStop - helpTime);
    data->cgeqr2.calls[0]++;
    
    return;
}


/* Wrapper for cgeqr2p */
void hook_cgeqr2p(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgeqr2p((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeqr2p.timings[0] += (helpTimeStop - helpTime);
    data->cgeqr2p.calls[0]++;
    
    return;
}


/* Wrapper for cgeqrf */
void hook_cgeqrf(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgeqrf((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeqrf.timings[0] += (helpTimeStop - helpTime);
    data->cgeqrf.calls[0]++;
    
    return;
}


/* Wrapper for cgeqrfp */
void hook_cgeqrfp(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgeqrfp((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeqrfp.timings[0] += (helpTimeStop - helpTime);
    data->cgeqrfp.calls[0]++;
    
    return;
}


/* Wrapper for cgeqrt */
void hook_cgeqrt(blasint *m, blasint *n, blasint *nb, lapack_complex_float *a, blasint *lda, lapack_complex_float *t, blasint *ldt, lapack_complex_float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgeqrt((void *) m, (void *) n, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeqrt.timings[0] += (helpTimeStop - helpTime);
    data->cgeqrt.calls[0]++;
    
    return;
}


/* Wrapper for cgeqrt2 */
void hook_cgeqrt2(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *t, blasint *ldt, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgeqrt2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeqrt2.timings[0] += (helpTimeStop - helpTime);
    data->cgeqrt2.calls[0]++;
    
    return;
}


/* Wrapper for cgeqrt3 */
void hook_cgeqrt3(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *t, blasint *ldt, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgeqrt3((void *) m, (void *) n, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgeqrt3.timings[0] += (helpTimeStop - helpTime);
    data->cgeqrt3.calls[0]++;
    
    return;
}


/* Wrapper for cgerfs */
void hook_cgerfs(char *trans, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *af, blasint *ldaf, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgerfs((void *) trans, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) af, (void *) ldaf, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cgerfs.timings[0] += (helpTimeStop - helpTime);
    data->cgerfs.calls[0]++;
    
    return;
}


/* Wrapper for cgerq2 */
void hook_cgerq2(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgerq2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgerq2.timings[0] += (helpTimeStop - helpTime);
    data->cgerq2.calls[0]++;
    
    return;
}


/* Wrapper for cgerqf */
void hook_cgerqf(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgerqf((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgerqf.timings[0] += (helpTimeStop - helpTime);
    data->cgerqf.calls[0]++;
    
    return;
}


/* Wrapper for cgesc2 */
void hook_cgesc2(blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *rhs, blasint *ipiv, blasint *jpiv, float *scale)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgesc2((void *) n, (void *) a, (void *) lda, (void *) rhs, (void *) ipiv, (void *) jpiv, (void *) scale);

    helpTimeStop = flexiblas_wtime();

    data->cgesc2.timings[0] += (helpTimeStop - helpTime);
    data->cgesc2.calls[0]++;
    
    return;
}


/* Wrapper for cgesdd */
void hook_cgesdd(char *jobz, blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, float *s, lapack_complex_float *u, blasint *ldu, lapack_complex_float *vt, blasint *ldvt, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t jobz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgesdd((void *) jobz, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) s, (void *) u, (void *) ldu, (void *) vt, (void *) ldvt, (void *) work, (void *) lwork, (void *) rwork, (void *) iwork, (void *) info, jobz_len);

    helpTimeStop = flexiblas_wtime();

    data->cgesdd.timings[0] += (helpTimeStop - helpTime);
    data->cgesdd.calls[0]++;
    
    return;
}


/* Wrapper for cgesv */
void hook_cgesv(blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *b, blasint *ldb, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgesv((void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgesv.timings[0] += (helpTimeStop - helpTime);
    data->cgesv.calls[0]++;
    
    return;
}


/* Wrapper for cgesvd */
void hook_cgesvd(char *jobu, char *jobvt, blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, float *s, lapack_complex_float *u, blasint *ldu, lapack_complex_float *vt, blasint *ldvt, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *info, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobvt_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgesvd((void *) jobu, (void *) jobvt, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) s, (void *) u, (void *) ldu, (void *) vt, (void *) ldvt, (void *) work, (void *) lwork, (void *) rwork, (void *) info, jobu_len, jobvt_len);

    helpTimeStop = flexiblas_wtime();

    data->cgesvd.timings[0] += (helpTimeStop - helpTime);
    data->cgesvd.calls[0]++;
    
    return;
}


/* Wrapper for cgesvdq */
void hook_cgesvdq(char *joba, char *jobp, char *jobr, char *jobu, char *jobv, blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, float *s, lapack_complex_float *u, blasint *ldu, lapack_complex_float *v, blasint *ldv, blasint *numrank, blasint *iwork, blasint *liwork, lapack_complex_float *cwork, blasint *lcwork, float *rwork, blasint *lrwork, blasint *info, flexiblas_fortran_charlen_t joba_len, flexiblas_fortran_charlen_t jobp_len, flexiblas_fortran_charlen_t jobr_len, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobv_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgesvdq((void *) joba, (void *) jobp, (void *) jobr, (void *) jobu, (void *) jobv, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) s, (void *) u, (void *) ldu, (void *) v, (void *) ldv, (void *) numrank, (void *) iwork, (void *) liwork, (void *) cwork, (void *) lcwork, (void *) rwork, (void *) lrwork, (void *) info, joba_len, jobp_len, jobr_len, jobu_len, jobv_len);

    helpTimeStop = flexiblas_wtime();

    data->cgesvdq.timings[0] += (helpTimeStop - helpTime);
    data->cgesvdq.calls[0]++;
    
    return;
}


/* Wrapper for cgesvdx */
void hook_cgesvdx(char *jobu, char *jobvt, char *range, blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, float *vl, float *vu, blasint *il, blasint *iu, blasint *ns, float *s, lapack_complex_float *u, blasint *ldu, lapack_complex_float *vt, blasint *ldvt, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobvt_len, flexiblas_fortran_charlen_t range_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgesvdx((void *) jobu, (void *) jobvt, (void *) range, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) ns, (void *) s, (void *) u, (void *) ldu, (void *) vt, (void *) ldvt, (void *) work, (void *) lwork, (void *) rwork, (void *) iwork, (void *) info, jobu_len, jobvt_len, range_len);

    helpTimeStop = flexiblas_wtime();

    data->cgesvdx.timings[0] += (helpTimeStop - helpTime);
    data->cgesvdx.calls[0]++;
    
    return;
}


/* Wrapper for cgesvj */
void hook_cgesvj(char *joba, char *jobu, char *jobv, blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, float *sva, blasint *mv, lapack_complex_float *v, blasint *ldv, lapack_complex_float *cwork, blasint *lwork, float *rwork, blasint *lrwork, blasint *info, flexiblas_fortran_charlen_t joba_len, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobv_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgesvj((void *) joba, (void *) jobu, (void *) jobv, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) sva, (void *) mv, (void *) v, (void *) ldv, (void *) cwork, (void *) lwork, (void *) rwork, (void *) lrwork, (void *) info, joba_len, jobu_len, jobv_len);

    helpTimeStop = flexiblas_wtime();

    data->cgesvj.timings[0] += (helpTimeStop - helpTime);
    data->cgesvj.calls[0]++;
    
    return;
}


/* Wrapper for cgesvx */
void hook_cgesvx(char *fact, char *trans, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *af, blasint *ldaf, blasint *ipiv, char *equed, float *r, float *c, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgesvx((void *) fact, (void *) trans, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) af, (void *) ldaf, (void *) ipiv, (void *) equed, (void *) r, (void *) c, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, fact_len, trans_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->cgesvx.timings[0] += (helpTimeStop - helpTime);
    data->cgesvx.calls[0]++;
    
    return;
}


/* Wrapper for cgetc2 */
void hook_cgetc2(blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, blasint *jpiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgetc2((void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) jpiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgetc2.timings[0] += (helpTimeStop - helpTime);
    data->cgetc2.calls[0]++;
    
    return;
}


/* Wrapper for cgetf2 */
void hook_cgetf2(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgetf2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgetf2.timings[0] += (helpTimeStop - helpTime);
    data->cgetf2.calls[0]++;
    
    return;
}


/* Wrapper for cgetrf */
void hook_cgetrf(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgetrf((void *) m, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgetrf.timings[0] += (helpTimeStop - helpTime);
    data->cgetrf.calls[0]++;
    
    return;
}


/* Wrapper for cgetrf2 */
void hook_cgetrf2(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgetrf2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgetrf2.timings[0] += (helpTimeStop - helpTime);
    data->cgetrf2.calls[0]++;
    
    return;
}


/* Wrapper for cgetri */
void hook_cgetri(blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgetri((void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgetri.timings[0] += (helpTimeStop - helpTime);
    data->cgetri.calls[0]++;
    
    return;
}


/* Wrapper for cgetrs */
void hook_cgetrs(char *trans, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgetrs((void *) trans, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cgetrs.timings[0] += (helpTimeStop - helpTime);
    data->cgetrs.calls[0]++;
    
    return;
}


/* Wrapper for cgetsls */
void hook_cgetsls(char *trans, blasint *m, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgetsls((void *) trans, (void *) m, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cgetsls.timings[0] += (helpTimeStop - helpTime);
    data->cgetsls.calls[0]++;
    
    return;
}


/* Wrapper for cgetsqrhrt */
void hook_cgetsqrhrt(blasint *m, blasint *n, blasint *mb1, blasint *nb1, blasint *nb2, lapack_complex_float *a, blasint *lda, lapack_complex_float *t, blasint *ldt, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgetsqrhrt((void *) m, (void *) n, (void *) mb1, (void *) nb1, (void *) nb2, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgetsqrhrt.timings[0] += (helpTimeStop - helpTime);
    data->cgetsqrhrt.calls[0]++;
    
    return;
}


/* Wrapper for cggbak */
void hook_cggbak(char *job, char *side, blasint *n, blasint *ilo, blasint *ihi, float *lscale, float *rscale, blasint *m, lapack_complex_float *v, blasint *ldv, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t side_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cggbak((void *) job, (void *) side, (void *) n, (void *) ilo, (void *) ihi, (void *) lscale, (void *) rscale, (void *) m, (void *) v, (void *) ldv, (void *) info, job_len, side_len);

    helpTimeStop = flexiblas_wtime();

    data->cggbak.timings[0] += (helpTimeStop - helpTime);
    data->cggbak.calls[0]++;
    
    return;
}


/* Wrapper for cggbal */
void hook_cggbal(char *job, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, blasint *ilo, blasint *ihi, float *lscale, float *rscale, float *work, blasint *info, flexiblas_fortran_charlen_t job_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cggbal((void *) job, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) ilo, (void *) ihi, (void *) lscale, (void *) rscale, (void *) work, (void *) info, job_len);

    helpTimeStop = flexiblas_wtime();

    data->cggbal.timings[0] += (helpTimeStop - helpTime);
    data->cggbal.calls[0]++;
    
    return;
}


/* Wrapper for cgges */
void hook_cgges(char *jobvsl, char *jobvsr, char *sort, blaslogical *selctg, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, blasint *sdim, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vsl, blasint *ldvsl, lapack_complex_float *vsr, blasint *ldvsr, lapack_complex_float *work, blasint *lwork, float *rwork, blaslogical *bwork, blasint *info, flexiblas_fortran_charlen_t jobvsl_len, flexiblas_fortran_charlen_t jobvsr_len, flexiblas_fortran_charlen_t sort_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgges((void *) jobvsl, (void *) jobvsr, (void *) sort, (void *) selctg, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) sdim, (void *) alpha, (void *) beta, (void *) vsl, (void *) ldvsl, (void *) vsr, (void *) ldvsr, (void *) work, (void *) lwork, (void *) rwork, (void *) bwork, (void *) info, jobvsl_len, jobvsr_len, sort_len);

    helpTimeStop = flexiblas_wtime();

    data->cgges.timings[0] += (helpTimeStop - helpTime);
    data->cgges.calls[0]++;
    
    return;
}


/* Wrapper for cgges3 */
void hook_cgges3(char *jobvsl, char *jobvsr, char *sort, blaslogical *selctg, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, blasint *sdim, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vsl, blasint *ldvsl, lapack_complex_float *vsr, blasint *ldvsr, lapack_complex_float *work, blasint *lwork, float *rwork, blaslogical *bwork, blasint *info, flexiblas_fortran_charlen_t jobvsl_len, flexiblas_fortran_charlen_t jobvsr_len, flexiblas_fortran_charlen_t sort_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgges3((void *) jobvsl, (void *) jobvsr, (void *) sort, (void *) selctg, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) sdim, (void *) alpha, (void *) beta, (void *) vsl, (void *) ldvsl, (void *) vsr, (void *) ldvsr, (void *) work, (void *) lwork, (void *) rwork, (void *) bwork, (void *) info, jobvsl_len, jobvsr_len, sort_len);

    helpTimeStop = flexiblas_wtime();

    data->cgges3.timings[0] += (helpTimeStop - helpTime);
    data->cgges3.calls[0]++;
    
    return;
}


/* Wrapper for cggesx */
void hook_cggesx(char *jobvsl, char *jobvsr, char *sort, blaslogical *selctg, char *sense, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, blasint *sdim, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vsl, blasint *ldvsl, lapack_complex_float *vsr, blasint *ldvsr, float *rconde, float *rcondv, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *iwork, blasint *liwork, blaslogical *bwork, blasint *info, flexiblas_fortran_charlen_t jobvsl_len, flexiblas_fortran_charlen_t jobvsr_len, flexiblas_fortran_charlen_t sort_len, flexiblas_fortran_charlen_t sense_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cggesx((void *) jobvsl, (void *) jobvsr, (void *) sort, (void *) selctg, (void *) sense, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) sdim, (void *) alpha, (void *) beta, (void *) vsl, (void *) ldvsl, (void *) vsr, (void *) ldvsr, (void *) rconde, (void *) rcondv, (void *) work, (void *) lwork, (void *) rwork, (void *) iwork, (void *) liwork, (void *) bwork, (void *) info, jobvsl_len, jobvsr_len, sort_len, sense_len);

    helpTimeStop = flexiblas_wtime();

    data->cggesx.timings[0] += (helpTimeStop - helpTime);
    data->cggesx.calls[0]++;
    
    return;
}


/* Wrapper for cggev */
void hook_cggev(char *jobvl, char *jobvr, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vl, blasint *ldvl, lapack_complex_float *vr, blasint *ldvr, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *info, flexiblas_fortran_charlen_t jobvl_len, flexiblas_fortran_charlen_t jobvr_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cggev((void *) jobvl, (void *) jobvr, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) alpha, (void *) beta, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) work, (void *) lwork, (void *) rwork, (void *) info, jobvl_len, jobvr_len);

    helpTimeStop = flexiblas_wtime();

    data->cggev.timings[0] += (helpTimeStop - helpTime);
    data->cggev.calls[0]++;
    
    return;
}


/* Wrapper for cggev3 */
void hook_cggev3(char *jobvl, char *jobvr, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vl, blasint *ldvl, lapack_complex_float *vr, blasint *ldvr, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *info, flexiblas_fortran_charlen_t jobvl_len, flexiblas_fortran_charlen_t jobvr_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cggev3((void *) jobvl, (void *) jobvr, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) alpha, (void *) beta, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) work, (void *) lwork, (void *) rwork, (void *) info, jobvl_len, jobvr_len);

    helpTimeStop = flexiblas_wtime();

    data->cggev3.timings[0] += (helpTimeStop - helpTime);
    data->cggev3.calls[0]++;
    
    return;
}


/* Wrapper for cggevx */
void hook_cggevx(char *balanc, char *jobvl, char *jobvr, char *sense, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vl, blasint *ldvl, lapack_complex_float *vr, blasint *ldvr, blasint *ilo, blasint *ihi, float *lscale, float *rscale, float *abnrm, float *bbnrm, float *rconde, float *rcondv, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *iwork, blaslogical *bwork, blasint *info, flexiblas_fortran_charlen_t balanc_len, flexiblas_fortran_charlen_t jobvl_len, flexiblas_fortran_charlen_t jobvr_len, flexiblas_fortran_charlen_t sense_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cggevx((void *) balanc, (void *) jobvl, (void *) jobvr, (void *) sense, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) alpha, (void *) beta, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) ilo, (void *) ihi, (void *) lscale, (void *) rscale, (void *) abnrm, (void *) bbnrm, (void *) rconde, (void *) rcondv, (void *) work, (void *) lwork, (void *) rwork, (void *) iwork, (void *) bwork, (void *) info, balanc_len, jobvl_len, jobvr_len, sense_len);

    helpTimeStop = flexiblas_wtime();

    data->cggevx.timings[0] += (helpTimeStop - helpTime);
    data->cggevx.calls[0]++;
    
    return;
}


/* Wrapper for cggglm */
void hook_cggglm(blasint *n, blasint *m, blasint *p, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *d, lapack_complex_float *x, lapack_complex_float *y, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cggglm((void *) n, (void *) m, (void *) p, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) d, (void *) x, (void *) y, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cggglm.timings[0] += (helpTimeStop - helpTime);
    data->cggglm.calls[0]++;
    
    return;
}


/* Wrapper for cgghd3 */
void hook_cgghd3(char *compq, char *compz, blasint *n, blasint *ilo, blasint *ihi, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *q, blasint *ldq, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t compq_len, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgghd3((void *) compq, (void *) compz, (void *) n, (void *) ilo, (void *) ihi, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) q, (void *) ldq, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) info, compq_len, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->cgghd3.timings[0] += (helpTimeStop - helpTime);
    data->cgghd3.calls[0]++;
    
    return;
}


/* Wrapper for cgghrd */
void hook_cgghrd(char *compq, char *compz, blasint *n, blasint *ilo, blasint *ihi, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *q, blasint *ldq, lapack_complex_float *z, blasint *ldz, blasint *info, flexiblas_fortran_charlen_t compq_len, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgghrd((void *) compq, (void *) compz, (void *) n, (void *) ilo, (void *) ihi, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) q, (void *) ldq, (void *) z, (void *) ldz, (void *) info, compq_len, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->cgghrd.timings[0] += (helpTimeStop - helpTime);
    data->cgghrd.calls[0]++;
    
    return;
}


/* Wrapper for cgglse */
void hook_cgglse(blasint *m, blasint *n, blasint *p, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *c, lapack_complex_float *d, lapack_complex_float *x, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgglse((void *) m, (void *) n, (void *) p, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) c, (void *) d, (void *) x, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgglse.timings[0] += (helpTimeStop - helpTime);
    data->cgglse.calls[0]++;
    
    return;
}


/* Wrapper for cggqrf */
void hook_cggqrf(blasint *n, blasint *m, blasint *p, lapack_complex_float *a, blasint *lda, lapack_complex_float *taua, lapack_complex_float *b, blasint *ldb, lapack_complex_float *taub, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cggqrf((void *) n, (void *) m, (void *) p, (void *) a, (void *) lda, (void *) taua, (void *) b, (void *) ldb, (void *) taub, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cggqrf.timings[0] += (helpTimeStop - helpTime);
    data->cggqrf.calls[0]++;
    
    return;
}


/* Wrapper for cggrqf */
void hook_cggrqf(blasint *m, blasint *p, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *taua, lapack_complex_float *b, blasint *ldb, lapack_complex_float *taub, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cggrqf((void *) m, (void *) p, (void *) n, (void *) a, (void *) lda, (void *) taua, (void *) b, (void *) ldb, (void *) taub, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cggrqf.timings[0] += (helpTimeStop - helpTime);
    data->cggrqf.calls[0]++;
    
    return;
}


/* Wrapper for cggsvd3 */
void hook_cggsvd3(char *jobu, char *jobv, char *jobq, blasint *m, blasint *n, blasint *p, blasint *k, blasint *l, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, float *alpha, float *beta, lapack_complex_float *u, blasint *ldu, lapack_complex_float *v, blasint *ldv, lapack_complex_float *q, blasint *ldq, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobv_len, flexiblas_fortran_charlen_t jobq_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cggsvd3((void *) jobu, (void *) jobv, (void *) jobq, (void *) m, (void *) n, (void *) p, (void *) k, (void *) l, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) alpha, (void *) beta, (void *) u, (void *) ldu, (void *) v, (void *) ldv, (void *) q, (void *) ldq, (void *) work, (void *) lwork, (void *) rwork, (void *) iwork, (void *) info, jobu_len, jobv_len, jobq_len);

    helpTimeStop = flexiblas_wtime();

    data->cggsvd3.timings[0] += (helpTimeStop - helpTime);
    data->cggsvd3.calls[0]++;
    
    return;
}


/* Wrapper for cggsvp3 */
void hook_cggsvp3(char *jobu, char *jobv, char *jobq, blasint *m, blasint *p, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, float *tola, float *tolb, blasint *k, blasint *l, lapack_complex_float *u, blasint *ldu, lapack_complex_float *v, blasint *ldv, lapack_complex_float *q, blasint *ldq, blasint *iwork, float *rwork, lapack_complex_float *tau, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobv_len, flexiblas_fortran_charlen_t jobq_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cggsvp3((void *) jobu, (void *) jobv, (void *) jobq, (void *) m, (void *) p, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) tola, (void *) tolb, (void *) k, (void *) l, (void *) u, (void *) ldu, (void *) v, (void *) ldv, (void *) q, (void *) ldq, (void *) iwork, (void *) rwork, (void *) tau, (void *) work, (void *) lwork, (void *) info, jobu_len, jobv_len, jobq_len);

    helpTimeStop = flexiblas_wtime();

    data->cggsvp3.timings[0] += (helpTimeStop - helpTime);
    data->cggsvp3.calls[0]++;
    
    return;
}


/* Wrapper for cgsvj0 */
void hook_cgsvj0(char *jobv, blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *d, float *sva, blasint *mv, lapack_complex_float *v, blasint *ldv, float *eps, float *sfmin, float *tol, blasint *nsweep, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobv_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgsvj0((void *) jobv, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) sva, (void *) mv, (void *) v, (void *) ldv, (void *) eps, (void *) sfmin, (void *) tol, (void *) nsweep, (void *) work, (void *) lwork, (void *) info, jobv_len);

    helpTimeStop = flexiblas_wtime();

    data->cgsvj0.timings[0] += (helpTimeStop - helpTime);
    data->cgsvj0.calls[0]++;
    
    return;
}


/* Wrapper for cgsvj1 */
void hook_cgsvj1(char *jobv, blasint *m, blasint *n, blasint *n1, lapack_complex_float *a, blasint *lda, lapack_complex_float *d, float *sva, blasint *mv, lapack_complex_float *v, blasint *ldv, float *eps, float *sfmin, float *tol, blasint *nsweep, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobv_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgsvj1((void *) jobv, (void *) m, (void *) n, (void *) n1, (void *) a, (void *) lda, (void *) d, (void *) sva, (void *) mv, (void *) v, (void *) ldv, (void *) eps, (void *) sfmin, (void *) tol, (void *) nsweep, (void *) work, (void *) lwork, (void *) info, jobv_len);

    helpTimeStop = flexiblas_wtime();

    data->cgsvj1.timings[0] += (helpTimeStop - helpTime);
    data->cgsvj1.calls[0]++;
    
    return;
}


/* Wrapper for cgtcon */
void hook_cgtcon(char *norm, blasint *n, lapack_complex_float *dl, lapack_complex_float *d, lapack_complex_float *du, lapack_complex_float *du2, blasint *ipiv, float *anorm, float *rcond, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgtcon((void *) norm, (void *) n, (void *) dl, (void *) d, (void *) du, (void *) du2, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) info, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->cgtcon.timings[0] += (helpTimeStop - helpTime);
    data->cgtcon.calls[0]++;
    
    return;
}


/* Wrapper for cgtrfs */
void hook_cgtrfs(char *trans, blasint *n, blasint *nrhs, lapack_complex_float *dl, lapack_complex_float *d, lapack_complex_float *du, lapack_complex_float *dlf, lapack_complex_float *df, lapack_complex_float *duf, lapack_complex_float *du2, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgtrfs((void *) trans, (void *) n, (void *) nrhs, (void *) dl, (void *) d, (void *) du, (void *) dlf, (void *) df, (void *) duf, (void *) du2, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cgtrfs.timings[0] += (helpTimeStop - helpTime);
    data->cgtrfs.calls[0]++;
    
    return;
}


/* Wrapper for cgtsv */
void hook_cgtsv(blasint *n, blasint *nrhs, lapack_complex_float *dl, lapack_complex_float *d, lapack_complex_float *du, lapack_complex_float *b, blasint *ldb, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgtsv((void *) n, (void *) nrhs, (void *) dl, (void *) d, (void *) du, (void *) b, (void *) ldb, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgtsv.timings[0] += (helpTimeStop - helpTime);
    data->cgtsv.calls[0]++;
    
    return;
}


/* Wrapper for cgtsvx */
void hook_cgtsvx(char *fact, char *trans, blasint *n, blasint *nrhs, lapack_complex_float *dl, lapack_complex_float *d, lapack_complex_float *du, lapack_complex_float *dlf, lapack_complex_float *df, lapack_complex_float *duf, lapack_complex_float *du2, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgtsvx((void *) fact, (void *) trans, (void *) n, (void *) nrhs, (void *) dl, (void *) d, (void *) du, (void *) dlf, (void *) df, (void *) duf, (void *) du2, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, fact_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cgtsvx.timings[0] += (helpTimeStop - helpTime);
    data->cgtsvx.calls[0]++;
    
    return;
}


/* Wrapper for cgttrf */
void hook_cgttrf(blasint *n, lapack_complex_float *dl, lapack_complex_float *d, lapack_complex_float *du, lapack_complex_float *du2, blasint *ipiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgttrf((void *) n, (void *) dl, (void *) d, (void *) du, (void *) du2, (void *) ipiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cgttrf.timings[0] += (helpTimeStop - helpTime);
    data->cgttrf.calls[0]++;
    
    return;
}


/* Wrapper for cgttrs */
void hook_cgttrs(char *trans, blasint *n, blasint *nrhs, lapack_complex_float *dl, lapack_complex_float *d, lapack_complex_float *du, lapack_complex_float *du2, blasint *ipiv, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgttrs((void *) trans, (void *) n, (void *) nrhs, (void *) dl, (void *) d, (void *) du, (void *) du2, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cgttrs.timings[0] += (helpTimeStop - helpTime);
    data->cgttrs.calls[0]++;
    
    return;
}


/* Wrapper for cgtts2 */
void hook_cgtts2(blasint *itrans, blasint *n, blasint *nrhs, lapack_complex_float *dl, lapack_complex_float *d, lapack_complex_float *du, lapack_complex_float *du2, blasint *ipiv, lapack_complex_float *b, blasint *ldb)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cgtts2((void *) itrans, (void *) n, (void *) nrhs, (void *) dl, (void *) d, (void *) du, (void *) du2, (void *) ipiv, (void *) b, (void *) ldb);

    helpTimeStop = flexiblas_wtime();

    data->cgtts2.timings[0] += (helpTimeStop - helpTime);
    data->cgtts2.calls[0]++;
    
    return;
}


/* Wrapper for chb2st_kernels */
void hook_chb2st_kernels(char *uplo, blaslogical *wantz, blasint *ttype, blasint *st, blasint *ed, blasint *sweep, blasint *n, blasint *nb, blasint *ib, lapack_complex_float *a, blasint *lda, lapack_complex_float *v, lapack_complex_float *tau, blasint *ldvt, lapack_complex_float *work, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chb2st_kernels((void *) uplo, (void *) wantz, (void *) ttype, (void *) st, (void *) ed, (void *) sweep, (void *) n, (void *) nb, (void *) ib, (void *) a, (void *) lda, (void *) v, (void *) tau, (void *) ldvt, (void *) work, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chb2st_kernels.timings[0] += (helpTimeStop - helpTime);
    data->chb2st_kernels.calls[0]++;
    
    return;
}


/* Wrapper for chbev */
void hook_chbev(char *jobz, char *uplo, blasint *n, blasint *kd, lapack_complex_float *ab, blasint *ldab, float *w, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chbev((void *) jobz, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) rwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chbev.timings[0] += (helpTimeStop - helpTime);
    data->chbev.calls[0]++;
    
    return;
}


/* Wrapper for chbev_2stage */
void hook_chbev_2stage(char *jobz, char *uplo, blasint *n, blasint *kd, lapack_complex_float *ab, blasint *ldab, float *w, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chbev_2stage((void *) jobz, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) rwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chbev_2stage.timings[0] += (helpTimeStop - helpTime);
    data->chbev_2stage.calls[0]++;
    
    return;
}


/* Wrapper for chbevd */
void hook_chbevd(char *jobz, char *uplo, blasint *n, blasint *kd, lapack_complex_float *ab, blasint *ldab, float *w, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *lrwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chbevd((void *) jobz, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) rwork, (void *) lrwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chbevd.timings[0] += (helpTimeStop - helpTime);
    data->chbevd.calls[0]++;
    
    return;
}


/* Wrapper for chbevd_2stage */
void hook_chbevd_2stage(char *jobz, char *uplo, blasint *n, blasint *kd, lapack_complex_float *ab, blasint *ldab, float *w, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *lrwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chbevd_2stage((void *) jobz, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) rwork, (void *) lrwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chbevd_2stage.timings[0] += (helpTimeStop - helpTime);
    data->chbevd_2stage.calls[0]++;
    
    return;
}


/* Wrapper for chbevx */
void hook_chbevx(char *jobz, char *range, char *uplo, blasint *n, blasint *kd, lapack_complex_float *ab, blasint *ldab, lapack_complex_float *q, blasint *ldq, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, float *rwork, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chbevx((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) q, (void *) ldq, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) rwork, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chbevx.timings[0] += (helpTimeStop - helpTime);
    data->chbevx.calls[0]++;
    
    return;
}


/* Wrapper for chbevx_2stage */
void hook_chbevx_2stage(char *jobz, char *range, char *uplo, blasint *n, blasint *kd, lapack_complex_float *ab, blasint *ldab, lapack_complex_float *q, blasint *ldq, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chbevx_2stage((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) q, (void *) ldq, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) rwork, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chbevx_2stage.timings[0] += (helpTimeStop - helpTime);
    data->chbevx_2stage.calls[0]++;
    
    return;
}


/* Wrapper for chbgst */
void hook_chbgst(char *vect, char *uplo, blasint *n, blasint *ka, blasint *kb, lapack_complex_float *ab, blasint *ldab, lapack_complex_float *bb, blasint *ldbb, lapack_complex_float *x, blasint *ldx, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t vect_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chbgst((void *) vect, (void *) uplo, (void *) n, (void *) ka, (void *) kb, (void *) ab, (void *) ldab, (void *) bb, (void *) ldbb, (void *) x, (void *) ldx, (void *) work, (void *) rwork, (void *) info, vect_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chbgst.timings[0] += (helpTimeStop - helpTime);
    data->chbgst.calls[0]++;
    
    return;
}


/* Wrapper for chbgv */
void hook_chbgv(char *jobz, char *uplo, blasint *n, blasint *ka, blasint *kb, lapack_complex_float *ab, blasint *ldab, lapack_complex_float *bb, blasint *ldbb, float *w, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chbgv((void *) jobz, (void *) uplo, (void *) n, (void *) ka, (void *) kb, (void *) ab, (void *) ldab, (void *) bb, (void *) ldbb, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) rwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chbgv.timings[0] += (helpTimeStop - helpTime);
    data->chbgv.calls[0]++;
    
    return;
}


/* Wrapper for chbgvd */
void hook_chbgvd(char *jobz, char *uplo, blasint *n, blasint *ka, blasint *kb, lapack_complex_float *ab, blasint *ldab, lapack_complex_float *bb, blasint *ldbb, float *w, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *lrwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chbgvd((void *) jobz, (void *) uplo, (void *) n, (void *) ka, (void *) kb, (void *) ab, (void *) ldab, (void *) bb, (void *) ldbb, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) rwork, (void *) lrwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chbgvd.timings[0] += (helpTimeStop - helpTime);
    data->chbgvd.calls[0]++;
    
    return;
}


/* Wrapper for chbgvx */
void hook_chbgvx(char *jobz, char *range, char *uplo, blasint *n, blasint *ka, blasint *kb, lapack_complex_float *ab, blasint *ldab, lapack_complex_float *bb, blasint *ldbb, lapack_complex_float *q, blasint *ldq, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, float *rwork, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chbgvx((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) ka, (void *) kb, (void *) ab, (void *) ldab, (void *) bb, (void *) ldbb, (void *) q, (void *) ldq, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) rwork, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chbgvx.timings[0] += (helpTimeStop - helpTime);
    data->chbgvx.calls[0]++;
    
    return;
}


/* Wrapper for chbtrd */
void hook_chbtrd(char *vect, char *uplo, blasint *n, blasint *kd, lapack_complex_float *ab, blasint *ldab, float *d, float *e, lapack_complex_float *q, blasint *ldq, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t vect_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chbtrd((void *) vect, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) d, (void *) e, (void *) q, (void *) ldq, (void *) work, (void *) info, vect_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chbtrd.timings[0] += (helpTimeStop - helpTime);
    data->chbtrd.calls[0]++;
    
    return;
}


/* Wrapper for checon */
void hook_checon(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, float *anorm, float *rcond, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_checon((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->checon.timings[0] += (helpTimeStop - helpTime);
    data->checon.calls[0]++;
    
    return;
}


/* Wrapper for checon_3 */
void hook_checon_3(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *e, blasint *ipiv, float *anorm, float *rcond, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_checon_3((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->checon_3.timings[0] += (helpTimeStop - helpTime);
    data->checon_3.calls[0]++;
    
    return;
}


/* Wrapper for checon_rook */
void hook_checon_rook(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, float *anorm, float *rcond, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_checon_rook((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->checon_rook.timings[0] += (helpTimeStop - helpTime);
    data->checon_rook.calls[0]++;
    
    return;
}


/* Wrapper for cheequb */
void hook_cheequb(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, float *s, float *scond, float *amax, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cheequb((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) s, (void *) scond, (void *) amax, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cheequb.timings[0] += (helpTimeStop - helpTime);
    data->cheequb.calls[0]++;
    
    return;
}


/* Wrapper for cheev */
void hook_cheev(char *jobz, char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, float *w, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cheev((void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) w, (void *) work, (void *) lwork, (void *) rwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cheev.timings[0] += (helpTimeStop - helpTime);
    data->cheev.calls[0]++;
    
    return;
}


/* Wrapper for cheev_2stage */
void hook_cheev_2stage(char *jobz, char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, float *w, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cheev_2stage((void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) w, (void *) work, (void *) lwork, (void *) rwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cheev_2stage.timings[0] += (helpTimeStop - helpTime);
    data->cheev_2stage.calls[0]++;
    
    return;
}


/* Wrapper for cheevd */
void hook_cheevd(char *jobz, char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, float *w, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *lrwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cheevd((void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) w, (void *) work, (void *) lwork, (void *) rwork, (void *) lrwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cheevd.timings[0] += (helpTimeStop - helpTime);
    data->cheevd.calls[0]++;
    
    return;
}


/* Wrapper for cheevd_2stage */
void hook_cheevd_2stage(char *jobz, char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, float *w, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *lrwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cheevd_2stage((void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) w, (void *) work, (void *) lwork, (void *) rwork, (void *) lrwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cheevd_2stage.timings[0] += (helpTimeStop - helpTime);
    data->cheevd_2stage.calls[0]++;
    
    return;
}


/* Wrapper for cheevr */
void hook_cheevr(char *jobz, char *range, char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, lapack_complex_float *z, blasint *ldz, blasint *isuppz, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *lrwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cheevr((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) isuppz, (void *) work, (void *) lwork, (void *) rwork, (void *) lrwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cheevr.timings[0] += (helpTimeStop - helpTime);
    data->cheevr.calls[0]++;
    
    return;
}


/* Wrapper for cheevr_2stage */
void hook_cheevr_2stage(char *jobz, char *range, char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, lapack_complex_float *z, blasint *ldz, blasint *isuppz, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *lrwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cheevr_2stage((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) isuppz, (void *) work, (void *) lwork, (void *) rwork, (void *) lrwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cheevr_2stage.timings[0] += (helpTimeStop - helpTime);
    data->cheevr_2stage.calls[0]++;
    
    return;
}


/* Wrapper for cheevx */
void hook_cheevx(char *jobz, char *range, char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cheevx((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) rwork, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cheevx.timings[0] += (helpTimeStop - helpTime);
    data->cheevx.calls[0]++;
    
    return;
}


/* Wrapper for cheevx_2stage */
void hook_cheevx_2stage(char *jobz, char *range, char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cheevx_2stage((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) rwork, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cheevx_2stage.timings[0] += (helpTimeStop - helpTime);
    data->cheevx_2stage.calls[0]++;
    
    return;
}


/* Wrapper for chegs2 */
void hook_chegs2(blasint *itype, char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chegs2((void *) itype, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chegs2.timings[0] += (helpTimeStop - helpTime);
    data->chegs2.calls[0]++;
    
    return;
}


/* Wrapper for chegst */
void hook_chegst(blasint *itype, char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chegst((void *) itype, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chegst.timings[0] += (helpTimeStop - helpTime);
    data->chegst.calls[0]++;
    
    return;
}


/* Wrapper for chegv */
void hook_chegv(blasint *itype, char *jobz, char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, float *w, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chegv((void *) itype, (void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) w, (void *) work, (void *) lwork, (void *) rwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chegv.timings[0] += (helpTimeStop - helpTime);
    data->chegv.calls[0]++;
    
    return;
}


/* Wrapper for chegv_2stage */
void hook_chegv_2stage(blasint *itype, char *jobz, char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, float *w, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chegv_2stage((void *) itype, (void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) w, (void *) work, (void *) lwork, (void *) rwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chegv_2stage.timings[0] += (helpTimeStop - helpTime);
    data->chegv_2stage.calls[0]++;
    
    return;
}


/* Wrapper for chegvd */
void hook_chegvd(blasint *itype, char *jobz, char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, float *w, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *lrwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chegvd((void *) itype, (void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) w, (void *) work, (void *) lwork, (void *) rwork, (void *) lrwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chegvd.timings[0] += (helpTimeStop - helpTime);
    data->chegvd.calls[0]++;
    
    return;
}


/* Wrapper for chegvx */
void hook_chegvx(blasint *itype, char *jobz, char *range, char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chegvx((void *) itype, (void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) rwork, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chegvx.timings[0] += (helpTimeStop - helpTime);
    data->chegvx.calls[0]++;
    
    return;
}


/* Wrapper for cherfs */
void hook_cherfs(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *af, blasint *ldaf, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cherfs((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) af, (void *) ldaf, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cherfs.timings[0] += (helpTimeStop - helpTime);
    data->cherfs.calls[0]++;
    
    return;
}


/* Wrapper for chesv */
void hook_chesv(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chesv((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chesv.timings[0] += (helpTimeStop - helpTime);
    data->chesv.calls[0]++;
    
    return;
}


/* Wrapper for chesv_aa */
void hook_chesv_aa(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chesv_aa((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chesv_aa.timings[0] += (helpTimeStop - helpTime);
    data->chesv_aa.calls[0]++;
    
    return;
}


/* Wrapper for chesv_aa_2stage */
void hook_chesv_aa_2stage(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *tb, blasint *ltb, blasint *ipiv, blasint *ipiv2, lapack_complex_float *b, blasint *ldb, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chesv_aa_2stage((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) tb, (void *) ltb, (void *) ipiv, (void *) ipiv2, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chesv_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->chesv_aa_2stage.calls[0]++;
    
    return;
}


/* Wrapper for chesv_rk */
void hook_chesv_rk(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *e, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chesv_rk((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chesv_rk.timings[0] += (helpTimeStop - helpTime);
    data->chesv_rk.calls[0]++;
    
    return;
}


/* Wrapper for chesv_rook */
void hook_chesv_rook(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chesv_rook((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chesv_rook.timings[0] += (helpTimeStop - helpTime);
    data->chesv_rook.calls[0]++;
    
    return;
}


/* Wrapper for chesvx */
void hook_chesvx(char *fact, char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *af, blasint *ldaf, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chesvx((void *) fact, (void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) af, (void *) ldaf, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) lwork, (void *) rwork, (void *) info, fact_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chesvx.timings[0] += (helpTimeStop - helpTime);
    data->chesvx.calls[0]++;
    
    return;
}


/* Wrapper for cheswapr */
void hook_cheswapr(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *i1, blasint *i2, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cheswapr((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) i1, (void *) i2, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cheswapr.timings[0] += (helpTimeStop - helpTime);
    data->cheswapr.calls[0]++;
    
    return;
}


/* Wrapper for chetd2 */
void hook_chetd2(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, float *d, float *e, lapack_complex_float *tau, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetd2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) e, (void *) tau, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetd2.timings[0] += (helpTimeStop - helpTime);
    data->chetd2.calls[0]++;
    
    return;
}


/* Wrapper for chetf2 */
void hook_chetf2(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetf2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetf2.timings[0] += (helpTimeStop - helpTime);
    data->chetf2.calls[0]++;
    
    return;
}


/* Wrapper for chetf2_rk */
void hook_chetf2_rk(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *e, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetf2_rk((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetf2_rk.timings[0] += (helpTimeStop - helpTime);
    data->chetf2_rk.calls[0]++;
    
    return;
}


/* Wrapper for chetf2_rook */
void hook_chetf2_rook(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetf2_rook((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetf2_rook.timings[0] += (helpTimeStop - helpTime);
    data->chetf2_rook.calls[0]++;
    
    return;
}


/* Wrapper for chetrd */
void hook_chetrd(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, float *d, float *e, lapack_complex_float *tau, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetrd((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) e, (void *) tau, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetrd.timings[0] += (helpTimeStop - helpTime);
    data->chetrd.calls[0]++;
    
    return;
}


/* Wrapper for chetrd_2stage */
void hook_chetrd_2stage(char *vect, char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, float *d, float *e, lapack_complex_float *tau, lapack_complex_float *hous2, blasint *lhous2, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t vect_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetrd_2stage((void *) vect, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) e, (void *) tau, (void *) hous2, (void *) lhous2, (void *) work, (void *) lwork, (void *) info, vect_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetrd_2stage.timings[0] += (helpTimeStop - helpTime);
    data->chetrd_2stage.calls[0]++;
    
    return;
}


/* Wrapper for chetrd_hb2st */
void hook_chetrd_hb2st(char *stage1, char *vect, char *uplo, blasint *n, blasint *kd, lapack_complex_float *ab, blasint *ldab, float *d, float *e, lapack_complex_float *hous, blasint *lhous, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t stage1_len, flexiblas_fortran_charlen_t vect_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetrd_hb2st((void *) stage1, (void *) vect, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) d, (void *) e, (void *) hous, (void *) lhous, (void *) work, (void *) lwork, (void *) info, stage1_len, vect_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetrd_hb2st.timings[0] += (helpTimeStop - helpTime);
    data->chetrd_hb2st.calls[0]++;
    
    return;
}


/* Wrapper for chetrd_he2hb */
void hook_chetrd_he2hb(char *uplo, blasint *n, blasint *kd, lapack_complex_float *a, blasint *lda, lapack_complex_float *ab, blasint *ldab, lapack_complex_float *tau, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetrd_he2hb((void *) uplo, (void *) n, (void *) kd, (void *) a, (void *) lda, (void *) ab, (void *) ldab, (void *) tau, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetrd_he2hb.timings[0] += (helpTimeStop - helpTime);
    data->chetrd_he2hb.calls[0]++;
    
    return;
}


/* Wrapper for chetrf */
void hook_chetrf(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetrf((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetrf.timings[0] += (helpTimeStop - helpTime);
    data->chetrf.calls[0]++;
    
    return;
}


/* Wrapper for chetrf_aa */
void hook_chetrf_aa(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetrf_aa((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetrf_aa.timings[0] += (helpTimeStop - helpTime);
    data->chetrf_aa.calls[0]++;
    
    return;
}


/* Wrapper for chetrf_aa_2stage */
void hook_chetrf_aa_2stage(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *tb, blasint *ltb, blasint *ipiv, blasint *ipiv2, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetrf_aa_2stage((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) tb, (void *) ltb, (void *) ipiv, (void *) ipiv2, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetrf_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->chetrf_aa_2stage.calls[0]++;
    
    return;
}


/* Wrapper for chetrf_rk */
void hook_chetrf_rk(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *e, blasint *ipiv, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetrf_rk((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetrf_rk.timings[0] += (helpTimeStop - helpTime);
    data->chetrf_rk.calls[0]++;
    
    return;
}


/* Wrapper for chetrf_rook */
void hook_chetrf_rook(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetrf_rook((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetrf_rook.timings[0] += (helpTimeStop - helpTime);
    data->chetrf_rook.calls[0]++;
    
    return;
}


/* Wrapper for chetri */
void hook_chetri(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetri((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetri.timings[0] += (helpTimeStop - helpTime);
    data->chetri.calls[0]++;
    
    return;
}


/* Wrapper for chetri2 */
void hook_chetri2(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetri2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetri2.timings[0] += (helpTimeStop - helpTime);
    data->chetri2.calls[0]++;
    
    return;
}


/* Wrapper for chetri2x */
void hook_chetri2x(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *work, blasint *nb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetri2x((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) nb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetri2x.timings[0] += (helpTimeStop - helpTime);
    data->chetri2x.calls[0]++;
    
    return;
}


/* Wrapper for chetri_3 */
void hook_chetri_3(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *e, blasint *ipiv, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetri_3((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetri_3.timings[0] += (helpTimeStop - helpTime);
    data->chetri_3.calls[0]++;
    
    return;
}


/* Wrapper for chetri_3x */
void hook_chetri_3x(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *e, blasint *ipiv, lapack_complex_float *work, blasint *nb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetri_3x((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) work, (void *) nb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetri_3x.timings[0] += (helpTimeStop - helpTime);
    data->chetri_3x.calls[0]++;
    
    return;
}


/* Wrapper for chetri_rook */
void hook_chetri_rook(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetri_rook((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetri_rook.timings[0] += (helpTimeStop - helpTime);
    data->chetri_rook.calls[0]++;
    
    return;
}


/* Wrapper for chetrs */
void hook_chetrs(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetrs((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetrs.timings[0] += (helpTimeStop - helpTime);
    data->chetrs.calls[0]++;
    
    return;
}


/* Wrapper for chetrs2 */
void hook_chetrs2(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetrs2((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetrs2.timings[0] += (helpTimeStop - helpTime);
    data->chetrs2.calls[0]++;
    
    return;
}


/* Wrapper for chetrs_3 */
void hook_chetrs_3(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *e, blasint *ipiv, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetrs_3((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetrs_3.timings[0] += (helpTimeStop - helpTime);
    data->chetrs_3.calls[0]++;
    
    return;
}


/* Wrapper for chetrs_aa */
void hook_chetrs_aa(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetrs_aa((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetrs_aa.timings[0] += (helpTimeStop - helpTime);
    data->chetrs_aa.calls[0]++;
    
    return;
}


/* Wrapper for chetrs_aa_2stage */
void hook_chetrs_aa_2stage(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *tb, blasint *ltb, blasint *ipiv, blasint *ipiv2, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetrs_aa_2stage((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) tb, (void *) ltb, (void *) ipiv, (void *) ipiv2, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetrs_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->chetrs_aa_2stage.calls[0]++;
    
    return;
}


/* Wrapper for chetrs_rook */
void hook_chetrs_rook(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chetrs_rook((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chetrs_rook.timings[0] += (helpTimeStop - helpTime);
    data->chetrs_rook.calls[0]++;
    
    return;
}


/* Wrapper for chfrk */
void hook_chfrk(char *transr, char *uplo, char *trans, blasint *n, blasint *k, float *alpha, lapack_complex_float *a, blasint *lda, float *beta, lapack_complex_float *c, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chfrk((void *) transr, (void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, transr_len, uplo_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->chfrk.timings[0] += (helpTimeStop - helpTime);
    data->chfrk.calls[0]++;
    
    return;
}


/* Wrapper for chgeqz */
void hook_chgeqz(char *job, char *compq, char *compz, blasint *n, blasint *ilo, blasint *ihi, lapack_complex_float *h, blasint *ldh, lapack_complex_float *t, blasint *ldt, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *q, blasint *ldq, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t compq_len, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chgeqz((void *) job, (void *) compq, (void *) compz, (void *) n, (void *) ilo, (void *) ihi, (void *) h, (void *) ldh, (void *) t, (void *) ldt, (void *) alpha, (void *) beta, (void *) q, (void *) ldq, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) rwork, (void *) info, job_len, compq_len, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->chgeqz.timings[0] += (helpTimeStop - helpTime);
    data->chgeqz.calls[0]++;
    
    return;
}


/* Wrapper for chla_transtype */
void hook_chla_transtype(char *result_chla_transtype, flexiblas_fortran_charlen_t result_chla_transtype_len, blasint *trans)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chla_transtype((void *) result_chla_transtype, result_chla_transtype_len, (void *) trans);

    helpTimeStop = flexiblas_wtime();

    data->chla_transtype.timings[0] += (helpTimeStop - helpTime);
    data->chla_transtype.calls[0]++;
    
    return;
}


/* Wrapper for chpcon */
void hook_chpcon(char *uplo, blasint *n, lapack_complex_float *ap, blasint *ipiv, float *anorm, float *rcond, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chpcon((void *) uplo, (void *) n, (void *) ap, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chpcon.timings[0] += (helpTimeStop - helpTime);
    data->chpcon.calls[0]++;
    
    return;
}


/* Wrapper for chpev */
void hook_chpev(char *jobz, char *uplo, blasint *n, lapack_complex_float *ap, float *w, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chpev((void *) jobz, (void *) uplo, (void *) n, (void *) ap, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) rwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chpev.timings[0] += (helpTimeStop - helpTime);
    data->chpev.calls[0]++;
    
    return;
}


/* Wrapper for chpevd */
void hook_chpevd(char *jobz, char *uplo, blasint *n, lapack_complex_float *ap, float *w, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *lrwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chpevd((void *) jobz, (void *) uplo, (void *) n, (void *) ap, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) rwork, (void *) lrwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chpevd.timings[0] += (helpTimeStop - helpTime);
    data->chpevd.calls[0]++;
    
    return;
}


/* Wrapper for chpevx */
void hook_chpevx(char *jobz, char *range, char *uplo, blasint *n, lapack_complex_float *ap, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, float *rwork, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chpevx((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) ap, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) rwork, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chpevx.timings[0] += (helpTimeStop - helpTime);
    data->chpevx.calls[0]++;
    
    return;
}


/* Wrapper for chpgst */
void hook_chpgst(blasint *itype, char *uplo, blasint *n, lapack_complex_float *ap, lapack_complex_float *bp, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chpgst((void *) itype, (void *) uplo, (void *) n, (void *) ap, (void *) bp, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chpgst.timings[0] += (helpTimeStop - helpTime);
    data->chpgst.calls[0]++;
    
    return;
}


/* Wrapper for chpgv */
void hook_chpgv(blasint *itype, char *jobz, char *uplo, blasint *n, lapack_complex_float *ap, lapack_complex_float *bp, float *w, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chpgv((void *) itype, (void *) jobz, (void *) uplo, (void *) n, (void *) ap, (void *) bp, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) rwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chpgv.timings[0] += (helpTimeStop - helpTime);
    data->chpgv.calls[0]++;
    
    return;
}


/* Wrapper for chpgvd */
void hook_chpgvd(blasint *itype, char *jobz, char *uplo, blasint *n, lapack_complex_float *ap, lapack_complex_float *bp, float *w, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *lrwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chpgvd((void *) itype, (void *) jobz, (void *) uplo, (void *) n, (void *) ap, (void *) bp, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) rwork, (void *) lrwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chpgvd.timings[0] += (helpTimeStop - helpTime);
    data->chpgvd.calls[0]++;
    
    return;
}


/* Wrapper for chpgvx */
void hook_chpgvx(blasint *itype, char *jobz, char *range, char *uplo, blasint *n, lapack_complex_float *ap, lapack_complex_float *bp, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, float *rwork, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chpgvx((void *) itype, (void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) ap, (void *) bp, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) rwork, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chpgvx.timings[0] += (helpTimeStop - helpTime);
    data->chpgvx.calls[0]++;
    
    return;
}


/* Wrapper for chprfs */
void hook_chprfs(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *ap, lapack_complex_float *afp, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chprfs((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) afp, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chprfs.timings[0] += (helpTimeStop - helpTime);
    data->chprfs.calls[0]++;
    
    return;
}


/* Wrapper for chpsv */
void hook_chpsv(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *ap, blasint *ipiv, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chpsv((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chpsv.timings[0] += (helpTimeStop - helpTime);
    data->chpsv.calls[0]++;
    
    return;
}


/* Wrapper for chpsvx */
void hook_chpsvx(char *fact, char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *ap, lapack_complex_float *afp, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chpsvx((void *) fact, (void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) afp, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, fact_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chpsvx.timings[0] += (helpTimeStop - helpTime);
    data->chpsvx.calls[0]++;
    
    return;
}


/* Wrapper for chptrd */
void hook_chptrd(char *uplo, blasint *n, lapack_complex_float *ap, float *d, float *e, lapack_complex_float *tau, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chptrd((void *) uplo, (void *) n, (void *) ap, (void *) d, (void *) e, (void *) tau, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chptrd.timings[0] += (helpTimeStop - helpTime);
    data->chptrd.calls[0]++;
    
    return;
}


/* Wrapper for chptrf */
void hook_chptrf(char *uplo, blasint *n, lapack_complex_float *ap, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chptrf((void *) uplo, (void *) n, (void *) ap, (void *) ipiv, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chptrf.timings[0] += (helpTimeStop - helpTime);
    data->chptrf.calls[0]++;
    
    return;
}


/* Wrapper for chptri */
void hook_chptri(char *uplo, blasint *n, lapack_complex_float *ap, blasint *ipiv, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chptri((void *) uplo, (void *) n, (void *) ap, (void *) ipiv, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chptri.timings[0] += (helpTimeStop - helpTime);
    data->chptri.calls[0]++;
    
    return;
}


/* Wrapper for chptrs */
void hook_chptrs(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *ap, blasint *ipiv, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chptrs((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->chptrs.timings[0] += (helpTimeStop - helpTime);
    data->chptrs.calls[0]++;
    
    return;
}


/* Wrapper for chsein */
void hook_chsein(char *side, char *eigsrc, char *initv, blaslogical *select, blasint *n, lapack_complex_float *h, blasint *ldh, lapack_complex_float *w, lapack_complex_float *vl, blasint *ldvl, lapack_complex_float *vr, blasint *ldvr, blasint *mm, blasint *m, lapack_complex_float *work, float *rwork, blasint *ifaill, blasint *ifailr, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t eigsrc_len, flexiblas_fortran_charlen_t initv_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chsein((void *) side, (void *) eigsrc, (void *) initv, (void *) select, (void *) n, (void *) h, (void *) ldh, (void *) w, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) mm, (void *) m, (void *) work, (void *) rwork, (void *) ifaill, (void *) ifailr, (void *) info, side_len, eigsrc_len, initv_len);

    helpTimeStop = flexiblas_wtime();

    data->chsein.timings[0] += (helpTimeStop - helpTime);
    data->chsein.calls[0]++;
    
    return;
}


/* Wrapper for chseqr */
void hook_chseqr(char *job, char *compz, blasint *n, blasint *ilo, blasint *ihi, lapack_complex_float *h, blasint *ldh, lapack_complex_float *w, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_chseqr((void *) job, (void *) compz, (void *) n, (void *) ilo, (void *) ihi, (void *) h, (void *) ldh, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) info, job_len, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->chseqr.timings[0] += (helpTimeStop - helpTime);
    data->chseqr.calls[0]++;
    
    return;
}


/* Wrapper for clabrd */
void hook_clabrd(blasint *m, blasint *n, blasint *nb, lapack_complex_float *a, blasint *lda, float *d, float *e, lapack_complex_float *tauq, lapack_complex_float *taup, lapack_complex_float *x, blasint *ldx, lapack_complex_float *y, blasint *ldy)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clabrd((void *) m, (void *) n, (void *) nb, (void *) a, (void *) lda, (void *) d, (void *) e, (void *) tauq, (void *) taup, (void *) x, (void *) ldx, (void *) y, (void *) ldy);

    helpTimeStop = flexiblas_wtime();

    data->clabrd.timings[0] += (helpTimeStop - helpTime);
    data->clabrd.calls[0]++;
    
    return;
}


/* Wrapper for clacgv */
void hook_clacgv(blasint *n, lapack_complex_float *x, blasint *incx)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clacgv((void *) n, (void *) x, (void *) incx);

    helpTimeStop = flexiblas_wtime();

    data->clacgv.timings[0] += (helpTimeStop - helpTime);
    data->clacgv.calls[0]++;
    
    return;
}


/* Wrapper for clacn2 */
void hook_clacn2(blasint *n, lapack_complex_float *v, lapack_complex_float *x, float *est, blasint *kase, blasint *isave)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clacn2((void *) n, (void *) v, (void *) x, (void *) est, (void *) kase, (void *) isave);

    helpTimeStop = flexiblas_wtime();

    data->clacn2.timings[0] += (helpTimeStop - helpTime);
    data->clacn2.calls[0]++;
    
    return;
}


/* Wrapper for clacon */
void hook_clacon(blasint *n, lapack_complex_float *v, lapack_complex_float *x, float *est, blasint *kase)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clacon((void *) n, (void *) v, (void *) x, (void *) est, (void *) kase);

    helpTimeStop = flexiblas_wtime();

    data->clacon.timings[0] += (helpTimeStop - helpTime);
    data->clacon.calls[0]++;
    
    return;
}


/* Wrapper for clacp2 */
void hook_clacp2(char *uplo, blasint *m, blasint *n, float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clacp2((void *) uplo, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clacp2.timings[0] += (helpTimeStop - helpTime);
    data->clacp2.calls[0]++;
    
    return;
}


/* Wrapper for clacpy */
void hook_clacpy(char *uplo, blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clacpy((void *) uplo, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clacpy.timings[0] += (helpTimeStop - helpTime);
    data->clacpy.calls[0]++;
    
    return;
}


/* Wrapper for clacrm */
void hook_clacrm(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, float *b, blasint *ldb, lapack_complex_float *c, blasint *ldc, float *rwork)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clacrm((void *) m, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) c, (void *) ldc, (void *) rwork);

    helpTimeStop = flexiblas_wtime();

    data->clacrm.timings[0] += (helpTimeStop - helpTime);
    data->clacrm.calls[0]++;
    
    return;
}


/* Wrapper for clacrt */
void hook_clacrt(blasint *n, lapack_complex_float *cx, blasint *incx, lapack_complex_float *cy, blasint *incy, lapack_complex_float *c, lapack_complex_float *s)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clacrt((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy, (void *) c, (void *) s);

    helpTimeStop = flexiblas_wtime();

    data->clacrt.timings[0] += (helpTimeStop - helpTime);
    data->clacrt.calls[0]++;
    
    return;
}

#ifdef FLEXIBLAS_ABI_GNU
/* Wrapper for cladiv */
float _Complex hook_cladiv(lapack_complex_float *x, lapack_complex_float *y)
{
    double helpTime;
    double helpTimeStop;
    float _Complex return_value;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cladiv((void *) &return_value, (void *) x, (void *) y);

    helpTimeStop = flexiblas_wtime();

    data->cladiv.timings[0] += (helpTimeStop - helpTime);
    data->cladiv.calls[0]++;
    
    return return_value;
}


#else
/* Wrapper for cladiv */
void hook_cladiv(float _Complex *return_value, lapack_complex_float *x, lapack_complex_float *y)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cladiv((void *) return_value, (void *) x, (void *) y);

    helpTimeStop = flexiblas_wtime();

    data->cladiv.timings[0] += (helpTimeStop - helpTime);
    data->cladiv.calls[0]++;
    
}

#endif

/* Wrapper for claed0 */
void hook_claed0(blasint *qsiz, blasint *n, float *d, float *e, lapack_complex_float *q, blasint *ldq, lapack_complex_float *qstore, blasint *ldqs, float *rwork, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claed0((void *) qsiz, (void *) n, (void *) d, (void *) e, (void *) q, (void *) ldq, (void *) qstore, (void *) ldqs, (void *) rwork, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->claed0.timings[0] += (helpTimeStop - helpTime);
    data->claed0.calls[0]++;
    
    return;
}


/* Wrapper for claed7 */
void hook_claed7(blasint *n, blasint *cutpnt, blasint *qsiz, blasint *tlvls, blasint *curlvl, blasint *curpbm, float *d, lapack_complex_float *q, blasint *ldq, float *rho, blasint *indxq, float *qstore, blasint *qptr, blasint *prmptr, blasint *perm, blasint *givptr, blasint *givcol, float *givnum, lapack_complex_float *work, float *rwork, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claed7((void *) n, (void *) cutpnt, (void *) qsiz, (void *) tlvls, (void *) curlvl, (void *) curpbm, (void *) d, (void *) q, (void *) ldq, (void *) rho, (void *) indxq, (void *) qstore, (void *) qptr, (void *) prmptr, (void *) perm, (void *) givptr, (void *) givcol, (void *) givnum, (void *) work, (void *) rwork, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->claed7.timings[0] += (helpTimeStop - helpTime);
    data->claed7.calls[0]++;
    
    return;
}


/* Wrapper for claed8 */
void hook_claed8(blasint *k, blasint *n, blasint *qsiz, lapack_complex_float *q, blasint *ldq, float *d, float *rho, blasint *cutpnt, float *z, float *dlamda, lapack_complex_float *q2, blasint *ldq2, float *w, blasint *indxp, blasint *indx, blasint *indxq, blasint *perm, blasint *givptr, blasint *givcol, float *givnum, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claed8((void *) k, (void *) n, (void *) qsiz, (void *) q, (void *) ldq, (void *) d, (void *) rho, (void *) cutpnt, (void *) z, (void *) dlamda, (void *) q2, (void *) ldq2, (void *) w, (void *) indxp, (void *) indx, (void *) indxq, (void *) perm, (void *) givptr, (void *) givcol, (void *) givnum, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->claed8.timings[0] += (helpTimeStop - helpTime);
    data->claed8.calls[0]++;
    
    return;
}


/* Wrapper for claein */
void hook_claein(blaslogical *rightv, blaslogical *noinit, blasint *n, lapack_complex_float *h, blasint *ldh, lapack_complex_float *w, lapack_complex_float *v, lapack_complex_float *b, blasint *ldb, float *rwork, float *eps3, float *smlnum, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claein((void *) rightv, (void *) noinit, (void *) n, (void *) h, (void *) ldh, (void *) w, (void *) v, (void *) b, (void *) ldb, (void *) rwork, (void *) eps3, (void *) smlnum, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->claein.timings[0] += (helpTimeStop - helpTime);
    data->claein.calls[0]++;
    
    return;
}


/* Wrapper for claesy */
void hook_claesy(lapack_complex_float *a, lapack_complex_float *b, lapack_complex_float *c, lapack_complex_float *rt1, lapack_complex_float *rt2, lapack_complex_float *evscal, lapack_complex_float *cs1, lapack_complex_float *sn1)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claesy((void *) a, (void *) b, (void *) c, (void *) rt1, (void *) rt2, (void *) evscal, (void *) cs1, (void *) sn1);

    helpTimeStop = flexiblas_wtime();

    data->claesy.timings[0] += (helpTimeStop - helpTime);
    data->claesy.calls[0]++;
    
    return;
}


/* Wrapper for claev2 */
void hook_claev2(lapack_complex_float *a, lapack_complex_float *b, lapack_complex_float *c, float *rt1, float *rt2, float *cs1, lapack_complex_float *sn1)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claev2((void *) a, (void *) b, (void *) c, (void *) rt1, (void *) rt2, (void *) cs1, (void *) sn1);

    helpTimeStop = flexiblas_wtime();

    data->claev2.timings[0] += (helpTimeStop - helpTime);
    data->claev2.calls[0]++;
    
    return;
}


/* Wrapper for clag2z */
void hook_clag2z(blasint *m, blasint *n, lapack_complex_float *sa, blasint *ldsa, lapack_complex_double *a, blasint *lda, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clag2z((void *) m, (void *) n, (void *) sa, (void *) ldsa, (void *) a, (void *) lda, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->clag2z.timings[0] += (helpTimeStop - helpTime);
    data->clag2z.calls[0]++;
    
    return;
}


/* Wrapper for clags2 */
void hook_clags2(blaslogical *upper, float *a1, lapack_complex_float *a2, float *a3, float *b1, lapack_complex_float *b2, float *b3, float *csu, lapack_complex_float *snu, float *csv, lapack_complex_float *snv, float *csq, lapack_complex_float *snq)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clags2((void *) upper, (void *) a1, (void *) a2, (void *) a3, (void *) b1, (void *) b2, (void *) b3, (void *) csu, (void *) snu, (void *) csv, (void *) snv, (void *) csq, (void *) snq);

    helpTimeStop = flexiblas_wtime();

    data->clags2.timings[0] += (helpTimeStop - helpTime);
    data->clags2.calls[0]++;
    
    return;
}


/* Wrapper for clagtm */
void hook_clagtm(char *trans, blasint *n, blasint *nrhs, float *alpha, lapack_complex_float *dl, lapack_complex_float *d, lapack_complex_float *du, lapack_complex_float *x, blasint *ldx, float *beta, lapack_complex_float *b, blasint *ldb, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clagtm((void *) trans, (void *) n, (void *) nrhs, (void *) alpha, (void *) dl, (void *) d, (void *) du, (void *) x, (void *) ldx, (void *) beta, (void *) b, (void *) ldb, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->clagtm.timings[0] += (helpTimeStop - helpTime);
    data->clagtm.calls[0]++;
    
    return;
}


/* Wrapper for clahef */
void hook_clahef(char *uplo, blasint *n, blasint *nb, blasint *kb, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *w, blasint *ldw, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clahef((void *) uplo, (void *) n, (void *) nb, (void *) kb, (void *) a, (void *) lda, (void *) ipiv, (void *) w, (void *) ldw, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clahef.timings[0] += (helpTimeStop - helpTime);
    data->clahef.calls[0]++;
    
    return;
}


/* Wrapper for clahef_aa */
void hook_clahef_aa(char *uplo, blasint *j1, blasint *m, blasint *nb, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *h, blasint *ldh, lapack_complex_float *work, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clahef_aa((void *) uplo, (void *) j1, (void *) m, (void *) nb, (void *) a, (void *) lda, (void *) ipiv, (void *) h, (void *) ldh, (void *) work, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clahef_aa.timings[0] += (helpTimeStop - helpTime);
    data->clahef_aa.calls[0]++;
    
    return;
}


/* Wrapper for clahef_rk */
void hook_clahef_rk(char *uplo, blasint *n, blasint *nb, blasint *kb, lapack_complex_float *a, blasint *lda, lapack_complex_float *e, blasint *ipiv, lapack_complex_float *w, blasint *ldw, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clahef_rk((void *) uplo, (void *) n, (void *) nb, (void *) kb, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) w, (void *) ldw, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clahef_rk.timings[0] += (helpTimeStop - helpTime);
    data->clahef_rk.calls[0]++;
    
    return;
}


/* Wrapper for clahef_rook */
void hook_clahef_rook(char *uplo, blasint *n, blasint *nb, blasint *kb, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *w, blasint *ldw, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clahef_rook((void *) uplo, (void *) n, (void *) nb, (void *) kb, (void *) a, (void *) lda, (void *) ipiv, (void *) w, (void *) ldw, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clahef_rook.timings[0] += (helpTimeStop - helpTime);
    data->clahef_rook.calls[0]++;
    
    return;
}


/* Wrapper for clahqr */
void hook_clahqr(blaslogical *wantt, blaslogical *wantz, blasint *n, blasint *ilo, blasint *ihi, lapack_complex_float *h, blasint *ldh, lapack_complex_float *w, blasint *iloz, blasint *ihiz, lapack_complex_float *z, blasint *ldz, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clahqr((void *) wantt, (void *) wantz, (void *) n, (void *) ilo, (void *) ihi, (void *) h, (void *) ldh, (void *) w, (void *) iloz, (void *) ihiz, (void *) z, (void *) ldz, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->clahqr.timings[0] += (helpTimeStop - helpTime);
    data->clahqr.calls[0]++;
    
    return;
}


/* Wrapper for clahr2 */
void hook_clahr2(blasint *n, blasint *k, blasint *nb, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *t, blasint *ldt, lapack_complex_float *y, blasint *ldy)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clahr2((void *) n, (void *) k, (void *) nb, (void *) a, (void *) lda, (void *) tau, (void *) t, (void *) ldt, (void *) y, (void *) ldy);

    helpTimeStop = flexiblas_wtime();

    data->clahr2.timings[0] += (helpTimeStop - helpTime);
    data->clahr2.calls[0]++;
    
    return;
}


/* Wrapper for claic1 */
void hook_claic1(blasint *job, blasint *j, lapack_complex_float *x, float *sest, lapack_complex_float *w, lapack_complex_float *gamma, float *sestpr, lapack_complex_float *s, lapack_complex_float *c)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claic1((void *) job, (void *) j, (void *) x, (void *) sest, (void *) w, (void *) gamma, (void *) sestpr, (void *) s, (void *) c);

    helpTimeStop = flexiblas_wtime();

    data->claic1.timings[0] += (helpTimeStop - helpTime);
    data->claic1.calls[0]++;
    
    return;
}


/* Wrapper for clals0 */
void hook_clals0(blasint *icompq, blasint *nl, blasint *nr, blasint *sqre, blasint *nrhs, lapack_complex_float *b, blasint *ldb, lapack_complex_float *bx, blasint *ldbx, blasint *perm, blasint *givptr, blasint *givcol, blasint *ldgcol, float *givnum, blasint *ldgnum, float *poles, float *difl, float *difr, float *z, blasint *k, float *c, float *s, float *rwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clals0((void *) icompq, (void *) nl, (void *) nr, (void *) sqre, (void *) nrhs, (void *) b, (void *) ldb, (void *) bx, (void *) ldbx, (void *) perm, (void *) givptr, (void *) givcol, (void *) ldgcol, (void *) givnum, (void *) ldgnum, (void *) poles, (void *) difl, (void *) difr, (void *) z, (void *) k, (void *) c, (void *) s, (void *) rwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->clals0.timings[0] += (helpTimeStop - helpTime);
    data->clals0.calls[0]++;
    
    return;
}


/* Wrapper for clalsa */
void hook_clalsa(blasint *icompq, blasint *smlsiz, blasint *n, blasint *nrhs, lapack_complex_float *b, blasint *ldb, lapack_complex_float *bx, blasint *ldbx, float *u, blasint *ldu, float *vt, blasint *k, float *difl, float *difr, float *z, float *poles, blasint *givptr, blasint *givcol, blasint *ldgcol, blasint *perm, float *givnum, float *c, float *s, float *rwork, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clalsa((void *) icompq, (void *) smlsiz, (void *) n, (void *) nrhs, (void *) b, (void *) ldb, (void *) bx, (void *) ldbx, (void *) u, (void *) ldu, (void *) vt, (void *) k, (void *) difl, (void *) difr, (void *) z, (void *) poles, (void *) givptr, (void *) givcol, (void *) ldgcol, (void *) perm, (void *) givnum, (void *) c, (void *) s, (void *) rwork, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->clalsa.timings[0] += (helpTimeStop - helpTime);
    data->clalsa.calls[0]++;
    
    return;
}


/* Wrapper for clalsd */
void hook_clalsd(char *uplo, blasint *smlsiz, blasint *n, blasint *nrhs, float *d, float *e, lapack_complex_float *b, blasint *ldb, float *rcond, blasint *rank, lapack_complex_float *work, float *rwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clalsd((void *) uplo, (void *) smlsiz, (void *) n, (void *) nrhs, (void *) d, (void *) e, (void *) b, (void *) ldb, (void *) rcond, (void *) rank, (void *) work, (void *) rwork, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clalsd.timings[0] += (helpTimeStop - helpTime);
    data->clalsd.calls[0]++;
    
    return;
}


/* Wrapper for clamswlq */
void hook_clamswlq(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *mb, blasint *nb, lapack_complex_float *a, blasint *lda, lapack_complex_float *t, blasint *ldt, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clamswlq((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) mb, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->clamswlq.timings[0] += (helpTimeStop - helpTime);
    data->clamswlq.calls[0]++;
    
    return;
}


/* Wrapper for clamtsqr */
void hook_clamtsqr(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *mb, blasint *nb, lapack_complex_float *a, blasint *lda, lapack_complex_float *t, blasint *ldt, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clamtsqr((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) mb, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->clamtsqr.timings[0] += (helpTimeStop - helpTime);
    data->clamtsqr.calls[0]++;
    
    return;
}

/* Wrapper for clangb */
float hook_clangb(char *norm, blasint *n, blasint *kl, blasint *ku, lapack_complex_float *ab, blasint *ldab, float *work, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_clangb((void *) norm, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) work, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->clangb.timings[0] += (helpTimeStop - helpTime);
    data->clangb.calls[0]++;
    
    return return_value;
}


/* Wrapper for clange */
float hook_clange(char *norm, blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, float *work, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_clange((void *) norm, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) work, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->clange.timings[0] += (helpTimeStop - helpTime);
    data->clange.calls[0]++;
    
    return return_value;
}


/* Wrapper for clangt */
float hook_clangt(char *norm, blasint *n, lapack_complex_float *dl, lapack_complex_float *d, lapack_complex_float *du, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_clangt((void *) norm, (void *) n, (void *) dl, (void *) d, (void *) du, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->clangt.timings[0] += (helpTimeStop - helpTime);
    data->clangt.calls[0]++;
    
    return return_value;
}


/* Wrapper for clanhb */
float hook_clanhb(char *norm, char *uplo, blasint *n, blasint *k, lapack_complex_float *ab, blasint *ldab, float *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_clanhb((void *) norm, (void *) uplo, (void *) n, (void *) k, (void *) ab, (void *) ldab, (void *) work, norm_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clanhb.timings[0] += (helpTimeStop - helpTime);
    data->clanhb.calls[0]++;
    
    return return_value;
}


/* Wrapper for clanhe */
float hook_clanhe(char *norm, char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, float *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_clanhe((void *) norm, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) work, norm_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clanhe.timings[0] += (helpTimeStop - helpTime);
    data->clanhe.calls[0]++;
    
    return return_value;
}


/* Wrapper for clanhf */
float hook_clanhf(char *norm, char *transr, char *uplo, blasint *n, lapack_complex_float *a, float *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_clanhf((void *) norm, (void *) transr, (void *) uplo, (void *) n, (void *) a, (void *) work, norm_len, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clanhf.timings[0] += (helpTimeStop - helpTime);
    data->clanhf.calls[0]++;
    
    return return_value;
}


/* Wrapper for clanhp */
float hook_clanhp(char *norm, char *uplo, blasint *n, lapack_complex_float *ap, float *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_clanhp((void *) norm, (void *) uplo, (void *) n, (void *) ap, (void *) work, norm_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clanhp.timings[0] += (helpTimeStop - helpTime);
    data->clanhp.calls[0]++;
    
    return return_value;
}


/* Wrapper for clanhs */
float hook_clanhs(char *norm, blasint *n, lapack_complex_float *a, blasint *lda, float *work, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_clanhs((void *) norm, (void *) n, (void *) a, (void *) lda, (void *) work, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->clanhs.timings[0] += (helpTimeStop - helpTime);
    data->clanhs.calls[0]++;
    
    return return_value;
}


/* Wrapper for clanht */
float hook_clanht(char *norm, blasint *n, float *d, lapack_complex_float *e, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_clanht((void *) norm, (void *) n, (void *) d, (void *) e, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->clanht.timings[0] += (helpTimeStop - helpTime);
    data->clanht.calls[0]++;
    
    return return_value;
}


/* Wrapper for clansb */
float hook_clansb(char *norm, char *uplo, blasint *n, blasint *k, lapack_complex_float *ab, blasint *ldab, float *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_clansb((void *) norm, (void *) uplo, (void *) n, (void *) k, (void *) ab, (void *) ldab, (void *) work, norm_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clansb.timings[0] += (helpTimeStop - helpTime);
    data->clansb.calls[0]++;
    
    return return_value;
}


/* Wrapper for clansp */
float hook_clansp(char *norm, char *uplo, blasint *n, lapack_complex_float *ap, float *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_clansp((void *) norm, (void *) uplo, (void *) n, (void *) ap, (void *) work, norm_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clansp.timings[0] += (helpTimeStop - helpTime);
    data->clansp.calls[0]++;
    
    return return_value;
}


/* Wrapper for clansy */
float hook_clansy(char *norm, char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, float *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_clansy((void *) norm, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) work, norm_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clansy.timings[0] += (helpTimeStop - helpTime);
    data->clansy.calls[0]++;
    
    return return_value;
}


/* Wrapper for clantb */
float hook_clantb(char *norm, char *uplo, char *diag, blasint *n, blasint *k, lapack_complex_float *ab, blasint *ldab, float *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_clantb((void *) norm, (void *) uplo, (void *) diag, (void *) n, (void *) k, (void *) ab, (void *) ldab, (void *) work, norm_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->clantb.timings[0] += (helpTimeStop - helpTime);
    data->clantb.calls[0]++;
    
    return return_value;
}


/* Wrapper for clantp */
float hook_clantp(char *norm, char *uplo, char *diag, blasint *n, lapack_complex_float *ap, float *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_clantp((void *) norm, (void *) uplo, (void *) diag, (void *) n, (void *) ap, (void *) work, norm_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->clantp.timings[0] += (helpTimeStop - helpTime);
    data->clantp.calls[0]++;
    
    return return_value;
}


/* Wrapper for clantr */
float hook_clantr(char *norm, char *uplo, char *diag, blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, float *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_clantr((void *) norm, (void *) uplo, (void *) diag, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) work, norm_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->clantr.timings[0] += (helpTimeStop - helpTime);
    data->clantr.calls[0]++;
    
    return return_value;
}



/* Wrapper for clapll */
void hook_clapll(blasint *n, lapack_complex_float *x, blasint *incx, lapack_complex_float *y, blasint *incy, float *ssmin)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clapll((void *) n, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ssmin);

    helpTimeStop = flexiblas_wtime();

    data->clapll.timings[0] += (helpTimeStop - helpTime);
    data->clapll.calls[0]++;
    
    return;
}


/* Wrapper for clapmr */
void hook_clapmr(blaslogical *forwrd, blasint *m, blasint *n, lapack_complex_float *x, blasint *ldx, blasint *k)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clapmr((void *) forwrd, (void *) m, (void *) n, (void *) x, (void *) ldx, (void *) k);

    helpTimeStop = flexiblas_wtime();

    data->clapmr.timings[0] += (helpTimeStop - helpTime);
    data->clapmr.calls[0]++;
    
    return;
}


/* Wrapper for clapmt */
void hook_clapmt(blaslogical *forwrd, blasint *m, blasint *n, lapack_complex_float *x, blasint *ldx, blasint *k)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clapmt((void *) forwrd, (void *) m, (void *) n, (void *) x, (void *) ldx, (void *) k);

    helpTimeStop = flexiblas_wtime();

    data->clapmt.timings[0] += (helpTimeStop - helpTime);
    data->clapmt.calls[0]++;
    
    return;
}


/* Wrapper for claqgb */
void hook_claqgb(blasint *m, blasint *n, blasint *kl, blasint *ku, lapack_complex_float *ab, blasint *ldab, float *r, float *c, float *rowcnd, float *colcnd, float *amax, char *equed, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claqgb((void *) m, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) equed, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->claqgb.timings[0] += (helpTimeStop - helpTime);
    data->claqgb.calls[0]++;
    
    return;
}


/* Wrapper for claqge */
void hook_claqge(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, float *r, float *c, float *rowcnd, float *colcnd, float *amax, char *equed, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claqge((void *) m, (void *) n, (void *) a, (void *) lda, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) equed, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->claqge.timings[0] += (helpTimeStop - helpTime);
    data->claqge.calls[0]++;
    
    return;
}


/* Wrapper for claqhb */
void hook_claqhb(char *uplo, blasint *n, blasint *kd, lapack_complex_float *ab, blasint *ldab, float *s, float *scond, float *amax, char *equed, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claqhb((void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) s, (void *) scond, (void *) amax, (void *) equed, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->claqhb.timings[0] += (helpTimeStop - helpTime);
    data->claqhb.calls[0]++;
    
    return;
}


/* Wrapper for claqhe */
void hook_claqhe(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, float *s, float *scond, float *amax, char *equed, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claqhe((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) s, (void *) scond, (void *) amax, (void *) equed, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->claqhe.timings[0] += (helpTimeStop - helpTime);
    data->claqhe.calls[0]++;
    
    return;
}


/* Wrapper for claqhp */
void hook_claqhp(char *uplo, blasint *n, lapack_complex_float *ap, float *s, float *scond, float *amax, char *equed, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claqhp((void *) uplo, (void *) n, (void *) ap, (void *) s, (void *) scond, (void *) amax, (void *) equed, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->claqhp.timings[0] += (helpTimeStop - helpTime);
    data->claqhp.calls[0]++;
    
    return;
}


/* Wrapper for claqp2 */
void hook_claqp2(blasint *m, blasint *n, blasint *offset, lapack_complex_float *a, blasint *lda, blasint *jpvt, lapack_complex_float *tau, float *vn1, float *vn2, lapack_complex_float *work)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claqp2((void *) m, (void *) n, (void *) offset, (void *) a, (void *) lda, (void *) jpvt, (void *) tau, (void *) vn1, (void *) vn2, (void *) work);

    helpTimeStop = flexiblas_wtime();

    data->claqp2.timings[0] += (helpTimeStop - helpTime);
    data->claqp2.calls[0]++;
    
    return;
}


/* Wrapper for claqps */
void hook_claqps(blasint *m, blasint *n, blasint *offset, blasint *nb, blasint *kb, lapack_complex_float *a, blasint *lda, blasint *jpvt, lapack_complex_float *tau, float *vn1, float *vn2, lapack_complex_float *auxv, lapack_complex_float *f, blasint *ldf)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claqps((void *) m, (void *) n, (void *) offset, (void *) nb, (void *) kb, (void *) a, (void *) lda, (void *) jpvt, (void *) tau, (void *) vn1, (void *) vn2, (void *) auxv, (void *) f, (void *) ldf);

    helpTimeStop = flexiblas_wtime();

    data->claqps.timings[0] += (helpTimeStop - helpTime);
    data->claqps.calls[0]++;
    
    return;
}


/* Wrapper for claqr0 */
void hook_claqr0(blaslogical *wantt, blaslogical *wantz, blasint *n, blasint *ilo, blasint *ihi, lapack_complex_float *h, blasint *ldh, lapack_complex_float *w, blasint *iloz, blasint *ihiz, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claqr0((void *) wantt, (void *) wantz, (void *) n, (void *) ilo, (void *) ihi, (void *) h, (void *) ldh, (void *) w, (void *) iloz, (void *) ihiz, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->claqr0.timings[0] += (helpTimeStop - helpTime);
    data->claqr0.calls[0]++;
    
    return;
}


/* Wrapper for claqr1 */
void hook_claqr1(blasint *n, lapack_complex_float *h, blasint *ldh, lapack_complex_float *s1, lapack_complex_float *s2, lapack_complex_float *v)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claqr1((void *) n, (void *) h, (void *) ldh, (void *) s1, (void *) s2, (void *) v);

    helpTimeStop = flexiblas_wtime();

    data->claqr1.timings[0] += (helpTimeStop - helpTime);
    data->claqr1.calls[0]++;
    
    return;
}


/* Wrapper for claqr2 */
void hook_claqr2(blaslogical *wantt, blaslogical *wantz, blasint *n, blasint *ktop, blasint *kbot, blasint *nw, lapack_complex_float *h, blasint *ldh, blasint *iloz, blasint *ihiz, lapack_complex_float *z, blasint *ldz, blasint *ns, blasint *nd, lapack_complex_float *sh, lapack_complex_float *v, blasint *ldv, blasint *nh, lapack_complex_float *t, blasint *ldt, blasint *nv, lapack_complex_float *wv, blasint *ldwv, lapack_complex_float *work, blasint *lwork)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claqr2((void *) wantt, (void *) wantz, (void *) n, (void *) ktop, (void *) kbot, (void *) nw, (void *) h, (void *) ldh, (void *) iloz, (void *) ihiz, (void *) z, (void *) ldz, (void *) ns, (void *) nd, (void *) sh, (void *) v, (void *) ldv, (void *) nh, (void *) t, (void *) ldt, (void *) nv, (void *) wv, (void *) ldwv, (void *) work, (void *) lwork);

    helpTimeStop = flexiblas_wtime();

    data->claqr2.timings[0] += (helpTimeStop - helpTime);
    data->claqr2.calls[0]++;
    
    return;
}


/* Wrapper for claqr3 */
void hook_claqr3(blaslogical *wantt, blaslogical *wantz, blasint *n, blasint *ktop, blasint *kbot, blasint *nw, lapack_complex_float *h, blasint *ldh, blasint *iloz, blasint *ihiz, lapack_complex_float *z, blasint *ldz, blasint *ns, blasint *nd, lapack_complex_float *sh, lapack_complex_float *v, blasint *ldv, blasint *nh, lapack_complex_float *t, blasint *ldt, blasint *nv, lapack_complex_float *wv, blasint *ldwv, lapack_complex_float *work, blasint *lwork)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claqr3((void *) wantt, (void *) wantz, (void *) n, (void *) ktop, (void *) kbot, (void *) nw, (void *) h, (void *) ldh, (void *) iloz, (void *) ihiz, (void *) z, (void *) ldz, (void *) ns, (void *) nd, (void *) sh, (void *) v, (void *) ldv, (void *) nh, (void *) t, (void *) ldt, (void *) nv, (void *) wv, (void *) ldwv, (void *) work, (void *) lwork);

    helpTimeStop = flexiblas_wtime();

    data->claqr3.timings[0] += (helpTimeStop - helpTime);
    data->claqr3.calls[0]++;
    
    return;
}


/* Wrapper for claqr4 */
void hook_claqr4(blaslogical *wantt, blaslogical *wantz, blasint *n, blasint *ilo, blasint *ihi, lapack_complex_float *h, blasint *ldh, lapack_complex_float *w, blasint *iloz, blasint *ihiz, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claqr4((void *) wantt, (void *) wantz, (void *) n, (void *) ilo, (void *) ihi, (void *) h, (void *) ldh, (void *) w, (void *) iloz, (void *) ihiz, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->claqr4.timings[0] += (helpTimeStop - helpTime);
    data->claqr4.calls[0]++;
    
    return;
}


/* Wrapper for claqr5 */
void hook_claqr5(blaslogical *wantt, blaslogical *wantz, blasint *kacc22, blasint *n, blasint *ktop, blasint *kbot, blasint *nshfts, lapack_complex_float *s, lapack_complex_float *h, blasint *ldh, blasint *iloz, blasint *ihiz, lapack_complex_float *z, blasint *ldz, lapack_complex_float *v, blasint *ldv, lapack_complex_float *u, blasint *ldu, blasint *nv, lapack_complex_float *wv, blasint *ldwv, blasint *nh, lapack_complex_float *wh, blasint *ldwh)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claqr5((void *) wantt, (void *) wantz, (void *) kacc22, (void *) n, (void *) ktop, (void *) kbot, (void *) nshfts, (void *) s, (void *) h, (void *) ldh, (void *) iloz, (void *) ihiz, (void *) z, (void *) ldz, (void *) v, (void *) ldv, (void *) u, (void *) ldu, (void *) nv, (void *) wv, (void *) ldwv, (void *) nh, (void *) wh, (void *) ldwh);

    helpTimeStop = flexiblas_wtime();

    data->claqr5.timings[0] += (helpTimeStop - helpTime);
    data->claqr5.calls[0]++;
    
    return;
}


/* Wrapper for claqsb */
void hook_claqsb(char *uplo, blasint *n, blasint *kd, lapack_complex_float *ab, blasint *ldab, float *s, float *scond, float *amax, char *equed, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claqsb((void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) s, (void *) scond, (void *) amax, (void *) equed, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->claqsb.timings[0] += (helpTimeStop - helpTime);
    data->claqsb.calls[0]++;
    
    return;
}


/* Wrapper for claqsp */
void hook_claqsp(char *uplo, blasint *n, lapack_complex_float *ap, float *s, float *scond, float *amax, char *equed, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claqsp((void *) uplo, (void *) n, (void *) ap, (void *) s, (void *) scond, (void *) amax, (void *) equed, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->claqsp.timings[0] += (helpTimeStop - helpTime);
    data->claqsp.calls[0]++;
    
    return;
}


/* Wrapper for claqsy */
void hook_claqsy(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, float *s, float *scond, float *amax, char *equed, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claqsy((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) s, (void *) scond, (void *) amax, (void *) equed, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->claqsy.timings[0] += (helpTimeStop - helpTime);
    data->claqsy.calls[0]++;
    
    return;
}


/* Wrapper for clar1v */
void hook_clar1v(blasint *n, blasint *b1, blasint *bn, float *lambda, float *d, float *l, float *ld, float *lld, float *pivmin, float *gaptol, lapack_complex_float *z, blaslogical *wantnc, blasint *negcnt, float *ztz, float *mingma, blasint *r, blasint *isuppz, float *nrminv, float *resid, float *rqcorr, float *work)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clar1v((void *) n, (void *) b1, (void *) bn, (void *) lambda, (void *) d, (void *) l, (void *) ld, (void *) lld, (void *) pivmin, (void *) gaptol, (void *) z, (void *) wantnc, (void *) negcnt, (void *) ztz, (void *) mingma, (void *) r, (void *) isuppz, (void *) nrminv, (void *) resid, (void *) rqcorr, (void *) work);

    helpTimeStop = flexiblas_wtime();

    data->clar1v.timings[0] += (helpTimeStop - helpTime);
    data->clar1v.calls[0]++;
    
    return;
}


/* Wrapper for clar2v */
void hook_clar2v(blasint *n, lapack_complex_float *x, lapack_complex_float *y, lapack_complex_float *z, blasint *incx, float *c, lapack_complex_float *s, blasint *incc)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clar2v((void *) n, (void *) x, (void *) y, (void *) z, (void *) incx, (void *) c, (void *) s, (void *) incc);

    helpTimeStop = flexiblas_wtime();

    data->clar2v.timings[0] += (helpTimeStop - helpTime);
    data->clar2v.calls[0]++;
    
    return;
}


/* Wrapper for clarcm */
void hook_clarcm(blasint *m, blasint *n, float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *c, blasint *ldc, float *rwork)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clarcm((void *) m, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) c, (void *) ldc, (void *) rwork);

    helpTimeStop = flexiblas_wtime();

    data->clarcm.timings[0] += (helpTimeStop - helpTime);
    data->clarcm.calls[0]++;
    
    return;
}


/* Wrapper for clarf */
void hook_clarf(char *side, blasint *m, blasint *n, lapack_complex_float *v, blasint *incv, lapack_complex_float *tau, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, flexiblas_fortran_charlen_t side_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clarf((void *) side, (void *) m, (void *) n, (void *) v, (void *) incv, (void *) tau, (void *) c, (void *) ldc, (void *) work, side_len);

    helpTimeStop = flexiblas_wtime();

    data->clarf.timings[0] += (helpTimeStop - helpTime);
    data->clarf.calls[0]++;
    
    return;
}


/* Wrapper for clarfb */
void hook_clarfb(char *side, char *trans, char *direct, char *storev, blasint *m, blasint *n, blasint *k, lapack_complex_float *v, blasint *ldv, lapack_complex_float *t, blasint *ldt, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *ldwork, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t direct_len, flexiblas_fortran_charlen_t storev_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clarfb((void *) side, (void *) trans, (void *) direct, (void *) storev, (void *) m, (void *) n, (void *) k, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) c, (void *) ldc, (void *) work, (void *) ldwork, side_len, trans_len, direct_len, storev_len);

    helpTimeStop = flexiblas_wtime();

    data->clarfb.timings[0] += (helpTimeStop - helpTime);
    data->clarfb.calls[0]++;
    
    return;
}


/* Wrapper for clarfb_gett */
void hook_clarfb_gett(char *ident, blasint *m, blasint *n, blasint *k, lapack_complex_float *t, blasint *ldt, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *work, blasint *ldwork, flexiblas_fortran_charlen_t ident_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clarfb_gett((void *) ident, (void *) m, (void *) n, (void *) k, (void *) t, (void *) ldt, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) work, (void *) ldwork, ident_len);

    helpTimeStop = flexiblas_wtime();

    data->clarfb_gett.timings[0] += (helpTimeStop - helpTime);
    data->clarfb_gett.calls[0]++;
    
    return;
}


/* Wrapper for clarfg */
void hook_clarfg(blasint *n, lapack_complex_float *alpha, lapack_complex_float *x, blasint *incx, lapack_complex_float *tau)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clarfg((void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) tau);

    helpTimeStop = flexiblas_wtime();

    data->clarfg.timings[0] += (helpTimeStop - helpTime);
    data->clarfg.calls[0]++;
    
    return;
}


/* Wrapper for clarfgp */
void hook_clarfgp(blasint *n, lapack_complex_float *alpha, lapack_complex_float *x, blasint *incx, lapack_complex_float *tau)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clarfgp((void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) tau);

    helpTimeStop = flexiblas_wtime();

    data->clarfgp.timings[0] += (helpTimeStop - helpTime);
    data->clarfgp.calls[0]++;
    
    return;
}


/* Wrapper for clarft */
void hook_clarft(char *direct, char *storev, blasint *n, blasint *k, lapack_complex_float *v, blasint *ldv, lapack_complex_float *tau, lapack_complex_float *t, blasint *ldt, flexiblas_fortran_charlen_t direct_len, flexiblas_fortran_charlen_t storev_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clarft((void *) direct, (void *) storev, (void *) n, (void *) k, (void *) v, (void *) ldv, (void *) tau, (void *) t, (void *) ldt, direct_len, storev_len);

    helpTimeStop = flexiblas_wtime();

    data->clarft.timings[0] += (helpTimeStop - helpTime);
    data->clarft.calls[0]++;
    
    return;
}


/* Wrapper for clarfx */
void hook_clarfx(char *side, blasint *m, blasint *n, lapack_complex_float *v, lapack_complex_float *tau, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, flexiblas_fortran_charlen_t side_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clarfx((void *) side, (void *) m, (void *) n, (void *) v, (void *) tau, (void *) c, (void *) ldc, (void *) work, side_len);

    helpTimeStop = flexiblas_wtime();

    data->clarfx.timings[0] += (helpTimeStop - helpTime);
    data->clarfx.calls[0]++;
    
    return;
}


/* Wrapper for clarfy */
void hook_clarfy(char *uplo, blasint *n, lapack_complex_float *v, blasint *incv, lapack_complex_float *tau, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clarfy((void *) uplo, (void *) n, (void *) v, (void *) incv, (void *) tau, (void *) c, (void *) ldc, (void *) work, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clarfy.timings[0] += (helpTimeStop - helpTime);
    data->clarfy.calls[0]++;
    
    return;
}


/* Wrapper for clargv */
void hook_clargv(blasint *n, lapack_complex_float *x, blasint *incx, lapack_complex_float *y, blasint *incy, float *c, blasint *incc)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clargv((void *) n, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) c, (void *) incc);

    helpTimeStop = flexiblas_wtime();

    data->clargv.timings[0] += (helpTimeStop - helpTime);
    data->clargv.calls[0]++;
    
    return;
}


/* Wrapper for clarnv */
void hook_clarnv(blasint *idist, blasint *iseed, blasint *n, lapack_complex_float *x)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clarnv((void *) idist, (void *) iseed, (void *) n, (void *) x);

    helpTimeStop = flexiblas_wtime();

    data->clarnv.timings[0] += (helpTimeStop - helpTime);
    data->clarnv.calls[0]++;
    
    return;
}


/* Wrapper for clarrv */
void hook_clarrv(blasint *n, float *vl, float *vu, float *d, float *l, float *pivmin, blasint *isplit, blasint *m, blasint *dol, blasint *dou, float *minrgp, float *rtol1, float *rtol2, float *w, float *werr, float *wgap, blasint *iblock, blasint *indexw, float *gers, lapack_complex_float *z, blasint *ldz, blasint *isuppz, float *work, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clarrv((void *) n, (void *) vl, (void *) vu, (void *) d, (void *) l, (void *) pivmin, (void *) isplit, (void *) m, (void *) dol, (void *) dou, (void *) minrgp, (void *) rtol1, (void *) rtol2, (void *) w, (void *) werr, (void *) wgap, (void *) iblock, (void *) indexw, (void *) gers, (void *) z, (void *) ldz, (void *) isuppz, (void *) work, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->clarrv.timings[0] += (helpTimeStop - helpTime);
    data->clarrv.calls[0]++;
    
    return;
}


/* Wrapper for clartg */
void hook_clartg(lapack_complex_float *f, lapack_complex_float *g, float *cs, lapack_complex_float *sn, lapack_complex_float *r)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clartg((void *) f, (void *) g, (void *) cs, (void *) sn, (void *) r);

    helpTimeStop = flexiblas_wtime();

    data->clartg.timings[0] += (helpTimeStop - helpTime);
    data->clartg.calls[0]++;
    
    return;
}


/* Wrapper for clartv */
void hook_clartv(blasint *n, lapack_complex_float *x, blasint *incx, lapack_complex_float *y, blasint *incy, float *c, lapack_complex_float *s, blasint *incc)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clartv((void *) n, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) c, (void *) s, (void *) incc);

    helpTimeStop = flexiblas_wtime();

    data->clartv.timings[0] += (helpTimeStop - helpTime);
    data->clartv.calls[0]++;
    
    return;
}


/* Wrapper for clarz */
void hook_clarz(char *side, blasint *m, blasint *n, blasint *l, lapack_complex_float *v, blasint *incv, lapack_complex_float *tau, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, flexiblas_fortran_charlen_t side_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clarz((void *) side, (void *) m, (void *) n, (void *) l, (void *) v, (void *) incv, (void *) tau, (void *) c, (void *) ldc, (void *) work, side_len);

    helpTimeStop = flexiblas_wtime();

    data->clarz.timings[0] += (helpTimeStop - helpTime);
    data->clarz.calls[0]++;
    
    return;
}


/* Wrapper for clarzb */
void hook_clarzb(char *side, char *trans, char *direct, char *storev, blasint *m, blasint *n, blasint *k, blasint *l, lapack_complex_float *v, blasint *ldv, lapack_complex_float *t, blasint *ldt, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *ldwork, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t direct_len, flexiblas_fortran_charlen_t storev_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clarzb((void *) side, (void *) trans, (void *) direct, (void *) storev, (void *) m, (void *) n, (void *) k, (void *) l, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) c, (void *) ldc, (void *) work, (void *) ldwork, side_len, trans_len, direct_len, storev_len);

    helpTimeStop = flexiblas_wtime();

    data->clarzb.timings[0] += (helpTimeStop - helpTime);
    data->clarzb.calls[0]++;
    
    return;
}


/* Wrapper for clarzt */
void hook_clarzt(char *direct, char *storev, blasint *n, blasint *k, lapack_complex_float *v, blasint *ldv, lapack_complex_float *tau, lapack_complex_float *t, blasint *ldt, flexiblas_fortran_charlen_t direct_len, flexiblas_fortran_charlen_t storev_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clarzt((void *) direct, (void *) storev, (void *) n, (void *) k, (void *) v, (void *) ldv, (void *) tau, (void *) t, (void *) ldt, direct_len, storev_len);

    helpTimeStop = flexiblas_wtime();

    data->clarzt.timings[0] += (helpTimeStop - helpTime);
    data->clarzt.calls[0]++;
    
    return;
}


/* Wrapper for clascl */
void hook_clascl(char *type, blasint *kl, blasint *ku, float *cfrom, float *cto, blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t type_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clascl((void *) type, (void *) kl, (void *) ku, (void *) cfrom, (void *) cto, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) info, type_len);

    helpTimeStop = flexiblas_wtime();

    data->clascl.timings[0] += (helpTimeStop - helpTime);
    data->clascl.calls[0]++;
    
    return;
}


/* Wrapper for claset */
void hook_claset(char *uplo, blasint *m, blasint *n, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claset((void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) beta, (void *) a, (void *) lda, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->claset.timings[0] += (helpTimeStop - helpTime);
    data->claset.calls[0]++;
    
    return;
}


/* Wrapper for clasr */
void hook_clasr(char *side, char *pivot, char *direct, blasint *m, blasint *n, float *c, float *s, lapack_complex_float *a, blasint *lda, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t pivot_len, flexiblas_fortran_charlen_t direct_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clasr((void *) side, (void *) pivot, (void *) direct, (void *) m, (void *) n, (void *) c, (void *) s, (void *) a, (void *) lda, side_len, pivot_len, direct_len);

    helpTimeStop = flexiblas_wtime();

    data->clasr.timings[0] += (helpTimeStop - helpTime);
    data->clasr.calls[0]++;
    
    return;
}


/* Wrapper for classq */
void hook_classq(blasint *n, lapack_complex_float *x, blasint *incx, float *scale, float *sumsq)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_classq((void *) n, (void *) x, (void *) incx, (void *) scale, (void *) sumsq);

    helpTimeStop = flexiblas_wtime();

    data->classq.timings[0] += (helpTimeStop - helpTime);
    data->classq.calls[0]++;
    
    return;
}


/* Wrapper for claswlq */
void hook_claswlq(blasint *m, blasint *n, blasint *mb, blasint *nb, lapack_complex_float *a, blasint *lda, lapack_complex_float *t, blasint *ldt, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claswlq((void *) m, (void *) n, (void *) mb, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->claswlq.timings[0] += (helpTimeStop - helpTime);
    data->claswlq.calls[0]++;
    
    return;
}


/* Wrapper for claswp */
void hook_claswp(blasint *n, lapack_complex_float *a, blasint *lda, blasint *k1, blasint *k2, blasint *ipiv, blasint *incx)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claswp((void *) n, (void *) a, (void *) lda, (void *) k1, (void *) k2, (void *) ipiv, (void *) incx);

    helpTimeStop = flexiblas_wtime();

    data->claswp.timings[0] += (helpTimeStop - helpTime);
    data->claswp.calls[0]++;
    
    return;
}


/* Wrapper for clasyf */
void hook_clasyf(char *uplo, blasint *n, blasint *nb, blasint *kb, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *w, blasint *ldw, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clasyf((void *) uplo, (void *) n, (void *) nb, (void *) kb, (void *) a, (void *) lda, (void *) ipiv, (void *) w, (void *) ldw, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clasyf.timings[0] += (helpTimeStop - helpTime);
    data->clasyf.calls[0]++;
    
    return;
}


/* Wrapper for clasyf_aa */
void hook_clasyf_aa(char *uplo, blasint *j1, blasint *m, blasint *nb, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *h, blasint *ldh, lapack_complex_float *work, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clasyf_aa((void *) uplo, (void *) j1, (void *) m, (void *) nb, (void *) a, (void *) lda, (void *) ipiv, (void *) h, (void *) ldh, (void *) work, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clasyf_aa.timings[0] += (helpTimeStop - helpTime);
    data->clasyf_aa.calls[0]++;
    
    return;
}


/* Wrapper for clasyf_rk */
void hook_clasyf_rk(char *uplo, blasint *n, blasint *nb, blasint *kb, lapack_complex_float *a, blasint *lda, lapack_complex_float *e, blasint *ipiv, lapack_complex_float *w, blasint *ldw, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clasyf_rk((void *) uplo, (void *) n, (void *) nb, (void *) kb, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) w, (void *) ldw, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clasyf_rk.timings[0] += (helpTimeStop - helpTime);
    data->clasyf_rk.calls[0]++;
    
    return;
}


/* Wrapper for clasyf_rook */
void hook_clasyf_rook(char *uplo, blasint *n, blasint *nb, blasint *kb, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *w, blasint *ldw, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clasyf_rook((void *) uplo, (void *) n, (void *) nb, (void *) kb, (void *) a, (void *) lda, (void *) ipiv, (void *) w, (void *) ldw, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clasyf_rook.timings[0] += (helpTimeStop - helpTime);
    data->clasyf_rook.calls[0]++;
    
    return;
}


/* Wrapper for clatbs */
void hook_clatbs(char *uplo, char *trans, char *diag, char *normin, blasint *n, blasint *kd, lapack_complex_float *ab, blasint *ldab, lapack_complex_float *x, float *scale, float *cnorm, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len, flexiblas_fortran_charlen_t normin_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clatbs((void *) uplo, (void *) trans, (void *) diag, (void *) normin, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) x, (void *) scale, (void *) cnorm, (void *) info, uplo_len, trans_len, diag_len, normin_len);

    helpTimeStop = flexiblas_wtime();

    data->clatbs.timings[0] += (helpTimeStop - helpTime);
    data->clatbs.calls[0]++;
    
    return;
}


/* Wrapper for clatdf */
void hook_clatdf(blasint *ijob, blasint *n, lapack_complex_float *z, blasint *ldz, lapack_complex_float *rhs, float *rdsum, float *rdscal, blasint *ipiv, blasint *jpiv)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clatdf((void *) ijob, (void *) n, (void *) z, (void *) ldz, (void *) rhs, (void *) rdsum, (void *) rdscal, (void *) ipiv, (void *) jpiv);

    helpTimeStop = flexiblas_wtime();

    data->clatdf.timings[0] += (helpTimeStop - helpTime);
    data->clatdf.calls[0]++;
    
    return;
}


/* Wrapper for clatps */
void hook_clatps(char *uplo, char *trans, char *diag, char *normin, blasint *n, lapack_complex_float *ap, lapack_complex_float *x, float *scale, float *cnorm, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len, flexiblas_fortran_charlen_t normin_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clatps((void *) uplo, (void *) trans, (void *) diag, (void *) normin, (void *) n, (void *) ap, (void *) x, (void *) scale, (void *) cnorm, (void *) info, uplo_len, trans_len, diag_len, normin_len);

    helpTimeStop = flexiblas_wtime();

    data->clatps.timings[0] += (helpTimeStop - helpTime);
    data->clatps.calls[0]++;
    
    return;
}


/* Wrapper for clatrd */
void hook_clatrd(char *uplo, blasint *n, blasint *nb, lapack_complex_float *a, blasint *lda, float *e, lapack_complex_float *tau, lapack_complex_float *w, blasint *ldw, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clatrd((void *) uplo, (void *) n, (void *) nb, (void *) a, (void *) lda, (void *) e, (void *) tau, (void *) w, (void *) ldw, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clatrd.timings[0] += (helpTimeStop - helpTime);
    data->clatrd.calls[0]++;
    
    return;
}


/* Wrapper for clatrs */
void hook_clatrs(char *uplo, char *trans, char *diag, char *normin, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *x, float *scale, float *cnorm, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len, flexiblas_fortran_charlen_t normin_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clatrs((void *) uplo, (void *) trans, (void *) diag, (void *) normin, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) scale, (void *) cnorm, (void *) info, uplo_len, trans_len, diag_len, normin_len);

    helpTimeStop = flexiblas_wtime();

    data->clatrs.timings[0] += (helpTimeStop - helpTime);
    data->clatrs.calls[0]++;
    
    return;
}


/* Wrapper for clatrz */
void hook_clatrz(blasint *m, blasint *n, blasint *l, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clatrz((void *) m, (void *) n, (void *) l, (void *) a, (void *) lda, (void *) tau, (void *) work);

    helpTimeStop = flexiblas_wtime();

    data->clatrz.timings[0] += (helpTimeStop - helpTime);
    data->clatrz.calls[0]++;
    
    return;
}


/* Wrapper for clatsqr */
void hook_clatsqr(blasint *m, blasint *n, blasint *mb, blasint *nb, lapack_complex_float *a, blasint *lda, lapack_complex_float *t, blasint *ldt, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clatsqr((void *) m, (void *) n, (void *) mb, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->clatsqr.timings[0] += (helpTimeStop - helpTime);
    data->clatsqr.calls[0]++;
    
    return;
}


/* Wrapper for claunhr_col_getrfnp */
void hook_claunhr_col_getrfnp(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *d, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claunhr_col_getrfnp((void *) m, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->claunhr_col_getrfnp.timings[0] += (helpTimeStop - helpTime);
    data->claunhr_col_getrfnp.calls[0]++;
    
    return;
}


/* Wrapper for claunhr_col_getrfnp2 */
void hook_claunhr_col_getrfnp2(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *d, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_claunhr_col_getrfnp2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->claunhr_col_getrfnp2.timings[0] += (helpTimeStop - helpTime);
    data->claunhr_col_getrfnp2.calls[0]++;
    
    return;
}


/* Wrapper for clauu2 */
void hook_clauu2(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clauu2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clauu2.timings[0] += (helpTimeStop - helpTime);
    data->clauu2.calls[0]++;
    
    return;
}


/* Wrapper for clauum */
void hook_clauum(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_clauum((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->clauum.timings[0] += (helpTimeStop - helpTime);
    data->clauum.calls[0]++;
    
    return;
}


/* Wrapper for cpbcon */
void hook_cpbcon(char *uplo, blasint *n, blasint *kd, lapack_complex_float *ab, blasint *ldab, float *anorm, float *rcond, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpbcon((void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) anorm, (void *) rcond, (void *) work, (void *) rwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpbcon.timings[0] += (helpTimeStop - helpTime);
    data->cpbcon.calls[0]++;
    
    return;
}


/* Wrapper for cpbequ */
void hook_cpbequ(char *uplo, blasint *n, blasint *kd, lapack_complex_float *ab, blasint *ldab, float *s, float *scond, float *amax, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpbequ((void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) s, (void *) scond, (void *) amax, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpbequ.timings[0] += (helpTimeStop - helpTime);
    data->cpbequ.calls[0]++;
    
    return;
}


/* Wrapper for cpbrfs */
void hook_cpbrfs(char *uplo, blasint *n, blasint *kd, blasint *nrhs, lapack_complex_float *ab, blasint *ldab, lapack_complex_float *afb, blasint *ldafb, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpbrfs((void *) uplo, (void *) n, (void *) kd, (void *) nrhs, (void *) ab, (void *) ldab, (void *) afb, (void *) ldafb, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpbrfs.timings[0] += (helpTimeStop - helpTime);
    data->cpbrfs.calls[0]++;
    
    return;
}


/* Wrapper for cpbstf */
void hook_cpbstf(char *uplo, blasint *n, blasint *kd, lapack_complex_float *ab, blasint *ldab, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpbstf((void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpbstf.timings[0] += (helpTimeStop - helpTime);
    data->cpbstf.calls[0]++;
    
    return;
}


/* Wrapper for cpbsv */
void hook_cpbsv(char *uplo, blasint *n, blasint *kd, blasint *nrhs, lapack_complex_float *ab, blasint *ldab, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpbsv((void *) uplo, (void *) n, (void *) kd, (void *) nrhs, (void *) ab, (void *) ldab, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpbsv.timings[0] += (helpTimeStop - helpTime);
    data->cpbsv.calls[0]++;
    
    return;
}


/* Wrapper for cpbsvx */
void hook_cpbsvx(char *fact, char *uplo, blasint *n, blasint *kd, blasint *nrhs, lapack_complex_float *ab, blasint *ldab, lapack_complex_float *afb, blasint *ldafb, char *equed, float *s, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpbsvx((void *) fact, (void *) uplo, (void *) n, (void *) kd, (void *) nrhs, (void *) ab, (void *) ldab, (void *) afb, (void *) ldafb, (void *) equed, (void *) s, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, fact_len, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->cpbsvx.timings[0] += (helpTimeStop - helpTime);
    data->cpbsvx.calls[0]++;
    
    return;
}


/* Wrapper for cpbtf2 */
void hook_cpbtf2(char *uplo, blasint *n, blasint *kd, lapack_complex_float *ab, blasint *ldab, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpbtf2((void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpbtf2.timings[0] += (helpTimeStop - helpTime);
    data->cpbtf2.calls[0]++;
    
    return;
}


/* Wrapper for cpbtrf */
void hook_cpbtrf(char *uplo, blasint *n, blasint *kd, lapack_complex_float *ab, blasint *ldab, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpbtrf((void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpbtrf.timings[0] += (helpTimeStop - helpTime);
    data->cpbtrf.calls[0]++;
    
    return;
}


/* Wrapper for cpbtrs */
void hook_cpbtrs(char *uplo, blasint *n, blasint *kd, blasint *nrhs, lapack_complex_float *ab, blasint *ldab, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpbtrs((void *) uplo, (void *) n, (void *) kd, (void *) nrhs, (void *) ab, (void *) ldab, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpbtrs.timings[0] += (helpTimeStop - helpTime);
    data->cpbtrs.calls[0]++;
    
    return;
}


/* Wrapper for cpftrf */
void hook_cpftrf(char *transr, char *uplo, blasint *n, lapack_complex_float *a, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpftrf((void *) transr, (void *) uplo, (void *) n, (void *) a, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpftrf.timings[0] += (helpTimeStop - helpTime);
    data->cpftrf.calls[0]++;
    
    return;
}


/* Wrapper for cpftri */
void hook_cpftri(char *transr, char *uplo, blasint *n, lapack_complex_float *a, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpftri((void *) transr, (void *) uplo, (void *) n, (void *) a, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpftri.timings[0] += (helpTimeStop - helpTime);
    data->cpftri.calls[0]++;
    
    return;
}


/* Wrapper for cpftrs */
void hook_cpftrs(char *transr, char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpftrs((void *) transr, (void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) b, (void *) ldb, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpftrs.timings[0] += (helpTimeStop - helpTime);
    data->cpftrs.calls[0]++;
    
    return;
}


/* Wrapper for cpocon */
void hook_cpocon(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, float *anorm, float *rcond, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpocon((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) anorm, (void *) rcond, (void *) work, (void *) rwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpocon.timings[0] += (helpTimeStop - helpTime);
    data->cpocon.calls[0]++;
    
    return;
}


/* Wrapper for cpoequ */
void hook_cpoequ(blasint *n, lapack_complex_float *a, blasint *lda, float *s, float *scond, float *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpoequ((void *) n, (void *) a, (void *) lda, (void *) s, (void *) scond, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cpoequ.timings[0] += (helpTimeStop - helpTime);
    data->cpoequ.calls[0]++;
    
    return;
}


/* Wrapper for cpoequb */
void hook_cpoequb(blasint *n, lapack_complex_float *a, blasint *lda, float *s, float *scond, float *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpoequb((void *) n, (void *) a, (void *) lda, (void *) s, (void *) scond, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cpoequb.timings[0] += (helpTimeStop - helpTime);
    data->cpoequb.calls[0]++;
    
    return;
}


/* Wrapper for cporfs */
void hook_cporfs(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *af, blasint *ldaf, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cporfs((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) af, (void *) ldaf, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cporfs.timings[0] += (helpTimeStop - helpTime);
    data->cporfs.calls[0]++;
    
    return;
}


/* Wrapper for cposv */
void hook_cposv(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cposv((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cposv.timings[0] += (helpTimeStop - helpTime);
    data->cposv.calls[0]++;
    
    return;
}


/* Wrapper for cposvx */
void hook_cposvx(char *fact, char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *af, blasint *ldaf, char *equed, float *s, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cposvx((void *) fact, (void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) af, (void *) ldaf, (void *) equed, (void *) s, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, fact_len, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->cposvx.timings[0] += (helpTimeStop - helpTime);
    data->cposvx.calls[0]++;
    
    return;
}


/* Wrapper for cpotf2 */
void hook_cpotf2(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpotf2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpotf2.timings[0] += (helpTimeStop - helpTime);
    data->cpotf2.calls[0]++;
    
    return;
}


/* Wrapper for cpotrf */
void hook_cpotrf(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpotrf((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpotrf.timings[0] += (helpTimeStop - helpTime);
    data->cpotrf.calls[0]++;
    
    return;
}


/* Wrapper for cpotrf2 */
void hook_cpotrf2(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpotrf2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpotrf2.timings[0] += (helpTimeStop - helpTime);
    data->cpotrf2.calls[0]++;
    
    return;
}


/* Wrapper for cpotri */
void hook_cpotri(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpotri((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpotri.timings[0] += (helpTimeStop - helpTime);
    data->cpotri.calls[0]++;
    
    return;
}


/* Wrapper for cpotrs */
void hook_cpotrs(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpotrs((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpotrs.timings[0] += (helpTimeStop - helpTime);
    data->cpotrs.calls[0]++;
    
    return;
}


/* Wrapper for cppcon */
void hook_cppcon(char *uplo, blasint *n, lapack_complex_float *ap, float *anorm, float *rcond, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cppcon((void *) uplo, (void *) n, (void *) ap, (void *) anorm, (void *) rcond, (void *) work, (void *) rwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cppcon.timings[0] += (helpTimeStop - helpTime);
    data->cppcon.calls[0]++;
    
    return;
}


/* Wrapper for cppequ */
void hook_cppequ(char *uplo, blasint *n, lapack_complex_float *ap, float *s, float *scond, float *amax, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cppequ((void *) uplo, (void *) n, (void *) ap, (void *) s, (void *) scond, (void *) amax, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cppequ.timings[0] += (helpTimeStop - helpTime);
    data->cppequ.calls[0]++;
    
    return;
}


/* Wrapper for cpprfs */
void hook_cpprfs(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *ap, lapack_complex_float *afp, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpprfs((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) afp, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpprfs.timings[0] += (helpTimeStop - helpTime);
    data->cpprfs.calls[0]++;
    
    return;
}


/* Wrapper for cppsv */
void hook_cppsv(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *ap, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cppsv((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cppsv.timings[0] += (helpTimeStop - helpTime);
    data->cppsv.calls[0]++;
    
    return;
}


/* Wrapper for cppsvx */
void hook_cppsvx(char *fact, char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *ap, lapack_complex_float *afp, char *equed, float *s, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cppsvx((void *) fact, (void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) afp, (void *) equed, (void *) s, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, fact_len, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->cppsvx.timings[0] += (helpTimeStop - helpTime);
    data->cppsvx.calls[0]++;
    
    return;
}


/* Wrapper for cpptrf */
void hook_cpptrf(char *uplo, blasint *n, lapack_complex_float *ap, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpptrf((void *) uplo, (void *) n, (void *) ap, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpptrf.timings[0] += (helpTimeStop - helpTime);
    data->cpptrf.calls[0]++;
    
    return;
}


/* Wrapper for cpptri */
void hook_cpptri(char *uplo, blasint *n, lapack_complex_float *ap, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpptri((void *) uplo, (void *) n, (void *) ap, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpptri.timings[0] += (helpTimeStop - helpTime);
    data->cpptri.calls[0]++;
    
    return;
}


/* Wrapper for cpptrs */
void hook_cpptrs(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *ap, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpptrs((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpptrs.timings[0] += (helpTimeStop - helpTime);
    data->cpptrs.calls[0]++;
    
    return;
}


/* Wrapper for cpstf2 */
void hook_cpstf2(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *piv, blasint *rank, float *tol, float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpstf2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) piv, (void *) rank, (void *) tol, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpstf2.timings[0] += (helpTimeStop - helpTime);
    data->cpstf2.calls[0]++;
    
    return;
}


/* Wrapper for cpstrf */
void hook_cpstrf(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *piv, blasint *rank, float *tol, float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpstrf((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) piv, (void *) rank, (void *) tol, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpstrf.timings[0] += (helpTimeStop - helpTime);
    data->cpstrf.calls[0]++;
    
    return;
}


/* Wrapper for cptcon */
void hook_cptcon(blasint *n, float *d, lapack_complex_float *e, float *anorm, float *rcond, float *rwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cptcon((void *) n, (void *) d, (void *) e, (void *) anorm, (void *) rcond, (void *) rwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cptcon.timings[0] += (helpTimeStop - helpTime);
    data->cptcon.calls[0]++;
    
    return;
}


/* Wrapper for cpteqr */
void hook_cpteqr(char *compz, blasint *n, float *d, float *e, lapack_complex_float *z, blasint *ldz, float *work, blasint *info, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpteqr((void *) compz, (void *) n, (void *) d, (void *) e, (void *) z, (void *) ldz, (void *) work, (void *) info, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->cpteqr.timings[0] += (helpTimeStop - helpTime);
    data->cpteqr.calls[0]++;
    
    return;
}


/* Wrapper for cptrfs */
void hook_cptrfs(char *uplo, blasint *n, blasint *nrhs, float *d, lapack_complex_float *e, float *df, lapack_complex_float *ef, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cptrfs((void *) uplo, (void *) n, (void *) nrhs, (void *) d, (void *) e, (void *) df, (void *) ef, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cptrfs.timings[0] += (helpTimeStop - helpTime);
    data->cptrfs.calls[0]++;
    
    return;
}


/* Wrapper for cptsv */
void hook_cptsv(blasint *n, blasint *nrhs, float *d, lapack_complex_float *e, lapack_complex_float *b, blasint *ldb, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cptsv((void *) n, (void *) nrhs, (void *) d, (void *) e, (void *) b, (void *) ldb, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cptsv.timings[0] += (helpTimeStop - helpTime);
    data->cptsv.calls[0]++;
    
    return;
}


/* Wrapper for cptsvx */
void hook_cptsvx(char *fact, blasint *n, blasint *nrhs, float *d, lapack_complex_float *e, float *df, lapack_complex_float *ef, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t fact_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cptsvx((void *) fact, (void *) n, (void *) nrhs, (void *) d, (void *) e, (void *) df, (void *) ef, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, fact_len);

    helpTimeStop = flexiblas_wtime();

    data->cptsvx.timings[0] += (helpTimeStop - helpTime);
    data->cptsvx.calls[0]++;
    
    return;
}


/* Wrapper for cpttrf */
void hook_cpttrf(blasint *n, float *d, lapack_complex_float *e, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpttrf((void *) n, (void *) d, (void *) e, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cpttrf.timings[0] += (helpTimeStop - helpTime);
    data->cpttrf.calls[0]++;
    
    return;
}


/* Wrapper for cpttrs */
void hook_cpttrs(char *uplo, blasint *n, blasint *nrhs, float *d, lapack_complex_float *e, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cpttrs((void *) uplo, (void *) n, (void *) nrhs, (void *) d, (void *) e, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cpttrs.timings[0] += (helpTimeStop - helpTime);
    data->cpttrs.calls[0]++;
    
    return;
}


/* Wrapper for cptts2 */
void hook_cptts2(blasint *iuplo, blasint *n, blasint *nrhs, float *d, lapack_complex_float *e, lapack_complex_float *b, blasint *ldb)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cptts2((void *) iuplo, (void *) n, (void *) nrhs, (void *) d, (void *) e, (void *) b, (void *) ldb);

    helpTimeStop = flexiblas_wtime();

    data->cptts2.timings[0] += (helpTimeStop - helpTime);
    data->cptts2.calls[0]++;
    
    return;
}


/* Wrapper for crot */
void hook_crot(blasint *n, lapack_complex_float *cx, blasint *incx, lapack_complex_float *cy, blasint *incy, float *c, lapack_complex_float *s)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_crot((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy, (void *) c, (void *) s);

    helpTimeStop = flexiblas_wtime();

    data->crot.timings[0] += (helpTimeStop - helpTime);
    data->crot.calls[0]++;
    
    return;
}


/* Wrapper for cspcon */
void hook_cspcon(char *uplo, blasint *n, lapack_complex_float *ap, blasint *ipiv, float *anorm, float *rcond, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cspcon((void *) uplo, (void *) n, (void *) ap, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cspcon.timings[0] += (helpTimeStop - helpTime);
    data->cspcon.calls[0]++;
    
    return;
}


/* Wrapper for cspmv */
void hook_cspmv(char *uplo, blasint *n, lapack_complex_float *alpha, lapack_complex_float *ap, lapack_complex_float *x, blasint *incx, lapack_complex_float *beta, lapack_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cspmv((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cspmv.timings[0] += (helpTimeStop - helpTime);
    data->cspmv.calls[0]++;
    
    return;
}


/* Wrapper for cspr */
void hook_cspr(char *uplo, blasint *n, lapack_complex_float *alpha, lapack_complex_float *x, blasint *incx, lapack_complex_float *ap, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cspr((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cspr.timings[0] += (helpTimeStop - helpTime);
    data->cspr.calls[0]++;
    
    return;
}


/* Wrapper for csprfs */
void hook_csprfs(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *ap, lapack_complex_float *afp, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csprfs((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) afp, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csprfs.timings[0] += (helpTimeStop - helpTime);
    data->csprfs.calls[0]++;
    
    return;
}


/* Wrapper for cspsv */
void hook_cspsv(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *ap, blasint *ipiv, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cspsv((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cspsv.timings[0] += (helpTimeStop - helpTime);
    data->cspsv.calls[0]++;
    
    return;
}


/* Wrapper for cspsvx */
void hook_cspsvx(char *fact, char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *ap, lapack_complex_float *afp, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cspsvx((void *) fact, (void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) afp, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, fact_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cspsvx.timings[0] += (helpTimeStop - helpTime);
    data->cspsvx.calls[0]++;
    
    return;
}


/* Wrapper for csptrf */
void hook_csptrf(char *uplo, blasint *n, lapack_complex_float *ap, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csptrf((void *) uplo, (void *) n, (void *) ap, (void *) ipiv, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csptrf.timings[0] += (helpTimeStop - helpTime);
    data->csptrf.calls[0]++;
    
    return;
}


/* Wrapper for csptri */
void hook_csptri(char *uplo, blasint *n, lapack_complex_float *ap, blasint *ipiv, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csptri((void *) uplo, (void *) n, (void *) ap, (void *) ipiv, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csptri.timings[0] += (helpTimeStop - helpTime);
    data->csptri.calls[0]++;
    
    return;
}


/* Wrapper for csptrs */
void hook_csptrs(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *ap, blasint *ipiv, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csptrs((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csptrs.timings[0] += (helpTimeStop - helpTime);
    data->csptrs.calls[0]++;
    
    return;
}


/* Wrapper for csrscl */
void hook_csrscl(blasint *n, float *sa, lapack_complex_float *sx, blasint *incx)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csrscl((void *) n, (void *) sa, (void *) sx, (void *) incx);

    helpTimeStop = flexiblas_wtime();

    data->csrscl.timings[0] += (helpTimeStop - helpTime);
    data->csrscl.calls[0]++;
    
    return;
}


/* Wrapper for cstedc */
void hook_cstedc(char *compz, blasint *n, float *d, float *e, lapack_complex_float *z, blasint *ldz, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *lrwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cstedc((void *) compz, (void *) n, (void *) d, (void *) e, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) rwork, (void *) lrwork, (void *) iwork, (void *) liwork, (void *) info, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->cstedc.timings[0] += (helpTimeStop - helpTime);
    data->cstedc.calls[0]++;
    
    return;
}


/* Wrapper for cstegr */
void hook_cstegr(char *jobz, char *range, blasint *n, float *d, float *e, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, lapack_complex_float *z, blasint *ldz, blasint *isuppz, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cstegr((void *) jobz, (void *) range, (void *) n, (void *) d, (void *) e, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) isuppz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, range_len);

    helpTimeStop = flexiblas_wtime();

    data->cstegr.timings[0] += (helpTimeStop - helpTime);
    data->cstegr.calls[0]++;
    
    return;
}


/* Wrapper for cstein */
void hook_cstein(blasint *n, float *d, float *e, blasint *m, float *w, blasint *iblock, blasint *isplit, lapack_complex_float *z, blasint *ldz, float *work, blasint *iwork, blasint *ifail, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cstein((void *) n, (void *) d, (void *) e, (void *) m, (void *) w, (void *) iblock, (void *) isplit, (void *) z, (void *) ldz, (void *) work, (void *) iwork, (void *) ifail, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cstein.timings[0] += (helpTimeStop - helpTime);
    data->cstein.calls[0]++;
    
    return;
}


/* Wrapper for cstemr */
void hook_cstemr(char *jobz, char *range, blasint *n, float *d, float *e, float *vl, float *vu, blasint *il, blasint *iu, blasint *m, float *w, lapack_complex_float *z, blasint *ldz, blasint *nzc, blasint *isuppz, blaslogical *tryrac, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cstemr((void *) jobz, (void *) range, (void *) n, (void *) d, (void *) e, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) nzc, (void *) isuppz, (void *) tryrac, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, range_len);

    helpTimeStop = flexiblas_wtime();

    data->cstemr.timings[0] += (helpTimeStop - helpTime);
    data->cstemr.calls[0]++;
    
    return;
}


/* Wrapper for csteqr */
void hook_csteqr(char *compz, blasint *n, float *d, float *e, lapack_complex_float *z, blasint *ldz, float *work, blasint *info, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csteqr((void *) compz, (void *) n, (void *) d, (void *) e, (void *) z, (void *) ldz, (void *) work, (void *) info, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->csteqr.timings[0] += (helpTimeStop - helpTime);
    data->csteqr.calls[0]++;
    
    return;
}


/* Wrapper for csycon */
void hook_csycon(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, float *anorm, float *rcond, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csycon((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csycon.timings[0] += (helpTimeStop - helpTime);
    data->csycon.calls[0]++;
    
    return;
}


/* Wrapper for csycon_3 */
void hook_csycon_3(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *e, blasint *ipiv, float *anorm, float *rcond, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csycon_3((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csycon_3.timings[0] += (helpTimeStop - helpTime);
    data->csycon_3.calls[0]++;
    
    return;
}


/* Wrapper for csycon_rook */
void hook_csycon_rook(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, float *anorm, float *rcond, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csycon_rook((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csycon_rook.timings[0] += (helpTimeStop - helpTime);
    data->csycon_rook.calls[0]++;
    
    return;
}


/* Wrapper for csyconv */
void hook_csyconv(char *uplo, char *way, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *e, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t way_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csyconv((void *) uplo, (void *) way, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) e, (void *) info, uplo_len, way_len);

    helpTimeStop = flexiblas_wtime();

    data->csyconv.timings[0] += (helpTimeStop - helpTime);
    data->csyconv.calls[0]++;
    
    return;
}


/* Wrapper for csyconvf */
void hook_csyconvf(char *uplo, char *way, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *e, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t way_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csyconvf((void *) uplo, (void *) way, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) info, uplo_len, way_len);

    helpTimeStop = flexiblas_wtime();

    data->csyconvf.timings[0] += (helpTimeStop - helpTime);
    data->csyconvf.calls[0]++;
    
    return;
}


/* Wrapper for csyconvf_rook */
void hook_csyconvf_rook(char *uplo, char *way, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *e, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t way_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csyconvf_rook((void *) uplo, (void *) way, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) info, uplo_len, way_len);

    helpTimeStop = flexiblas_wtime();

    data->csyconvf_rook.timings[0] += (helpTimeStop - helpTime);
    data->csyconvf_rook.calls[0]++;
    
    return;
}


/* Wrapper for csyequb */
void hook_csyequb(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, float *s, float *scond, float *amax, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csyequb((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) s, (void *) scond, (void *) amax, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csyequb.timings[0] += (helpTimeStop - helpTime);
    data->csyequb.calls[0]++;
    
    return;
}


/* Wrapper for csymv */
void hook_csymv(char *uplo, blasint *n, lapack_complex_float *alpha, lapack_complex_float *a, blasint *lda, lapack_complex_float *x, blasint *incx, lapack_complex_float *beta, lapack_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csymv((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csymv.timings[0] += (helpTimeStop - helpTime);
    data->csymv.calls[0]++;
    
    return;
}


/* Wrapper for csyr */
void hook_csyr(char *uplo, blasint *n, lapack_complex_float *alpha, lapack_complex_float *x, blasint *incx, lapack_complex_float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csyr((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csyr.timings[0] += (helpTimeStop - helpTime);
    data->csyr.calls[0]++;
    
    return;
}


/* Wrapper for csyrfs */
void hook_csyrfs(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *af, blasint *ldaf, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csyrfs((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) af, (void *) ldaf, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csyrfs.timings[0] += (helpTimeStop - helpTime);
    data->csyrfs.calls[0]++;
    
    return;
}


/* Wrapper for csysv */
void hook_csysv(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csysv((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csysv.timings[0] += (helpTimeStop - helpTime);
    data->csysv.calls[0]++;
    
    return;
}


/* Wrapper for csysv_aa */
void hook_csysv_aa(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csysv_aa((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csysv_aa.timings[0] += (helpTimeStop - helpTime);
    data->csysv_aa.calls[0]++;
    
    return;
}


/* Wrapper for csysv_aa_2stage */
void hook_csysv_aa_2stage(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *tb, blasint *ltb, blasint *ipiv, blasint *ipiv2, lapack_complex_float *b, blasint *ldb, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csysv_aa_2stage((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) tb, (void *) ltb, (void *) ipiv, (void *) ipiv2, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csysv_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->csysv_aa_2stage.calls[0]++;
    
    return;
}


/* Wrapper for csysv_rk */
void hook_csysv_rk(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *e, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csysv_rk((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csysv_rk.timings[0] += (helpTimeStop - helpTime);
    data->csysv_rk.calls[0]++;
    
    return;
}


/* Wrapper for csysv_rook */
void hook_csysv_rook(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csysv_rook((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csysv_rook.timings[0] += (helpTimeStop - helpTime);
    data->csysv_rook.calls[0]++;
    
    return;
}


/* Wrapper for csysvx */
void hook_csysvx(char *fact, char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *af, blasint *ldaf, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csysvx((void *) fact, (void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) af, (void *) ldaf, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) lwork, (void *) rwork, (void *) info, fact_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csysvx.timings[0] += (helpTimeStop - helpTime);
    data->csysvx.calls[0]++;
    
    return;
}


/* Wrapper for csyswapr */
void hook_csyswapr(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *i1, blasint *i2, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csyswapr((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) i1, (void *) i2, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csyswapr.timings[0] += (helpTimeStop - helpTime);
    data->csyswapr.calls[0]++;
    
    return;
}


/* Wrapper for csytf2 */
void hook_csytf2(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csytf2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csytf2.timings[0] += (helpTimeStop - helpTime);
    data->csytf2.calls[0]++;
    
    return;
}


/* Wrapper for csytf2_rk */
void hook_csytf2_rk(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *e, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csytf2_rk((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csytf2_rk.timings[0] += (helpTimeStop - helpTime);
    data->csytf2_rk.calls[0]++;
    
    return;
}


/* Wrapper for csytf2_rook */
void hook_csytf2_rook(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csytf2_rook((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csytf2_rook.timings[0] += (helpTimeStop - helpTime);
    data->csytf2_rook.calls[0]++;
    
    return;
}


/* Wrapper for csytrf */
void hook_csytrf(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csytrf((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csytrf.timings[0] += (helpTimeStop - helpTime);
    data->csytrf.calls[0]++;
    
    return;
}


/* Wrapper for csytrf_aa */
void hook_csytrf_aa(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csytrf_aa((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csytrf_aa.timings[0] += (helpTimeStop - helpTime);
    data->csytrf_aa.calls[0]++;
    
    return;
}


/* Wrapper for csytrf_aa_2stage */
void hook_csytrf_aa_2stage(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *tb, blasint *ltb, blasint *ipiv, blasint *ipiv2, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csytrf_aa_2stage((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) tb, (void *) ltb, (void *) ipiv, (void *) ipiv2, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csytrf_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->csytrf_aa_2stage.calls[0]++;
    
    return;
}


/* Wrapper for csytrf_rk */
void hook_csytrf_rk(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *e, blasint *ipiv, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csytrf_rk((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csytrf_rk.timings[0] += (helpTimeStop - helpTime);
    data->csytrf_rk.calls[0]++;
    
    return;
}


/* Wrapper for csytrf_rook */
void hook_csytrf_rook(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csytrf_rook((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csytrf_rook.timings[0] += (helpTimeStop - helpTime);
    data->csytrf_rook.calls[0]++;
    
    return;
}


/* Wrapper for csytri */
void hook_csytri(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csytri((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csytri.timings[0] += (helpTimeStop - helpTime);
    data->csytri.calls[0]++;
    
    return;
}


/* Wrapper for csytri2 */
void hook_csytri2(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csytri2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csytri2.timings[0] += (helpTimeStop - helpTime);
    data->csytri2.calls[0]++;
    
    return;
}


/* Wrapper for csytri2x */
void hook_csytri2x(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *work, blasint *nb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csytri2x((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) nb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csytri2x.timings[0] += (helpTimeStop - helpTime);
    data->csytri2x.calls[0]++;
    
    return;
}


/* Wrapper for csytri_3 */
void hook_csytri_3(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *e, blasint *ipiv, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csytri_3((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csytri_3.timings[0] += (helpTimeStop - helpTime);
    data->csytri_3.calls[0]++;
    
    return;
}


/* Wrapper for csytri_3x */
void hook_csytri_3x(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *e, blasint *ipiv, lapack_complex_float *work, blasint *nb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csytri_3x((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) work, (void *) nb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csytri_3x.timings[0] += (helpTimeStop - helpTime);
    data->csytri_3x.calls[0]++;
    
    return;
}


/* Wrapper for csytri_rook */
void hook_csytri_rook(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csytri_rook((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csytri_rook.timings[0] += (helpTimeStop - helpTime);
    data->csytri_rook.calls[0]++;
    
    return;
}


/* Wrapper for csytrs */
void hook_csytrs(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csytrs((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csytrs.timings[0] += (helpTimeStop - helpTime);
    data->csytrs.calls[0]++;
    
    return;
}


/* Wrapper for csytrs2 */
void hook_csytrs2(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csytrs2((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csytrs2.timings[0] += (helpTimeStop - helpTime);
    data->csytrs2.calls[0]++;
    
    return;
}


/* Wrapper for csytrs_3 */
void hook_csytrs_3(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *e, blasint *ipiv, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csytrs_3((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csytrs_3.timings[0] += (helpTimeStop - helpTime);
    data->csytrs_3.calls[0]++;
    
    return;
}


/* Wrapper for csytrs_aa */
void hook_csytrs_aa(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *b, blasint *ldb, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csytrs_aa((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csytrs_aa.timings[0] += (helpTimeStop - helpTime);
    data->csytrs_aa.calls[0]++;
    
    return;
}


/* Wrapper for csytrs_aa_2stage */
void hook_csytrs_aa_2stage(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *tb, blasint *ltb, blasint *ipiv, blasint *ipiv2, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csytrs_aa_2stage((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) tb, (void *) ltb, (void *) ipiv, (void *) ipiv2, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csytrs_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->csytrs_aa_2stage.calls[0]++;
    
    return;
}


/* Wrapper for csytrs_rook */
void hook_csytrs_rook(char *uplo, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, blasint *ipiv, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_csytrs_rook((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->csytrs_rook.timings[0] += (helpTimeStop - helpTime);
    data->csytrs_rook.calls[0]++;
    
    return;
}


/* Wrapper for ctbcon */
void hook_ctbcon(char *norm, char *uplo, char *diag, blasint *n, blasint *kd, lapack_complex_float *ab, blasint *ldab, float *rcond, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctbcon((void *) norm, (void *) uplo, (void *) diag, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) rcond, (void *) work, (void *) rwork, (void *) info, norm_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->ctbcon.timings[0] += (helpTimeStop - helpTime);
    data->ctbcon.calls[0]++;
    
    return;
}


/* Wrapper for ctbrfs */
void hook_ctbrfs(char *uplo, char *trans, char *diag, blasint *n, blasint *kd, blasint *nrhs, lapack_complex_float *ab, blasint *ldab, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctbrfs((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) kd, (void *) nrhs, (void *) ab, (void *) ldab, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->ctbrfs.timings[0] += (helpTimeStop - helpTime);
    data->ctbrfs.calls[0]++;
    
    return;
}


/* Wrapper for ctbtrs */
void hook_ctbtrs(char *uplo, char *trans, char *diag, blasint *n, blasint *kd, blasint *nrhs, lapack_complex_float *ab, blasint *ldab, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctbtrs((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) kd, (void *) nrhs, (void *) ab, (void *) ldab, (void *) b, (void *) ldb, (void *) info, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->ctbtrs.timings[0] += (helpTimeStop - helpTime);
    data->ctbtrs.calls[0]++;
    
    return;
}


/* Wrapper for ctfsm */
void hook_ctfsm(char *transr, char *side, char *uplo, char *trans, char *diag, blasint *m, blasint *n, lapack_complex_float *alpha, lapack_complex_float *a, lapack_complex_float *b, blasint *ldb, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctfsm((void *) transr, (void *) side, (void *) uplo, (void *) trans, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) b, (void *) ldb, transr_len, side_len, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->ctfsm.timings[0] += (helpTimeStop - helpTime);
    data->ctfsm.calls[0]++;
    
    return;
}


/* Wrapper for ctftri */
void hook_ctftri(char *transr, char *uplo, char *diag, blasint *n, lapack_complex_float *a, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctftri((void *) transr, (void *) uplo, (void *) diag, (void *) n, (void *) a, (void *) info, transr_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->ctftri.timings[0] += (helpTimeStop - helpTime);
    data->ctftri.calls[0]++;
    
    return;
}


/* Wrapper for ctfttp */
void hook_ctfttp(char *transr, char *uplo, blasint *n, lapack_complex_float *arf, lapack_complex_float *ap, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctfttp((void *) transr, (void *) uplo, (void *) n, (void *) arf, (void *) ap, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ctfttp.timings[0] += (helpTimeStop - helpTime);
    data->ctfttp.calls[0]++;
    
    return;
}


/* Wrapper for ctfttr */
void hook_ctfttr(char *transr, char *uplo, blasint *n, lapack_complex_float *arf, lapack_complex_float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctfttr((void *) transr, (void *) uplo, (void *) n, (void *) arf, (void *) a, (void *) lda, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ctfttr.timings[0] += (helpTimeStop - helpTime);
    data->ctfttr.calls[0]++;
    
    return;
}


/* Wrapper for ctgevc */
void hook_ctgevc(char *side, char *howmny, blaslogical *select, blasint *n, lapack_complex_float *s, blasint *lds, lapack_complex_float *p, blasint *ldp, lapack_complex_float *vl, blasint *ldvl, lapack_complex_float *vr, blasint *ldvr, blasint *mm, blasint *m, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t howmny_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctgevc((void *) side, (void *) howmny, (void *) select, (void *) n, (void *) s, (void *) lds, (void *) p, (void *) ldp, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) mm, (void *) m, (void *) work, (void *) rwork, (void *) info, side_len, howmny_len);

    helpTimeStop = flexiblas_wtime();

    data->ctgevc.timings[0] += (helpTimeStop - helpTime);
    data->ctgevc.calls[0]++;
    
    return;
}


/* Wrapper for ctgex2 */
void hook_ctgex2(blaslogical *wantq, blaslogical *wantz, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *q, blasint *ldq, lapack_complex_float *z, blasint *ldz, blasint *j1, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctgex2((void *) wantq, (void *) wantz, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) q, (void *) ldq, (void *) z, (void *) ldz, (void *) j1, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->ctgex2.timings[0] += (helpTimeStop - helpTime);
    data->ctgex2.calls[0]++;
    
    return;
}


/* Wrapper for ctgexc */
void hook_ctgexc(blaslogical *wantq, blaslogical *wantz, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *q, blasint *ldq, lapack_complex_float *z, blasint *ldz, blasint *ifst, blasint *ilst, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctgexc((void *) wantq, (void *) wantz, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) q, (void *) ldq, (void *) z, (void *) ldz, (void *) ifst, (void *) ilst, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->ctgexc.timings[0] += (helpTimeStop - helpTime);
    data->ctgexc.calls[0]++;
    
    return;
}


/* Wrapper for ctgsen */
void hook_ctgsen(blasint *ijob, blaslogical *wantq, blaslogical *wantz, blaslogical *select, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *q, blasint *ldq, lapack_complex_float *z, blasint *ldz, blasint *m, float *pl, float *pr, float *dif, lapack_complex_float *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctgsen((void *) ijob, (void *) wantq, (void *) wantz, (void *) select, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) alpha, (void *) beta, (void *) q, (void *) ldq, (void *) z, (void *) ldz, (void *) m, (void *) pl, (void *) pr, (void *) dif, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->ctgsen.timings[0] += (helpTimeStop - helpTime);
    data->ctgsen.calls[0]++;
    
    return;
}


/* Wrapper for ctgsja */
void hook_ctgsja(char *jobu, char *jobv, char *jobq, blasint *m, blasint *p, blasint *n, blasint *k, blasint *l, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, float *tola, float *tolb, float *alpha, float *beta, lapack_complex_float *u, blasint *ldu, lapack_complex_float *v, blasint *ldv, lapack_complex_float *q, blasint *ldq, lapack_complex_float *work, blasint *ncycle, blasint *info, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobv_len, flexiblas_fortran_charlen_t jobq_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctgsja((void *) jobu, (void *) jobv, (void *) jobq, (void *) m, (void *) p, (void *) n, (void *) k, (void *) l, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) tola, (void *) tolb, (void *) alpha, (void *) beta, (void *) u, (void *) ldu, (void *) v, (void *) ldv, (void *) q, (void *) ldq, (void *) work, (void *) ncycle, (void *) info, jobu_len, jobv_len, jobq_len);

    helpTimeStop = flexiblas_wtime();

    data->ctgsja.timings[0] += (helpTimeStop - helpTime);
    data->ctgsja.calls[0]++;
    
    return;
}


/* Wrapper for ctgsna */
void hook_ctgsna(char *job, char *howmny, blaslogical *select, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *vl, blasint *ldvl, lapack_complex_float *vr, blasint *ldvr, float *s, float *dif, blasint *mm, blasint *m, lapack_complex_float *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t howmny_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctgsna((void *) job, (void *) howmny, (void *) select, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) s, (void *) dif, (void *) mm, (void *) m, (void *) work, (void *) lwork, (void *) iwork, (void *) info, job_len, howmny_len);

    helpTimeStop = flexiblas_wtime();

    data->ctgsna.timings[0] += (helpTimeStop - helpTime);
    data->ctgsna.calls[0]++;
    
    return;
}


/* Wrapper for ctgsy2 */
void hook_ctgsy2(char *trans, blasint *ijob, blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *c, blasint *ldc, lapack_complex_float *d, blasint *ldd, lapack_complex_float *e, blasint *lde, lapack_complex_float *f, blasint *ldf, float *scale, float *rdsum, float *rdscal, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctgsy2((void *) trans, (void *) ijob, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) c, (void *) ldc, (void *) d, (void *) ldd, (void *) e, (void *) lde, (void *) f, (void *) ldf, (void *) scale, (void *) rdsum, (void *) rdscal, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->ctgsy2.timings[0] += (helpTimeStop - helpTime);
    data->ctgsy2.calls[0]++;
    
    return;
}


/* Wrapper for ctgsyl */
void hook_ctgsyl(char *trans, blasint *ijob, blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *c, blasint *ldc, lapack_complex_float *d, blasint *ldd, lapack_complex_float *e, blasint *lde, lapack_complex_float *f, blasint *ldf, float *scale, float *dif, lapack_complex_float *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctgsyl((void *) trans, (void *) ijob, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) c, (void *) ldc, (void *) d, (void *) ldd, (void *) e, (void *) lde, (void *) f, (void *) ldf, (void *) scale, (void *) dif, (void *) work, (void *) lwork, (void *) iwork, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->ctgsyl.timings[0] += (helpTimeStop - helpTime);
    data->ctgsyl.calls[0]++;
    
    return;
}


/* Wrapper for ctpcon */
void hook_ctpcon(char *norm, char *uplo, char *diag, blasint *n, lapack_complex_float *ap, float *rcond, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctpcon((void *) norm, (void *) uplo, (void *) diag, (void *) n, (void *) ap, (void *) rcond, (void *) work, (void *) rwork, (void *) info, norm_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->ctpcon.timings[0] += (helpTimeStop - helpTime);
    data->ctpcon.calls[0]++;
    
    return;
}


/* Wrapper for ctplqt */
void hook_ctplqt(blasint *m, blasint *n, blasint *l, blasint *mb, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *t, blasint *ldt, lapack_complex_float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctplqt((void *) m, (void *) n, (void *) l, (void *) mb, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) t, (void *) ldt, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->ctplqt.timings[0] += (helpTimeStop - helpTime);
    data->ctplqt.calls[0]++;
    
    return;
}


/* Wrapper for ctplqt2 */
void hook_ctplqt2(blasint *m, blasint *n, blasint *l, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *t, blasint *ldt, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctplqt2((void *) m, (void *) n, (void *) l, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) t, (void *) ldt, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->ctplqt2.timings[0] += (helpTimeStop - helpTime);
    data->ctplqt2.calls[0]++;
    
    return;
}


/* Wrapper for ctpmlqt */
void hook_ctpmlqt(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *l, blasint *mb, lapack_complex_float *v, blasint *ldv, lapack_complex_float *t, blasint *ldt, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctpmlqt((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) l, (void *) mb, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->ctpmlqt.timings[0] += (helpTimeStop - helpTime);
    data->ctpmlqt.calls[0]++;
    
    return;
}


/* Wrapper for ctpmqrt */
void hook_ctpmqrt(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *l, blasint *nb, lapack_complex_float *v, blasint *ldv, lapack_complex_float *t, blasint *ldt, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctpmqrt((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) l, (void *) nb, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->ctpmqrt.timings[0] += (helpTimeStop - helpTime);
    data->ctpmqrt.calls[0]++;
    
    return;
}


/* Wrapper for ctpqrt */
void hook_ctpqrt(blasint *m, blasint *n, blasint *l, blasint *nb, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *t, blasint *ldt, lapack_complex_float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctpqrt((void *) m, (void *) n, (void *) l, (void *) nb, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) t, (void *) ldt, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->ctpqrt.timings[0] += (helpTimeStop - helpTime);
    data->ctpqrt.calls[0]++;
    
    return;
}


/* Wrapper for ctpqrt2 */
void hook_ctpqrt2(blasint *m, blasint *n, blasint *l, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *t, blasint *ldt, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctpqrt2((void *) m, (void *) n, (void *) l, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) t, (void *) ldt, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->ctpqrt2.timings[0] += (helpTimeStop - helpTime);
    data->ctpqrt2.calls[0]++;
    
    return;
}


/* Wrapper for ctprfb */
void hook_ctprfb(char *side, char *trans, char *direct, char *storev, blasint *m, blasint *n, blasint *k, blasint *l, lapack_complex_float *v, blasint *ldv, lapack_complex_float *t, blasint *ldt, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *work, blasint *ldwork, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t direct_len, flexiblas_fortran_charlen_t storev_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctprfb((void *) side, (void *) trans, (void *) direct, (void *) storev, (void *) m, (void *) n, (void *) k, (void *) l, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) work, (void *) ldwork, side_len, trans_len, direct_len, storev_len);

    helpTimeStop = flexiblas_wtime();

    data->ctprfb.timings[0] += (helpTimeStop - helpTime);
    data->ctprfb.calls[0]++;
    
    return;
}


/* Wrapper for ctprfs */
void hook_ctprfs(char *uplo, char *trans, char *diag, blasint *n, blasint *nrhs, lapack_complex_float *ap, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctprfs((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) nrhs, (void *) ap, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->ctprfs.timings[0] += (helpTimeStop - helpTime);
    data->ctprfs.calls[0]++;
    
    return;
}


/* Wrapper for ctptri */
void hook_ctptri(char *uplo, char *diag, blasint *n, lapack_complex_float *ap, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctptri((void *) uplo, (void *) diag, (void *) n, (void *) ap, (void *) info, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->ctptri.timings[0] += (helpTimeStop - helpTime);
    data->ctptri.calls[0]++;
    
    return;
}


/* Wrapper for ctptrs */
void hook_ctptrs(char *uplo, char *trans, char *diag, blasint *n, blasint *nrhs, lapack_complex_float *ap, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctptrs((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) nrhs, (void *) ap, (void *) b, (void *) ldb, (void *) info, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->ctptrs.timings[0] += (helpTimeStop - helpTime);
    data->ctptrs.calls[0]++;
    
    return;
}


/* Wrapper for ctpttf */
void hook_ctpttf(char *transr, char *uplo, blasint *n, lapack_complex_float *ap, lapack_complex_float *arf, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctpttf((void *) transr, (void *) uplo, (void *) n, (void *) ap, (void *) arf, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ctpttf.timings[0] += (helpTimeStop - helpTime);
    data->ctpttf.calls[0]++;
    
    return;
}


/* Wrapper for ctpttr */
void hook_ctpttr(char *uplo, blasint *n, lapack_complex_float *ap, lapack_complex_float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctpttr((void *) uplo, (void *) n, (void *) ap, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ctpttr.timings[0] += (helpTimeStop - helpTime);
    data->ctpttr.calls[0]++;
    
    return;
}


/* Wrapper for ctrcon */
void hook_ctrcon(char *norm, char *uplo, char *diag, blasint *n, lapack_complex_float *a, blasint *lda, float *rcond, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctrcon((void *) norm, (void *) uplo, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) rcond, (void *) work, (void *) rwork, (void *) info, norm_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->ctrcon.timings[0] += (helpTimeStop - helpTime);
    data->ctrcon.calls[0]++;
    
    return;
}


/* Wrapper for ctrevc */
void hook_ctrevc(char *side, char *howmny, blaslogical *select, blasint *n, lapack_complex_float *t, blasint *ldt, lapack_complex_float *vl, blasint *ldvl, lapack_complex_float *vr, blasint *ldvr, blasint *mm, blasint *m, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t howmny_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctrevc((void *) side, (void *) howmny, (void *) select, (void *) n, (void *) t, (void *) ldt, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) mm, (void *) m, (void *) work, (void *) rwork, (void *) info, side_len, howmny_len);

    helpTimeStop = flexiblas_wtime();

    data->ctrevc.timings[0] += (helpTimeStop - helpTime);
    data->ctrevc.calls[0]++;
    
    return;
}


/* Wrapper for ctrevc3 */
void hook_ctrevc3(char *side, char *howmny, blaslogical *select, blasint *n, lapack_complex_float *t, blasint *ldt, lapack_complex_float *vl, blasint *ldvl, lapack_complex_float *vr, blasint *ldvr, blasint *mm, blasint *m, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *lrwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t howmny_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctrevc3((void *) side, (void *) howmny, (void *) select, (void *) n, (void *) t, (void *) ldt, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) mm, (void *) m, (void *) work, (void *) lwork, (void *) rwork, (void *) lrwork, (void *) info, side_len, howmny_len);

    helpTimeStop = flexiblas_wtime();

    data->ctrevc3.timings[0] += (helpTimeStop - helpTime);
    data->ctrevc3.calls[0]++;
    
    return;
}


/* Wrapper for ctrexc */
void hook_ctrexc(char *compq, blasint *n, lapack_complex_float *t, blasint *ldt, lapack_complex_float *q, blasint *ldq, blasint *ifst, blasint *ilst, blasint *info, flexiblas_fortran_charlen_t compq_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctrexc((void *) compq, (void *) n, (void *) t, (void *) ldt, (void *) q, (void *) ldq, (void *) ifst, (void *) ilst, (void *) info, compq_len);

    helpTimeStop = flexiblas_wtime();

    data->ctrexc.timings[0] += (helpTimeStop - helpTime);
    data->ctrexc.calls[0]++;
    
    return;
}


/* Wrapper for ctrrfs */
void hook_ctrrfs(char *uplo, char *trans, char *diag, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *x, blasint *ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctrrfs((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->ctrrfs.timings[0] += (helpTimeStop - helpTime);
    data->ctrrfs.calls[0]++;
    
    return;
}


/* Wrapper for ctrsen */
void hook_ctrsen(char *job, char *compq, blaslogical *select, blasint *n, lapack_complex_float *t, blasint *ldt, lapack_complex_float *q, blasint *ldq, lapack_complex_float *w, blasint *m, float *s, float *sep, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t compq_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctrsen((void *) job, (void *) compq, (void *) select, (void *) n, (void *) t, (void *) ldt, (void *) q, (void *) ldq, (void *) w, (void *) m, (void *) s, (void *) sep, (void *) work, (void *) lwork, (void *) info, job_len, compq_len);

    helpTimeStop = flexiblas_wtime();

    data->ctrsen.timings[0] += (helpTimeStop - helpTime);
    data->ctrsen.calls[0]++;
    
    return;
}


/* Wrapper for ctrsna */
void hook_ctrsna(char *job, char *howmny, blaslogical *select, blasint *n, lapack_complex_float *t, blasint *ldt, lapack_complex_float *vl, blasint *ldvl, lapack_complex_float *vr, blasint *ldvr, float *s, float *sep, blasint *mm, blasint *m, lapack_complex_float *work, blasint *ldwork, float *rwork, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t howmny_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctrsna((void *) job, (void *) howmny, (void *) select, (void *) n, (void *) t, (void *) ldt, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) s, (void *) sep, (void *) mm, (void *) m, (void *) work, (void *) ldwork, (void *) rwork, (void *) info, job_len, howmny_len);

    helpTimeStop = flexiblas_wtime();

    data->ctrsna.timings[0] += (helpTimeStop - helpTime);
    data->ctrsna.calls[0]++;
    
    return;
}


/* Wrapper for ctrsyl */
void hook_ctrsyl(char *trana, char *tranb, blasint *isgn, blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, lapack_complex_float *c, blasint *ldc, float *scale, blasint *info, flexiblas_fortran_charlen_t trana_len, flexiblas_fortran_charlen_t tranb_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctrsyl((void *) trana, (void *) tranb, (void *) isgn, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) c, (void *) ldc, (void *) scale, (void *) info, trana_len, tranb_len);

    helpTimeStop = flexiblas_wtime();

    data->ctrsyl.timings[0] += (helpTimeStop - helpTime);
    data->ctrsyl.calls[0]++;
    
    return;
}


/* Wrapper for ctrti2 */
void hook_ctrti2(char *uplo, char *diag, blasint *n, lapack_complex_float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctrti2((void *) uplo, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->ctrti2.timings[0] += (helpTimeStop - helpTime);
    data->ctrti2.calls[0]++;
    
    return;
}


/* Wrapper for ctrtri */
void hook_ctrtri(char *uplo, char *diag, blasint *n, lapack_complex_float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctrtri((void *) uplo, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->ctrtri.timings[0] += (helpTimeStop - helpTime);
    data->ctrtri.calls[0]++;
    
    return;
}


/* Wrapper for ctrtrs */
void hook_ctrtrs(char *uplo, char *trans, char *diag, blasint *n, blasint *nrhs, lapack_complex_float *a, blasint *lda, lapack_complex_float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctrtrs((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) info, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->ctrtrs.timings[0] += (helpTimeStop - helpTime);
    data->ctrtrs.calls[0]++;
    
    return;
}


/* Wrapper for ctrttf */
void hook_ctrttf(char *transr, char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *arf, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctrttf((void *) transr, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) arf, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ctrttf.timings[0] += (helpTimeStop - helpTime);
    data->ctrttf.calls[0]++;
    
    return;
}


/* Wrapper for ctrttp */
void hook_ctrttp(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *ap, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctrttp((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ap, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ctrttp.timings[0] += (helpTimeStop - helpTime);
    data->ctrttp.calls[0]++;
    
    return;
}


/* Wrapper for ctzrzf */
void hook_ctzrzf(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ctzrzf((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->ctzrzf.timings[0] += (helpTimeStop - helpTime);
    data->ctzrzf.calls[0]++;
    
    return;
}


/* Wrapper for cunbdb */
void hook_cunbdb(char *trans, char *signs, blasint *m, blasint *p, blasint *q, lapack_complex_float *x11, blasint *ldx11, lapack_complex_float *x12, blasint *ldx12, lapack_complex_float *x21, blasint *ldx21, lapack_complex_float *x22, blasint *ldx22, float *theta, float *phi, lapack_complex_float *taup1, lapack_complex_float *taup2, lapack_complex_float *tauq1, lapack_complex_float *tauq2, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t signs_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunbdb((void *) trans, (void *) signs, (void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x12, (void *) ldx12, (void *) x21, (void *) ldx21, (void *) x22, (void *) ldx22, (void *) theta, (void *) phi, (void *) taup1, (void *) taup2, (void *) tauq1, (void *) tauq2, (void *) work, (void *) lwork, (void *) info, trans_len, signs_len);

    helpTimeStop = flexiblas_wtime();

    data->cunbdb.timings[0] += (helpTimeStop - helpTime);
    data->cunbdb.calls[0]++;
    
    return;
}


/* Wrapper for cunbdb1 */
void hook_cunbdb1(blasint *m, blasint *p, blasint *q, lapack_complex_float *x11, blasint *ldx11, lapack_complex_float *x21, blasint *ldx21, float *theta, float *phi, lapack_complex_float *taup1, lapack_complex_float *taup2, lapack_complex_float *tauq1, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunbdb1((void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x21, (void *) ldx21, (void *) theta, (void *) phi, (void *) taup1, (void *) taup2, (void *) tauq1, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cunbdb1.timings[0] += (helpTimeStop - helpTime);
    data->cunbdb1.calls[0]++;
    
    return;
}


/* Wrapper for cunbdb2 */
void hook_cunbdb2(blasint *m, blasint *p, blasint *q, lapack_complex_float *x11, blasint *ldx11, lapack_complex_float *x21, blasint *ldx21, float *theta, float *phi, lapack_complex_float *taup1, lapack_complex_float *taup2, lapack_complex_float *tauq1, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunbdb2((void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x21, (void *) ldx21, (void *) theta, (void *) phi, (void *) taup1, (void *) taup2, (void *) tauq1, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cunbdb2.timings[0] += (helpTimeStop - helpTime);
    data->cunbdb2.calls[0]++;
    
    return;
}


/* Wrapper for cunbdb3 */
void hook_cunbdb3(blasint *m, blasint *p, blasint *q, lapack_complex_float *x11, blasint *ldx11, lapack_complex_float *x21, blasint *ldx21, float *theta, float *phi, lapack_complex_float *taup1, lapack_complex_float *taup2, lapack_complex_float *tauq1, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunbdb3((void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x21, (void *) ldx21, (void *) theta, (void *) phi, (void *) taup1, (void *) taup2, (void *) tauq1, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cunbdb3.timings[0] += (helpTimeStop - helpTime);
    data->cunbdb3.calls[0]++;
    
    return;
}


/* Wrapper for cunbdb4 */
void hook_cunbdb4(blasint *m, blasint *p, blasint *q, lapack_complex_float *x11, blasint *ldx11, lapack_complex_float *x21, blasint *ldx21, float *theta, float *phi, lapack_complex_float *taup1, lapack_complex_float *taup2, lapack_complex_float *tauq1, lapack_complex_float *phantom, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunbdb4((void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x21, (void *) ldx21, (void *) theta, (void *) phi, (void *) taup1, (void *) taup2, (void *) tauq1, (void *) phantom, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cunbdb4.timings[0] += (helpTimeStop - helpTime);
    data->cunbdb4.calls[0]++;
    
    return;
}


/* Wrapper for cunbdb5 */
void hook_cunbdb5(blasint *m1, blasint *m2, blasint *n, lapack_complex_float *x1, blasint *incx1, lapack_complex_float *x2, blasint *incx2, lapack_complex_float *q1, blasint *ldq1, lapack_complex_float *q2, blasint *ldq2, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunbdb5((void *) m1, (void *) m2, (void *) n, (void *) x1, (void *) incx1, (void *) x2, (void *) incx2, (void *) q1, (void *) ldq1, (void *) q2, (void *) ldq2, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cunbdb5.timings[0] += (helpTimeStop - helpTime);
    data->cunbdb5.calls[0]++;
    
    return;
}


/* Wrapper for cunbdb6 */
void hook_cunbdb6(blasint *m1, blasint *m2, blasint *n, lapack_complex_float *x1, blasint *incx1, lapack_complex_float *x2, blasint *incx2, lapack_complex_float *q1, blasint *ldq1, lapack_complex_float *q2, blasint *ldq2, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunbdb6((void *) m1, (void *) m2, (void *) n, (void *) x1, (void *) incx1, (void *) x2, (void *) incx2, (void *) q1, (void *) ldq1, (void *) q2, (void *) ldq2, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cunbdb6.timings[0] += (helpTimeStop - helpTime);
    data->cunbdb6.calls[0]++;
    
    return;
}


/* Wrapper for cuncsd */
void hook_cuncsd(char *jobu1, char *jobu2, char *jobv1t, char *jobv2t, char *trans, char *signs, blasint *m, blasint *p, blasint *q, lapack_complex_float *x11, blasint *ldx11, lapack_complex_float *x12, blasint *ldx12, lapack_complex_float *x21, blasint *ldx21, lapack_complex_float *x22, blasint *ldx22, float *theta, lapack_complex_float *u1, blasint *ldu1, lapack_complex_float *u2, blasint *ldu2, lapack_complex_float *v1t, blasint *ldv1t, lapack_complex_float *v2t, blasint *ldv2t, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *lrwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t jobu1_len, flexiblas_fortran_charlen_t jobu2_len, flexiblas_fortran_charlen_t jobv1t_len, flexiblas_fortran_charlen_t jobv2t_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t signs_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cuncsd((void *) jobu1, (void *) jobu2, (void *) jobv1t, (void *) jobv2t, (void *) trans, (void *) signs, (void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x12, (void *) ldx12, (void *) x21, (void *) ldx21, (void *) x22, (void *) ldx22, (void *) theta, (void *) u1, (void *) ldu1, (void *) u2, (void *) ldu2, (void *) v1t, (void *) ldv1t, (void *) v2t, (void *) ldv2t, (void *) work, (void *) lwork, (void *) rwork, (void *) lrwork, (void *) iwork, (void *) info, jobu1_len, jobu2_len, jobv1t_len, jobv2t_len, trans_len, signs_len);

    helpTimeStop = flexiblas_wtime();

    data->cuncsd.timings[0] += (helpTimeStop - helpTime);
    data->cuncsd.calls[0]++;
    
    return;
}


/* Wrapper for cuncsd2by1 */
void hook_cuncsd2by1(char *jobu1, char *jobu2, char *jobv1t, blasint *m, blasint *p, blasint *q, lapack_complex_float *x11, blasint *ldx11, lapack_complex_float *x21, blasint *ldx21, float *theta, lapack_complex_float *u1, blasint *ldu1, lapack_complex_float *u2, blasint *ldu2, lapack_complex_float *v1t, blasint *ldv1t, lapack_complex_float *work, blasint *lwork, float *rwork, blasint *lrwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t jobu1_len, flexiblas_fortran_charlen_t jobu2_len, flexiblas_fortran_charlen_t jobv1t_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cuncsd2by1((void *) jobu1, (void *) jobu2, (void *) jobv1t, (void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x21, (void *) ldx21, (void *) theta, (void *) u1, (void *) ldu1, (void *) u2, (void *) ldu2, (void *) v1t, (void *) ldv1t, (void *) work, (void *) lwork, (void *) rwork, (void *) lrwork, (void *) iwork, (void *) info, jobu1_len, jobu2_len, jobv1t_len);

    helpTimeStop = flexiblas_wtime();

    data->cuncsd2by1.timings[0] += (helpTimeStop - helpTime);
    data->cuncsd2by1.calls[0]++;
    
    return;
}


/* Wrapper for cung2l */
void hook_cung2l(blasint *m, blasint *n, blasint *k, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cung2l((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cung2l.timings[0] += (helpTimeStop - helpTime);
    data->cung2l.calls[0]++;
    
    return;
}


/* Wrapper for cung2r */
void hook_cung2r(blasint *m, blasint *n, blasint *k, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cung2r((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cung2r.timings[0] += (helpTimeStop - helpTime);
    data->cung2r.calls[0]++;
    
    return;
}


/* Wrapper for cungbr */
void hook_cungbr(char *vect, blasint *m, blasint *n, blasint *k, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t vect_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cungbr((void *) vect, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info, vect_len);

    helpTimeStop = flexiblas_wtime();

    data->cungbr.timings[0] += (helpTimeStop - helpTime);
    data->cungbr.calls[0]++;
    
    return;
}


/* Wrapper for cunghr */
void hook_cunghr(blasint *n, blasint *ilo, blasint *ihi, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunghr((void *) n, (void *) ilo, (void *) ihi, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cunghr.timings[0] += (helpTimeStop - helpTime);
    data->cunghr.calls[0]++;
    
    return;
}


/* Wrapper for cungl2 */
void hook_cungl2(blasint *m, blasint *n, blasint *k, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cungl2((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cungl2.timings[0] += (helpTimeStop - helpTime);
    data->cungl2.calls[0]++;
    
    return;
}


/* Wrapper for cunglq */
void hook_cunglq(blasint *m, blasint *n, blasint *k, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunglq((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cunglq.timings[0] += (helpTimeStop - helpTime);
    data->cunglq.calls[0]++;
    
    return;
}


/* Wrapper for cungql */
void hook_cungql(blasint *m, blasint *n, blasint *k, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cungql((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cungql.timings[0] += (helpTimeStop - helpTime);
    data->cungql.calls[0]++;
    
    return;
}


/* Wrapper for cungqr */
void hook_cungqr(blasint *m, blasint *n, blasint *k, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cungqr((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cungqr.timings[0] += (helpTimeStop - helpTime);
    data->cungqr.calls[0]++;
    
    return;
}


/* Wrapper for cungr2 */
void hook_cungr2(blasint *m, blasint *n, blasint *k, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cungr2((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cungr2.timings[0] += (helpTimeStop - helpTime);
    data->cungr2.calls[0]++;
    
    return;
}


/* Wrapper for cungrq */
void hook_cungrq(blasint *m, blasint *n, blasint *k, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cungrq((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cungrq.timings[0] += (helpTimeStop - helpTime);
    data->cungrq.calls[0]++;
    
    return;
}


/* Wrapper for cungtr */
void hook_cungtr(char *uplo, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cungtr((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cungtr.timings[0] += (helpTimeStop - helpTime);
    data->cungtr.calls[0]++;
    
    return;
}


/* Wrapper for cungtsqr */
void hook_cungtsqr(blasint *m, blasint *n, blasint *mb, blasint *nb, lapack_complex_float *a, blasint *lda, lapack_complex_float *t, blasint *ldt, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cungtsqr((void *) m, (void *) n, (void *) mb, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cungtsqr.timings[0] += (helpTimeStop - helpTime);
    data->cungtsqr.calls[0]++;
    
    return;
}


/* Wrapper for cungtsqr_row */
void hook_cungtsqr_row(blasint *m, blasint *n, blasint *mb, blasint *nb, lapack_complex_float *a, blasint *lda, lapack_complex_float *t, blasint *ldt, lapack_complex_float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cungtsqr_row((void *) m, (void *) n, (void *) mb, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cungtsqr_row.timings[0] += (helpTimeStop - helpTime);
    data->cungtsqr_row.calls[0]++;
    
    return;
}


/* Wrapper for cunhr_col */
void hook_cunhr_col(blasint *m, blasint *n, blasint *nb, lapack_complex_float *a, blasint *lda, lapack_complex_float *t, blasint *ldt, lapack_complex_float *d, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunhr_col((void *) m, (void *) n, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) d, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->cunhr_col.timings[0] += (helpTimeStop - helpTime);
    data->cunhr_col.calls[0]++;
    
    return;
}


/* Wrapper for cunm22 */
void hook_cunm22(char *side, char *trans, blasint *m, blasint *n, blasint *n1, blasint *n2, lapack_complex_float *q, blasint *ldq, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunm22((void *) side, (void *) trans, (void *) m, (void *) n, (void *) n1, (void *) n2, (void *) q, (void *) ldq, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cunm22.timings[0] += (helpTimeStop - helpTime);
    data->cunm22.calls[0]++;
    
    return;
}


/* Wrapper for cunm2l */
void hook_cunm2l(char *side, char *trans, blasint *m, blasint *n, blasint *k, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunm2l((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cunm2l.timings[0] += (helpTimeStop - helpTime);
    data->cunm2l.calls[0]++;
    
    return;
}


/* Wrapper for cunm2r */
void hook_cunm2r(char *side, char *trans, blasint *m, blasint *n, blasint *k, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunm2r((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cunm2r.timings[0] += (helpTimeStop - helpTime);
    data->cunm2r.calls[0]++;
    
    return;
}


/* Wrapper for cunmbr */
void hook_cunmbr(char *vect, char *side, char *trans, blasint *m, blasint *n, blasint *k, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t vect_len, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunmbr((void *) vect, (void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, vect_len, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cunmbr.timings[0] += (helpTimeStop - helpTime);
    data->cunmbr.calls[0]++;
    
    return;
}


/* Wrapper for cunmhr */
void hook_cunmhr(char *side, char *trans, blasint *m, blasint *n, blasint *ilo, blasint *ihi, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunmhr((void *) side, (void *) trans, (void *) m, (void *) n, (void *) ilo, (void *) ihi, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cunmhr.timings[0] += (helpTimeStop - helpTime);
    data->cunmhr.calls[0]++;
    
    return;
}


/* Wrapper for cunml2 */
void hook_cunml2(char *side, char *trans, blasint *m, blasint *n, blasint *k, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunml2((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cunml2.timings[0] += (helpTimeStop - helpTime);
    data->cunml2.calls[0]++;
    
    return;
}


/* Wrapper for cunmlq */
void hook_cunmlq(char *side, char *trans, blasint *m, blasint *n, blasint *k, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunmlq((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cunmlq.timings[0] += (helpTimeStop - helpTime);
    data->cunmlq.calls[0]++;
    
    return;
}


/* Wrapper for cunmql */
void hook_cunmql(char *side, char *trans, blasint *m, blasint *n, blasint *k, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunmql((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cunmql.timings[0] += (helpTimeStop - helpTime);
    data->cunmql.calls[0]++;
    
    return;
}


/* Wrapper for cunmqr */
void hook_cunmqr(char *side, char *trans, blasint *m, blasint *n, blasint *k, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunmqr((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cunmqr.timings[0] += (helpTimeStop - helpTime);
    data->cunmqr.calls[0]++;
    
    return;
}


/* Wrapper for cunmr2 */
void hook_cunmr2(char *side, char *trans, blasint *m, blasint *n, blasint *k, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunmr2((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cunmr2.timings[0] += (helpTimeStop - helpTime);
    data->cunmr2.calls[0]++;
    
    return;
}


/* Wrapper for cunmr3 */
void hook_cunmr3(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *l, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunmr3((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) l, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cunmr3.timings[0] += (helpTimeStop - helpTime);
    data->cunmr3.calls[0]++;
    
    return;
}


/* Wrapper for cunmrq */
void hook_cunmrq(char *side, char *trans, blasint *m, blasint *n, blasint *k, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunmrq((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cunmrq.timings[0] += (helpTimeStop - helpTime);
    data->cunmrq.calls[0]++;
    
    return;
}


/* Wrapper for cunmrz */
void hook_cunmrz(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *l, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunmrz((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) l, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cunmrz.timings[0] += (helpTimeStop - helpTime);
    data->cunmrz.calls[0]++;
    
    return;
}


/* Wrapper for cunmtr */
void hook_cunmtr(char *side, char *uplo, char *trans, blasint *m, blasint *n, lapack_complex_float *a, blasint *lda, lapack_complex_float *tau, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cunmtr((void *) side, (void *) uplo, (void *) trans, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, uplo_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cunmtr.timings[0] += (helpTimeStop - helpTime);
    data->cunmtr.calls[0]++;
    
    return;
}


/* Wrapper for cupgtr */
void hook_cupgtr(char *uplo, blasint *n, lapack_complex_float *ap, lapack_complex_float *tau, lapack_complex_float *q, blasint *ldq, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cupgtr((void *) uplo, (void *) n, (void *) ap, (void *) tau, (void *) q, (void *) ldq, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->cupgtr.timings[0] += (helpTimeStop - helpTime);
    data->cupgtr.calls[0]++;
    
    return;
}


/* Wrapper for cupmtr */
void hook_cupmtr(char *side, char *uplo, char *trans, blasint *m, blasint *n, lapack_complex_float *ap, lapack_complex_float *tau, lapack_complex_float *c, blasint *ldc, lapack_complex_float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_cupmtr((void *) side, (void *) uplo, (void *) trans, (void *) m, (void *) n, (void *) ap, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, uplo_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->cupmtr.timings[0] += (helpTimeStop - helpTime);
    data->cupmtr.calls[0]++;
    
    return;
}


/* Wrapper for dbbcsd */
void hook_dbbcsd(char *jobu1, char *jobu2, char *jobv1t, char *jobv2t, char *trans, blasint *m, blasint *p, blasint *q, double *theta, double *phi, double *u1, blasint *ldu1, double *u2, blasint *ldu2, double *v1t, blasint *ldv1t, double *v2t, blasint *ldv2t, double *b11d, double *b11e, double *b12d, double *b12e, double *b21d, double *b21e, double *b22d, double *b22e, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobu1_len, flexiblas_fortran_charlen_t jobu2_len, flexiblas_fortran_charlen_t jobv1t_len, flexiblas_fortran_charlen_t jobv2t_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dbbcsd((void *) jobu1, (void *) jobu2, (void *) jobv1t, (void *) jobv2t, (void *) trans, (void *) m, (void *) p, (void *) q, (void *) theta, (void *) phi, (void *) u1, (void *) ldu1, (void *) u2, (void *) ldu2, (void *) v1t, (void *) ldv1t, (void *) v2t, (void *) ldv2t, (void *) b11d, (void *) b11e, (void *) b12d, (void *) b12e, (void *) b21d, (void *) b21e, (void *) b22d, (void *) b22e, (void *) work, (void *) lwork, (void *) info, jobu1_len, jobu2_len, jobv1t_len, jobv2t_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dbbcsd.timings[0] += (helpTimeStop - helpTime);
    data->dbbcsd.calls[0]++;
    
    return;
}


/* Wrapper for dbdsdc */
void hook_dbdsdc(char *uplo, char *compq, blasint *n, double *d, double *e, double *u, blasint *ldu, double *vt, blasint *ldvt, double *q, blasint *iq, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t compq_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dbdsdc((void *) uplo, (void *) compq, (void *) n, (void *) d, (void *) e, (void *) u, (void *) ldu, (void *) vt, (void *) ldvt, (void *) q, (void *) iq, (void *) work, (void *) iwork, (void *) info, uplo_len, compq_len);

    helpTimeStop = flexiblas_wtime();

    data->dbdsdc.timings[0] += (helpTimeStop - helpTime);
    data->dbdsdc.calls[0]++;
    
    return;
}


/* Wrapper for dbdsqr */
void hook_dbdsqr(char *uplo, blasint *n, blasint *ncvt, blasint *nru, blasint *ncc, double *d, double *e, double *vt, blasint *ldvt, double *u, blasint *ldu, double *c, blasint *ldc, double *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dbdsqr((void *) uplo, (void *) n, (void *) ncvt, (void *) nru, (void *) ncc, (void *) d, (void *) e, (void *) vt, (void *) ldvt, (void *) u, (void *) ldu, (void *) c, (void *) ldc, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dbdsqr.timings[0] += (helpTimeStop - helpTime);
    data->dbdsqr.calls[0]++;
    
    return;
}


/* Wrapper for dbdsvdx */
void hook_dbdsvdx(char *uplo, char *jobz, char *range, blasint *n, double *d, double *e, double *vl, double *vu, blasint *il, blasint *iu, blasint *ns, double *s, double *z, blasint *ldz, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dbdsvdx((void *) uplo, (void *) jobz, (void *) range, (void *) n, (void *) d, (void *) e, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) ns, (void *) s, (void *) z, (void *) ldz, (void *) work, (void *) iwork, (void *) info, uplo_len, jobz_len, range_len);

    helpTimeStop = flexiblas_wtime();

    data->dbdsvdx.timings[0] += (helpTimeStop - helpTime);
    data->dbdsvdx.calls[0]++;
    
    return;
}


/* Wrapper for dcombssq */
void hook_dcombssq(double *v1, double *v2)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dcombssq((void *) v1, (void *) v2);

    helpTimeStop = flexiblas_wtime();

    data->dcombssq.timings[0] += (helpTimeStop - helpTime);
    data->dcombssq.calls[0]++;
    
    return;
}


/* Wrapper for ddisna */
void hook_ddisna(char *job, blasint *m, blasint *n, double *d, double *sep, blasint *info, flexiblas_fortran_charlen_t job_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ddisna((void *) job, (void *) m, (void *) n, (void *) d, (void *) sep, (void *) info, job_len);

    helpTimeStop = flexiblas_wtime();

    data->ddisna.timings[0] += (helpTimeStop - helpTime);
    data->ddisna.calls[0]++;
    
    return;
}


/* Wrapper for dgbbrd */
void hook_dgbbrd(char *vect, blasint *m, blasint *n, blasint *ncc, blasint *kl, blasint *ku, double *ab, blasint *ldab, double *d, double *e, double *q, blasint *ldq, double *pt, blasint *ldpt, double *c, blasint *ldc, double *work, blasint *info, flexiblas_fortran_charlen_t vect_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgbbrd((void *) vect, (void *) m, (void *) n, (void *) ncc, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) d, (void *) e, (void *) q, (void *) ldq, (void *) pt, (void *) ldpt, (void *) c, (void *) ldc, (void *) work, (void *) info, vect_len);

    helpTimeStop = flexiblas_wtime();

    data->dgbbrd.timings[0] += (helpTimeStop - helpTime);
    data->dgbbrd.calls[0]++;
    
    return;
}


/* Wrapper for dgbcon */
void hook_dgbcon(char *norm, blasint *n, blasint *kl, blasint *ku, double *ab, blasint *ldab, blasint *ipiv, double *anorm, double *rcond, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgbcon((void *) norm, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) iwork, (void *) info, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->dgbcon.timings[0] += (helpTimeStop - helpTime);
    data->dgbcon.calls[0]++;
    
    return;
}


/* Wrapper for dgbequ */
void hook_dgbequ(blasint *m, blasint *n, blasint *kl, blasint *ku, double *ab, blasint *ldab, double *r, double *c, double *rowcnd, double *colcnd, double *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgbequ((void *) m, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgbequ.timings[0] += (helpTimeStop - helpTime);
    data->dgbequ.calls[0]++;
    
    return;
}


/* Wrapper for dgbequb */
void hook_dgbequb(blasint *m, blasint *n, blasint *kl, blasint *ku, double *ab, blasint *ldab, double *r, double *c, double *rowcnd, double *colcnd, double *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgbequb((void *) m, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgbequb.timings[0] += (helpTimeStop - helpTime);
    data->dgbequb.calls[0]++;
    
    return;
}


/* Wrapper for dgbrfs */
void hook_dgbrfs(char *trans, blasint *n, blasint *kl, blasint *ku, blasint *nrhs, double *ab, blasint *ldab, double *afb, blasint *ldafb, blasint *ipiv, double *b, blasint *ldb, double *x, blasint *ldx, double *ferr, double *berr, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgbrfs((void *) trans, (void *) n, (void *) kl, (void *) ku, (void *) nrhs, (void *) ab, (void *) ldab, (void *) afb, (void *) ldafb, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dgbrfs.timings[0] += (helpTimeStop - helpTime);
    data->dgbrfs.calls[0]++;
    
    return;
}


/* Wrapper for dgbsv */
void hook_dgbsv(blasint *n, blasint *kl, blasint *ku, blasint *nrhs, double *ab, blasint *ldab, blasint *ipiv, double *b, blasint *ldb, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgbsv((void *) n, (void *) kl, (void *) ku, (void *) nrhs, (void *) ab, (void *) ldab, (void *) ipiv, (void *) b, (void *) ldb, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgbsv.timings[0] += (helpTimeStop - helpTime);
    data->dgbsv.calls[0]++;
    
    return;
}


/* Wrapper for dgbsvx */
void hook_dgbsvx(char *fact, char *trans, blasint *n, blasint *kl, blasint *ku, blasint *nrhs, double *ab, blasint *ldab, double *afb, blasint *ldafb, blasint *ipiv, char *equed, double *r, double *c, double *b, blasint *ldb, double *x, blasint *ldx, double *rcond, double *ferr, double *berr, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgbsvx((void *) fact, (void *) trans, (void *) n, (void *) kl, (void *) ku, (void *) nrhs, (void *) ab, (void *) ldab, (void *) afb, (void *) ldafb, (void *) ipiv, (void *) equed, (void *) r, (void *) c, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, fact_len, trans_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->dgbsvx.timings[0] += (helpTimeStop - helpTime);
    data->dgbsvx.calls[0]++;
    
    return;
}


/* Wrapper for dgbtf2 */
void hook_dgbtf2(blasint *m, blasint *n, blasint *kl, blasint *ku, double *ab, blasint *ldab, blasint *ipiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgbtf2((void *) m, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) ipiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgbtf2.timings[0] += (helpTimeStop - helpTime);
    data->dgbtf2.calls[0]++;
    
    return;
}


/* Wrapper for dgbtrf */
void hook_dgbtrf(blasint *m, blasint *n, blasint *kl, blasint *ku, double *ab, blasint *ldab, blasint *ipiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgbtrf((void *) m, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) ipiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgbtrf.timings[0] += (helpTimeStop - helpTime);
    data->dgbtrf.calls[0]++;
    
    return;
}


/* Wrapper for dgbtrs */
void hook_dgbtrs(char *trans, blasint *n, blasint *kl, blasint *ku, blasint *nrhs, double *ab, blasint *ldab, blasint *ipiv, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgbtrs((void *) trans, (void *) n, (void *) kl, (void *) ku, (void *) nrhs, (void *) ab, (void *) ldab, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dgbtrs.timings[0] += (helpTimeStop - helpTime);
    data->dgbtrs.calls[0]++;
    
    return;
}


/* Wrapper for dgebak */
void hook_dgebak(char *job, char *side, blasint *n, blasint *ilo, blasint *ihi, double *scale, blasint *m, double *v, blasint *ldv, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t side_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgebak((void *) job, (void *) side, (void *) n, (void *) ilo, (void *) ihi, (void *) scale, (void *) m, (void *) v, (void *) ldv, (void *) info, job_len, side_len);

    helpTimeStop = flexiblas_wtime();

    data->dgebak.timings[0] += (helpTimeStop - helpTime);
    data->dgebak.calls[0]++;
    
    return;
}


/* Wrapper for dgebal */
void hook_dgebal(char *job, blasint *n, double *a, blasint *lda, blasint *ilo, blasint *ihi, double *scale, blasint *info, flexiblas_fortran_charlen_t job_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgebal((void *) job, (void *) n, (void *) a, (void *) lda, (void *) ilo, (void *) ihi, (void *) scale, (void *) info, job_len);

    helpTimeStop = flexiblas_wtime();

    data->dgebal.timings[0] += (helpTimeStop - helpTime);
    data->dgebal.calls[0]++;
    
    return;
}


/* Wrapper for dgebd2 */
void hook_dgebd2(blasint *m, blasint *n, double *a, blasint *lda, double *d, double *e, double *tauq, double *taup, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgebd2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) e, (void *) tauq, (void *) taup, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgebd2.timings[0] += (helpTimeStop - helpTime);
    data->dgebd2.calls[0]++;
    
    return;
}


/* Wrapper for dgebrd */
void hook_dgebrd(blasint *m, blasint *n, double *a, blasint *lda, double *d, double *e, double *tauq, double *taup, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgebrd((void *) m, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) e, (void *) tauq, (void *) taup, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgebrd.timings[0] += (helpTimeStop - helpTime);
    data->dgebrd.calls[0]++;
    
    return;
}


/* Wrapper for dgecon */
void hook_dgecon(char *norm, blasint *n, double *a, blasint *lda, double *anorm, double *rcond, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgecon((void *) norm, (void *) n, (void *) a, (void *) lda, (void *) anorm, (void *) rcond, (void *) work, (void *) iwork, (void *) info, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->dgecon.timings[0] += (helpTimeStop - helpTime);
    data->dgecon.calls[0]++;
    
    return;
}


/* Wrapper for dgeequ */
void hook_dgeequ(blasint *m, blasint *n, double *a, blasint *lda, double *r, double *c, double *rowcnd, double *colcnd, double *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgeequ((void *) m, (void *) n, (void *) a, (void *) lda, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeequ.timings[0] += (helpTimeStop - helpTime);
    data->dgeequ.calls[0]++;
    
    return;
}


/* Wrapper for dgeequb */
void hook_dgeequb(blasint *m, blasint *n, double *a, blasint *lda, double *r, double *c, double *rowcnd, double *colcnd, double *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgeequb((void *) m, (void *) n, (void *) a, (void *) lda, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeequb.timings[0] += (helpTimeStop - helpTime);
    data->dgeequb.calls[0]++;
    
    return;
}


/* Wrapper for dgees */
void hook_dgees(char *jobvs, char *sort, blaslogical *select, blasint *n, double *a, blasint *lda, blasint *sdim, double *wr, double *wi, double *vs, blasint *ldvs, double *work, blasint *lwork, blaslogical *bwork, blasint *info, flexiblas_fortran_charlen_t jobvs_len, flexiblas_fortran_charlen_t sort_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgees((void *) jobvs, (void *) sort, (void *) select, (void *) n, (void *) a, (void *) lda, (void *) sdim, (void *) wr, (void *) wi, (void *) vs, (void *) ldvs, (void *) work, (void *) lwork, (void *) bwork, (void *) info, jobvs_len, sort_len);

    helpTimeStop = flexiblas_wtime();

    data->dgees.timings[0] += (helpTimeStop - helpTime);
    data->dgees.calls[0]++;
    
    return;
}


/* Wrapper for dgeesx */
void hook_dgeesx(char *jobvs, char *sort, blaslogical *select, char *sense, blasint *n, double *a, blasint *lda, blasint *sdim, double *wr, double *wi, double *vs, blasint *ldvs, double *rconde, double *rcondv, double *work, blasint *lwork, blasint *iwork, blasint *liwork, blaslogical *bwork, blasint *info, flexiblas_fortran_charlen_t jobvs_len, flexiblas_fortran_charlen_t sort_len, flexiblas_fortran_charlen_t sense_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgeesx((void *) jobvs, (void *) sort, (void *) select, (void *) sense, (void *) n, (void *) a, (void *) lda, (void *) sdim, (void *) wr, (void *) wi, (void *) vs, (void *) ldvs, (void *) rconde, (void *) rcondv, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) bwork, (void *) info, jobvs_len, sort_len, sense_len);

    helpTimeStop = flexiblas_wtime();

    data->dgeesx.timings[0] += (helpTimeStop - helpTime);
    data->dgeesx.calls[0]++;
    
    return;
}


/* Wrapper for dgeev */
void hook_dgeev(char *jobvl, char *jobvr, blasint *n, double *a, blasint *lda, double *wr, double *wi, double *vl, blasint *ldvl, double *vr, blasint *ldvr, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobvl_len, flexiblas_fortran_charlen_t jobvr_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgeev((void *) jobvl, (void *) jobvr, (void *) n, (void *) a, (void *) lda, (void *) wr, (void *) wi, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) work, (void *) lwork, (void *) info, jobvl_len, jobvr_len);

    helpTimeStop = flexiblas_wtime();

    data->dgeev.timings[0] += (helpTimeStop - helpTime);
    data->dgeev.calls[0]++;
    
    return;
}


/* Wrapper for dgeevx */
void hook_dgeevx(char *balanc, char *jobvl, char *jobvr, char *sense, blasint *n, double *a, blasint *lda, double *wr, double *wi, double *vl, blasint *ldvl, double *vr, blasint *ldvr, blasint *ilo, blasint *ihi, double *scale, double *abnrm, double *rconde, double *rcondv, double *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t balanc_len, flexiblas_fortran_charlen_t jobvl_len, flexiblas_fortran_charlen_t jobvr_len, flexiblas_fortran_charlen_t sense_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgeevx((void *) balanc, (void *) jobvl, (void *) jobvr, (void *) sense, (void *) n, (void *) a, (void *) lda, (void *) wr, (void *) wi, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) ilo, (void *) ihi, (void *) scale, (void *) abnrm, (void *) rconde, (void *) rcondv, (void *) work, (void *) lwork, (void *) iwork, (void *) info, balanc_len, jobvl_len, jobvr_len, sense_len);

    helpTimeStop = flexiblas_wtime();

    data->dgeevx.timings[0] += (helpTimeStop - helpTime);
    data->dgeevx.calls[0]++;
    
    return;
}


/* Wrapper for dgehd2 */
void hook_dgehd2(blasint *n, blasint *ilo, blasint *ihi, double *a, blasint *lda, double *tau, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgehd2((void *) n, (void *) ilo, (void *) ihi, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgehd2.timings[0] += (helpTimeStop - helpTime);
    data->dgehd2.calls[0]++;
    
    return;
}


/* Wrapper for dgehrd */
void hook_dgehrd(blasint *n, blasint *ilo, blasint *ihi, double *a, blasint *lda, double *tau, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgehrd((void *) n, (void *) ilo, (void *) ihi, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgehrd.timings[0] += (helpTimeStop - helpTime);
    data->dgehrd.calls[0]++;
    
    return;
}


/* Wrapper for dgejsv */
void hook_dgejsv(char *joba, char *jobu, char *jobv, char *jobr, char *jobt, char *jobp, blasint *m, blasint *n, double *a, blasint *lda, double *sva, double *u, blasint *ldu, double *v, blasint *ldv, double *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t joba_len, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobv_len, flexiblas_fortran_charlen_t jobr_len, flexiblas_fortran_charlen_t jobt_len, flexiblas_fortran_charlen_t jobp_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgejsv((void *) joba, (void *) jobu, (void *) jobv, (void *) jobr, (void *) jobt, (void *) jobp, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) sva, (void *) u, (void *) ldu, (void *) v, (void *) ldv, (void *) work, (void *) lwork, (void *) iwork, (void *) info, joba_len, jobu_len, jobv_len, jobr_len, jobt_len, jobp_len);

    helpTimeStop = flexiblas_wtime();

    data->dgejsv.timings[0] += (helpTimeStop - helpTime);
    data->dgejsv.calls[0]++;
    
    return;
}


/* Wrapper for dgelq */
void hook_dgelq(blasint *m, blasint *n, double *a, blasint *lda, double *t, blasint *tsize, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgelq((void *) m, (void *) n, (void *) a, (void *) lda, (void *) t, (void *) tsize, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgelq.timings[0] += (helpTimeStop - helpTime);
    data->dgelq.calls[0]++;
    
    return;
}


/* Wrapper for dgelq2 */
void hook_dgelq2(blasint *m, blasint *n, double *a, blasint *lda, double *tau, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgelq2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgelq2.timings[0] += (helpTimeStop - helpTime);
    data->dgelq2.calls[0]++;
    
    return;
}


/* Wrapper for dgelqf */
void hook_dgelqf(blasint *m, blasint *n, double *a, blasint *lda, double *tau, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgelqf((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgelqf.timings[0] += (helpTimeStop - helpTime);
    data->dgelqf.calls[0]++;
    
    return;
}


/* Wrapper for dgelqt */
void hook_dgelqt(blasint *m, blasint *n, blasint *mb, double *a, blasint *lda, double *t, blasint *ldt, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgelqt((void *) m, (void *) n, (void *) mb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgelqt.timings[0] += (helpTimeStop - helpTime);
    data->dgelqt.calls[0]++;
    
    return;
}


/* Wrapper for dgelqt3 */
void hook_dgelqt3(blasint *m, blasint *n, double *a, blasint *lda, double *t, blasint *ldt, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgelqt3((void *) m, (void *) n, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgelqt3.timings[0] += (helpTimeStop - helpTime);
    data->dgelqt3.calls[0]++;
    
    return;
}


/* Wrapper for dgels */
void hook_dgels(char *trans, blasint *m, blasint *n, blasint *nrhs, double *a, blasint *lda, double *b, blasint *ldb, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgels((void *) trans, (void *) m, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dgels.timings[0] += (helpTimeStop - helpTime);
    data->dgels.calls[0]++;
    
    return;
}


/* Wrapper for dgelsd */
void hook_dgelsd(blasint *m, blasint *n, blasint *nrhs, double *a, blasint *lda, double *b, blasint *ldb, double *s, double *rcond, blasint *rank, double *work, blasint *lwork, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgelsd((void *) m, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) s, (void *) rcond, (void *) rank, (void *) work, (void *) lwork, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgelsd.timings[0] += (helpTimeStop - helpTime);
    data->dgelsd.calls[0]++;
    
    return;
}


/* Wrapper for dgelss */
void hook_dgelss(blasint *m, blasint *n, blasint *nrhs, double *a, blasint *lda, double *b, blasint *ldb, double *s, double *rcond, blasint *rank, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgelss((void *) m, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) s, (void *) rcond, (void *) rank, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgelss.timings[0] += (helpTimeStop - helpTime);
    data->dgelss.calls[0]++;
    
    return;
}


/* Wrapper for dgelsy */
void hook_dgelsy(blasint *m, blasint *n, blasint *nrhs, double *a, blasint *lda, double *b, blasint *ldb, blasint *jpvt, double *rcond, blasint *rank, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgelsy((void *) m, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) jpvt, (void *) rcond, (void *) rank, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgelsy.timings[0] += (helpTimeStop - helpTime);
    data->dgelsy.calls[0]++;
    
    return;
}


/* Wrapper for dgemlq */
void hook_dgemlq(char *side, char *trans, blasint *m, blasint *n, blasint *k, double *a, blasint *lda, double *t, blasint *tsize, double *c, blasint *ldc, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgemlq((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) t, (void *) tsize, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dgemlq.timings[0] += (helpTimeStop - helpTime);
    data->dgemlq.calls[0]++;
    
    return;
}


/* Wrapper for dgemlqt */
void hook_dgemlqt(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *mb, double *v, blasint *ldv, double *t, blasint *ldt, double *c, blasint *ldc, double *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgemlqt((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) mb, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dgemlqt.timings[0] += (helpTimeStop - helpTime);
    data->dgemlqt.calls[0]++;
    
    return;
}


/* Wrapper for dgemqr */
void hook_dgemqr(char *side, char *trans, blasint *m, blasint *n, blasint *k, double *a, blasint *lda, double *t, blasint *tsize, double *c, blasint *ldc, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgemqr((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) t, (void *) tsize, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dgemqr.timings[0] += (helpTimeStop - helpTime);
    data->dgemqr.calls[0]++;
    
    return;
}


/* Wrapper for dgemqrt */
void hook_dgemqrt(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *nb, double *v, blasint *ldv, double *t, blasint *ldt, double *c, blasint *ldc, double *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgemqrt((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) nb, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dgemqrt.timings[0] += (helpTimeStop - helpTime);
    data->dgemqrt.calls[0]++;
    
    return;
}


/* Wrapper for dgeql2 */
void hook_dgeql2(blasint *m, blasint *n, double *a, blasint *lda, double *tau, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgeql2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeql2.timings[0] += (helpTimeStop - helpTime);
    data->dgeql2.calls[0]++;
    
    return;
}


/* Wrapper for dgeqlf */
void hook_dgeqlf(blasint *m, blasint *n, double *a, blasint *lda, double *tau, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgeqlf((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeqlf.timings[0] += (helpTimeStop - helpTime);
    data->dgeqlf.calls[0]++;
    
    return;
}


/* Wrapper for dgeqp3 */
void hook_dgeqp3(blasint *m, blasint *n, double *a, blasint *lda, blasint *jpvt, double *tau, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgeqp3((void *) m, (void *) n, (void *) a, (void *) lda, (void *) jpvt, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeqp3.timings[0] += (helpTimeStop - helpTime);
    data->dgeqp3.calls[0]++;
    
    return;
}


/* Wrapper for dgeqr */
void hook_dgeqr(blasint *m, blasint *n, double *a, blasint *lda, double *t, blasint *tsize, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgeqr((void *) m, (void *) n, (void *) a, (void *) lda, (void *) t, (void *) tsize, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeqr.timings[0] += (helpTimeStop - helpTime);
    data->dgeqr.calls[0]++;
    
    return;
}


/* Wrapper for dgeqr2 */
void hook_dgeqr2(blasint *m, blasint *n, double *a, blasint *lda, double *tau, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgeqr2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeqr2.timings[0] += (helpTimeStop - helpTime);
    data->dgeqr2.calls[0]++;
    
    return;
}


/* Wrapper for dgeqr2p */
void hook_dgeqr2p(blasint *m, blasint *n, double *a, blasint *lda, double *tau, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgeqr2p((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeqr2p.timings[0] += (helpTimeStop - helpTime);
    data->dgeqr2p.calls[0]++;
    
    return;
}


/* Wrapper for dgeqrf */
void hook_dgeqrf(blasint *m, blasint *n, double *a, blasint *lda, double *tau, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgeqrf((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeqrf.timings[0] += (helpTimeStop - helpTime);
    data->dgeqrf.calls[0]++;
    
    return;
}


/* Wrapper for dgeqrfp */
void hook_dgeqrfp(blasint *m, blasint *n, double *a, blasint *lda, double *tau, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgeqrfp((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeqrfp.timings[0] += (helpTimeStop - helpTime);
    data->dgeqrfp.calls[0]++;
    
    return;
}


/* Wrapper for dgeqrt */
void hook_dgeqrt(blasint *m, blasint *n, blasint *nb, double *a, blasint *lda, double *t, blasint *ldt, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgeqrt((void *) m, (void *) n, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeqrt.timings[0] += (helpTimeStop - helpTime);
    data->dgeqrt.calls[0]++;
    
    return;
}


/* Wrapper for dgeqrt2 */
void hook_dgeqrt2(blasint *m, blasint *n, double *a, blasint *lda, double *t, blasint *ldt, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgeqrt2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeqrt2.timings[0] += (helpTimeStop - helpTime);
    data->dgeqrt2.calls[0]++;
    
    return;
}


/* Wrapper for dgeqrt3 */
void hook_dgeqrt3(blasint *m, blasint *n, double *a, blasint *lda, double *t, blasint *ldt, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgeqrt3((void *) m, (void *) n, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgeqrt3.timings[0] += (helpTimeStop - helpTime);
    data->dgeqrt3.calls[0]++;
    
    return;
}


/* Wrapper for dgerfs */
void hook_dgerfs(char *trans, blasint *n, blasint *nrhs, double *a, blasint *lda, double *af, blasint *ldaf, blasint *ipiv, double *b, blasint *ldb, double *x, blasint *ldx, double *ferr, double *berr, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgerfs((void *) trans, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) af, (void *) ldaf, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dgerfs.timings[0] += (helpTimeStop - helpTime);
    data->dgerfs.calls[0]++;
    
    return;
}


/* Wrapper for dgerq2 */
void hook_dgerq2(blasint *m, blasint *n, double *a, blasint *lda, double *tau, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgerq2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgerq2.timings[0] += (helpTimeStop - helpTime);
    data->dgerq2.calls[0]++;
    
    return;
}


/* Wrapper for dgerqf */
void hook_dgerqf(blasint *m, blasint *n, double *a, blasint *lda, double *tau, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgerqf((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgerqf.timings[0] += (helpTimeStop - helpTime);
    data->dgerqf.calls[0]++;
    
    return;
}


/* Wrapper for dgesc2 */
void hook_dgesc2(blasint *n, double *a, blasint *lda, double *rhs, blasint *ipiv, blasint *jpiv, double *scale)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgesc2((void *) n, (void *) a, (void *) lda, (void *) rhs, (void *) ipiv, (void *) jpiv, (void *) scale);

    helpTimeStop = flexiblas_wtime();

    data->dgesc2.timings[0] += (helpTimeStop - helpTime);
    data->dgesc2.calls[0]++;
    
    return;
}


/* Wrapper for dgesdd */
void hook_dgesdd(char *jobz, blasint *m, blasint *n, double *a, blasint *lda, double *s, double *u, blasint *ldu, double *vt, blasint *ldvt, double *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t jobz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgesdd((void *) jobz, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) s, (void *) u, (void *) ldu, (void *) vt, (void *) ldvt, (void *) work, (void *) lwork, (void *) iwork, (void *) info, jobz_len);

    helpTimeStop = flexiblas_wtime();

    data->dgesdd.timings[0] += (helpTimeStop - helpTime);
    data->dgesdd.calls[0]++;
    
    return;
}


/* Wrapper for dgesv */
void hook_dgesv(blasint *n, blasint *nrhs, double *a, blasint *lda, blasint *ipiv, double *b, blasint *ldb, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgesv((void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgesv.timings[0] += (helpTimeStop - helpTime);
    data->dgesv.calls[0]++;
    
    return;
}


/* Wrapper for dgesvd */
void hook_dgesvd(char *jobu, char *jobvt, blasint *m, blasint *n, double *a, blasint *lda, double *s, double *u, blasint *ldu, double *vt, blasint *ldvt, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobvt_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgesvd((void *) jobu, (void *) jobvt, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) s, (void *) u, (void *) ldu, (void *) vt, (void *) ldvt, (void *) work, (void *) lwork, (void *) info, jobu_len, jobvt_len);

    helpTimeStop = flexiblas_wtime();

    data->dgesvd.timings[0] += (helpTimeStop - helpTime);
    data->dgesvd.calls[0]++;
    
    return;
}


/* Wrapper for dgesvdq */
void hook_dgesvdq(char *joba, char *jobp, char *jobr, char *jobu, char *jobv, blasint *m, blasint *n, double *a, blasint *lda, double *s, double *u, blasint *ldu, double *v, blasint *ldv, blasint *numrank, blasint *iwork, blasint *liwork, double *work, blasint *lwork, double *rwork, blasint *lrwork, blasint *info, flexiblas_fortran_charlen_t joba_len, flexiblas_fortran_charlen_t jobp_len, flexiblas_fortran_charlen_t jobr_len, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobv_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgesvdq((void *) joba, (void *) jobp, (void *) jobr, (void *) jobu, (void *) jobv, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) s, (void *) u, (void *) ldu, (void *) v, (void *) ldv, (void *) numrank, (void *) iwork, (void *) liwork, (void *) work, (void *) lwork, (void *) rwork, (void *) lrwork, (void *) info, joba_len, jobp_len, jobr_len, jobu_len, jobv_len);

    helpTimeStop = flexiblas_wtime();

    data->dgesvdq.timings[0] += (helpTimeStop - helpTime);
    data->dgesvdq.calls[0]++;
    
    return;
}


/* Wrapper for dgesvdx */
void hook_dgesvdx(char *jobu, char *jobvt, char *range, blasint *m, blasint *n, double *a, blasint *lda, double *vl, double *vu, blasint *il, blasint *iu, blasint *ns, double *s, double *u, blasint *ldu, double *vt, blasint *ldvt, double *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobvt_len, flexiblas_fortran_charlen_t range_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgesvdx((void *) jobu, (void *) jobvt, (void *) range, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) ns, (void *) s, (void *) u, (void *) ldu, (void *) vt, (void *) ldvt, (void *) work, (void *) lwork, (void *) iwork, (void *) info, jobu_len, jobvt_len, range_len);

    helpTimeStop = flexiblas_wtime();

    data->dgesvdx.timings[0] += (helpTimeStop - helpTime);
    data->dgesvdx.calls[0]++;
    
    return;
}


/* Wrapper for dgesvj */
void hook_dgesvj(char *joba, char *jobu, char *jobv, blasint *m, blasint *n, double *a, blasint *lda, double *sva, blasint *mv, double *v, blasint *ldv, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t joba_len, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobv_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgesvj((void *) joba, (void *) jobu, (void *) jobv, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) sva, (void *) mv, (void *) v, (void *) ldv, (void *) work, (void *) lwork, (void *) info, joba_len, jobu_len, jobv_len);

    helpTimeStop = flexiblas_wtime();

    data->dgesvj.timings[0] += (helpTimeStop - helpTime);
    data->dgesvj.calls[0]++;
    
    return;
}


/* Wrapper for dgesvx */
void hook_dgesvx(char *fact, char *trans, blasint *n, blasint *nrhs, double *a, blasint *lda, double *af, blasint *ldaf, blasint *ipiv, char *equed, double *r, double *c, double *b, blasint *ldb, double *x, blasint *ldx, double *rcond, double *ferr, double *berr, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgesvx((void *) fact, (void *) trans, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) af, (void *) ldaf, (void *) ipiv, (void *) equed, (void *) r, (void *) c, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, fact_len, trans_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->dgesvx.timings[0] += (helpTimeStop - helpTime);
    data->dgesvx.calls[0]++;
    
    return;
}


/* Wrapper for dgetc2 */
void hook_dgetc2(blasint *n, double *a, blasint *lda, blasint *ipiv, blasint *jpiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgetc2((void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) jpiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgetc2.timings[0] += (helpTimeStop - helpTime);
    data->dgetc2.calls[0]++;
    
    return;
}


/* Wrapper for dgetf2 */
void hook_dgetf2(blasint *m, blasint *n, double *a, blasint *lda, blasint *ipiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgetf2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgetf2.timings[0] += (helpTimeStop - helpTime);
    data->dgetf2.calls[0]++;
    
    return;
}


/* Wrapper for dgetrf */
void hook_dgetrf(blasint *m, blasint *n, double *a, blasint *lda, blasint *ipiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgetrf((void *) m, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgetrf.timings[0] += (helpTimeStop - helpTime);
    data->dgetrf.calls[0]++;
    
    return;
}


/* Wrapper for dgetrf2 */
void hook_dgetrf2(blasint *m, blasint *n, double *a, blasint *lda, blasint *ipiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgetrf2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgetrf2.timings[0] += (helpTimeStop - helpTime);
    data->dgetrf2.calls[0]++;
    
    return;
}


/* Wrapper for dgetri */
void hook_dgetri(blasint *n, double *a, blasint *lda, blasint *ipiv, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgetri((void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgetri.timings[0] += (helpTimeStop - helpTime);
    data->dgetri.calls[0]++;
    
    return;
}


/* Wrapper for dgetrs */
void hook_dgetrs(char *trans, blasint *n, blasint *nrhs, double *a, blasint *lda, blasint *ipiv, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgetrs((void *) trans, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dgetrs.timings[0] += (helpTimeStop - helpTime);
    data->dgetrs.calls[0]++;
    
    return;
}


/* Wrapper for dgetsls */
void hook_dgetsls(char *trans, blasint *m, blasint *n, blasint *nrhs, double *a, blasint *lda, double *b, blasint *ldb, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgetsls((void *) trans, (void *) m, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dgetsls.timings[0] += (helpTimeStop - helpTime);
    data->dgetsls.calls[0]++;
    
    return;
}


/* Wrapper for dgetsqrhrt */
void hook_dgetsqrhrt(blasint *m, blasint *n, blasint *mb1, blasint *nb1, blasint *nb2, double *a, blasint *lda, double *t, blasint *ldt, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgetsqrhrt((void *) m, (void *) n, (void *) mb1, (void *) nb1, (void *) nb2, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgetsqrhrt.timings[0] += (helpTimeStop - helpTime);
    data->dgetsqrhrt.calls[0]++;
    
    return;
}


/* Wrapper for dggbak */
void hook_dggbak(char *job, char *side, blasint *n, blasint *ilo, blasint *ihi, double *lscale, double *rscale, blasint *m, double *v, blasint *ldv, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t side_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dggbak((void *) job, (void *) side, (void *) n, (void *) ilo, (void *) ihi, (void *) lscale, (void *) rscale, (void *) m, (void *) v, (void *) ldv, (void *) info, job_len, side_len);

    helpTimeStop = flexiblas_wtime();

    data->dggbak.timings[0] += (helpTimeStop - helpTime);
    data->dggbak.calls[0]++;
    
    return;
}


/* Wrapper for dggbal */
void hook_dggbal(char *job, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, blasint *ilo, blasint *ihi, double *lscale, double *rscale, double *work, blasint *info, flexiblas_fortran_charlen_t job_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dggbal((void *) job, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) ilo, (void *) ihi, (void *) lscale, (void *) rscale, (void *) work, (void *) info, job_len);

    helpTimeStop = flexiblas_wtime();

    data->dggbal.timings[0] += (helpTimeStop - helpTime);
    data->dggbal.calls[0]++;
    
    return;
}


/* Wrapper for dgges */
void hook_dgges(char *jobvsl, char *jobvsr, char *sort, blaslogical *selctg, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, blasint *sdim, double *alphar, double *alphai, double *beta, double *vsl, blasint *ldvsl, double *vsr, blasint *ldvsr, double *work, blasint *lwork, blaslogical *bwork, blasint *info, flexiblas_fortran_charlen_t jobvsl_len, flexiblas_fortran_charlen_t jobvsr_len, flexiblas_fortran_charlen_t sort_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgges((void *) jobvsl, (void *) jobvsr, (void *) sort, (void *) selctg, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) sdim, (void *) alphar, (void *) alphai, (void *) beta, (void *) vsl, (void *) ldvsl, (void *) vsr, (void *) ldvsr, (void *) work, (void *) lwork, (void *) bwork, (void *) info, jobvsl_len, jobvsr_len, sort_len);

    helpTimeStop = flexiblas_wtime();

    data->dgges.timings[0] += (helpTimeStop - helpTime);
    data->dgges.calls[0]++;
    
    return;
}


/* Wrapper for dgges3 */
void hook_dgges3(char *jobvsl, char *jobvsr, char *sort, blaslogical *selctg, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, blasint *sdim, double *alphar, double *alphai, double *beta, double *vsl, blasint *ldvsl, double *vsr, blasint *ldvsr, double *work, blasint *lwork, blaslogical *bwork, blasint *info, flexiblas_fortran_charlen_t jobvsl_len, flexiblas_fortran_charlen_t jobvsr_len, flexiblas_fortran_charlen_t sort_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgges3((void *) jobvsl, (void *) jobvsr, (void *) sort, (void *) selctg, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) sdim, (void *) alphar, (void *) alphai, (void *) beta, (void *) vsl, (void *) ldvsl, (void *) vsr, (void *) ldvsr, (void *) work, (void *) lwork, (void *) bwork, (void *) info, jobvsl_len, jobvsr_len, sort_len);

    helpTimeStop = flexiblas_wtime();

    data->dgges3.timings[0] += (helpTimeStop - helpTime);
    data->dgges3.calls[0]++;
    
    return;
}


/* Wrapper for dggesx */
void hook_dggesx(char *jobvsl, char *jobvsr, char *sort, blaslogical *selctg, char *sense, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, blasint *sdim, double *alphar, double *alphai, double *beta, double *vsl, blasint *ldvsl, double *vsr, blasint *ldvsr, double *rconde, double *rcondv, double *work, blasint *lwork, blasint *iwork, blasint *liwork, blaslogical *bwork, blasint *info, flexiblas_fortran_charlen_t jobvsl_len, flexiblas_fortran_charlen_t jobvsr_len, flexiblas_fortran_charlen_t sort_len, flexiblas_fortran_charlen_t sense_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dggesx((void *) jobvsl, (void *) jobvsr, (void *) sort, (void *) selctg, (void *) sense, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) sdim, (void *) alphar, (void *) alphai, (void *) beta, (void *) vsl, (void *) ldvsl, (void *) vsr, (void *) ldvsr, (void *) rconde, (void *) rcondv, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) bwork, (void *) info, jobvsl_len, jobvsr_len, sort_len, sense_len);

    helpTimeStop = flexiblas_wtime();

    data->dggesx.timings[0] += (helpTimeStop - helpTime);
    data->dggesx.calls[0]++;
    
    return;
}


/* Wrapper for dggev */
void hook_dggev(char *jobvl, char *jobvr, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, double *alphar, double *alphai, double *beta, double *vl, blasint *ldvl, double *vr, blasint *ldvr, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobvl_len, flexiblas_fortran_charlen_t jobvr_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dggev((void *) jobvl, (void *) jobvr, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) alphar, (void *) alphai, (void *) beta, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) work, (void *) lwork, (void *) info, jobvl_len, jobvr_len);

    helpTimeStop = flexiblas_wtime();

    data->dggev.timings[0] += (helpTimeStop - helpTime);
    data->dggev.calls[0]++;
    
    return;
}


/* Wrapper for dggev3 */
void hook_dggev3(char *jobvl, char *jobvr, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, double *alphar, double *alphai, double *beta, double *vl, blasint *ldvl, double *vr, blasint *ldvr, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobvl_len, flexiblas_fortran_charlen_t jobvr_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dggev3((void *) jobvl, (void *) jobvr, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) alphar, (void *) alphai, (void *) beta, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) work, (void *) lwork, (void *) info, jobvl_len, jobvr_len);

    helpTimeStop = flexiblas_wtime();

    data->dggev3.timings[0] += (helpTimeStop - helpTime);
    data->dggev3.calls[0]++;
    
    return;
}


/* Wrapper for dggevx */
void hook_dggevx(char *balanc, char *jobvl, char *jobvr, char *sense, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, double *alphar, double *alphai, double *beta, double *vl, blasint *ldvl, double *vr, blasint *ldvr, blasint *ilo, blasint *ihi, double *lscale, double *rscale, double *abnrm, double *bbnrm, double *rconde, double *rcondv, double *work, blasint *lwork, blasint *iwork, blaslogical *bwork, blasint *info, flexiblas_fortran_charlen_t balanc_len, flexiblas_fortran_charlen_t jobvl_len, flexiblas_fortran_charlen_t jobvr_len, flexiblas_fortran_charlen_t sense_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dggevx((void *) balanc, (void *) jobvl, (void *) jobvr, (void *) sense, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) alphar, (void *) alphai, (void *) beta, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) ilo, (void *) ihi, (void *) lscale, (void *) rscale, (void *) abnrm, (void *) bbnrm, (void *) rconde, (void *) rcondv, (void *) work, (void *) lwork, (void *) iwork, (void *) bwork, (void *) info, balanc_len, jobvl_len, jobvr_len, sense_len);

    helpTimeStop = flexiblas_wtime();

    data->dggevx.timings[0] += (helpTimeStop - helpTime);
    data->dggevx.calls[0]++;
    
    return;
}


/* Wrapper for dggglm */
void hook_dggglm(blasint *n, blasint *m, blasint *p, double *a, blasint *lda, double *b, blasint *ldb, double *d, double *x, double *y, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dggglm((void *) n, (void *) m, (void *) p, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) d, (void *) x, (void *) y, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dggglm.timings[0] += (helpTimeStop - helpTime);
    data->dggglm.calls[0]++;
    
    return;
}


/* Wrapper for dgghd3 */
void hook_dgghd3(char *compq, char *compz, blasint *n, blasint *ilo, blasint *ihi, double *a, blasint *lda, double *b, blasint *ldb, double *q, blasint *ldq, double *z, blasint *ldz, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t compq_len, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgghd3((void *) compq, (void *) compz, (void *) n, (void *) ilo, (void *) ihi, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) q, (void *) ldq, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) info, compq_len, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->dgghd3.timings[0] += (helpTimeStop - helpTime);
    data->dgghd3.calls[0]++;
    
    return;
}


/* Wrapper for dgghrd */
void hook_dgghrd(char *compq, char *compz, blasint *n, blasint *ilo, blasint *ihi, double *a, blasint *lda, double *b, blasint *ldb, double *q, blasint *ldq, double *z, blasint *ldz, blasint *info, flexiblas_fortran_charlen_t compq_len, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgghrd((void *) compq, (void *) compz, (void *) n, (void *) ilo, (void *) ihi, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) q, (void *) ldq, (void *) z, (void *) ldz, (void *) info, compq_len, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->dgghrd.timings[0] += (helpTimeStop - helpTime);
    data->dgghrd.calls[0]++;
    
    return;
}


/* Wrapper for dgglse */
void hook_dgglse(blasint *m, blasint *n, blasint *p, double *a, blasint *lda, double *b, blasint *ldb, double *c, double *d, double *x, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgglse((void *) m, (void *) n, (void *) p, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) c, (void *) d, (void *) x, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgglse.timings[0] += (helpTimeStop - helpTime);
    data->dgglse.calls[0]++;
    
    return;
}


/* Wrapper for dggqrf */
void hook_dggqrf(blasint *n, blasint *m, blasint *p, double *a, blasint *lda, double *taua, double *b, blasint *ldb, double *taub, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dggqrf((void *) n, (void *) m, (void *) p, (void *) a, (void *) lda, (void *) taua, (void *) b, (void *) ldb, (void *) taub, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dggqrf.timings[0] += (helpTimeStop - helpTime);
    data->dggqrf.calls[0]++;
    
    return;
}


/* Wrapper for dggrqf */
void hook_dggrqf(blasint *m, blasint *p, blasint *n, double *a, blasint *lda, double *taua, double *b, blasint *ldb, double *taub, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dggrqf((void *) m, (void *) p, (void *) n, (void *) a, (void *) lda, (void *) taua, (void *) b, (void *) ldb, (void *) taub, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dggrqf.timings[0] += (helpTimeStop - helpTime);
    data->dggrqf.calls[0]++;
    
    return;
}


/* Wrapper for dggsvd3 */
void hook_dggsvd3(char *jobu, char *jobv, char *jobq, blasint *m, blasint *n, blasint *p, blasint *k, blasint *l, double *a, blasint *lda, double *b, blasint *ldb, double *alpha, double *beta, double *u, blasint *ldu, double *v, blasint *ldv, double *q, blasint *ldq, double *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobv_len, flexiblas_fortran_charlen_t jobq_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dggsvd3((void *) jobu, (void *) jobv, (void *) jobq, (void *) m, (void *) n, (void *) p, (void *) k, (void *) l, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) alpha, (void *) beta, (void *) u, (void *) ldu, (void *) v, (void *) ldv, (void *) q, (void *) ldq, (void *) work, (void *) lwork, (void *) iwork, (void *) info, jobu_len, jobv_len, jobq_len);

    helpTimeStop = flexiblas_wtime();

    data->dggsvd3.timings[0] += (helpTimeStop - helpTime);
    data->dggsvd3.calls[0]++;
    
    return;
}


/* Wrapper for dggsvp3 */
void hook_dggsvp3(char *jobu, char *jobv, char *jobq, blasint *m, blasint *p, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, double *tola, double *tolb, blasint *k, blasint *l, double *u, blasint *ldu, double *v, blasint *ldv, double *q, blasint *ldq, blasint *iwork, double *tau, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobv_len, flexiblas_fortran_charlen_t jobq_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dggsvp3((void *) jobu, (void *) jobv, (void *) jobq, (void *) m, (void *) p, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) tola, (void *) tolb, (void *) k, (void *) l, (void *) u, (void *) ldu, (void *) v, (void *) ldv, (void *) q, (void *) ldq, (void *) iwork, (void *) tau, (void *) work, (void *) lwork, (void *) info, jobu_len, jobv_len, jobq_len);

    helpTimeStop = flexiblas_wtime();

    data->dggsvp3.timings[0] += (helpTimeStop - helpTime);
    data->dggsvp3.calls[0]++;
    
    return;
}


/* Wrapper for dgsvj0 */
void hook_dgsvj0(char *jobv, blasint *m, blasint *n, double *a, blasint *lda, double *d, double *sva, blasint *mv, double *v, blasint *ldv, double *eps, double *sfmin, double *tol, blasint *nsweep, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobv_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgsvj0((void *) jobv, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) sva, (void *) mv, (void *) v, (void *) ldv, (void *) eps, (void *) sfmin, (void *) tol, (void *) nsweep, (void *) work, (void *) lwork, (void *) info, jobv_len);

    helpTimeStop = flexiblas_wtime();

    data->dgsvj0.timings[0] += (helpTimeStop - helpTime);
    data->dgsvj0.calls[0]++;
    
    return;
}


/* Wrapper for dgsvj1 */
void hook_dgsvj1(char *jobv, blasint *m, blasint *n, blasint *n1, double *a, blasint *lda, double *d, double *sva, blasint *mv, double *v, blasint *ldv, double *eps, double *sfmin, double *tol, blasint *nsweep, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobv_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgsvj1((void *) jobv, (void *) m, (void *) n, (void *) n1, (void *) a, (void *) lda, (void *) d, (void *) sva, (void *) mv, (void *) v, (void *) ldv, (void *) eps, (void *) sfmin, (void *) tol, (void *) nsweep, (void *) work, (void *) lwork, (void *) info, jobv_len);

    helpTimeStop = flexiblas_wtime();

    data->dgsvj1.timings[0] += (helpTimeStop - helpTime);
    data->dgsvj1.calls[0]++;
    
    return;
}


/* Wrapper for dgtcon */
void hook_dgtcon(char *norm, blasint *n, double *dl, double *d, double *du, double *du2, blasint *ipiv, double *anorm, double *rcond, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgtcon((void *) norm, (void *) n, (void *) dl, (void *) d, (void *) du, (void *) du2, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) iwork, (void *) info, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->dgtcon.timings[0] += (helpTimeStop - helpTime);
    data->dgtcon.calls[0]++;
    
    return;
}


/* Wrapper for dgtrfs */
void hook_dgtrfs(char *trans, blasint *n, blasint *nrhs, double *dl, double *d, double *du, double *dlf, double *df, double *duf, double *du2, blasint *ipiv, double *b, blasint *ldb, double *x, blasint *ldx, double *ferr, double *berr, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgtrfs((void *) trans, (void *) n, (void *) nrhs, (void *) dl, (void *) d, (void *) du, (void *) dlf, (void *) df, (void *) duf, (void *) du2, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dgtrfs.timings[0] += (helpTimeStop - helpTime);
    data->dgtrfs.calls[0]++;
    
    return;
}


/* Wrapper for dgtsv */
void hook_dgtsv(blasint *n, blasint *nrhs, double *dl, double *d, double *du, double *b, blasint *ldb, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgtsv((void *) n, (void *) nrhs, (void *) dl, (void *) d, (void *) du, (void *) b, (void *) ldb, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgtsv.timings[0] += (helpTimeStop - helpTime);
    data->dgtsv.calls[0]++;
    
    return;
}


/* Wrapper for dgtsvx */
void hook_dgtsvx(char *fact, char *trans, blasint *n, blasint *nrhs, double *dl, double *d, double *du, double *dlf, double *df, double *duf, double *du2, blasint *ipiv, double *b, blasint *ldb, double *x, blasint *ldx, double *rcond, double *ferr, double *berr, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgtsvx((void *) fact, (void *) trans, (void *) n, (void *) nrhs, (void *) dl, (void *) d, (void *) du, (void *) dlf, (void *) df, (void *) duf, (void *) du2, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, fact_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dgtsvx.timings[0] += (helpTimeStop - helpTime);
    data->dgtsvx.calls[0]++;
    
    return;
}


/* Wrapper for dgttrf */
void hook_dgttrf(blasint *n, double *dl, double *d, double *du, double *du2, blasint *ipiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgttrf((void *) n, (void *) dl, (void *) d, (void *) du, (void *) du2, (void *) ipiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dgttrf.timings[0] += (helpTimeStop - helpTime);
    data->dgttrf.calls[0]++;
    
    return;
}


/* Wrapper for dgttrs */
void hook_dgttrs(char *trans, blasint *n, blasint *nrhs, double *dl, double *d, double *du, double *du2, blasint *ipiv, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgttrs((void *) trans, (void *) n, (void *) nrhs, (void *) dl, (void *) d, (void *) du, (void *) du2, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dgttrs.timings[0] += (helpTimeStop - helpTime);
    data->dgttrs.calls[0]++;
    
    return;
}


/* Wrapper for dgtts2 */
void hook_dgtts2(blasint *itrans, blasint *n, blasint *nrhs, double *dl, double *d, double *du, double *du2, blasint *ipiv, double *b, blasint *ldb)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dgtts2((void *) itrans, (void *) n, (void *) nrhs, (void *) dl, (void *) d, (void *) du, (void *) du2, (void *) ipiv, (void *) b, (void *) ldb);

    helpTimeStop = flexiblas_wtime();

    data->dgtts2.timings[0] += (helpTimeStop - helpTime);
    data->dgtts2.calls[0]++;
    
    return;
}


/* Wrapper for dhgeqz */
void hook_dhgeqz(char *job, char *compq, char *compz, blasint *n, blasint *ilo, blasint *ihi, double *h, blasint *ldh, double *t, blasint *ldt, double *alphar, double *alphai, double *beta, double *q, blasint *ldq, double *z, blasint *ldz, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t compq_len, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dhgeqz((void *) job, (void *) compq, (void *) compz, (void *) n, (void *) ilo, (void *) ihi, (void *) h, (void *) ldh, (void *) t, (void *) ldt, (void *) alphar, (void *) alphai, (void *) beta, (void *) q, (void *) ldq, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) info, job_len, compq_len, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->dhgeqz.timings[0] += (helpTimeStop - helpTime);
    data->dhgeqz.calls[0]++;
    
    return;
}


/* Wrapper for dhsein */
void hook_dhsein(char *side, char *eigsrc, char *initv, blaslogical *select, blasint *n, double *h, blasint *ldh, double *wr, double *wi, double *vl, blasint *ldvl, double *vr, blasint *ldvr, blasint *mm, blasint *m, double *work, blasint *ifaill, blasint *ifailr, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t eigsrc_len, flexiblas_fortran_charlen_t initv_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dhsein((void *) side, (void *) eigsrc, (void *) initv, (void *) select, (void *) n, (void *) h, (void *) ldh, (void *) wr, (void *) wi, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) mm, (void *) m, (void *) work, (void *) ifaill, (void *) ifailr, (void *) info, side_len, eigsrc_len, initv_len);

    helpTimeStop = flexiblas_wtime();

    data->dhsein.timings[0] += (helpTimeStop - helpTime);
    data->dhsein.calls[0]++;
    
    return;
}


/* Wrapper for dhseqr */
void hook_dhseqr(char *job, char *compz, blasint *n, blasint *ilo, blasint *ihi, double *h, blasint *ldh, double *wr, double *wi, double *z, blasint *ldz, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dhseqr((void *) job, (void *) compz, (void *) n, (void *) ilo, (void *) ihi, (void *) h, (void *) ldh, (void *) wr, (void *) wi, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) info, job_len, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->dhseqr.timings[0] += (helpTimeStop - helpTime);
    data->dhseqr.calls[0]++;
    
    return;
}

/* Wrapper for disnan */
int_least32_t hook_disnan(double *din)
{
    double helpTime;
    double helpTimeStop;
    int_least32_t return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_disnan((void *) din);

    helpTimeStop = flexiblas_wtime();

    data->disnan.timings[0] += (helpTimeStop - helpTime);
    data->disnan.calls[0]++;
    
    return return_value;
}



/* Wrapper for dlabad */
void hook_dlabad(double *small, double *large)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlabad((void *) small, (void *) large);

    helpTimeStop = flexiblas_wtime();

    data->dlabad.timings[0] += (helpTimeStop - helpTime);
    data->dlabad.calls[0]++;
    
    return;
}


/* Wrapper for dlabrd */
void hook_dlabrd(blasint *m, blasint *n, blasint *nb, double *a, blasint *lda, double *d, double *e, double *tauq, double *taup, double *x, blasint *ldx, double *y, blasint *ldy)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlabrd((void *) m, (void *) n, (void *) nb, (void *) a, (void *) lda, (void *) d, (void *) e, (void *) tauq, (void *) taup, (void *) x, (void *) ldx, (void *) y, (void *) ldy);

    helpTimeStop = flexiblas_wtime();

    data->dlabrd.timings[0] += (helpTimeStop - helpTime);
    data->dlabrd.calls[0]++;
    
    return;
}


/* Wrapper for dlacn2 */
void hook_dlacn2(blasint *n, double *v, double *x, blasint *isgn, double *est, blasint *kase, blasint *isave)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlacn2((void *) n, (void *) v, (void *) x, (void *) isgn, (void *) est, (void *) kase, (void *) isave);

    helpTimeStop = flexiblas_wtime();

    data->dlacn2.timings[0] += (helpTimeStop - helpTime);
    data->dlacn2.calls[0]++;
    
    return;
}


/* Wrapper for dlacon */
void hook_dlacon(blasint *n, double *v, double *x, blasint *isgn, double *est, blasint *kase)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlacon((void *) n, (void *) v, (void *) x, (void *) isgn, (void *) est, (void *) kase);

    helpTimeStop = flexiblas_wtime();

    data->dlacon.timings[0] += (helpTimeStop - helpTime);
    data->dlacon.calls[0]++;
    
    return;
}


/* Wrapper for dlacpy */
void hook_dlacpy(char *uplo, blasint *m, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlacpy((void *) uplo, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dlacpy.timings[0] += (helpTimeStop - helpTime);
    data->dlacpy.calls[0]++;
    
    return;
}


/* Wrapper for dladiv */
void hook_dladiv(double *a, double *b, double *c, double *d, double *p, double *q)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dladiv((void *) a, (void *) b, (void *) c, (void *) d, (void *) p, (void *) q);

    helpTimeStop = flexiblas_wtime();

    data->dladiv.timings[0] += (helpTimeStop - helpTime);
    data->dladiv.calls[0]++;
    
    return;
}


/* Wrapper for dladiv1 */
void hook_dladiv1(double *a, double *b, double *c, double *d, double *p, double *q)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dladiv1((void *) a, (void *) b, (void *) c, (void *) d, (void *) p, (void *) q);

    helpTimeStop = flexiblas_wtime();

    data->dladiv1.timings[0] += (helpTimeStop - helpTime);
    data->dladiv1.calls[0]++;
    
    return;
}

/* Wrapper for dladiv2 */
double hook_dladiv2(double *a, double *b, double *c, double *d, double *r, double *t)
{
    double helpTime;
    double helpTimeStop;
    double return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dladiv2((void *) a, (void *) b, (void *) c, (void *) d, (void *) r, (void *) t);

    helpTimeStop = flexiblas_wtime();

    data->dladiv2.timings[0] += (helpTimeStop - helpTime);
    data->dladiv2.calls[0]++;
    
    return return_value;
}



/* Wrapper for dlae2 */
void hook_dlae2(double *a, double *b, double *c, double *rt1, double *rt2)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlae2((void *) a, (void *) b, (void *) c, (void *) rt1, (void *) rt2);

    helpTimeStop = flexiblas_wtime();

    data->dlae2.timings[0] += (helpTimeStop - helpTime);
    data->dlae2.calls[0]++;
    
    return;
}


/* Wrapper for dlaebz */
void hook_dlaebz(blasint *ijob, blasint *nitmax, blasint *n, blasint *mmax, blasint *minp, blasint *nbmin, double *abstol, double *reltol, double *pivmin, double *d, double *e, double *e2, blasint *nval, double *ab, double *c, blasint *mout, blasint *nab, double *work, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaebz((void *) ijob, (void *) nitmax, (void *) n, (void *) mmax, (void *) minp, (void *) nbmin, (void *) abstol, (void *) reltol, (void *) pivmin, (void *) d, (void *) e, (void *) e2, (void *) nval, (void *) ab, (void *) c, (void *) mout, (void *) nab, (void *) work, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaebz.timings[0] += (helpTimeStop - helpTime);
    data->dlaebz.calls[0]++;
    
    return;
}


/* Wrapper for dlaed0 */
void hook_dlaed0(blasint *icompq, blasint *qsiz, blasint *n, double *d, double *e, double *q, blasint *ldq, double *qstore, blasint *ldqs, double *work, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaed0((void *) icompq, (void *) qsiz, (void *) n, (void *) d, (void *) e, (void *) q, (void *) ldq, (void *) qstore, (void *) ldqs, (void *) work, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaed0.timings[0] += (helpTimeStop - helpTime);
    data->dlaed0.calls[0]++;
    
    return;
}


/* Wrapper for dlaed1 */
void hook_dlaed1(blasint *n, double *d, double *q, blasint *ldq, blasint *indxq, double *rho, blasint *cutpnt, double *work, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaed1((void *) n, (void *) d, (void *) q, (void *) ldq, (void *) indxq, (void *) rho, (void *) cutpnt, (void *) work, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaed1.timings[0] += (helpTimeStop - helpTime);
    data->dlaed1.calls[0]++;
    
    return;
}


/* Wrapper for dlaed2 */
void hook_dlaed2(blasint *k, blasint *n, blasint *n1, double *d, double *q, blasint *ldq, blasint *indxq, double *rho, double *z, double *dlamda, double *w, double *q2, blasint *indx, blasint *indxc, blasint *indxp, blasint *coltyp, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaed2((void *) k, (void *) n, (void *) n1, (void *) d, (void *) q, (void *) ldq, (void *) indxq, (void *) rho, (void *) z, (void *) dlamda, (void *) w, (void *) q2, (void *) indx, (void *) indxc, (void *) indxp, (void *) coltyp, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaed2.timings[0] += (helpTimeStop - helpTime);
    data->dlaed2.calls[0]++;
    
    return;
}


/* Wrapper for dlaed3 */
void hook_dlaed3(blasint *k, blasint *n, blasint *n1, double *d, double *q, blasint *ldq, double *rho, double *dlamda, double *q2, blasint *indx, blasint *ctot, double *w, double *s, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaed3((void *) k, (void *) n, (void *) n1, (void *) d, (void *) q, (void *) ldq, (void *) rho, (void *) dlamda, (void *) q2, (void *) indx, (void *) ctot, (void *) w, (void *) s, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaed3.timings[0] += (helpTimeStop - helpTime);
    data->dlaed3.calls[0]++;
    
    return;
}


/* Wrapper for dlaed4 */
void hook_dlaed4(blasint *n, blasint *i, double *d, double *z, double *delta, double *rho, double *dlam, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaed4((void *) n, (void *) i, (void *) d, (void *) z, (void *) delta, (void *) rho, (void *) dlam, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaed4.timings[0] += (helpTimeStop - helpTime);
    data->dlaed4.calls[0]++;
    
    return;
}


/* Wrapper for dlaed5 */
void hook_dlaed5(blasint *i, double *d, double *z, double *delta, double *rho, double *dlam)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaed5((void *) i, (void *) d, (void *) z, (void *) delta, (void *) rho, (void *) dlam);

    helpTimeStop = flexiblas_wtime();

    data->dlaed5.timings[0] += (helpTimeStop - helpTime);
    data->dlaed5.calls[0]++;
    
    return;
}


/* Wrapper for dlaed6 */
void hook_dlaed6(blasint *kniter, blaslogical *orgati, double *rho, double *d, double *z, double *finit, double *tau, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaed6((void *) kniter, (void *) orgati, (void *) rho, (void *) d, (void *) z, (void *) finit, (void *) tau, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaed6.timings[0] += (helpTimeStop - helpTime);
    data->dlaed6.calls[0]++;
    
    return;
}


/* Wrapper for dlaed7 */
void hook_dlaed7(blasint *icompq, blasint *n, blasint *qsiz, blasint *tlvls, blasint *curlvl, blasint *curpbm, double *d, double *q, blasint *ldq, blasint *indxq, double *rho, blasint *cutpnt, double *qstore, blasint *qptr, blasint *prmptr, blasint *perm, blasint *givptr, blasint *givcol, double *givnum, double *work, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaed7((void *) icompq, (void *) n, (void *) qsiz, (void *) tlvls, (void *) curlvl, (void *) curpbm, (void *) d, (void *) q, (void *) ldq, (void *) indxq, (void *) rho, (void *) cutpnt, (void *) qstore, (void *) qptr, (void *) prmptr, (void *) perm, (void *) givptr, (void *) givcol, (void *) givnum, (void *) work, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaed7.timings[0] += (helpTimeStop - helpTime);
    data->dlaed7.calls[0]++;
    
    return;
}


/* Wrapper for dlaed8 */
void hook_dlaed8(blasint *icompq, blasint *k, blasint *n, blasint *qsiz, double *d, double *q, blasint *ldq, blasint *indxq, double *rho, blasint *cutpnt, double *z, double *dlamda, double *q2, blasint *ldq2, double *w, blasint *perm, blasint *givptr, blasint *givcol, double *givnum, blasint *indxp, blasint *indx, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaed8((void *) icompq, (void *) k, (void *) n, (void *) qsiz, (void *) d, (void *) q, (void *) ldq, (void *) indxq, (void *) rho, (void *) cutpnt, (void *) z, (void *) dlamda, (void *) q2, (void *) ldq2, (void *) w, (void *) perm, (void *) givptr, (void *) givcol, (void *) givnum, (void *) indxp, (void *) indx, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaed8.timings[0] += (helpTimeStop - helpTime);
    data->dlaed8.calls[0]++;
    
    return;
}


/* Wrapper for dlaed9 */
void hook_dlaed9(blasint *k, blasint *kstart, blasint *kstop, blasint *n, double *d, double *q, blasint *ldq, double *rho, double *dlamda, double *w, double *s, blasint *lds, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaed9((void *) k, (void *) kstart, (void *) kstop, (void *) n, (void *) d, (void *) q, (void *) ldq, (void *) rho, (void *) dlamda, (void *) w, (void *) s, (void *) lds, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaed9.timings[0] += (helpTimeStop - helpTime);
    data->dlaed9.calls[0]++;
    
    return;
}


/* Wrapper for dlaeda */
void hook_dlaeda(blasint *n, blasint *tlvls, blasint *curlvl, blasint *curpbm, blasint *prmptr, blasint *perm, blasint *givptr, blasint *givcol, double *givnum, double *q, blasint *qptr, double *z, double *ztemp, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaeda((void *) n, (void *) tlvls, (void *) curlvl, (void *) curpbm, (void *) prmptr, (void *) perm, (void *) givptr, (void *) givcol, (void *) givnum, (void *) q, (void *) qptr, (void *) z, (void *) ztemp, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaeda.timings[0] += (helpTimeStop - helpTime);
    data->dlaeda.calls[0]++;
    
    return;
}


/* Wrapper for dlaein */
void hook_dlaein(blaslogical *rightv, blaslogical *noinit, blasint *n, double *h, blasint *ldh, double *wr, double *wi, double *vr, double *vi, double *b, blasint *ldb, double *work, double *eps3, double *smlnum, double *bignum, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaein((void *) rightv, (void *) noinit, (void *) n, (void *) h, (void *) ldh, (void *) wr, (void *) wi, (void *) vr, (void *) vi, (void *) b, (void *) ldb, (void *) work, (void *) eps3, (void *) smlnum, (void *) bignum, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaein.timings[0] += (helpTimeStop - helpTime);
    data->dlaein.calls[0]++;
    
    return;
}


/* Wrapper for dlaev2 */
void hook_dlaev2(double *a, double *b, double *c, double *rt1, double *rt2, double *cs1, double *sn1)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaev2((void *) a, (void *) b, (void *) c, (void *) rt1, (void *) rt2, (void *) cs1, (void *) sn1);

    helpTimeStop = flexiblas_wtime();

    data->dlaev2.timings[0] += (helpTimeStop - helpTime);
    data->dlaev2.calls[0]++;
    
    return;
}


/* Wrapper for dlaexc */
void hook_dlaexc(blaslogical *wantq, blasint *n, double *t, blasint *ldt, double *q, blasint *ldq, blasint *j1, blasint *n1, blasint *n2, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaexc((void *) wantq, (void *) n, (void *) t, (void *) ldt, (void *) q, (void *) ldq, (void *) j1, (void *) n1, (void *) n2, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaexc.timings[0] += (helpTimeStop - helpTime);
    data->dlaexc.calls[0]++;
    
    return;
}


/* Wrapper for dlag2 */
void hook_dlag2(double *a, blasint *lda, double *b, blasint *ldb, double *safmin, double *scale1, double *scale2, double *wr1, double *wr2, double *wi)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlag2((void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) safmin, (void *) scale1, (void *) scale2, (void *) wr1, (void *) wr2, (void *) wi);

    helpTimeStop = flexiblas_wtime();

    data->dlag2.timings[0] += (helpTimeStop - helpTime);
    data->dlag2.calls[0]++;
    
    return;
}


/* Wrapper for dlag2s */
void hook_dlag2s(blasint *m, blasint *n, double *a, blasint *lda, float *sa, blasint *ldsa, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlag2s((void *) m, (void *) n, (void *) a, (void *) lda, (void *) sa, (void *) ldsa, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlag2s.timings[0] += (helpTimeStop - helpTime);
    data->dlag2s.calls[0]++;
    
    return;
}


/* Wrapper for dlags2 */
void hook_dlags2(blaslogical *upper, double *a1, double *a2, double *a3, double *b1, double *b2, double *b3, double *csu, double *snu, double *csv, double *snv, double *csq, double *snq)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlags2((void *) upper, (void *) a1, (void *) a2, (void *) a3, (void *) b1, (void *) b2, (void *) b3, (void *) csu, (void *) snu, (void *) csv, (void *) snv, (void *) csq, (void *) snq);

    helpTimeStop = flexiblas_wtime();

    data->dlags2.timings[0] += (helpTimeStop - helpTime);
    data->dlags2.calls[0]++;
    
    return;
}


/* Wrapper for dlagtf */
void hook_dlagtf(blasint *n, double *a, double *lambda, double *b, double *c, double *tol, double *d, blasint *in, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlagtf((void *) n, (void *) a, (void *) lambda, (void *) b, (void *) c, (void *) tol, (void *) d, (void *) in, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlagtf.timings[0] += (helpTimeStop - helpTime);
    data->dlagtf.calls[0]++;
    
    return;
}


/* Wrapper for dlagtm */
void hook_dlagtm(char *trans, blasint *n, blasint *nrhs, double *alpha, double *dl, double *d, double *du, double *x, blasint *ldx, double *beta, double *b, blasint *ldb, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlagtm((void *) trans, (void *) n, (void *) nrhs, (void *) alpha, (void *) dl, (void *) d, (void *) du, (void *) x, (void *) ldx, (void *) beta, (void *) b, (void *) ldb, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dlagtm.timings[0] += (helpTimeStop - helpTime);
    data->dlagtm.calls[0]++;
    
    return;
}


/* Wrapper for dlagts */
void hook_dlagts(blasint *job, blasint *n, double *a, double *b, double *c, double *d, blasint *in, double *y, double *tol, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlagts((void *) job, (void *) n, (void *) a, (void *) b, (void *) c, (void *) d, (void *) in, (void *) y, (void *) tol, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlagts.timings[0] += (helpTimeStop - helpTime);
    data->dlagts.calls[0]++;
    
    return;
}


/* Wrapper for dlagv2 */
void hook_dlagv2(double *a, blasint *lda, double *b, blasint *ldb, double *alphar, double *alphai, double *beta, double *csl, double *snl, double *csr, double *snr)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlagv2((void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) alphar, (void *) alphai, (void *) beta, (void *) csl, (void *) snl, (void *) csr, (void *) snr);

    helpTimeStop = flexiblas_wtime();

    data->dlagv2.timings[0] += (helpTimeStop - helpTime);
    data->dlagv2.calls[0]++;
    
    return;
}


/* Wrapper for dlahqr */
void hook_dlahqr(blaslogical *wantt, blaslogical *wantz, blasint *n, blasint *ilo, blasint *ihi, double *h, blasint *ldh, double *wr, double *wi, blasint *iloz, blasint *ihiz, double *z, blasint *ldz, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlahqr((void *) wantt, (void *) wantz, (void *) n, (void *) ilo, (void *) ihi, (void *) h, (void *) ldh, (void *) wr, (void *) wi, (void *) iloz, (void *) ihiz, (void *) z, (void *) ldz, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlahqr.timings[0] += (helpTimeStop - helpTime);
    data->dlahqr.calls[0]++;
    
    return;
}


/* Wrapper for dlahr2 */
void hook_dlahr2(blasint *n, blasint *k, blasint *nb, double *a, blasint *lda, double *tau, double *t, blasint *ldt, double *y, blasint *ldy)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlahr2((void *) n, (void *) k, (void *) nb, (void *) a, (void *) lda, (void *) tau, (void *) t, (void *) ldt, (void *) y, (void *) ldy);

    helpTimeStop = flexiblas_wtime();

    data->dlahr2.timings[0] += (helpTimeStop - helpTime);
    data->dlahr2.calls[0]++;
    
    return;
}


/* Wrapper for dlaic1 */
void hook_dlaic1(blasint *job, blasint *j, double *x, double *sest, double *w, double *gamma, double *sestpr, double *s, double *c)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaic1((void *) job, (void *) j, (void *) x, (void *) sest, (void *) w, (void *) gamma, (void *) sestpr, (void *) s, (void *) c);

    helpTimeStop = flexiblas_wtime();

    data->dlaic1.timings[0] += (helpTimeStop - helpTime);
    data->dlaic1.calls[0]++;
    
    return;
}

/* Wrapper for dlaisnan */
int_least32_t hook_dlaisnan(double *din1, double *din2)
{
    double helpTime;
    double helpTimeStop;
    int_least32_t return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dlaisnan((void *) din1, (void *) din2);

    helpTimeStop = flexiblas_wtime();

    data->dlaisnan.timings[0] += (helpTimeStop - helpTime);
    data->dlaisnan.calls[0]++;
    
    return return_value;
}



/* Wrapper for dlaln2 */
void hook_dlaln2(blaslogical *ltrans, blasint *na, blasint *nw, double *smin, double *ca, double *a, blasint *lda, double *d1, double *d2, double *b, blasint *ldb, double *wr, double *wi, double *x, blasint *ldx, double *scale, double *xnorm, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaln2((void *) ltrans, (void *) na, (void *) nw, (void *) smin, (void *) ca, (void *) a, (void *) lda, (void *) d1, (void *) d2, (void *) b, (void *) ldb, (void *) wr, (void *) wi, (void *) x, (void *) ldx, (void *) scale, (void *) xnorm, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaln2.timings[0] += (helpTimeStop - helpTime);
    data->dlaln2.calls[0]++;
    
    return;
}


/* Wrapper for dlals0 */
void hook_dlals0(blasint *icompq, blasint *nl, blasint *nr, blasint *sqre, blasint *nrhs, double *b, blasint *ldb, double *bx, blasint *ldbx, blasint *perm, blasint *givptr, blasint *givcol, blasint *ldgcol, double *givnum, blasint *ldgnum, double *poles, double *difl, double *difr, double *z, blasint *k, double *c, double *s, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlals0((void *) icompq, (void *) nl, (void *) nr, (void *) sqre, (void *) nrhs, (void *) b, (void *) ldb, (void *) bx, (void *) ldbx, (void *) perm, (void *) givptr, (void *) givcol, (void *) ldgcol, (void *) givnum, (void *) ldgnum, (void *) poles, (void *) difl, (void *) difr, (void *) z, (void *) k, (void *) c, (void *) s, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlals0.timings[0] += (helpTimeStop - helpTime);
    data->dlals0.calls[0]++;
    
    return;
}


/* Wrapper for dlalsa */
void hook_dlalsa(blasint *icompq, blasint *smlsiz, blasint *n, blasint *nrhs, double *b, blasint *ldb, double *bx, blasint *ldbx, double *u, blasint *ldu, double *vt, blasint *k, double *difl, double *difr, double *z, double *poles, blasint *givptr, blasint *givcol, blasint *ldgcol, blasint *perm, double *givnum, double *c, double *s, double *work, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlalsa((void *) icompq, (void *) smlsiz, (void *) n, (void *) nrhs, (void *) b, (void *) ldb, (void *) bx, (void *) ldbx, (void *) u, (void *) ldu, (void *) vt, (void *) k, (void *) difl, (void *) difr, (void *) z, (void *) poles, (void *) givptr, (void *) givcol, (void *) ldgcol, (void *) perm, (void *) givnum, (void *) c, (void *) s, (void *) work, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlalsa.timings[0] += (helpTimeStop - helpTime);
    data->dlalsa.calls[0]++;
    
    return;
}


/* Wrapper for dlalsd */
void hook_dlalsd(char *uplo, blasint *smlsiz, blasint *n, blasint *nrhs, double *d, double *e, double *b, blasint *ldb, double *rcond, blasint *rank, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlalsd((void *) uplo, (void *) smlsiz, (void *) n, (void *) nrhs, (void *) d, (void *) e, (void *) b, (void *) ldb, (void *) rcond, (void *) rank, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dlalsd.timings[0] += (helpTimeStop - helpTime);
    data->dlalsd.calls[0]++;
    
    return;
}

/* Wrapper for dlamc3 */
double hook_dlamc3(double *a, double *b)
{
    double helpTime;
    double helpTimeStop;
    double return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dlamc3((void *) a, (void *) b);

    helpTimeStop = flexiblas_wtime();

    data->dlamc3.timings[0] += (helpTimeStop - helpTime);
    data->dlamc3.calls[0]++;
    
    return return_value;
}


/* Wrapper for dlamch */
double hook_dlamch(char *cmach, flexiblas_fortran_charlen_t cmach_len)
{
    double helpTime;
    double helpTimeStop;
    double return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dlamch((void *) cmach, cmach_len);

    helpTimeStop = flexiblas_wtime();

    data->dlamch.timings[0] += (helpTimeStop - helpTime);
    data->dlamch.calls[0]++;
    
    return return_value;
}



/* Wrapper for dlamrg */
void hook_dlamrg(blasint *n1, blasint *n2, double *a, blasint *dtrd1, blasint *dtrd2, blasint *index)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlamrg((void *) n1, (void *) n2, (void *) a, (void *) dtrd1, (void *) dtrd2, (void *) index);

    helpTimeStop = flexiblas_wtime();

    data->dlamrg.timings[0] += (helpTimeStop - helpTime);
    data->dlamrg.calls[0]++;
    
    return;
}


/* Wrapper for dlamswlq */
void hook_dlamswlq(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *mb, blasint *nb, double *a, blasint *lda, double *t, blasint *ldt, double *c, blasint *ldc, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlamswlq((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) mb, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dlamswlq.timings[0] += (helpTimeStop - helpTime);
    data->dlamswlq.calls[0]++;
    
    return;
}


/* Wrapper for dlamtsqr */
void hook_dlamtsqr(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *mb, blasint *nb, double *a, blasint *lda, double *t, blasint *ldt, double *c, blasint *ldc, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlamtsqr((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) mb, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dlamtsqr.timings[0] += (helpTimeStop - helpTime);
    data->dlamtsqr.calls[0]++;
    
    return;
}

/* Wrapper for dlaneg */
blasint hook_dlaneg(blasint *n, double *d, double *lld, double *sigma, double *pivmin, blasint *r)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dlaneg((void *) n, (void *) d, (void *) lld, (void *) sigma, (void *) pivmin, (void *) r);

    helpTimeStop = flexiblas_wtime();

    data->dlaneg.timings[0] += (helpTimeStop - helpTime);
    data->dlaneg.calls[0]++;
    
    return return_value;
}


/* Wrapper for dlangb */
double hook_dlangb(char *norm, blasint *n, blasint *kl, blasint *ku, double *ab, blasint *ldab, double *work, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;
    double return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dlangb((void *) norm, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) work, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->dlangb.timings[0] += (helpTimeStop - helpTime);
    data->dlangb.calls[0]++;
    
    return return_value;
}


/* Wrapper for dlange */
double hook_dlange(char *norm, blasint *m, blasint *n, double *a, blasint *lda, double *work, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;
    double return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dlange((void *) norm, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) work, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->dlange.timings[0] += (helpTimeStop - helpTime);
    data->dlange.calls[0]++;
    
    return return_value;
}


/* Wrapper for dlangt */
double hook_dlangt(char *norm, blasint *n, double *dl, double *d, double *du, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;
    double return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dlangt((void *) norm, (void *) n, (void *) dl, (void *) d, (void *) du, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->dlangt.timings[0] += (helpTimeStop - helpTime);
    data->dlangt.calls[0]++;
    
    return return_value;
}


/* Wrapper for dlanhs */
double hook_dlanhs(char *norm, blasint *n, double *a, blasint *lda, double *work, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;
    double return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dlanhs((void *) norm, (void *) n, (void *) a, (void *) lda, (void *) work, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->dlanhs.timings[0] += (helpTimeStop - helpTime);
    data->dlanhs.calls[0]++;
    
    return return_value;
}


/* Wrapper for dlansb */
double hook_dlansb(char *norm, char *uplo, blasint *n, blasint *k, double *ab, blasint *ldab, double *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;
    double return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dlansb((void *) norm, (void *) uplo, (void *) n, (void *) k, (void *) ab, (void *) ldab, (void *) work, norm_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dlansb.timings[0] += (helpTimeStop - helpTime);
    data->dlansb.calls[0]++;
    
    return return_value;
}


/* Wrapper for dlansf */
double hook_dlansf(char *norm, char *transr, char *uplo, blasint *n, double *a, double *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;
    double return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dlansf((void *) norm, (void *) transr, (void *) uplo, (void *) n, (void *) a, (void *) work, norm_len, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dlansf.timings[0] += (helpTimeStop - helpTime);
    data->dlansf.calls[0]++;
    
    return return_value;
}


/* Wrapper for dlansp */
double hook_dlansp(char *norm, char *uplo, blasint *n, double *ap, double *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;
    double return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dlansp((void *) norm, (void *) uplo, (void *) n, (void *) ap, (void *) work, norm_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dlansp.timings[0] += (helpTimeStop - helpTime);
    data->dlansp.calls[0]++;
    
    return return_value;
}


/* Wrapper for dlanst */
double hook_dlanst(char *norm, blasint *n, double *d, double *e, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;
    double return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dlanst((void *) norm, (void *) n, (void *) d, (void *) e, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->dlanst.timings[0] += (helpTimeStop - helpTime);
    data->dlanst.calls[0]++;
    
    return return_value;
}


/* Wrapper for dlansy */
double hook_dlansy(char *norm, char *uplo, blasint *n, double *a, blasint *lda, double *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;
    double return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dlansy((void *) norm, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) work, norm_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dlansy.timings[0] += (helpTimeStop - helpTime);
    data->dlansy.calls[0]++;
    
    return return_value;
}


/* Wrapper for dlantb */
double hook_dlantb(char *norm, char *uplo, char *diag, blasint *n, blasint *k, double *ab, blasint *ldab, double *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;
    double return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dlantb((void *) norm, (void *) uplo, (void *) diag, (void *) n, (void *) k, (void *) ab, (void *) ldab, (void *) work, norm_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->dlantb.timings[0] += (helpTimeStop - helpTime);
    data->dlantb.calls[0]++;
    
    return return_value;
}


/* Wrapper for dlantp */
double hook_dlantp(char *norm, char *uplo, char *diag, blasint *n, double *ap, double *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;
    double return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dlantp((void *) norm, (void *) uplo, (void *) diag, (void *) n, (void *) ap, (void *) work, norm_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->dlantp.timings[0] += (helpTimeStop - helpTime);
    data->dlantp.calls[0]++;
    
    return return_value;
}


/* Wrapper for dlantr */
double hook_dlantr(char *norm, char *uplo, char *diag, blasint *m, blasint *n, double *a, blasint *lda, double *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;
    double return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dlantr((void *) norm, (void *) uplo, (void *) diag, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) work, norm_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->dlantr.timings[0] += (helpTimeStop - helpTime);
    data->dlantr.calls[0]++;
    
    return return_value;
}



/* Wrapper for dlanv2 */
void hook_dlanv2(double *a, double *b, double *c, double *d, double *rt1r, double *rt1i, double *rt2r, double *rt2i, double *cs, double *sn)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlanv2((void *) a, (void *) b, (void *) c, (void *) d, (void *) rt1r, (void *) rt1i, (void *) rt2r, (void *) rt2i, (void *) cs, (void *) sn);

    helpTimeStop = flexiblas_wtime();

    data->dlanv2.timings[0] += (helpTimeStop - helpTime);
    data->dlanv2.calls[0]++;
    
    return;
}


/* Wrapper for dlaorhr_col_getrfnp */
void hook_dlaorhr_col_getrfnp(blasint *m, blasint *n, double *a, blasint *lda, double *d, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaorhr_col_getrfnp((void *) m, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaorhr_col_getrfnp.timings[0] += (helpTimeStop - helpTime);
    data->dlaorhr_col_getrfnp.calls[0]++;
    
    return;
}


/* Wrapper for dlaorhr_col_getrfnp2 */
void hook_dlaorhr_col_getrfnp2(blasint *m, blasint *n, double *a, blasint *lda, double *d, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaorhr_col_getrfnp2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaorhr_col_getrfnp2.timings[0] += (helpTimeStop - helpTime);
    data->dlaorhr_col_getrfnp2.calls[0]++;
    
    return;
}


/* Wrapper for dlapll */
void hook_dlapll(blasint *n, double *x, blasint *incx, double *y, blasint *incy, double *ssmin)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlapll((void *) n, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ssmin);

    helpTimeStop = flexiblas_wtime();

    data->dlapll.timings[0] += (helpTimeStop - helpTime);
    data->dlapll.calls[0]++;
    
    return;
}


/* Wrapper for dlapmr */
void hook_dlapmr(blaslogical *forwrd, blasint *m, blasint *n, double *x, blasint *ldx, blasint *k)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlapmr((void *) forwrd, (void *) m, (void *) n, (void *) x, (void *) ldx, (void *) k);

    helpTimeStop = flexiblas_wtime();

    data->dlapmr.timings[0] += (helpTimeStop - helpTime);
    data->dlapmr.calls[0]++;
    
    return;
}


/* Wrapper for dlapmt */
void hook_dlapmt(blaslogical *forwrd, blasint *m, blasint *n, double *x, blasint *ldx, blasint *k)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlapmt((void *) forwrd, (void *) m, (void *) n, (void *) x, (void *) ldx, (void *) k);

    helpTimeStop = flexiblas_wtime();

    data->dlapmt.timings[0] += (helpTimeStop - helpTime);
    data->dlapmt.calls[0]++;
    
    return;
}

/* Wrapper for dlapy2 */
double hook_dlapy2(double *x, double *y)
{
    double helpTime;
    double helpTimeStop;
    double return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dlapy2((void *) x, (void *) y);

    helpTimeStop = flexiblas_wtime();

    data->dlapy2.timings[0] += (helpTimeStop - helpTime);
    data->dlapy2.calls[0]++;
    
    return return_value;
}


/* Wrapper for dlapy3 */
double hook_dlapy3(double *x, double *y, double *z)
{
    double helpTime;
    double helpTimeStop;
    double return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dlapy3((void *) x, (void *) y, (void *) z);

    helpTimeStop = flexiblas_wtime();

    data->dlapy3.timings[0] += (helpTimeStop - helpTime);
    data->dlapy3.calls[0]++;
    
    return return_value;
}



/* Wrapper for dlaqgb */
void hook_dlaqgb(blasint *m, blasint *n, blasint *kl, blasint *ku, double *ab, blasint *ldab, double *r, double *c, double *rowcnd, double *colcnd, double *amax, char *equed, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaqgb((void *) m, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) equed, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->dlaqgb.timings[0] += (helpTimeStop - helpTime);
    data->dlaqgb.calls[0]++;
    
    return;
}


/* Wrapper for dlaqge */
void hook_dlaqge(blasint *m, blasint *n, double *a, blasint *lda, double *r, double *c, double *rowcnd, double *colcnd, double *amax, char *equed, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaqge((void *) m, (void *) n, (void *) a, (void *) lda, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) equed, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->dlaqge.timings[0] += (helpTimeStop - helpTime);
    data->dlaqge.calls[0]++;
    
    return;
}


/* Wrapper for dlaqp2 */
void hook_dlaqp2(blasint *m, blasint *n, blasint *offset, double *a, blasint *lda, blasint *jpvt, double *tau, double *vn1, double *vn2, double *work)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaqp2((void *) m, (void *) n, (void *) offset, (void *) a, (void *) lda, (void *) jpvt, (void *) tau, (void *) vn1, (void *) vn2, (void *) work);

    helpTimeStop = flexiblas_wtime();

    data->dlaqp2.timings[0] += (helpTimeStop - helpTime);
    data->dlaqp2.calls[0]++;
    
    return;
}


/* Wrapper for dlaqps */
void hook_dlaqps(blasint *m, blasint *n, blasint *offset, blasint *nb, blasint *kb, double *a, blasint *lda, blasint *jpvt, double *tau, double *vn1, double *vn2, double *auxv, double *f, blasint *ldf)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaqps((void *) m, (void *) n, (void *) offset, (void *) nb, (void *) kb, (void *) a, (void *) lda, (void *) jpvt, (void *) tau, (void *) vn1, (void *) vn2, (void *) auxv, (void *) f, (void *) ldf);

    helpTimeStop = flexiblas_wtime();

    data->dlaqps.timings[0] += (helpTimeStop - helpTime);
    data->dlaqps.calls[0]++;
    
    return;
}


/* Wrapper for dlaqr0 */
void hook_dlaqr0(blaslogical *wantt, blaslogical *wantz, blasint *n, blasint *ilo, blasint *ihi, double *h, blasint *ldh, double *wr, double *wi, blasint *iloz, blasint *ihiz, double *z, blasint *ldz, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaqr0((void *) wantt, (void *) wantz, (void *) n, (void *) ilo, (void *) ihi, (void *) h, (void *) ldh, (void *) wr, (void *) wi, (void *) iloz, (void *) ihiz, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaqr0.timings[0] += (helpTimeStop - helpTime);
    data->dlaqr0.calls[0]++;
    
    return;
}


/* Wrapper for dlaqr1 */
void hook_dlaqr1(blasint *n, double *h, blasint *ldh, double *sr1, double *si1, double *sr2, double *si2, double *v)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaqr1((void *) n, (void *) h, (void *) ldh, (void *) sr1, (void *) si1, (void *) sr2, (void *) si2, (void *) v);

    helpTimeStop = flexiblas_wtime();

    data->dlaqr1.timings[0] += (helpTimeStop - helpTime);
    data->dlaqr1.calls[0]++;
    
    return;
}


/* Wrapper for dlaqr2 */
void hook_dlaqr2(blaslogical *wantt, blaslogical *wantz, blasint *n, blasint *ktop, blasint *kbot, blasint *nw, double *h, blasint *ldh, blasint *iloz, blasint *ihiz, double *z, blasint *ldz, blasint *ns, blasint *nd, double *sr, double *si, double *v, blasint *ldv, blasint *nh, double *t, blasint *ldt, blasint *nv, double *wv, blasint *ldwv, double *work, blasint *lwork)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaqr2((void *) wantt, (void *) wantz, (void *) n, (void *) ktop, (void *) kbot, (void *) nw, (void *) h, (void *) ldh, (void *) iloz, (void *) ihiz, (void *) z, (void *) ldz, (void *) ns, (void *) nd, (void *) sr, (void *) si, (void *) v, (void *) ldv, (void *) nh, (void *) t, (void *) ldt, (void *) nv, (void *) wv, (void *) ldwv, (void *) work, (void *) lwork);

    helpTimeStop = flexiblas_wtime();

    data->dlaqr2.timings[0] += (helpTimeStop - helpTime);
    data->dlaqr2.calls[0]++;
    
    return;
}


/* Wrapper for dlaqr3 */
void hook_dlaqr3(blaslogical *wantt, blaslogical *wantz, blasint *n, blasint *ktop, blasint *kbot, blasint *nw, double *h, blasint *ldh, blasint *iloz, blasint *ihiz, double *z, blasint *ldz, blasint *ns, blasint *nd, double *sr, double *si, double *v, blasint *ldv, blasint *nh, double *t, blasint *ldt, blasint *nv, double *wv, blasint *ldwv, double *work, blasint *lwork)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaqr3((void *) wantt, (void *) wantz, (void *) n, (void *) ktop, (void *) kbot, (void *) nw, (void *) h, (void *) ldh, (void *) iloz, (void *) ihiz, (void *) z, (void *) ldz, (void *) ns, (void *) nd, (void *) sr, (void *) si, (void *) v, (void *) ldv, (void *) nh, (void *) t, (void *) ldt, (void *) nv, (void *) wv, (void *) ldwv, (void *) work, (void *) lwork);

    helpTimeStop = flexiblas_wtime();

    data->dlaqr3.timings[0] += (helpTimeStop - helpTime);
    data->dlaqr3.calls[0]++;
    
    return;
}


/* Wrapper for dlaqr4 */
void hook_dlaqr4(blaslogical *wantt, blaslogical *wantz, blasint *n, blasint *ilo, blasint *ihi, double *h, blasint *ldh, double *wr, double *wi, blasint *iloz, blasint *ihiz, double *z, blasint *ldz, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaqr4((void *) wantt, (void *) wantz, (void *) n, (void *) ilo, (void *) ihi, (void *) h, (void *) ldh, (void *) wr, (void *) wi, (void *) iloz, (void *) ihiz, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaqr4.timings[0] += (helpTimeStop - helpTime);
    data->dlaqr4.calls[0]++;
    
    return;
}


/* Wrapper for dlaqr5 */
void hook_dlaqr5(blaslogical *wantt, blaslogical *wantz, blasint *kacc22, blasint *n, blasint *ktop, blasint *kbot, blasint *nshfts, double *sr, double *si, double *h, blasint *ldh, blasint *iloz, blasint *ihiz, double *z, blasint *ldz, double *v, blasint *ldv, double *u, blasint *ldu, blasint *nv, double *wv, blasint *ldwv, blasint *nh, double *wh, blasint *ldwh)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaqr5((void *) wantt, (void *) wantz, (void *) kacc22, (void *) n, (void *) ktop, (void *) kbot, (void *) nshfts, (void *) sr, (void *) si, (void *) h, (void *) ldh, (void *) iloz, (void *) ihiz, (void *) z, (void *) ldz, (void *) v, (void *) ldv, (void *) u, (void *) ldu, (void *) nv, (void *) wv, (void *) ldwv, (void *) nh, (void *) wh, (void *) ldwh);

    helpTimeStop = flexiblas_wtime();

    data->dlaqr5.timings[0] += (helpTimeStop - helpTime);
    data->dlaqr5.calls[0]++;
    
    return;
}


/* Wrapper for dlaqsb */
void hook_dlaqsb(char *uplo, blasint *n, blasint *kd, double *ab, blasint *ldab, double *s, double *scond, double *amax, char *equed, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaqsb((void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) s, (void *) scond, (void *) amax, (void *) equed, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->dlaqsb.timings[0] += (helpTimeStop - helpTime);
    data->dlaqsb.calls[0]++;
    
    return;
}


/* Wrapper for dlaqsp */
void hook_dlaqsp(char *uplo, blasint *n, double *ap, double *s, double *scond, double *amax, char *equed, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaqsp((void *) uplo, (void *) n, (void *) ap, (void *) s, (void *) scond, (void *) amax, (void *) equed, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->dlaqsp.timings[0] += (helpTimeStop - helpTime);
    data->dlaqsp.calls[0]++;
    
    return;
}


/* Wrapper for dlaqsy */
void hook_dlaqsy(char *uplo, blasint *n, double *a, blasint *lda, double *s, double *scond, double *amax, char *equed, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaqsy((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) s, (void *) scond, (void *) amax, (void *) equed, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->dlaqsy.timings[0] += (helpTimeStop - helpTime);
    data->dlaqsy.calls[0]++;
    
    return;
}


/* Wrapper for dlaqtr */
void hook_dlaqtr(blaslogical *ltran, blaslogical *lreal, blasint *n, double *t, blasint *ldt, double *b, double *w, double *scale, double *x, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaqtr((void *) ltran, (void *) lreal, (void *) n, (void *) t, (void *) ldt, (void *) b, (void *) w, (void *) scale, (void *) x, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaqtr.timings[0] += (helpTimeStop - helpTime);
    data->dlaqtr.calls[0]++;
    
    return;
}


/* Wrapper for dlar1v */
void hook_dlar1v(blasint *n, blasint *b1, blasint *bn, double *lambda, double *d, double *l, double *ld, double *lld, double *pivmin, double *gaptol, double *z, blaslogical *wantnc, blasint *negcnt, double *ztz, double *mingma, blasint *r, blasint *isuppz, double *nrminv, double *resid, double *rqcorr, double *work)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlar1v((void *) n, (void *) b1, (void *) bn, (void *) lambda, (void *) d, (void *) l, (void *) ld, (void *) lld, (void *) pivmin, (void *) gaptol, (void *) z, (void *) wantnc, (void *) negcnt, (void *) ztz, (void *) mingma, (void *) r, (void *) isuppz, (void *) nrminv, (void *) resid, (void *) rqcorr, (void *) work);

    helpTimeStop = flexiblas_wtime();

    data->dlar1v.timings[0] += (helpTimeStop - helpTime);
    data->dlar1v.calls[0]++;
    
    return;
}


/* Wrapper for dlar2v */
void hook_dlar2v(blasint *n, double *x, double *y, double *z, blasint *incx, double *c, double *s, blasint *incc)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlar2v((void *) n, (void *) x, (void *) y, (void *) z, (void *) incx, (void *) c, (void *) s, (void *) incc);

    helpTimeStop = flexiblas_wtime();

    data->dlar2v.timings[0] += (helpTimeStop - helpTime);
    data->dlar2v.calls[0]++;
    
    return;
}


/* Wrapper for dlarf */
void hook_dlarf(char *side, blasint *m, blasint *n, double *v, blasint *incv, double *tau, double *c, blasint *ldc, double *work, flexiblas_fortran_charlen_t side_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarf((void *) side, (void *) m, (void *) n, (void *) v, (void *) incv, (void *) tau, (void *) c, (void *) ldc, (void *) work, side_len);

    helpTimeStop = flexiblas_wtime();

    data->dlarf.timings[0] += (helpTimeStop - helpTime);
    data->dlarf.calls[0]++;
    
    return;
}


/* Wrapper for dlarfb */
void hook_dlarfb(char *side, char *trans, char *direct, char *storev, blasint *m, blasint *n, blasint *k, double *v, blasint *ldv, double *t, blasint *ldt, double *c, blasint *ldc, double *work, blasint *ldwork, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t direct_len, flexiblas_fortran_charlen_t storev_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarfb((void *) side, (void *) trans, (void *) direct, (void *) storev, (void *) m, (void *) n, (void *) k, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) c, (void *) ldc, (void *) work, (void *) ldwork, side_len, trans_len, direct_len, storev_len);

    helpTimeStop = flexiblas_wtime();

    data->dlarfb.timings[0] += (helpTimeStop - helpTime);
    data->dlarfb.calls[0]++;
    
    return;
}


/* Wrapper for dlarfb_gett */
void hook_dlarfb_gett(char *ident, blasint *m, blasint *n, blasint *k, double *t, blasint *ldt, double *a, blasint *lda, double *b, blasint *ldb, double *work, blasint *ldwork, flexiblas_fortran_charlen_t ident_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarfb_gett((void *) ident, (void *) m, (void *) n, (void *) k, (void *) t, (void *) ldt, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) work, (void *) ldwork, ident_len);

    helpTimeStop = flexiblas_wtime();

    data->dlarfb_gett.timings[0] += (helpTimeStop - helpTime);
    data->dlarfb_gett.calls[0]++;
    
    return;
}


/* Wrapper for dlarfg */
void hook_dlarfg(blasint *n, double *alpha, double *x, blasint *incx, double *tau)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarfg((void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) tau);

    helpTimeStop = flexiblas_wtime();

    data->dlarfg.timings[0] += (helpTimeStop - helpTime);
    data->dlarfg.calls[0]++;
    
    return;
}


/* Wrapper for dlarfgp */
void hook_dlarfgp(blasint *n, double *alpha, double *x, blasint *incx, double *tau)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarfgp((void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) tau);

    helpTimeStop = flexiblas_wtime();

    data->dlarfgp.timings[0] += (helpTimeStop - helpTime);
    data->dlarfgp.calls[0]++;
    
    return;
}


/* Wrapper for dlarft */
void hook_dlarft(char *direct, char *storev, blasint *n, blasint *k, double *v, blasint *ldv, double *tau, double *t, blasint *ldt, flexiblas_fortran_charlen_t direct_len, flexiblas_fortran_charlen_t storev_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarft((void *) direct, (void *) storev, (void *) n, (void *) k, (void *) v, (void *) ldv, (void *) tau, (void *) t, (void *) ldt, direct_len, storev_len);

    helpTimeStop = flexiblas_wtime();

    data->dlarft.timings[0] += (helpTimeStop - helpTime);
    data->dlarft.calls[0]++;
    
    return;
}


/* Wrapper for dlarfx */
void hook_dlarfx(char *side, blasint *m, blasint *n, double *v, double *tau, double *c, blasint *ldc, double *work, flexiblas_fortran_charlen_t side_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarfx((void *) side, (void *) m, (void *) n, (void *) v, (void *) tau, (void *) c, (void *) ldc, (void *) work, side_len);

    helpTimeStop = flexiblas_wtime();

    data->dlarfx.timings[0] += (helpTimeStop - helpTime);
    data->dlarfx.calls[0]++;
    
    return;
}


/* Wrapper for dlarfy */
void hook_dlarfy(char *uplo, blasint *n, double *v, blasint *incv, double *tau, double *c, blasint *ldc, double *work, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarfy((void *) uplo, (void *) n, (void *) v, (void *) incv, (void *) tau, (void *) c, (void *) ldc, (void *) work, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dlarfy.timings[0] += (helpTimeStop - helpTime);
    data->dlarfy.calls[0]++;
    
    return;
}


/* Wrapper for dlargv */
void hook_dlargv(blasint *n, double *x, blasint *incx, double *y, blasint *incy, double *c, blasint *incc)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlargv((void *) n, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) c, (void *) incc);

    helpTimeStop = flexiblas_wtime();

    data->dlargv.timings[0] += (helpTimeStop - helpTime);
    data->dlargv.calls[0]++;
    
    return;
}


/* Wrapper for dlarnv */
void hook_dlarnv(blasint *idist, blasint *iseed, blasint *n, double *x)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarnv((void *) idist, (void *) iseed, (void *) n, (void *) x);

    helpTimeStop = flexiblas_wtime();

    data->dlarnv.timings[0] += (helpTimeStop - helpTime);
    data->dlarnv.calls[0]++;
    
    return;
}


/* Wrapper for dlarra */
void hook_dlarra(blasint *n, double *d, double *e, double *e2, double *spltol, double *tnrm, blasint *nsplit, blasint *isplit, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarra((void *) n, (void *) d, (void *) e, (void *) e2, (void *) spltol, (void *) tnrm, (void *) nsplit, (void *) isplit, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlarra.timings[0] += (helpTimeStop - helpTime);
    data->dlarra.calls[0]++;
    
    return;
}


/* Wrapper for dlarrb */
void hook_dlarrb(blasint *n, double *d, double *lld, blasint *ifirst, blasint *ilast, double *rtol1, double *rtol2, blasint *offset, double *w, double *wgap, double *werr, double *work, blasint *iwork, double *pivmin, double *spdiam, blasint *twist, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarrb((void *) n, (void *) d, (void *) lld, (void *) ifirst, (void *) ilast, (void *) rtol1, (void *) rtol2, (void *) offset, (void *) w, (void *) wgap, (void *) werr, (void *) work, (void *) iwork, (void *) pivmin, (void *) spdiam, (void *) twist, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlarrb.timings[0] += (helpTimeStop - helpTime);
    data->dlarrb.calls[0]++;
    
    return;
}


/* Wrapper for dlarrc */
void hook_dlarrc(char *jobt, blasint *n, double *vl, double *vu, double *d, double *e, double *pivmin, blasint *eigcnt, blasint *lcnt, blasint *rcnt, blasint *info, flexiblas_fortran_charlen_t jobt_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarrc((void *) jobt, (void *) n, (void *) vl, (void *) vu, (void *) d, (void *) e, (void *) pivmin, (void *) eigcnt, (void *) lcnt, (void *) rcnt, (void *) info, jobt_len);

    helpTimeStop = flexiblas_wtime();

    data->dlarrc.timings[0] += (helpTimeStop - helpTime);
    data->dlarrc.calls[0]++;
    
    return;
}


/* Wrapper for dlarrd */
void hook_dlarrd(char *range, char *order, blasint *n, double *vl, double *vu, blasint *il, blasint *iu, double *gers, double *reltol, double *d, double *e, double *e2, double *pivmin, blasint *nsplit, blasint *isplit, blasint *m, double *w, double *werr, double *wl, double *wu, blasint *iblock, blasint *indexw, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t order_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarrd((void *) range, (void *) order, (void *) n, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) gers, (void *) reltol, (void *) d, (void *) e, (void *) e2, (void *) pivmin, (void *) nsplit, (void *) isplit, (void *) m, (void *) w, (void *) werr, (void *) wl, (void *) wu, (void *) iblock, (void *) indexw, (void *) work, (void *) iwork, (void *) info, range_len, order_len);

    helpTimeStop = flexiblas_wtime();

    data->dlarrd.timings[0] += (helpTimeStop - helpTime);
    data->dlarrd.calls[0]++;
    
    return;
}


/* Wrapper for dlarre */
void hook_dlarre(char *range, blasint *n, double *vl, double *vu, blasint *il, blasint *iu, double *d, double *e, double *e2, double *rtol1, double *rtol2, double *spltol, blasint *nsplit, blasint *isplit, blasint *m, double *w, double *werr, double *wgap, blasint *iblock, blasint *indexw, double *gers, double *pivmin, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t range_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarre((void *) range, (void *) n, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) d, (void *) e, (void *) e2, (void *) rtol1, (void *) rtol2, (void *) spltol, (void *) nsplit, (void *) isplit, (void *) m, (void *) w, (void *) werr, (void *) wgap, (void *) iblock, (void *) indexw, (void *) gers, (void *) pivmin, (void *) work, (void *) iwork, (void *) info, range_len);

    helpTimeStop = flexiblas_wtime();

    data->dlarre.timings[0] += (helpTimeStop - helpTime);
    data->dlarre.calls[0]++;
    
    return;
}


/* Wrapper for dlarrf */
void hook_dlarrf(blasint *n, double *d, double *l, double *ld, blasint *clstrt, blasint *clend, double *w, double *wgap, double *werr, double *spdiam, double *clgapl, double *clgapr, double *pivmin, double *sigma, double *dplus, double *lplus, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarrf((void *) n, (void *) d, (void *) l, (void *) ld, (void *) clstrt, (void *) clend, (void *) w, (void *) wgap, (void *) werr, (void *) spdiam, (void *) clgapl, (void *) clgapr, (void *) pivmin, (void *) sigma, (void *) dplus, (void *) lplus, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlarrf.timings[0] += (helpTimeStop - helpTime);
    data->dlarrf.calls[0]++;
    
    return;
}


/* Wrapper for dlarrj */
void hook_dlarrj(blasint *n, double *d, double *e2, blasint *ifirst, blasint *ilast, double *rtol, blasint *offset, double *w, double *werr, double *work, blasint *iwork, double *pivmin, double *spdiam, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarrj((void *) n, (void *) d, (void *) e2, (void *) ifirst, (void *) ilast, (void *) rtol, (void *) offset, (void *) w, (void *) werr, (void *) work, (void *) iwork, (void *) pivmin, (void *) spdiam, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlarrj.timings[0] += (helpTimeStop - helpTime);
    data->dlarrj.calls[0]++;
    
    return;
}


/* Wrapper for dlarrk */
void hook_dlarrk(blasint *n, blasint *iw, double *gl, double *gu, double *d, double *e2, double *pivmin, double *reltol, double *w, double *werr, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarrk((void *) n, (void *) iw, (void *) gl, (void *) gu, (void *) d, (void *) e2, (void *) pivmin, (void *) reltol, (void *) w, (void *) werr, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlarrk.timings[0] += (helpTimeStop - helpTime);
    data->dlarrk.calls[0]++;
    
    return;
}


/* Wrapper for dlarrr */
void hook_dlarrr(blasint *n, double *d, double *e, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarrr((void *) n, (void *) d, (void *) e, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlarrr.timings[0] += (helpTimeStop - helpTime);
    data->dlarrr.calls[0]++;
    
    return;
}


/* Wrapper for dlarrv */
void hook_dlarrv(blasint *n, double *vl, double *vu, double *d, double *l, double *pivmin, blasint *isplit, blasint *m, blasint *dol, blasint *dou, double *minrgp, double *rtol1, double *rtol2, double *w, double *werr, double *wgap, blasint *iblock, blasint *indexw, double *gers, double *z, blasint *ldz, blasint *isuppz, double *work, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarrv((void *) n, (void *) vl, (void *) vu, (void *) d, (void *) l, (void *) pivmin, (void *) isplit, (void *) m, (void *) dol, (void *) dou, (void *) minrgp, (void *) rtol1, (void *) rtol2, (void *) w, (void *) werr, (void *) wgap, (void *) iblock, (void *) indexw, (void *) gers, (void *) z, (void *) ldz, (void *) isuppz, (void *) work, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlarrv.timings[0] += (helpTimeStop - helpTime);
    data->dlarrv.calls[0]++;
    
    return;
}


/* Wrapper for dlartg */
void hook_dlartg(double *f, double *g, double *cs, double *sn, double *r)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlartg((void *) f, (void *) g, (void *) cs, (void *) sn, (void *) r);

    helpTimeStop = flexiblas_wtime();

    data->dlartg.timings[0] += (helpTimeStop - helpTime);
    data->dlartg.calls[0]++;
    
    return;
}


/* Wrapper for dlartgp */
void hook_dlartgp(double *f, double *g, double *cs, double *sn, double *r)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlartgp((void *) f, (void *) g, (void *) cs, (void *) sn, (void *) r);

    helpTimeStop = flexiblas_wtime();

    data->dlartgp.timings[0] += (helpTimeStop - helpTime);
    data->dlartgp.calls[0]++;
    
    return;
}


/* Wrapper for dlartgs */
void hook_dlartgs(double *x, double *y, double *sigma, double *cs, double *sn)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlartgs((void *) x, (void *) y, (void *) sigma, (void *) cs, (void *) sn);

    helpTimeStop = flexiblas_wtime();

    data->dlartgs.timings[0] += (helpTimeStop - helpTime);
    data->dlartgs.calls[0]++;
    
    return;
}


/* Wrapper for dlartv */
void hook_dlartv(blasint *n, double *x, blasint *incx, double *y, blasint *incy, double *c, double *s, blasint *incc)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlartv((void *) n, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) c, (void *) s, (void *) incc);

    helpTimeStop = flexiblas_wtime();

    data->dlartv.timings[0] += (helpTimeStop - helpTime);
    data->dlartv.calls[0]++;
    
    return;
}


/* Wrapper for dlaruv */
void hook_dlaruv(blasint *iseed, blasint *n, double *x)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaruv((void *) iseed, (void *) n, (void *) x);

    helpTimeStop = flexiblas_wtime();

    data->dlaruv.timings[0] += (helpTimeStop - helpTime);
    data->dlaruv.calls[0]++;
    
    return;
}


/* Wrapper for dlarz */
void hook_dlarz(char *side, blasint *m, blasint *n, blasint *l, double *v, blasint *incv, double *tau, double *c, blasint *ldc, double *work, flexiblas_fortran_charlen_t side_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarz((void *) side, (void *) m, (void *) n, (void *) l, (void *) v, (void *) incv, (void *) tau, (void *) c, (void *) ldc, (void *) work, side_len);

    helpTimeStop = flexiblas_wtime();

    data->dlarz.timings[0] += (helpTimeStop - helpTime);
    data->dlarz.calls[0]++;
    
    return;
}


/* Wrapper for dlarzb */
void hook_dlarzb(char *side, char *trans, char *direct, char *storev, blasint *m, blasint *n, blasint *k, blasint *l, double *v, blasint *ldv, double *t, blasint *ldt, double *c, blasint *ldc, double *work, blasint *ldwork, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t direct_len, flexiblas_fortran_charlen_t storev_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarzb((void *) side, (void *) trans, (void *) direct, (void *) storev, (void *) m, (void *) n, (void *) k, (void *) l, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) c, (void *) ldc, (void *) work, (void *) ldwork, side_len, trans_len, direct_len, storev_len);

    helpTimeStop = flexiblas_wtime();

    data->dlarzb.timings[0] += (helpTimeStop - helpTime);
    data->dlarzb.calls[0]++;
    
    return;
}


/* Wrapper for dlarzt */
void hook_dlarzt(char *direct, char *storev, blasint *n, blasint *k, double *v, blasint *ldv, double *tau, double *t, blasint *ldt, flexiblas_fortran_charlen_t direct_len, flexiblas_fortran_charlen_t storev_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlarzt((void *) direct, (void *) storev, (void *) n, (void *) k, (void *) v, (void *) ldv, (void *) tau, (void *) t, (void *) ldt, direct_len, storev_len);

    helpTimeStop = flexiblas_wtime();

    data->dlarzt.timings[0] += (helpTimeStop - helpTime);
    data->dlarzt.calls[0]++;
    
    return;
}


/* Wrapper for dlas2 */
void hook_dlas2(double *f, double *g, double *h, double *ssmin, double *ssmax)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlas2((void *) f, (void *) g, (void *) h, (void *) ssmin, (void *) ssmax);

    helpTimeStop = flexiblas_wtime();

    data->dlas2.timings[0] += (helpTimeStop - helpTime);
    data->dlas2.calls[0]++;
    
    return;
}


/* Wrapper for dlascl */
void hook_dlascl(char *type, blasint *kl, blasint *ku, double *cfrom, double *cto, blasint *m, blasint *n, double *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t type_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlascl((void *) type, (void *) kl, (void *) ku, (void *) cfrom, (void *) cto, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) info, type_len);

    helpTimeStop = flexiblas_wtime();

    data->dlascl.timings[0] += (helpTimeStop - helpTime);
    data->dlascl.calls[0]++;
    
    return;
}


/* Wrapper for dlasd0 */
void hook_dlasd0(blasint *n, blasint *sqre, double *d, double *e, double *u, blasint *ldu, double *vt, blasint *ldvt, blasint *smlsiz, blasint *iwork, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasd0((void *) n, (void *) sqre, (void *) d, (void *) e, (void *) u, (void *) ldu, (void *) vt, (void *) ldvt, (void *) smlsiz, (void *) iwork, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasd0.timings[0] += (helpTimeStop - helpTime);
    data->dlasd0.calls[0]++;
    
    return;
}


/* Wrapper for dlasd1 */
void hook_dlasd1(blasint *nl, blasint *nr, blasint *sqre, double *d, double *alpha, double *beta, double *u, blasint *ldu, double *vt, blasint *ldvt, blasint *idxq, blasint *iwork, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasd1((void *) nl, (void *) nr, (void *) sqre, (void *) d, (void *) alpha, (void *) beta, (void *) u, (void *) ldu, (void *) vt, (void *) ldvt, (void *) idxq, (void *) iwork, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasd1.timings[0] += (helpTimeStop - helpTime);
    data->dlasd1.calls[0]++;
    
    return;
}


/* Wrapper for dlasd2 */
void hook_dlasd2(blasint *nl, blasint *nr, blasint *sqre, blasint *k, double *d, double *z, double *alpha, double *beta, double *u, blasint *ldu, double *vt, blasint *ldvt, double *dsigma, double *u2, blasint *ldu2, double *vt2, blasint *ldvt2, blasint *idxp, blasint *idx, blasint *idxc, blasint *idxq, blasint *coltyp, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasd2((void *) nl, (void *) nr, (void *) sqre, (void *) k, (void *) d, (void *) z, (void *) alpha, (void *) beta, (void *) u, (void *) ldu, (void *) vt, (void *) ldvt, (void *) dsigma, (void *) u2, (void *) ldu2, (void *) vt2, (void *) ldvt2, (void *) idxp, (void *) idx, (void *) idxc, (void *) idxq, (void *) coltyp, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasd2.timings[0] += (helpTimeStop - helpTime);
    data->dlasd2.calls[0]++;
    
    return;
}


/* Wrapper for dlasd3 */
void hook_dlasd3(blasint *nl, blasint *nr, blasint *sqre, blasint *k, double *d, double *q, blasint *ldq, double *dsigma, double *u, blasint *ldu, double *u2, blasint *ldu2, double *vt, blasint *ldvt, double *vt2, blasint *ldvt2, blasint *idxc, blasint *ctot, double *z, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasd3((void *) nl, (void *) nr, (void *) sqre, (void *) k, (void *) d, (void *) q, (void *) ldq, (void *) dsigma, (void *) u, (void *) ldu, (void *) u2, (void *) ldu2, (void *) vt, (void *) ldvt, (void *) vt2, (void *) ldvt2, (void *) idxc, (void *) ctot, (void *) z, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasd3.timings[0] += (helpTimeStop - helpTime);
    data->dlasd3.calls[0]++;
    
    return;
}


/* Wrapper for dlasd4 */
void hook_dlasd4(blasint *n, blasint *i, double *d, double *z, double *delta, double *rho, double *sigma, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasd4((void *) n, (void *) i, (void *) d, (void *) z, (void *) delta, (void *) rho, (void *) sigma, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasd4.timings[0] += (helpTimeStop - helpTime);
    data->dlasd4.calls[0]++;
    
    return;
}


/* Wrapper for dlasd5 */
void hook_dlasd5(blasint *i, double *d, double *z, double *delta, double *rho, double *dsigma, double *work)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasd5((void *) i, (void *) d, (void *) z, (void *) delta, (void *) rho, (void *) dsigma, (void *) work);

    helpTimeStop = flexiblas_wtime();

    data->dlasd5.timings[0] += (helpTimeStop - helpTime);
    data->dlasd5.calls[0]++;
    
    return;
}


/* Wrapper for dlasd6 */
void hook_dlasd6(blasint *icompq, blasint *nl, blasint *nr, blasint *sqre, double *d, double *vf, double *vl, double *alpha, double *beta, blasint *idxq, blasint *perm, blasint *givptr, blasint *givcol, blasint *ldgcol, double *givnum, blasint *ldgnum, double *poles, double *difl, double *difr, double *z, blasint *k, double *c, double *s, double *work, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasd6((void *) icompq, (void *) nl, (void *) nr, (void *) sqre, (void *) d, (void *) vf, (void *) vl, (void *) alpha, (void *) beta, (void *) idxq, (void *) perm, (void *) givptr, (void *) givcol, (void *) ldgcol, (void *) givnum, (void *) ldgnum, (void *) poles, (void *) difl, (void *) difr, (void *) z, (void *) k, (void *) c, (void *) s, (void *) work, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasd6.timings[0] += (helpTimeStop - helpTime);
    data->dlasd6.calls[0]++;
    
    return;
}


/* Wrapper for dlasd7 */
void hook_dlasd7(blasint *icompq, blasint *nl, blasint *nr, blasint *sqre, blasint *k, double *d, double *z, double *zw, double *vf, double *vfw, double *vl, double *vlw, double *alpha, double *beta, double *dsigma, blasint *idx, blasint *idxp, blasint *idxq, blasint *perm, blasint *givptr, blasint *givcol, blasint *ldgcol, double *givnum, blasint *ldgnum, double *c, double *s, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasd7((void *) icompq, (void *) nl, (void *) nr, (void *) sqre, (void *) k, (void *) d, (void *) z, (void *) zw, (void *) vf, (void *) vfw, (void *) vl, (void *) vlw, (void *) alpha, (void *) beta, (void *) dsigma, (void *) idx, (void *) idxp, (void *) idxq, (void *) perm, (void *) givptr, (void *) givcol, (void *) ldgcol, (void *) givnum, (void *) ldgnum, (void *) c, (void *) s, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasd7.timings[0] += (helpTimeStop - helpTime);
    data->dlasd7.calls[0]++;
    
    return;
}


/* Wrapper for dlasd8 */
void hook_dlasd8(blasint *icompq, blasint *k, double *d, double *z, double *vf, double *vl, double *difl, double *difr, blasint *lddifr, double *dsigma, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasd8((void *) icompq, (void *) k, (void *) d, (void *) z, (void *) vf, (void *) vl, (void *) difl, (void *) difr, (void *) lddifr, (void *) dsigma, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasd8.timings[0] += (helpTimeStop - helpTime);
    data->dlasd8.calls[0]++;
    
    return;
}


/* Wrapper for dlasda */
void hook_dlasda(blasint *icompq, blasint *smlsiz, blasint *n, blasint *sqre, double *d, double *e, double *u, blasint *ldu, double *vt, blasint *k, double *difl, double *difr, double *z, double *poles, blasint *givptr, blasint *givcol, blasint *ldgcol, blasint *perm, double *givnum, double *c, double *s, double *work, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasda((void *) icompq, (void *) smlsiz, (void *) n, (void *) sqre, (void *) d, (void *) e, (void *) u, (void *) ldu, (void *) vt, (void *) k, (void *) difl, (void *) difr, (void *) z, (void *) poles, (void *) givptr, (void *) givcol, (void *) ldgcol, (void *) perm, (void *) givnum, (void *) c, (void *) s, (void *) work, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasda.timings[0] += (helpTimeStop - helpTime);
    data->dlasda.calls[0]++;
    
    return;
}


/* Wrapper for dlasdq */
void hook_dlasdq(char *uplo, blasint *sqre, blasint *n, blasint *ncvt, blasint *nru, blasint *ncc, double *d, double *e, double *vt, blasint *ldvt, double *u, blasint *ldu, double *c, blasint *ldc, double *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasdq((void *) uplo, (void *) sqre, (void *) n, (void *) ncvt, (void *) nru, (void *) ncc, (void *) d, (void *) e, (void *) vt, (void *) ldvt, (void *) u, (void *) ldu, (void *) c, (void *) ldc, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dlasdq.timings[0] += (helpTimeStop - helpTime);
    data->dlasdq.calls[0]++;
    
    return;
}


/* Wrapper for dlasdt */
void hook_dlasdt(blasint *n, blasint *lvl, blasint *nd, blasint *inode, blasint *ndiml, blasint *ndimr, blasint *msub)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasdt((void *) n, (void *) lvl, (void *) nd, (void *) inode, (void *) ndiml, (void *) ndimr, (void *) msub);

    helpTimeStop = flexiblas_wtime();

    data->dlasdt.timings[0] += (helpTimeStop - helpTime);
    data->dlasdt.calls[0]++;
    
    return;
}


/* Wrapper for dlaset */
void hook_dlaset(char *uplo, blasint *m, blasint *n, double *alpha, double *beta, double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaset((void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) beta, (void *) a, (void *) lda, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dlaset.timings[0] += (helpTimeStop - helpTime);
    data->dlaset.calls[0]++;
    
    return;
}


/* Wrapper for dlasq1 */
void hook_dlasq1(blasint *n, double *d, double *e, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasq1((void *) n, (void *) d, (void *) e, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasq1.timings[0] += (helpTimeStop - helpTime);
    data->dlasq1.calls[0]++;
    
    return;
}


/* Wrapper for dlasq2 */
void hook_dlasq2(blasint *n, double *z, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasq2((void *) n, (void *) z, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasq2.timings[0] += (helpTimeStop - helpTime);
    data->dlasq2.calls[0]++;
    
    return;
}


/* Wrapper for dlasq3 */
void hook_dlasq3(blasint *i0, blasint *n0, double *z, blasint *pp, double *dmin, double *sigma, double *desig, double *qmax, blasint *nfail, blasint *iter, blasint *ndiv, blaslogical *ieee, blasint *ttype, double *dmin1, double *dmin2, double *dn, double *dn1, double *dn2, double *g, double *tau)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasq3((void *) i0, (void *) n0, (void *) z, (void *) pp, (void *) dmin, (void *) sigma, (void *) desig, (void *) qmax, (void *) nfail, (void *) iter, (void *) ndiv, (void *) ieee, (void *) ttype, (void *) dmin1, (void *) dmin2, (void *) dn, (void *) dn1, (void *) dn2, (void *) g, (void *) tau);

    helpTimeStop = flexiblas_wtime();

    data->dlasq3.timings[0] += (helpTimeStop - helpTime);
    data->dlasq3.calls[0]++;
    
    return;
}


/* Wrapper for dlasq4 */
void hook_dlasq4(blasint *i0, blasint *n0, double *z, blasint *pp, blasint *n0in, double *dmin, double *dmin1, double *dmin2, double *dn, double *dn1, double *dn2, double *tau, blasint *ttype, double *g)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasq4((void *) i0, (void *) n0, (void *) z, (void *) pp, (void *) n0in, (void *) dmin, (void *) dmin1, (void *) dmin2, (void *) dn, (void *) dn1, (void *) dn2, (void *) tau, (void *) ttype, (void *) g);

    helpTimeStop = flexiblas_wtime();

    data->dlasq4.timings[0] += (helpTimeStop - helpTime);
    data->dlasq4.calls[0]++;
    
    return;
}


/* Wrapper for dlasq5 */
void hook_dlasq5(blasint *i0, blasint *n0, double *z, blasint *pp, double *tau, double *sigma, double *dmin, double *dmin1, double *dmin2, double *dn, double *dnm1, double *dnm2, blaslogical *ieee, double *eps)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasq5((void *) i0, (void *) n0, (void *) z, (void *) pp, (void *) tau, (void *) sigma, (void *) dmin, (void *) dmin1, (void *) dmin2, (void *) dn, (void *) dnm1, (void *) dnm2, (void *) ieee, (void *) eps);

    helpTimeStop = flexiblas_wtime();

    data->dlasq5.timings[0] += (helpTimeStop - helpTime);
    data->dlasq5.calls[0]++;
    
    return;
}


/* Wrapper for dlasq6 */
void hook_dlasq6(blasint *i0, blasint *n0, double *z, blasint *pp, double *dmin, double *dmin1, double *dmin2, double *dn, double *dnm1, double *dnm2)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasq6((void *) i0, (void *) n0, (void *) z, (void *) pp, (void *) dmin, (void *) dmin1, (void *) dmin2, (void *) dn, (void *) dnm1, (void *) dnm2);

    helpTimeStop = flexiblas_wtime();

    data->dlasq6.timings[0] += (helpTimeStop - helpTime);
    data->dlasq6.calls[0]++;
    
    return;
}


/* Wrapper for dlasr */
void hook_dlasr(char *side, char *pivot, char *direct, blasint *m, blasint *n, double *c, double *s, double *a, blasint *lda, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t pivot_len, flexiblas_fortran_charlen_t direct_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasr((void *) side, (void *) pivot, (void *) direct, (void *) m, (void *) n, (void *) c, (void *) s, (void *) a, (void *) lda, side_len, pivot_len, direct_len);

    helpTimeStop = flexiblas_wtime();

    data->dlasr.timings[0] += (helpTimeStop - helpTime);
    data->dlasr.calls[0]++;
    
    return;
}


/* Wrapper for dlasrt */
void hook_dlasrt(char *id, blasint *n, double *d, blasint *info, flexiblas_fortran_charlen_t id_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasrt((void *) id, (void *) n, (void *) d, (void *) info, id_len);

    helpTimeStop = flexiblas_wtime();

    data->dlasrt.timings[0] += (helpTimeStop - helpTime);
    data->dlasrt.calls[0]++;
    
    return;
}


/* Wrapper for dlassq */
void hook_dlassq(blasint *n, double *x, blasint *incx, double *scale, double *sumsq)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlassq((void *) n, (void *) x, (void *) incx, (void *) scale, (void *) sumsq);

    helpTimeStop = flexiblas_wtime();

    data->dlassq.timings[0] += (helpTimeStop - helpTime);
    data->dlassq.calls[0]++;
    
    return;
}


/* Wrapper for dlasv2 */
void hook_dlasv2(double *f, double *g, double *h, double *ssmin, double *ssmax, double *snr, double *csr, double *snl, double *csl)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasv2((void *) f, (void *) g, (void *) h, (void *) ssmin, (void *) ssmax, (void *) snr, (void *) csr, (void *) snl, (void *) csl);

    helpTimeStop = flexiblas_wtime();

    data->dlasv2.timings[0] += (helpTimeStop - helpTime);
    data->dlasv2.calls[0]++;
    
    return;
}


/* Wrapper for dlaswlq */
void hook_dlaswlq(blasint *m, blasint *n, blasint *mb, blasint *nb, double *a, blasint *lda, double *t, blasint *ldt, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaswlq((void *) m, (void *) n, (void *) mb, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlaswlq.timings[0] += (helpTimeStop - helpTime);
    data->dlaswlq.calls[0]++;
    
    return;
}


/* Wrapper for dlaswp */
void hook_dlaswp(blasint *n, double *a, blasint *lda, blasint *k1, blasint *k2, blasint *ipiv, blasint *incx)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlaswp((void *) n, (void *) a, (void *) lda, (void *) k1, (void *) k2, (void *) ipiv, (void *) incx);

    helpTimeStop = flexiblas_wtime();

    data->dlaswp.timings[0] += (helpTimeStop - helpTime);
    data->dlaswp.calls[0]++;
    
    return;
}


/* Wrapper for dlasy2 */
void hook_dlasy2(blaslogical *ltranl, blaslogical *ltranr, blasint *isgn, blasint *n1, blasint *n2, double *tl, blasint *ldtl, double *tr, blasint *ldtr, double *b, blasint *ldb, double *scale, double *x, blasint *ldx, double *xnorm, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasy2((void *) ltranl, (void *) ltranr, (void *) isgn, (void *) n1, (void *) n2, (void *) tl, (void *) ldtl, (void *) tr, (void *) ldtr, (void *) b, (void *) ldb, (void *) scale, (void *) x, (void *) ldx, (void *) xnorm, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlasy2.timings[0] += (helpTimeStop - helpTime);
    data->dlasy2.calls[0]++;
    
    return;
}


/* Wrapper for dlasyf */
void hook_dlasyf(char *uplo, blasint *n, blasint *nb, blasint *kb, double *a, blasint *lda, blasint *ipiv, double *w, blasint *ldw, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasyf((void *) uplo, (void *) n, (void *) nb, (void *) kb, (void *) a, (void *) lda, (void *) ipiv, (void *) w, (void *) ldw, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dlasyf.timings[0] += (helpTimeStop - helpTime);
    data->dlasyf.calls[0]++;
    
    return;
}


/* Wrapper for dlasyf_aa */
void hook_dlasyf_aa(char *uplo, blasint *j1, blasint *m, blasint *nb, double *a, blasint *lda, blasint *ipiv, double *h, blasint *ldh, double *work, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasyf_aa((void *) uplo, (void *) j1, (void *) m, (void *) nb, (void *) a, (void *) lda, (void *) ipiv, (void *) h, (void *) ldh, (void *) work, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dlasyf_aa.timings[0] += (helpTimeStop - helpTime);
    data->dlasyf_aa.calls[0]++;
    
    return;
}


/* Wrapper for dlasyf_rk */
void hook_dlasyf_rk(char *uplo, blasint *n, blasint *nb, blasint *kb, double *a, blasint *lda, double *e, blasint *ipiv, double *w, blasint *ldw, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasyf_rk((void *) uplo, (void *) n, (void *) nb, (void *) kb, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) w, (void *) ldw, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dlasyf_rk.timings[0] += (helpTimeStop - helpTime);
    data->dlasyf_rk.calls[0]++;
    
    return;
}


/* Wrapper for dlasyf_rook */
void hook_dlasyf_rook(char *uplo, blasint *n, blasint *nb, blasint *kb, double *a, blasint *lda, blasint *ipiv, double *w, blasint *ldw, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlasyf_rook((void *) uplo, (void *) n, (void *) nb, (void *) kb, (void *) a, (void *) lda, (void *) ipiv, (void *) w, (void *) ldw, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dlasyf_rook.timings[0] += (helpTimeStop - helpTime);
    data->dlasyf_rook.calls[0]++;
    
    return;
}


/* Wrapper for dlat2s */
void hook_dlat2s(char *uplo, blasint *n, double *a, blasint *lda, float *sa, blasint *ldsa, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlat2s((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) sa, (void *) ldsa, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dlat2s.timings[0] += (helpTimeStop - helpTime);
    data->dlat2s.calls[0]++;
    
    return;
}


/* Wrapper for dlatbs */
void hook_dlatbs(char *uplo, char *trans, char *diag, char *normin, blasint *n, blasint *kd, double *ab, blasint *ldab, double *x, double *scale, double *cnorm, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len, flexiblas_fortran_charlen_t normin_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlatbs((void *) uplo, (void *) trans, (void *) diag, (void *) normin, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) x, (void *) scale, (void *) cnorm, (void *) info, uplo_len, trans_len, diag_len, normin_len);

    helpTimeStop = flexiblas_wtime();

    data->dlatbs.timings[0] += (helpTimeStop - helpTime);
    data->dlatbs.calls[0]++;
    
    return;
}


/* Wrapper for dlatdf */
void hook_dlatdf(blasint *ijob, blasint *n, double *z, blasint *ldz, double *rhs, double *rdsum, double *rdscal, blasint *ipiv, blasint *jpiv)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlatdf((void *) ijob, (void *) n, (void *) z, (void *) ldz, (void *) rhs, (void *) rdsum, (void *) rdscal, (void *) ipiv, (void *) jpiv);

    helpTimeStop = flexiblas_wtime();

    data->dlatdf.timings[0] += (helpTimeStop - helpTime);
    data->dlatdf.calls[0]++;
    
    return;
}


/* Wrapper for dlatps */
void hook_dlatps(char *uplo, char *trans, char *diag, char *normin, blasint *n, double *ap, double *x, double *scale, double *cnorm, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len, flexiblas_fortran_charlen_t normin_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlatps((void *) uplo, (void *) trans, (void *) diag, (void *) normin, (void *) n, (void *) ap, (void *) x, (void *) scale, (void *) cnorm, (void *) info, uplo_len, trans_len, diag_len, normin_len);

    helpTimeStop = flexiblas_wtime();

    data->dlatps.timings[0] += (helpTimeStop - helpTime);
    data->dlatps.calls[0]++;
    
    return;
}


/* Wrapper for dlatrd */
void hook_dlatrd(char *uplo, blasint *n, blasint *nb, double *a, blasint *lda, double *e, double *tau, double *w, blasint *ldw, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlatrd((void *) uplo, (void *) n, (void *) nb, (void *) a, (void *) lda, (void *) e, (void *) tau, (void *) w, (void *) ldw, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dlatrd.timings[0] += (helpTimeStop - helpTime);
    data->dlatrd.calls[0]++;
    
    return;
}


/* Wrapper for dlatrs */
void hook_dlatrs(char *uplo, char *trans, char *diag, char *normin, blasint *n, double *a, blasint *lda, double *x, double *scale, double *cnorm, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len, flexiblas_fortran_charlen_t normin_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlatrs((void *) uplo, (void *) trans, (void *) diag, (void *) normin, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) scale, (void *) cnorm, (void *) info, uplo_len, trans_len, diag_len, normin_len);

    helpTimeStop = flexiblas_wtime();

    data->dlatrs.timings[0] += (helpTimeStop - helpTime);
    data->dlatrs.calls[0]++;
    
    return;
}


/* Wrapper for dlatrz */
void hook_dlatrz(blasint *m, blasint *n, blasint *l, double *a, blasint *lda, double *tau, double *work)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlatrz((void *) m, (void *) n, (void *) l, (void *) a, (void *) lda, (void *) tau, (void *) work);

    helpTimeStop = flexiblas_wtime();

    data->dlatrz.timings[0] += (helpTimeStop - helpTime);
    data->dlatrz.calls[0]++;
    
    return;
}


/* Wrapper for dlatsqr */
void hook_dlatsqr(blasint *m, blasint *n, blasint *mb, blasint *nb, double *a, blasint *lda, double *t, blasint *ldt, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlatsqr((void *) m, (void *) n, (void *) mb, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dlatsqr.timings[0] += (helpTimeStop - helpTime);
    data->dlatsqr.calls[0]++;
    
    return;
}


/* Wrapper for dlauu2 */
void hook_dlauu2(char *uplo, blasint *n, double *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlauu2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dlauu2.timings[0] += (helpTimeStop - helpTime);
    data->dlauu2.calls[0]++;
    
    return;
}


/* Wrapper for dlauum */
void hook_dlauum(char *uplo, blasint *n, double *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dlauum((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dlauum.timings[0] += (helpTimeStop - helpTime);
    data->dlauum.calls[0]++;
    
    return;
}


/* Wrapper for dopgtr */
void hook_dopgtr(char *uplo, blasint *n, double *ap, double *tau, double *q, blasint *ldq, double *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dopgtr((void *) uplo, (void *) n, (void *) ap, (void *) tau, (void *) q, (void *) ldq, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dopgtr.timings[0] += (helpTimeStop - helpTime);
    data->dopgtr.calls[0]++;
    
    return;
}


/* Wrapper for dopmtr */
void hook_dopmtr(char *side, char *uplo, char *trans, blasint *m, blasint *n, double *ap, double *tau, double *c, blasint *ldc, double *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dopmtr((void *) side, (void *) uplo, (void *) trans, (void *) m, (void *) n, (void *) ap, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, uplo_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dopmtr.timings[0] += (helpTimeStop - helpTime);
    data->dopmtr.calls[0]++;
    
    return;
}


/* Wrapper for dorbdb */
void hook_dorbdb(char *trans, char *signs, blasint *m, blasint *p, blasint *q, double *x11, blasint *ldx11, double *x12, blasint *ldx12, double *x21, blasint *ldx21, double *x22, blasint *ldx22, double *theta, double *phi, double *taup1, double *taup2, double *tauq1, double *tauq2, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t signs_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorbdb((void *) trans, (void *) signs, (void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x12, (void *) ldx12, (void *) x21, (void *) ldx21, (void *) x22, (void *) ldx22, (void *) theta, (void *) phi, (void *) taup1, (void *) taup2, (void *) tauq1, (void *) tauq2, (void *) work, (void *) lwork, (void *) info, trans_len, signs_len);

    helpTimeStop = flexiblas_wtime();

    data->dorbdb.timings[0] += (helpTimeStop - helpTime);
    data->dorbdb.calls[0]++;
    
    return;
}


/* Wrapper for dorbdb1 */
void hook_dorbdb1(blasint *m, blasint *p, blasint *q, double *x11, blasint *ldx11, double *x21, blasint *ldx21, double *theta, double *phi, double *taup1, double *taup2, double *tauq1, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorbdb1((void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x21, (void *) ldx21, (void *) theta, (void *) phi, (void *) taup1, (void *) taup2, (void *) tauq1, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dorbdb1.timings[0] += (helpTimeStop - helpTime);
    data->dorbdb1.calls[0]++;
    
    return;
}


/* Wrapper for dorbdb2 */
void hook_dorbdb2(blasint *m, blasint *p, blasint *q, double *x11, blasint *ldx11, double *x21, blasint *ldx21, double *theta, double *phi, double *taup1, double *taup2, double *tauq1, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorbdb2((void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x21, (void *) ldx21, (void *) theta, (void *) phi, (void *) taup1, (void *) taup2, (void *) tauq1, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dorbdb2.timings[0] += (helpTimeStop - helpTime);
    data->dorbdb2.calls[0]++;
    
    return;
}


/* Wrapper for dorbdb3 */
void hook_dorbdb3(blasint *m, blasint *p, blasint *q, double *x11, blasint *ldx11, double *x21, blasint *ldx21, double *theta, double *phi, double *taup1, double *taup2, double *tauq1, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorbdb3((void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x21, (void *) ldx21, (void *) theta, (void *) phi, (void *) taup1, (void *) taup2, (void *) tauq1, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dorbdb3.timings[0] += (helpTimeStop - helpTime);
    data->dorbdb3.calls[0]++;
    
    return;
}


/* Wrapper for dorbdb4 */
void hook_dorbdb4(blasint *m, blasint *p, blasint *q, double *x11, blasint *ldx11, double *x21, blasint *ldx21, double *theta, double *phi, double *taup1, double *taup2, double *tauq1, double *phantom, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorbdb4((void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x21, (void *) ldx21, (void *) theta, (void *) phi, (void *) taup1, (void *) taup2, (void *) tauq1, (void *) phantom, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dorbdb4.timings[0] += (helpTimeStop - helpTime);
    data->dorbdb4.calls[0]++;
    
    return;
}


/* Wrapper for dorbdb5 */
void hook_dorbdb5(blasint *m1, blasint *m2, blasint *n, double *x1, blasint *incx1, double *x2, blasint *incx2, double *q1, blasint *ldq1, double *q2, blasint *ldq2, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorbdb5((void *) m1, (void *) m2, (void *) n, (void *) x1, (void *) incx1, (void *) x2, (void *) incx2, (void *) q1, (void *) ldq1, (void *) q2, (void *) ldq2, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dorbdb5.timings[0] += (helpTimeStop - helpTime);
    data->dorbdb5.calls[0]++;
    
    return;
}


/* Wrapper for dorbdb6 */
void hook_dorbdb6(blasint *m1, blasint *m2, blasint *n, double *x1, blasint *incx1, double *x2, blasint *incx2, double *q1, blasint *ldq1, double *q2, blasint *ldq2, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorbdb6((void *) m1, (void *) m2, (void *) n, (void *) x1, (void *) incx1, (void *) x2, (void *) incx2, (void *) q1, (void *) ldq1, (void *) q2, (void *) ldq2, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dorbdb6.timings[0] += (helpTimeStop - helpTime);
    data->dorbdb6.calls[0]++;
    
    return;
}


/* Wrapper for dorcsd */
void hook_dorcsd(char *jobu1, char *jobu2, char *jobv1t, char *jobv2t, char *trans, char *signs, blasint *m, blasint *p, blasint *q, double *x11, blasint *ldx11, double *x12, blasint *ldx12, double *x21, blasint *ldx21, double *x22, blasint *ldx22, double *theta, double *u1, blasint *ldu1, double *u2, blasint *ldu2, double *v1t, blasint *ldv1t, double *v2t, blasint *ldv2t, double *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t jobu1_len, flexiblas_fortran_charlen_t jobu2_len, flexiblas_fortran_charlen_t jobv1t_len, flexiblas_fortran_charlen_t jobv2t_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t signs_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorcsd((void *) jobu1, (void *) jobu2, (void *) jobv1t, (void *) jobv2t, (void *) trans, (void *) signs, (void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x12, (void *) ldx12, (void *) x21, (void *) ldx21, (void *) x22, (void *) ldx22, (void *) theta, (void *) u1, (void *) ldu1, (void *) u2, (void *) ldu2, (void *) v1t, (void *) ldv1t, (void *) v2t, (void *) ldv2t, (void *) work, (void *) lwork, (void *) iwork, (void *) info, jobu1_len, jobu2_len, jobv1t_len, jobv2t_len, trans_len, signs_len);

    helpTimeStop = flexiblas_wtime();

    data->dorcsd.timings[0] += (helpTimeStop - helpTime);
    data->dorcsd.calls[0]++;
    
    return;
}


/* Wrapper for dorcsd2by1 */
void hook_dorcsd2by1(char *jobu1, char *jobu2, char *jobv1t, blasint *m, blasint *p, blasint *q, double *x11, blasint *ldx11, double *x21, blasint *ldx21, double *theta, double *u1, blasint *ldu1, double *u2, blasint *ldu2, double *v1t, blasint *ldv1t, double *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t jobu1_len, flexiblas_fortran_charlen_t jobu2_len, flexiblas_fortran_charlen_t jobv1t_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorcsd2by1((void *) jobu1, (void *) jobu2, (void *) jobv1t, (void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x21, (void *) ldx21, (void *) theta, (void *) u1, (void *) ldu1, (void *) u2, (void *) ldu2, (void *) v1t, (void *) ldv1t, (void *) work, (void *) lwork, (void *) iwork, (void *) info, jobu1_len, jobu2_len, jobv1t_len);

    helpTimeStop = flexiblas_wtime();

    data->dorcsd2by1.timings[0] += (helpTimeStop - helpTime);
    data->dorcsd2by1.calls[0]++;
    
    return;
}


/* Wrapper for dorg2l */
void hook_dorg2l(blasint *m, blasint *n, blasint *k, double *a, blasint *lda, double *tau, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorg2l((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dorg2l.timings[0] += (helpTimeStop - helpTime);
    data->dorg2l.calls[0]++;
    
    return;
}


/* Wrapper for dorg2r */
void hook_dorg2r(blasint *m, blasint *n, blasint *k, double *a, blasint *lda, double *tau, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorg2r((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dorg2r.timings[0] += (helpTimeStop - helpTime);
    data->dorg2r.calls[0]++;
    
    return;
}


/* Wrapper for dorgbr */
void hook_dorgbr(char *vect, blasint *m, blasint *n, blasint *k, double *a, blasint *lda, double *tau, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t vect_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorgbr((void *) vect, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info, vect_len);

    helpTimeStop = flexiblas_wtime();

    data->dorgbr.timings[0] += (helpTimeStop - helpTime);
    data->dorgbr.calls[0]++;
    
    return;
}


/* Wrapper for dorghr */
void hook_dorghr(blasint *n, blasint *ilo, blasint *ihi, double *a, blasint *lda, double *tau, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorghr((void *) n, (void *) ilo, (void *) ihi, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dorghr.timings[0] += (helpTimeStop - helpTime);
    data->dorghr.calls[0]++;
    
    return;
}


/* Wrapper for dorgl2 */
void hook_dorgl2(blasint *m, blasint *n, blasint *k, double *a, blasint *lda, double *tau, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorgl2((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dorgl2.timings[0] += (helpTimeStop - helpTime);
    data->dorgl2.calls[0]++;
    
    return;
}


/* Wrapper for dorglq */
void hook_dorglq(blasint *m, blasint *n, blasint *k, double *a, blasint *lda, double *tau, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorglq((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dorglq.timings[0] += (helpTimeStop - helpTime);
    data->dorglq.calls[0]++;
    
    return;
}


/* Wrapper for dorgql */
void hook_dorgql(blasint *m, blasint *n, blasint *k, double *a, blasint *lda, double *tau, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorgql((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dorgql.timings[0] += (helpTimeStop - helpTime);
    data->dorgql.calls[0]++;
    
    return;
}


/* Wrapper for dorgqr */
void hook_dorgqr(blasint *m, blasint *n, blasint *k, double *a, blasint *lda, double *tau, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorgqr((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dorgqr.timings[0] += (helpTimeStop - helpTime);
    data->dorgqr.calls[0]++;
    
    return;
}


/* Wrapper for dorgr2 */
void hook_dorgr2(blasint *m, blasint *n, blasint *k, double *a, blasint *lda, double *tau, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorgr2((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dorgr2.timings[0] += (helpTimeStop - helpTime);
    data->dorgr2.calls[0]++;
    
    return;
}


/* Wrapper for dorgrq */
void hook_dorgrq(blasint *m, blasint *n, blasint *k, double *a, blasint *lda, double *tau, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorgrq((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dorgrq.timings[0] += (helpTimeStop - helpTime);
    data->dorgrq.calls[0]++;
    
    return;
}


/* Wrapper for dorgtr */
void hook_dorgtr(char *uplo, blasint *n, double *a, blasint *lda, double *tau, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorgtr((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dorgtr.timings[0] += (helpTimeStop - helpTime);
    data->dorgtr.calls[0]++;
    
    return;
}


/* Wrapper for dorgtsqr */
void hook_dorgtsqr(blasint *m, blasint *n, blasint *mb, blasint *nb, double *a, blasint *lda, double *t, blasint *ldt, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorgtsqr((void *) m, (void *) n, (void *) mb, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dorgtsqr.timings[0] += (helpTimeStop - helpTime);
    data->dorgtsqr.calls[0]++;
    
    return;
}


/* Wrapper for dorgtsqr_row */
void hook_dorgtsqr_row(blasint *m, blasint *n, blasint *mb, blasint *nb, double *a, blasint *lda, double *t, blasint *ldt, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorgtsqr_row((void *) m, (void *) n, (void *) mb, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dorgtsqr_row.timings[0] += (helpTimeStop - helpTime);
    data->dorgtsqr_row.calls[0]++;
    
    return;
}


/* Wrapper for dorhr_col */
void hook_dorhr_col(blasint *m, blasint *n, blasint *nb, double *a, blasint *lda, double *t, blasint *ldt, double *d, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorhr_col((void *) m, (void *) n, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) d, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dorhr_col.timings[0] += (helpTimeStop - helpTime);
    data->dorhr_col.calls[0]++;
    
    return;
}


/* Wrapper for dorm22 */
void hook_dorm22(char *side, char *trans, blasint *m, blasint *n, blasint *n1, blasint *n2, double *q, blasint *ldq, double *c, blasint *ldc, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorm22((void *) side, (void *) trans, (void *) m, (void *) n, (void *) n1, (void *) n2, (void *) q, (void *) ldq, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dorm22.timings[0] += (helpTimeStop - helpTime);
    data->dorm22.calls[0]++;
    
    return;
}


/* Wrapper for dorm2l */
void hook_dorm2l(char *side, char *trans, blasint *m, blasint *n, blasint *k, double *a, blasint *lda, double *tau, double *c, blasint *ldc, double *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorm2l((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dorm2l.timings[0] += (helpTimeStop - helpTime);
    data->dorm2l.calls[0]++;
    
    return;
}


/* Wrapper for dorm2r */
void hook_dorm2r(char *side, char *trans, blasint *m, blasint *n, blasint *k, double *a, blasint *lda, double *tau, double *c, blasint *ldc, double *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorm2r((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dorm2r.timings[0] += (helpTimeStop - helpTime);
    data->dorm2r.calls[0]++;
    
    return;
}


/* Wrapper for dormbr */
void hook_dormbr(char *vect, char *side, char *trans, blasint *m, blasint *n, blasint *k, double *a, blasint *lda, double *tau, double *c, blasint *ldc, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t vect_len, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dormbr((void *) vect, (void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, vect_len, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dormbr.timings[0] += (helpTimeStop - helpTime);
    data->dormbr.calls[0]++;
    
    return;
}


/* Wrapper for dormhr */
void hook_dormhr(char *side, char *trans, blasint *m, blasint *n, blasint *ilo, blasint *ihi, double *a, blasint *lda, double *tau, double *c, blasint *ldc, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dormhr((void *) side, (void *) trans, (void *) m, (void *) n, (void *) ilo, (void *) ihi, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dormhr.timings[0] += (helpTimeStop - helpTime);
    data->dormhr.calls[0]++;
    
    return;
}


/* Wrapper for dorml2 */
void hook_dorml2(char *side, char *trans, blasint *m, blasint *n, blasint *k, double *a, blasint *lda, double *tau, double *c, blasint *ldc, double *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dorml2((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dorml2.timings[0] += (helpTimeStop - helpTime);
    data->dorml2.calls[0]++;
    
    return;
}


/* Wrapper for dormlq */
void hook_dormlq(char *side, char *trans, blasint *m, blasint *n, blasint *k, double *a, blasint *lda, double *tau, double *c, blasint *ldc, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dormlq((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dormlq.timings[0] += (helpTimeStop - helpTime);
    data->dormlq.calls[0]++;
    
    return;
}


/* Wrapper for dormql */
void hook_dormql(char *side, char *trans, blasint *m, blasint *n, blasint *k, double *a, blasint *lda, double *tau, double *c, blasint *ldc, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dormql((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dormql.timings[0] += (helpTimeStop - helpTime);
    data->dormql.calls[0]++;
    
    return;
}


/* Wrapper for dormqr */
void hook_dormqr(char *side, char *trans, blasint *m, blasint *n, blasint *k, double *a, blasint *lda, double *tau, double *c, blasint *ldc, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dormqr((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dormqr.timings[0] += (helpTimeStop - helpTime);
    data->dormqr.calls[0]++;
    
    return;
}


/* Wrapper for dormr2 */
void hook_dormr2(char *side, char *trans, blasint *m, blasint *n, blasint *k, double *a, blasint *lda, double *tau, double *c, blasint *ldc, double *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dormr2((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dormr2.timings[0] += (helpTimeStop - helpTime);
    data->dormr2.calls[0]++;
    
    return;
}


/* Wrapper for dormr3 */
void hook_dormr3(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *l, double *a, blasint *lda, double *tau, double *c, blasint *ldc, double *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dormr3((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) l, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dormr3.timings[0] += (helpTimeStop - helpTime);
    data->dormr3.calls[0]++;
    
    return;
}


/* Wrapper for dormrq */
void hook_dormrq(char *side, char *trans, blasint *m, blasint *n, blasint *k, double *a, blasint *lda, double *tau, double *c, blasint *ldc, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dormrq((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dormrq.timings[0] += (helpTimeStop - helpTime);
    data->dormrq.calls[0]++;
    
    return;
}


/* Wrapper for dormrz */
void hook_dormrz(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *l, double *a, blasint *lda, double *tau, double *c, blasint *ldc, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dormrz((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) l, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dormrz.timings[0] += (helpTimeStop - helpTime);
    data->dormrz.calls[0]++;
    
    return;
}


/* Wrapper for dormtr */
void hook_dormtr(char *side, char *uplo, char *trans, blasint *m, blasint *n, double *a, blasint *lda, double *tau, double *c, blasint *ldc, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dormtr((void *) side, (void *) uplo, (void *) trans, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, uplo_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dormtr.timings[0] += (helpTimeStop - helpTime);
    data->dormtr.calls[0]++;
    
    return;
}


/* Wrapper for dpbcon */
void hook_dpbcon(char *uplo, blasint *n, blasint *kd, double *ab, blasint *ldab, double *anorm, double *rcond, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpbcon((void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) anorm, (void *) rcond, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpbcon.timings[0] += (helpTimeStop - helpTime);
    data->dpbcon.calls[0]++;
    
    return;
}


/* Wrapper for dpbequ */
void hook_dpbequ(char *uplo, blasint *n, blasint *kd, double *ab, blasint *ldab, double *s, double *scond, double *amax, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpbequ((void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) s, (void *) scond, (void *) amax, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpbequ.timings[0] += (helpTimeStop - helpTime);
    data->dpbequ.calls[0]++;
    
    return;
}


/* Wrapper for dpbrfs */
void hook_dpbrfs(char *uplo, blasint *n, blasint *kd, blasint *nrhs, double *ab, blasint *ldab, double *afb, blasint *ldafb, double *b, blasint *ldb, double *x, blasint *ldx, double *ferr, double *berr, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpbrfs((void *) uplo, (void *) n, (void *) kd, (void *) nrhs, (void *) ab, (void *) ldab, (void *) afb, (void *) ldafb, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpbrfs.timings[0] += (helpTimeStop - helpTime);
    data->dpbrfs.calls[0]++;
    
    return;
}


/* Wrapper for dpbstf */
void hook_dpbstf(char *uplo, blasint *n, blasint *kd, double *ab, blasint *ldab, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpbstf((void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpbstf.timings[0] += (helpTimeStop - helpTime);
    data->dpbstf.calls[0]++;
    
    return;
}


/* Wrapper for dpbsv */
void hook_dpbsv(char *uplo, blasint *n, blasint *kd, blasint *nrhs, double *ab, blasint *ldab, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpbsv((void *) uplo, (void *) n, (void *) kd, (void *) nrhs, (void *) ab, (void *) ldab, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpbsv.timings[0] += (helpTimeStop - helpTime);
    data->dpbsv.calls[0]++;
    
    return;
}


/* Wrapper for dpbsvx */
void hook_dpbsvx(char *fact, char *uplo, blasint *n, blasint *kd, blasint *nrhs, double *ab, blasint *ldab, double *afb, blasint *ldafb, char *equed, double *s, double *b, blasint *ldb, double *x, blasint *ldx, double *rcond, double *ferr, double *berr, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpbsvx((void *) fact, (void *) uplo, (void *) n, (void *) kd, (void *) nrhs, (void *) ab, (void *) ldab, (void *) afb, (void *) ldafb, (void *) equed, (void *) s, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, fact_len, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->dpbsvx.timings[0] += (helpTimeStop - helpTime);
    data->dpbsvx.calls[0]++;
    
    return;
}


/* Wrapper for dpbtf2 */
void hook_dpbtf2(char *uplo, blasint *n, blasint *kd, double *ab, blasint *ldab, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpbtf2((void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpbtf2.timings[0] += (helpTimeStop - helpTime);
    data->dpbtf2.calls[0]++;
    
    return;
}


/* Wrapper for dpbtrf */
void hook_dpbtrf(char *uplo, blasint *n, blasint *kd, double *ab, blasint *ldab, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpbtrf((void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpbtrf.timings[0] += (helpTimeStop - helpTime);
    data->dpbtrf.calls[0]++;
    
    return;
}


/* Wrapper for dpbtrs */
void hook_dpbtrs(char *uplo, blasint *n, blasint *kd, blasint *nrhs, double *ab, blasint *ldab, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpbtrs((void *) uplo, (void *) n, (void *) kd, (void *) nrhs, (void *) ab, (void *) ldab, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpbtrs.timings[0] += (helpTimeStop - helpTime);
    data->dpbtrs.calls[0]++;
    
    return;
}


/* Wrapper for dpftrf */
void hook_dpftrf(char *transr, char *uplo, blasint *n, double *a, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpftrf((void *) transr, (void *) uplo, (void *) n, (void *) a, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpftrf.timings[0] += (helpTimeStop - helpTime);
    data->dpftrf.calls[0]++;
    
    return;
}


/* Wrapper for dpftri */
void hook_dpftri(char *transr, char *uplo, blasint *n, double *a, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpftri((void *) transr, (void *) uplo, (void *) n, (void *) a, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpftri.timings[0] += (helpTimeStop - helpTime);
    data->dpftri.calls[0]++;
    
    return;
}


/* Wrapper for dpftrs */
void hook_dpftrs(char *transr, char *uplo, blasint *n, blasint *nrhs, double *a, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpftrs((void *) transr, (void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) b, (void *) ldb, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpftrs.timings[0] += (helpTimeStop - helpTime);
    data->dpftrs.calls[0]++;
    
    return;
}


/* Wrapper for dpocon */
void hook_dpocon(char *uplo, blasint *n, double *a, blasint *lda, double *anorm, double *rcond, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpocon((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) anorm, (void *) rcond, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpocon.timings[0] += (helpTimeStop - helpTime);
    data->dpocon.calls[0]++;
    
    return;
}


/* Wrapper for dpoequ */
void hook_dpoequ(blasint *n, double *a, blasint *lda, double *s, double *scond, double *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpoequ((void *) n, (void *) a, (void *) lda, (void *) s, (void *) scond, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dpoequ.timings[0] += (helpTimeStop - helpTime);
    data->dpoequ.calls[0]++;
    
    return;
}


/* Wrapper for dpoequb */
void hook_dpoequb(blasint *n, double *a, blasint *lda, double *s, double *scond, double *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpoequb((void *) n, (void *) a, (void *) lda, (void *) s, (void *) scond, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dpoequb.timings[0] += (helpTimeStop - helpTime);
    data->dpoequb.calls[0]++;
    
    return;
}


/* Wrapper for dporfs */
void hook_dporfs(char *uplo, blasint *n, blasint *nrhs, double *a, blasint *lda, double *af, blasint *ldaf, double *b, blasint *ldb, double *x, blasint *ldx, double *ferr, double *berr, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dporfs((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) af, (void *) ldaf, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dporfs.timings[0] += (helpTimeStop - helpTime);
    data->dporfs.calls[0]++;
    
    return;
}


/* Wrapper for dposv */
void hook_dposv(char *uplo, blasint *n, blasint *nrhs, double *a, blasint *lda, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dposv((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dposv.timings[0] += (helpTimeStop - helpTime);
    data->dposv.calls[0]++;
    
    return;
}


/* Wrapper for dposvx */
void hook_dposvx(char *fact, char *uplo, blasint *n, blasint *nrhs, double *a, blasint *lda, double *af, blasint *ldaf, char *equed, double *s, double *b, blasint *ldb, double *x, blasint *ldx, double *rcond, double *ferr, double *berr, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dposvx((void *) fact, (void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) af, (void *) ldaf, (void *) equed, (void *) s, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, fact_len, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->dposvx.timings[0] += (helpTimeStop - helpTime);
    data->dposvx.calls[0]++;
    
    return;
}


/* Wrapper for dpotf2 */
void hook_dpotf2(char *uplo, blasint *n, double *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpotf2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpotf2.timings[0] += (helpTimeStop - helpTime);
    data->dpotf2.calls[0]++;
    
    return;
}


/* Wrapper for dpotrf */
void hook_dpotrf(char *uplo, blasint *n, double *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpotrf((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpotrf.timings[0] += (helpTimeStop - helpTime);
    data->dpotrf.calls[0]++;
    
    return;
}


/* Wrapper for dpotrf2 */
void hook_dpotrf2(char *uplo, blasint *n, double *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpotrf2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpotrf2.timings[0] += (helpTimeStop - helpTime);
    data->dpotrf2.calls[0]++;
    
    return;
}


/* Wrapper for dpotri */
void hook_dpotri(char *uplo, blasint *n, double *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpotri((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpotri.timings[0] += (helpTimeStop - helpTime);
    data->dpotri.calls[0]++;
    
    return;
}


/* Wrapper for dpotrs */
void hook_dpotrs(char *uplo, blasint *n, blasint *nrhs, double *a, blasint *lda, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpotrs((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpotrs.timings[0] += (helpTimeStop - helpTime);
    data->dpotrs.calls[0]++;
    
    return;
}


/* Wrapper for dppcon */
void hook_dppcon(char *uplo, blasint *n, double *ap, double *anorm, double *rcond, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dppcon((void *) uplo, (void *) n, (void *) ap, (void *) anorm, (void *) rcond, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dppcon.timings[0] += (helpTimeStop - helpTime);
    data->dppcon.calls[0]++;
    
    return;
}


/* Wrapper for dppequ */
void hook_dppequ(char *uplo, blasint *n, double *ap, double *s, double *scond, double *amax, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dppequ((void *) uplo, (void *) n, (void *) ap, (void *) s, (void *) scond, (void *) amax, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dppequ.timings[0] += (helpTimeStop - helpTime);
    data->dppequ.calls[0]++;
    
    return;
}


/* Wrapper for dpprfs */
void hook_dpprfs(char *uplo, blasint *n, blasint *nrhs, double *ap, double *afp, double *b, blasint *ldb, double *x, blasint *ldx, double *ferr, double *berr, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpprfs((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) afp, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpprfs.timings[0] += (helpTimeStop - helpTime);
    data->dpprfs.calls[0]++;
    
    return;
}


/* Wrapper for dppsv */
void hook_dppsv(char *uplo, blasint *n, blasint *nrhs, double *ap, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dppsv((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dppsv.timings[0] += (helpTimeStop - helpTime);
    data->dppsv.calls[0]++;
    
    return;
}


/* Wrapper for dppsvx */
void hook_dppsvx(char *fact, char *uplo, blasint *n, blasint *nrhs, double *ap, double *afp, char *equed, double *s, double *b, blasint *ldb, double *x, blasint *ldx, double *rcond, double *ferr, double *berr, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dppsvx((void *) fact, (void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) afp, (void *) equed, (void *) s, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, fact_len, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->dppsvx.timings[0] += (helpTimeStop - helpTime);
    data->dppsvx.calls[0]++;
    
    return;
}


/* Wrapper for dpptrf */
void hook_dpptrf(char *uplo, blasint *n, double *ap, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpptrf((void *) uplo, (void *) n, (void *) ap, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpptrf.timings[0] += (helpTimeStop - helpTime);
    data->dpptrf.calls[0]++;
    
    return;
}


/* Wrapper for dpptri */
void hook_dpptri(char *uplo, blasint *n, double *ap, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpptri((void *) uplo, (void *) n, (void *) ap, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpptri.timings[0] += (helpTimeStop - helpTime);
    data->dpptri.calls[0]++;
    
    return;
}


/* Wrapper for dpptrs */
void hook_dpptrs(char *uplo, blasint *n, blasint *nrhs, double *ap, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpptrs((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpptrs.timings[0] += (helpTimeStop - helpTime);
    data->dpptrs.calls[0]++;
    
    return;
}


/* Wrapper for dpstf2 */
void hook_dpstf2(char *uplo, blasint *n, double *a, blasint *lda, blasint *piv, blasint *rank, double *tol, double *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpstf2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) piv, (void *) rank, (void *) tol, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpstf2.timings[0] += (helpTimeStop - helpTime);
    data->dpstf2.calls[0]++;
    
    return;
}


/* Wrapper for dpstrf */
void hook_dpstrf(char *uplo, blasint *n, double *a, blasint *lda, blasint *piv, blasint *rank, double *tol, double *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpstrf((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) piv, (void *) rank, (void *) tol, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dpstrf.timings[0] += (helpTimeStop - helpTime);
    data->dpstrf.calls[0]++;
    
    return;
}


/* Wrapper for dptcon */
void hook_dptcon(blasint *n, double *d, double *e, double *anorm, double *rcond, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dptcon((void *) n, (void *) d, (void *) e, (void *) anorm, (void *) rcond, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dptcon.timings[0] += (helpTimeStop - helpTime);
    data->dptcon.calls[0]++;
    
    return;
}


/* Wrapper for dpteqr */
void hook_dpteqr(char *compz, blasint *n, double *d, double *e, double *z, blasint *ldz, double *work, blasint *info, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpteqr((void *) compz, (void *) n, (void *) d, (void *) e, (void *) z, (void *) ldz, (void *) work, (void *) info, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->dpteqr.timings[0] += (helpTimeStop - helpTime);
    data->dpteqr.calls[0]++;
    
    return;
}


/* Wrapper for dptrfs */
void hook_dptrfs(blasint *n, blasint *nrhs, double *d, double *e, double *df, double *ef, double *b, blasint *ldb, double *x, blasint *ldx, double *ferr, double *berr, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dptrfs((void *) n, (void *) nrhs, (void *) d, (void *) e, (void *) df, (void *) ef, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dptrfs.timings[0] += (helpTimeStop - helpTime);
    data->dptrfs.calls[0]++;
    
    return;
}


/* Wrapper for dptsv */
void hook_dptsv(blasint *n, blasint *nrhs, double *d, double *e, double *b, blasint *ldb, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dptsv((void *) n, (void *) nrhs, (void *) d, (void *) e, (void *) b, (void *) ldb, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dptsv.timings[0] += (helpTimeStop - helpTime);
    data->dptsv.calls[0]++;
    
    return;
}


/* Wrapper for dptsvx */
void hook_dptsvx(char *fact, blasint *n, blasint *nrhs, double *d, double *e, double *df, double *ef, double *b, blasint *ldb, double *x, blasint *ldx, double *rcond, double *ferr, double *berr, double *work, blasint *info, flexiblas_fortran_charlen_t fact_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dptsvx((void *) fact, (void *) n, (void *) nrhs, (void *) d, (void *) e, (void *) df, (void *) ef, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) info, fact_len);

    helpTimeStop = flexiblas_wtime();

    data->dptsvx.timings[0] += (helpTimeStop - helpTime);
    data->dptsvx.calls[0]++;
    
    return;
}


/* Wrapper for dpttrf */
void hook_dpttrf(blasint *n, double *d, double *e, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpttrf((void *) n, (void *) d, (void *) e, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dpttrf.timings[0] += (helpTimeStop - helpTime);
    data->dpttrf.calls[0]++;
    
    return;
}


/* Wrapper for dpttrs */
void hook_dpttrs(blasint *n, blasint *nrhs, double *d, double *e, double *b, blasint *ldb, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dpttrs((void *) n, (void *) nrhs, (void *) d, (void *) e, (void *) b, (void *) ldb, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dpttrs.timings[0] += (helpTimeStop - helpTime);
    data->dpttrs.calls[0]++;
    
    return;
}


/* Wrapper for dptts2 */
void hook_dptts2(blasint *n, blasint *nrhs, double *d, double *e, double *b, blasint *ldb)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dptts2((void *) n, (void *) nrhs, (void *) d, (void *) e, (void *) b, (void *) ldb);

    helpTimeStop = flexiblas_wtime();

    data->dptts2.timings[0] += (helpTimeStop - helpTime);
    data->dptts2.calls[0]++;
    
    return;
}


/* Wrapper for drscl */
void hook_drscl(blasint *n, double *sa, double *sx, blasint *incx)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_drscl((void *) n, (void *) sa, (void *) sx, (void *) incx);

    helpTimeStop = flexiblas_wtime();

    data->drscl.timings[0] += (helpTimeStop - helpTime);
    data->drscl.calls[0]++;
    
    return;
}


/* Wrapper for dsb2st_kernels */
void hook_dsb2st_kernels(char *uplo, blaslogical *wantz, blasint *ttype, blasint *st, blasint *ed, blasint *sweep, blasint *n, blasint *nb, blasint *ib, double *a, blasint *lda, double *v, double *tau, blasint *ldvt, double *work, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsb2st_kernels((void *) uplo, (void *) wantz, (void *) ttype, (void *) st, (void *) ed, (void *) sweep, (void *) n, (void *) nb, (void *) ib, (void *) a, (void *) lda, (void *) v, (void *) tau, (void *) ldvt, (void *) work, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsb2st_kernels.timings[0] += (helpTimeStop - helpTime);
    data->dsb2st_kernels.calls[0]++;
    
    return;
}


/* Wrapper for dsbev */
void hook_dsbev(char *jobz, char *uplo, blasint *n, blasint *kd, double *ab, blasint *ldab, double *w, double *z, blasint *ldz, double *work, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsbev((void *) jobz, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsbev.timings[0] += (helpTimeStop - helpTime);
    data->dsbev.calls[0]++;
    
    return;
}


/* Wrapper for dsbev_2stage */
void hook_dsbev_2stage(char *jobz, char *uplo, blasint *n, blasint *kd, double *ab, blasint *ldab, double *w, double *z, blasint *ldz, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsbev_2stage((void *) jobz, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsbev_2stage.timings[0] += (helpTimeStop - helpTime);
    data->dsbev_2stage.calls[0]++;
    
    return;
}


/* Wrapper for dsbevd */
void hook_dsbevd(char *jobz, char *uplo, blasint *n, blasint *kd, double *ab, blasint *ldab, double *w, double *z, blasint *ldz, double *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsbevd((void *) jobz, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsbevd.timings[0] += (helpTimeStop - helpTime);
    data->dsbevd.calls[0]++;
    
    return;
}


/* Wrapper for dsbevd_2stage */
void hook_dsbevd_2stage(char *jobz, char *uplo, blasint *n, blasint *kd, double *ab, blasint *ldab, double *w, double *z, blasint *ldz, double *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsbevd_2stage((void *) jobz, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsbevd_2stage.timings[0] += (helpTimeStop - helpTime);
    data->dsbevd_2stage.calls[0]++;
    
    return;
}


/* Wrapper for dsbevx */
void hook_dsbevx(char *jobz, char *range, char *uplo, blasint *n, blasint *kd, double *ab, blasint *ldab, double *q, blasint *ldq, double *vl, double *vu, blasint *il, blasint *iu, double *abstol, blasint *m, double *w, double *z, blasint *ldz, double *work, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsbevx((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) q, (void *) ldq, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsbevx.timings[0] += (helpTimeStop - helpTime);
    data->dsbevx.calls[0]++;
    
    return;
}


/* Wrapper for dsbevx_2stage */
void hook_dsbevx_2stage(char *jobz, char *range, char *uplo, blasint *n, blasint *kd, double *ab, blasint *ldab, double *q, blasint *ldq, double *vl, double *vu, blasint *il, blasint *iu, double *abstol, blasint *m, double *w, double *z, blasint *ldz, double *work, blasint *lwork, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsbevx_2stage((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) q, (void *) ldq, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsbevx_2stage.timings[0] += (helpTimeStop - helpTime);
    data->dsbevx_2stage.calls[0]++;
    
    return;
}


/* Wrapper for dsbgst */
void hook_dsbgst(char *vect, char *uplo, blasint *n, blasint *ka, blasint *kb, double *ab, blasint *ldab, double *bb, blasint *ldbb, double *x, blasint *ldx, double *work, blasint *info, flexiblas_fortran_charlen_t vect_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsbgst((void *) vect, (void *) uplo, (void *) n, (void *) ka, (void *) kb, (void *) ab, (void *) ldab, (void *) bb, (void *) ldbb, (void *) x, (void *) ldx, (void *) work, (void *) info, vect_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsbgst.timings[0] += (helpTimeStop - helpTime);
    data->dsbgst.calls[0]++;
    
    return;
}


/* Wrapper for dsbgv */
void hook_dsbgv(char *jobz, char *uplo, blasint *n, blasint *ka, blasint *kb, double *ab, blasint *ldab, double *bb, blasint *ldbb, double *w, double *z, blasint *ldz, double *work, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsbgv((void *) jobz, (void *) uplo, (void *) n, (void *) ka, (void *) kb, (void *) ab, (void *) ldab, (void *) bb, (void *) ldbb, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsbgv.timings[0] += (helpTimeStop - helpTime);
    data->dsbgv.calls[0]++;
    
    return;
}


/* Wrapper for dsbgvd */
void hook_dsbgvd(char *jobz, char *uplo, blasint *n, blasint *ka, blasint *kb, double *ab, blasint *ldab, double *bb, blasint *ldbb, double *w, double *z, blasint *ldz, double *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsbgvd((void *) jobz, (void *) uplo, (void *) n, (void *) ka, (void *) kb, (void *) ab, (void *) ldab, (void *) bb, (void *) ldbb, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsbgvd.timings[0] += (helpTimeStop - helpTime);
    data->dsbgvd.calls[0]++;
    
    return;
}


/* Wrapper for dsbgvx */
void hook_dsbgvx(char *jobz, char *range, char *uplo, blasint *n, blasint *ka, blasint *kb, double *ab, blasint *ldab, double *bb, blasint *ldbb, double *q, blasint *ldq, double *vl, double *vu, blasint *il, blasint *iu, double *abstol, blasint *m, double *w, double *z, blasint *ldz, double *work, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsbgvx((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) ka, (void *) kb, (void *) ab, (void *) ldab, (void *) bb, (void *) ldbb, (void *) q, (void *) ldq, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsbgvx.timings[0] += (helpTimeStop - helpTime);
    data->dsbgvx.calls[0]++;
    
    return;
}


/* Wrapper for dsbtrd */
void hook_dsbtrd(char *vect, char *uplo, blasint *n, blasint *kd, double *ab, blasint *ldab, double *d, double *e, double *q, blasint *ldq, double *work, blasint *info, flexiblas_fortran_charlen_t vect_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsbtrd((void *) vect, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) d, (void *) e, (void *) q, (void *) ldq, (void *) work, (void *) info, vect_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsbtrd.timings[0] += (helpTimeStop - helpTime);
    data->dsbtrd.calls[0]++;
    
    return;
}

/* Wrapper for dsecnd */
double hook_dsecnd(void)
{
    double helpTime;
    double helpTimeStop;
    double return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dsecnd();

    helpTimeStop = flexiblas_wtime();

    data->dsecnd.timings[0] += (helpTimeStop - helpTime);
    data->dsecnd.calls[0]++;
    
    return return_value;
}



/* Wrapper for dsfrk */
void hook_dsfrk(char *transr, char *uplo, char *trans, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *beta, double *c, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsfrk((void *) transr, (void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, transr_len, uplo_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dsfrk.timings[0] += (helpTimeStop - helpTime);
    data->dsfrk.calls[0]++;
    
    return;
}


/* Wrapper for dsgesv */
void hook_dsgesv(blasint *n, blasint *nrhs, double *a, blasint *lda, blasint *ipiv, double *b, blasint *ldb, double *x, blasint *ldx, double *work, float *swork, blasint *iter, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsgesv((void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) work, (void *) swork, (void *) iter, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dsgesv.timings[0] += (helpTimeStop - helpTime);
    data->dsgesv.calls[0]++;
    
    return;
}


/* Wrapper for dspcon */
void hook_dspcon(char *uplo, blasint *n, double *ap, blasint *ipiv, double *anorm, double *rcond, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dspcon((void *) uplo, (void *) n, (void *) ap, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dspcon.timings[0] += (helpTimeStop - helpTime);
    data->dspcon.calls[0]++;
    
    return;
}


/* Wrapper for dspev */
void hook_dspev(char *jobz, char *uplo, blasint *n, double *ap, double *w, double *z, blasint *ldz, double *work, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dspev((void *) jobz, (void *) uplo, (void *) n, (void *) ap, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dspev.timings[0] += (helpTimeStop - helpTime);
    data->dspev.calls[0]++;
    
    return;
}


/* Wrapper for dspevd */
void hook_dspevd(char *jobz, char *uplo, blasint *n, double *ap, double *w, double *z, blasint *ldz, double *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dspevd((void *) jobz, (void *) uplo, (void *) n, (void *) ap, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dspevd.timings[0] += (helpTimeStop - helpTime);
    data->dspevd.calls[0]++;
    
    return;
}


/* Wrapper for dspevx */
void hook_dspevx(char *jobz, char *range, char *uplo, blasint *n, double *ap, double *vl, double *vu, blasint *il, blasint *iu, double *abstol, blasint *m, double *w, double *z, blasint *ldz, double *work, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dspevx((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) ap, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dspevx.timings[0] += (helpTimeStop - helpTime);
    data->dspevx.calls[0]++;
    
    return;
}


/* Wrapper for dspgst */
void hook_dspgst(blasint *itype, char *uplo, blasint *n, double *ap, double *bp, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dspgst((void *) itype, (void *) uplo, (void *) n, (void *) ap, (void *) bp, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dspgst.timings[0] += (helpTimeStop - helpTime);
    data->dspgst.calls[0]++;
    
    return;
}


/* Wrapper for dspgv */
void hook_dspgv(blasint *itype, char *jobz, char *uplo, blasint *n, double *ap, double *bp, double *w, double *z, blasint *ldz, double *work, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dspgv((void *) itype, (void *) jobz, (void *) uplo, (void *) n, (void *) ap, (void *) bp, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dspgv.timings[0] += (helpTimeStop - helpTime);
    data->dspgv.calls[0]++;
    
    return;
}


/* Wrapper for dspgvd */
void hook_dspgvd(blasint *itype, char *jobz, char *uplo, blasint *n, double *ap, double *bp, double *w, double *z, blasint *ldz, double *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dspgvd((void *) itype, (void *) jobz, (void *) uplo, (void *) n, (void *) ap, (void *) bp, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dspgvd.timings[0] += (helpTimeStop - helpTime);
    data->dspgvd.calls[0]++;
    
    return;
}


/* Wrapper for dspgvx */
void hook_dspgvx(blasint *itype, char *jobz, char *range, char *uplo, blasint *n, double *ap, double *bp, double *vl, double *vu, blasint *il, blasint *iu, double *abstol, blasint *m, double *w, double *z, blasint *ldz, double *work, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dspgvx((void *) itype, (void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) ap, (void *) bp, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dspgvx.timings[0] += (helpTimeStop - helpTime);
    data->dspgvx.calls[0]++;
    
    return;
}


/* Wrapper for dsposv */
void hook_dsposv(char *uplo, blasint *n, blasint *nrhs, double *a, blasint *lda, double *b, blasint *ldb, double *x, blasint *ldx, double *work, float *swork, blasint *iter, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsposv((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) work, (void *) swork, (void *) iter, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsposv.timings[0] += (helpTimeStop - helpTime);
    data->dsposv.calls[0]++;
    
    return;
}


/* Wrapper for dsprfs */
void hook_dsprfs(char *uplo, blasint *n, blasint *nrhs, double *ap, double *afp, blasint *ipiv, double *b, blasint *ldb, double *x, blasint *ldx, double *ferr, double *berr, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsprfs((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) afp, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsprfs.timings[0] += (helpTimeStop - helpTime);
    data->dsprfs.calls[0]++;
    
    return;
}


/* Wrapper for dspsv */
void hook_dspsv(char *uplo, blasint *n, blasint *nrhs, double *ap, blasint *ipiv, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dspsv((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dspsv.timings[0] += (helpTimeStop - helpTime);
    data->dspsv.calls[0]++;
    
    return;
}


/* Wrapper for dspsvx */
void hook_dspsvx(char *fact, char *uplo, blasint *n, blasint *nrhs, double *ap, double *afp, blasint *ipiv, double *b, blasint *ldb, double *x, blasint *ldx, double *rcond, double *ferr, double *berr, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dspsvx((void *) fact, (void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) afp, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, fact_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dspsvx.timings[0] += (helpTimeStop - helpTime);
    data->dspsvx.calls[0]++;
    
    return;
}


/* Wrapper for dsptrd */
void hook_dsptrd(char *uplo, blasint *n, double *ap, double *d, double *e, double *tau, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsptrd((void *) uplo, (void *) n, (void *) ap, (void *) d, (void *) e, (void *) tau, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsptrd.timings[0] += (helpTimeStop - helpTime);
    data->dsptrd.calls[0]++;
    
    return;
}


/* Wrapper for dsptrf */
void hook_dsptrf(char *uplo, blasint *n, double *ap, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsptrf((void *) uplo, (void *) n, (void *) ap, (void *) ipiv, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsptrf.timings[0] += (helpTimeStop - helpTime);
    data->dsptrf.calls[0]++;
    
    return;
}


/* Wrapper for dsptri */
void hook_dsptri(char *uplo, blasint *n, double *ap, blasint *ipiv, double *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsptri((void *) uplo, (void *) n, (void *) ap, (void *) ipiv, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsptri.timings[0] += (helpTimeStop - helpTime);
    data->dsptri.calls[0]++;
    
    return;
}


/* Wrapper for dsptrs */
void hook_dsptrs(char *uplo, blasint *n, blasint *nrhs, double *ap, blasint *ipiv, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsptrs((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsptrs.timings[0] += (helpTimeStop - helpTime);
    data->dsptrs.calls[0]++;
    
    return;
}


/* Wrapper for dstebz */
void hook_dstebz(char *range, char *order, blasint *n, double *vl, double *vu, blasint *il, blasint *iu, double *abstol, double *d, double *e, blasint *m, blasint *nsplit, double *w, blasint *iblock, blasint *isplit, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t order_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dstebz((void *) range, (void *) order, (void *) n, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) d, (void *) e, (void *) m, (void *) nsplit, (void *) w, (void *) iblock, (void *) isplit, (void *) work, (void *) iwork, (void *) info, range_len, order_len);

    helpTimeStop = flexiblas_wtime();

    data->dstebz.timings[0] += (helpTimeStop - helpTime);
    data->dstebz.calls[0]++;
    
    return;
}


/* Wrapper for dstedc */
void hook_dstedc(char *compz, blasint *n, double *d, double *e, double *z, blasint *ldz, double *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dstedc((void *) compz, (void *) n, (void *) d, (void *) e, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->dstedc.timings[0] += (helpTimeStop - helpTime);
    data->dstedc.calls[0]++;
    
    return;
}


/* Wrapper for dstegr */
void hook_dstegr(char *jobz, char *range, blasint *n, double *d, double *e, double *vl, double *vu, blasint *il, blasint *iu, double *abstol, blasint *m, double *w, double *z, blasint *ldz, blasint *isuppz, double *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dstegr((void *) jobz, (void *) range, (void *) n, (void *) d, (void *) e, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) isuppz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, range_len);

    helpTimeStop = flexiblas_wtime();

    data->dstegr.timings[0] += (helpTimeStop - helpTime);
    data->dstegr.calls[0]++;
    
    return;
}


/* Wrapper for dstein */
void hook_dstein(blasint *n, double *d, double *e, blasint *m, double *w, blasint *iblock, blasint *isplit, double *z, blasint *ldz, double *work, blasint *iwork, blasint *ifail, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dstein((void *) n, (void *) d, (void *) e, (void *) m, (void *) w, (void *) iblock, (void *) isplit, (void *) z, (void *) ldz, (void *) work, (void *) iwork, (void *) ifail, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dstein.timings[0] += (helpTimeStop - helpTime);
    data->dstein.calls[0]++;
    
    return;
}


/* Wrapper for dstemr */
void hook_dstemr(char *jobz, char *range, blasint *n, double *d, double *e, double *vl, double *vu, blasint *il, blasint *iu, blasint *m, double *w, double *z, blasint *ldz, blasint *nzc, blasint *isuppz, blaslogical *tryrac, double *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dstemr((void *) jobz, (void *) range, (void *) n, (void *) d, (void *) e, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) nzc, (void *) isuppz, (void *) tryrac, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, range_len);

    helpTimeStop = flexiblas_wtime();

    data->dstemr.timings[0] += (helpTimeStop - helpTime);
    data->dstemr.calls[0]++;
    
    return;
}


/* Wrapper for dsteqr */
void hook_dsteqr(char *compz, blasint *n, double *d, double *e, double *z, blasint *ldz, double *work, blasint *info, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsteqr((void *) compz, (void *) n, (void *) d, (void *) e, (void *) z, (void *) ldz, (void *) work, (void *) info, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->dsteqr.timings[0] += (helpTimeStop - helpTime);
    data->dsteqr.calls[0]++;
    
    return;
}


/* Wrapper for dsterf */
void hook_dsterf(blasint *n, double *d, double *e, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsterf((void *) n, (void *) d, (void *) e, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dsterf.timings[0] += (helpTimeStop - helpTime);
    data->dsterf.calls[0]++;
    
    return;
}


/* Wrapper for dstev */
void hook_dstev(char *jobz, blasint *n, double *d, double *e, double *z, blasint *ldz, double *work, blasint *info, flexiblas_fortran_charlen_t jobz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dstev((void *) jobz, (void *) n, (void *) d, (void *) e, (void *) z, (void *) ldz, (void *) work, (void *) info, jobz_len);

    helpTimeStop = flexiblas_wtime();

    data->dstev.timings[0] += (helpTimeStop - helpTime);
    data->dstev.calls[0]++;
    
    return;
}


/* Wrapper for dstevd */
void hook_dstevd(char *jobz, blasint *n, double *d, double *e, double *z, blasint *ldz, double *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dstevd((void *) jobz, (void *) n, (void *) d, (void *) e, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len);

    helpTimeStop = flexiblas_wtime();

    data->dstevd.timings[0] += (helpTimeStop - helpTime);
    data->dstevd.calls[0]++;
    
    return;
}


/* Wrapper for dstevr */
void hook_dstevr(char *jobz, char *range, blasint *n, double *d, double *e, double *vl, double *vu, blasint *il, blasint *iu, double *abstol, blasint *m, double *w, double *z, blasint *ldz, blasint *isuppz, double *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dstevr((void *) jobz, (void *) range, (void *) n, (void *) d, (void *) e, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) isuppz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, range_len);

    helpTimeStop = flexiblas_wtime();

    data->dstevr.timings[0] += (helpTimeStop - helpTime);
    data->dstevr.calls[0]++;
    
    return;
}


/* Wrapper for dstevx */
void hook_dstevx(char *jobz, char *range, blasint *n, double *d, double *e, double *vl, double *vu, blasint *il, blasint *iu, double *abstol, blasint *m, double *w, double *z, blasint *ldz, double *work, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dstevx((void *) jobz, (void *) range, (void *) n, (void *) d, (void *) e, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len);

    helpTimeStop = flexiblas_wtime();

    data->dstevx.timings[0] += (helpTimeStop - helpTime);
    data->dstevx.calls[0]++;
    
    return;
}


/* Wrapper for dsycon */
void hook_dsycon(char *uplo, blasint *n, double *a, blasint *lda, blasint *ipiv, double *anorm, double *rcond, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsycon((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsycon.timings[0] += (helpTimeStop - helpTime);
    data->dsycon.calls[0]++;
    
    return;
}


/* Wrapper for dsycon_3 */
void hook_dsycon_3(char *uplo, blasint *n, double *a, blasint *lda, double *e, blasint *ipiv, double *anorm, double *rcond, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsycon_3((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsycon_3.timings[0] += (helpTimeStop - helpTime);
    data->dsycon_3.calls[0]++;
    
    return;
}


/* Wrapper for dsycon_rook */
void hook_dsycon_rook(char *uplo, blasint *n, double *a, blasint *lda, blasint *ipiv, double *anorm, double *rcond, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsycon_rook((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsycon_rook.timings[0] += (helpTimeStop - helpTime);
    data->dsycon_rook.calls[0]++;
    
    return;
}


/* Wrapper for dsyconv */
void hook_dsyconv(char *uplo, char *way, blasint *n, double *a, blasint *lda, blasint *ipiv, double *e, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t way_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsyconv((void *) uplo, (void *) way, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) e, (void *) info, uplo_len, way_len);

    helpTimeStop = flexiblas_wtime();

    data->dsyconv.timings[0] += (helpTimeStop - helpTime);
    data->dsyconv.calls[0]++;
    
    return;
}


/* Wrapper for dsyconvf */
void hook_dsyconvf(char *uplo, char *way, blasint *n, double *a, blasint *lda, double *e, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t way_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsyconvf((void *) uplo, (void *) way, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) info, uplo_len, way_len);

    helpTimeStop = flexiblas_wtime();

    data->dsyconvf.timings[0] += (helpTimeStop - helpTime);
    data->dsyconvf.calls[0]++;
    
    return;
}


/* Wrapper for dsyconvf_rook */
void hook_dsyconvf_rook(char *uplo, char *way, blasint *n, double *a, blasint *lda, double *e, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t way_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsyconvf_rook((void *) uplo, (void *) way, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) info, uplo_len, way_len);

    helpTimeStop = flexiblas_wtime();

    data->dsyconvf_rook.timings[0] += (helpTimeStop - helpTime);
    data->dsyconvf_rook.calls[0]++;
    
    return;
}


/* Wrapper for dsyequb */
void hook_dsyequb(char *uplo, blasint *n, double *a, blasint *lda, double *s, double *scond, double *amax, double *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsyequb((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) s, (void *) scond, (void *) amax, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsyequb.timings[0] += (helpTimeStop - helpTime);
    data->dsyequb.calls[0]++;
    
    return;
}


/* Wrapper for dsyev */
void hook_dsyev(char *jobz, char *uplo, blasint *n, double *a, blasint *lda, double *w, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsyev((void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) w, (void *) work, (void *) lwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsyev.timings[0] += (helpTimeStop - helpTime);
    data->dsyev.calls[0]++;
    
    return;
}


/* Wrapper for dsyev_2stage */
void hook_dsyev_2stage(char *jobz, char *uplo, blasint *n, double *a, blasint *lda, double *w, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsyev_2stage((void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) w, (void *) work, (void *) lwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsyev_2stage.timings[0] += (helpTimeStop - helpTime);
    data->dsyev_2stage.calls[0]++;
    
    return;
}


/* Wrapper for dsyevd */
void hook_dsyevd(char *jobz, char *uplo, blasint *n, double *a, blasint *lda, double *w, double *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsyevd((void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) w, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsyevd.timings[0] += (helpTimeStop - helpTime);
    data->dsyevd.calls[0]++;
    
    return;
}


/* Wrapper for dsyevd_2stage */
void hook_dsyevd_2stage(char *jobz, char *uplo, blasint *n, double *a, blasint *lda, double *w, double *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsyevd_2stage((void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) w, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsyevd_2stage.timings[0] += (helpTimeStop - helpTime);
    data->dsyevd_2stage.calls[0]++;
    
    return;
}


/* Wrapper for dsyevr */
void hook_dsyevr(char *jobz, char *range, char *uplo, blasint *n, double *a, blasint *lda, double *vl, double *vu, blasint *il, blasint *iu, double *abstol, blasint *m, double *w, double *z, blasint *ldz, blasint *isuppz, double *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsyevr((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) isuppz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsyevr.timings[0] += (helpTimeStop - helpTime);
    data->dsyevr.calls[0]++;
    
    return;
}


/* Wrapper for dsyevr_2stage */
void hook_dsyevr_2stage(char *jobz, char *range, char *uplo, blasint *n, double *a, blasint *lda, double *vl, double *vu, blasint *il, blasint *iu, double *abstol, blasint *m, double *w, double *z, blasint *ldz, blasint *isuppz, double *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsyevr_2stage((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) isuppz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsyevr_2stage.timings[0] += (helpTimeStop - helpTime);
    data->dsyevr_2stage.calls[0]++;
    
    return;
}


/* Wrapper for dsyevx */
void hook_dsyevx(char *jobz, char *range, char *uplo, blasint *n, double *a, blasint *lda, double *vl, double *vu, blasint *il, blasint *iu, double *abstol, blasint *m, double *w, double *z, blasint *ldz, double *work, blasint *lwork, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsyevx((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsyevx.timings[0] += (helpTimeStop - helpTime);
    data->dsyevx.calls[0]++;
    
    return;
}


/* Wrapper for dsyevx_2stage */
void hook_dsyevx_2stage(char *jobz, char *range, char *uplo, blasint *n, double *a, blasint *lda, double *vl, double *vu, blasint *il, blasint *iu, double *abstol, blasint *m, double *w, double *z, blasint *ldz, double *work, blasint *lwork, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsyevx_2stage((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsyevx_2stage.timings[0] += (helpTimeStop - helpTime);
    data->dsyevx_2stage.calls[0]++;
    
    return;
}


/* Wrapper for dsygs2 */
void hook_dsygs2(blasint *itype, char *uplo, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsygs2((void *) itype, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsygs2.timings[0] += (helpTimeStop - helpTime);
    data->dsygs2.calls[0]++;
    
    return;
}


/* Wrapper for dsygst */
void hook_dsygst(blasint *itype, char *uplo, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsygst((void *) itype, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsygst.timings[0] += (helpTimeStop - helpTime);
    data->dsygst.calls[0]++;
    
    return;
}


/* Wrapper for dsygv */
void hook_dsygv(blasint *itype, char *jobz, char *uplo, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, double *w, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsygv((void *) itype, (void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) w, (void *) work, (void *) lwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsygv.timings[0] += (helpTimeStop - helpTime);
    data->dsygv.calls[0]++;
    
    return;
}


/* Wrapper for dsygv_2stage */
void hook_dsygv_2stage(blasint *itype, char *jobz, char *uplo, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, double *w, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsygv_2stage((void *) itype, (void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) w, (void *) work, (void *) lwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsygv_2stage.timings[0] += (helpTimeStop - helpTime);
    data->dsygv_2stage.calls[0]++;
    
    return;
}


/* Wrapper for dsygvd */
void hook_dsygvd(blasint *itype, char *jobz, char *uplo, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, double *w, double *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsygvd((void *) itype, (void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) w, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsygvd.timings[0] += (helpTimeStop - helpTime);
    data->dsygvd.calls[0]++;
    
    return;
}


/* Wrapper for dsygvx */
void hook_dsygvx(blasint *itype, char *jobz, char *range, char *uplo, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, double *vl, double *vu, blasint *il, blasint *iu, double *abstol, blasint *m, double *w, double *z, blasint *ldz, double *work, blasint *lwork, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsygvx((void *) itype, (void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsygvx.timings[0] += (helpTimeStop - helpTime);
    data->dsygvx.calls[0]++;
    
    return;
}


/* Wrapper for dsyrfs */
void hook_dsyrfs(char *uplo, blasint *n, blasint *nrhs, double *a, blasint *lda, double *af, blasint *ldaf, blasint *ipiv, double *b, blasint *ldb, double *x, blasint *ldx, double *ferr, double *berr, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsyrfs((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) af, (void *) ldaf, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsyrfs.timings[0] += (helpTimeStop - helpTime);
    data->dsyrfs.calls[0]++;
    
    return;
}


/* Wrapper for dsysv */
void hook_dsysv(char *uplo, blasint *n, blasint *nrhs, double *a, blasint *lda, blasint *ipiv, double *b, blasint *ldb, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsysv((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsysv.timings[0] += (helpTimeStop - helpTime);
    data->dsysv.calls[0]++;
    
    return;
}


/* Wrapper for dsysv_aa */
void hook_dsysv_aa(char *uplo, blasint *n, blasint *nrhs, double *a, blasint *lda, blasint *ipiv, double *b, blasint *ldb, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsysv_aa((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsysv_aa.timings[0] += (helpTimeStop - helpTime);
    data->dsysv_aa.calls[0]++;
    
    return;
}


/* Wrapper for dsysv_aa_2stage */
void hook_dsysv_aa_2stage(char *uplo, blasint *n, blasint *nrhs, double *a, blasint *lda, double *tb, blasint *ltb, blasint *ipiv, blasint *ipiv2, double *b, blasint *ldb, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsysv_aa_2stage((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) tb, (void *) ltb, (void *) ipiv, (void *) ipiv2, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsysv_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->dsysv_aa_2stage.calls[0]++;
    
    return;
}


/* Wrapper for dsysv_rk */
void hook_dsysv_rk(char *uplo, blasint *n, blasint *nrhs, double *a, blasint *lda, double *e, blasint *ipiv, double *b, blasint *ldb, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsysv_rk((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsysv_rk.timings[0] += (helpTimeStop - helpTime);
    data->dsysv_rk.calls[0]++;
    
    return;
}


/* Wrapper for dsysv_rook */
void hook_dsysv_rook(char *uplo, blasint *n, blasint *nrhs, double *a, blasint *lda, blasint *ipiv, double *b, blasint *ldb, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsysv_rook((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsysv_rook.timings[0] += (helpTimeStop - helpTime);
    data->dsysv_rook.calls[0]++;
    
    return;
}


/* Wrapper for dsysvx */
void hook_dsysvx(char *fact, char *uplo, blasint *n, blasint *nrhs, double *a, blasint *lda, double *af, blasint *ldaf, blasint *ipiv, double *b, blasint *ldb, double *x, blasint *ldx, double *rcond, double *ferr, double *berr, double *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsysvx((void *) fact, (void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) af, (void *) ldaf, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) lwork, (void *) iwork, (void *) info, fact_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsysvx.timings[0] += (helpTimeStop - helpTime);
    data->dsysvx.calls[0]++;
    
    return;
}


/* Wrapper for dsyswapr */
void hook_dsyswapr(char *uplo, blasint *n, double *a, blasint *lda, blasint *i1, blasint *i2, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsyswapr((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) i1, (void *) i2, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsyswapr.timings[0] += (helpTimeStop - helpTime);
    data->dsyswapr.calls[0]++;
    
    return;
}


/* Wrapper for dsytd2 */
void hook_dsytd2(char *uplo, blasint *n, double *a, blasint *lda, double *d, double *e, double *tau, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytd2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) e, (void *) tau, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytd2.timings[0] += (helpTimeStop - helpTime);
    data->dsytd2.calls[0]++;
    
    return;
}


/* Wrapper for dsytf2 */
void hook_dsytf2(char *uplo, blasint *n, double *a, blasint *lda, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytf2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytf2.timings[0] += (helpTimeStop - helpTime);
    data->dsytf2.calls[0]++;
    
    return;
}


/* Wrapper for dsytf2_rk */
void hook_dsytf2_rk(char *uplo, blasint *n, double *a, blasint *lda, double *e, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytf2_rk((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytf2_rk.timings[0] += (helpTimeStop - helpTime);
    data->dsytf2_rk.calls[0]++;
    
    return;
}


/* Wrapper for dsytf2_rook */
void hook_dsytf2_rook(char *uplo, blasint *n, double *a, blasint *lda, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytf2_rook((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytf2_rook.timings[0] += (helpTimeStop - helpTime);
    data->dsytf2_rook.calls[0]++;
    
    return;
}


/* Wrapper for dsytrd */
void hook_dsytrd(char *uplo, blasint *n, double *a, blasint *lda, double *d, double *e, double *tau, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytrd((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) e, (void *) tau, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytrd.timings[0] += (helpTimeStop - helpTime);
    data->dsytrd.calls[0]++;
    
    return;
}


/* Wrapper for dsytrd_2stage */
void hook_dsytrd_2stage(char *vect, char *uplo, blasint *n, double *a, blasint *lda, double *d, double *e, double *tau, double *hous2, blasint *lhous2, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t vect_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytrd_2stage((void *) vect, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) e, (void *) tau, (void *) hous2, (void *) lhous2, (void *) work, (void *) lwork, (void *) info, vect_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytrd_2stage.timings[0] += (helpTimeStop - helpTime);
    data->dsytrd_2stage.calls[0]++;
    
    return;
}


/* Wrapper for dsytrd_sb2st */
void hook_dsytrd_sb2st(char *stage1, char *vect, char *uplo, blasint *n, blasint *kd, double *ab, blasint *ldab, double *d, double *e, double *hous, blasint *lhous, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t stage1_len, flexiblas_fortran_charlen_t vect_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytrd_sb2st((void *) stage1, (void *) vect, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) d, (void *) e, (void *) hous, (void *) lhous, (void *) work, (void *) lwork, (void *) info, stage1_len, vect_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytrd_sb2st.timings[0] += (helpTimeStop - helpTime);
    data->dsytrd_sb2st.calls[0]++;
    
    return;
}


/* Wrapper for dsytrd_sy2sb */
void hook_dsytrd_sy2sb(char *uplo, blasint *n, blasint *kd, double *a, blasint *lda, double *ab, blasint *ldab, double *tau, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytrd_sy2sb((void *) uplo, (void *) n, (void *) kd, (void *) a, (void *) lda, (void *) ab, (void *) ldab, (void *) tau, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytrd_sy2sb.timings[0] += (helpTimeStop - helpTime);
    data->dsytrd_sy2sb.calls[0]++;
    
    return;
}


/* Wrapper for dsytrf */
void hook_dsytrf(char *uplo, blasint *n, double *a, blasint *lda, blasint *ipiv, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytrf((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytrf.timings[0] += (helpTimeStop - helpTime);
    data->dsytrf.calls[0]++;
    
    return;
}


/* Wrapper for dsytrf_aa */
void hook_dsytrf_aa(char *uplo, blasint *n, double *a, blasint *lda, blasint *ipiv, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytrf_aa((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytrf_aa.timings[0] += (helpTimeStop - helpTime);
    data->dsytrf_aa.calls[0]++;
    
    return;
}


/* Wrapper for dsytrf_aa_2stage */
void hook_dsytrf_aa_2stage(char *uplo, blasint *n, double *a, blasint *lda, double *tb, blasint *ltb, blasint *ipiv, blasint *ipiv2, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytrf_aa_2stage((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) tb, (void *) ltb, (void *) ipiv, (void *) ipiv2, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytrf_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->dsytrf_aa_2stage.calls[0]++;
    
    return;
}


/* Wrapper for dsytrf_rk */
void hook_dsytrf_rk(char *uplo, blasint *n, double *a, blasint *lda, double *e, blasint *ipiv, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytrf_rk((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytrf_rk.timings[0] += (helpTimeStop - helpTime);
    data->dsytrf_rk.calls[0]++;
    
    return;
}


/* Wrapper for dsytrf_rook */
void hook_dsytrf_rook(char *uplo, blasint *n, double *a, blasint *lda, blasint *ipiv, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytrf_rook((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytrf_rook.timings[0] += (helpTimeStop - helpTime);
    data->dsytrf_rook.calls[0]++;
    
    return;
}


/* Wrapper for dsytri */
void hook_dsytri(char *uplo, blasint *n, double *a, blasint *lda, blasint *ipiv, double *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytri((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytri.timings[0] += (helpTimeStop - helpTime);
    data->dsytri.calls[0]++;
    
    return;
}


/* Wrapper for dsytri2 */
void hook_dsytri2(char *uplo, blasint *n, double *a, blasint *lda, blasint *ipiv, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytri2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytri2.timings[0] += (helpTimeStop - helpTime);
    data->dsytri2.calls[0]++;
    
    return;
}


/* Wrapper for dsytri2x */
void hook_dsytri2x(char *uplo, blasint *n, double *a, blasint *lda, blasint *ipiv, double *work, blasint *nb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytri2x((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) nb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytri2x.timings[0] += (helpTimeStop - helpTime);
    data->dsytri2x.calls[0]++;
    
    return;
}


/* Wrapper for dsytri_3 */
void hook_dsytri_3(char *uplo, blasint *n, double *a, blasint *lda, double *e, blasint *ipiv, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytri_3((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytri_3.timings[0] += (helpTimeStop - helpTime);
    data->dsytri_3.calls[0]++;
    
    return;
}


/* Wrapper for dsytri_3x */
void hook_dsytri_3x(char *uplo, blasint *n, double *a, blasint *lda, double *e, blasint *ipiv, double *work, blasint *nb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytri_3x((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) work, (void *) nb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytri_3x.timings[0] += (helpTimeStop - helpTime);
    data->dsytri_3x.calls[0]++;
    
    return;
}


/* Wrapper for dsytri_rook */
void hook_dsytri_rook(char *uplo, blasint *n, double *a, blasint *lda, blasint *ipiv, double *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytri_rook((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytri_rook.timings[0] += (helpTimeStop - helpTime);
    data->dsytri_rook.calls[0]++;
    
    return;
}


/* Wrapper for dsytrs */
void hook_dsytrs(char *uplo, blasint *n, blasint *nrhs, double *a, blasint *lda, blasint *ipiv, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytrs((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytrs.timings[0] += (helpTimeStop - helpTime);
    data->dsytrs.calls[0]++;
    
    return;
}


/* Wrapper for dsytrs2 */
void hook_dsytrs2(char *uplo, blasint *n, blasint *nrhs, double *a, blasint *lda, blasint *ipiv, double *b, blasint *ldb, double *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytrs2((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytrs2.timings[0] += (helpTimeStop - helpTime);
    data->dsytrs2.calls[0]++;
    
    return;
}


/* Wrapper for dsytrs_3 */
void hook_dsytrs_3(char *uplo, blasint *n, blasint *nrhs, double *a, blasint *lda, double *e, blasint *ipiv, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytrs_3((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytrs_3.timings[0] += (helpTimeStop - helpTime);
    data->dsytrs_3.calls[0]++;
    
    return;
}


/* Wrapper for dsytrs_aa */
void hook_dsytrs_aa(char *uplo, blasint *n, blasint *nrhs, double *a, blasint *lda, blasint *ipiv, double *b, blasint *ldb, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytrs_aa((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytrs_aa.timings[0] += (helpTimeStop - helpTime);
    data->dsytrs_aa.calls[0]++;
    
    return;
}


/* Wrapper for dsytrs_aa_2stage */
void hook_dsytrs_aa_2stage(char *uplo, blasint *n, blasint *nrhs, double *a, blasint *lda, double *tb, blasint *ltb, blasint *ipiv, blasint *ipiv2, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytrs_aa_2stage((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) tb, (void *) ltb, (void *) ipiv, (void *) ipiv2, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytrs_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->dsytrs_aa_2stage.calls[0]++;
    
    return;
}


/* Wrapper for dsytrs_rook */
void hook_dsytrs_rook(char *uplo, blasint *n, blasint *nrhs, double *a, blasint *lda, blasint *ipiv, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dsytrs_rook((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dsytrs_rook.timings[0] += (helpTimeStop - helpTime);
    data->dsytrs_rook.calls[0]++;
    
    return;
}


/* Wrapper for dtbcon */
void hook_dtbcon(char *norm, char *uplo, char *diag, blasint *n, blasint *kd, double *ab, blasint *ldab, double *rcond, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtbcon((void *) norm, (void *) uplo, (void *) diag, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) rcond, (void *) work, (void *) iwork, (void *) info, norm_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->dtbcon.timings[0] += (helpTimeStop - helpTime);
    data->dtbcon.calls[0]++;
    
    return;
}


/* Wrapper for dtbrfs */
void hook_dtbrfs(char *uplo, char *trans, char *diag, blasint *n, blasint *kd, blasint *nrhs, double *ab, blasint *ldab, double *b, blasint *ldb, double *x, blasint *ldx, double *ferr, double *berr, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtbrfs((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) kd, (void *) nrhs, (void *) ab, (void *) ldab, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->dtbrfs.timings[0] += (helpTimeStop - helpTime);
    data->dtbrfs.calls[0]++;
    
    return;
}


/* Wrapper for dtbtrs */
void hook_dtbtrs(char *uplo, char *trans, char *diag, blasint *n, blasint *kd, blasint *nrhs, double *ab, blasint *ldab, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtbtrs((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) kd, (void *) nrhs, (void *) ab, (void *) ldab, (void *) b, (void *) ldb, (void *) info, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->dtbtrs.timings[0] += (helpTimeStop - helpTime);
    data->dtbtrs.calls[0]++;
    
    return;
}


/* Wrapper for dtfsm */
void hook_dtfsm(char *transr, char *side, char *uplo, char *trans, char *diag, blasint *m, blasint *n, double *alpha, double *a, double *b, blasint *ldb, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtfsm((void *) transr, (void *) side, (void *) uplo, (void *) trans, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) b, (void *) ldb, transr_len, side_len, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->dtfsm.timings[0] += (helpTimeStop - helpTime);
    data->dtfsm.calls[0]++;
    
    return;
}


/* Wrapper for dtftri */
void hook_dtftri(char *transr, char *uplo, char *diag, blasint *n, double *a, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtftri((void *) transr, (void *) uplo, (void *) diag, (void *) n, (void *) a, (void *) info, transr_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->dtftri.timings[0] += (helpTimeStop - helpTime);
    data->dtftri.calls[0]++;
    
    return;
}


/* Wrapper for dtfttp */
void hook_dtfttp(char *transr, char *uplo, blasint *n, double *arf, double *ap, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtfttp((void *) transr, (void *) uplo, (void *) n, (void *) arf, (void *) ap, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dtfttp.timings[0] += (helpTimeStop - helpTime);
    data->dtfttp.calls[0]++;
    
    return;
}


/* Wrapper for dtfttr */
void hook_dtfttr(char *transr, char *uplo, blasint *n, double *arf, double *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtfttr((void *) transr, (void *) uplo, (void *) n, (void *) arf, (void *) a, (void *) lda, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dtfttr.timings[0] += (helpTimeStop - helpTime);
    data->dtfttr.calls[0]++;
    
    return;
}


/* Wrapper for dtgevc */
void hook_dtgevc(char *side, char *howmny, blaslogical *select, blasint *n, double *s, blasint *lds, double *p, blasint *ldp, double *vl, blasint *ldvl, double *vr, blasint *ldvr, blasint *mm, blasint *m, double *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t howmny_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtgevc((void *) side, (void *) howmny, (void *) select, (void *) n, (void *) s, (void *) lds, (void *) p, (void *) ldp, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) mm, (void *) m, (void *) work, (void *) info, side_len, howmny_len);

    helpTimeStop = flexiblas_wtime();

    data->dtgevc.timings[0] += (helpTimeStop - helpTime);
    data->dtgevc.calls[0]++;
    
    return;
}


/* Wrapper for dtgex2 */
void hook_dtgex2(blaslogical *wantq, blaslogical *wantz, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, double *q, blasint *ldq, double *z, blasint *ldz, blasint *j1, blasint *n1, blasint *n2, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtgex2((void *) wantq, (void *) wantz, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) q, (void *) ldq, (void *) z, (void *) ldz, (void *) j1, (void *) n1, (void *) n2, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dtgex2.timings[0] += (helpTimeStop - helpTime);
    data->dtgex2.calls[0]++;
    
    return;
}


/* Wrapper for dtgexc */
void hook_dtgexc(blaslogical *wantq, blaslogical *wantz, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, double *q, blasint *ldq, double *z, blasint *ldz, blasint *ifst, blasint *ilst, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtgexc((void *) wantq, (void *) wantz, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) q, (void *) ldq, (void *) z, (void *) ldz, (void *) ifst, (void *) ilst, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dtgexc.timings[0] += (helpTimeStop - helpTime);
    data->dtgexc.calls[0]++;
    
    return;
}


/* Wrapper for dtgsen */
void hook_dtgsen(blasint *ijob, blaslogical *wantq, blaslogical *wantz, blaslogical *select, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, double *alphar, double *alphai, double *beta, double *q, blasint *ldq, double *z, blasint *ldz, blasint *m, double *pl, double *pr, double *dif, double *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtgsen((void *) ijob, (void *) wantq, (void *) wantz, (void *) select, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) alphar, (void *) alphai, (void *) beta, (void *) q, (void *) ldq, (void *) z, (void *) ldz, (void *) m, (void *) pl, (void *) pr, (void *) dif, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dtgsen.timings[0] += (helpTimeStop - helpTime);
    data->dtgsen.calls[0]++;
    
    return;
}


/* Wrapper for dtgsja */
void hook_dtgsja(char *jobu, char *jobv, char *jobq, blasint *m, blasint *p, blasint *n, blasint *k, blasint *l, double *a, blasint *lda, double *b, blasint *ldb, double *tola, double *tolb, double *alpha, double *beta, double *u, blasint *ldu, double *v, blasint *ldv, double *q, blasint *ldq, double *work, blasint *ncycle, blasint *info, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobv_len, flexiblas_fortran_charlen_t jobq_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtgsja((void *) jobu, (void *) jobv, (void *) jobq, (void *) m, (void *) p, (void *) n, (void *) k, (void *) l, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) tola, (void *) tolb, (void *) alpha, (void *) beta, (void *) u, (void *) ldu, (void *) v, (void *) ldv, (void *) q, (void *) ldq, (void *) work, (void *) ncycle, (void *) info, jobu_len, jobv_len, jobq_len);

    helpTimeStop = flexiblas_wtime();

    data->dtgsja.timings[0] += (helpTimeStop - helpTime);
    data->dtgsja.calls[0]++;
    
    return;
}


/* Wrapper for dtgsna */
void hook_dtgsna(char *job, char *howmny, blaslogical *select, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, double *vl, blasint *ldvl, double *vr, blasint *ldvr, double *s, double *dif, blasint *mm, blasint *m, double *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t howmny_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtgsna((void *) job, (void *) howmny, (void *) select, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) s, (void *) dif, (void *) mm, (void *) m, (void *) work, (void *) lwork, (void *) iwork, (void *) info, job_len, howmny_len);

    helpTimeStop = flexiblas_wtime();

    data->dtgsna.timings[0] += (helpTimeStop - helpTime);
    data->dtgsna.calls[0]++;
    
    return;
}


/* Wrapper for dtgsy2 */
void hook_dtgsy2(char *trans, blasint *ijob, blasint *m, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, double *c, blasint *ldc, double *d, blasint *ldd, double *e, blasint *lde, double *f, blasint *ldf, double *scale, double *rdsum, double *rdscal, blasint *iwork, blasint *pq, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtgsy2((void *) trans, (void *) ijob, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) c, (void *) ldc, (void *) d, (void *) ldd, (void *) e, (void *) lde, (void *) f, (void *) ldf, (void *) scale, (void *) rdsum, (void *) rdscal, (void *) iwork, (void *) pq, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dtgsy2.timings[0] += (helpTimeStop - helpTime);
    data->dtgsy2.calls[0]++;
    
    return;
}


/* Wrapper for dtgsyl */
void hook_dtgsyl(char *trans, blasint *ijob, blasint *m, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, double *c, blasint *ldc, double *d, blasint *ldd, double *e, blasint *lde, double *f, blasint *ldf, double *scale, double *dif, double *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtgsyl((void *) trans, (void *) ijob, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) c, (void *) ldc, (void *) d, (void *) ldd, (void *) e, (void *) lde, (void *) f, (void *) ldf, (void *) scale, (void *) dif, (void *) work, (void *) lwork, (void *) iwork, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dtgsyl.timings[0] += (helpTimeStop - helpTime);
    data->dtgsyl.calls[0]++;
    
    return;
}


/* Wrapper for dtpcon */
void hook_dtpcon(char *norm, char *uplo, char *diag, blasint *n, double *ap, double *rcond, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtpcon((void *) norm, (void *) uplo, (void *) diag, (void *) n, (void *) ap, (void *) rcond, (void *) work, (void *) iwork, (void *) info, norm_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->dtpcon.timings[0] += (helpTimeStop - helpTime);
    data->dtpcon.calls[0]++;
    
    return;
}


/* Wrapper for dtplqt */
void hook_dtplqt(blasint *m, blasint *n, blasint *l, blasint *mb, double *a, blasint *lda, double *b, blasint *ldb, double *t, blasint *ldt, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtplqt((void *) m, (void *) n, (void *) l, (void *) mb, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) t, (void *) ldt, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dtplqt.timings[0] += (helpTimeStop - helpTime);
    data->dtplqt.calls[0]++;
    
    return;
}


/* Wrapper for dtplqt2 */
void hook_dtplqt2(blasint *m, blasint *n, blasint *l, double *a, blasint *lda, double *b, blasint *ldb, double *t, blasint *ldt, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtplqt2((void *) m, (void *) n, (void *) l, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) t, (void *) ldt, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dtplqt2.timings[0] += (helpTimeStop - helpTime);
    data->dtplqt2.calls[0]++;
    
    return;
}


/* Wrapper for dtpmlqt */
void hook_dtpmlqt(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *l, blasint *mb, double *v, blasint *ldv, double *t, blasint *ldt, double *a, blasint *lda, double *b, blasint *ldb, double *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtpmlqt((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) l, (void *) mb, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dtpmlqt.timings[0] += (helpTimeStop - helpTime);
    data->dtpmlqt.calls[0]++;
    
    return;
}


/* Wrapper for dtpmqrt */
void hook_dtpmqrt(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *l, blasint *nb, double *v, blasint *ldv, double *t, blasint *ldt, double *a, blasint *lda, double *b, blasint *ldb, double *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtpmqrt((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) l, (void *) nb, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->dtpmqrt.timings[0] += (helpTimeStop - helpTime);
    data->dtpmqrt.calls[0]++;
    
    return;
}


/* Wrapper for dtpqrt */
void hook_dtpqrt(blasint *m, blasint *n, blasint *l, blasint *nb, double *a, blasint *lda, double *b, blasint *ldb, double *t, blasint *ldt, double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtpqrt((void *) m, (void *) n, (void *) l, (void *) nb, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) t, (void *) ldt, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dtpqrt.timings[0] += (helpTimeStop - helpTime);
    data->dtpqrt.calls[0]++;
    
    return;
}


/* Wrapper for dtpqrt2 */
void hook_dtpqrt2(blasint *m, blasint *n, blasint *l, double *a, blasint *lda, double *b, blasint *ldb, double *t, blasint *ldt, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtpqrt2((void *) m, (void *) n, (void *) l, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) t, (void *) ldt, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dtpqrt2.timings[0] += (helpTimeStop - helpTime);
    data->dtpqrt2.calls[0]++;
    
    return;
}


/* Wrapper for dtprfb */
void hook_dtprfb(char *side, char *trans, char *direct, char *storev, blasint *m, blasint *n, blasint *k, blasint *l, double *v, blasint *ldv, double *t, blasint *ldt, double *a, blasint *lda, double *b, blasint *ldb, double *work, blasint *ldwork, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t direct_len, flexiblas_fortran_charlen_t storev_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtprfb((void *) side, (void *) trans, (void *) direct, (void *) storev, (void *) m, (void *) n, (void *) k, (void *) l, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) work, (void *) ldwork, side_len, trans_len, direct_len, storev_len);

    helpTimeStop = flexiblas_wtime();

    data->dtprfb.timings[0] += (helpTimeStop - helpTime);
    data->dtprfb.calls[0]++;
    
    return;
}


/* Wrapper for dtprfs */
void hook_dtprfs(char *uplo, char *trans, char *diag, blasint *n, blasint *nrhs, double *ap, double *b, blasint *ldb, double *x, blasint *ldx, double *ferr, double *berr, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtprfs((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) nrhs, (void *) ap, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->dtprfs.timings[0] += (helpTimeStop - helpTime);
    data->dtprfs.calls[0]++;
    
    return;
}


/* Wrapper for dtptri */
void hook_dtptri(char *uplo, char *diag, blasint *n, double *ap, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtptri((void *) uplo, (void *) diag, (void *) n, (void *) ap, (void *) info, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->dtptri.timings[0] += (helpTimeStop - helpTime);
    data->dtptri.calls[0]++;
    
    return;
}


/* Wrapper for dtptrs */
void hook_dtptrs(char *uplo, char *trans, char *diag, blasint *n, blasint *nrhs, double *ap, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtptrs((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) nrhs, (void *) ap, (void *) b, (void *) ldb, (void *) info, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->dtptrs.timings[0] += (helpTimeStop - helpTime);
    data->dtptrs.calls[0]++;
    
    return;
}


/* Wrapper for dtpttf */
void hook_dtpttf(char *transr, char *uplo, blasint *n, double *ap, double *arf, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtpttf((void *) transr, (void *) uplo, (void *) n, (void *) ap, (void *) arf, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dtpttf.timings[0] += (helpTimeStop - helpTime);
    data->dtpttf.calls[0]++;
    
    return;
}


/* Wrapper for dtpttr */
void hook_dtpttr(char *uplo, blasint *n, double *ap, double *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtpttr((void *) uplo, (void *) n, (void *) ap, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dtpttr.timings[0] += (helpTimeStop - helpTime);
    data->dtpttr.calls[0]++;
    
    return;
}


/* Wrapper for dtrcon */
void hook_dtrcon(char *norm, char *uplo, char *diag, blasint *n, double *a, blasint *lda, double *rcond, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtrcon((void *) norm, (void *) uplo, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) rcond, (void *) work, (void *) iwork, (void *) info, norm_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->dtrcon.timings[0] += (helpTimeStop - helpTime);
    data->dtrcon.calls[0]++;
    
    return;
}


/* Wrapper for dtrevc */
void hook_dtrevc(char *side, char *howmny, blaslogical *select, blasint *n, double *t, blasint *ldt, double *vl, blasint *ldvl, double *vr, blasint *ldvr, blasint *mm, blasint *m, double *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t howmny_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtrevc((void *) side, (void *) howmny, (void *) select, (void *) n, (void *) t, (void *) ldt, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) mm, (void *) m, (void *) work, (void *) info, side_len, howmny_len);

    helpTimeStop = flexiblas_wtime();

    data->dtrevc.timings[0] += (helpTimeStop - helpTime);
    data->dtrevc.calls[0]++;
    
    return;
}


/* Wrapper for dtrevc3 */
void hook_dtrevc3(char *side, char *howmny, blaslogical *select, blasint *n, double *t, blasint *ldt, double *vl, blasint *ldvl, double *vr, blasint *ldvr, blasint *mm, blasint *m, double *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t howmny_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtrevc3((void *) side, (void *) howmny, (void *) select, (void *) n, (void *) t, (void *) ldt, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) mm, (void *) m, (void *) work, (void *) lwork, (void *) info, side_len, howmny_len);

    helpTimeStop = flexiblas_wtime();

    data->dtrevc3.timings[0] += (helpTimeStop - helpTime);
    data->dtrevc3.calls[0]++;
    
    return;
}


/* Wrapper for dtrexc */
void hook_dtrexc(char *compq, blasint *n, double *t, blasint *ldt, double *q, blasint *ldq, blasint *ifst, blasint *ilst, double *work, blasint *info, flexiblas_fortran_charlen_t compq_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtrexc((void *) compq, (void *) n, (void *) t, (void *) ldt, (void *) q, (void *) ldq, (void *) ifst, (void *) ilst, (void *) work, (void *) info, compq_len);

    helpTimeStop = flexiblas_wtime();

    data->dtrexc.timings[0] += (helpTimeStop - helpTime);
    data->dtrexc.calls[0]++;
    
    return;
}


/* Wrapper for dtrrfs */
void hook_dtrrfs(char *uplo, char *trans, char *diag, blasint *n, blasint *nrhs, double *a, blasint *lda, double *b, blasint *ldb, double *x, blasint *ldx, double *ferr, double *berr, double *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtrrfs((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->dtrrfs.timings[0] += (helpTimeStop - helpTime);
    data->dtrrfs.calls[0]++;
    
    return;
}


/* Wrapper for dtrsen */
void hook_dtrsen(char *job, char *compq, blaslogical *select, blasint *n, double *t, blasint *ldt, double *q, blasint *ldq, double *wr, double *wi, blasint *m, double *s, double *sep, double *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t compq_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtrsen((void *) job, (void *) compq, (void *) select, (void *) n, (void *) t, (void *) ldt, (void *) q, (void *) ldq, (void *) wr, (void *) wi, (void *) m, (void *) s, (void *) sep, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, job_len, compq_len);

    helpTimeStop = flexiblas_wtime();

    data->dtrsen.timings[0] += (helpTimeStop - helpTime);
    data->dtrsen.calls[0]++;
    
    return;
}


/* Wrapper for dtrsna */
void hook_dtrsna(char *job, char *howmny, blaslogical *select, blasint *n, double *t, blasint *ldt, double *vl, blasint *ldvl, double *vr, blasint *ldvr, double *s, double *sep, blasint *mm, blasint *m, double *work, blasint *ldwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t howmny_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtrsna((void *) job, (void *) howmny, (void *) select, (void *) n, (void *) t, (void *) ldt, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) s, (void *) sep, (void *) mm, (void *) m, (void *) work, (void *) ldwork, (void *) iwork, (void *) info, job_len, howmny_len);

    helpTimeStop = flexiblas_wtime();

    data->dtrsna.timings[0] += (helpTimeStop - helpTime);
    data->dtrsna.calls[0]++;
    
    return;
}


/* Wrapper for dtrsyl */
void hook_dtrsyl(char *trana, char *tranb, blasint *isgn, blasint *m, blasint *n, double *a, blasint *lda, double *b, blasint *ldb, double *c, blasint *ldc, double *scale, blasint *info, flexiblas_fortran_charlen_t trana_len, flexiblas_fortran_charlen_t tranb_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtrsyl((void *) trana, (void *) tranb, (void *) isgn, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) c, (void *) ldc, (void *) scale, (void *) info, trana_len, tranb_len);

    helpTimeStop = flexiblas_wtime();

    data->dtrsyl.timings[0] += (helpTimeStop - helpTime);
    data->dtrsyl.calls[0]++;
    
    return;
}


/* Wrapper for dtrti2 */
void hook_dtrti2(char *uplo, char *diag, blasint *n, double *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtrti2((void *) uplo, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->dtrti2.timings[0] += (helpTimeStop - helpTime);
    data->dtrti2.calls[0]++;
    
    return;
}


/* Wrapper for dtrtri */
void hook_dtrtri(char *uplo, char *diag, blasint *n, double *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtrtri((void *) uplo, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->dtrtri.timings[0] += (helpTimeStop - helpTime);
    data->dtrtri.calls[0]++;
    
    return;
}


/* Wrapper for dtrtrs */
void hook_dtrtrs(char *uplo, char *trans, char *diag, blasint *n, blasint *nrhs, double *a, blasint *lda, double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtrtrs((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) info, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->dtrtrs.timings[0] += (helpTimeStop - helpTime);
    data->dtrtrs.calls[0]++;
    
    return;
}


/* Wrapper for dtrttf */
void hook_dtrttf(char *transr, char *uplo, blasint *n, double *a, blasint *lda, double *arf, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtrttf((void *) transr, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) arf, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dtrttf.timings[0] += (helpTimeStop - helpTime);
    data->dtrttf.calls[0]++;
    
    return;
}


/* Wrapper for dtrttp */
void hook_dtrttp(char *uplo, blasint *n, double *a, blasint *lda, double *ap, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtrttp((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ap, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->dtrttp.timings[0] += (helpTimeStop - helpTime);
    data->dtrttp.calls[0]++;
    
    return;
}


/* Wrapper for dtzrzf */
void hook_dtzrzf(blasint *m, blasint *n, double *a, blasint *lda, double *tau, double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_dtzrzf((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->dtzrzf.timings[0] += (helpTimeStop - helpTime);
    data->dtzrzf.calls[0]++;
    
    return;
}

/* Wrapper for dzsum1 */
double hook_dzsum1(blasint *n, lapack_complex_double *cx, blasint *incx)
{
    double helpTime;
    double helpTimeStop;
    double return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_dzsum1((void *) n, (void *) cx, (void *) incx);

    helpTimeStop = flexiblas_wtime();

    data->dzsum1.timings[0] += (helpTimeStop - helpTime);
    data->dzsum1.calls[0]++;
    
    return return_value;
}


/* Wrapper for icmax1 */
blasint hook_icmax1(blasint *n, lapack_complex_float *cx, blasint *incx)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_icmax1((void *) n, (void *) cx, (void *) incx);

    helpTimeStop = flexiblas_wtime();

    data->icmax1.timings[0] += (helpTimeStop - helpTime);
    data->icmax1.calls[0]++;
    
    return return_value;
}


/* Wrapper for ieeeck */
blasint hook_ieeeck(blasint *ispec, float *zero, float *one)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_ieeeck((void *) ispec, (void *) zero, (void *) one);

    helpTimeStop = flexiblas_wtime();

    data->ieeeck.timings[0] += (helpTimeStop - helpTime);
    data->ieeeck.calls[0]++;
    
    return return_value;
}


/* Wrapper for ilaclc */
blasint hook_ilaclc(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_ilaclc((void *) m, (void *) n, (void *) a, (void *) lda);

    helpTimeStop = flexiblas_wtime();

    data->ilaclc.timings[0] += (helpTimeStop - helpTime);
    data->ilaclc.calls[0]++;
    
    return return_value;
}


/* Wrapper for ilaclr */
blasint hook_ilaclr(blasint *m, blasint *n, lapack_complex_float *a, blasint *lda)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_ilaclr((void *) m, (void *) n, (void *) a, (void *) lda);

    helpTimeStop = flexiblas_wtime();

    data->ilaclr.timings[0] += (helpTimeStop - helpTime);
    data->ilaclr.calls[0]++;
    
    return return_value;
}


/* Wrapper for iladiag */
blasint hook_iladiag(char *diag, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_iladiag((void *) diag, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->iladiag.timings[0] += (helpTimeStop - helpTime);
    data->iladiag.calls[0]++;
    
    return return_value;
}


/* Wrapper for iladlc */
blasint hook_iladlc(blasint *m, blasint *n, double *a, blasint *lda)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_iladlc((void *) m, (void *) n, (void *) a, (void *) lda);

    helpTimeStop = flexiblas_wtime();

    data->iladlc.timings[0] += (helpTimeStop - helpTime);
    data->iladlc.calls[0]++;
    
    return return_value;
}


/* Wrapper for iladlr */
blasint hook_iladlr(blasint *m, blasint *n, double *a, blasint *lda)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_iladlr((void *) m, (void *) n, (void *) a, (void *) lda);

    helpTimeStop = flexiblas_wtime();

    data->iladlr.timings[0] += (helpTimeStop - helpTime);
    data->iladlr.calls[0]++;
    
    return return_value;
}


/* Wrapper for ilaenv */
blasint hook_ilaenv(blasint *ispec, char *name, char *opts, blasint *n1, blasint *n2, blasint *n3, blasint *n4, flexiblas_fortran_charlen_t name_len, flexiblas_fortran_charlen_t opts_len)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_ilaenv((void *) ispec, (void *) name, (void *) opts, (void *) n1, (void *) n2, (void *) n3, (void *) n4, name_len, opts_len);

    helpTimeStop = flexiblas_wtime();

    data->ilaenv.timings[0] += (helpTimeStop - helpTime);
    data->ilaenv.calls[0]++;
    
    return return_value;
}


/* Wrapper for ilaenv2stage */
blasint hook_ilaenv2stage(blasint *ispec, char *name, char *opts, blasint *n1, blasint *n2, blasint *n3, blasint *n4, flexiblas_fortran_charlen_t name_len, flexiblas_fortran_charlen_t opts_len)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_ilaenv2stage((void *) ispec, (void *) name, (void *) opts, (void *) n1, (void *) n2, (void *) n3, (void *) n4, name_len, opts_len);

    helpTimeStop = flexiblas_wtime();

    data->ilaenv2stage.timings[0] += (helpTimeStop - helpTime);
    data->ilaenv2stage.calls[0]++;
    
    return return_value;
}


/* Wrapper for ilaprec */
blasint hook_ilaprec(char *prec, flexiblas_fortran_charlen_t prec_len)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_ilaprec((void *) prec, prec_len);

    helpTimeStop = flexiblas_wtime();

    data->ilaprec.timings[0] += (helpTimeStop - helpTime);
    data->ilaprec.calls[0]++;
    
    return return_value;
}


/* Wrapper for ilaslc */
blasint hook_ilaslc(blasint *m, blasint *n, float *a, blasint *lda)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_ilaslc((void *) m, (void *) n, (void *) a, (void *) lda);

    helpTimeStop = flexiblas_wtime();

    data->ilaslc.timings[0] += (helpTimeStop - helpTime);
    data->ilaslc.calls[0]++;
    
    return return_value;
}


/* Wrapper for ilaslr */
blasint hook_ilaslr(blasint *m, blasint *n, float *a, blasint *lda)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_ilaslr((void *) m, (void *) n, (void *) a, (void *) lda);

    helpTimeStop = flexiblas_wtime();

    data->ilaslr.timings[0] += (helpTimeStop - helpTime);
    data->ilaslr.calls[0]++;
    
    return return_value;
}


/* Wrapper for ilatrans */
blasint hook_ilatrans(char *trans, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_ilatrans((void *) trans, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->ilatrans.timings[0] += (helpTimeStop - helpTime);
    data->ilatrans.calls[0]++;
    
    return return_value;
}


/* Wrapper for ilauplo */
blasint hook_ilauplo(char *uplo, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_ilauplo((void *) uplo, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ilauplo.timings[0] += (helpTimeStop - helpTime);
    data->ilauplo.calls[0]++;
    
    return return_value;
}


/* Wrapper for ilazlc */
blasint hook_ilazlc(blasint *m, blasint *n, lapack_complex_double *a, blasint *lda)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_ilazlc((void *) m, (void *) n, (void *) a, (void *) lda);

    helpTimeStop = flexiblas_wtime();

    data->ilazlc.timings[0] += (helpTimeStop - helpTime);
    data->ilazlc.calls[0]++;
    
    return return_value;
}


/* Wrapper for ilazlr */
blasint hook_ilazlr(blasint *m, blasint *n, lapack_complex_double *a, blasint *lda)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_ilazlr((void *) m, (void *) n, (void *) a, (void *) lda);

    helpTimeStop = flexiblas_wtime();

    data->ilazlr.timings[0] += (helpTimeStop - helpTime);
    data->ilazlr.calls[0]++;
    
    return return_value;
}


/* Wrapper for iparam2stage */
blasint hook_iparam2stage(blasint *ispec, char *name, char *opts, blasint *ni, blasint *nbi, blasint *ibi, blasint *nxi, flexiblas_fortran_charlen_t name_len, flexiblas_fortran_charlen_t opts_len)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_iparam2stage((void *) ispec, (void *) name, (void *) opts, (void *) ni, (void *) nbi, (void *) ibi, (void *) nxi, name_len, opts_len);

    helpTimeStop = flexiblas_wtime();

    data->iparam2stage.timings[0] += (helpTimeStop - helpTime);
    data->iparam2stage.calls[0]++;
    
    return return_value;
}


/* Wrapper for iparmq */
blasint hook_iparmq(blasint *ispec, char *name, char *opts, blasint *n, blasint *ilo, blasint *ihi, blasint *lwork, flexiblas_fortran_charlen_t name_len, flexiblas_fortran_charlen_t opts_len)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_iparmq((void *) ispec, (void *) name, (void *) opts, (void *) n, (void *) ilo, (void *) ihi, (void *) lwork, name_len, opts_len);

    helpTimeStop = flexiblas_wtime();

    data->iparmq.timings[0] += (helpTimeStop - helpTime);
    data->iparmq.calls[0]++;
    
    return return_value;
}


/* Wrapper for izmax1 */
blasint hook_izmax1(blasint *n, lapack_complex_double *zx, blasint *incx)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_izmax1((void *) n, (void *) zx, (void *) incx);

    helpTimeStop = flexiblas_wtime();

    data->izmax1.timings[0] += (helpTimeStop - helpTime);
    data->izmax1.calls[0]++;
    
    return return_value;
}



/* Wrapper for sbbcsd */
void hook_sbbcsd(char *jobu1, char *jobu2, char *jobv1t, char *jobv2t, char *trans, blasint *m, blasint *p, blasint *q, float *theta, float *phi, float *u1, blasint *ldu1, float *u2, blasint *ldu2, float *v1t, blasint *ldv1t, float *v2t, blasint *ldv2t, float *b11d, float *b11e, float *b12d, float *b12e, float *b21d, float *b21e, float *b22d, float *b22e, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobu1_len, flexiblas_fortran_charlen_t jobu2_len, flexiblas_fortran_charlen_t jobv1t_len, flexiblas_fortran_charlen_t jobv2t_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sbbcsd((void *) jobu1, (void *) jobu2, (void *) jobv1t, (void *) jobv2t, (void *) trans, (void *) m, (void *) p, (void *) q, (void *) theta, (void *) phi, (void *) u1, (void *) ldu1, (void *) u2, (void *) ldu2, (void *) v1t, (void *) ldv1t, (void *) v2t, (void *) ldv2t, (void *) b11d, (void *) b11e, (void *) b12d, (void *) b12e, (void *) b21d, (void *) b21e, (void *) b22d, (void *) b22e, (void *) work, (void *) lwork, (void *) info, jobu1_len, jobu2_len, jobv1t_len, jobv2t_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sbbcsd.timings[0] += (helpTimeStop - helpTime);
    data->sbbcsd.calls[0]++;
    
    return;
}


/* Wrapper for sbdsdc */
void hook_sbdsdc(char *uplo, char *compq, blasint *n, float *d, float *e, float *u, blasint *ldu, float *vt, blasint *ldvt, float *q, blasint *iq, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t compq_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sbdsdc((void *) uplo, (void *) compq, (void *) n, (void *) d, (void *) e, (void *) u, (void *) ldu, (void *) vt, (void *) ldvt, (void *) q, (void *) iq, (void *) work, (void *) iwork, (void *) info, uplo_len, compq_len);

    helpTimeStop = flexiblas_wtime();

    data->sbdsdc.timings[0] += (helpTimeStop - helpTime);
    data->sbdsdc.calls[0]++;
    
    return;
}


/* Wrapper for sbdsqr */
void hook_sbdsqr(char *uplo, blasint *n, blasint *ncvt, blasint *nru, blasint *ncc, float *d, float *e, float *vt, blasint *ldvt, float *u, blasint *ldu, float *c, blasint *ldc, float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sbdsqr((void *) uplo, (void *) n, (void *) ncvt, (void *) nru, (void *) ncc, (void *) d, (void *) e, (void *) vt, (void *) ldvt, (void *) u, (void *) ldu, (void *) c, (void *) ldc, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->sbdsqr.timings[0] += (helpTimeStop - helpTime);
    data->sbdsqr.calls[0]++;
    
    return;
}


/* Wrapper for sbdsvdx */
void hook_sbdsvdx(char *uplo, char *jobz, char *range, blasint *n, float *d, float *e, float *vl, float *vu, blasint *il, blasint *iu, blasint *ns, float *s, float *z, blasint *ldz, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sbdsvdx((void *) uplo, (void *) jobz, (void *) range, (void *) n, (void *) d, (void *) e, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) ns, (void *) s, (void *) z, (void *) ldz, (void *) work, (void *) iwork, (void *) info, uplo_len, jobz_len, range_len);

    helpTimeStop = flexiblas_wtime();

    data->sbdsvdx.timings[0] += (helpTimeStop - helpTime);
    data->sbdsvdx.calls[0]++;
    
    return;
}


/* Wrapper for scombssq */
void hook_scombssq(float *v1, float *v2)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_scombssq((void *) v1, (void *) v2);

    helpTimeStop = flexiblas_wtime();

    data->scombssq.timings[0] += (helpTimeStop - helpTime);
    data->scombssq.calls[0]++;
    
    return;
}

/* Wrapper for scsum1 */
float hook_scsum1(blasint *n, lapack_complex_float *cx, blasint *incx)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_scsum1((void *) n, (void *) cx, (void *) incx);

    helpTimeStop = flexiblas_wtime();

    data->scsum1.timings[0] += (helpTimeStop - helpTime);
    data->scsum1.calls[0]++;
    
    return return_value;
}



/* Wrapper for sdisna */
void hook_sdisna(char *job, blasint *m, blasint *n, float *d, float *sep, blasint *info, flexiblas_fortran_charlen_t job_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sdisna((void *) job, (void *) m, (void *) n, (void *) d, (void *) sep, (void *) info, job_len);

    helpTimeStop = flexiblas_wtime();

    data->sdisna.timings[0] += (helpTimeStop - helpTime);
    data->sdisna.calls[0]++;
    
    return;
}

/* Wrapper for second */
float hook_second(void)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_second();

    helpTimeStop = flexiblas_wtime();

    data->second.timings[0] += (helpTimeStop - helpTime);
    data->second.calls[0]++;
    
    return return_value;
}



/* Wrapper for sgbbrd */
void hook_sgbbrd(char *vect, blasint *m, blasint *n, blasint *ncc, blasint *kl, blasint *ku, float *ab, blasint *ldab, float *d, float *e, float *q, blasint *ldq, float *pt, blasint *ldpt, float *c, blasint *ldc, float *work, blasint *info, flexiblas_fortran_charlen_t vect_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgbbrd((void *) vect, (void *) m, (void *) n, (void *) ncc, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) d, (void *) e, (void *) q, (void *) ldq, (void *) pt, (void *) ldpt, (void *) c, (void *) ldc, (void *) work, (void *) info, vect_len);

    helpTimeStop = flexiblas_wtime();

    data->sgbbrd.timings[0] += (helpTimeStop - helpTime);
    data->sgbbrd.calls[0]++;
    
    return;
}


/* Wrapper for sgbcon */
void hook_sgbcon(char *norm, blasint *n, blasint *kl, blasint *ku, float *ab, blasint *ldab, blasint *ipiv, float *anorm, float *rcond, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgbcon((void *) norm, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) iwork, (void *) info, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->sgbcon.timings[0] += (helpTimeStop - helpTime);
    data->sgbcon.calls[0]++;
    
    return;
}


/* Wrapper for sgbequ */
void hook_sgbequ(blasint *m, blasint *n, blasint *kl, blasint *ku, float *ab, blasint *ldab, float *r, float *c, float *rowcnd, float *colcnd, float *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgbequ((void *) m, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgbequ.timings[0] += (helpTimeStop - helpTime);
    data->sgbequ.calls[0]++;
    
    return;
}


/* Wrapper for sgbequb */
void hook_sgbequb(blasint *m, blasint *n, blasint *kl, blasint *ku, float *ab, blasint *ldab, float *r, float *c, float *rowcnd, float *colcnd, float *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgbequb((void *) m, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgbequb.timings[0] += (helpTimeStop - helpTime);
    data->sgbequb.calls[0]++;
    
    return;
}


/* Wrapper for sgbrfs */
void hook_sgbrfs(char *trans, blasint *n, blasint *kl, blasint *ku, blasint *nrhs, float *ab, blasint *ldab, float *afb, blasint *ldafb, blasint *ipiv, float *b, blasint *ldb, float *x, blasint *ldx, float *ferr, float *berr, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgbrfs((void *) trans, (void *) n, (void *) kl, (void *) ku, (void *) nrhs, (void *) ab, (void *) ldab, (void *) afb, (void *) ldafb, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sgbrfs.timings[0] += (helpTimeStop - helpTime);
    data->sgbrfs.calls[0]++;
    
    return;
}


/* Wrapper for sgbsv */
void hook_sgbsv(blasint *n, blasint *kl, blasint *ku, blasint *nrhs, float *ab, blasint *ldab, blasint *ipiv, float *b, blasint *ldb, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgbsv((void *) n, (void *) kl, (void *) ku, (void *) nrhs, (void *) ab, (void *) ldab, (void *) ipiv, (void *) b, (void *) ldb, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgbsv.timings[0] += (helpTimeStop - helpTime);
    data->sgbsv.calls[0]++;
    
    return;
}


/* Wrapper for sgbsvx */
void hook_sgbsvx(char *fact, char *trans, blasint *n, blasint *kl, blasint *ku, blasint *nrhs, float *ab, blasint *ldab, float *afb, blasint *ldafb, blasint *ipiv, char *equed, float *r, float *c, float *b, blasint *ldb, float *x, blasint *ldx, float *rcond, float *ferr, float *berr, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgbsvx((void *) fact, (void *) trans, (void *) n, (void *) kl, (void *) ku, (void *) nrhs, (void *) ab, (void *) ldab, (void *) afb, (void *) ldafb, (void *) ipiv, (void *) equed, (void *) r, (void *) c, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, fact_len, trans_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->sgbsvx.timings[0] += (helpTimeStop - helpTime);
    data->sgbsvx.calls[0]++;
    
    return;
}


/* Wrapper for sgbtf2 */
void hook_sgbtf2(blasint *m, blasint *n, blasint *kl, blasint *ku, float *ab, blasint *ldab, blasint *ipiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgbtf2((void *) m, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) ipiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgbtf2.timings[0] += (helpTimeStop - helpTime);
    data->sgbtf2.calls[0]++;
    
    return;
}


/* Wrapper for sgbtrf */
void hook_sgbtrf(blasint *m, blasint *n, blasint *kl, blasint *ku, float *ab, blasint *ldab, blasint *ipiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgbtrf((void *) m, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) ipiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgbtrf.timings[0] += (helpTimeStop - helpTime);
    data->sgbtrf.calls[0]++;
    
    return;
}


/* Wrapper for sgbtrs */
void hook_sgbtrs(char *trans, blasint *n, blasint *kl, blasint *ku, blasint *nrhs, float *ab, blasint *ldab, blasint *ipiv, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgbtrs((void *) trans, (void *) n, (void *) kl, (void *) ku, (void *) nrhs, (void *) ab, (void *) ldab, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sgbtrs.timings[0] += (helpTimeStop - helpTime);
    data->sgbtrs.calls[0]++;
    
    return;
}


/* Wrapper for sgebak */
void hook_sgebak(char *job, char *side, blasint *n, blasint *ilo, blasint *ihi, float *scale, blasint *m, float *v, blasint *ldv, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t side_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgebak((void *) job, (void *) side, (void *) n, (void *) ilo, (void *) ihi, (void *) scale, (void *) m, (void *) v, (void *) ldv, (void *) info, job_len, side_len);

    helpTimeStop = flexiblas_wtime();

    data->sgebak.timings[0] += (helpTimeStop - helpTime);
    data->sgebak.calls[0]++;
    
    return;
}


/* Wrapper for sgebal */
void hook_sgebal(char *job, blasint *n, float *a, blasint *lda, blasint *ilo, blasint *ihi, float *scale, blasint *info, flexiblas_fortran_charlen_t job_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgebal((void *) job, (void *) n, (void *) a, (void *) lda, (void *) ilo, (void *) ihi, (void *) scale, (void *) info, job_len);

    helpTimeStop = flexiblas_wtime();

    data->sgebal.timings[0] += (helpTimeStop - helpTime);
    data->sgebal.calls[0]++;
    
    return;
}


/* Wrapper for sgebd2 */
void hook_sgebd2(blasint *m, blasint *n, float *a, blasint *lda, float *d, float *e, float *tauq, float *taup, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgebd2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) e, (void *) tauq, (void *) taup, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgebd2.timings[0] += (helpTimeStop - helpTime);
    data->sgebd2.calls[0]++;
    
    return;
}


/* Wrapper for sgebrd */
void hook_sgebrd(blasint *m, blasint *n, float *a, blasint *lda, float *d, float *e, float *tauq, float *taup, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgebrd((void *) m, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) e, (void *) tauq, (void *) taup, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgebrd.timings[0] += (helpTimeStop - helpTime);
    data->sgebrd.calls[0]++;
    
    return;
}


/* Wrapper for sgecon */
void hook_sgecon(char *norm, blasint *n, float *a, blasint *lda, float *anorm, float *rcond, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgecon((void *) norm, (void *) n, (void *) a, (void *) lda, (void *) anorm, (void *) rcond, (void *) work, (void *) iwork, (void *) info, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->sgecon.timings[0] += (helpTimeStop - helpTime);
    data->sgecon.calls[0]++;
    
    return;
}


/* Wrapper for sgeequ */
void hook_sgeequ(blasint *m, blasint *n, float *a, blasint *lda, float *r, float *c, float *rowcnd, float *colcnd, float *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgeequ((void *) m, (void *) n, (void *) a, (void *) lda, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeequ.timings[0] += (helpTimeStop - helpTime);
    data->sgeequ.calls[0]++;
    
    return;
}


/* Wrapper for sgeequb */
void hook_sgeequb(blasint *m, blasint *n, float *a, blasint *lda, float *r, float *c, float *rowcnd, float *colcnd, float *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgeequb((void *) m, (void *) n, (void *) a, (void *) lda, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeequb.timings[0] += (helpTimeStop - helpTime);
    data->sgeequb.calls[0]++;
    
    return;
}


/* Wrapper for sgees */
void hook_sgees(char *jobvs, char *sort, blaslogical *select, blasint *n, float *a, blasint *lda, blasint *sdim, float *wr, float *wi, float *vs, blasint *ldvs, float *work, blasint *lwork, blaslogical *bwork, blasint *info, flexiblas_fortran_charlen_t jobvs_len, flexiblas_fortran_charlen_t sort_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgees((void *) jobvs, (void *) sort, (void *) select, (void *) n, (void *) a, (void *) lda, (void *) sdim, (void *) wr, (void *) wi, (void *) vs, (void *) ldvs, (void *) work, (void *) lwork, (void *) bwork, (void *) info, jobvs_len, sort_len);

    helpTimeStop = flexiblas_wtime();

    data->sgees.timings[0] += (helpTimeStop - helpTime);
    data->sgees.calls[0]++;
    
    return;
}


/* Wrapper for sgeesx */
void hook_sgeesx(char *jobvs, char *sort, blaslogical *select, char *sense, blasint *n, float *a, blasint *lda, blasint *sdim, float *wr, float *wi, float *vs, blasint *ldvs, float *rconde, float *rcondv, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blaslogical *bwork, blasint *info, flexiblas_fortran_charlen_t jobvs_len, flexiblas_fortran_charlen_t sort_len, flexiblas_fortran_charlen_t sense_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgeesx((void *) jobvs, (void *) sort, (void *) select, (void *) sense, (void *) n, (void *) a, (void *) lda, (void *) sdim, (void *) wr, (void *) wi, (void *) vs, (void *) ldvs, (void *) rconde, (void *) rcondv, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) bwork, (void *) info, jobvs_len, sort_len, sense_len);

    helpTimeStop = flexiblas_wtime();

    data->sgeesx.timings[0] += (helpTimeStop - helpTime);
    data->sgeesx.calls[0]++;
    
    return;
}


/* Wrapper for sgeev */
void hook_sgeev(char *jobvl, char *jobvr, blasint *n, float *a, blasint *lda, float *wr, float *wi, float *vl, blasint *ldvl, float *vr, blasint *ldvr, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobvl_len, flexiblas_fortran_charlen_t jobvr_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgeev((void *) jobvl, (void *) jobvr, (void *) n, (void *) a, (void *) lda, (void *) wr, (void *) wi, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) work, (void *) lwork, (void *) info, jobvl_len, jobvr_len);

    helpTimeStop = flexiblas_wtime();

    data->sgeev.timings[0] += (helpTimeStop - helpTime);
    data->sgeev.calls[0]++;
    
    return;
}


/* Wrapper for sgeevx */
void hook_sgeevx(char *balanc, char *jobvl, char *jobvr, char *sense, blasint *n, float *a, blasint *lda, float *wr, float *wi, float *vl, blasint *ldvl, float *vr, blasint *ldvr, blasint *ilo, blasint *ihi, float *scale, float *abnrm, float *rconde, float *rcondv, float *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t balanc_len, flexiblas_fortran_charlen_t jobvl_len, flexiblas_fortran_charlen_t jobvr_len, flexiblas_fortran_charlen_t sense_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgeevx((void *) balanc, (void *) jobvl, (void *) jobvr, (void *) sense, (void *) n, (void *) a, (void *) lda, (void *) wr, (void *) wi, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) ilo, (void *) ihi, (void *) scale, (void *) abnrm, (void *) rconde, (void *) rcondv, (void *) work, (void *) lwork, (void *) iwork, (void *) info, balanc_len, jobvl_len, jobvr_len, sense_len);

    helpTimeStop = flexiblas_wtime();

    data->sgeevx.timings[0] += (helpTimeStop - helpTime);
    data->sgeevx.calls[0]++;
    
    return;
}


/* Wrapper for sgehd2 */
void hook_sgehd2(blasint *n, blasint *ilo, blasint *ihi, float *a, blasint *lda, float *tau, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgehd2((void *) n, (void *) ilo, (void *) ihi, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgehd2.timings[0] += (helpTimeStop - helpTime);
    data->sgehd2.calls[0]++;
    
    return;
}


/* Wrapper for sgehrd */
void hook_sgehrd(blasint *n, blasint *ilo, blasint *ihi, float *a, blasint *lda, float *tau, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgehrd((void *) n, (void *) ilo, (void *) ihi, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgehrd.timings[0] += (helpTimeStop - helpTime);
    data->sgehrd.calls[0]++;
    
    return;
}


/* Wrapper for sgejsv */
void hook_sgejsv(char *joba, char *jobu, char *jobv, char *jobr, char *jobt, char *jobp, blasint *m, blasint *n, float *a, blasint *lda, float *sva, float *u, blasint *ldu, float *v, blasint *ldv, float *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t joba_len, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobv_len, flexiblas_fortran_charlen_t jobr_len, flexiblas_fortran_charlen_t jobt_len, flexiblas_fortran_charlen_t jobp_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgejsv((void *) joba, (void *) jobu, (void *) jobv, (void *) jobr, (void *) jobt, (void *) jobp, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) sva, (void *) u, (void *) ldu, (void *) v, (void *) ldv, (void *) work, (void *) lwork, (void *) iwork, (void *) info, joba_len, jobu_len, jobv_len, jobr_len, jobt_len, jobp_len);

    helpTimeStop = flexiblas_wtime();

    data->sgejsv.timings[0] += (helpTimeStop - helpTime);
    data->sgejsv.calls[0]++;
    
    return;
}


/* Wrapper for sgelq */
void hook_sgelq(blasint *m, blasint *n, float *a, blasint *lda, float *t, blasint *tsize, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgelq((void *) m, (void *) n, (void *) a, (void *) lda, (void *) t, (void *) tsize, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgelq.timings[0] += (helpTimeStop - helpTime);
    data->sgelq.calls[0]++;
    
    return;
}


/* Wrapper for sgelq2 */
void hook_sgelq2(blasint *m, blasint *n, float *a, blasint *lda, float *tau, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgelq2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgelq2.timings[0] += (helpTimeStop - helpTime);
    data->sgelq2.calls[0]++;
    
    return;
}


/* Wrapper for sgelqf */
void hook_sgelqf(blasint *m, blasint *n, float *a, blasint *lda, float *tau, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgelqf((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgelqf.timings[0] += (helpTimeStop - helpTime);
    data->sgelqf.calls[0]++;
    
    return;
}


/* Wrapper for sgelqt */
void hook_sgelqt(blasint *m, blasint *n, blasint *mb, float *a, blasint *lda, float *t, blasint *ldt, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgelqt((void *) m, (void *) n, (void *) mb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgelqt.timings[0] += (helpTimeStop - helpTime);
    data->sgelqt.calls[0]++;
    
    return;
}


/* Wrapper for sgelqt3 */
void hook_sgelqt3(blasint *m, blasint *n, float *a, blasint *lda, float *t, blasint *ldt, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgelqt3((void *) m, (void *) n, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgelqt3.timings[0] += (helpTimeStop - helpTime);
    data->sgelqt3.calls[0]++;
    
    return;
}


/* Wrapper for sgels */
void hook_sgels(char *trans, blasint *m, blasint *n, blasint *nrhs, float *a, blasint *lda, float *b, blasint *ldb, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgels((void *) trans, (void *) m, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sgels.timings[0] += (helpTimeStop - helpTime);
    data->sgels.calls[0]++;
    
    return;
}


/* Wrapper for sgelsd */
void hook_sgelsd(blasint *m, blasint *n, blasint *nrhs, float *a, blasint *lda, float *b, blasint *ldb, float *s, float *rcond, blasint *rank, float *work, blasint *lwork, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgelsd((void *) m, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) s, (void *) rcond, (void *) rank, (void *) work, (void *) lwork, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgelsd.timings[0] += (helpTimeStop - helpTime);
    data->sgelsd.calls[0]++;
    
    return;
}


/* Wrapper for sgelss */
void hook_sgelss(blasint *m, blasint *n, blasint *nrhs, float *a, blasint *lda, float *b, blasint *ldb, float *s, float *rcond, blasint *rank, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgelss((void *) m, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) s, (void *) rcond, (void *) rank, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgelss.timings[0] += (helpTimeStop - helpTime);
    data->sgelss.calls[0]++;
    
    return;
}


/* Wrapper for sgelsy */
void hook_sgelsy(blasint *m, blasint *n, blasint *nrhs, float *a, blasint *lda, float *b, blasint *ldb, blasint *jpvt, float *rcond, blasint *rank, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgelsy((void *) m, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) jpvt, (void *) rcond, (void *) rank, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgelsy.timings[0] += (helpTimeStop - helpTime);
    data->sgelsy.calls[0]++;
    
    return;
}


/* Wrapper for sgemlq */
void hook_sgemlq(char *side, char *trans, blasint *m, blasint *n, blasint *k, float *a, blasint *lda, float *t, blasint *tsize, float *c, blasint *ldc, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgemlq((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) t, (void *) tsize, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sgemlq.timings[0] += (helpTimeStop - helpTime);
    data->sgemlq.calls[0]++;
    
    return;
}


/* Wrapper for sgemlqt */
void hook_sgemlqt(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *mb, float *v, blasint *ldv, float *t, blasint *ldt, float *c, blasint *ldc, float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgemlqt((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) mb, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sgemlqt.timings[0] += (helpTimeStop - helpTime);
    data->sgemlqt.calls[0]++;
    
    return;
}


/* Wrapper for sgemqr */
void hook_sgemqr(char *side, char *trans, blasint *m, blasint *n, blasint *k, float *a, blasint *lda, float *t, blasint *tsize, float *c, blasint *ldc, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgemqr((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) t, (void *) tsize, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sgemqr.timings[0] += (helpTimeStop - helpTime);
    data->sgemqr.calls[0]++;
    
    return;
}


/* Wrapper for sgemqrt */
void hook_sgemqrt(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *nb, float *v, blasint *ldv, float *t, blasint *ldt, float *c, blasint *ldc, float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgemqrt((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) nb, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sgemqrt.timings[0] += (helpTimeStop - helpTime);
    data->sgemqrt.calls[0]++;
    
    return;
}


/* Wrapper for sgeql2 */
void hook_sgeql2(blasint *m, blasint *n, float *a, blasint *lda, float *tau, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgeql2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeql2.timings[0] += (helpTimeStop - helpTime);
    data->sgeql2.calls[0]++;
    
    return;
}


/* Wrapper for sgeqlf */
void hook_sgeqlf(blasint *m, blasint *n, float *a, blasint *lda, float *tau, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgeqlf((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeqlf.timings[0] += (helpTimeStop - helpTime);
    data->sgeqlf.calls[0]++;
    
    return;
}


/* Wrapper for sgeqp3 */
void hook_sgeqp3(blasint *m, blasint *n, float *a, blasint *lda, blasint *jpvt, float *tau, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgeqp3((void *) m, (void *) n, (void *) a, (void *) lda, (void *) jpvt, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeqp3.timings[0] += (helpTimeStop - helpTime);
    data->sgeqp3.calls[0]++;
    
    return;
}


/* Wrapper for sgeqr */
void hook_sgeqr(blasint *m, blasint *n, float *a, blasint *lda, float *t, blasint *tsize, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgeqr((void *) m, (void *) n, (void *) a, (void *) lda, (void *) t, (void *) tsize, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeqr.timings[0] += (helpTimeStop - helpTime);
    data->sgeqr.calls[0]++;
    
    return;
}


/* Wrapper for sgeqr2 */
void hook_sgeqr2(blasint *m, blasint *n, float *a, blasint *lda, float *tau, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgeqr2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeqr2.timings[0] += (helpTimeStop - helpTime);
    data->sgeqr2.calls[0]++;
    
    return;
}


/* Wrapper for sgeqr2p */
void hook_sgeqr2p(blasint *m, blasint *n, float *a, blasint *lda, float *tau, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgeqr2p((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeqr2p.timings[0] += (helpTimeStop - helpTime);
    data->sgeqr2p.calls[0]++;
    
    return;
}


/* Wrapper for sgeqrf */
void hook_sgeqrf(blasint *m, blasint *n, float *a, blasint *lda, float *tau, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgeqrf((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeqrf.timings[0] += (helpTimeStop - helpTime);
    data->sgeqrf.calls[0]++;
    
    return;
}


/* Wrapper for sgeqrfp */
void hook_sgeqrfp(blasint *m, blasint *n, float *a, blasint *lda, float *tau, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgeqrfp((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeqrfp.timings[0] += (helpTimeStop - helpTime);
    data->sgeqrfp.calls[0]++;
    
    return;
}


/* Wrapper for sgeqrt */
void hook_sgeqrt(blasint *m, blasint *n, blasint *nb, float *a, blasint *lda, float *t, blasint *ldt, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgeqrt((void *) m, (void *) n, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeqrt.timings[0] += (helpTimeStop - helpTime);
    data->sgeqrt.calls[0]++;
    
    return;
}


/* Wrapper for sgeqrt2 */
void hook_sgeqrt2(blasint *m, blasint *n, float *a, blasint *lda, float *t, blasint *ldt, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgeqrt2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeqrt2.timings[0] += (helpTimeStop - helpTime);
    data->sgeqrt2.calls[0]++;
    
    return;
}


/* Wrapper for sgeqrt3 */
void hook_sgeqrt3(blasint *m, blasint *n, float *a, blasint *lda, float *t, blasint *ldt, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgeqrt3((void *) m, (void *) n, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgeqrt3.timings[0] += (helpTimeStop - helpTime);
    data->sgeqrt3.calls[0]++;
    
    return;
}


/* Wrapper for sgerfs */
void hook_sgerfs(char *trans, blasint *n, blasint *nrhs, float *a, blasint *lda, float *af, blasint *ldaf, blasint *ipiv, float *b, blasint *ldb, float *x, blasint *ldx, float *ferr, float *berr, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgerfs((void *) trans, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) af, (void *) ldaf, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sgerfs.timings[0] += (helpTimeStop - helpTime);
    data->sgerfs.calls[0]++;
    
    return;
}


/* Wrapper for sgerq2 */
void hook_sgerq2(blasint *m, blasint *n, float *a, blasint *lda, float *tau, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgerq2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgerq2.timings[0] += (helpTimeStop - helpTime);
    data->sgerq2.calls[0]++;
    
    return;
}


/* Wrapper for sgerqf */
void hook_sgerqf(blasint *m, blasint *n, float *a, blasint *lda, float *tau, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgerqf((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgerqf.timings[0] += (helpTimeStop - helpTime);
    data->sgerqf.calls[0]++;
    
    return;
}


/* Wrapper for sgesc2 */
void hook_sgesc2(blasint *n, float *a, blasint *lda, float *rhs, blasint *ipiv, blasint *jpiv, float *scale)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgesc2((void *) n, (void *) a, (void *) lda, (void *) rhs, (void *) ipiv, (void *) jpiv, (void *) scale);

    helpTimeStop = flexiblas_wtime();

    data->sgesc2.timings[0] += (helpTimeStop - helpTime);
    data->sgesc2.calls[0]++;
    
    return;
}


/* Wrapper for sgesdd */
void hook_sgesdd(char *jobz, blasint *m, blasint *n, float *a, blasint *lda, float *s, float *u, blasint *ldu, float *vt, blasint *ldvt, float *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t jobz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgesdd((void *) jobz, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) s, (void *) u, (void *) ldu, (void *) vt, (void *) ldvt, (void *) work, (void *) lwork, (void *) iwork, (void *) info, jobz_len);

    helpTimeStop = flexiblas_wtime();

    data->sgesdd.timings[0] += (helpTimeStop - helpTime);
    data->sgesdd.calls[0]++;
    
    return;
}


/* Wrapper for sgesv */
void hook_sgesv(blasint *n, blasint *nrhs, float *a, blasint *lda, blasint *ipiv, float *b, blasint *ldb, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgesv((void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgesv.timings[0] += (helpTimeStop - helpTime);
    data->sgesv.calls[0]++;
    
    return;
}


/* Wrapper for sgesvd */
void hook_sgesvd(char *jobu, char *jobvt, blasint *m, blasint *n, float *a, blasint *lda, float *s, float *u, blasint *ldu, float *vt, blasint *ldvt, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobvt_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgesvd((void *) jobu, (void *) jobvt, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) s, (void *) u, (void *) ldu, (void *) vt, (void *) ldvt, (void *) work, (void *) lwork, (void *) info, jobu_len, jobvt_len);

    helpTimeStop = flexiblas_wtime();

    data->sgesvd.timings[0] += (helpTimeStop - helpTime);
    data->sgesvd.calls[0]++;
    
    return;
}


/* Wrapper for sgesvdq */
void hook_sgesvdq(char *joba, char *jobp, char *jobr, char *jobu, char *jobv, blasint *m, blasint *n, float *a, blasint *lda, float *s, float *u, blasint *ldu, float *v, blasint *ldv, blasint *numrank, blasint *iwork, blasint *liwork, float *work, blasint *lwork, float *rwork, blasint *lrwork, blasint *info, flexiblas_fortran_charlen_t joba_len, flexiblas_fortran_charlen_t jobp_len, flexiblas_fortran_charlen_t jobr_len, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobv_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgesvdq((void *) joba, (void *) jobp, (void *) jobr, (void *) jobu, (void *) jobv, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) s, (void *) u, (void *) ldu, (void *) v, (void *) ldv, (void *) numrank, (void *) iwork, (void *) liwork, (void *) work, (void *) lwork, (void *) rwork, (void *) lrwork, (void *) info, joba_len, jobp_len, jobr_len, jobu_len, jobv_len);

    helpTimeStop = flexiblas_wtime();

    data->sgesvdq.timings[0] += (helpTimeStop - helpTime);
    data->sgesvdq.calls[0]++;
    
    return;
}


/* Wrapper for sgesvdx */
void hook_sgesvdx(char *jobu, char *jobvt, char *range, blasint *m, blasint *n, float *a, blasint *lda, float *vl, float *vu, blasint *il, blasint *iu, blasint *ns, float *s, float *u, blasint *ldu, float *vt, blasint *ldvt, float *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobvt_len, flexiblas_fortran_charlen_t range_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgesvdx((void *) jobu, (void *) jobvt, (void *) range, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) ns, (void *) s, (void *) u, (void *) ldu, (void *) vt, (void *) ldvt, (void *) work, (void *) lwork, (void *) iwork, (void *) info, jobu_len, jobvt_len, range_len);

    helpTimeStop = flexiblas_wtime();

    data->sgesvdx.timings[0] += (helpTimeStop - helpTime);
    data->sgesvdx.calls[0]++;
    
    return;
}


/* Wrapper for sgesvj */
void hook_sgesvj(char *joba, char *jobu, char *jobv, blasint *m, blasint *n, float *a, blasint *lda, float *sva, blasint *mv, float *v, blasint *ldv, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t joba_len, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobv_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgesvj((void *) joba, (void *) jobu, (void *) jobv, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) sva, (void *) mv, (void *) v, (void *) ldv, (void *) work, (void *) lwork, (void *) info, joba_len, jobu_len, jobv_len);

    helpTimeStop = flexiblas_wtime();

    data->sgesvj.timings[0] += (helpTimeStop - helpTime);
    data->sgesvj.calls[0]++;
    
    return;
}


/* Wrapper for sgesvx */
void hook_sgesvx(char *fact, char *trans, blasint *n, blasint *nrhs, float *a, blasint *lda, float *af, blasint *ldaf, blasint *ipiv, char *equed, float *r, float *c, float *b, blasint *ldb, float *x, blasint *ldx, float *rcond, float *ferr, float *berr, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgesvx((void *) fact, (void *) trans, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) af, (void *) ldaf, (void *) ipiv, (void *) equed, (void *) r, (void *) c, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, fact_len, trans_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->sgesvx.timings[0] += (helpTimeStop - helpTime);
    data->sgesvx.calls[0]++;
    
    return;
}


/* Wrapper for sgetc2 */
void hook_sgetc2(blasint *n, float *a, blasint *lda, blasint *ipiv, blasint *jpiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgetc2((void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) jpiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgetc2.timings[0] += (helpTimeStop - helpTime);
    data->sgetc2.calls[0]++;
    
    return;
}


/* Wrapper for sgetf2 */
void hook_sgetf2(blasint *m, blasint *n, float *a, blasint *lda, blasint *ipiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgetf2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgetf2.timings[0] += (helpTimeStop - helpTime);
    data->sgetf2.calls[0]++;
    
    return;
}


/* Wrapper for sgetrf */
void hook_sgetrf(blasint *m, blasint *n, float *a, blasint *lda, blasint *ipiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgetrf((void *) m, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgetrf.timings[0] += (helpTimeStop - helpTime);
    data->sgetrf.calls[0]++;
    
    return;
}


/* Wrapper for sgetrf2 */
void hook_sgetrf2(blasint *m, blasint *n, float *a, blasint *lda, blasint *ipiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgetrf2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgetrf2.timings[0] += (helpTimeStop - helpTime);
    data->sgetrf2.calls[0]++;
    
    return;
}


/* Wrapper for sgetri */
void hook_sgetri(blasint *n, float *a, blasint *lda, blasint *ipiv, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgetri((void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgetri.timings[0] += (helpTimeStop - helpTime);
    data->sgetri.calls[0]++;
    
    return;
}


/* Wrapper for sgetrs */
void hook_sgetrs(char *trans, blasint *n, blasint *nrhs, float *a, blasint *lda, blasint *ipiv, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgetrs((void *) trans, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sgetrs.timings[0] += (helpTimeStop - helpTime);
    data->sgetrs.calls[0]++;
    
    return;
}


/* Wrapper for sgetsls */
void hook_sgetsls(char *trans, blasint *m, blasint *n, blasint *nrhs, float *a, blasint *lda, float *b, blasint *ldb, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgetsls((void *) trans, (void *) m, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sgetsls.timings[0] += (helpTimeStop - helpTime);
    data->sgetsls.calls[0]++;
    
    return;
}


/* Wrapper for sgetsqrhrt */
void hook_sgetsqrhrt(blasint *m, blasint *n, blasint *mb1, blasint *nb1, blasint *nb2, float *a, blasint *lda, float *t, blasint *ldt, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgetsqrhrt((void *) m, (void *) n, (void *) mb1, (void *) nb1, (void *) nb2, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgetsqrhrt.timings[0] += (helpTimeStop - helpTime);
    data->sgetsqrhrt.calls[0]++;
    
    return;
}


/* Wrapper for sggbak */
void hook_sggbak(char *job, char *side, blasint *n, blasint *ilo, blasint *ihi, float *lscale, float *rscale, blasint *m, float *v, blasint *ldv, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t side_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sggbak((void *) job, (void *) side, (void *) n, (void *) ilo, (void *) ihi, (void *) lscale, (void *) rscale, (void *) m, (void *) v, (void *) ldv, (void *) info, job_len, side_len);

    helpTimeStop = flexiblas_wtime();

    data->sggbak.timings[0] += (helpTimeStop - helpTime);
    data->sggbak.calls[0]++;
    
    return;
}


/* Wrapper for sggbal */
void hook_sggbal(char *job, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, blasint *ilo, blasint *ihi, float *lscale, float *rscale, float *work, blasint *info, flexiblas_fortran_charlen_t job_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sggbal((void *) job, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) ilo, (void *) ihi, (void *) lscale, (void *) rscale, (void *) work, (void *) info, job_len);

    helpTimeStop = flexiblas_wtime();

    data->sggbal.timings[0] += (helpTimeStop - helpTime);
    data->sggbal.calls[0]++;
    
    return;
}


/* Wrapper for sgges */
void hook_sgges(char *jobvsl, char *jobvsr, char *sort, blaslogical *selctg, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, blasint *sdim, float *alphar, float *alphai, float *beta, float *vsl, blasint *ldvsl, float *vsr, blasint *ldvsr, float *work, blasint *lwork, blaslogical *bwork, blasint *info, flexiblas_fortran_charlen_t jobvsl_len, flexiblas_fortran_charlen_t jobvsr_len, flexiblas_fortran_charlen_t sort_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgges((void *) jobvsl, (void *) jobvsr, (void *) sort, (void *) selctg, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) sdim, (void *) alphar, (void *) alphai, (void *) beta, (void *) vsl, (void *) ldvsl, (void *) vsr, (void *) ldvsr, (void *) work, (void *) lwork, (void *) bwork, (void *) info, jobvsl_len, jobvsr_len, sort_len);

    helpTimeStop = flexiblas_wtime();

    data->sgges.timings[0] += (helpTimeStop - helpTime);
    data->sgges.calls[0]++;
    
    return;
}


/* Wrapper for sgges3 */
void hook_sgges3(char *jobvsl, char *jobvsr, char *sort, blaslogical *selctg, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, blasint *sdim, float *alphar, float *alphai, float *beta, float *vsl, blasint *ldvsl, float *vsr, blasint *ldvsr, float *work, blasint *lwork, blaslogical *bwork, blasint *info, flexiblas_fortran_charlen_t jobvsl_len, flexiblas_fortran_charlen_t jobvsr_len, flexiblas_fortran_charlen_t sort_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgges3((void *) jobvsl, (void *) jobvsr, (void *) sort, (void *) selctg, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) sdim, (void *) alphar, (void *) alphai, (void *) beta, (void *) vsl, (void *) ldvsl, (void *) vsr, (void *) ldvsr, (void *) work, (void *) lwork, (void *) bwork, (void *) info, jobvsl_len, jobvsr_len, sort_len);

    helpTimeStop = flexiblas_wtime();

    data->sgges3.timings[0] += (helpTimeStop - helpTime);
    data->sgges3.calls[0]++;
    
    return;
}


/* Wrapper for sggesx */
void hook_sggesx(char *jobvsl, char *jobvsr, char *sort, blaslogical *selctg, char *sense, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, blasint *sdim, float *alphar, float *alphai, float *beta, float *vsl, blasint *ldvsl, float *vsr, blasint *ldvsr, float *rconde, float *rcondv, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blaslogical *bwork, blasint *info, flexiblas_fortran_charlen_t jobvsl_len, flexiblas_fortran_charlen_t jobvsr_len, flexiblas_fortran_charlen_t sort_len, flexiblas_fortran_charlen_t sense_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sggesx((void *) jobvsl, (void *) jobvsr, (void *) sort, (void *) selctg, (void *) sense, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) sdim, (void *) alphar, (void *) alphai, (void *) beta, (void *) vsl, (void *) ldvsl, (void *) vsr, (void *) ldvsr, (void *) rconde, (void *) rcondv, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) bwork, (void *) info, jobvsl_len, jobvsr_len, sort_len, sense_len);

    helpTimeStop = flexiblas_wtime();

    data->sggesx.timings[0] += (helpTimeStop - helpTime);
    data->sggesx.calls[0]++;
    
    return;
}


/* Wrapper for sggev */
void hook_sggev(char *jobvl, char *jobvr, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, float *alphar, float *alphai, float *beta, float *vl, blasint *ldvl, float *vr, blasint *ldvr, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobvl_len, flexiblas_fortran_charlen_t jobvr_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sggev((void *) jobvl, (void *) jobvr, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) alphar, (void *) alphai, (void *) beta, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) work, (void *) lwork, (void *) info, jobvl_len, jobvr_len);

    helpTimeStop = flexiblas_wtime();

    data->sggev.timings[0] += (helpTimeStop - helpTime);
    data->sggev.calls[0]++;
    
    return;
}


/* Wrapper for sggev3 */
void hook_sggev3(char *jobvl, char *jobvr, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, float *alphar, float *alphai, float *beta, float *vl, blasint *ldvl, float *vr, blasint *ldvr, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobvl_len, flexiblas_fortran_charlen_t jobvr_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sggev3((void *) jobvl, (void *) jobvr, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) alphar, (void *) alphai, (void *) beta, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) work, (void *) lwork, (void *) info, jobvl_len, jobvr_len);

    helpTimeStop = flexiblas_wtime();

    data->sggev3.timings[0] += (helpTimeStop - helpTime);
    data->sggev3.calls[0]++;
    
    return;
}


/* Wrapper for sggevx */
void hook_sggevx(char *balanc, char *jobvl, char *jobvr, char *sense, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, float *alphar, float *alphai, float *beta, float *vl, blasint *ldvl, float *vr, blasint *ldvr, blasint *ilo, blasint *ihi, float *lscale, float *rscale, float *abnrm, float *bbnrm, float *rconde, float *rcondv, float *work, blasint *lwork, blasint *iwork, blaslogical *bwork, blasint *info, flexiblas_fortran_charlen_t balanc_len, flexiblas_fortran_charlen_t jobvl_len, flexiblas_fortran_charlen_t jobvr_len, flexiblas_fortran_charlen_t sense_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sggevx((void *) balanc, (void *) jobvl, (void *) jobvr, (void *) sense, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) alphar, (void *) alphai, (void *) beta, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) ilo, (void *) ihi, (void *) lscale, (void *) rscale, (void *) abnrm, (void *) bbnrm, (void *) rconde, (void *) rcondv, (void *) work, (void *) lwork, (void *) iwork, (void *) bwork, (void *) info, balanc_len, jobvl_len, jobvr_len, sense_len);

    helpTimeStop = flexiblas_wtime();

    data->sggevx.timings[0] += (helpTimeStop - helpTime);
    data->sggevx.calls[0]++;
    
    return;
}


/* Wrapper for sggglm */
void hook_sggglm(blasint *n, blasint *m, blasint *p, float *a, blasint *lda, float *b, blasint *ldb, float *d, float *x, float *y, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sggglm((void *) n, (void *) m, (void *) p, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) d, (void *) x, (void *) y, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sggglm.timings[0] += (helpTimeStop - helpTime);
    data->sggglm.calls[0]++;
    
    return;
}


/* Wrapper for sgghd3 */
void hook_sgghd3(char *compq, char *compz, blasint *n, blasint *ilo, blasint *ihi, float *a, blasint *lda, float *b, blasint *ldb, float *q, blasint *ldq, float *z, blasint *ldz, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t compq_len, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgghd3((void *) compq, (void *) compz, (void *) n, (void *) ilo, (void *) ihi, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) q, (void *) ldq, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) info, compq_len, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->sgghd3.timings[0] += (helpTimeStop - helpTime);
    data->sgghd3.calls[0]++;
    
    return;
}


/* Wrapper for sgghrd */
void hook_sgghrd(char *compq, char *compz, blasint *n, blasint *ilo, blasint *ihi, float *a, blasint *lda, float *b, blasint *ldb, float *q, blasint *ldq, float *z, blasint *ldz, blasint *info, flexiblas_fortran_charlen_t compq_len, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgghrd((void *) compq, (void *) compz, (void *) n, (void *) ilo, (void *) ihi, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) q, (void *) ldq, (void *) z, (void *) ldz, (void *) info, compq_len, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->sgghrd.timings[0] += (helpTimeStop - helpTime);
    data->sgghrd.calls[0]++;
    
    return;
}


/* Wrapper for sgglse */
void hook_sgglse(blasint *m, blasint *n, blasint *p, float *a, blasint *lda, float *b, blasint *ldb, float *c, float *d, float *x, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgglse((void *) m, (void *) n, (void *) p, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) c, (void *) d, (void *) x, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgglse.timings[0] += (helpTimeStop - helpTime);
    data->sgglse.calls[0]++;
    
    return;
}


/* Wrapper for sggqrf */
void hook_sggqrf(blasint *n, blasint *m, blasint *p, float *a, blasint *lda, float *taua, float *b, blasint *ldb, float *taub, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sggqrf((void *) n, (void *) m, (void *) p, (void *) a, (void *) lda, (void *) taua, (void *) b, (void *) ldb, (void *) taub, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sggqrf.timings[0] += (helpTimeStop - helpTime);
    data->sggqrf.calls[0]++;
    
    return;
}


/* Wrapper for sggrqf */
void hook_sggrqf(blasint *m, blasint *p, blasint *n, float *a, blasint *lda, float *taua, float *b, blasint *ldb, float *taub, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sggrqf((void *) m, (void *) p, (void *) n, (void *) a, (void *) lda, (void *) taua, (void *) b, (void *) ldb, (void *) taub, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sggrqf.timings[0] += (helpTimeStop - helpTime);
    data->sggrqf.calls[0]++;
    
    return;
}


/* Wrapper for sggsvd3 */
void hook_sggsvd3(char *jobu, char *jobv, char *jobq, blasint *m, blasint *n, blasint *p, blasint *k, blasint *l, float *a, blasint *lda, float *b, blasint *ldb, float *alpha, float *beta, float *u, blasint *ldu, float *v, blasint *ldv, float *q, blasint *ldq, float *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobv_len, flexiblas_fortran_charlen_t jobq_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sggsvd3((void *) jobu, (void *) jobv, (void *) jobq, (void *) m, (void *) n, (void *) p, (void *) k, (void *) l, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) alpha, (void *) beta, (void *) u, (void *) ldu, (void *) v, (void *) ldv, (void *) q, (void *) ldq, (void *) work, (void *) lwork, (void *) iwork, (void *) info, jobu_len, jobv_len, jobq_len);

    helpTimeStop = flexiblas_wtime();

    data->sggsvd3.timings[0] += (helpTimeStop - helpTime);
    data->sggsvd3.calls[0]++;
    
    return;
}


/* Wrapper for sggsvp3 */
void hook_sggsvp3(char *jobu, char *jobv, char *jobq, blasint *m, blasint *p, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, float *tola, float *tolb, blasint *k, blasint *l, float *u, blasint *ldu, float *v, blasint *ldv, float *q, blasint *ldq, blasint *iwork, float *tau, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobv_len, flexiblas_fortran_charlen_t jobq_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sggsvp3((void *) jobu, (void *) jobv, (void *) jobq, (void *) m, (void *) p, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) tola, (void *) tolb, (void *) k, (void *) l, (void *) u, (void *) ldu, (void *) v, (void *) ldv, (void *) q, (void *) ldq, (void *) iwork, (void *) tau, (void *) work, (void *) lwork, (void *) info, jobu_len, jobv_len, jobq_len);

    helpTimeStop = flexiblas_wtime();

    data->sggsvp3.timings[0] += (helpTimeStop - helpTime);
    data->sggsvp3.calls[0]++;
    
    return;
}


/* Wrapper for sgsvj0 */
void hook_sgsvj0(char *jobv, blasint *m, blasint *n, float *a, blasint *lda, float *d, float *sva, blasint *mv, float *v, blasint *ldv, float *eps, float *sfmin, float *tol, blasint *nsweep, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobv_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgsvj0((void *) jobv, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) sva, (void *) mv, (void *) v, (void *) ldv, (void *) eps, (void *) sfmin, (void *) tol, (void *) nsweep, (void *) work, (void *) lwork, (void *) info, jobv_len);

    helpTimeStop = flexiblas_wtime();

    data->sgsvj0.timings[0] += (helpTimeStop - helpTime);
    data->sgsvj0.calls[0]++;
    
    return;
}


/* Wrapper for sgsvj1 */
void hook_sgsvj1(char *jobv, blasint *m, blasint *n, blasint *n1, float *a, blasint *lda, float *d, float *sva, blasint *mv, float *v, blasint *ldv, float *eps, float *sfmin, float *tol, blasint *nsweep, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobv_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgsvj1((void *) jobv, (void *) m, (void *) n, (void *) n1, (void *) a, (void *) lda, (void *) d, (void *) sva, (void *) mv, (void *) v, (void *) ldv, (void *) eps, (void *) sfmin, (void *) tol, (void *) nsweep, (void *) work, (void *) lwork, (void *) info, jobv_len);

    helpTimeStop = flexiblas_wtime();

    data->sgsvj1.timings[0] += (helpTimeStop - helpTime);
    data->sgsvj1.calls[0]++;
    
    return;
}


/* Wrapper for sgtcon */
void hook_sgtcon(char *norm, blasint *n, float *dl, float *d, float *du, float *du2, blasint *ipiv, float *anorm, float *rcond, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgtcon((void *) norm, (void *) n, (void *) dl, (void *) d, (void *) du, (void *) du2, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) iwork, (void *) info, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->sgtcon.timings[0] += (helpTimeStop - helpTime);
    data->sgtcon.calls[0]++;
    
    return;
}


/* Wrapper for sgtrfs */
void hook_sgtrfs(char *trans, blasint *n, blasint *nrhs, float *dl, float *d, float *du, float *dlf, float *df, float *duf, float *du2, blasint *ipiv, float *b, blasint *ldb, float *x, blasint *ldx, float *ferr, float *berr, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgtrfs((void *) trans, (void *) n, (void *) nrhs, (void *) dl, (void *) d, (void *) du, (void *) dlf, (void *) df, (void *) duf, (void *) du2, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sgtrfs.timings[0] += (helpTimeStop - helpTime);
    data->sgtrfs.calls[0]++;
    
    return;
}


/* Wrapper for sgtsv */
void hook_sgtsv(blasint *n, blasint *nrhs, float *dl, float *d, float *du, float *b, blasint *ldb, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgtsv((void *) n, (void *) nrhs, (void *) dl, (void *) d, (void *) du, (void *) b, (void *) ldb, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgtsv.timings[0] += (helpTimeStop - helpTime);
    data->sgtsv.calls[0]++;
    
    return;
}


/* Wrapper for sgtsvx */
void hook_sgtsvx(char *fact, char *trans, blasint *n, blasint *nrhs, float *dl, float *d, float *du, float *dlf, float *df, float *duf, float *du2, blasint *ipiv, float *b, blasint *ldb, float *x, blasint *ldx, float *rcond, float *ferr, float *berr, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgtsvx((void *) fact, (void *) trans, (void *) n, (void *) nrhs, (void *) dl, (void *) d, (void *) du, (void *) dlf, (void *) df, (void *) duf, (void *) du2, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, fact_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sgtsvx.timings[0] += (helpTimeStop - helpTime);
    data->sgtsvx.calls[0]++;
    
    return;
}


/* Wrapper for sgttrf */
void hook_sgttrf(blasint *n, float *dl, float *d, float *du, float *du2, blasint *ipiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgttrf((void *) n, (void *) dl, (void *) d, (void *) du, (void *) du2, (void *) ipiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sgttrf.timings[0] += (helpTimeStop - helpTime);
    data->sgttrf.calls[0]++;
    
    return;
}


/* Wrapper for sgttrs */
void hook_sgttrs(char *trans, blasint *n, blasint *nrhs, float *dl, float *d, float *du, float *du2, blasint *ipiv, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgttrs((void *) trans, (void *) n, (void *) nrhs, (void *) dl, (void *) d, (void *) du, (void *) du2, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sgttrs.timings[0] += (helpTimeStop - helpTime);
    data->sgttrs.calls[0]++;
    
    return;
}


/* Wrapper for sgtts2 */
void hook_sgtts2(blasint *itrans, blasint *n, blasint *nrhs, float *dl, float *d, float *du, float *du2, blasint *ipiv, float *b, blasint *ldb)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sgtts2((void *) itrans, (void *) n, (void *) nrhs, (void *) dl, (void *) d, (void *) du, (void *) du2, (void *) ipiv, (void *) b, (void *) ldb);

    helpTimeStop = flexiblas_wtime();

    data->sgtts2.timings[0] += (helpTimeStop - helpTime);
    data->sgtts2.calls[0]++;
    
    return;
}


/* Wrapper for shgeqz */
void hook_shgeqz(char *job, char *compq, char *compz, blasint *n, blasint *ilo, blasint *ihi, float *h, blasint *ldh, float *t, blasint *ldt, float *alphar, float *alphai, float *beta, float *q, blasint *ldq, float *z, blasint *ldz, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t compq_len, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_shgeqz((void *) job, (void *) compq, (void *) compz, (void *) n, (void *) ilo, (void *) ihi, (void *) h, (void *) ldh, (void *) t, (void *) ldt, (void *) alphar, (void *) alphai, (void *) beta, (void *) q, (void *) ldq, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) info, job_len, compq_len, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->shgeqz.timings[0] += (helpTimeStop - helpTime);
    data->shgeqz.calls[0]++;
    
    return;
}


/* Wrapper for shsein */
void hook_shsein(char *side, char *eigsrc, char *initv, blaslogical *select, blasint *n, float *h, blasint *ldh, float *wr, float *wi, float *vl, blasint *ldvl, float *vr, blasint *ldvr, blasint *mm, blasint *m, float *work, blasint *ifaill, blasint *ifailr, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t eigsrc_len, flexiblas_fortran_charlen_t initv_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_shsein((void *) side, (void *) eigsrc, (void *) initv, (void *) select, (void *) n, (void *) h, (void *) ldh, (void *) wr, (void *) wi, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) mm, (void *) m, (void *) work, (void *) ifaill, (void *) ifailr, (void *) info, side_len, eigsrc_len, initv_len);

    helpTimeStop = flexiblas_wtime();

    data->shsein.timings[0] += (helpTimeStop - helpTime);
    data->shsein.calls[0]++;
    
    return;
}


/* Wrapper for shseqr */
void hook_shseqr(char *job, char *compz, blasint *n, blasint *ilo, blasint *ihi, float *h, blasint *ldh, float *wr, float *wi, float *z, blasint *ldz, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_shseqr((void *) job, (void *) compz, (void *) n, (void *) ilo, (void *) ihi, (void *) h, (void *) ldh, (void *) wr, (void *) wi, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) info, job_len, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->shseqr.timings[0] += (helpTimeStop - helpTime);
    data->shseqr.calls[0]++;
    
    return;
}

/* Wrapper for sisnan */
int_least32_t hook_sisnan(float *sin)
{
    double helpTime;
    double helpTimeStop;
    int_least32_t return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_sisnan((void *) sin);

    helpTimeStop = flexiblas_wtime();

    data->sisnan.timings[0] += (helpTimeStop - helpTime);
    data->sisnan.calls[0]++;
    
    return return_value;
}



/* Wrapper for slabad */
void hook_slabad(float *small, float *large)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slabad((void *) small, (void *) large);

    helpTimeStop = flexiblas_wtime();

    data->slabad.timings[0] += (helpTimeStop - helpTime);
    data->slabad.calls[0]++;
    
    return;
}


/* Wrapper for slabrd */
void hook_slabrd(blasint *m, blasint *n, blasint *nb, float *a, blasint *lda, float *d, float *e, float *tauq, float *taup, float *x, blasint *ldx, float *y, blasint *ldy)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slabrd((void *) m, (void *) n, (void *) nb, (void *) a, (void *) lda, (void *) d, (void *) e, (void *) tauq, (void *) taup, (void *) x, (void *) ldx, (void *) y, (void *) ldy);

    helpTimeStop = flexiblas_wtime();

    data->slabrd.timings[0] += (helpTimeStop - helpTime);
    data->slabrd.calls[0]++;
    
    return;
}


/* Wrapper for slacn2 */
void hook_slacn2(blasint *n, float *v, float *x, blasint *isgn, float *est, blasint *kase, blasint *isave)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slacn2((void *) n, (void *) v, (void *) x, (void *) isgn, (void *) est, (void *) kase, (void *) isave);

    helpTimeStop = flexiblas_wtime();

    data->slacn2.timings[0] += (helpTimeStop - helpTime);
    data->slacn2.calls[0]++;
    
    return;
}


/* Wrapper for slacon */
void hook_slacon(blasint *n, float *v, float *x, blasint *isgn, float *est, blasint *kase)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slacon((void *) n, (void *) v, (void *) x, (void *) isgn, (void *) est, (void *) kase);

    helpTimeStop = flexiblas_wtime();

    data->slacon.timings[0] += (helpTimeStop - helpTime);
    data->slacon.calls[0]++;
    
    return;
}


/* Wrapper for slacpy */
void hook_slacpy(char *uplo, blasint *m, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slacpy((void *) uplo, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->slacpy.timings[0] += (helpTimeStop - helpTime);
    data->slacpy.calls[0]++;
    
    return;
}


/* Wrapper for sladiv */
void hook_sladiv(float *a, float *b, float *c, float *d, float *p, float *q)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sladiv((void *) a, (void *) b, (void *) c, (void *) d, (void *) p, (void *) q);

    helpTimeStop = flexiblas_wtime();

    data->sladiv.timings[0] += (helpTimeStop - helpTime);
    data->sladiv.calls[0]++;
    
    return;
}


/* Wrapper for sladiv1 */
void hook_sladiv1(float *a, float *b, float *c, float *d, float *p, float *q)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sladiv1((void *) a, (void *) b, (void *) c, (void *) d, (void *) p, (void *) q);

    helpTimeStop = flexiblas_wtime();

    data->sladiv1.timings[0] += (helpTimeStop - helpTime);
    data->sladiv1.calls[0]++;
    
    return;
}

/* Wrapper for sladiv2 */
float hook_sladiv2(float *a, float *b, float *c, float *d, float *r, float *t)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_sladiv2((void *) a, (void *) b, (void *) c, (void *) d, (void *) r, (void *) t);

    helpTimeStop = flexiblas_wtime();

    data->sladiv2.timings[0] += (helpTimeStop - helpTime);
    data->sladiv2.calls[0]++;
    
    return return_value;
}



/* Wrapper for slae2 */
void hook_slae2(float *a, float *b, float *c, float *rt1, float *rt2)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slae2((void *) a, (void *) b, (void *) c, (void *) rt1, (void *) rt2);

    helpTimeStop = flexiblas_wtime();

    data->slae2.timings[0] += (helpTimeStop - helpTime);
    data->slae2.calls[0]++;
    
    return;
}


/* Wrapper for slaebz */
void hook_slaebz(blasint *ijob, blasint *nitmax, blasint *n, blasint *mmax, blasint *minp, blasint *nbmin, float *abstol, float *reltol, float *pivmin, float *d, float *e, float *e2, blasint *nval, float *ab, float *c, blasint *mout, blasint *nab, float *work, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaebz((void *) ijob, (void *) nitmax, (void *) n, (void *) mmax, (void *) minp, (void *) nbmin, (void *) abstol, (void *) reltol, (void *) pivmin, (void *) d, (void *) e, (void *) e2, (void *) nval, (void *) ab, (void *) c, (void *) mout, (void *) nab, (void *) work, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slaebz.timings[0] += (helpTimeStop - helpTime);
    data->slaebz.calls[0]++;
    
    return;
}


/* Wrapper for slaed0 */
void hook_slaed0(blasint *icompq, blasint *qsiz, blasint *n, float *d, float *e, float *q, blasint *ldq, float *qstore, blasint *ldqs, float *work, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaed0((void *) icompq, (void *) qsiz, (void *) n, (void *) d, (void *) e, (void *) q, (void *) ldq, (void *) qstore, (void *) ldqs, (void *) work, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slaed0.timings[0] += (helpTimeStop - helpTime);
    data->slaed0.calls[0]++;
    
    return;
}


/* Wrapper for slaed1 */
void hook_slaed1(blasint *n, float *d, float *q, blasint *ldq, blasint *indxq, float *rho, blasint *cutpnt, float *work, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaed1((void *) n, (void *) d, (void *) q, (void *) ldq, (void *) indxq, (void *) rho, (void *) cutpnt, (void *) work, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slaed1.timings[0] += (helpTimeStop - helpTime);
    data->slaed1.calls[0]++;
    
    return;
}


/* Wrapper for slaed2 */
void hook_slaed2(blasint *k, blasint *n, blasint *n1, float *d, float *q, blasint *ldq, blasint *indxq, float *rho, float *z, float *dlamda, float *w, float *q2, blasint *indx, blasint *indxc, blasint *indxp, blasint *coltyp, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaed2((void *) k, (void *) n, (void *) n1, (void *) d, (void *) q, (void *) ldq, (void *) indxq, (void *) rho, (void *) z, (void *) dlamda, (void *) w, (void *) q2, (void *) indx, (void *) indxc, (void *) indxp, (void *) coltyp, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slaed2.timings[0] += (helpTimeStop - helpTime);
    data->slaed2.calls[0]++;
    
    return;
}


/* Wrapper for slaed3 */
void hook_slaed3(blasint *k, blasint *n, blasint *n1, float *d, float *q, blasint *ldq, float *rho, float *dlamda, float *q2, blasint *indx, blasint *ctot, float *w, float *s, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaed3((void *) k, (void *) n, (void *) n1, (void *) d, (void *) q, (void *) ldq, (void *) rho, (void *) dlamda, (void *) q2, (void *) indx, (void *) ctot, (void *) w, (void *) s, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slaed3.timings[0] += (helpTimeStop - helpTime);
    data->slaed3.calls[0]++;
    
    return;
}


/* Wrapper for slaed4 */
void hook_slaed4(blasint *n, blasint *i, float *d, float *z, float *delta, float *rho, float *dlam, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaed4((void *) n, (void *) i, (void *) d, (void *) z, (void *) delta, (void *) rho, (void *) dlam, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slaed4.timings[0] += (helpTimeStop - helpTime);
    data->slaed4.calls[0]++;
    
    return;
}


/* Wrapper for slaed5 */
void hook_slaed5(blasint *i, float *d, float *z, float *delta, float *rho, float *dlam)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaed5((void *) i, (void *) d, (void *) z, (void *) delta, (void *) rho, (void *) dlam);

    helpTimeStop = flexiblas_wtime();

    data->slaed5.timings[0] += (helpTimeStop - helpTime);
    data->slaed5.calls[0]++;
    
    return;
}


/* Wrapper for slaed6 */
void hook_slaed6(blasint *kniter, blaslogical *orgati, float *rho, float *d, float *z, float *finit, float *tau, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaed6((void *) kniter, (void *) orgati, (void *) rho, (void *) d, (void *) z, (void *) finit, (void *) tau, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slaed6.timings[0] += (helpTimeStop - helpTime);
    data->slaed6.calls[0]++;
    
    return;
}


/* Wrapper for slaed7 */
void hook_slaed7(blasint *icompq, blasint *n, blasint *qsiz, blasint *tlvls, blasint *curlvl, blasint *curpbm, float *d, float *q, blasint *ldq, blasint *indxq, float *rho, blasint *cutpnt, float *qstore, blasint *qptr, blasint *prmptr, blasint *perm, blasint *givptr, blasint *givcol, float *givnum, float *work, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaed7((void *) icompq, (void *) n, (void *) qsiz, (void *) tlvls, (void *) curlvl, (void *) curpbm, (void *) d, (void *) q, (void *) ldq, (void *) indxq, (void *) rho, (void *) cutpnt, (void *) qstore, (void *) qptr, (void *) prmptr, (void *) perm, (void *) givptr, (void *) givcol, (void *) givnum, (void *) work, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slaed7.timings[0] += (helpTimeStop - helpTime);
    data->slaed7.calls[0]++;
    
    return;
}


/* Wrapper for slaed8 */
void hook_slaed8(blasint *icompq, blasint *k, blasint *n, blasint *qsiz, float *d, float *q, blasint *ldq, blasint *indxq, float *rho, blasint *cutpnt, float *z, float *dlamda, float *q2, blasint *ldq2, float *w, blasint *perm, blasint *givptr, blasint *givcol, float *givnum, blasint *indxp, blasint *indx, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaed8((void *) icompq, (void *) k, (void *) n, (void *) qsiz, (void *) d, (void *) q, (void *) ldq, (void *) indxq, (void *) rho, (void *) cutpnt, (void *) z, (void *) dlamda, (void *) q2, (void *) ldq2, (void *) w, (void *) perm, (void *) givptr, (void *) givcol, (void *) givnum, (void *) indxp, (void *) indx, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slaed8.timings[0] += (helpTimeStop - helpTime);
    data->slaed8.calls[0]++;
    
    return;
}


/* Wrapper for slaed9 */
void hook_slaed9(blasint *k, blasint *kstart, blasint *kstop, blasint *n, float *d, float *q, blasint *ldq, float *rho, float *dlamda, float *w, float *s, blasint *lds, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaed9((void *) k, (void *) kstart, (void *) kstop, (void *) n, (void *) d, (void *) q, (void *) ldq, (void *) rho, (void *) dlamda, (void *) w, (void *) s, (void *) lds, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slaed9.timings[0] += (helpTimeStop - helpTime);
    data->slaed9.calls[0]++;
    
    return;
}


/* Wrapper for slaeda */
void hook_slaeda(blasint *n, blasint *tlvls, blasint *curlvl, blasint *curpbm, blasint *prmptr, blasint *perm, blasint *givptr, blasint *givcol, float *givnum, float *q, blasint *qptr, float *z, float *ztemp, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaeda((void *) n, (void *) tlvls, (void *) curlvl, (void *) curpbm, (void *) prmptr, (void *) perm, (void *) givptr, (void *) givcol, (void *) givnum, (void *) q, (void *) qptr, (void *) z, (void *) ztemp, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slaeda.timings[0] += (helpTimeStop - helpTime);
    data->slaeda.calls[0]++;
    
    return;
}


/* Wrapper for slaein */
void hook_slaein(blaslogical *rightv, blaslogical *noinit, blasint *n, float *h, blasint *ldh, float *wr, float *wi, float *vr, float *vi, float *b, blasint *ldb, float *work, float *eps3, float *smlnum, float *bignum, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaein((void *) rightv, (void *) noinit, (void *) n, (void *) h, (void *) ldh, (void *) wr, (void *) wi, (void *) vr, (void *) vi, (void *) b, (void *) ldb, (void *) work, (void *) eps3, (void *) smlnum, (void *) bignum, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slaein.timings[0] += (helpTimeStop - helpTime);
    data->slaein.calls[0]++;
    
    return;
}


/* Wrapper for slaev2 */
void hook_slaev2(float *a, float *b, float *c, float *rt1, float *rt2, float *cs1, float *sn1)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaev2((void *) a, (void *) b, (void *) c, (void *) rt1, (void *) rt2, (void *) cs1, (void *) sn1);

    helpTimeStop = flexiblas_wtime();

    data->slaev2.timings[0] += (helpTimeStop - helpTime);
    data->slaev2.calls[0]++;
    
    return;
}


/* Wrapper for slaexc */
void hook_slaexc(blaslogical *wantq, blasint *n, float *t, blasint *ldt, float *q, blasint *ldq, blasint *j1, blasint *n1, blasint *n2, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaexc((void *) wantq, (void *) n, (void *) t, (void *) ldt, (void *) q, (void *) ldq, (void *) j1, (void *) n1, (void *) n2, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slaexc.timings[0] += (helpTimeStop - helpTime);
    data->slaexc.calls[0]++;
    
    return;
}


/* Wrapper for slag2 */
void hook_slag2(float *a, blasint *lda, float *b, blasint *ldb, float *safmin, float *scale1, float *scale2, float *wr1, float *wr2, float *wi)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slag2((void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) safmin, (void *) scale1, (void *) scale2, (void *) wr1, (void *) wr2, (void *) wi);

    helpTimeStop = flexiblas_wtime();

    data->slag2.timings[0] += (helpTimeStop - helpTime);
    data->slag2.calls[0]++;
    
    return;
}


/* Wrapper for slag2d */
void hook_slag2d(blasint *m, blasint *n, float *sa, blasint *ldsa, double *a, blasint *lda, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slag2d((void *) m, (void *) n, (void *) sa, (void *) ldsa, (void *) a, (void *) lda, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slag2d.timings[0] += (helpTimeStop - helpTime);
    data->slag2d.calls[0]++;
    
    return;
}


/* Wrapper for slags2 */
void hook_slags2(blaslogical *upper, float *a1, float *a2, float *a3, float *b1, float *b2, float *b3, float *csu, float *snu, float *csv, float *snv, float *csq, float *snq)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slags2((void *) upper, (void *) a1, (void *) a2, (void *) a3, (void *) b1, (void *) b2, (void *) b3, (void *) csu, (void *) snu, (void *) csv, (void *) snv, (void *) csq, (void *) snq);

    helpTimeStop = flexiblas_wtime();

    data->slags2.timings[0] += (helpTimeStop - helpTime);
    data->slags2.calls[0]++;
    
    return;
}


/* Wrapper for slagtf */
void hook_slagtf(blasint *n, float *a, float *lambda, float *b, float *c, float *tol, float *d, blasint *in, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slagtf((void *) n, (void *) a, (void *) lambda, (void *) b, (void *) c, (void *) tol, (void *) d, (void *) in, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slagtf.timings[0] += (helpTimeStop - helpTime);
    data->slagtf.calls[0]++;
    
    return;
}


/* Wrapper for slagtm */
void hook_slagtm(char *trans, blasint *n, blasint *nrhs, float *alpha, float *dl, float *d, float *du, float *x, blasint *ldx, float *beta, float *b, blasint *ldb, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slagtm((void *) trans, (void *) n, (void *) nrhs, (void *) alpha, (void *) dl, (void *) d, (void *) du, (void *) x, (void *) ldx, (void *) beta, (void *) b, (void *) ldb, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->slagtm.timings[0] += (helpTimeStop - helpTime);
    data->slagtm.calls[0]++;
    
    return;
}


/* Wrapper for slagts */
void hook_slagts(blasint *job, blasint *n, float *a, float *b, float *c, float *d, blasint *in, float *y, float *tol, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slagts((void *) job, (void *) n, (void *) a, (void *) b, (void *) c, (void *) d, (void *) in, (void *) y, (void *) tol, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slagts.timings[0] += (helpTimeStop - helpTime);
    data->slagts.calls[0]++;
    
    return;
}


/* Wrapper for slagv2 */
void hook_slagv2(float *a, blasint *lda, float *b, blasint *ldb, float *alphar, float *alphai, float *beta, float *csl, float *snl, float *csr, float *snr)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slagv2((void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) alphar, (void *) alphai, (void *) beta, (void *) csl, (void *) snl, (void *) csr, (void *) snr);

    helpTimeStop = flexiblas_wtime();

    data->slagv2.timings[0] += (helpTimeStop - helpTime);
    data->slagv2.calls[0]++;
    
    return;
}


/* Wrapper for slahqr */
void hook_slahqr(blaslogical *wantt, blaslogical *wantz, blasint *n, blasint *ilo, blasint *ihi, float *h, blasint *ldh, float *wr, float *wi, blasint *iloz, blasint *ihiz, float *z, blasint *ldz, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slahqr((void *) wantt, (void *) wantz, (void *) n, (void *) ilo, (void *) ihi, (void *) h, (void *) ldh, (void *) wr, (void *) wi, (void *) iloz, (void *) ihiz, (void *) z, (void *) ldz, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slahqr.timings[0] += (helpTimeStop - helpTime);
    data->slahqr.calls[0]++;
    
    return;
}


/* Wrapper for slahr2 */
void hook_slahr2(blasint *n, blasint *k, blasint *nb, float *a, blasint *lda, float *tau, float *t, blasint *ldt, float *y, blasint *ldy)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slahr2((void *) n, (void *) k, (void *) nb, (void *) a, (void *) lda, (void *) tau, (void *) t, (void *) ldt, (void *) y, (void *) ldy);

    helpTimeStop = flexiblas_wtime();

    data->slahr2.timings[0] += (helpTimeStop - helpTime);
    data->slahr2.calls[0]++;
    
    return;
}


/* Wrapper for slaic1 */
void hook_slaic1(blasint *job, blasint *j, float *x, float *sest, float *w, float *gamma, float *sestpr, float *s, float *c)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaic1((void *) job, (void *) j, (void *) x, (void *) sest, (void *) w, (void *) gamma, (void *) sestpr, (void *) s, (void *) c);

    helpTimeStop = flexiblas_wtime();

    data->slaic1.timings[0] += (helpTimeStop - helpTime);
    data->slaic1.calls[0]++;
    
    return;
}

/* Wrapper for slaisnan */
int_least32_t hook_slaisnan(float *sin1, float *sin2)
{
    double helpTime;
    double helpTimeStop;
    int_least32_t return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_slaisnan((void *) sin1, (void *) sin2);

    helpTimeStop = flexiblas_wtime();

    data->slaisnan.timings[0] += (helpTimeStop - helpTime);
    data->slaisnan.calls[0]++;
    
    return return_value;
}



/* Wrapper for slaln2 */
void hook_slaln2(blaslogical *ltrans, blasint *na, blasint *nw, float *smin, float *ca, float *a, blasint *lda, float *d1, float *d2, float *b, blasint *ldb, float *wr, float *wi, float *x, blasint *ldx, float *scale, float *xnorm, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaln2((void *) ltrans, (void *) na, (void *) nw, (void *) smin, (void *) ca, (void *) a, (void *) lda, (void *) d1, (void *) d2, (void *) b, (void *) ldb, (void *) wr, (void *) wi, (void *) x, (void *) ldx, (void *) scale, (void *) xnorm, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slaln2.timings[0] += (helpTimeStop - helpTime);
    data->slaln2.calls[0]++;
    
    return;
}


/* Wrapper for slals0 */
void hook_slals0(blasint *icompq, blasint *nl, blasint *nr, blasint *sqre, blasint *nrhs, float *b, blasint *ldb, float *bx, blasint *ldbx, blasint *perm, blasint *givptr, blasint *givcol, blasint *ldgcol, float *givnum, blasint *ldgnum, float *poles, float *difl, float *difr, float *z, blasint *k, float *c, float *s, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slals0((void *) icompq, (void *) nl, (void *) nr, (void *) sqre, (void *) nrhs, (void *) b, (void *) ldb, (void *) bx, (void *) ldbx, (void *) perm, (void *) givptr, (void *) givcol, (void *) ldgcol, (void *) givnum, (void *) ldgnum, (void *) poles, (void *) difl, (void *) difr, (void *) z, (void *) k, (void *) c, (void *) s, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slals0.timings[0] += (helpTimeStop - helpTime);
    data->slals0.calls[0]++;
    
    return;
}


/* Wrapper for slalsa */
void hook_slalsa(blasint *icompq, blasint *smlsiz, blasint *n, blasint *nrhs, float *b, blasint *ldb, float *bx, blasint *ldbx, float *u, blasint *ldu, float *vt, blasint *k, float *difl, float *difr, float *z, float *poles, blasint *givptr, blasint *givcol, blasint *ldgcol, blasint *perm, float *givnum, float *c, float *s, float *work, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slalsa((void *) icompq, (void *) smlsiz, (void *) n, (void *) nrhs, (void *) b, (void *) ldb, (void *) bx, (void *) ldbx, (void *) u, (void *) ldu, (void *) vt, (void *) k, (void *) difl, (void *) difr, (void *) z, (void *) poles, (void *) givptr, (void *) givcol, (void *) ldgcol, (void *) perm, (void *) givnum, (void *) c, (void *) s, (void *) work, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slalsa.timings[0] += (helpTimeStop - helpTime);
    data->slalsa.calls[0]++;
    
    return;
}


/* Wrapper for slalsd */
void hook_slalsd(char *uplo, blasint *smlsiz, blasint *n, blasint *nrhs, float *d, float *e, float *b, blasint *ldb, float *rcond, blasint *rank, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slalsd((void *) uplo, (void *) smlsiz, (void *) n, (void *) nrhs, (void *) d, (void *) e, (void *) b, (void *) ldb, (void *) rcond, (void *) rank, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->slalsd.timings[0] += (helpTimeStop - helpTime);
    data->slalsd.calls[0]++;
    
    return;
}

/* Wrapper for slamc3 */
float hook_slamc3(float *a, float *b)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_slamc3((void *) a, (void *) b);

    helpTimeStop = flexiblas_wtime();

    data->slamc3.timings[0] += (helpTimeStop - helpTime);
    data->slamc3.calls[0]++;
    
    return return_value;
}


/* Wrapper for slamch */
float hook_slamch(char *cmach, flexiblas_fortran_charlen_t cmach_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_slamch((void *) cmach, cmach_len);

    helpTimeStop = flexiblas_wtime();

    data->slamch.timings[0] += (helpTimeStop - helpTime);
    data->slamch.calls[0]++;
    
    return return_value;
}



/* Wrapper for slamrg */
void hook_slamrg(blasint *n1, blasint *n2, float *a, blasint *strd1, blasint *strd2, blasint *index)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slamrg((void *) n1, (void *) n2, (void *) a, (void *) strd1, (void *) strd2, (void *) index);

    helpTimeStop = flexiblas_wtime();

    data->slamrg.timings[0] += (helpTimeStop - helpTime);
    data->slamrg.calls[0]++;
    
    return;
}


/* Wrapper for slamswlq */
void hook_slamswlq(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *mb, blasint *nb, float *a, blasint *lda, float *t, blasint *ldt, float *c, blasint *ldc, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slamswlq((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) mb, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->slamswlq.timings[0] += (helpTimeStop - helpTime);
    data->slamswlq.calls[0]++;
    
    return;
}


/* Wrapper for slamtsqr */
void hook_slamtsqr(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *mb, blasint *nb, float *a, blasint *lda, float *t, blasint *ldt, float *c, blasint *ldc, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slamtsqr((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) mb, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->slamtsqr.timings[0] += (helpTimeStop - helpTime);
    data->slamtsqr.calls[0]++;
    
    return;
}

/* Wrapper for slaneg */
blasint hook_slaneg(blasint *n, float *d, float *lld, float *sigma, float *pivmin, blasint *r)
{
    double helpTime;
    double helpTimeStop;
    blasint return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_slaneg((void *) n, (void *) d, (void *) lld, (void *) sigma, (void *) pivmin, (void *) r);

    helpTimeStop = flexiblas_wtime();

    data->slaneg.timings[0] += (helpTimeStop - helpTime);
    data->slaneg.calls[0]++;
    
    return return_value;
}


/* Wrapper for slangb */
float hook_slangb(char *norm, blasint *n, blasint *kl, blasint *ku, float *ab, blasint *ldab, float *work, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_slangb((void *) norm, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) work, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->slangb.timings[0] += (helpTimeStop - helpTime);
    data->slangb.calls[0]++;
    
    return return_value;
}


/* Wrapper for slange */
float hook_slange(char *norm, blasint *m, blasint *n, float *a, blasint *lda, float *work, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_slange((void *) norm, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) work, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->slange.timings[0] += (helpTimeStop - helpTime);
    data->slange.calls[0]++;
    
    return return_value;
}


/* Wrapper for slangt */
float hook_slangt(char *norm, blasint *n, float *dl, float *d, float *du, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_slangt((void *) norm, (void *) n, (void *) dl, (void *) d, (void *) du, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->slangt.timings[0] += (helpTimeStop - helpTime);
    data->slangt.calls[0]++;
    
    return return_value;
}


/* Wrapper for slanhs */
float hook_slanhs(char *norm, blasint *n, float *a, blasint *lda, float *work, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_slanhs((void *) norm, (void *) n, (void *) a, (void *) lda, (void *) work, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->slanhs.timings[0] += (helpTimeStop - helpTime);
    data->slanhs.calls[0]++;
    
    return return_value;
}


/* Wrapper for slansb */
float hook_slansb(char *norm, char *uplo, blasint *n, blasint *k, float *ab, blasint *ldab, float *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_slansb((void *) norm, (void *) uplo, (void *) n, (void *) k, (void *) ab, (void *) ldab, (void *) work, norm_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->slansb.timings[0] += (helpTimeStop - helpTime);
    data->slansb.calls[0]++;
    
    return return_value;
}


/* Wrapper for slansf */
float hook_slansf(char *norm, char *transr, char *uplo, blasint *n, float *a, float *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_slansf((void *) norm, (void *) transr, (void *) uplo, (void *) n, (void *) a, (void *) work, norm_len, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->slansf.timings[0] += (helpTimeStop - helpTime);
    data->slansf.calls[0]++;
    
    return return_value;
}


/* Wrapper for slansp */
float hook_slansp(char *norm, char *uplo, blasint *n, float *ap, float *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_slansp((void *) norm, (void *) uplo, (void *) n, (void *) ap, (void *) work, norm_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->slansp.timings[0] += (helpTimeStop - helpTime);
    data->slansp.calls[0]++;
    
    return return_value;
}


/* Wrapper for slanst */
float hook_slanst(char *norm, blasint *n, float *d, float *e, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_slanst((void *) norm, (void *) n, (void *) d, (void *) e, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->slanst.timings[0] += (helpTimeStop - helpTime);
    data->slanst.calls[0]++;
    
    return return_value;
}


/* Wrapper for slansy */
float hook_slansy(char *norm, char *uplo, blasint *n, float *a, blasint *lda, float *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_slansy((void *) norm, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) work, norm_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->slansy.timings[0] += (helpTimeStop - helpTime);
    data->slansy.calls[0]++;
    
    return return_value;
}


/* Wrapper for slantb */
float hook_slantb(char *norm, char *uplo, char *diag, blasint *n, blasint *k, float *ab, blasint *ldab, float *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_slantb((void *) norm, (void *) uplo, (void *) diag, (void *) n, (void *) k, (void *) ab, (void *) ldab, (void *) work, norm_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->slantb.timings[0] += (helpTimeStop - helpTime);
    data->slantb.calls[0]++;
    
    return return_value;
}


/* Wrapper for slantp */
float hook_slantp(char *norm, char *uplo, char *diag, blasint *n, float *ap, float *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_slantp((void *) norm, (void *) uplo, (void *) diag, (void *) n, (void *) ap, (void *) work, norm_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->slantp.timings[0] += (helpTimeStop - helpTime);
    data->slantp.calls[0]++;
    
    return return_value;
}


/* Wrapper for slantr */
float hook_slantr(char *norm, char *uplo, char *diag, blasint *m, blasint *n, float *a, blasint *lda, float *work, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_slantr((void *) norm, (void *) uplo, (void *) diag, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) work, norm_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->slantr.timings[0] += (helpTimeStop - helpTime);
    data->slantr.calls[0]++;
    
    return return_value;
}



/* Wrapper for slanv2 */
void hook_slanv2(float *a, float *b, float *c, float *d, float *rt1r, float *rt1i, float *rt2r, float *rt2i, float *cs, float *sn)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slanv2((void *) a, (void *) b, (void *) c, (void *) d, (void *) rt1r, (void *) rt1i, (void *) rt2r, (void *) rt2i, (void *) cs, (void *) sn);

    helpTimeStop = flexiblas_wtime();

    data->slanv2.timings[0] += (helpTimeStop - helpTime);
    data->slanv2.calls[0]++;
    
    return;
}


/* Wrapper for slaorhr_col_getrfnp */
void hook_slaorhr_col_getrfnp(blasint *m, blasint *n, float *a, blasint *lda, float *d, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaorhr_col_getrfnp((void *) m, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slaorhr_col_getrfnp.timings[0] += (helpTimeStop - helpTime);
    data->slaorhr_col_getrfnp.calls[0]++;
    
    return;
}


/* Wrapper for slaorhr_col_getrfnp2 */
void hook_slaorhr_col_getrfnp2(blasint *m, blasint *n, float *a, blasint *lda, float *d, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaorhr_col_getrfnp2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slaorhr_col_getrfnp2.timings[0] += (helpTimeStop - helpTime);
    data->slaorhr_col_getrfnp2.calls[0]++;
    
    return;
}


/* Wrapper for slapll */
void hook_slapll(blasint *n, float *x, blasint *incx, float *y, blasint *incy, float *ssmin)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slapll((void *) n, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ssmin);

    helpTimeStop = flexiblas_wtime();

    data->slapll.timings[0] += (helpTimeStop - helpTime);
    data->slapll.calls[0]++;
    
    return;
}


/* Wrapper for slapmr */
void hook_slapmr(blaslogical *forwrd, blasint *m, blasint *n, float *x, blasint *ldx, blasint *k)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slapmr((void *) forwrd, (void *) m, (void *) n, (void *) x, (void *) ldx, (void *) k);

    helpTimeStop = flexiblas_wtime();

    data->slapmr.timings[0] += (helpTimeStop - helpTime);
    data->slapmr.calls[0]++;
    
    return;
}


/* Wrapper for slapmt */
void hook_slapmt(blaslogical *forwrd, blasint *m, blasint *n, float *x, blasint *ldx, blasint *k)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slapmt((void *) forwrd, (void *) m, (void *) n, (void *) x, (void *) ldx, (void *) k);

    helpTimeStop = flexiblas_wtime();

    data->slapmt.timings[0] += (helpTimeStop - helpTime);
    data->slapmt.calls[0]++;
    
    return;
}

/* Wrapper for slapy2 */
float hook_slapy2(float *x, float *y)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_slapy2((void *) x, (void *) y);

    helpTimeStop = flexiblas_wtime();

    data->slapy2.timings[0] += (helpTimeStop - helpTime);
    data->slapy2.calls[0]++;
    
    return return_value;
}


/* Wrapper for slapy3 */
float hook_slapy3(float *x, float *y, float *z)
{
    double helpTime;
    double helpTimeStop;
    float return_value;

    helpTime = flexiblas_wtime();
    
    return_value = flexiblas_chain_slapy3((void *) x, (void *) y, (void *) z);

    helpTimeStop = flexiblas_wtime();

    data->slapy3.timings[0] += (helpTimeStop - helpTime);
    data->slapy3.calls[0]++;
    
    return return_value;
}



/* Wrapper for slaqgb */
void hook_slaqgb(blasint *m, blasint *n, blasint *kl, blasint *ku, float *ab, blasint *ldab, float *r, float *c, float *rowcnd, float *colcnd, float *amax, char *equed, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaqgb((void *) m, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) equed, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->slaqgb.timings[0] += (helpTimeStop - helpTime);
    data->slaqgb.calls[0]++;
    
    return;
}


/* Wrapper for slaqge */
void hook_slaqge(blasint *m, blasint *n, float *a, blasint *lda, float *r, float *c, float *rowcnd, float *colcnd, float *amax, char *equed, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaqge((void *) m, (void *) n, (void *) a, (void *) lda, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) equed, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->slaqge.timings[0] += (helpTimeStop - helpTime);
    data->slaqge.calls[0]++;
    
    return;
}


/* Wrapper for slaqp2 */
void hook_slaqp2(blasint *m, blasint *n, blasint *offset, float *a, blasint *lda, blasint *jpvt, float *tau, float *vn1, float *vn2, float *work)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaqp2((void *) m, (void *) n, (void *) offset, (void *) a, (void *) lda, (void *) jpvt, (void *) tau, (void *) vn1, (void *) vn2, (void *) work);

    helpTimeStop = flexiblas_wtime();

    data->slaqp2.timings[0] += (helpTimeStop - helpTime);
    data->slaqp2.calls[0]++;
    
    return;
}


/* Wrapper for slaqps */
void hook_slaqps(blasint *m, blasint *n, blasint *offset, blasint *nb, blasint *kb, float *a, blasint *lda, blasint *jpvt, float *tau, float *vn1, float *vn2, float *auxv, float *f, blasint *ldf)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaqps((void *) m, (void *) n, (void *) offset, (void *) nb, (void *) kb, (void *) a, (void *) lda, (void *) jpvt, (void *) tau, (void *) vn1, (void *) vn2, (void *) auxv, (void *) f, (void *) ldf);

    helpTimeStop = flexiblas_wtime();

    data->slaqps.timings[0] += (helpTimeStop - helpTime);
    data->slaqps.calls[0]++;
    
    return;
}


/* Wrapper for slaqr0 */
void hook_slaqr0(blaslogical *wantt, blaslogical *wantz, blasint *n, blasint *ilo, blasint *ihi, float *h, blasint *ldh, float *wr, float *wi, blasint *iloz, blasint *ihiz, float *z, blasint *ldz, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaqr0((void *) wantt, (void *) wantz, (void *) n, (void *) ilo, (void *) ihi, (void *) h, (void *) ldh, (void *) wr, (void *) wi, (void *) iloz, (void *) ihiz, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slaqr0.timings[0] += (helpTimeStop - helpTime);
    data->slaqr0.calls[0]++;
    
    return;
}


/* Wrapper for slaqr1 */
void hook_slaqr1(blasint *n, float *h, blasint *ldh, float *sr1, float *si1, float *sr2, float *si2, float *v)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaqr1((void *) n, (void *) h, (void *) ldh, (void *) sr1, (void *) si1, (void *) sr2, (void *) si2, (void *) v);

    helpTimeStop = flexiblas_wtime();

    data->slaqr1.timings[0] += (helpTimeStop - helpTime);
    data->slaqr1.calls[0]++;
    
    return;
}


/* Wrapper for slaqr2 */
void hook_slaqr2(blaslogical *wantt, blaslogical *wantz, blasint *n, blasint *ktop, blasint *kbot, blasint *nw, float *h, blasint *ldh, blasint *iloz, blasint *ihiz, float *z, blasint *ldz, blasint *ns, blasint *nd, float *sr, float *si, float *v, blasint *ldv, blasint *nh, float *t, blasint *ldt, blasint *nv, float *wv, blasint *ldwv, float *work, blasint *lwork)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaqr2((void *) wantt, (void *) wantz, (void *) n, (void *) ktop, (void *) kbot, (void *) nw, (void *) h, (void *) ldh, (void *) iloz, (void *) ihiz, (void *) z, (void *) ldz, (void *) ns, (void *) nd, (void *) sr, (void *) si, (void *) v, (void *) ldv, (void *) nh, (void *) t, (void *) ldt, (void *) nv, (void *) wv, (void *) ldwv, (void *) work, (void *) lwork);

    helpTimeStop = flexiblas_wtime();

    data->slaqr2.timings[0] += (helpTimeStop - helpTime);
    data->slaqr2.calls[0]++;
    
    return;
}


/* Wrapper for slaqr3 */
void hook_slaqr3(blaslogical *wantt, blaslogical *wantz, blasint *n, blasint *ktop, blasint *kbot, blasint *nw, float *h, blasint *ldh, blasint *iloz, blasint *ihiz, float *z, blasint *ldz, blasint *ns, blasint *nd, float *sr, float *si, float *v, blasint *ldv, blasint *nh, float *t, blasint *ldt, blasint *nv, float *wv, blasint *ldwv, float *work, blasint *lwork)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaqr3((void *) wantt, (void *) wantz, (void *) n, (void *) ktop, (void *) kbot, (void *) nw, (void *) h, (void *) ldh, (void *) iloz, (void *) ihiz, (void *) z, (void *) ldz, (void *) ns, (void *) nd, (void *) sr, (void *) si, (void *) v, (void *) ldv, (void *) nh, (void *) t, (void *) ldt, (void *) nv, (void *) wv, (void *) ldwv, (void *) work, (void *) lwork);

    helpTimeStop = flexiblas_wtime();

    data->slaqr3.timings[0] += (helpTimeStop - helpTime);
    data->slaqr3.calls[0]++;
    
    return;
}


/* Wrapper for slaqr4 */
void hook_slaqr4(blaslogical *wantt, blaslogical *wantz, blasint *n, blasint *ilo, blasint *ihi, float *h, blasint *ldh, float *wr, float *wi, blasint *iloz, blasint *ihiz, float *z, blasint *ldz, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaqr4((void *) wantt, (void *) wantz, (void *) n, (void *) ilo, (void *) ihi, (void *) h, (void *) ldh, (void *) wr, (void *) wi, (void *) iloz, (void *) ihiz, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slaqr4.timings[0] += (helpTimeStop - helpTime);
    data->slaqr4.calls[0]++;
    
    return;
}


/* Wrapper for slaqr5 */
void hook_slaqr5(blaslogical *wantt, blaslogical *wantz, blasint *kacc22, blasint *n, blasint *ktop, blasint *kbot, blasint *nshfts, float *sr, float *si, float *h, blasint *ldh, blasint *iloz, blasint *ihiz, float *z, blasint *ldz, float *v, blasint *ldv, float *u, blasint *ldu, blasint *nv, float *wv, blasint *ldwv, blasint *nh, float *wh, blasint *ldwh)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaqr5((void *) wantt, (void *) wantz, (void *) kacc22, (void *) n, (void *) ktop, (void *) kbot, (void *) nshfts, (void *) sr, (void *) si, (void *) h, (void *) ldh, (void *) iloz, (void *) ihiz, (void *) z, (void *) ldz, (void *) v, (void *) ldv, (void *) u, (void *) ldu, (void *) nv, (void *) wv, (void *) ldwv, (void *) nh, (void *) wh, (void *) ldwh);

    helpTimeStop = flexiblas_wtime();

    data->slaqr5.timings[0] += (helpTimeStop - helpTime);
    data->slaqr5.calls[0]++;
    
    return;
}


/* Wrapper for slaqsb */
void hook_slaqsb(char *uplo, blasint *n, blasint *kd, float *ab, blasint *ldab, float *s, float *scond, float *amax, char *equed, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaqsb((void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) s, (void *) scond, (void *) amax, (void *) equed, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->slaqsb.timings[0] += (helpTimeStop - helpTime);
    data->slaqsb.calls[0]++;
    
    return;
}


/* Wrapper for slaqsp */
void hook_slaqsp(char *uplo, blasint *n, float *ap, float *s, float *scond, float *amax, char *equed, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaqsp((void *) uplo, (void *) n, (void *) ap, (void *) s, (void *) scond, (void *) amax, (void *) equed, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->slaqsp.timings[0] += (helpTimeStop - helpTime);
    data->slaqsp.calls[0]++;
    
    return;
}


/* Wrapper for slaqsy */
void hook_slaqsy(char *uplo, blasint *n, float *a, blasint *lda, float *s, float *scond, float *amax, char *equed, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaqsy((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) s, (void *) scond, (void *) amax, (void *) equed, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->slaqsy.timings[0] += (helpTimeStop - helpTime);
    data->slaqsy.calls[0]++;
    
    return;
}


/* Wrapper for slaqtr */
void hook_slaqtr(blaslogical *ltran, blaslogical *lreal, blasint *n, float *t, blasint *ldt, float *b, float *w, float *scale, float *x, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaqtr((void *) ltran, (void *) lreal, (void *) n, (void *) t, (void *) ldt, (void *) b, (void *) w, (void *) scale, (void *) x, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slaqtr.timings[0] += (helpTimeStop - helpTime);
    data->slaqtr.calls[0]++;
    
    return;
}


/* Wrapper for slar1v */
void hook_slar1v(blasint *n, blasint *b1, blasint *bn, float *lambda, float *d, float *l, float *ld, float *lld, float *pivmin, float *gaptol, float *z, blaslogical *wantnc, blasint *negcnt, float *ztz, float *mingma, blasint *r, blasint *isuppz, float *nrminv, float *resid, float *rqcorr, float *work)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slar1v((void *) n, (void *) b1, (void *) bn, (void *) lambda, (void *) d, (void *) l, (void *) ld, (void *) lld, (void *) pivmin, (void *) gaptol, (void *) z, (void *) wantnc, (void *) negcnt, (void *) ztz, (void *) mingma, (void *) r, (void *) isuppz, (void *) nrminv, (void *) resid, (void *) rqcorr, (void *) work);

    helpTimeStop = flexiblas_wtime();

    data->slar1v.timings[0] += (helpTimeStop - helpTime);
    data->slar1v.calls[0]++;
    
    return;
}


/* Wrapper for slar2v */
void hook_slar2v(blasint *n, float *x, float *y, float *z, blasint *incx, float *c, float *s, blasint *incc)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slar2v((void *) n, (void *) x, (void *) y, (void *) z, (void *) incx, (void *) c, (void *) s, (void *) incc);

    helpTimeStop = flexiblas_wtime();

    data->slar2v.timings[0] += (helpTimeStop - helpTime);
    data->slar2v.calls[0]++;
    
    return;
}


/* Wrapper for slarf */
void hook_slarf(char *side, blasint *m, blasint *n, float *v, blasint *incv, float *tau, float *c, blasint *ldc, float *work, flexiblas_fortran_charlen_t side_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarf((void *) side, (void *) m, (void *) n, (void *) v, (void *) incv, (void *) tau, (void *) c, (void *) ldc, (void *) work, side_len);

    helpTimeStop = flexiblas_wtime();

    data->slarf.timings[0] += (helpTimeStop - helpTime);
    data->slarf.calls[0]++;
    
    return;
}


/* Wrapper for slarfb */
void hook_slarfb(char *side, char *trans, char *direct, char *storev, blasint *m, blasint *n, blasint *k, float *v, blasint *ldv, float *t, blasint *ldt, float *c, blasint *ldc, float *work, blasint *ldwork, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t direct_len, flexiblas_fortran_charlen_t storev_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarfb((void *) side, (void *) trans, (void *) direct, (void *) storev, (void *) m, (void *) n, (void *) k, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) c, (void *) ldc, (void *) work, (void *) ldwork, side_len, trans_len, direct_len, storev_len);

    helpTimeStop = flexiblas_wtime();

    data->slarfb.timings[0] += (helpTimeStop - helpTime);
    data->slarfb.calls[0]++;
    
    return;
}


/* Wrapper for slarfb_gett */
void hook_slarfb_gett(char *ident, blasint *m, blasint *n, blasint *k, float *t, blasint *ldt, float *a, blasint *lda, float *b, blasint *ldb, float *work, blasint *ldwork, flexiblas_fortran_charlen_t ident_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarfb_gett((void *) ident, (void *) m, (void *) n, (void *) k, (void *) t, (void *) ldt, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) work, (void *) ldwork, ident_len);

    helpTimeStop = flexiblas_wtime();

    data->slarfb_gett.timings[0] += (helpTimeStop - helpTime);
    data->slarfb_gett.calls[0]++;
    
    return;
}


/* Wrapper for slarfg */
void hook_slarfg(blasint *n, float *alpha, float *x, blasint *incx, float *tau)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarfg((void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) tau);

    helpTimeStop = flexiblas_wtime();

    data->slarfg.timings[0] += (helpTimeStop - helpTime);
    data->slarfg.calls[0]++;
    
    return;
}


/* Wrapper for slarfgp */
void hook_slarfgp(blasint *n, float *alpha, float *x, blasint *incx, float *tau)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarfgp((void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) tau);

    helpTimeStop = flexiblas_wtime();

    data->slarfgp.timings[0] += (helpTimeStop - helpTime);
    data->slarfgp.calls[0]++;
    
    return;
}


/* Wrapper for slarft */
void hook_slarft(char *direct, char *storev, blasint *n, blasint *k, float *v, blasint *ldv, float *tau, float *t, blasint *ldt, flexiblas_fortran_charlen_t direct_len, flexiblas_fortran_charlen_t storev_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarft((void *) direct, (void *) storev, (void *) n, (void *) k, (void *) v, (void *) ldv, (void *) tau, (void *) t, (void *) ldt, direct_len, storev_len);

    helpTimeStop = flexiblas_wtime();

    data->slarft.timings[0] += (helpTimeStop - helpTime);
    data->slarft.calls[0]++;
    
    return;
}


/* Wrapper for slarfx */
void hook_slarfx(char *side, blasint *m, blasint *n, float *v, float *tau, float *c, blasint *ldc, float *work, flexiblas_fortran_charlen_t side_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarfx((void *) side, (void *) m, (void *) n, (void *) v, (void *) tau, (void *) c, (void *) ldc, (void *) work, side_len);

    helpTimeStop = flexiblas_wtime();

    data->slarfx.timings[0] += (helpTimeStop - helpTime);
    data->slarfx.calls[0]++;
    
    return;
}


/* Wrapper for slarfy */
void hook_slarfy(char *uplo, blasint *n, float *v, blasint *incv, float *tau, float *c, blasint *ldc, float *work, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarfy((void *) uplo, (void *) n, (void *) v, (void *) incv, (void *) tau, (void *) c, (void *) ldc, (void *) work, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->slarfy.timings[0] += (helpTimeStop - helpTime);
    data->slarfy.calls[0]++;
    
    return;
}


/* Wrapper for slargv */
void hook_slargv(blasint *n, float *x, blasint *incx, float *y, blasint *incy, float *c, blasint *incc)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slargv((void *) n, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) c, (void *) incc);

    helpTimeStop = flexiblas_wtime();

    data->slargv.timings[0] += (helpTimeStop - helpTime);
    data->slargv.calls[0]++;
    
    return;
}


/* Wrapper for slarnv */
void hook_slarnv(blasint *idist, blasint *iseed, blasint *n, float *x)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarnv((void *) idist, (void *) iseed, (void *) n, (void *) x);

    helpTimeStop = flexiblas_wtime();

    data->slarnv.timings[0] += (helpTimeStop - helpTime);
    data->slarnv.calls[0]++;
    
    return;
}


/* Wrapper for slarra */
void hook_slarra(blasint *n, float *d, float *e, float *e2, float *spltol, float *tnrm, blasint *nsplit, blasint *isplit, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarra((void *) n, (void *) d, (void *) e, (void *) e2, (void *) spltol, (void *) tnrm, (void *) nsplit, (void *) isplit, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slarra.timings[0] += (helpTimeStop - helpTime);
    data->slarra.calls[0]++;
    
    return;
}


/* Wrapper for slarrb */
void hook_slarrb(blasint *n, float *d, float *lld, blasint *ifirst, blasint *ilast, float *rtol1, float *rtol2, blasint *offset, float *w, float *wgap, float *werr, float *work, blasint *iwork, float *pivmin, float *spdiam, blasint *twist, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarrb((void *) n, (void *) d, (void *) lld, (void *) ifirst, (void *) ilast, (void *) rtol1, (void *) rtol2, (void *) offset, (void *) w, (void *) wgap, (void *) werr, (void *) work, (void *) iwork, (void *) pivmin, (void *) spdiam, (void *) twist, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slarrb.timings[0] += (helpTimeStop - helpTime);
    data->slarrb.calls[0]++;
    
    return;
}


/* Wrapper for slarrc */
void hook_slarrc(char *jobt, blasint *n, float *vl, float *vu, float *d, float *e, float *pivmin, blasint *eigcnt, blasint *lcnt, blasint *rcnt, blasint *info, flexiblas_fortran_charlen_t jobt_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarrc((void *) jobt, (void *) n, (void *) vl, (void *) vu, (void *) d, (void *) e, (void *) pivmin, (void *) eigcnt, (void *) lcnt, (void *) rcnt, (void *) info, jobt_len);

    helpTimeStop = flexiblas_wtime();

    data->slarrc.timings[0] += (helpTimeStop - helpTime);
    data->slarrc.calls[0]++;
    
    return;
}


/* Wrapper for slarrd */
void hook_slarrd(char *range, char *order, blasint *n, float *vl, float *vu, blasint *il, blasint *iu, float *gers, float *reltol, float *d, float *e, float *e2, float *pivmin, blasint *nsplit, blasint *isplit, blasint *m, float *w, float *werr, float *wl, float *wu, blasint *iblock, blasint *indexw, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t order_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarrd((void *) range, (void *) order, (void *) n, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) gers, (void *) reltol, (void *) d, (void *) e, (void *) e2, (void *) pivmin, (void *) nsplit, (void *) isplit, (void *) m, (void *) w, (void *) werr, (void *) wl, (void *) wu, (void *) iblock, (void *) indexw, (void *) work, (void *) iwork, (void *) info, range_len, order_len);

    helpTimeStop = flexiblas_wtime();

    data->slarrd.timings[0] += (helpTimeStop - helpTime);
    data->slarrd.calls[0]++;
    
    return;
}


/* Wrapper for slarre */
void hook_slarre(char *range, blasint *n, float *vl, float *vu, blasint *il, blasint *iu, float *d, float *e, float *e2, float *rtol1, float *rtol2, float *spltol, blasint *nsplit, blasint *isplit, blasint *m, float *w, float *werr, float *wgap, blasint *iblock, blasint *indexw, float *gers, float *pivmin, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t range_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarre((void *) range, (void *) n, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) d, (void *) e, (void *) e2, (void *) rtol1, (void *) rtol2, (void *) spltol, (void *) nsplit, (void *) isplit, (void *) m, (void *) w, (void *) werr, (void *) wgap, (void *) iblock, (void *) indexw, (void *) gers, (void *) pivmin, (void *) work, (void *) iwork, (void *) info, range_len);

    helpTimeStop = flexiblas_wtime();

    data->slarre.timings[0] += (helpTimeStop - helpTime);
    data->slarre.calls[0]++;
    
    return;
}


/* Wrapper for slarrf */
void hook_slarrf(blasint *n, float *d, float *l, float *ld, blasint *clstrt, blasint *clend, float *w, float *wgap, float *werr, float *spdiam, float *clgapl, float *clgapr, float *pivmin, float *sigma, float *dplus, float *lplus, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarrf((void *) n, (void *) d, (void *) l, (void *) ld, (void *) clstrt, (void *) clend, (void *) w, (void *) wgap, (void *) werr, (void *) spdiam, (void *) clgapl, (void *) clgapr, (void *) pivmin, (void *) sigma, (void *) dplus, (void *) lplus, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slarrf.timings[0] += (helpTimeStop - helpTime);
    data->slarrf.calls[0]++;
    
    return;
}


/* Wrapper for slarrj */
void hook_slarrj(blasint *n, float *d, float *e2, blasint *ifirst, blasint *ilast, float *rtol, blasint *offset, float *w, float *werr, float *work, blasint *iwork, float *pivmin, float *spdiam, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarrj((void *) n, (void *) d, (void *) e2, (void *) ifirst, (void *) ilast, (void *) rtol, (void *) offset, (void *) w, (void *) werr, (void *) work, (void *) iwork, (void *) pivmin, (void *) spdiam, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slarrj.timings[0] += (helpTimeStop - helpTime);
    data->slarrj.calls[0]++;
    
    return;
}


/* Wrapper for slarrk */
void hook_slarrk(blasint *n, blasint *iw, float *gl, float *gu, float *d, float *e2, float *pivmin, float *reltol, float *w, float *werr, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarrk((void *) n, (void *) iw, (void *) gl, (void *) gu, (void *) d, (void *) e2, (void *) pivmin, (void *) reltol, (void *) w, (void *) werr, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slarrk.timings[0] += (helpTimeStop - helpTime);
    data->slarrk.calls[0]++;
    
    return;
}


/* Wrapper for slarrr */
void hook_slarrr(blasint *n, float *d, float *e, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarrr((void *) n, (void *) d, (void *) e, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slarrr.timings[0] += (helpTimeStop - helpTime);
    data->slarrr.calls[0]++;
    
    return;
}


/* Wrapper for slarrv */
void hook_slarrv(blasint *n, float *vl, float *vu, float *d, float *l, float *pivmin, blasint *isplit, blasint *m, blasint *dol, blasint *dou, float *minrgp, float *rtol1, float *rtol2, float *w, float *werr, float *wgap, blasint *iblock, blasint *indexw, float *gers, float *z, blasint *ldz, blasint *isuppz, float *work, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarrv((void *) n, (void *) vl, (void *) vu, (void *) d, (void *) l, (void *) pivmin, (void *) isplit, (void *) m, (void *) dol, (void *) dou, (void *) minrgp, (void *) rtol1, (void *) rtol2, (void *) w, (void *) werr, (void *) wgap, (void *) iblock, (void *) indexw, (void *) gers, (void *) z, (void *) ldz, (void *) isuppz, (void *) work, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slarrv.timings[0] += (helpTimeStop - helpTime);
    data->slarrv.calls[0]++;
    
    return;
}


/* Wrapper for slartg */
void hook_slartg(float *f, float *g, float *cs, float *sn, float *r)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slartg((void *) f, (void *) g, (void *) cs, (void *) sn, (void *) r);

    helpTimeStop = flexiblas_wtime();

    data->slartg.timings[0] += (helpTimeStop - helpTime);
    data->slartg.calls[0]++;
    
    return;
}


/* Wrapper for slartgp */
void hook_slartgp(float *f, float *g, float *cs, float *sn, float *r)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slartgp((void *) f, (void *) g, (void *) cs, (void *) sn, (void *) r);

    helpTimeStop = flexiblas_wtime();

    data->slartgp.timings[0] += (helpTimeStop - helpTime);
    data->slartgp.calls[0]++;
    
    return;
}


/* Wrapper for slartgs */
void hook_slartgs(float *x, float *y, float *sigma, float *cs, float *sn)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slartgs((void *) x, (void *) y, (void *) sigma, (void *) cs, (void *) sn);

    helpTimeStop = flexiblas_wtime();

    data->slartgs.timings[0] += (helpTimeStop - helpTime);
    data->slartgs.calls[0]++;
    
    return;
}


/* Wrapper for slartv */
void hook_slartv(blasint *n, float *x, blasint *incx, float *y, blasint *incy, float *c, float *s, blasint *incc)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slartv((void *) n, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) c, (void *) s, (void *) incc);

    helpTimeStop = flexiblas_wtime();

    data->slartv.timings[0] += (helpTimeStop - helpTime);
    data->slartv.calls[0]++;
    
    return;
}


/* Wrapper for slaruv */
void hook_slaruv(blasint *iseed, blasint *n, float *x)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaruv((void *) iseed, (void *) n, (void *) x);

    helpTimeStop = flexiblas_wtime();

    data->slaruv.timings[0] += (helpTimeStop - helpTime);
    data->slaruv.calls[0]++;
    
    return;
}


/* Wrapper for slarz */
void hook_slarz(char *side, blasint *m, blasint *n, blasint *l, float *v, blasint *incv, float *tau, float *c, blasint *ldc, float *work, flexiblas_fortran_charlen_t side_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarz((void *) side, (void *) m, (void *) n, (void *) l, (void *) v, (void *) incv, (void *) tau, (void *) c, (void *) ldc, (void *) work, side_len);

    helpTimeStop = flexiblas_wtime();

    data->slarz.timings[0] += (helpTimeStop - helpTime);
    data->slarz.calls[0]++;
    
    return;
}


/* Wrapper for slarzb */
void hook_slarzb(char *side, char *trans, char *direct, char *storev, blasint *m, blasint *n, blasint *k, blasint *l, float *v, blasint *ldv, float *t, blasint *ldt, float *c, blasint *ldc, float *work, blasint *ldwork, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t direct_len, flexiblas_fortran_charlen_t storev_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarzb((void *) side, (void *) trans, (void *) direct, (void *) storev, (void *) m, (void *) n, (void *) k, (void *) l, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) c, (void *) ldc, (void *) work, (void *) ldwork, side_len, trans_len, direct_len, storev_len);

    helpTimeStop = flexiblas_wtime();

    data->slarzb.timings[0] += (helpTimeStop - helpTime);
    data->slarzb.calls[0]++;
    
    return;
}


/* Wrapper for slarzt */
void hook_slarzt(char *direct, char *storev, blasint *n, blasint *k, float *v, blasint *ldv, float *tau, float *t, blasint *ldt, flexiblas_fortran_charlen_t direct_len, flexiblas_fortran_charlen_t storev_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slarzt((void *) direct, (void *) storev, (void *) n, (void *) k, (void *) v, (void *) ldv, (void *) tau, (void *) t, (void *) ldt, direct_len, storev_len);

    helpTimeStop = flexiblas_wtime();

    data->slarzt.timings[0] += (helpTimeStop - helpTime);
    data->slarzt.calls[0]++;
    
    return;
}


/* Wrapper for slas2 */
void hook_slas2(float *f, float *g, float *h, float *ssmin, float *ssmax)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slas2((void *) f, (void *) g, (void *) h, (void *) ssmin, (void *) ssmax);

    helpTimeStop = flexiblas_wtime();

    data->slas2.timings[0] += (helpTimeStop - helpTime);
    data->slas2.calls[0]++;
    
    return;
}


/* Wrapper for slascl */
void hook_slascl(char *type, blasint *kl, blasint *ku, float *cfrom, float *cto, blasint *m, blasint *n, float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t type_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slascl((void *) type, (void *) kl, (void *) ku, (void *) cfrom, (void *) cto, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) info, type_len);

    helpTimeStop = flexiblas_wtime();

    data->slascl.timings[0] += (helpTimeStop - helpTime);
    data->slascl.calls[0]++;
    
    return;
}


/* Wrapper for slasd0 */
void hook_slasd0(blasint *n, blasint *sqre, float *d, float *e, float *u, blasint *ldu, float *vt, blasint *ldvt, blasint *smlsiz, blasint *iwork, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasd0((void *) n, (void *) sqre, (void *) d, (void *) e, (void *) u, (void *) ldu, (void *) vt, (void *) ldvt, (void *) smlsiz, (void *) iwork, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slasd0.timings[0] += (helpTimeStop - helpTime);
    data->slasd0.calls[0]++;
    
    return;
}


/* Wrapper for slasd1 */
void hook_slasd1(blasint *nl, blasint *nr, blasint *sqre, float *d, float *alpha, float *beta, float *u, blasint *ldu, float *vt, blasint *ldvt, blasint *idxq, blasint *iwork, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasd1((void *) nl, (void *) nr, (void *) sqre, (void *) d, (void *) alpha, (void *) beta, (void *) u, (void *) ldu, (void *) vt, (void *) ldvt, (void *) idxq, (void *) iwork, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slasd1.timings[0] += (helpTimeStop - helpTime);
    data->slasd1.calls[0]++;
    
    return;
}


/* Wrapper for slasd2 */
void hook_slasd2(blasint *nl, blasint *nr, blasint *sqre, blasint *k, float *d, float *z, float *alpha, float *beta, float *u, blasint *ldu, float *vt, blasint *ldvt, float *dsigma, float *u2, blasint *ldu2, float *vt2, blasint *ldvt2, blasint *idxp, blasint *idx, blasint *idxc, blasint *idxq, blasint *coltyp, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasd2((void *) nl, (void *) nr, (void *) sqre, (void *) k, (void *) d, (void *) z, (void *) alpha, (void *) beta, (void *) u, (void *) ldu, (void *) vt, (void *) ldvt, (void *) dsigma, (void *) u2, (void *) ldu2, (void *) vt2, (void *) ldvt2, (void *) idxp, (void *) idx, (void *) idxc, (void *) idxq, (void *) coltyp, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slasd2.timings[0] += (helpTimeStop - helpTime);
    data->slasd2.calls[0]++;
    
    return;
}


/* Wrapper for slasd3 */
void hook_slasd3(blasint *nl, blasint *nr, blasint *sqre, blasint *k, float *d, float *q, blasint *ldq, float *dsigma, float *u, blasint *ldu, float *u2, blasint *ldu2, float *vt, blasint *ldvt, float *vt2, blasint *ldvt2, blasint *idxc, blasint *ctot, float *z, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasd3((void *) nl, (void *) nr, (void *) sqre, (void *) k, (void *) d, (void *) q, (void *) ldq, (void *) dsigma, (void *) u, (void *) ldu, (void *) u2, (void *) ldu2, (void *) vt, (void *) ldvt, (void *) vt2, (void *) ldvt2, (void *) idxc, (void *) ctot, (void *) z, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slasd3.timings[0] += (helpTimeStop - helpTime);
    data->slasd3.calls[0]++;
    
    return;
}


/* Wrapper for slasd4 */
void hook_slasd4(blasint *n, blasint *i, float *d, float *z, float *delta, float *rho, float *sigma, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasd4((void *) n, (void *) i, (void *) d, (void *) z, (void *) delta, (void *) rho, (void *) sigma, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slasd4.timings[0] += (helpTimeStop - helpTime);
    data->slasd4.calls[0]++;
    
    return;
}


/* Wrapper for slasd5 */
void hook_slasd5(blasint *i, float *d, float *z, float *delta, float *rho, float *dsigma, float *work)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasd5((void *) i, (void *) d, (void *) z, (void *) delta, (void *) rho, (void *) dsigma, (void *) work);

    helpTimeStop = flexiblas_wtime();

    data->slasd5.timings[0] += (helpTimeStop - helpTime);
    data->slasd5.calls[0]++;
    
    return;
}


/* Wrapper for slasd6 */
void hook_slasd6(blasint *icompq, blasint *nl, blasint *nr, blasint *sqre, float *d, float *vf, float *vl, float *alpha, float *beta, blasint *idxq, blasint *perm, blasint *givptr, blasint *givcol, blasint *ldgcol, float *givnum, blasint *ldgnum, float *poles, float *difl, float *difr, float *z, blasint *k, float *c, float *s, float *work, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasd6((void *) icompq, (void *) nl, (void *) nr, (void *) sqre, (void *) d, (void *) vf, (void *) vl, (void *) alpha, (void *) beta, (void *) idxq, (void *) perm, (void *) givptr, (void *) givcol, (void *) ldgcol, (void *) givnum, (void *) ldgnum, (void *) poles, (void *) difl, (void *) difr, (void *) z, (void *) k, (void *) c, (void *) s, (void *) work, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slasd6.timings[0] += (helpTimeStop - helpTime);
    data->slasd6.calls[0]++;
    
    return;
}


/* Wrapper for slasd7 */
void hook_slasd7(blasint *icompq, blasint *nl, blasint *nr, blasint *sqre, blasint *k, float *d, float *z, float *zw, float *vf, float *vfw, float *vl, float *vlw, float *alpha, float *beta, float *dsigma, blasint *idx, blasint *idxp, blasint *idxq, blasint *perm, blasint *givptr, blasint *givcol, blasint *ldgcol, float *givnum, blasint *ldgnum, float *c, float *s, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasd7((void *) icompq, (void *) nl, (void *) nr, (void *) sqre, (void *) k, (void *) d, (void *) z, (void *) zw, (void *) vf, (void *) vfw, (void *) vl, (void *) vlw, (void *) alpha, (void *) beta, (void *) dsigma, (void *) idx, (void *) idxp, (void *) idxq, (void *) perm, (void *) givptr, (void *) givcol, (void *) ldgcol, (void *) givnum, (void *) ldgnum, (void *) c, (void *) s, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slasd7.timings[0] += (helpTimeStop - helpTime);
    data->slasd7.calls[0]++;
    
    return;
}


/* Wrapper for slasd8 */
void hook_slasd8(blasint *icompq, blasint *k, float *d, float *z, float *vf, float *vl, float *difl, float *difr, blasint *lddifr, float *dsigma, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasd8((void *) icompq, (void *) k, (void *) d, (void *) z, (void *) vf, (void *) vl, (void *) difl, (void *) difr, (void *) lddifr, (void *) dsigma, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slasd8.timings[0] += (helpTimeStop - helpTime);
    data->slasd8.calls[0]++;
    
    return;
}


/* Wrapper for slasda */
void hook_slasda(blasint *icompq, blasint *smlsiz, blasint *n, blasint *sqre, float *d, float *e, float *u, blasint *ldu, float *vt, blasint *k, float *difl, float *difr, float *z, float *poles, blasint *givptr, blasint *givcol, blasint *ldgcol, blasint *perm, float *givnum, float *c, float *s, float *work, blasint *iwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasda((void *) icompq, (void *) smlsiz, (void *) n, (void *) sqre, (void *) d, (void *) e, (void *) u, (void *) ldu, (void *) vt, (void *) k, (void *) difl, (void *) difr, (void *) z, (void *) poles, (void *) givptr, (void *) givcol, (void *) ldgcol, (void *) perm, (void *) givnum, (void *) c, (void *) s, (void *) work, (void *) iwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slasda.timings[0] += (helpTimeStop - helpTime);
    data->slasda.calls[0]++;
    
    return;
}


/* Wrapper for slasdq */
void hook_slasdq(char *uplo, blasint *sqre, blasint *n, blasint *ncvt, blasint *nru, blasint *ncc, float *d, float *e, float *vt, blasint *ldvt, float *u, blasint *ldu, float *c, blasint *ldc, float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasdq((void *) uplo, (void *) sqre, (void *) n, (void *) ncvt, (void *) nru, (void *) ncc, (void *) d, (void *) e, (void *) vt, (void *) ldvt, (void *) u, (void *) ldu, (void *) c, (void *) ldc, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->slasdq.timings[0] += (helpTimeStop - helpTime);
    data->slasdq.calls[0]++;
    
    return;
}


/* Wrapper for slasdt */
void hook_slasdt(blasint *n, blasint *lvl, blasint *nd, blasint *inode, blasint *ndiml, blasint *ndimr, blasint *msub)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasdt((void *) n, (void *) lvl, (void *) nd, (void *) inode, (void *) ndiml, (void *) ndimr, (void *) msub);

    helpTimeStop = flexiblas_wtime();

    data->slasdt.timings[0] += (helpTimeStop - helpTime);
    data->slasdt.calls[0]++;
    
    return;
}


/* Wrapper for slaset */
void hook_slaset(char *uplo, blasint *m, blasint *n, float *alpha, float *beta, float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaset((void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) beta, (void *) a, (void *) lda, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->slaset.timings[0] += (helpTimeStop - helpTime);
    data->slaset.calls[0]++;
    
    return;
}


/* Wrapper for slasq1 */
void hook_slasq1(blasint *n, float *d, float *e, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasq1((void *) n, (void *) d, (void *) e, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slasq1.timings[0] += (helpTimeStop - helpTime);
    data->slasq1.calls[0]++;
    
    return;
}


/* Wrapper for slasq2 */
void hook_slasq2(blasint *n, float *z, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasq2((void *) n, (void *) z, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slasq2.timings[0] += (helpTimeStop - helpTime);
    data->slasq2.calls[0]++;
    
    return;
}


/* Wrapper for slasq3 */
void hook_slasq3(blasint *i0, blasint *n0, float *z, blasint *pp, float *dmin, float *sigma, float *desig, float *qmax, blasint *nfail, blasint *iter, blasint *ndiv, blaslogical *ieee, blasint *ttype, float *dmin1, float *dmin2, float *dn, float *dn1, float *dn2, float *g, float *tau)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasq3((void *) i0, (void *) n0, (void *) z, (void *) pp, (void *) dmin, (void *) sigma, (void *) desig, (void *) qmax, (void *) nfail, (void *) iter, (void *) ndiv, (void *) ieee, (void *) ttype, (void *) dmin1, (void *) dmin2, (void *) dn, (void *) dn1, (void *) dn2, (void *) g, (void *) tau);

    helpTimeStop = flexiblas_wtime();

    data->slasq3.timings[0] += (helpTimeStop - helpTime);
    data->slasq3.calls[0]++;
    
    return;
}


/* Wrapper for slasq4 */
void hook_slasq4(blasint *i0, blasint *n0, float *z, blasint *pp, blasint *n0in, float *dmin, float *dmin1, float *dmin2, float *dn, float *dn1, float *dn2, float *tau, blasint *ttype, float *g)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasq4((void *) i0, (void *) n0, (void *) z, (void *) pp, (void *) n0in, (void *) dmin, (void *) dmin1, (void *) dmin2, (void *) dn, (void *) dn1, (void *) dn2, (void *) tau, (void *) ttype, (void *) g);

    helpTimeStop = flexiblas_wtime();

    data->slasq4.timings[0] += (helpTimeStop - helpTime);
    data->slasq4.calls[0]++;
    
    return;
}


/* Wrapper for slasq5 */
void hook_slasq5(blasint *i0, blasint *n0, float *z, blasint *pp, float *tau, float *sigma, float *dmin, float *dmin1, float *dmin2, float *dn, float *dnm1, float *dnm2, blaslogical *ieee, float *eps)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasq5((void *) i0, (void *) n0, (void *) z, (void *) pp, (void *) tau, (void *) sigma, (void *) dmin, (void *) dmin1, (void *) dmin2, (void *) dn, (void *) dnm1, (void *) dnm2, (void *) ieee, (void *) eps);

    helpTimeStop = flexiblas_wtime();

    data->slasq5.timings[0] += (helpTimeStop - helpTime);
    data->slasq5.calls[0]++;
    
    return;
}


/* Wrapper for slasq6 */
void hook_slasq6(blasint *i0, blasint *n0, float *z, blasint *pp, float *dmin, float *dmin1, float *dmin2, float *dn, float *dnm1, float *dnm2)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasq6((void *) i0, (void *) n0, (void *) z, (void *) pp, (void *) dmin, (void *) dmin1, (void *) dmin2, (void *) dn, (void *) dnm1, (void *) dnm2);

    helpTimeStop = flexiblas_wtime();

    data->slasq6.timings[0] += (helpTimeStop - helpTime);
    data->slasq6.calls[0]++;
    
    return;
}


/* Wrapper for slasr */
void hook_slasr(char *side, char *pivot, char *direct, blasint *m, blasint *n, float *c, float *s, float *a, blasint *lda, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t pivot_len, flexiblas_fortran_charlen_t direct_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasr((void *) side, (void *) pivot, (void *) direct, (void *) m, (void *) n, (void *) c, (void *) s, (void *) a, (void *) lda, side_len, pivot_len, direct_len);

    helpTimeStop = flexiblas_wtime();

    data->slasr.timings[0] += (helpTimeStop - helpTime);
    data->slasr.calls[0]++;
    
    return;
}


/* Wrapper for slasrt */
void hook_slasrt(char *id, blasint *n, float *d, blasint *info, flexiblas_fortran_charlen_t id_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasrt((void *) id, (void *) n, (void *) d, (void *) info, id_len);

    helpTimeStop = flexiblas_wtime();

    data->slasrt.timings[0] += (helpTimeStop - helpTime);
    data->slasrt.calls[0]++;
    
    return;
}


/* Wrapper for slassq */
void hook_slassq(blasint *n, float *x, blasint *incx, float *scale, float *sumsq)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slassq((void *) n, (void *) x, (void *) incx, (void *) scale, (void *) sumsq);

    helpTimeStop = flexiblas_wtime();

    data->slassq.timings[0] += (helpTimeStop - helpTime);
    data->slassq.calls[0]++;
    
    return;
}


/* Wrapper for slasv2 */
void hook_slasv2(float *f, float *g, float *h, float *ssmin, float *ssmax, float *snr, float *csr, float *snl, float *csl)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasv2((void *) f, (void *) g, (void *) h, (void *) ssmin, (void *) ssmax, (void *) snr, (void *) csr, (void *) snl, (void *) csl);

    helpTimeStop = flexiblas_wtime();

    data->slasv2.timings[0] += (helpTimeStop - helpTime);
    data->slasv2.calls[0]++;
    
    return;
}


/* Wrapper for slaswlq */
void hook_slaswlq(blasint *m, blasint *n, blasint *mb, blasint *nb, float *a, blasint *lda, float *t, blasint *ldt, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaswlq((void *) m, (void *) n, (void *) mb, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slaswlq.timings[0] += (helpTimeStop - helpTime);
    data->slaswlq.calls[0]++;
    
    return;
}


/* Wrapper for slaswp */
void hook_slaswp(blasint *n, float *a, blasint *lda, blasint *k1, blasint *k2, blasint *ipiv, blasint *incx)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slaswp((void *) n, (void *) a, (void *) lda, (void *) k1, (void *) k2, (void *) ipiv, (void *) incx);

    helpTimeStop = flexiblas_wtime();

    data->slaswp.timings[0] += (helpTimeStop - helpTime);
    data->slaswp.calls[0]++;
    
    return;
}


/* Wrapper for slasy2 */
void hook_slasy2(blaslogical *ltranl, blaslogical *ltranr, blasint *isgn, blasint *n1, blasint *n2, float *tl, blasint *ldtl, float *tr, blasint *ldtr, float *b, blasint *ldb, float *scale, float *x, blasint *ldx, float *xnorm, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasy2((void *) ltranl, (void *) ltranr, (void *) isgn, (void *) n1, (void *) n2, (void *) tl, (void *) ldtl, (void *) tr, (void *) ldtr, (void *) b, (void *) ldb, (void *) scale, (void *) x, (void *) ldx, (void *) xnorm, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slasy2.timings[0] += (helpTimeStop - helpTime);
    data->slasy2.calls[0]++;
    
    return;
}


/* Wrapper for slasyf */
void hook_slasyf(char *uplo, blasint *n, blasint *nb, blasint *kb, float *a, blasint *lda, blasint *ipiv, float *w, blasint *ldw, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasyf((void *) uplo, (void *) n, (void *) nb, (void *) kb, (void *) a, (void *) lda, (void *) ipiv, (void *) w, (void *) ldw, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->slasyf.timings[0] += (helpTimeStop - helpTime);
    data->slasyf.calls[0]++;
    
    return;
}


/* Wrapper for slasyf_aa */
void hook_slasyf_aa(char *uplo, blasint *j1, blasint *m, blasint *nb, float *a, blasint *lda, blasint *ipiv, float *h, blasint *ldh, float *work, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasyf_aa((void *) uplo, (void *) j1, (void *) m, (void *) nb, (void *) a, (void *) lda, (void *) ipiv, (void *) h, (void *) ldh, (void *) work, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->slasyf_aa.timings[0] += (helpTimeStop - helpTime);
    data->slasyf_aa.calls[0]++;
    
    return;
}


/* Wrapper for slasyf_rk */
void hook_slasyf_rk(char *uplo, blasint *n, blasint *nb, blasint *kb, float *a, blasint *lda, float *e, blasint *ipiv, float *w, blasint *ldw, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasyf_rk((void *) uplo, (void *) n, (void *) nb, (void *) kb, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) w, (void *) ldw, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->slasyf_rk.timings[0] += (helpTimeStop - helpTime);
    data->slasyf_rk.calls[0]++;
    
    return;
}


/* Wrapper for slasyf_rook */
void hook_slasyf_rook(char *uplo, blasint *n, blasint *nb, blasint *kb, float *a, blasint *lda, blasint *ipiv, float *w, blasint *ldw, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slasyf_rook((void *) uplo, (void *) n, (void *) nb, (void *) kb, (void *) a, (void *) lda, (void *) ipiv, (void *) w, (void *) ldw, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->slasyf_rook.timings[0] += (helpTimeStop - helpTime);
    data->slasyf_rook.calls[0]++;
    
    return;
}


/* Wrapper for slatbs */
void hook_slatbs(char *uplo, char *trans, char *diag, char *normin, blasint *n, blasint *kd, float *ab, blasint *ldab, float *x, float *scale, float *cnorm, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len, flexiblas_fortran_charlen_t normin_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slatbs((void *) uplo, (void *) trans, (void *) diag, (void *) normin, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) x, (void *) scale, (void *) cnorm, (void *) info, uplo_len, trans_len, diag_len, normin_len);

    helpTimeStop = flexiblas_wtime();

    data->slatbs.timings[0] += (helpTimeStop - helpTime);
    data->slatbs.calls[0]++;
    
    return;
}


/* Wrapper for slatdf */
void hook_slatdf(blasint *ijob, blasint *n, float *z, blasint *ldz, float *rhs, float *rdsum, float *rdscal, blasint *ipiv, blasint *jpiv)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slatdf((void *) ijob, (void *) n, (void *) z, (void *) ldz, (void *) rhs, (void *) rdsum, (void *) rdscal, (void *) ipiv, (void *) jpiv);

    helpTimeStop = flexiblas_wtime();

    data->slatdf.timings[0] += (helpTimeStop - helpTime);
    data->slatdf.calls[0]++;
    
    return;
}


/* Wrapper for slatps */
void hook_slatps(char *uplo, char *trans, char *diag, char *normin, blasint *n, float *ap, float *x, float *scale, float *cnorm, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len, flexiblas_fortran_charlen_t normin_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slatps((void *) uplo, (void *) trans, (void *) diag, (void *) normin, (void *) n, (void *) ap, (void *) x, (void *) scale, (void *) cnorm, (void *) info, uplo_len, trans_len, diag_len, normin_len);

    helpTimeStop = flexiblas_wtime();

    data->slatps.timings[0] += (helpTimeStop - helpTime);
    data->slatps.calls[0]++;
    
    return;
}


/* Wrapper for slatrd */
void hook_slatrd(char *uplo, blasint *n, blasint *nb, float *a, blasint *lda, float *e, float *tau, float *w, blasint *ldw, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slatrd((void *) uplo, (void *) n, (void *) nb, (void *) a, (void *) lda, (void *) e, (void *) tau, (void *) w, (void *) ldw, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->slatrd.timings[0] += (helpTimeStop - helpTime);
    data->slatrd.calls[0]++;
    
    return;
}


/* Wrapper for slatrs */
void hook_slatrs(char *uplo, char *trans, char *diag, char *normin, blasint *n, float *a, blasint *lda, float *x, float *scale, float *cnorm, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len, flexiblas_fortran_charlen_t normin_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slatrs((void *) uplo, (void *) trans, (void *) diag, (void *) normin, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) scale, (void *) cnorm, (void *) info, uplo_len, trans_len, diag_len, normin_len);

    helpTimeStop = flexiblas_wtime();

    data->slatrs.timings[0] += (helpTimeStop - helpTime);
    data->slatrs.calls[0]++;
    
    return;
}


/* Wrapper for slatrz */
void hook_slatrz(blasint *m, blasint *n, blasint *l, float *a, blasint *lda, float *tau, float *work)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slatrz((void *) m, (void *) n, (void *) l, (void *) a, (void *) lda, (void *) tau, (void *) work);

    helpTimeStop = flexiblas_wtime();

    data->slatrz.timings[0] += (helpTimeStop - helpTime);
    data->slatrz.calls[0]++;
    
    return;
}


/* Wrapper for slatsqr */
void hook_slatsqr(blasint *m, blasint *n, blasint *mb, blasint *nb, float *a, blasint *lda, float *t, blasint *ldt, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slatsqr((void *) m, (void *) n, (void *) mb, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->slatsqr.timings[0] += (helpTimeStop - helpTime);
    data->slatsqr.calls[0]++;
    
    return;
}


/* Wrapper for slauu2 */
void hook_slauu2(char *uplo, blasint *n, float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slauu2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->slauu2.timings[0] += (helpTimeStop - helpTime);
    data->slauu2.calls[0]++;
    
    return;
}


/* Wrapper for slauum */
void hook_slauum(char *uplo, blasint *n, float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_slauum((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->slauum.timings[0] += (helpTimeStop - helpTime);
    data->slauum.calls[0]++;
    
    return;
}


/* Wrapper for sopgtr */
void hook_sopgtr(char *uplo, blasint *n, float *ap, float *tau, float *q, blasint *ldq, float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sopgtr((void *) uplo, (void *) n, (void *) ap, (void *) tau, (void *) q, (void *) ldq, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->sopgtr.timings[0] += (helpTimeStop - helpTime);
    data->sopgtr.calls[0]++;
    
    return;
}


/* Wrapper for sopmtr */
void hook_sopmtr(char *side, char *uplo, char *trans, blasint *m, blasint *n, float *ap, float *tau, float *c, blasint *ldc, float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sopmtr((void *) side, (void *) uplo, (void *) trans, (void *) m, (void *) n, (void *) ap, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, uplo_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sopmtr.timings[0] += (helpTimeStop - helpTime);
    data->sopmtr.calls[0]++;
    
    return;
}


/* Wrapper for sorbdb */
void hook_sorbdb(char *trans, char *signs, blasint *m, blasint *p, blasint *q, float *x11, blasint *ldx11, float *x12, blasint *ldx12, float *x21, blasint *ldx21, float *x22, blasint *ldx22, float *theta, float *phi, float *taup1, float *taup2, float *tauq1, float *tauq2, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t signs_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorbdb((void *) trans, (void *) signs, (void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x12, (void *) ldx12, (void *) x21, (void *) ldx21, (void *) x22, (void *) ldx22, (void *) theta, (void *) phi, (void *) taup1, (void *) taup2, (void *) tauq1, (void *) tauq2, (void *) work, (void *) lwork, (void *) info, trans_len, signs_len);

    helpTimeStop = flexiblas_wtime();

    data->sorbdb.timings[0] += (helpTimeStop - helpTime);
    data->sorbdb.calls[0]++;
    
    return;
}


/* Wrapper for sorbdb1 */
void hook_sorbdb1(blasint *m, blasint *p, blasint *q, float *x11, blasint *ldx11, float *x21, blasint *ldx21, float *theta, float *phi, float *taup1, float *taup2, float *tauq1, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorbdb1((void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x21, (void *) ldx21, (void *) theta, (void *) phi, (void *) taup1, (void *) taup2, (void *) tauq1, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sorbdb1.timings[0] += (helpTimeStop - helpTime);
    data->sorbdb1.calls[0]++;
    
    return;
}


/* Wrapper for sorbdb2 */
void hook_sorbdb2(blasint *m, blasint *p, blasint *q, float *x11, blasint *ldx11, float *x21, blasint *ldx21, float *theta, float *phi, float *taup1, float *taup2, float *tauq1, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorbdb2((void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x21, (void *) ldx21, (void *) theta, (void *) phi, (void *) taup1, (void *) taup2, (void *) tauq1, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sorbdb2.timings[0] += (helpTimeStop - helpTime);
    data->sorbdb2.calls[0]++;
    
    return;
}


/* Wrapper for sorbdb3 */
void hook_sorbdb3(blasint *m, blasint *p, blasint *q, float *x11, blasint *ldx11, float *x21, blasint *ldx21, float *theta, float *phi, float *taup1, float *taup2, float *tauq1, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorbdb3((void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x21, (void *) ldx21, (void *) theta, (void *) phi, (void *) taup1, (void *) taup2, (void *) tauq1, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sorbdb3.timings[0] += (helpTimeStop - helpTime);
    data->sorbdb3.calls[0]++;
    
    return;
}


/* Wrapper for sorbdb4 */
void hook_sorbdb4(blasint *m, blasint *p, blasint *q, float *x11, blasint *ldx11, float *x21, blasint *ldx21, float *theta, float *phi, float *taup1, float *taup2, float *tauq1, float *phantom, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorbdb4((void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x21, (void *) ldx21, (void *) theta, (void *) phi, (void *) taup1, (void *) taup2, (void *) tauq1, (void *) phantom, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sorbdb4.timings[0] += (helpTimeStop - helpTime);
    data->sorbdb4.calls[0]++;
    
    return;
}


/* Wrapper for sorbdb5 */
void hook_sorbdb5(blasint *m1, blasint *m2, blasint *n, float *x1, blasint *incx1, float *x2, blasint *incx2, float *q1, blasint *ldq1, float *q2, blasint *ldq2, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorbdb5((void *) m1, (void *) m2, (void *) n, (void *) x1, (void *) incx1, (void *) x2, (void *) incx2, (void *) q1, (void *) ldq1, (void *) q2, (void *) ldq2, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sorbdb5.timings[0] += (helpTimeStop - helpTime);
    data->sorbdb5.calls[0]++;
    
    return;
}


/* Wrapper for sorbdb6 */
void hook_sorbdb6(blasint *m1, blasint *m2, blasint *n, float *x1, blasint *incx1, float *x2, blasint *incx2, float *q1, blasint *ldq1, float *q2, blasint *ldq2, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorbdb6((void *) m1, (void *) m2, (void *) n, (void *) x1, (void *) incx1, (void *) x2, (void *) incx2, (void *) q1, (void *) ldq1, (void *) q2, (void *) ldq2, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sorbdb6.timings[0] += (helpTimeStop - helpTime);
    data->sorbdb6.calls[0]++;
    
    return;
}


/* Wrapper for sorcsd */
void hook_sorcsd(char *jobu1, char *jobu2, char *jobv1t, char *jobv2t, char *trans, char *signs, blasint *m, blasint *p, blasint *q, float *x11, blasint *ldx11, float *x12, blasint *ldx12, float *x21, blasint *ldx21, float *x22, blasint *ldx22, float *theta, float *u1, blasint *ldu1, float *u2, blasint *ldu2, float *v1t, blasint *ldv1t, float *v2t, blasint *ldv2t, float *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t jobu1_len, flexiblas_fortran_charlen_t jobu2_len, flexiblas_fortran_charlen_t jobv1t_len, flexiblas_fortran_charlen_t jobv2t_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t signs_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorcsd((void *) jobu1, (void *) jobu2, (void *) jobv1t, (void *) jobv2t, (void *) trans, (void *) signs, (void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x12, (void *) ldx12, (void *) x21, (void *) ldx21, (void *) x22, (void *) ldx22, (void *) theta, (void *) u1, (void *) ldu1, (void *) u2, (void *) ldu2, (void *) v1t, (void *) ldv1t, (void *) v2t, (void *) ldv2t, (void *) work, (void *) lwork, (void *) iwork, (void *) info, jobu1_len, jobu2_len, jobv1t_len, jobv2t_len, trans_len, signs_len);

    helpTimeStop = flexiblas_wtime();

    data->sorcsd.timings[0] += (helpTimeStop - helpTime);
    data->sorcsd.calls[0]++;
    
    return;
}


/* Wrapper for sorcsd2by1 */
void hook_sorcsd2by1(char *jobu1, char *jobu2, char *jobv1t, blasint *m, blasint *p, blasint *q, float *x11, blasint *ldx11, float *x21, blasint *ldx21, float *theta, float *u1, blasint *ldu1, float *u2, blasint *ldu2, float *v1t, blasint *ldv1t, float *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t jobu1_len, flexiblas_fortran_charlen_t jobu2_len, flexiblas_fortran_charlen_t jobv1t_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorcsd2by1((void *) jobu1, (void *) jobu2, (void *) jobv1t, (void *) m, (void *) p, (void *) q, (void *) x11, (void *) ldx11, (void *) x21, (void *) ldx21, (void *) theta, (void *) u1, (void *) ldu1, (void *) u2, (void *) ldu2, (void *) v1t, (void *) ldv1t, (void *) work, (void *) lwork, (void *) iwork, (void *) info, jobu1_len, jobu2_len, jobv1t_len);

    helpTimeStop = flexiblas_wtime();

    data->sorcsd2by1.timings[0] += (helpTimeStop - helpTime);
    data->sorcsd2by1.calls[0]++;
    
    return;
}


/* Wrapper for sorg2l */
void hook_sorg2l(blasint *m, blasint *n, blasint *k, float *a, blasint *lda, float *tau, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorg2l((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sorg2l.timings[0] += (helpTimeStop - helpTime);
    data->sorg2l.calls[0]++;
    
    return;
}


/* Wrapper for sorg2r */
void hook_sorg2r(blasint *m, blasint *n, blasint *k, float *a, blasint *lda, float *tau, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorg2r((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sorg2r.timings[0] += (helpTimeStop - helpTime);
    data->sorg2r.calls[0]++;
    
    return;
}


/* Wrapper for sorgbr */
void hook_sorgbr(char *vect, blasint *m, blasint *n, blasint *k, float *a, blasint *lda, float *tau, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t vect_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorgbr((void *) vect, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info, vect_len);

    helpTimeStop = flexiblas_wtime();

    data->sorgbr.timings[0] += (helpTimeStop - helpTime);
    data->sorgbr.calls[0]++;
    
    return;
}


/* Wrapper for sorghr */
void hook_sorghr(blasint *n, blasint *ilo, blasint *ihi, float *a, blasint *lda, float *tau, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorghr((void *) n, (void *) ilo, (void *) ihi, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sorghr.timings[0] += (helpTimeStop - helpTime);
    data->sorghr.calls[0]++;
    
    return;
}


/* Wrapper for sorgl2 */
void hook_sorgl2(blasint *m, blasint *n, blasint *k, float *a, blasint *lda, float *tau, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorgl2((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sorgl2.timings[0] += (helpTimeStop - helpTime);
    data->sorgl2.calls[0]++;
    
    return;
}


/* Wrapper for sorglq */
void hook_sorglq(blasint *m, blasint *n, blasint *k, float *a, blasint *lda, float *tau, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorglq((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sorglq.timings[0] += (helpTimeStop - helpTime);
    data->sorglq.calls[0]++;
    
    return;
}


/* Wrapper for sorgql */
void hook_sorgql(blasint *m, blasint *n, blasint *k, float *a, blasint *lda, float *tau, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorgql((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sorgql.timings[0] += (helpTimeStop - helpTime);
    data->sorgql.calls[0]++;
    
    return;
}


/* Wrapper for sorgqr */
void hook_sorgqr(blasint *m, blasint *n, blasint *k, float *a, blasint *lda, float *tau, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorgqr((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sorgqr.timings[0] += (helpTimeStop - helpTime);
    data->sorgqr.calls[0]++;
    
    return;
}


/* Wrapper for sorgr2 */
void hook_sorgr2(blasint *m, blasint *n, blasint *k, float *a, blasint *lda, float *tau, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorgr2((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sorgr2.timings[0] += (helpTimeStop - helpTime);
    data->sorgr2.calls[0]++;
    
    return;
}


/* Wrapper for sorgrq */
void hook_sorgrq(blasint *m, blasint *n, blasint *k, float *a, blasint *lda, float *tau, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorgrq((void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sorgrq.timings[0] += (helpTimeStop - helpTime);
    data->sorgrq.calls[0]++;
    
    return;
}


/* Wrapper for sorgtr */
void hook_sorgtr(char *uplo, blasint *n, float *a, blasint *lda, float *tau, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorgtr((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->sorgtr.timings[0] += (helpTimeStop - helpTime);
    data->sorgtr.calls[0]++;
    
    return;
}


/* Wrapper for sorgtsqr */
void hook_sorgtsqr(blasint *m, blasint *n, blasint *mb, blasint *nb, float *a, blasint *lda, float *t, blasint *ldt, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorgtsqr((void *) m, (void *) n, (void *) mb, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sorgtsqr.timings[0] += (helpTimeStop - helpTime);
    data->sorgtsqr.calls[0]++;
    
    return;
}


/* Wrapper for sorgtsqr_row */
void hook_sorgtsqr_row(blasint *m, blasint *n, blasint *mb, blasint *nb, float *a, blasint *lda, float *t, blasint *ldt, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorgtsqr_row((void *) m, (void *) n, (void *) mb, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sorgtsqr_row.timings[0] += (helpTimeStop - helpTime);
    data->sorgtsqr_row.calls[0]++;
    
    return;
}


/* Wrapper for sorhr_col */
void hook_sorhr_col(blasint *m, blasint *n, blasint *nb, float *a, blasint *lda, float *t, blasint *ldt, float *d, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorhr_col((void *) m, (void *) n, (void *) nb, (void *) a, (void *) lda, (void *) t, (void *) ldt, (void *) d, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sorhr_col.timings[0] += (helpTimeStop - helpTime);
    data->sorhr_col.calls[0]++;
    
    return;
}


/* Wrapper for sorm22 */
void hook_sorm22(char *side, char *trans, blasint *m, blasint *n, blasint *n1, blasint *n2, float *q, blasint *ldq, float *c, blasint *ldc, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorm22((void *) side, (void *) trans, (void *) m, (void *) n, (void *) n1, (void *) n2, (void *) q, (void *) ldq, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sorm22.timings[0] += (helpTimeStop - helpTime);
    data->sorm22.calls[0]++;
    
    return;
}


/* Wrapper for sorm2l */
void hook_sorm2l(char *side, char *trans, blasint *m, blasint *n, blasint *k, float *a, blasint *lda, float *tau, float *c, blasint *ldc, float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorm2l((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sorm2l.timings[0] += (helpTimeStop - helpTime);
    data->sorm2l.calls[0]++;
    
    return;
}


/* Wrapper for sorm2r */
void hook_sorm2r(char *side, char *trans, blasint *m, blasint *n, blasint *k, float *a, blasint *lda, float *tau, float *c, blasint *ldc, float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorm2r((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sorm2r.timings[0] += (helpTimeStop - helpTime);
    data->sorm2r.calls[0]++;
    
    return;
}


/* Wrapper for sormbr */
void hook_sormbr(char *vect, char *side, char *trans, blasint *m, blasint *n, blasint *k, float *a, blasint *lda, float *tau, float *c, blasint *ldc, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t vect_len, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sormbr((void *) vect, (void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, vect_len, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sormbr.timings[0] += (helpTimeStop - helpTime);
    data->sormbr.calls[0]++;
    
    return;
}


/* Wrapper for sormhr */
void hook_sormhr(char *side, char *trans, blasint *m, blasint *n, blasint *ilo, blasint *ihi, float *a, blasint *lda, float *tau, float *c, blasint *ldc, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sormhr((void *) side, (void *) trans, (void *) m, (void *) n, (void *) ilo, (void *) ihi, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sormhr.timings[0] += (helpTimeStop - helpTime);
    data->sormhr.calls[0]++;
    
    return;
}


/* Wrapper for sorml2 */
void hook_sorml2(char *side, char *trans, blasint *m, blasint *n, blasint *k, float *a, blasint *lda, float *tau, float *c, blasint *ldc, float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sorml2((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sorml2.timings[0] += (helpTimeStop - helpTime);
    data->sorml2.calls[0]++;
    
    return;
}


/* Wrapper for sormlq */
void hook_sormlq(char *side, char *trans, blasint *m, blasint *n, blasint *k, float *a, blasint *lda, float *tau, float *c, blasint *ldc, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sormlq((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sormlq.timings[0] += (helpTimeStop - helpTime);
    data->sormlq.calls[0]++;
    
    return;
}


/* Wrapper for sormql */
void hook_sormql(char *side, char *trans, blasint *m, blasint *n, blasint *k, float *a, blasint *lda, float *tau, float *c, blasint *ldc, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sormql((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sormql.timings[0] += (helpTimeStop - helpTime);
    data->sormql.calls[0]++;
    
    return;
}


/* Wrapper for sormqr */
void hook_sormqr(char *side, char *trans, blasint *m, blasint *n, blasint *k, float *a, blasint *lda, float *tau, float *c, blasint *ldc, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sormqr((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sormqr.timings[0] += (helpTimeStop - helpTime);
    data->sormqr.calls[0]++;
    
    return;
}


/* Wrapper for sormr2 */
void hook_sormr2(char *side, char *trans, blasint *m, blasint *n, blasint *k, float *a, blasint *lda, float *tau, float *c, blasint *ldc, float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sormr2((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sormr2.timings[0] += (helpTimeStop - helpTime);
    data->sormr2.calls[0]++;
    
    return;
}


/* Wrapper for sormr3 */
void hook_sormr3(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *l, float *a, blasint *lda, float *tau, float *c, blasint *ldc, float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sormr3((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) l, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sormr3.timings[0] += (helpTimeStop - helpTime);
    data->sormr3.calls[0]++;
    
    return;
}


/* Wrapper for sormrq */
void hook_sormrq(char *side, char *trans, blasint *m, blasint *n, blasint *k, float *a, blasint *lda, float *tau, float *c, blasint *ldc, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sormrq((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sormrq.timings[0] += (helpTimeStop - helpTime);
    data->sormrq.calls[0]++;
    
    return;
}


/* Wrapper for sormrz */
void hook_sormrz(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *l, float *a, blasint *lda, float *tau, float *c, blasint *ldc, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sormrz((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) l, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sormrz.timings[0] += (helpTimeStop - helpTime);
    data->sormrz.calls[0]++;
    
    return;
}


/* Wrapper for sormtr */
void hook_sormtr(char *side, char *uplo, char *trans, blasint *m, blasint *n, float *a, blasint *lda, float *tau, float *c, blasint *ldc, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sormtr((void *) side, (void *) uplo, (void *) trans, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) c, (void *) ldc, (void *) work, (void *) lwork, (void *) info, side_len, uplo_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->sormtr.timings[0] += (helpTimeStop - helpTime);
    data->sormtr.calls[0]++;
    
    return;
}


/* Wrapper for spbcon */
void hook_spbcon(char *uplo, blasint *n, blasint *kd, float *ab, blasint *ldab, float *anorm, float *rcond, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spbcon((void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) anorm, (void *) rcond, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spbcon.timings[0] += (helpTimeStop - helpTime);
    data->spbcon.calls[0]++;
    
    return;
}


/* Wrapper for spbequ */
void hook_spbequ(char *uplo, blasint *n, blasint *kd, float *ab, blasint *ldab, float *s, float *scond, float *amax, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spbequ((void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) s, (void *) scond, (void *) amax, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spbequ.timings[0] += (helpTimeStop - helpTime);
    data->spbequ.calls[0]++;
    
    return;
}


/* Wrapper for spbrfs */
void hook_spbrfs(char *uplo, blasint *n, blasint *kd, blasint *nrhs, float *ab, blasint *ldab, float *afb, blasint *ldafb, float *b, blasint *ldb, float *x, blasint *ldx, float *ferr, float *berr, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spbrfs((void *) uplo, (void *) n, (void *) kd, (void *) nrhs, (void *) ab, (void *) ldab, (void *) afb, (void *) ldafb, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spbrfs.timings[0] += (helpTimeStop - helpTime);
    data->spbrfs.calls[0]++;
    
    return;
}


/* Wrapper for spbstf */
void hook_spbstf(char *uplo, blasint *n, blasint *kd, float *ab, blasint *ldab, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spbstf((void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spbstf.timings[0] += (helpTimeStop - helpTime);
    data->spbstf.calls[0]++;
    
    return;
}


/* Wrapper for spbsv */
void hook_spbsv(char *uplo, blasint *n, blasint *kd, blasint *nrhs, float *ab, blasint *ldab, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spbsv((void *) uplo, (void *) n, (void *) kd, (void *) nrhs, (void *) ab, (void *) ldab, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spbsv.timings[0] += (helpTimeStop - helpTime);
    data->spbsv.calls[0]++;
    
    return;
}


/* Wrapper for spbsvx */
void hook_spbsvx(char *fact, char *uplo, blasint *n, blasint *kd, blasint *nrhs, float *ab, blasint *ldab, float *afb, blasint *ldafb, char *equed, float *s, float *b, blasint *ldb, float *x, blasint *ldx, float *rcond, float *ferr, float *berr, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spbsvx((void *) fact, (void *) uplo, (void *) n, (void *) kd, (void *) nrhs, (void *) ab, (void *) ldab, (void *) afb, (void *) ldafb, (void *) equed, (void *) s, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, fact_len, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->spbsvx.timings[0] += (helpTimeStop - helpTime);
    data->spbsvx.calls[0]++;
    
    return;
}


/* Wrapper for spbtf2 */
void hook_spbtf2(char *uplo, blasint *n, blasint *kd, float *ab, blasint *ldab, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spbtf2((void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spbtf2.timings[0] += (helpTimeStop - helpTime);
    data->spbtf2.calls[0]++;
    
    return;
}


/* Wrapper for spbtrf */
void hook_spbtrf(char *uplo, blasint *n, blasint *kd, float *ab, blasint *ldab, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spbtrf((void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spbtrf.timings[0] += (helpTimeStop - helpTime);
    data->spbtrf.calls[0]++;
    
    return;
}


/* Wrapper for spbtrs */
void hook_spbtrs(char *uplo, blasint *n, blasint *kd, blasint *nrhs, float *ab, blasint *ldab, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spbtrs((void *) uplo, (void *) n, (void *) kd, (void *) nrhs, (void *) ab, (void *) ldab, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spbtrs.timings[0] += (helpTimeStop - helpTime);
    data->spbtrs.calls[0]++;
    
    return;
}


/* Wrapper for spftrf */
void hook_spftrf(char *transr, char *uplo, blasint *n, float *a, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spftrf((void *) transr, (void *) uplo, (void *) n, (void *) a, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spftrf.timings[0] += (helpTimeStop - helpTime);
    data->spftrf.calls[0]++;
    
    return;
}


/* Wrapper for spftri */
void hook_spftri(char *transr, char *uplo, blasint *n, float *a, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spftri((void *) transr, (void *) uplo, (void *) n, (void *) a, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spftri.timings[0] += (helpTimeStop - helpTime);
    data->spftri.calls[0]++;
    
    return;
}


/* Wrapper for spftrs */
void hook_spftrs(char *transr, char *uplo, blasint *n, blasint *nrhs, float *a, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spftrs((void *) transr, (void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) b, (void *) ldb, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spftrs.timings[0] += (helpTimeStop - helpTime);
    data->spftrs.calls[0]++;
    
    return;
}


/* Wrapper for spocon */
void hook_spocon(char *uplo, blasint *n, float *a, blasint *lda, float *anorm, float *rcond, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spocon((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) anorm, (void *) rcond, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spocon.timings[0] += (helpTimeStop - helpTime);
    data->spocon.calls[0]++;
    
    return;
}


/* Wrapper for spoequ */
void hook_spoequ(blasint *n, float *a, blasint *lda, float *s, float *scond, float *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spoequ((void *) n, (void *) a, (void *) lda, (void *) s, (void *) scond, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->spoequ.timings[0] += (helpTimeStop - helpTime);
    data->spoequ.calls[0]++;
    
    return;
}


/* Wrapper for spoequb */
void hook_spoequb(blasint *n, float *a, blasint *lda, float *s, float *scond, float *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spoequb((void *) n, (void *) a, (void *) lda, (void *) s, (void *) scond, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->spoequb.timings[0] += (helpTimeStop - helpTime);
    data->spoequb.calls[0]++;
    
    return;
}


/* Wrapper for sporfs */
void hook_sporfs(char *uplo, blasint *n, blasint *nrhs, float *a, blasint *lda, float *af, blasint *ldaf, float *b, blasint *ldb, float *x, blasint *ldx, float *ferr, float *berr, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sporfs((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) af, (void *) ldaf, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->sporfs.timings[0] += (helpTimeStop - helpTime);
    data->sporfs.calls[0]++;
    
    return;
}


/* Wrapper for sposv */
void hook_sposv(char *uplo, blasint *n, blasint *nrhs, float *a, blasint *lda, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sposv((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->sposv.timings[0] += (helpTimeStop - helpTime);
    data->sposv.calls[0]++;
    
    return;
}


/* Wrapper for sposvx */
void hook_sposvx(char *fact, char *uplo, blasint *n, blasint *nrhs, float *a, blasint *lda, float *af, blasint *ldaf, char *equed, float *s, float *b, blasint *ldb, float *x, blasint *ldx, float *rcond, float *ferr, float *berr, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sposvx((void *) fact, (void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) af, (void *) ldaf, (void *) equed, (void *) s, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, fact_len, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->sposvx.timings[0] += (helpTimeStop - helpTime);
    data->sposvx.calls[0]++;
    
    return;
}


/* Wrapper for spotf2 */
void hook_spotf2(char *uplo, blasint *n, float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spotf2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spotf2.timings[0] += (helpTimeStop - helpTime);
    data->spotf2.calls[0]++;
    
    return;
}


/* Wrapper for spotrf */
void hook_spotrf(char *uplo, blasint *n, float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spotrf((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spotrf.timings[0] += (helpTimeStop - helpTime);
    data->spotrf.calls[0]++;
    
    return;
}


/* Wrapper for spotrf2 */
void hook_spotrf2(char *uplo, blasint *n, float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spotrf2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spotrf2.timings[0] += (helpTimeStop - helpTime);
    data->spotrf2.calls[0]++;
    
    return;
}


/* Wrapper for spotri */
void hook_spotri(char *uplo, blasint *n, float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spotri((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spotri.timings[0] += (helpTimeStop - helpTime);
    data->spotri.calls[0]++;
    
    return;
}


/* Wrapper for spotrs */
void hook_spotrs(char *uplo, blasint *n, blasint *nrhs, float *a, blasint *lda, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spotrs((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spotrs.timings[0] += (helpTimeStop - helpTime);
    data->spotrs.calls[0]++;
    
    return;
}


/* Wrapper for sppcon */
void hook_sppcon(char *uplo, blasint *n, float *ap, float *anorm, float *rcond, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sppcon((void *) uplo, (void *) n, (void *) ap, (void *) anorm, (void *) rcond, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->sppcon.timings[0] += (helpTimeStop - helpTime);
    data->sppcon.calls[0]++;
    
    return;
}


/* Wrapper for sppequ */
void hook_sppequ(char *uplo, blasint *n, float *ap, float *s, float *scond, float *amax, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sppequ((void *) uplo, (void *) n, (void *) ap, (void *) s, (void *) scond, (void *) amax, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->sppequ.timings[0] += (helpTimeStop - helpTime);
    data->sppequ.calls[0]++;
    
    return;
}


/* Wrapper for spprfs */
void hook_spprfs(char *uplo, blasint *n, blasint *nrhs, float *ap, float *afp, float *b, blasint *ldb, float *x, blasint *ldx, float *ferr, float *berr, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spprfs((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) afp, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spprfs.timings[0] += (helpTimeStop - helpTime);
    data->spprfs.calls[0]++;
    
    return;
}


/* Wrapper for sppsv */
void hook_sppsv(char *uplo, blasint *n, blasint *nrhs, float *ap, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sppsv((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->sppsv.timings[0] += (helpTimeStop - helpTime);
    data->sppsv.calls[0]++;
    
    return;
}


/* Wrapper for sppsvx */
void hook_sppsvx(char *fact, char *uplo, blasint *n, blasint *nrhs, float *ap, float *afp, char *equed, float *s, float *b, blasint *ldb, float *x, blasint *ldx, float *rcond, float *ferr, float *berr, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sppsvx((void *) fact, (void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) afp, (void *) equed, (void *) s, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, fact_len, uplo_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->sppsvx.timings[0] += (helpTimeStop - helpTime);
    data->sppsvx.calls[0]++;
    
    return;
}


/* Wrapper for spptrf */
void hook_spptrf(char *uplo, blasint *n, float *ap, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spptrf((void *) uplo, (void *) n, (void *) ap, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spptrf.timings[0] += (helpTimeStop - helpTime);
    data->spptrf.calls[0]++;
    
    return;
}


/* Wrapper for spptri */
void hook_spptri(char *uplo, blasint *n, float *ap, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spptri((void *) uplo, (void *) n, (void *) ap, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spptri.timings[0] += (helpTimeStop - helpTime);
    data->spptri.calls[0]++;
    
    return;
}


/* Wrapper for spptrs */
void hook_spptrs(char *uplo, blasint *n, blasint *nrhs, float *ap, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spptrs((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spptrs.timings[0] += (helpTimeStop - helpTime);
    data->spptrs.calls[0]++;
    
    return;
}


/* Wrapper for spstf2 */
void hook_spstf2(char *uplo, blasint *n, float *a, blasint *lda, blasint *piv, blasint *rank, float *tol, float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spstf2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) piv, (void *) rank, (void *) tol, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spstf2.timings[0] += (helpTimeStop - helpTime);
    data->spstf2.calls[0]++;
    
    return;
}


/* Wrapper for spstrf */
void hook_spstrf(char *uplo, blasint *n, float *a, blasint *lda, blasint *piv, blasint *rank, float *tol, float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spstrf((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) piv, (void *) rank, (void *) tol, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->spstrf.timings[0] += (helpTimeStop - helpTime);
    data->spstrf.calls[0]++;
    
    return;
}


/* Wrapper for sptcon */
void hook_sptcon(blasint *n, float *d, float *e, float *anorm, float *rcond, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sptcon((void *) n, (void *) d, (void *) e, (void *) anorm, (void *) rcond, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sptcon.timings[0] += (helpTimeStop - helpTime);
    data->sptcon.calls[0]++;
    
    return;
}


/* Wrapper for spteqr */
void hook_spteqr(char *compz, blasint *n, float *d, float *e, float *z, blasint *ldz, float *work, blasint *info, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spteqr((void *) compz, (void *) n, (void *) d, (void *) e, (void *) z, (void *) ldz, (void *) work, (void *) info, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->spteqr.timings[0] += (helpTimeStop - helpTime);
    data->spteqr.calls[0]++;
    
    return;
}


/* Wrapper for sptrfs */
void hook_sptrfs(blasint *n, blasint *nrhs, float *d, float *e, float *df, float *ef, float *b, blasint *ldb, float *x, blasint *ldx, float *ferr, float *berr, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sptrfs((void *) n, (void *) nrhs, (void *) d, (void *) e, (void *) df, (void *) ef, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sptrfs.timings[0] += (helpTimeStop - helpTime);
    data->sptrfs.calls[0]++;
    
    return;
}


/* Wrapper for sptsv */
void hook_sptsv(blasint *n, blasint *nrhs, float *d, float *e, float *b, blasint *ldb, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sptsv((void *) n, (void *) nrhs, (void *) d, (void *) e, (void *) b, (void *) ldb, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sptsv.timings[0] += (helpTimeStop - helpTime);
    data->sptsv.calls[0]++;
    
    return;
}


/* Wrapper for sptsvx */
void hook_sptsvx(char *fact, blasint *n, blasint *nrhs, float *d, float *e, float *df, float *ef, float *b, blasint *ldb, float *x, blasint *ldx, float *rcond, float *ferr, float *berr, float *work, blasint *info, flexiblas_fortran_charlen_t fact_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sptsvx((void *) fact, (void *) n, (void *) nrhs, (void *) d, (void *) e, (void *) df, (void *) ef, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) info, fact_len);

    helpTimeStop = flexiblas_wtime();

    data->sptsvx.timings[0] += (helpTimeStop - helpTime);
    data->sptsvx.calls[0]++;
    
    return;
}


/* Wrapper for spttrf */
void hook_spttrf(blasint *n, float *d, float *e, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spttrf((void *) n, (void *) d, (void *) e, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->spttrf.timings[0] += (helpTimeStop - helpTime);
    data->spttrf.calls[0]++;
    
    return;
}


/* Wrapper for spttrs */
void hook_spttrs(blasint *n, blasint *nrhs, float *d, float *e, float *b, blasint *ldb, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_spttrs((void *) n, (void *) nrhs, (void *) d, (void *) e, (void *) b, (void *) ldb, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->spttrs.timings[0] += (helpTimeStop - helpTime);
    data->spttrs.calls[0]++;
    
    return;
}


/* Wrapper for sptts2 */
void hook_sptts2(blasint *n, blasint *nrhs, float *d, float *e, float *b, blasint *ldb)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sptts2((void *) n, (void *) nrhs, (void *) d, (void *) e, (void *) b, (void *) ldb);

    helpTimeStop = flexiblas_wtime();

    data->sptts2.timings[0] += (helpTimeStop - helpTime);
    data->sptts2.calls[0]++;
    
    return;
}


/* Wrapper for srscl */
void hook_srscl(blasint *n, float *sa, float *sx, blasint *incx)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_srscl((void *) n, (void *) sa, (void *) sx, (void *) incx);

    helpTimeStop = flexiblas_wtime();

    data->srscl.timings[0] += (helpTimeStop - helpTime);
    data->srscl.calls[0]++;
    
    return;
}


/* Wrapper for ssb2st_kernels */
void hook_ssb2st_kernels(char *uplo, blaslogical *wantz, blasint *ttype, blasint *st, blasint *ed, blasint *sweep, blasint *n, blasint *nb, blasint *ib, float *a, blasint *lda, float *v, float *tau, blasint *ldvt, float *work, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssb2st_kernels((void *) uplo, (void *) wantz, (void *) ttype, (void *) st, (void *) ed, (void *) sweep, (void *) n, (void *) nb, (void *) ib, (void *) a, (void *) lda, (void *) v, (void *) tau, (void *) ldvt, (void *) work, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssb2st_kernels.timings[0] += (helpTimeStop - helpTime);
    data->ssb2st_kernels.calls[0]++;
    
    return;
}


/* Wrapper for ssbev */
void hook_ssbev(char *jobz, char *uplo, blasint *n, blasint *kd, float *ab, blasint *ldab, float *w, float *z, blasint *ldz, float *work, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssbev((void *) jobz, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssbev.timings[0] += (helpTimeStop - helpTime);
    data->ssbev.calls[0]++;
    
    return;
}


/* Wrapper for ssbev_2stage */
void hook_ssbev_2stage(char *jobz, char *uplo, blasint *n, blasint *kd, float *ab, blasint *ldab, float *w, float *z, blasint *ldz, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssbev_2stage((void *) jobz, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssbev_2stage.timings[0] += (helpTimeStop - helpTime);
    data->ssbev_2stage.calls[0]++;
    
    return;
}


/* Wrapper for ssbevd */
void hook_ssbevd(char *jobz, char *uplo, blasint *n, blasint *kd, float *ab, blasint *ldab, float *w, float *z, blasint *ldz, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssbevd((void *) jobz, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssbevd.timings[0] += (helpTimeStop - helpTime);
    data->ssbevd.calls[0]++;
    
    return;
}


/* Wrapper for ssbevd_2stage */
void hook_ssbevd_2stage(char *jobz, char *uplo, blasint *n, blasint *kd, float *ab, blasint *ldab, float *w, float *z, blasint *ldz, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssbevd_2stage((void *) jobz, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssbevd_2stage.timings[0] += (helpTimeStop - helpTime);
    data->ssbevd_2stage.calls[0]++;
    
    return;
}


/* Wrapper for ssbevx */
void hook_ssbevx(char *jobz, char *range, char *uplo, blasint *n, blasint *kd, float *ab, blasint *ldab, float *q, blasint *ldq, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, float *z, blasint *ldz, float *work, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssbevx((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) q, (void *) ldq, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssbevx.timings[0] += (helpTimeStop - helpTime);
    data->ssbevx.calls[0]++;
    
    return;
}


/* Wrapper for ssbevx_2stage */
void hook_ssbevx_2stage(char *jobz, char *range, char *uplo, blasint *n, blasint *kd, float *ab, blasint *ldab, float *q, blasint *ldq, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, float *z, blasint *ldz, float *work, blasint *lwork, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssbevx_2stage((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) q, (void *) ldq, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssbevx_2stage.timings[0] += (helpTimeStop - helpTime);
    data->ssbevx_2stage.calls[0]++;
    
    return;
}


/* Wrapper for ssbgst */
void hook_ssbgst(char *vect, char *uplo, blasint *n, blasint *ka, blasint *kb, float *ab, blasint *ldab, float *bb, blasint *ldbb, float *x, blasint *ldx, float *work, blasint *info, flexiblas_fortran_charlen_t vect_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssbgst((void *) vect, (void *) uplo, (void *) n, (void *) ka, (void *) kb, (void *) ab, (void *) ldab, (void *) bb, (void *) ldbb, (void *) x, (void *) ldx, (void *) work, (void *) info, vect_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssbgst.timings[0] += (helpTimeStop - helpTime);
    data->ssbgst.calls[0]++;
    
    return;
}


/* Wrapper for ssbgv */
void hook_ssbgv(char *jobz, char *uplo, blasint *n, blasint *ka, blasint *kb, float *ab, blasint *ldab, float *bb, blasint *ldbb, float *w, float *z, blasint *ldz, float *work, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssbgv((void *) jobz, (void *) uplo, (void *) n, (void *) ka, (void *) kb, (void *) ab, (void *) ldab, (void *) bb, (void *) ldbb, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssbgv.timings[0] += (helpTimeStop - helpTime);
    data->ssbgv.calls[0]++;
    
    return;
}


/* Wrapper for ssbgvd */
void hook_ssbgvd(char *jobz, char *uplo, blasint *n, blasint *ka, blasint *kb, float *ab, blasint *ldab, float *bb, blasint *ldbb, float *w, float *z, blasint *ldz, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssbgvd((void *) jobz, (void *) uplo, (void *) n, (void *) ka, (void *) kb, (void *) ab, (void *) ldab, (void *) bb, (void *) ldbb, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssbgvd.timings[0] += (helpTimeStop - helpTime);
    data->ssbgvd.calls[0]++;
    
    return;
}


/* Wrapper for ssbgvx */
void hook_ssbgvx(char *jobz, char *range, char *uplo, blasint *n, blasint *ka, blasint *kb, float *ab, blasint *ldab, float *bb, blasint *ldbb, float *q, blasint *ldq, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, float *z, blasint *ldz, float *work, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssbgvx((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) ka, (void *) kb, (void *) ab, (void *) ldab, (void *) bb, (void *) ldbb, (void *) q, (void *) ldq, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssbgvx.timings[0] += (helpTimeStop - helpTime);
    data->ssbgvx.calls[0]++;
    
    return;
}


/* Wrapper for ssbtrd */
void hook_ssbtrd(char *vect, char *uplo, blasint *n, blasint *kd, float *ab, blasint *ldab, float *d, float *e, float *q, blasint *ldq, float *work, blasint *info, flexiblas_fortran_charlen_t vect_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssbtrd((void *) vect, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) d, (void *) e, (void *) q, (void *) ldq, (void *) work, (void *) info, vect_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssbtrd.timings[0] += (helpTimeStop - helpTime);
    data->ssbtrd.calls[0]++;
    
    return;
}


/* Wrapper for ssfrk */
void hook_ssfrk(char *transr, char *uplo, char *trans, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *beta, float *c, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssfrk((void *) transr, (void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, transr_len, uplo_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->ssfrk.timings[0] += (helpTimeStop - helpTime);
    data->ssfrk.calls[0]++;
    
    return;
}


/* Wrapper for sspcon */
void hook_sspcon(char *uplo, blasint *n, float *ap, blasint *ipiv, float *anorm, float *rcond, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sspcon((void *) uplo, (void *) n, (void *) ap, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->sspcon.timings[0] += (helpTimeStop - helpTime);
    data->sspcon.calls[0]++;
    
    return;
}


/* Wrapper for sspev */
void hook_sspev(char *jobz, char *uplo, blasint *n, float *ap, float *w, float *z, blasint *ldz, float *work, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sspev((void *) jobz, (void *) uplo, (void *) n, (void *) ap, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->sspev.timings[0] += (helpTimeStop - helpTime);
    data->sspev.calls[0]++;
    
    return;
}


/* Wrapper for sspevd */
void hook_sspevd(char *jobz, char *uplo, blasint *n, float *ap, float *w, float *z, blasint *ldz, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sspevd((void *) jobz, (void *) uplo, (void *) n, (void *) ap, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->sspevd.timings[0] += (helpTimeStop - helpTime);
    data->sspevd.calls[0]++;
    
    return;
}


/* Wrapper for sspevx */
void hook_sspevx(char *jobz, char *range, char *uplo, blasint *n, float *ap, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, float *z, blasint *ldz, float *work, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sspevx((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) ap, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->sspevx.timings[0] += (helpTimeStop - helpTime);
    data->sspevx.calls[0]++;
    
    return;
}


/* Wrapper for sspgst */
void hook_sspgst(blasint *itype, char *uplo, blasint *n, float *ap, float *bp, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sspgst((void *) itype, (void *) uplo, (void *) n, (void *) ap, (void *) bp, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->sspgst.timings[0] += (helpTimeStop - helpTime);
    data->sspgst.calls[0]++;
    
    return;
}


/* Wrapper for sspgv */
void hook_sspgv(blasint *itype, char *jobz, char *uplo, blasint *n, float *ap, float *bp, float *w, float *z, blasint *ldz, float *work, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sspgv((void *) itype, (void *) jobz, (void *) uplo, (void *) n, (void *) ap, (void *) bp, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->sspgv.timings[0] += (helpTimeStop - helpTime);
    data->sspgv.calls[0]++;
    
    return;
}


/* Wrapper for sspgvd */
void hook_sspgvd(blasint *itype, char *jobz, char *uplo, blasint *n, float *ap, float *bp, float *w, float *z, blasint *ldz, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sspgvd((void *) itype, (void *) jobz, (void *) uplo, (void *) n, (void *) ap, (void *) bp, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->sspgvd.timings[0] += (helpTimeStop - helpTime);
    data->sspgvd.calls[0]++;
    
    return;
}


/* Wrapper for sspgvx */
void hook_sspgvx(blasint *itype, char *jobz, char *range, char *uplo, blasint *n, float *ap, float *bp, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, float *z, blasint *ldz, float *work, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sspgvx((void *) itype, (void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) ap, (void *) bp, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->sspgvx.timings[0] += (helpTimeStop - helpTime);
    data->sspgvx.calls[0]++;
    
    return;
}


/* Wrapper for ssprfs */
void hook_ssprfs(char *uplo, blasint *n, blasint *nrhs, float *ap, float *afp, blasint *ipiv, float *b, blasint *ldb, float *x, blasint *ldx, float *ferr, float *berr, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssprfs((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) afp, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssprfs.timings[0] += (helpTimeStop - helpTime);
    data->ssprfs.calls[0]++;
    
    return;
}


/* Wrapper for sspsv */
void hook_sspsv(char *uplo, blasint *n, blasint *nrhs, float *ap, blasint *ipiv, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sspsv((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->sspsv.timings[0] += (helpTimeStop - helpTime);
    data->sspsv.calls[0]++;
    
    return;
}


/* Wrapper for sspsvx */
void hook_sspsvx(char *fact, char *uplo, blasint *n, blasint *nrhs, float *ap, float *afp, blasint *ipiv, float *b, blasint *ldb, float *x, blasint *ldx, float *rcond, float *ferr, float *berr, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sspsvx((void *) fact, (void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) afp, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, fact_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->sspsvx.timings[0] += (helpTimeStop - helpTime);
    data->sspsvx.calls[0]++;
    
    return;
}


/* Wrapper for ssptrd */
void hook_ssptrd(char *uplo, blasint *n, float *ap, float *d, float *e, float *tau, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssptrd((void *) uplo, (void *) n, (void *) ap, (void *) d, (void *) e, (void *) tau, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssptrd.timings[0] += (helpTimeStop - helpTime);
    data->ssptrd.calls[0]++;
    
    return;
}


/* Wrapper for ssptrf */
void hook_ssptrf(char *uplo, blasint *n, float *ap, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssptrf((void *) uplo, (void *) n, (void *) ap, (void *) ipiv, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssptrf.timings[0] += (helpTimeStop - helpTime);
    data->ssptrf.calls[0]++;
    
    return;
}


/* Wrapper for ssptri */
void hook_ssptri(char *uplo, blasint *n, float *ap, blasint *ipiv, float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssptri((void *) uplo, (void *) n, (void *) ap, (void *) ipiv, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssptri.timings[0] += (helpTimeStop - helpTime);
    data->ssptri.calls[0]++;
    
    return;
}


/* Wrapper for ssptrs */
void hook_ssptrs(char *uplo, blasint *n, blasint *nrhs, float *ap, blasint *ipiv, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssptrs((void *) uplo, (void *) n, (void *) nrhs, (void *) ap, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssptrs.timings[0] += (helpTimeStop - helpTime);
    data->ssptrs.calls[0]++;
    
    return;
}


/* Wrapper for sstebz */
void hook_sstebz(char *range, char *order, blasint *n, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, float *d, float *e, blasint *m, blasint *nsplit, float *w, blasint *iblock, blasint *isplit, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t order_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sstebz((void *) range, (void *) order, (void *) n, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) d, (void *) e, (void *) m, (void *) nsplit, (void *) w, (void *) iblock, (void *) isplit, (void *) work, (void *) iwork, (void *) info, range_len, order_len);

    helpTimeStop = flexiblas_wtime();

    data->sstebz.timings[0] += (helpTimeStop - helpTime);
    data->sstebz.calls[0]++;
    
    return;
}


/* Wrapper for sstedc */
void hook_sstedc(char *compz, blasint *n, float *d, float *e, float *z, blasint *ldz, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sstedc((void *) compz, (void *) n, (void *) d, (void *) e, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->sstedc.timings[0] += (helpTimeStop - helpTime);
    data->sstedc.calls[0]++;
    
    return;
}


/* Wrapper for sstegr */
void hook_sstegr(char *jobz, char *range, blasint *n, float *d, float *e, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, float *z, blasint *ldz, blasint *isuppz, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sstegr((void *) jobz, (void *) range, (void *) n, (void *) d, (void *) e, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) isuppz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, range_len);

    helpTimeStop = flexiblas_wtime();

    data->sstegr.timings[0] += (helpTimeStop - helpTime);
    data->sstegr.calls[0]++;
    
    return;
}


/* Wrapper for sstein */
void hook_sstein(blasint *n, float *d, float *e, blasint *m, float *w, blasint *iblock, blasint *isplit, float *z, blasint *ldz, float *work, blasint *iwork, blasint *ifail, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sstein((void *) n, (void *) d, (void *) e, (void *) m, (void *) w, (void *) iblock, (void *) isplit, (void *) z, (void *) ldz, (void *) work, (void *) iwork, (void *) ifail, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->sstein.timings[0] += (helpTimeStop - helpTime);
    data->sstein.calls[0]++;
    
    return;
}


/* Wrapper for sstemr */
void hook_sstemr(char *jobz, char *range, blasint *n, float *d, float *e, float *vl, float *vu, blasint *il, blasint *iu, blasint *m, float *w, float *z, blasint *ldz, blasint *nzc, blasint *isuppz, blaslogical *tryrac, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sstemr((void *) jobz, (void *) range, (void *) n, (void *) d, (void *) e, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) nzc, (void *) isuppz, (void *) tryrac, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, range_len);

    helpTimeStop = flexiblas_wtime();

    data->sstemr.timings[0] += (helpTimeStop - helpTime);
    data->sstemr.calls[0]++;
    
    return;
}


/* Wrapper for ssteqr */
void hook_ssteqr(char *compz, blasint *n, float *d, float *e, float *z, blasint *ldz, float *work, blasint *info, flexiblas_fortran_charlen_t compz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssteqr((void *) compz, (void *) n, (void *) d, (void *) e, (void *) z, (void *) ldz, (void *) work, (void *) info, compz_len);

    helpTimeStop = flexiblas_wtime();

    data->ssteqr.timings[0] += (helpTimeStop - helpTime);
    data->ssteqr.calls[0]++;
    
    return;
}


/* Wrapper for ssterf */
void hook_ssterf(blasint *n, float *d, float *e, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssterf((void *) n, (void *) d, (void *) e, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->ssterf.timings[0] += (helpTimeStop - helpTime);
    data->ssterf.calls[0]++;
    
    return;
}


/* Wrapper for sstev */
void hook_sstev(char *jobz, blasint *n, float *d, float *e, float *z, blasint *ldz, float *work, blasint *info, flexiblas_fortran_charlen_t jobz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sstev((void *) jobz, (void *) n, (void *) d, (void *) e, (void *) z, (void *) ldz, (void *) work, (void *) info, jobz_len);

    helpTimeStop = flexiblas_wtime();

    data->sstev.timings[0] += (helpTimeStop - helpTime);
    data->sstev.calls[0]++;
    
    return;
}


/* Wrapper for sstevd */
void hook_sstevd(char *jobz, blasint *n, float *d, float *e, float *z, blasint *ldz, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sstevd((void *) jobz, (void *) n, (void *) d, (void *) e, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len);

    helpTimeStop = flexiblas_wtime();

    data->sstevd.timings[0] += (helpTimeStop - helpTime);
    data->sstevd.calls[0]++;
    
    return;
}


/* Wrapper for sstevr */
void hook_sstevr(char *jobz, char *range, blasint *n, float *d, float *e, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, float *z, blasint *ldz, blasint *isuppz, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sstevr((void *) jobz, (void *) range, (void *) n, (void *) d, (void *) e, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) isuppz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, range_len);

    helpTimeStop = flexiblas_wtime();

    data->sstevr.timings[0] += (helpTimeStop - helpTime);
    data->sstevr.calls[0]++;
    
    return;
}


/* Wrapper for sstevx */
void hook_sstevx(char *jobz, char *range, blasint *n, float *d, float *e, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, float *z, blasint *ldz, float *work, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_sstevx((void *) jobz, (void *) range, (void *) n, (void *) d, (void *) e, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len);

    helpTimeStop = flexiblas_wtime();

    data->sstevx.timings[0] += (helpTimeStop - helpTime);
    data->sstevx.calls[0]++;
    
    return;
}


/* Wrapper for ssycon */
void hook_ssycon(char *uplo, blasint *n, float *a, blasint *lda, blasint *ipiv, float *anorm, float *rcond, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssycon((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssycon.timings[0] += (helpTimeStop - helpTime);
    data->ssycon.calls[0]++;
    
    return;
}


/* Wrapper for ssycon_3 */
void hook_ssycon_3(char *uplo, blasint *n, float *a, blasint *lda, float *e, blasint *ipiv, float *anorm, float *rcond, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssycon_3((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssycon_3.timings[0] += (helpTimeStop - helpTime);
    data->ssycon_3.calls[0]++;
    
    return;
}


/* Wrapper for ssycon_rook */
void hook_ssycon_rook(char *uplo, blasint *n, float *a, blasint *lda, blasint *ipiv, float *anorm, float *rcond, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssycon_rook((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssycon_rook.timings[0] += (helpTimeStop - helpTime);
    data->ssycon_rook.calls[0]++;
    
    return;
}


/* Wrapper for ssyconv */
void hook_ssyconv(char *uplo, char *way, blasint *n, float *a, blasint *lda, blasint *ipiv, float *e, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t way_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssyconv((void *) uplo, (void *) way, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) e, (void *) info, uplo_len, way_len);

    helpTimeStop = flexiblas_wtime();

    data->ssyconv.timings[0] += (helpTimeStop - helpTime);
    data->ssyconv.calls[0]++;
    
    return;
}


/* Wrapper for ssyconvf */
void hook_ssyconvf(char *uplo, char *way, blasint *n, float *a, blasint *lda, float *e, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t way_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssyconvf((void *) uplo, (void *) way, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) info, uplo_len, way_len);

    helpTimeStop = flexiblas_wtime();

    data->ssyconvf.timings[0] += (helpTimeStop - helpTime);
    data->ssyconvf.calls[0]++;
    
    return;
}


/* Wrapper for ssyconvf_rook */
void hook_ssyconvf_rook(char *uplo, char *way, blasint *n, float *a, blasint *lda, float *e, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t way_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssyconvf_rook((void *) uplo, (void *) way, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) info, uplo_len, way_len);

    helpTimeStop = flexiblas_wtime();

    data->ssyconvf_rook.timings[0] += (helpTimeStop - helpTime);
    data->ssyconvf_rook.calls[0]++;
    
    return;
}


/* Wrapper for ssyequb */
void hook_ssyequb(char *uplo, blasint *n, float *a, blasint *lda, float *s, float *scond, float *amax, float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssyequb((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) s, (void *) scond, (void *) amax, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssyequb.timings[0] += (helpTimeStop - helpTime);
    data->ssyequb.calls[0]++;
    
    return;
}


/* Wrapper for ssyev */
void hook_ssyev(char *jobz, char *uplo, blasint *n, float *a, blasint *lda, float *w, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssyev((void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) w, (void *) work, (void *) lwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssyev.timings[0] += (helpTimeStop - helpTime);
    data->ssyev.calls[0]++;
    
    return;
}


/* Wrapper for ssyev_2stage */
void hook_ssyev_2stage(char *jobz, char *uplo, blasint *n, float *a, blasint *lda, float *w, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssyev_2stage((void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) w, (void *) work, (void *) lwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssyev_2stage.timings[0] += (helpTimeStop - helpTime);
    data->ssyev_2stage.calls[0]++;
    
    return;
}


/* Wrapper for ssyevd */
void hook_ssyevd(char *jobz, char *uplo, blasint *n, float *a, blasint *lda, float *w, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssyevd((void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) w, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssyevd.timings[0] += (helpTimeStop - helpTime);
    data->ssyevd.calls[0]++;
    
    return;
}


/* Wrapper for ssyevd_2stage */
void hook_ssyevd_2stage(char *jobz, char *uplo, blasint *n, float *a, blasint *lda, float *w, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssyevd_2stage((void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) w, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssyevd_2stage.timings[0] += (helpTimeStop - helpTime);
    data->ssyevd_2stage.calls[0]++;
    
    return;
}


/* Wrapper for ssyevr */
void hook_ssyevr(char *jobz, char *range, char *uplo, blasint *n, float *a, blasint *lda, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, float *z, blasint *ldz, blasint *isuppz, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssyevr((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) isuppz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssyevr.timings[0] += (helpTimeStop - helpTime);
    data->ssyevr.calls[0]++;
    
    return;
}


/* Wrapper for ssyevr_2stage */
void hook_ssyevr_2stage(char *jobz, char *range, char *uplo, blasint *n, float *a, blasint *lda, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, float *z, blasint *ldz, blasint *isuppz, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssyevr_2stage((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) isuppz, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssyevr_2stage.timings[0] += (helpTimeStop - helpTime);
    data->ssyevr_2stage.calls[0]++;
    
    return;
}


/* Wrapper for ssyevx */
void hook_ssyevx(char *jobz, char *range, char *uplo, blasint *n, float *a, blasint *lda, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, float *z, blasint *ldz, float *work, blasint *lwork, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssyevx((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssyevx.timings[0] += (helpTimeStop - helpTime);
    data->ssyevx.calls[0]++;
    
    return;
}


/* Wrapper for ssyevx_2stage */
void hook_ssyevx_2stage(char *jobz, char *range, char *uplo, blasint *n, float *a, blasint *lda, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, float *z, blasint *ldz, float *work, blasint *lwork, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssyevx_2stage((void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssyevx_2stage.timings[0] += (helpTimeStop - helpTime);
    data->ssyevx_2stage.calls[0]++;
    
    return;
}


/* Wrapper for ssygs2 */
void hook_ssygs2(blasint *itype, char *uplo, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssygs2((void *) itype, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssygs2.timings[0] += (helpTimeStop - helpTime);
    data->ssygs2.calls[0]++;
    
    return;
}


/* Wrapper for ssygst */
void hook_ssygst(blasint *itype, char *uplo, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssygst((void *) itype, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssygst.timings[0] += (helpTimeStop - helpTime);
    data->ssygst.calls[0]++;
    
    return;
}


/* Wrapper for ssygv */
void hook_ssygv(blasint *itype, char *jobz, char *uplo, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, float *w, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssygv((void *) itype, (void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) w, (void *) work, (void *) lwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssygv.timings[0] += (helpTimeStop - helpTime);
    data->ssygv.calls[0]++;
    
    return;
}


/* Wrapper for ssygv_2stage */
void hook_ssygv_2stage(blasint *itype, char *jobz, char *uplo, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, float *w, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssygv_2stage((void *) itype, (void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) w, (void *) work, (void *) lwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssygv_2stage.timings[0] += (helpTimeStop - helpTime);
    data->ssygv_2stage.calls[0]++;
    
    return;
}


/* Wrapper for ssygvd */
void hook_ssygvd(blasint *itype, char *jobz, char *uplo, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, float *w, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssygvd((void *) itype, (void *) jobz, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) w, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, jobz_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssygvd.timings[0] += (helpTimeStop - helpTime);
    data->ssygvd.calls[0]++;
    
    return;
}


/* Wrapper for ssygvx */
void hook_ssygvx(blasint *itype, char *jobz, char *range, char *uplo, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, float *vl, float *vu, blasint *il, blasint *iu, float *abstol, blasint *m, float *w, float *z, blasint *ldz, float *work, blasint *lwork, blasint *iwork, blasint *ifail, blasint *info, flexiblas_fortran_charlen_t jobz_len, flexiblas_fortran_charlen_t range_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssygvx((void *) itype, (void *) jobz, (void *) range, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) vl, (void *) vu, (void *) il, (void *) iu, (void *) abstol, (void *) m, (void *) w, (void *) z, (void *) ldz, (void *) work, (void *) lwork, (void *) iwork, (void *) ifail, (void *) info, jobz_len, range_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssygvx.timings[0] += (helpTimeStop - helpTime);
    data->ssygvx.calls[0]++;
    
    return;
}


/* Wrapper for ssyrfs */
void hook_ssyrfs(char *uplo, blasint *n, blasint *nrhs, float *a, blasint *lda, float *af, blasint *ldaf, blasint *ipiv, float *b, blasint *ldb, float *x, blasint *ldx, float *ferr, float *berr, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssyrfs((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) af, (void *) ldaf, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssyrfs.timings[0] += (helpTimeStop - helpTime);
    data->ssyrfs.calls[0]++;
    
    return;
}


/* Wrapper for ssysv */
void hook_ssysv(char *uplo, blasint *n, blasint *nrhs, float *a, blasint *lda, blasint *ipiv, float *b, blasint *ldb, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssysv((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssysv.timings[0] += (helpTimeStop - helpTime);
    data->ssysv.calls[0]++;
    
    return;
}


/* Wrapper for ssysv_aa */
void hook_ssysv_aa(char *uplo, blasint *n, blasint *nrhs, float *a, blasint *lda, blasint *ipiv, float *b, blasint *ldb, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssysv_aa((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssysv_aa.timings[0] += (helpTimeStop - helpTime);
    data->ssysv_aa.calls[0]++;
    
    return;
}


/* Wrapper for ssysv_aa_2stage */
void hook_ssysv_aa_2stage(char *uplo, blasint *n, blasint *nrhs, float *a, blasint *lda, float *tb, blasint *ltb, blasint *ipiv, blasint *ipiv2, float *b, blasint *ldb, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssysv_aa_2stage((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) tb, (void *) ltb, (void *) ipiv, (void *) ipiv2, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssysv_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->ssysv_aa_2stage.calls[0]++;
    
    return;
}


/* Wrapper for ssysv_rk */
void hook_ssysv_rk(char *uplo, blasint *n, blasint *nrhs, float *a, blasint *lda, float *e, blasint *ipiv, float *b, blasint *ldb, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssysv_rk((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssysv_rk.timings[0] += (helpTimeStop - helpTime);
    data->ssysv_rk.calls[0]++;
    
    return;
}


/* Wrapper for ssysv_rook */
void hook_ssysv_rook(char *uplo, blasint *n, blasint *nrhs, float *a, blasint *lda, blasint *ipiv, float *b, blasint *ldb, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssysv_rook((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssysv_rook.timings[0] += (helpTimeStop - helpTime);
    data->ssysv_rook.calls[0]++;
    
    return;
}


/* Wrapper for ssysvx */
void hook_ssysvx(char *fact, char *uplo, blasint *n, blasint *nrhs, float *a, blasint *lda, float *af, blasint *ldaf, blasint *ipiv, float *b, blasint *ldb, float *x, blasint *ldx, float *rcond, float *ferr, float *berr, float *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssysvx((void *) fact, (void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) af, (void *) ldaf, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) lwork, (void *) iwork, (void *) info, fact_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssysvx.timings[0] += (helpTimeStop - helpTime);
    data->ssysvx.calls[0]++;
    
    return;
}


/* Wrapper for ssyswapr */
void hook_ssyswapr(char *uplo, blasint *n, float *a, blasint *lda, blasint *i1, blasint *i2, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssyswapr((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) i1, (void *) i2, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssyswapr.timings[0] += (helpTimeStop - helpTime);
    data->ssyswapr.calls[0]++;
    
    return;
}


/* Wrapper for ssytd2 */
void hook_ssytd2(char *uplo, blasint *n, float *a, blasint *lda, float *d, float *e, float *tau, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytd2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) e, (void *) tau, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytd2.timings[0] += (helpTimeStop - helpTime);
    data->ssytd2.calls[0]++;
    
    return;
}


/* Wrapper for ssytf2 */
void hook_ssytf2(char *uplo, blasint *n, float *a, blasint *lda, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytf2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytf2.timings[0] += (helpTimeStop - helpTime);
    data->ssytf2.calls[0]++;
    
    return;
}


/* Wrapper for ssytf2_rk */
void hook_ssytf2_rk(char *uplo, blasint *n, float *a, blasint *lda, float *e, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytf2_rk((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytf2_rk.timings[0] += (helpTimeStop - helpTime);
    data->ssytf2_rk.calls[0]++;
    
    return;
}


/* Wrapper for ssytf2_rook */
void hook_ssytf2_rook(char *uplo, blasint *n, float *a, blasint *lda, blasint *ipiv, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytf2_rook((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytf2_rook.timings[0] += (helpTimeStop - helpTime);
    data->ssytf2_rook.calls[0]++;
    
    return;
}


/* Wrapper for ssytrd */
void hook_ssytrd(char *uplo, blasint *n, float *a, blasint *lda, float *d, float *e, float *tau, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytrd((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) e, (void *) tau, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytrd.timings[0] += (helpTimeStop - helpTime);
    data->ssytrd.calls[0]++;
    
    return;
}


/* Wrapper for ssytrd_2stage */
void hook_ssytrd_2stage(char *vect, char *uplo, blasint *n, float *a, blasint *lda, float *d, float *e, float *tau, float *hous2, blasint *lhous2, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t vect_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytrd_2stage((void *) vect, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) e, (void *) tau, (void *) hous2, (void *) lhous2, (void *) work, (void *) lwork, (void *) info, vect_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytrd_2stage.timings[0] += (helpTimeStop - helpTime);
    data->ssytrd_2stage.calls[0]++;
    
    return;
}


/* Wrapper for ssytrd_sb2st */
void hook_ssytrd_sb2st(char *stage1, char *vect, char *uplo, blasint *n, blasint *kd, float *ab, blasint *ldab, float *d, float *e, float *hous, blasint *lhous, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t stage1_len, flexiblas_fortran_charlen_t vect_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytrd_sb2st((void *) stage1, (void *) vect, (void *) uplo, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) d, (void *) e, (void *) hous, (void *) lhous, (void *) work, (void *) lwork, (void *) info, stage1_len, vect_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytrd_sb2st.timings[0] += (helpTimeStop - helpTime);
    data->ssytrd_sb2st.calls[0]++;
    
    return;
}


/* Wrapper for ssytrd_sy2sb */
void hook_ssytrd_sy2sb(char *uplo, blasint *n, blasint *kd, float *a, blasint *lda, float *ab, blasint *ldab, float *tau, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytrd_sy2sb((void *) uplo, (void *) n, (void *) kd, (void *) a, (void *) lda, (void *) ab, (void *) ldab, (void *) tau, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytrd_sy2sb.timings[0] += (helpTimeStop - helpTime);
    data->ssytrd_sy2sb.calls[0]++;
    
    return;
}


/* Wrapper for ssytrf */
void hook_ssytrf(char *uplo, blasint *n, float *a, blasint *lda, blasint *ipiv, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytrf((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytrf.timings[0] += (helpTimeStop - helpTime);
    data->ssytrf.calls[0]++;
    
    return;
}


/* Wrapper for ssytrf_aa */
void hook_ssytrf_aa(char *uplo, blasint *n, float *a, blasint *lda, blasint *ipiv, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytrf_aa((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytrf_aa.timings[0] += (helpTimeStop - helpTime);
    data->ssytrf_aa.calls[0]++;
    
    return;
}


/* Wrapper for ssytrf_aa_2stage */
void hook_ssytrf_aa_2stage(char *uplo, blasint *n, float *a, blasint *lda, float *tb, blasint *ltb, blasint *ipiv, blasint *ipiv2, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytrf_aa_2stage((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) tb, (void *) ltb, (void *) ipiv, (void *) ipiv2, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytrf_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->ssytrf_aa_2stage.calls[0]++;
    
    return;
}


/* Wrapper for ssytrf_rk */
void hook_ssytrf_rk(char *uplo, blasint *n, float *a, blasint *lda, float *e, blasint *ipiv, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytrf_rk((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytrf_rk.timings[0] += (helpTimeStop - helpTime);
    data->ssytrf_rk.calls[0]++;
    
    return;
}


/* Wrapper for ssytrf_rook */
void hook_ssytrf_rook(char *uplo, blasint *n, float *a, blasint *lda, blasint *ipiv, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytrf_rook((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytrf_rook.timings[0] += (helpTimeStop - helpTime);
    data->ssytrf_rook.calls[0]++;
    
    return;
}


/* Wrapper for ssytri */
void hook_ssytri(char *uplo, blasint *n, float *a, blasint *lda, blasint *ipiv, float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytri((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytri.timings[0] += (helpTimeStop - helpTime);
    data->ssytri.calls[0]++;
    
    return;
}


/* Wrapper for ssytri2 */
void hook_ssytri2(char *uplo, blasint *n, float *a, blasint *lda, blasint *ipiv, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytri2((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytri2.timings[0] += (helpTimeStop - helpTime);
    data->ssytri2.calls[0]++;
    
    return;
}


/* Wrapper for ssytri2x */
void hook_ssytri2x(char *uplo, blasint *n, float *a, blasint *lda, blasint *ipiv, float *work, blasint *nb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytri2x((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) nb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytri2x.timings[0] += (helpTimeStop - helpTime);
    data->ssytri2x.calls[0]++;
    
    return;
}


/* Wrapper for ssytri_3 */
void hook_ssytri_3(char *uplo, blasint *n, float *a, blasint *lda, float *e, blasint *ipiv, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytri_3((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytri_3.timings[0] += (helpTimeStop - helpTime);
    data->ssytri_3.calls[0]++;
    
    return;
}


/* Wrapper for ssytri_3x */
void hook_ssytri_3x(char *uplo, blasint *n, float *a, blasint *lda, float *e, blasint *ipiv, float *work, blasint *nb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytri_3x((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) work, (void *) nb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytri_3x.timings[0] += (helpTimeStop - helpTime);
    data->ssytri_3x.calls[0]++;
    
    return;
}


/* Wrapper for ssytri_rook */
void hook_ssytri_rook(char *uplo, blasint *n, float *a, blasint *lda, blasint *ipiv, float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytri_rook((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ipiv, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytri_rook.timings[0] += (helpTimeStop - helpTime);
    data->ssytri_rook.calls[0]++;
    
    return;
}


/* Wrapper for ssytrs */
void hook_ssytrs(char *uplo, blasint *n, blasint *nrhs, float *a, blasint *lda, blasint *ipiv, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytrs((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytrs.timings[0] += (helpTimeStop - helpTime);
    data->ssytrs.calls[0]++;
    
    return;
}


/* Wrapper for ssytrs2 */
void hook_ssytrs2(char *uplo, blasint *n, blasint *nrhs, float *a, blasint *lda, blasint *ipiv, float *b, blasint *ldb, float *work, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytrs2((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytrs2.timings[0] += (helpTimeStop - helpTime);
    data->ssytrs2.calls[0]++;
    
    return;
}


/* Wrapper for ssytrs_3 */
void hook_ssytrs_3(char *uplo, blasint *n, blasint *nrhs, float *a, blasint *lda, float *e, blasint *ipiv, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytrs_3((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) e, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytrs_3.timings[0] += (helpTimeStop - helpTime);
    data->ssytrs_3.calls[0]++;
    
    return;
}


/* Wrapper for ssytrs_aa */
void hook_ssytrs_aa(char *uplo, blasint *n, blasint *nrhs, float *a, blasint *lda, blasint *ipiv, float *b, blasint *ldb, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytrs_aa((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) work, (void *) lwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytrs_aa.timings[0] += (helpTimeStop - helpTime);
    data->ssytrs_aa.calls[0]++;
    
    return;
}


/* Wrapper for ssytrs_aa_2stage */
void hook_ssytrs_aa_2stage(char *uplo, blasint *n, blasint *nrhs, float *a, blasint *lda, float *tb, blasint *ltb, blasint *ipiv, blasint *ipiv2, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytrs_aa_2stage((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) tb, (void *) ltb, (void *) ipiv, (void *) ipiv2, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytrs_aa_2stage.timings[0] += (helpTimeStop - helpTime);
    data->ssytrs_aa_2stage.calls[0]++;
    
    return;
}


/* Wrapper for ssytrs_rook */
void hook_ssytrs_rook(char *uplo, blasint *n, blasint *nrhs, float *a, blasint *lda, blasint *ipiv, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_ssytrs_rook((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->ssytrs_rook.timings[0] += (helpTimeStop - helpTime);
    data->ssytrs_rook.calls[0]++;
    
    return;
}


/* Wrapper for stbcon */
void hook_stbcon(char *norm, char *uplo, char *diag, blasint *n, blasint *kd, float *ab, blasint *ldab, float *rcond, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stbcon((void *) norm, (void *) uplo, (void *) diag, (void *) n, (void *) kd, (void *) ab, (void *) ldab, (void *) rcond, (void *) work, (void *) iwork, (void *) info, norm_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->stbcon.timings[0] += (helpTimeStop - helpTime);
    data->stbcon.calls[0]++;
    
    return;
}


/* Wrapper for stbrfs */
void hook_stbrfs(char *uplo, char *trans, char *diag, blasint *n, blasint *kd, blasint *nrhs, float *ab, blasint *ldab, float *b, blasint *ldb, float *x, blasint *ldx, float *ferr, float *berr, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stbrfs((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) kd, (void *) nrhs, (void *) ab, (void *) ldab, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->stbrfs.timings[0] += (helpTimeStop - helpTime);
    data->stbrfs.calls[0]++;
    
    return;
}


/* Wrapper for stbtrs */
void hook_stbtrs(char *uplo, char *trans, char *diag, blasint *n, blasint *kd, blasint *nrhs, float *ab, blasint *ldab, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stbtrs((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) kd, (void *) nrhs, (void *) ab, (void *) ldab, (void *) b, (void *) ldb, (void *) info, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->stbtrs.timings[0] += (helpTimeStop - helpTime);
    data->stbtrs.calls[0]++;
    
    return;
}


/* Wrapper for stfsm */
void hook_stfsm(char *transr, char *side, char *uplo, char *trans, char *diag, blasint *m, blasint *n, float *alpha, float *a, float *b, blasint *ldb, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stfsm((void *) transr, (void *) side, (void *) uplo, (void *) trans, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) b, (void *) ldb, transr_len, side_len, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->stfsm.timings[0] += (helpTimeStop - helpTime);
    data->stfsm.calls[0]++;
    
    return;
}


/* Wrapper for stftri */
void hook_stftri(char *transr, char *uplo, char *diag, blasint *n, float *a, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stftri((void *) transr, (void *) uplo, (void *) diag, (void *) n, (void *) a, (void *) info, transr_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->stftri.timings[0] += (helpTimeStop - helpTime);
    data->stftri.calls[0]++;
    
    return;
}


/* Wrapper for stfttp */
void hook_stfttp(char *transr, char *uplo, blasint *n, float *arf, float *ap, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stfttp((void *) transr, (void *) uplo, (void *) n, (void *) arf, (void *) ap, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->stfttp.timings[0] += (helpTimeStop - helpTime);
    data->stfttp.calls[0]++;
    
    return;
}


/* Wrapper for stfttr */
void hook_stfttr(char *transr, char *uplo, blasint *n, float *arf, float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stfttr((void *) transr, (void *) uplo, (void *) n, (void *) arf, (void *) a, (void *) lda, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->stfttr.timings[0] += (helpTimeStop - helpTime);
    data->stfttr.calls[0]++;
    
    return;
}


/* Wrapper for stgevc */
void hook_stgevc(char *side, char *howmny, blaslogical *select, blasint *n, float *s, blasint *lds, float *p, blasint *ldp, float *vl, blasint *ldvl, float *vr, blasint *ldvr, blasint *mm, blasint *m, float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t howmny_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stgevc((void *) side, (void *) howmny, (void *) select, (void *) n, (void *) s, (void *) lds, (void *) p, (void *) ldp, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) mm, (void *) m, (void *) work, (void *) info, side_len, howmny_len);

    helpTimeStop = flexiblas_wtime();

    data->stgevc.timings[0] += (helpTimeStop - helpTime);
    data->stgevc.calls[0]++;
    
    return;
}


/* Wrapper for stgex2 */
void hook_stgex2(blaslogical *wantq, blaslogical *wantz, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, float *q, blasint *ldq, float *z, blasint *ldz, blasint *j1, blasint *n1, blasint *n2, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stgex2((void *) wantq, (void *) wantz, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) q, (void *) ldq, (void *) z, (void *) ldz, (void *) j1, (void *) n1, (void *) n2, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->stgex2.timings[0] += (helpTimeStop - helpTime);
    data->stgex2.calls[0]++;
    
    return;
}


/* Wrapper for stgexc */
void hook_stgexc(blaslogical *wantq, blaslogical *wantz, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, float *q, blasint *ldq, float *z, blasint *ldz, blasint *ifst, blasint *ilst, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stgexc((void *) wantq, (void *) wantz, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) q, (void *) ldq, (void *) z, (void *) ldz, (void *) ifst, (void *) ilst, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->stgexc.timings[0] += (helpTimeStop - helpTime);
    data->stgexc.calls[0]++;
    
    return;
}


/* Wrapper for stgsen */
void hook_stgsen(blasint *ijob, blaslogical *wantq, blaslogical *wantz, blaslogical *select, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, float *alphar, float *alphai, float *beta, float *q, blasint *ldq, float *z, blasint *ldz, blasint *m, float *pl, float *pr, float *dif, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stgsen((void *) ijob, (void *) wantq, (void *) wantz, (void *) select, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) alphar, (void *) alphai, (void *) beta, (void *) q, (void *) ldq, (void *) z, (void *) ldz, (void *) m, (void *) pl, (void *) pr, (void *) dif, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->stgsen.timings[0] += (helpTimeStop - helpTime);
    data->stgsen.calls[0]++;
    
    return;
}


/* Wrapper for stgsja */
void hook_stgsja(char *jobu, char *jobv, char *jobq, blasint *m, blasint *p, blasint *n, blasint *k, blasint *l, float *a, blasint *lda, float *b, blasint *ldb, float *tola, float *tolb, float *alpha, float *beta, float *u, blasint *ldu, float *v, blasint *ldv, float *q, blasint *ldq, float *work, blasint *ncycle, blasint *info, flexiblas_fortran_charlen_t jobu_len, flexiblas_fortran_charlen_t jobv_len, flexiblas_fortran_charlen_t jobq_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stgsja((void *) jobu, (void *) jobv, (void *) jobq, (void *) m, (void *) p, (void *) n, (void *) k, (void *) l, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) tola, (void *) tolb, (void *) alpha, (void *) beta, (void *) u, (void *) ldu, (void *) v, (void *) ldv, (void *) q, (void *) ldq, (void *) work, (void *) ncycle, (void *) info, jobu_len, jobv_len, jobq_len);

    helpTimeStop = flexiblas_wtime();

    data->stgsja.timings[0] += (helpTimeStop - helpTime);
    data->stgsja.calls[0]++;
    
    return;
}


/* Wrapper for stgsna */
void hook_stgsna(char *job, char *howmny, blaslogical *select, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, float *vl, blasint *ldvl, float *vr, blasint *ldvr, float *s, float *dif, blasint *mm, blasint *m, float *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t howmny_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stgsna((void *) job, (void *) howmny, (void *) select, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) s, (void *) dif, (void *) mm, (void *) m, (void *) work, (void *) lwork, (void *) iwork, (void *) info, job_len, howmny_len);

    helpTimeStop = flexiblas_wtime();

    data->stgsna.timings[0] += (helpTimeStop - helpTime);
    data->stgsna.calls[0]++;
    
    return;
}


/* Wrapper for stgsy2 */
void hook_stgsy2(char *trans, blasint *ijob, blasint *m, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, float *c, blasint *ldc, float *d, blasint *ldd, float *e, blasint *lde, float *f, blasint *ldf, float *scale, float *rdsum, float *rdscal, blasint *iwork, blasint *pq, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stgsy2((void *) trans, (void *) ijob, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) c, (void *) ldc, (void *) d, (void *) ldd, (void *) e, (void *) lde, (void *) f, (void *) ldf, (void *) scale, (void *) rdsum, (void *) rdscal, (void *) iwork, (void *) pq, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->stgsy2.timings[0] += (helpTimeStop - helpTime);
    data->stgsy2.calls[0]++;
    
    return;
}


/* Wrapper for stgsyl */
void hook_stgsyl(char *trans, blasint *ijob, blasint *m, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, float *c, blasint *ldc, float *d, blasint *ldd, float *e, blasint *lde, float *f, blasint *ldf, float *scale, float *dif, float *work, blasint *lwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stgsyl((void *) trans, (void *) ijob, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) c, (void *) ldc, (void *) d, (void *) ldd, (void *) e, (void *) lde, (void *) f, (void *) ldf, (void *) scale, (void *) dif, (void *) work, (void *) lwork, (void *) iwork, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->stgsyl.timings[0] += (helpTimeStop - helpTime);
    data->stgsyl.calls[0]++;
    
    return;
}


/* Wrapper for stpcon */
void hook_stpcon(char *norm, char *uplo, char *diag, blasint *n, float *ap, float *rcond, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stpcon((void *) norm, (void *) uplo, (void *) diag, (void *) n, (void *) ap, (void *) rcond, (void *) work, (void *) iwork, (void *) info, norm_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->stpcon.timings[0] += (helpTimeStop - helpTime);
    data->stpcon.calls[0]++;
    
    return;
}


/* Wrapper for stplqt */
void hook_stplqt(blasint *m, blasint *n, blasint *l, blasint *mb, float *a, blasint *lda, float *b, blasint *ldb, float *t, blasint *ldt, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stplqt((void *) m, (void *) n, (void *) l, (void *) mb, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) t, (void *) ldt, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->stplqt.timings[0] += (helpTimeStop - helpTime);
    data->stplqt.calls[0]++;
    
    return;
}


/* Wrapper for stplqt2 */
void hook_stplqt2(blasint *m, blasint *n, blasint *l, float *a, blasint *lda, float *b, blasint *ldb, float *t, blasint *ldt, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stplqt2((void *) m, (void *) n, (void *) l, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) t, (void *) ldt, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->stplqt2.timings[0] += (helpTimeStop - helpTime);
    data->stplqt2.calls[0]++;
    
    return;
}


/* Wrapper for stpmlqt */
void hook_stpmlqt(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *l, blasint *mb, float *v, blasint *ldv, float *t, blasint *ldt, float *a, blasint *lda, float *b, blasint *ldb, float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stpmlqt((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) l, (void *) mb, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->stpmlqt.timings[0] += (helpTimeStop - helpTime);
    data->stpmlqt.calls[0]++;
    
    return;
}


/* Wrapper for stpmqrt */
void hook_stpmqrt(char *side, char *trans, blasint *m, blasint *n, blasint *k, blasint *l, blasint *nb, float *v, blasint *ldv, float *t, blasint *ldt, float *a, blasint *lda, float *b, blasint *ldb, float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stpmqrt((void *) side, (void *) trans, (void *) m, (void *) n, (void *) k, (void *) l, (void *) nb, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) work, (void *) info, side_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->stpmqrt.timings[0] += (helpTimeStop - helpTime);
    data->stpmqrt.calls[0]++;
    
    return;
}


/* Wrapper for stpqrt */
void hook_stpqrt(blasint *m, blasint *n, blasint *l, blasint *nb, float *a, blasint *lda, float *b, blasint *ldb, float *t, blasint *ldt, float *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stpqrt((void *) m, (void *) n, (void *) l, (void *) nb, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) t, (void *) ldt, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->stpqrt.timings[0] += (helpTimeStop - helpTime);
    data->stpqrt.calls[0]++;
    
    return;
}


/* Wrapper for stpqrt2 */
void hook_stpqrt2(blasint *m, blasint *n, blasint *l, float *a, blasint *lda, float *b, blasint *ldb, float *t, blasint *ldt, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stpqrt2((void *) m, (void *) n, (void *) l, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) t, (void *) ldt, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->stpqrt2.timings[0] += (helpTimeStop - helpTime);
    data->stpqrt2.calls[0]++;
    
    return;
}


/* Wrapper for stprfb */
void hook_stprfb(char *side, char *trans, char *direct, char *storev, blasint *m, blasint *n, blasint *k, blasint *l, float *v, blasint *ldv, float *t, blasint *ldt, float *a, blasint *lda, float *b, blasint *ldb, float *work, blasint *ldwork, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t direct_len, flexiblas_fortran_charlen_t storev_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stprfb((void *) side, (void *) trans, (void *) direct, (void *) storev, (void *) m, (void *) n, (void *) k, (void *) l, (void *) v, (void *) ldv, (void *) t, (void *) ldt, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) work, (void *) ldwork, side_len, trans_len, direct_len, storev_len);

    helpTimeStop = flexiblas_wtime();

    data->stprfb.timings[0] += (helpTimeStop - helpTime);
    data->stprfb.calls[0]++;
    
    return;
}


/* Wrapper for stprfs */
void hook_stprfs(char *uplo, char *trans, char *diag, blasint *n, blasint *nrhs, float *ap, float *b, blasint *ldb, float *x, blasint *ldx, float *ferr, float *berr, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stprfs((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) nrhs, (void *) ap, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->stprfs.timings[0] += (helpTimeStop - helpTime);
    data->stprfs.calls[0]++;
    
    return;
}


/* Wrapper for stptri */
void hook_stptri(char *uplo, char *diag, blasint *n, float *ap, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stptri((void *) uplo, (void *) diag, (void *) n, (void *) ap, (void *) info, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->stptri.timings[0] += (helpTimeStop - helpTime);
    data->stptri.calls[0]++;
    
    return;
}


/* Wrapper for stptrs */
void hook_stptrs(char *uplo, char *trans, char *diag, blasint *n, blasint *nrhs, float *ap, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stptrs((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) nrhs, (void *) ap, (void *) b, (void *) ldb, (void *) info, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->stptrs.timings[0] += (helpTimeStop - helpTime);
    data->stptrs.calls[0]++;
    
    return;
}


/* Wrapper for stpttf */
void hook_stpttf(char *transr, char *uplo, blasint *n, float *ap, float *arf, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stpttf((void *) transr, (void *) uplo, (void *) n, (void *) ap, (void *) arf, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->stpttf.timings[0] += (helpTimeStop - helpTime);
    data->stpttf.calls[0]++;
    
    return;
}


/* Wrapper for stpttr */
void hook_stpttr(char *uplo, blasint *n, float *ap, float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stpttr((void *) uplo, (void *) n, (void *) ap, (void *) a, (void *) lda, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->stpttr.timings[0] += (helpTimeStop - helpTime);
    data->stpttr.calls[0]++;
    
    return;
}


/* Wrapper for strcon */
void hook_strcon(char *norm, char *uplo, char *diag, blasint *n, float *a, blasint *lda, float *rcond, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t norm_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_strcon((void *) norm, (void *) uplo, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) rcond, (void *) work, (void *) iwork, (void *) info, norm_len, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->strcon.timings[0] += (helpTimeStop - helpTime);
    data->strcon.calls[0]++;
    
    return;
}


/* Wrapper for strevc */
void hook_strevc(char *side, char *howmny, blaslogical *select, blasint *n, float *t, blasint *ldt, float *vl, blasint *ldvl, float *vr, blasint *ldvr, blasint *mm, blasint *m, float *work, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t howmny_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_strevc((void *) side, (void *) howmny, (void *) select, (void *) n, (void *) t, (void *) ldt, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) mm, (void *) m, (void *) work, (void *) info, side_len, howmny_len);

    helpTimeStop = flexiblas_wtime();

    data->strevc.timings[0] += (helpTimeStop - helpTime);
    data->strevc.calls[0]++;
    
    return;
}


/* Wrapper for strevc3 */
void hook_strevc3(char *side, char *howmny, blaslogical *select, blasint *n, float *t, blasint *ldt, float *vl, blasint *ldvl, float *vr, blasint *ldvr, blasint *mm, blasint *m, float *work, blasint *lwork, blasint *info, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t howmny_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_strevc3((void *) side, (void *) howmny, (void *) select, (void *) n, (void *) t, (void *) ldt, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) mm, (void *) m, (void *) work, (void *) lwork, (void *) info, side_len, howmny_len);

    helpTimeStop = flexiblas_wtime();

    data->strevc3.timings[0] += (helpTimeStop - helpTime);
    data->strevc3.calls[0]++;
    
    return;
}


/* Wrapper for strexc */
void hook_strexc(char *compq, blasint *n, float *t, blasint *ldt, float *q, blasint *ldq, blasint *ifst, blasint *ilst, float *work, blasint *info, flexiblas_fortran_charlen_t compq_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_strexc((void *) compq, (void *) n, (void *) t, (void *) ldt, (void *) q, (void *) ldq, (void *) ifst, (void *) ilst, (void *) work, (void *) info, compq_len);

    helpTimeStop = flexiblas_wtime();

    data->strexc.timings[0] += (helpTimeStop - helpTime);
    data->strexc.calls[0]++;
    
    return;
}


/* Wrapper for strrfs */
void hook_strrfs(char *uplo, char *trans, char *diag, blasint *n, blasint *nrhs, float *a, blasint *lda, float *b, blasint *ldb, float *x, blasint *ldx, float *ferr, float *berr, float *work, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_strrfs((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) iwork, (void *) info, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->strrfs.timings[0] += (helpTimeStop - helpTime);
    data->strrfs.calls[0]++;
    
    return;
}


/* Wrapper for strsen */
void hook_strsen(char *job, char *compq, blaslogical *select, blasint *n, float *t, blasint *ldt, float *q, blasint *ldq, float *wr, float *wi, blasint *m, float *s, float *sep, float *work, blasint *lwork, blasint *iwork, blasint *liwork, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t compq_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_strsen((void *) job, (void *) compq, (void *) select, (void *) n, (void *) t, (void *) ldt, (void *) q, (void *) ldq, (void *) wr, (void *) wi, (void *) m, (void *) s, (void *) sep, (void *) work, (void *) lwork, (void *) iwork, (void *) liwork, (void *) info, job_len, compq_len);

    helpTimeStop = flexiblas_wtime();

    data->strsen.timings[0] += (helpTimeStop - helpTime);
    data->strsen.calls[0]++;
    
    return;
}


/* Wrapper for strsna */
void hook_strsna(char *job, char *howmny, blaslogical *select, blasint *n, float *t, blasint *ldt, float *vl, blasint *ldvl, float *vr, blasint *ldvr, float *s, float *sep, blasint *mm, blasint *m, float *work, blasint *ldwork, blasint *iwork, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t howmny_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_strsna((void *) job, (void *) howmny, (void *) select, (void *) n, (void *) t, (void *) ldt, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) s, (void *) sep, (void *) mm, (void *) m, (void *) work, (void *) ldwork, (void *) iwork, (void *) info, job_len, howmny_len);

    helpTimeStop = flexiblas_wtime();

    data->strsna.timings[0] += (helpTimeStop - helpTime);
    data->strsna.calls[0]++;
    
    return;
}


/* Wrapper for strsyl */
void hook_strsyl(char *trana, char *tranb, blasint *isgn, blasint *m, blasint *n, float *a, blasint *lda, float *b, blasint *ldb, float *c, blasint *ldc, float *scale, blasint *info, flexiblas_fortran_charlen_t trana_len, flexiblas_fortran_charlen_t tranb_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_strsyl((void *) trana, (void *) tranb, (void *) isgn, (void *) m, (void *) n, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) c, (void *) ldc, (void *) scale, (void *) info, trana_len, tranb_len);

    helpTimeStop = flexiblas_wtime();

    data->strsyl.timings[0] += (helpTimeStop - helpTime);
    data->strsyl.calls[0]++;
    
    return;
}


/* Wrapper for strti2 */
void hook_strti2(char *uplo, char *diag, blasint *n, float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_strti2((void *) uplo, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->strti2.timings[0] += (helpTimeStop - helpTime);
    data->strti2.calls[0]++;
    
    return;
}


/* Wrapper for strtri */
void hook_strtri(char *uplo, char *diag, blasint *n, float *a, blasint *lda, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_strtri((void *) uplo, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) info, uplo_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->strtri.timings[0] += (helpTimeStop - helpTime);
    data->strtri.calls[0]++;
    
    return;
}


/* Wrapper for strtrs */
void hook_strtrs(char *uplo, char *trans, char *diag, blasint *n, blasint *nrhs, float *a, blasint *lda, float *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_strtrs((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) info, uplo_len, trans_len, diag_len);

    helpTimeStop = flexiblas_wtime();

    data->strtrs.timings[0] += (helpTimeStop - helpTime);
    data->strtrs.calls[0]++;
    
    return;
}


/* Wrapper for strttf */
void hook_strttf(char *transr, char *uplo, blasint *n, float *a, blasint *lda, float *arf, blasint *info, flexiblas_fortran_charlen_t transr_len, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_strttf((void *) transr, (void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) arf, (void *) info, transr_len, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->strttf.timings[0] += (helpTimeStop - helpTime);
    data->strttf.calls[0]++;
    
    return;
}


/* Wrapper for strttp */
void hook_strttp(char *uplo, blasint *n, float *a, blasint *lda, float *ap, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_strttp((void *) uplo, (void *) n, (void *) a, (void *) lda, (void *) ap, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->strttp.timings[0] += (helpTimeStop - helpTime);
    data->strttp.calls[0]++;
    
    return;
}


/* Wrapper for stzrzf */
void hook_stzrzf(blasint *m, blasint *n, float *a, blasint *lda, float *tau, float *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_stzrzf((void *) m, (void *) n, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->stzrzf.timings[0] += (helpTimeStop - helpTime);
    data->stzrzf.calls[0]++;
    
    return;
}


/* Wrapper for zbbcsd */
void hook_zbbcsd(char *jobu1, char *jobu2, char *jobv1t, char *jobv2t, char *trans, blasint *m, blasint *p, blasint *q, double *theta, double *phi, lapack_complex_double *u1, blasint *ldu1, lapack_complex_double *u2, blasint *ldu2, lapack_complex_double *v1t, blasint *ldv1t, lapack_complex_double *v2t, blasint *ldv2t, double *b11d, double *b11e, double *b12d, double *b12e, double *b21d, double *b21e, double *b22d, double *b22e, double *rwork, blasint *lrwork, blasint *info, flexiblas_fortran_charlen_t jobu1_len, flexiblas_fortran_charlen_t jobu2_len, flexiblas_fortran_charlen_t jobv1t_len, flexiblas_fortran_charlen_t jobv2t_len, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zbbcsd((void *) jobu1, (void *) jobu2, (void *) jobv1t, (void *) jobv2t, (void *) trans, (void *) m, (void *) p, (void *) q, (void *) theta, (void *) phi, (void *) u1, (void *) ldu1, (void *) u2, (void *) ldu2, (void *) v1t, (void *) ldv1t, (void *) v2t, (void *) ldv2t, (void *) b11d, (void *) b11e, (void *) b12d, (void *) b12e, (void *) b21d, (void *) b21e, (void *) b22d, (void *) b22e, (void *) rwork, (void *) lrwork, (void *) info, jobu1_len, jobu2_len, jobv1t_len, jobv2t_len, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->zbbcsd.timings[0] += (helpTimeStop - helpTime);
    data->zbbcsd.calls[0]++;
    
    return;
}


/* Wrapper for zbdsqr */
void hook_zbdsqr(char *uplo, blasint *n, blasint *ncvt, blasint *nru, blasint *ncc, double *d, double *e, lapack_complex_double *vt, blasint *ldvt, lapack_complex_double *u, blasint *ldu, lapack_complex_double *c, blasint *ldc, double *rwork, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zbdsqr((void *) uplo, (void *) n, (void *) ncvt, (void *) nru, (void *) ncc, (void *) d, (void *) e, (void *) vt, (void *) ldvt, (void *) u, (void *) ldu, (void *) c, (void *) ldc, (void *) rwork, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->zbdsqr.timings[0] += (helpTimeStop - helpTime);
    data->zbdsqr.calls[0]++;
    
    return;
}


/* Wrapper for zcgesv */
void hook_zcgesv(blasint *n, blasint *nrhs, lapack_complex_double *a, blasint *lda, blasint *ipiv, lapack_complex_double *b, blasint *ldb, lapack_complex_double *x, blasint *ldx, lapack_complex_double *work, lapack_complex_float *swork, double *rwork, blasint *iter, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zcgesv((void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) work, (void *) swork, (void *) rwork, (void *) iter, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->zcgesv.timings[0] += (helpTimeStop - helpTime);
    data->zcgesv.calls[0]++;
    
    return;
}


/* Wrapper for zcposv */
void hook_zcposv(char *uplo, blasint *n, blasint *nrhs, lapack_complex_double *a, blasint *lda, lapack_complex_double *b, blasint *ldb, lapack_complex_double *x, blasint *ldx, lapack_complex_double *work, lapack_complex_float *swork, double *rwork, blasint *iter, blasint *info, flexiblas_fortran_charlen_t uplo_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zcposv((void *) uplo, (void *) n, (void *) nrhs, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) work, (void *) swork, (void *) rwork, (void *) iter, (void *) info, uplo_len);

    helpTimeStop = flexiblas_wtime();

    data->zcposv.timings[0] += (helpTimeStop - helpTime);
    data->zcposv.calls[0]++;
    
    return;
}


/* Wrapper for zdrscl */
void hook_zdrscl(blasint *n, double *sa, lapack_complex_double *sx, blasint *incx)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zdrscl((void *) n, (void *) sa, (void *) sx, (void *) incx);

    helpTimeStop = flexiblas_wtime();

    data->zdrscl.timings[0] += (helpTimeStop - helpTime);
    data->zdrscl.calls[0]++;
    
    return;
}


/* Wrapper for zgbbrd */
void hook_zgbbrd(char *vect, blasint *m, blasint *n, blasint *ncc, blasint *kl, blasint *ku, lapack_complex_double *ab, blasint *ldab, double *d, double *e, lapack_complex_double *q, blasint *ldq, lapack_complex_double *pt, blasint *ldpt, lapack_complex_double *c, blasint *ldc, lapack_complex_double *work, double *rwork, blasint *info, flexiblas_fortran_charlen_t vect_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgbbrd((void *) vect, (void *) m, (void *) n, (void *) ncc, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) d, (void *) e, (void *) q, (void *) ldq, (void *) pt, (void *) ldpt, (void *) c, (void *) ldc, (void *) work, (void *) rwork, (void *) info, vect_len);

    helpTimeStop = flexiblas_wtime();

    data->zgbbrd.timings[0] += (helpTimeStop - helpTime);
    data->zgbbrd.calls[0]++;
    
    return;
}


/* Wrapper for zgbcon */
void hook_zgbcon(char *norm, blasint *n, blasint *kl, blasint *ku, lapack_complex_double *ab, blasint *ldab, blasint *ipiv, double *anorm, double *rcond, lapack_complex_double *work, double *rwork, blasint *info, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgbcon((void *) norm, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) ipiv, (void *) anorm, (void *) rcond, (void *) work, (void *) rwork, (void *) info, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->zgbcon.timings[0] += (helpTimeStop - helpTime);
    data->zgbcon.calls[0]++;
    
    return;
}


/* Wrapper for zgbequ */
void hook_zgbequ(blasint *m, blasint *n, blasint *kl, blasint *ku, lapack_complex_double *ab, blasint *ldab, double *r, double *c, double *rowcnd, double *colcnd, double *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgbequ((void *) m, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->zgbequ.timings[0] += (helpTimeStop - helpTime);
    data->zgbequ.calls[0]++;
    
    return;
}


/* Wrapper for zgbequb */
void hook_zgbequb(blasint *m, blasint *n, blasint *kl, blasint *ku, lapack_complex_double *ab, blasint *ldab, double *r, double *c, double *rowcnd, double *colcnd, double *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgbequb((void *) m, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->zgbequb.timings[0] += (helpTimeStop - helpTime);
    data->zgbequb.calls[0]++;
    
    return;
}


/* Wrapper for zgbrfs */
void hook_zgbrfs(char *trans, blasint *n, blasint *kl, blasint *ku, blasint *nrhs, lapack_complex_double *ab, blasint *ldab, lapack_complex_double *afb, blasint *ldafb, blasint *ipiv, lapack_complex_double *b, blasint *ldb, lapack_complex_double *x, blasint *ldx, double *ferr, double *berr, lapack_complex_double *work, double *rwork, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgbrfs((void *) trans, (void *) n, (void *) kl, (void *) ku, (void *) nrhs, (void *) ab, (void *) ldab, (void *) afb, (void *) ldafb, (void *) ipiv, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->zgbrfs.timings[0] += (helpTimeStop - helpTime);
    data->zgbrfs.calls[0]++;
    
    return;
}


/* Wrapper for zgbsv */
void hook_zgbsv(blasint *n, blasint *kl, blasint *ku, blasint *nrhs, lapack_complex_double *ab, blasint *ldab, blasint *ipiv, lapack_complex_double *b, blasint *ldb, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgbsv((void *) n, (void *) kl, (void *) ku, (void *) nrhs, (void *) ab, (void *) ldab, (void *) ipiv, (void *) b, (void *) ldb, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->zgbsv.timings[0] += (helpTimeStop - helpTime);
    data->zgbsv.calls[0]++;
    
    return;
}


/* Wrapper for zgbsvx */
void hook_zgbsvx(char *fact, char *trans, blasint *n, blasint *kl, blasint *ku, blasint *nrhs, lapack_complex_double *ab, blasint *ldab, lapack_complex_double *afb, blasint *ldafb, blasint *ipiv, char *equed, double *r, double *c, lapack_complex_double *b, blasint *ldb, lapack_complex_double *x, blasint *ldx, double *rcond, double *ferr, double *berr, lapack_complex_double *work, double *rwork, blasint *info, flexiblas_fortran_charlen_t fact_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t equed_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgbsvx((void *) fact, (void *) trans, (void *) n, (void *) kl, (void *) ku, (void *) nrhs, (void *) ab, (void *) ldab, (void *) afb, (void *) ldafb, (void *) ipiv, (void *) equed, (void *) r, (void *) c, (void *) b, (void *) ldb, (void *) x, (void *) ldx, (void *) rcond, (void *) ferr, (void *) berr, (void *) work, (void *) rwork, (void *) info, fact_len, trans_len, equed_len);

    helpTimeStop = flexiblas_wtime();

    data->zgbsvx.timings[0] += (helpTimeStop - helpTime);
    data->zgbsvx.calls[0]++;
    
    return;
}


/* Wrapper for zgbtf2 */
void hook_zgbtf2(blasint *m, blasint *n, blasint *kl, blasint *ku, lapack_complex_double *ab, blasint *ldab, blasint *ipiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgbtf2((void *) m, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) ipiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->zgbtf2.timings[0] += (helpTimeStop - helpTime);
    data->zgbtf2.calls[0]++;
    
    return;
}


/* Wrapper for zgbtrf */
void hook_zgbtrf(blasint *m, blasint *n, blasint *kl, blasint *ku, lapack_complex_double *ab, blasint *ldab, blasint *ipiv, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgbtrf((void *) m, (void *) n, (void *) kl, (void *) ku, (void *) ab, (void *) ldab, (void *) ipiv, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->zgbtrf.timings[0] += (helpTimeStop - helpTime);
    data->zgbtrf.calls[0]++;
    
    return;
}


/* Wrapper for zgbtrs */
void hook_zgbtrs(char *trans, blasint *n, blasint *kl, blasint *ku, blasint *nrhs, lapack_complex_double *ab, blasint *ldab, blasint *ipiv, lapack_complex_double *b, blasint *ldb, blasint *info, flexiblas_fortran_charlen_t trans_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgbtrs((void *) trans, (void *) n, (void *) kl, (void *) ku, (void *) nrhs, (void *) ab, (void *) ldab, (void *) ipiv, (void *) b, (void *) ldb, (void *) info, trans_len);

    helpTimeStop = flexiblas_wtime();

    data->zgbtrs.timings[0] += (helpTimeStop - helpTime);
    data->zgbtrs.calls[0]++;
    
    return;
}


/* Wrapper for zgebak */
void hook_zgebak(char *job, char *side, blasint *n, blasint *ilo, blasint *ihi, double *scale, blasint *m, lapack_complex_double *v, blasint *ldv, blasint *info, flexiblas_fortran_charlen_t job_len, flexiblas_fortran_charlen_t side_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgebak((void *) job, (void *) side, (void *) n, (void *) ilo, (void *) ihi, (void *) scale, (void *) m, (void *) v, (void *) ldv, (void *) info, job_len, side_len);

    helpTimeStop = flexiblas_wtime();

    data->zgebak.timings[0] += (helpTimeStop - helpTime);
    data->zgebak.calls[0]++;
    
    return;
}


/* Wrapper for zgebal */
void hook_zgebal(char *job, blasint *n, lapack_complex_double *a, blasint *lda, blasint *ilo, blasint *ihi, double *scale, blasint *info, flexiblas_fortran_charlen_t job_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgebal((void *) job, (void *) n, (void *) a, (void *) lda, (void *) ilo, (void *) ihi, (void *) scale, (void *) info, job_len);

    helpTimeStop = flexiblas_wtime();

    data->zgebal.timings[0] += (helpTimeStop - helpTime);
    data->zgebal.calls[0]++;
    
    return;
}


/* Wrapper for zgebd2 */
void hook_zgebd2(blasint *m, blasint *n, lapack_complex_double *a, blasint *lda, double *d, double *e, lapack_complex_double *tauq, lapack_complex_double *taup, lapack_complex_double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgebd2((void *) m, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) e, (void *) tauq, (void *) taup, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->zgebd2.timings[0] += (helpTimeStop - helpTime);
    data->zgebd2.calls[0]++;
    
    return;
}


/* Wrapper for zgebrd */
void hook_zgebrd(blasint *m, blasint *n, lapack_complex_double *a, blasint *lda, double *d, double *e, lapack_complex_double *tauq, lapack_complex_double *taup, lapack_complex_double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgebrd((void *) m, (void *) n, (void *) a, (void *) lda, (void *) d, (void *) e, (void *) tauq, (void *) taup, (void *) work, (void *) lwork, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->zgebrd.timings[0] += (helpTimeStop - helpTime);
    data->zgebrd.calls[0]++;
    
    return;
}


/* Wrapper for zgecon */
void hook_zgecon(char *norm, blasint *n, lapack_complex_double *a, blasint *lda, double *anorm, double *rcond, lapack_complex_double *work, double *rwork, blasint *info, flexiblas_fortran_charlen_t norm_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgecon((void *) norm, (void *) n, (void *) a, (void *) lda, (void *) anorm, (void *) rcond, (void *) work, (void *) rwork, (void *) info, norm_len);

    helpTimeStop = flexiblas_wtime();

    data->zgecon.timings[0] += (helpTimeStop - helpTime);
    data->zgecon.calls[0]++;
    
    return;
}


/* Wrapper for zgeequ */
void hook_zgeequ(blasint *m, blasint *n, lapack_complex_double *a, blasint *lda, double *r, double *c, double *rowcnd, double *colcnd, double *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgeequ((void *) m, (void *) n, (void *) a, (void *) lda, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->zgeequ.timings[0] += (helpTimeStop - helpTime);
    data->zgeequ.calls[0]++;
    
    return;
}


/* Wrapper for zgeequb */
void hook_zgeequb(blasint *m, blasint *n, lapack_complex_double *a, blasint *lda, double *r, double *c, double *rowcnd, double *colcnd, double *amax, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgeequb((void *) m, (void *) n, (void *) a, (void *) lda, (void *) r, (void *) c, (void *) rowcnd, (void *) colcnd, (void *) amax, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->zgeequb.timings[0] += (helpTimeStop - helpTime);
    data->zgeequb.calls[0]++;
    
    return;
}


/* Wrapper for zgees */
void hook_zgees(char *jobvs, char *sort, blaslogical *select, blasint *n, lapack_complex_double *a, blasint *lda, blasint *sdim, lapack_complex_double *w, lapack_complex_double *vs, blasint *ldvs, lapack_complex_double *work, blasint *lwork, double *rwork, blaslogical *bwork, blasint *info, flexiblas_fortran_charlen_t jobvs_len, flexiblas_fortran_charlen_t sort_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgees((void *) jobvs, (void *) sort, (void *) select, (void *) n, (void *) a, (void *) lda, (void *) sdim, (void *) w, (void *) vs, (void *) ldvs, (void *) work, (void *) lwork, (void *) rwork, (void *) bwork, (void *) info, jobvs_len, sort_len);

    helpTimeStop = flexiblas_wtime();

    data->zgees.timings[0] += (helpTimeStop - helpTime);
    data->zgees.calls[0]++;
    
    return;
}


/* Wrapper for zgeesx */
void hook_zgeesx(char *jobvs, char *sort, blaslogical *select, char *sense, blasint *n, lapack_complex_double *a, blasint *lda, blasint *sdim, lapack_complex_double *w, lapack_complex_double *vs, blasint *ldvs, double *rconde, double *rcondv, lapack_complex_double *work, blasint *lwork, double *rwork, blaslogical *bwork, blasint *info, flexiblas_fortran_charlen_t jobvs_len, flexiblas_fortran_charlen_t sort_len, flexiblas_fortran_charlen_t sense_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgeesx((void *) jobvs, (void *) sort, (void *) select, (void *) sense, (void *) n, (void *) a, (void *) lda, (void *) sdim, (void *) w, (void *) vs, (void *) ldvs, (void *) rconde, (void *) rcondv, (void *) work, (void *) lwork, (void *) rwork, (void *) bwork, (void *) info, jobvs_len, sort_len, sense_len);

    helpTimeStop = flexiblas_wtime();

    data->zgeesx.timings[0] += (helpTimeStop - helpTime);
    data->zgeesx.calls[0]++;
    
    return;
}


/* Wrapper for zgeev */
void hook_zgeev(char *jobvl, char *jobvr, blasint *n, lapack_complex_double *a, blasint *lda, lapack_complex_double *w, lapack_complex_double *vl, blasint *ldvl, lapack_complex_double *vr, blasint *ldvr, lapack_complex_double *work, blasint *lwork, double *rwork, blasint *info, flexiblas_fortran_charlen_t jobvl_len, flexiblas_fortran_charlen_t jobvr_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgeev((void *) jobvl, (void *) jobvr, (void *) n, (void *) a, (void *) lda, (void *) w, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) work, (void *) lwork, (void *) rwork, (void *) info, jobvl_len, jobvr_len);

    helpTimeStop = flexiblas_wtime();

    data->zgeev.timings[0] += (helpTimeStop - helpTime);
    data->zgeev.calls[0]++;
    
    return;
}


/* Wrapper for zgeevx */
void hook_zgeevx(char *balanc, char *jobvl, char *jobvr, char *sense, blasint *n, lapack_complex_double *a, blasint *lda, lapack_complex_double *w, lapack_complex_double *vl, blasint *ldvl, lapack_complex_double *vr, blasint *ldvr, blasint *ilo, blasint *ihi, double *scale, double *abnrm, double *rconde, double *rcondv, lapack_complex_double *work, blasint *lwork, double *rwork, blasint *info, flexiblas_fortran_charlen_t balanc_len, flexiblas_fortran_charlen_t jobvl_len, flexiblas_fortran_charlen_t jobvr_len, flexiblas_fortran_charlen_t sense_len)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgeevx((void *) balanc, (void *) jobvl, (void *) jobvr, (void *) sense, (void *) n, (void *) a, (void *) lda, (void *) w, (void *) vl, (void *) ldvl, (void *) vr, (void *) ldvr, (void *) ilo, (void *) ihi, (void *) scale, (void *) abnrm, (void *) rconde, (void *) rcondv, (void *) work, (void *) lwork, (void *) rwork, (void *) info, balanc_len, jobvl_len, jobvr_len, sense_len);

    helpTimeStop = flexiblas_wtime();

    data->zgeevx.timings[0] += (helpTimeStop - helpTime);
    data->zgeevx.calls[0]++;
    
    return;
}


/* Wrapper for zgehd2 */
void hook_zgehd2(blasint *n, blasint *ilo, blasint *ihi, lapack_complex_double *a, blasint *lda, lapack_complex_double *tau, lapack_complex_double *work, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgehd2((void *) n, (void *) ilo, (void *) ihi, (void *) a, (void *) lda, (void *) tau, (void *) work, (void *) info);

    helpTimeStop = flexiblas_wtime();

    data->zgehd2.timings[0] += (helpTimeStop - helpTime);
    data->zgehd2.calls[0]++;
    
    return;
}


/* Wrapper for zgehrd */
void hook_zgehrd(blasint *n, blasint *ilo, blasint *ihi, lapack_complex_double *a, blasint *lda, lapack_complex_double *tau, lapack_complex_double *work, blasint *lwork, blasint *info)
{
    double helpTime;
    double helpTimeStop;

    helpTime = flexiblas_wtime();
    
    flexiblas_chain_zgehrd((void *) n, (void *) ilo, (void 