/*++

Copyright (C) 2018 Autodesk Inc. (Original Author)

All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

--*/

//////////////////////////////////////////////////////////////////////////////////////////////////////
// buildbindingpascal.go
// functions to generate dynamic Pascal-bindings of a library's API in form of explicitly loaded
// function handles.
//////////////////////////////////////////////////////////////////////////////////////////////////////

package main

import (
	"fmt"
	"log"
	"path"
	"strings"
)

// BuildBindingPascalDynamic builds dynamic Pascal bindings of a library's API in form of explicitly loaded
// function handles.
func BuildBindingPascalDynamic(component ComponentDefinition, outputFolder string, outputFolderExample string, indentString string) error {
	forceRecreation := false

	namespace := component.NameSpace
	libraryname := component.LibraryName
	baseName := component.BaseName

	DynamicPascalImpl := path.Join(outputFolder, "Unit_"+namespace+".pas")
	log.Printf("Creating \"%s\"", DynamicPascalImpl)
	dynpascalfile, err := CreateLanguageFile(DynamicPascalImpl, indentString)
	if err != nil {
		return err
	}

	dynpascalfile.Writeln("{$IFDEF FPC}{$MODE DELPHI}{$ENDIF}")
	dynpascalfile.WritePascalLicenseHeader(component,
		fmt.Sprintf("This is an autogenerated Pascal Header file in order to allow an easy\n use of %s", libraryname),
		true)

	err = buildDynamicPascalImplementation(component, dynpascalfile, namespace, baseName)
	if err != nil {
		return err
	}

	if len(outputFolderExample) > 0 {
		DynamicPascalExample := path.Join(outputFolderExample, namespace+"_Example.lpr")
		if forceRecreation || !FileExists(DynamicPascalExample) {
			log.Printf("Creating \"%s\"", DynamicPascalExample)
			dynpascalexamplefile, err := CreateLanguageFile(DynamicPascalExample, indentString)
			dynpascalexamplefile.WritePascalLicenseHeader(component,
				fmt.Sprintf("This is an autogenerated Pascal application that demonstrates the\n usage of the Pascal bindings of %s", libraryname),
				true)
			err = buildDynamicPascalExample(dynpascalexamplefile, component, outputFolder)
			if err != nil {
				return err
			}
		} else {
			log.Printf("Omitting recreation of Pascal example \"%s\"", DynamicPascalExample)
		}

		DynamicPascalExampleLPI := path.Join(outputFolderExample, namespace+"_Example.lpi")
		if forceRecreation || !FileExists(DynamicPascalExampleLPI) {
			log.Printf("Creating \"%s\"", DynamicPascalExampleLPI)
			dynpascalexampleLPIfile, err := CreateLanguageFile(DynamicPascalExampleLPI, indentString)
			err = buildDynamicPascalExampleLPI(component, dynpascalexampleLPIfile, outputFolder)
			if err != nil {
				return err
			}
		} else {
			log.Printf("Omitting recreation of Pascal example \"%s\"", DynamicPascalExampleLPI)
		}
	}

	return nil
}

func writeEnumConversionInterface(component ComponentDefinition, w LanguageWriter, NameSpace string) error {

	if len(component.Enums) > 0 {
		w.Writeln("(*************************************************************************************************************************")
		w.Writeln(" Enum conversion")
		w.Writeln("**************************************************************************************************************************)")
		w.Writeln("")

		for i := 0; i < len(component.Enums); i++ {
			enum := component.Enums[i]
			w.Writeln("  function convert%sToConst(const AValue: T%s%s): Integer;", enum.Name, NameSpace, enum.Name)
			w.Writeln("  function convertConstTo%s(const AValue: Integer): T%s%s;", enum.Name, NameSpace, enum.Name)
		}

		w.Writeln("")
	}

	return nil
}

func writeEnumConversionImplementation(component ComponentDefinition, w LanguageWriter, NameSpace string) error {

	if len(component.Enums) > 0 {
		w.Writeln("(*************************************************************************************************************************")
		w.Writeln(" Enum conversion")
		w.Writeln("**************************************************************************************************************************)")
		w.Writeln("")

		for i := 0; i < len(component.Enums); i++ {
			enum := component.Enums[i]
			w.Writeln("  function convert%sToConst(const AValue: T%s%s): Integer;", enum.Name, NameSpace, enum.Name)
			w.Writeln("  begin")
			w.Writeln("    case AValue of")

			for j := 0; j < len(enum.Options); j++ {
				option := enum.Options[j]
				w.Writeln("      e%s%s: Result := %d;", enum.Name, option.Name, option.Value)
			}

			w.Writeln("      else ")
			w.Writeln("        raise E%sException.CreateCustomMessage(%s_ERROR_INVALIDPARAM, 'invalid enum value');", NameSpace, strings.ToUpper(NameSpace))
			w.Writeln("    end;")
			w.Writeln("  end;")
			w.Writeln("  ")
			w.Writeln("  function convertConstTo%s(const AValue: Integer): T%s%s;", enum.Name, NameSpace, enum.Name)
			w.Writeln("  begin")
			w.Writeln("    case AValue of")

			for j := 0; j < len(enum.Options); j++ {
				option := enum.Options[j]
				w.Writeln("      %d: Result := e%s%s;", option.Value, enum.Name, option.Name)
			}

			w.Writeln("      else ")
			w.Writeln("        raise E%sException.CreateCustomMessage(%s_ERROR_INVALIDPARAM, 'invalid enum constant');", NameSpace, strings.ToUpper(NameSpace))
			w.Writeln("    end;")
			w.Writeln("  end;")
			w.Writeln("  ")
			w.Writeln("  ")
		}

		w.Writeln("")
	}

	return nil
}

func buildDynamicPascalImplementation(component ComponentDefinition, w LanguageWriter, NameSpace string, BaseName string) error {

	w.Writeln("unit Unit_%s;", NameSpace)
	w.Writeln("")
	w.Writeln("interface")
	w.Writeln("")
	w.Writeln("uses")
	if len(component.ImportedComponentDefinitions) > 0 {
		for _, subComponent := range component.ImportedComponentDefinitions {
			w.Writeln("  Unit_%s,", subComponent.NameSpace)
		}
	}
	w.Writeln("  {$IFDEF WINDOWS}")
	w.Writeln("    Windows,")
	w.Writeln("  {$ELSE}")
	w.Writeln("    dynlibs,")
	w.Writeln("  {$ENDIF}")
	w.Writeln("  Types,")
	w.Writeln("  Classes,")
	w.Writeln("  SysUtils;")
	w.Writeln("")

	err := writePascalBaseTypeDefinitions(component, w, NameSpace, BaseName)
	if err != nil {
		return err
	}

	w.Writeln("(*************************************************************************************************************************")
	w.Writeln(" Declaration of handle classes ")
	w.Writeln("**************************************************************************************************************************)")
	w.Writeln("")
	w.Writeln("type")
	w.Writeln("  T%sWrapper = class;", NameSpace)

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		w.Writeln("  T%s%s = class;", NameSpace, class.ClassName)
	}
	w.Writeln("")

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]

		w.Writeln("")
		w.Writeln("(*************************************************************************************************************************")
		w.Writeln(" Function type definitions for %s", class.ClassName)
		w.Writeln("**************************************************************************************************************************)")
		w.Writeln("")

		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			err := writePascalFunctionType(method, w, NameSpace, class.ClassName, false, "  ")
			if err != nil {
				return err
			}
		}

	}

	w.Writeln("(*************************************************************************************************************************")
	w.Writeln(" Global function definitions ")
	w.Writeln("**************************************************************************************************************************)")
	w.Writeln("")

	global := component.Global
	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]

		err := writePascalFunctionType(method, w, NameSpace, "Wrapper", true, "  ")
		if err != nil {
			return err
		}
	}

	w.Writeln("")
	w.Writeln("(*************************************************************************************************************************")
	w.Writeln(" Helper function pointer definitions ")
	w.Writeln("**************************************************************************************************************************)")
	w.Writeln("T%sSymbolLookupMethod = function(const pSymbolName: PAnsiChar; out pValue: Pointer): T%sResult; cdecl;", NameSpace, NameSpace)
	w.Writeln("")

	w.Writeln("(*************************************************************************************************************************")
	w.Writeln(" Exception definition")
	w.Writeln("**************************************************************************************************************************)")
	w.Writeln("")
	w.Writeln("  E%sException = class(Exception)", NameSpace)
	w.Writeln("  private")
	w.Writeln("    FErrorCode: T%sResult;", NameSpace)
	w.Writeln("    FCustomMessage: String;")
	w.Writeln("  public")
	w.Writeln("    property ErrorCode: T%sResult read FErrorCode;", NameSpace)
	w.Writeln("    property CustomMessage: String read FCustomMessage;")
	w.Writeln("    constructor Create(AErrorCode: T%sResult; AMessage: String);", NameSpace)
	w.Writeln("    constructor CreateCustomMessage(AErrorCode: T%sResult; AMessage: String);", NameSpace)
	w.Writeln("  end;")
	w.Writeln("")

	pascalBaseClassName := "T" + NameSpace + component.Global.BaseClassName
	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]

		w.Writeln("")
		w.Writeln("(*************************************************************************************************************************")
		w.Writeln(" Class definition for %s", class.ClassName)
		w.Writeln("**************************************************************************************************************************)")
		w.Writeln("")

		pascalParentClassName := ""
		if !component.isBaseClass(class) {
			if class.ParentClass == "" {
				pascalParentClassName = pascalBaseClassName
			} else {
				pascalParentClassName = "T" + NameSpace + class.ParentClass
			}
		}

		parentClassName := class.ParentClass
		if parentClassName == "" {
			parentClassName = "BaseClass"
		}

		if component.isBaseClass(class) {
			w.Writeln(" %s = class(TObject)", pascalBaseClassName)
			w.Writeln("  private")
			w.Writeln("    FWrapper: T%sWrapper;", NameSpace)
			w.Writeln("    FHandle: T%sHandle;", NameSpace)
		} else {
			w.Writeln("  T%s%s = class(%s)", NameSpace, class.ClassName, pascalParentClassName)
		}
		w.Writeln("  public")
		w.Writeln("    constructor Create(AWrapper: T%sWrapper; AHandle: T%sHandle);", NameSpace, NameSpace)
		w.Writeln("    destructor Destroy; override;")
		if component.isBaseClass(class) {
			w.Writeln("    property TheHandle: T%sHandle read FHandle;", NameSpace)
		}

		w.AddIndentationLevel(2)
		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			err := writePascalClassMethodDefinition(method, w, NameSpace, class.ClassName, false, false)
			if err != nil {
				return err
			}
		}
		w.AddIndentationLevel(-2)

		w.Writeln("  end;")
		w.Writeln("")
	}

	w.Writeln("(*************************************************************************************************************************")
	w.Writeln(" Wrapper definition")
	w.Writeln("**************************************************************************************************************************)")
	w.Writeln("")
	w.Writeln("  T%sWrapper = class(TObject)", NameSpace)
	w.Writeln("  private")
	w.Writeln("    FModule: HMODULE;")
	if len(component.ImportedComponentDefinitions) > 0 {
		w.Writeln("    // Imported Components")
		for _, subComponent := range component.ImportedComponentDefinitions {
			w.Writeln("    F%sWrapper: T%sWrapper;", subComponent.NameSpace, subComponent.NameSpace)
		}
		w.Writeln("    ")
	}

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]

		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]

			w.Writeln("    F%s%s_%sFunc: T%s%s_%sFunc;", NameSpace, class.ClassName, method.MethodName, NameSpace, class.ClassName, method.MethodName)
		}
	}

	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]

		w.Writeln("    F%s%sFunc: T%s%sFunc;", NameSpace, method.MethodName, NameSpace, method.MethodName)
	}

	w.Writeln("")
	w.Writeln("    {$IFDEF MSWINDOWS}")
	w.Writeln("    function LoadFunction(AFunctionName: AnsiString; FailIfNotExistent: Boolean = True): FARPROC;")
	w.Writeln("    {$ELSE}")
	w.Writeln("    function LoadFunction(AFunctionName: AnsiString; FailIfNotExistent: Boolean = True): Pointer;")
	w.Writeln("    {$ENDIF MSWINDOWS}")
	w.Writeln("")
	w.Writeln("    procedure checkBinaryVersion();")
	w.Writeln("")
	if len(component.ImportedComponentDefinitions) > 0 {
		w.Writeln("    // Imported Components")
		for _, subComponent := range component.ImportedComponentDefinitions {
			w.Writeln("    function Get%sWrapper(): T%sWrapper;", subComponent.NameSpace, subComponent.NameSpace)
		}
		w.Writeln("    ")
	}

	w.Writeln("  protected")

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]

		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]

			w.Writeln("    property %s%s_%sFunc: T%s%s_%sFunc read F%s%s_%sFunc;", NameSpace, class.ClassName, method.MethodName, NameSpace, class.ClassName, method.MethodName, NameSpace, class.ClassName, method.MethodName)
		}

	}

	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]
		w.Writeln("    property %s%sFunc: T%s%sFunc read F%s%sFunc;", NameSpace, method.MethodName, NameSpace, method.MethodName, NameSpace, method.MethodName)
	}
	w.Writeln("    procedure CheckError(AInstance: %s; AErrorCode: T%sResult);", pascalBaseClassName, NameSpace)

	w.Writeln("  public")

	w.Writeln("    constructor Create(ADLLName: String);")
	w.Writeln("    constructor CreateFromSymbolLookupMethod(ALookupMethod: T%sSymbolLookupMethod);", NameSpace)
	w.Writeln("    destructor Destroy; override;")

	if len(component.ImportedComponentDefinitions) > 0 {
		w.Writeln("    // Imported Components")
		for _, subComponent := range component.ImportedComponentDefinitions {
			w.Writeln("    property %sWrapper: T%sWrapper read Get%sWrapper;", subComponent.NameSpace, subComponent.NameSpace, subComponent.NameSpace)
		}
		w.Writeln("    ")
	}

	w.AddIndentationLevel(2)
	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]

		err = writePascalClassMethodDefinition(method, w, NameSpace, "Wrapper", true, false)
		if err != nil {
			return err
		}

	}
	w.AddIndentationLevel(-2)
	w.Writeln("  end;")
	w.Writeln("")

	writeEnumConversionInterface(component, w, NameSpace)

	w.Writeln("")
	w.Writeln("implementation")
	w.Writeln("")

	writeEnumConversionImplementation(component, w, NameSpace)

	w.Writeln("")
	w.Writeln("(*************************************************************************************************************************")
	w.Writeln(" Exception implementation")
	w.Writeln("**************************************************************************************************************************)")
	w.Writeln("")
	w.Writeln("  constructor E%sException.Create(AErrorCode: T%sResult; AMessage: String);", NameSpace, NameSpace)
	w.Writeln("  var")
	w.Writeln("    ADescription: String;")
	w.Writeln("  begin")
	w.Writeln("    FErrorCode := AErrorCode;")
	w.Writeln("    case FErrorCode of")

	for _, error := range component.Errors.Errors {
		w.Writeln("      %s_ERROR_%s: ADescription := '%s';", strings.ToUpper(NameSpace), error.Name, error.Description)
	}

	w.Writeln("      else")
	w.Writeln("        ADescription := 'unknown';")
	w.Writeln("    end;")

	w.Writeln("")
	w.Writeln("    inherited Create(Format('%s Error - %%s (#%%d, %%s)', [ ADescription, AErrorCode, AMessage ]));", component.LibraryName)
	w.Writeln("  end;")
	w.Writeln("")
	w.Writeln("  constructor E%sException.CreateCustomMessage(AErrorCode: T%sResult; AMessage: String);", NameSpace, NameSpace)
	w.Writeln("  begin")
	w.Writeln("    FCustomMessage := AMessage;")
	w.Writeln("    FErrorCode := AErrorCode;")
	w.Writeln("    inherited Create(Format('%%s (%%d)', [FCustomMessage, AErrorCode]));")
	w.Writeln("  end;")
	w.Writeln("")

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]

		w.Writeln("(*************************************************************************************************************************")
		w.Writeln(" Class implementation for %s", class.ClassName)
		w.Writeln("**************************************************************************************************************************)")
		w.Writeln("")

		if component.isBaseClass(class) {
			w.Writeln("  constructor %s.Create(AWrapper: T%sWrapper; AHandle: T%sHandle);", pascalBaseClassName, NameSpace, NameSpace)
			w.Writeln("  begin")
			w.Writeln("    if not Assigned(AWrapper) then")
			w.Writeln("      raise E%sException.Create(%s_ERROR_INVALIDPARAM, '');", NameSpace, strings.ToUpper(NameSpace))
			w.Writeln("    if not Assigned(AHandle) then")
			w.Writeln("      raise E%sException.Create(%s_ERROR_INVALIDPARAM, '');", NameSpace, strings.ToUpper(NameSpace))
			w.Writeln("")
			w.Writeln("    inherited Create();")
			w.Writeln("    FWrapper := AWrapper;")
			w.Writeln("    FHandle := AHandle;")
			w.Writeln("  end;")
		} else {
			w.Writeln("  constructor T%s%s.Create(AWrapper: T%sWrapper; AHandle: T%sHandle);", NameSpace, class.ClassName, NameSpace, NameSpace)
			w.Writeln("  begin")
			w.Writeln("    inherited Create(AWrapper, AHandle);")
			w.Writeln("  end;")
		}

		w.Writeln("")
		w.Writeln("  destructor T%s%s.Destroy;", NameSpace, class.ClassName)
		w.Writeln("  begin")
		if component.isBaseClass(class) {
			w.Writeln("    FWrapper.%s(self);", component.Global.ReleaseMethod)
		}
		w.Writeln("    inherited;")
		w.Writeln("  end;")

		w.Writeln("")

		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]

			err := writePascalClassMethodImplementation(method, w, NameSpace, class.ClassName, make([]string, 0), make([]string, 0), false)
			if err != nil {
				return err
			}
		}

	}

	w.Writeln("(*************************************************************************************************************************")
	w.Writeln(" Wrapper class implementation")
	w.Writeln("**************************************************************************************************************************)")
	w.Writeln("")
	w.Writeln("  constructor T%sWrapper.Create(ADLLName: String);", NameSpace)
	w.Writeln("  {$IFDEF MSWINDOWS}")
	w.Writeln("  var")
	w.Writeln("    AWideString: WideString;")
	w.Writeln("  {$ENDIF MSWINDOWS}")
	w.Writeln("  begin")
	w.Writeln("    inherited Create;")
	w.Writeln("    ")
	for _, subComponent := range component.ImportedComponentDefinitions {
		w.Writeln("    F%sWrapper := nil;", subComponent.NameSpace)
	}
	w.Writeln("    ")
	w.Writeln("    {$IFDEF MSWINDOWS}")
	w.Writeln("      AWideString := UTF8Decode(ADLLName + #0);")
	w.Writeln("      FModule := LoadLibraryW(PWideChar(AWideString));")
	w.Writeln("    {$ELSE}")
	w.Writeln("      FModule := dynlibs.LoadLibrary(ADLLName);")
	w.Writeln("    {$ENDIF MSWINDOWS}")
	w.Writeln("    if FModule = 0 then")
	w.Writeln("      raise E%sException.Create(%s_ERROR_COULDNOTLOADLIBRARY, '');", NameSpace, strings.ToUpper(NameSpace))
	w.Writeln("")

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			w.Writeln("    F%s%s_%sFunc := LoadFunction('%s');", NameSpace, class.ClassName, method.MethodName, GetCExportName(NameSpace, class.ClassName, method, false))
		}
	}

	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]
		w.Writeln("    F%s%sFunc := LoadFunction('%s');", NameSpace, method.MethodName, GetCExportName(NameSpace, "Wrapper", method, true))
	}
	w.Writeln("    ")
	w.Writeln("    checkBinaryVersion();")
	w.Writeln("  end;")
	w.Writeln("")

	w.Writeln("  constructor T%sWrapper.CreateFromSymbolLookupMethod(ALookupMethod: T%sSymbolLookupMethod);", NameSpace, NameSpace)
	w.Writeln("  var")
	w.Writeln("    AResult : T%sResult;", NameSpace)
	w.Writeln("  begin")
	w.Writeln("    inherited Create;")
	w.Writeln("    ")
	for _, subComponent := range component.ImportedComponentDefinitions {
		w.Writeln("    F%sWrapper := nil;", subComponent.NameSpace)
	}
	w.Writeln("    ")

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			w.Writeln("    AResult := ALookupMethod(PAnsiChar('%s'), @F%s%s_%sFunc);", GetCExportName(NameSpace, class.ClassName, method, false), NameSpace, class.ClassName, method.MethodName)
			w.Writeln("    if AResult <> %s_SUCCESS then", strings.ToUpper(NameSpace))
			w.Writeln("      raise E%sException.CreateCustomMessage(%s_ERROR_COULDNOTLOADLIBRARY, '');", NameSpace, strings.ToUpper(NameSpace))
		}
	}

	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]
		w.Writeln("    AResult := ALookupMethod(PAnsiChar('%s'), @F%s%sFunc);", GetCExportName(NameSpace, "Wrapper", method, true), NameSpace, method.MethodName)
		w.Writeln("    if AResult <> %s_SUCCESS then", strings.ToUpper(NameSpace))
		w.Writeln("      raise E%sException.CreateCustomMessage(%s_ERROR_COULDNOTLOADLIBRARY, '');", NameSpace, strings.ToUpper(NameSpace))
	}
	w.Writeln("    ")
	w.Writeln("    checkBinaryVersion();")
	w.Writeln("  end;")
	w.Writeln("")

	w.Writeln("  destructor T%sWrapper.Destroy;", NameSpace)
	w.Writeln("  begin")
	w.Writeln("    {$IFDEF MSWINDOWS}")
	w.Writeln("      if FModule <> 0 then")
	w.Writeln("        FreeLibrary(FModule);")
	w.Writeln("    {$ELSE}")
	w.Writeln("      if FModule <> 0 then")
	w.Writeln("        UnloadLibrary(FModule);")
	w.Writeln("    {$ENDIF MSWINDOWS}")
	w.Writeln("    inherited;")
	w.Writeln("  end;")
	w.Writeln("")
	w.Writeln("  procedure T%sWrapper.CheckError(AInstance: %s; AErrorCode: T%sResult);", NameSpace, pascalBaseClassName, NameSpace)
	w.Writeln("  var")
	w.Writeln("    AErrorMessage: String;")
	w.Writeln("  begin")
	w.Writeln("    if AInstance <> nil then begin")
	w.Writeln("      if AInstance.FWrapper <> Self then")
	w.Writeln("        raise E%sException.CreateCustomMessage(%s_ERROR_INVALIDCAST, 'invalid wrapper call');", NameSpace, strings.ToUpper(NameSpace))
	w.Writeln("    end;")

	w.Writeln("    if AErrorCode <> %s_SUCCESS then begin", strings.ToUpper(NameSpace))
	w.Writeln("      AErrorMessage := '';")

	w.Writeln("      if Assigned(AInstance) then")
	w.Writeln("        %s(AInstance, AErrorMessage);", component.Global.ErrorMethod)

	w.Writeln("      raise E%sException.Create(AErrorCode, AErrorMessage);", NameSpace)
	w.Writeln("    end;")
	w.Writeln("  end;")
	w.Writeln("")

	w.Writeln("  {$IFDEF MSWINDOWS}")
	w.Writeln("  function T%sWrapper.LoadFunction(AFunctionName: AnsiString; FailIfNotExistent: Boolean): FARPROC;", NameSpace)
	w.Writeln("  begin")
	w.Writeln("    Result := GetProcAddress(FModule, PAnsiChar(AFunctionName));")
	w.Writeln("    if FailIfNotExistent and not Assigned(Result) then")
	w.Writeln("      raise E%sException.CreateCustomMessage(%s_ERROR_COULDNOTFINDLIBRARYEXPORT, 'could not find function ' + AFunctionName);", NameSpace, strings.ToUpper(NameSpace))
	w.Writeln("  end;")
	w.Writeln("  {$ELSE}")
	w.Writeln("  function T%sWrapper.LoadFunction(AFunctionName: AnsiString; FailIfNotExistent: Boolean): Pointer;", NameSpace)
	w.Writeln("  begin")
	w.Writeln("    Result := dynlibs.GetProcAddress(FModule, AFunctionName);")
	w.Writeln("    if FailIfNotExistent and not Assigned(Result) then")
	w.Writeln("      raise E%sException.CreateCustomMessage(%s_ERROR_COULDNOTFINDLIBRARYEXPORT, 'could not find function ' + AFunctionName);", NameSpace, strings.ToUpper(NameSpace))
	w.Writeln("  end;")
	w.Writeln("  {$ENDIF MSWINDOWS}")
	w.Writeln("")

	w.Writeln("  procedure T%sWrapper.checkBinaryVersion();", NameSpace)
	w.Writeln("  var")
	w.Writeln("    AMajor, AMinor, AMicro: Cardinal;")
	w.Writeln("  begin")
	w.Writeln("    %s(AMajor, AMinor, AMicro);", global.VersionMethod)
	if minorVersion(component.Version) == 0 {
		w.Writeln("    if (AMajor <> %s_VERSION_MAJOR) then", strings.ToUpper(NameSpace))
	} else {
		w.Writeln("    if (AMajor <> %s_VERSION_MAJOR) or (AMinor < %s_VERSION_MINOR) then", strings.ToUpper(NameSpace), strings.ToUpper(NameSpace))
	}
	w.Writeln("      raise E%sException.Create(%s_ERROR_INCOMPATIBLEBINARYVERSION, '');", NameSpace, strings.ToUpper(NameSpace))
	w.Writeln("  end;")
	w.Writeln("  ")

	if len(component.ImportedComponentDefinitions) > 0 {
		w.Writeln("    // Imported Components")
		for _, subComponent := range component.ImportedComponentDefinitions {
			w.Writeln("  function T%sWrapper.Get%sWrapper(): T%sWrapper;", NameSpace, subComponent.NameSpace, subComponent.NameSpace)
			w.Writeln("  begin")
			w.Writeln("    if not Assigned(F%sWrapper) then", subComponent.NameSpace)
			w.Writeln("      raise E%sException.Create(%s_ERROR_COULDNOTLOADLIBRARY, 'Injected library %s has not yet been loaded.');", NameSpace, strings.ToUpper(NameSpace), subComponent.NameSpace)
			w.Writeln("    result := F%sWrapper;", subComponent.NameSpace)
			w.Writeln("  end;")
		}
		w.Writeln("    ")
	}

	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]

		isSpecialFunction, err := CheckHeaderSpecialFunction(method, global)
		if err != nil {
			return err
		}

		definitionLines := make([]string, 0)
		implementationLines := make([]string, 0)
		if isSpecialFunction == eSpecialMethodInjection {
			sParamName := "A" + method.Params[0].ParamName
			for _, subComponent := range component.ImportedComponentDefinitions {
				theNameSpace := subComponent.NameSpace
				definitionLines = append(definitionLines, "ANameSpaceFound: boolean;")

				implementationLines = append(implementationLines, fmt.Sprintf("ANameSpaceFound := False;"))
				implementationLines = append(implementationLines, fmt.Sprintf("if (%s = '%s') then begin", sParamName, theNameSpace))
				implementationLines = append(implementationLines, fmt.Sprintf("  if assigned(F%sWrapper) then", theNameSpace))
				implementationLines = append(implementationLines, fmt.Sprintf("    raise E%sException.Create(%s_ERROR_COULDNOTLOADLIBRARY, 'Library with namespace ' + %s + ' is already registered.');", NameSpace, strings.ToUpper(NameSpace), sParamName))
				implementationLines = append(implementationLines, fmt.Sprintf("  F%sWrapper := T%sWrapper.CreateFromSymbolLookupMethod(A%s);", theNameSpace, theNameSpace, method.Params[1].ParamName))
				implementationLines = append(implementationLines, fmt.Sprintf("  ANameSpaceFound := True;"))
				implementationLines = append(implementationLines, fmt.Sprintf("end;"))
			}
			implementationLines = append(implementationLines, fmt.Sprintf("if not ANameSpaceFound then"))
			implementationLines = append(implementationLines, fmt.Sprintf("  raise E%sException.Create(%s_ERROR_COULDNOTLOADLIBRARY, 'Unknown namespace ' + %s);", NameSpace, strings.ToUpper(NameSpace), sParamName))
		}

		err = writePascalClassMethodImplementation(method, w, NameSpace, "Wrapper", definitionLines, implementationLines, true)
		if err != nil {
			return err
		}
	}

	w.Writeln("")
	w.Writeln("end.")

	return nil
}

func getPascalClassParameters(method ComponentDefinitionMethod, NameSpace string, ClassName string, isGlobal bool, isImplementation bool) (string, string, error) {
	parameters := ""
	returnType := ""

	for k := 0; k < len(method.Params); k++ {
		param := method.Params[k]
		ParamTypeName, err := getPascalParameterType(param.ParamType, NameSpace, param.ParamClass, false, isImplementation)
		if err != nil {
			return "", "", err
		}

		switch param.ParamPass {
		case "in":
			if parameters != "" {
				parameters = parameters + "; "
			}
			parameters = parameters + "const A" + param.ParamName + ": " + ParamTypeName

		case "out":
			if parameters != "" {
				parameters = parameters + "; "
			}
			parameters = parameters + "out A" + param.ParamName + ": " + ParamTypeName

		case "return":
			if returnType != "" {
				return "", "", fmt.Errorf("duplicate return value \"%s\" for Pascal method \"%s\"", param.ParamName, method.MethodName)
			}
			returnType = ParamTypeName
		}
	}

	return parameters, returnType, nil
}

func writePascalClassMethodDefinition(method ComponentDefinitionMethod, w LanguageWriter, NameSpace string, ClassName string, isGlobal bool, isImplementation bool) error {

	parameters, returnType, err := getPascalClassParameters(method, NameSpace, ClassName, isGlobal, isImplementation)
	if err != nil {
		return err
	}

	classPrefix := ""
	if isImplementation && isGlobal {
		classPrefix = "class "
	}

	if returnType == "" {
		w.Writeln("%sprocedure %s(%s);", classPrefix, method.MethodName, parameters)
	} else {
		w.Writeln("%sfunction %s(%s): %s;", classPrefix, method.MethodName, parameters, returnType)
	}

	return nil
}

func writePascalClassMethodImplementation(method ComponentDefinitionMethod, w LanguageWriter, NameSpace string, ClassName string, definitionLines []string, implementationLines []string, isGlobal bool) error {

	parameters, returnType, err := getPascalClassParameters(method, NameSpace, ClassName, isGlobal, false)
	if err != nil {
		return err
	}

	defineCommands := make([]string, 0)
	initCommands := make([]string, 0)
	resultCommands := make([]string, 0)
	postInitCommands := make([]string, 0)
	wrapperCallPrefix := ""
	wrapperInstanceName := ""

	doInitCall := false

	callFunctionName := ""
	callFunctionParameters := ""
	initCallParameters := ""
	errorInstanceHandle := ""

	if isGlobal {
		callFunctionName = fmt.Sprintf("%s%sFunc", NameSpace, method.MethodName)
		errorInstanceHandle = "nil"
		wrapperInstanceName = "Self"
	} else {
		callFunctionName = fmt.Sprintf("%s%s_%sFunc", NameSpace, ClassName, method.MethodName)
		callFunctionParameters = "FHandle"
		errorInstanceHandle = "Self"
		wrapperCallPrefix = "FWrapper."
		wrapperInstanceName = "FWrapper"
	}

	initCallParameters = callFunctionParameters

	for k := 0; k < len(method.Params); k++ {
		param := method.Params[k]
		PlainParamTypeName, err := getPascalParameterType(param.ParamType, NameSpace, param.ParamClass, true, false)
		if err != nil {
			return err
		}

		if callFunctionParameters != "" {
			callFunctionParameters = callFunctionParameters + ", "
		}

		if initCallParameters != "" {
			initCallParameters = initCallParameters + ", "
		}

		switch param.ParamPass {
		case "in":

			switch param.ParamType {
			case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64", "single", "double", "pointer":
				callFunctionParameters = callFunctionParameters + "A" + param.ParamName
				initCallParameters = initCallParameters + "A" + param.ParamName

			case "string":
				callFunctionParameters = callFunctionParameters + "PAnsiChar(A" + param.ParamName + ")"
				initCallParameters = initCallParameters + "PAnsiChar(A" + param.ParamName + ")"

			case "enum":
				callFunctionParameters = callFunctionParameters + "convert" + param.ParamClass + "ToConst(A" + param.ParamName + ")"
				initCallParameters = initCallParameters + "convert" + param.ParamClass + "ToConst(A" + param.ParamName + ")"

			case "bool":
				callFunctionParameters = callFunctionParameters + "Ord(A" + param.ParamName + ")"
				initCallParameters = initCallParameters + "Ord(A" + param.ParamName + ")"

			case "struct":
				callFunctionParameters = callFunctionParameters + "@A" + param.ParamName
				initCallParameters = initCallParameters + "@A" + param.ParamName

			case "basicarray":
				basicPlainTypeName, err := getPascalParameterType(param.ParamClass, NameSpace, "", true, false)
				if err != nil {
					return err
				}

				defineCommands = append(defineCommands, "Ptr"+param.ParamName+": P"+basicPlainTypeName+";")
				defineCommands = append(defineCommands, "Len"+param.ParamName+": QWord;")
				initCommands = append(initCommands, fmt.Sprintf("Len%s := Length(A%s);", param.ParamName, param.ParamName))
				initCommands = append(initCommands, fmt.Sprintf("if Len%s > $FFFFFFFF then", param.ParamName))
				initCommands = append(initCommands, fmt.Sprintf("  raise E%sException.CreateCustomMessage(%s_ERROR_INVALIDPARAM, 'array has too many entries.');", NameSpace, strings.ToUpper(NameSpace)))
				initCommands = append(initCommands, fmt.Sprintf("if Len%s > 0 then", param.ParamName))
				initCommands = append(initCommands, fmt.Sprintf("  Ptr%s := @A%s[0]", param.ParamName, param.ParamName))
				initCommands = append(initCommands, fmt.Sprintf("else"))
				initCommands = append(initCommands, fmt.Sprintf("  Ptr%s := nil;", param.ParamName))
				initCommands = append(initCommands, "")

				callFunctionParameters = callFunctionParameters + "QWord(Len" + param.ParamName + "), Ptr" + param.ParamName
				initCallParameters = initCallParameters + "QWord(Len" + param.ParamName + "), Ptr" + param.ParamName

			case "structarray":

				defineCommands = append(defineCommands, "Ptr"+param.ParamName+": P"+NameSpace+param.ParamClass+";")
				defineCommands = append(defineCommands, "Len"+param.ParamName+": QWord;")
				initCommands = append(initCommands, fmt.Sprintf("Len%s := Length(A%s);", param.ParamName, param.ParamName))
				initCommands = append(initCommands, fmt.Sprintf("if Len%s > $FFFFFFFF then", param.ParamName))
				initCommands = append(initCommands, fmt.Sprintf("  raise E%sException.CreateCustomMessage(%s_ERROR_INVALIDPARAM, 'array has too many entries.');", NameSpace, strings.ToUpper(NameSpace)))
				initCommands = append(initCommands, fmt.Sprintf("if Len%s > 0 then", param.ParamName))
				initCommands = append(initCommands, fmt.Sprintf("  Ptr%s := @A%s[0]", param.ParamName, param.ParamName))
				initCommands = append(initCommands, fmt.Sprintf("else"))
				initCommands = append(initCommands, fmt.Sprintf("  Ptr%s := nil;", param.ParamName))
				initCommands = append(initCommands, "")

				callFunctionParameters = callFunctionParameters + "QWord(Len" + param.ParamName + "), Ptr" + param.ParamName
				initCallParameters = initCallParameters + "QWord(Len" + param.ParamName + "), Ptr" + param.ParamName

			case "functiontype":
				initCommands = append(initCommands, fmt.Sprintf("if not Assigned(A%s) then", param.ParamName))
				initCommands = append(initCommands, fmt.Sprintf("  raise E%sException.CreateCustomMessage(%s_ERROR_INVALIDPARAM, 'A%s is a nil value.');", NameSpace, strings.ToUpper(NameSpace), param.ParamName))
				callFunctionParameters = callFunctionParameters + "A" + param.ParamName
				initCallParameters = initCallParameters + "A" + param.ParamName

			case "class", "optionalclass":
				defineCommands = append(defineCommands, "A"+param.ParamName+"Handle: T"+NameSpace+"Handle;")
				initCommands = append(initCommands, fmt.Sprintf("if Assigned(A%s) then", param.ParamName))
				initCommands = append(initCommands, "A"+param.ParamName+"Handle := A" + param.ParamName + ".TheHandle")
				initCommands = append(initCommands, fmt.Sprintf("else"))
				if (param.ParamType == "optionalclass") {
					initCommands = append(initCommands, "A"+param.ParamName+"Handle := nil;")
				} else {
					initCommands = append(initCommands, fmt.Sprintf("  raise E%sException.CreateCustomMessage(%s_ERROR_INVALIDPARAM, 'A%s is a nil value.');", NameSpace, strings.ToUpper(NameSpace), param.ParamName))
				}
				callFunctionParameters = callFunctionParameters + "A" + param.ParamName + "Handle"
				initCallParameters = initCallParameters + "A" + param.ParamName + "Handle"

			default:
				return fmt.Errorf("invalid method parameter type \"%s\" for %s.%s (%s)", param.ParamType, ClassName, method.MethodName, param.ParamName)
			}

		case "out":

			switch param.ParamType {
			case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64", "single", "double", "pointer":
				callFunctionParameters = callFunctionParameters + "A" + param.ParamName
				initCallParameters = initCallParameters + "A" + param.ParamName

			case "string":
				defineCommands = append(defineCommands, "bytesNeeded"+param.ParamName+": Cardinal;")
				defineCommands = append(defineCommands, "bytesWritten"+param.ParamName+": Cardinal;")
				defineCommands = append(defineCommands, "buffer"+param.ParamName+": array of Char;")
				initCommands = append(initCommands, "bytesNeeded"+param.ParamName+":= 0;")
				initCommands = append(initCommands, "bytesWritten"+param.ParamName+":= 0;")

				initCallParameters = initCallParameters + fmt.Sprintf("0, bytesNeeded%s, nil", param.ParamName)

				postInitCommands = append(postInitCommands, fmt.Sprintf("SetLength(buffer%s, bytesNeeded%s);", param.ParamName, param.ParamName))

				callFunctionParameters = callFunctionParameters + fmt.Sprintf("bytesNeeded%s, bytesWritten%s, @buffer%s[0]", param.ParamName, param.ParamName, param.ParamName)

				resultCommands = append(resultCommands, fmt.Sprintf("  A%s := StrPas(@buffer%s[0]);", param.ParamName, param.ParamName))

				doInitCall = true

			case "enum":
				defineCommands = append(defineCommands, "Result"+param.ParamName+": Integer;")
				initCommands = append(initCommands, "Result"+param.ParamName+" := 0;")

				callFunctionParameters = callFunctionParameters + "Result" + param.ParamName
				initCallParameters = initCallParameters + "Result" + param.ParamName
				resultCommands = append(resultCommands, fmt.Sprintf("  A%s := convertConstTo%s(Result%s);", param.ParamName, param.ParamClass, param.ParamName))

			case "bool":
				defineCommands = append(defineCommands, "Result"+param.ParamName+": Byte;")
				initCommands = append(initCommands, "Result"+param.ParamName+" := 0;")

				callFunctionParameters = callFunctionParameters + "Result" + param.ParamName
				initCallParameters = initCallParameters + "Result" + param.ParamName
				resultCommands = append(resultCommands, fmt.Sprintf("  A%s := Result%s <> 0;", param.ParamName, param.ParamName))

			case "struct":
				callFunctionParameters = callFunctionParameters + "@A" + param.ParamName
				initCallParameters = initCallParameters + "@A" + param.ParamName

			case "basicarray", "structarray":

				defineCommands = append(defineCommands, "countNeeded"+param.ParamName+": QWord;")
				defineCommands = append(defineCommands, "countWritten"+param.ParamName+": QWord;")
				initCommands = append(initCommands, "countNeeded"+param.ParamName+":= 0;")
				initCommands = append(initCommands, "countWritten"+param.ParamName+":= 0;")

				initCallParameters = initCallParameters + fmt.Sprintf("0, countNeeded%s, nil", param.ParamName)

				postInitCommands = append(postInitCommands, fmt.Sprintf("SetLength(A%s, countNeeded%s);", param.ParamName, param.ParamName))

				callFunctionParameters = callFunctionParameters + fmt.Sprintf("countNeeded%s, countWritten%s, @A%s[0]", param.ParamName, param.ParamName, param.ParamName)

				doInitCall = true

			case "class", "optionalclass":
				theNameSpace, theParamClass, _ := decomposeParamClassName(param.ParamClass)
				theWrapperInstance := wrapperInstanceName
				if len(theNameSpace) > 0 {
					theWrapperInstance = theWrapperInstance + "." + theNameSpace + "Wrapper"
				} else {
					theNameSpace = NameSpace
				}

				defineCommands = append(defineCommands, "H"+param.ParamName+": "+PlainParamTypeName+";")
				initCommands = append(initCommands, fmt.Sprintf("A%s := nil;", param.ParamName))
				initCommands = append(initCommands, "H"+param.ParamName+" := nil;")
				callFunctionParameters = callFunctionParameters + "H" + param.ParamName
				initCallParameters = initCallParameters + "nil"

				resultCommands = append(resultCommands, fmt.Sprintf("  if Assigned(H%s) then", param.ParamName))
				resultCommands = append(resultCommands, fmt.Sprintf("    A%s := T%s%s.Create(%s, H%s);", param.ParamName, theNameSpace, theParamClass, theWrapperInstance, param.ParamName))

			default:
				return fmt.Errorf("invalid method parameter type \"%s\" for %s.%s(%s)", param.ParamType, ClassName, method.MethodName, param.ParamName)
			}

		case "return":

			switch param.ParamType {
			case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64", "single", "double", "pointer":
				callFunctionParameters = callFunctionParameters + "Result"

			case "string":
				defineCommands = append(defineCommands, "bytesNeeded"+param.ParamName+": Cardinal;")
				defineCommands = append(defineCommands, "bytesWritten"+param.ParamName+": Cardinal;")
				defineCommands = append(defineCommands, "buffer"+param.ParamName+": array of Char;")
				initCommands = append(initCommands, "bytesNeeded"+param.ParamName+":= 0;")
				initCommands = append(initCommands, "bytesWritten"+param.ParamName+":= 0;")

				initCallParameters = initCallParameters + fmt.Sprintf("0, bytesNeeded%s, nil", param.ParamName)

				postInitCommands = append(postInitCommands, fmt.Sprintf("SetLength(buffer%s, bytesNeeded%s);", param.ParamName, param.ParamName))

				callFunctionParameters = callFunctionParameters + fmt.Sprintf("bytesNeeded%s, bytesWritten%s, @buffer%s[0]", param.ParamName, param.ParamName, param.ParamName)

				resultCommands = append(resultCommands, fmt.Sprintf("  Result := StrPas(@buffer%s[0]);", param.ParamName))

				doInitCall = true

			case "enum":
				defineCommands = append(defineCommands, "Result"+param.ParamName+": Integer;")
				initCommands = append(initCommands, "Result"+param.ParamName+" := 0;")

				callFunctionParameters = callFunctionParameters + "Result" + param.ParamName
				initCallParameters = initCallParameters + "Result" + param.ParamName
				resultCommands = append(resultCommands, fmt.Sprintf("  Result := convertConstTo%s(Result%s);", param.ParamClass, param.ParamName))

			case "bool":
				defineCommands = append(defineCommands, "Result"+param.ParamName+": Byte;")
				initCommands = append(initCommands, "Result"+param.ParamName+" := 0;")

				callFunctionParameters = callFunctionParameters + "Result" + param.ParamName
				initCallParameters = initCallParameters + "Result" + param.ParamName
				resultCommands = append(resultCommands, fmt.Sprintf("  Result := (Result%s <> 0);", param.ParamName))

			case "struct":
				callFunctionParameters = callFunctionParameters + "@Result"

			case "basicarray", "structarray":
				defineCommands = append(defineCommands, "countNeeded"+param.ParamName+": QWord;")
				defineCommands = append(defineCommands, "countWritten"+param.ParamName+": QWord;")
				initCommands = append(initCommands, "countNeeded"+param.ParamName+":= 0;")
				initCommands = append(initCommands, "countWritten"+param.ParamName+":= 0;")

				initCallParameters = initCallParameters + fmt.Sprintf("0, countNeeded%s, nil", param.ParamName)

				postInitCommands = append(postInitCommands, fmt.Sprintf("SetLength(Result, countNeeded%s);", param.ParamName))

				callFunctionParameters = callFunctionParameters + fmt.Sprintf("countNeeded%s, countWritten%s, @Result[0]", param.ParamName, param.ParamName)

				doInitCall = true

			case "class", "optionalclass":
				theNameSpace, theParamClass, _ := decomposeParamClassName(param.ParamClass)
				theWrapperInstance := wrapperInstanceName
				if len(theNameSpace) > 0 {
					theWrapperInstance = theWrapperInstance + "." + theNameSpace + "Wrapper"
				} else {
					theNameSpace = NameSpace
				}
				defineCommands = append(defineCommands, "H"+param.ParamName+": "+PlainParamTypeName+";")
				initCommands = append(initCommands, "Result := nil;")
				initCommands = append(initCommands, "H"+param.ParamName+" := nil;")
				callFunctionParameters = callFunctionParameters + "H" + param.ParamName
				resultCommands = append(resultCommands, fmt.Sprintf("  if Assigned(H%s) then", param.ParamName))
				resultCommands = append(resultCommands, fmt.Sprintf("    Result := T%s%s.Create(%s, H%s);", theNameSpace, theParamClass, theWrapperInstance, param.ParamName))

			default:
				return fmt.Errorf("invalid method parameter type \"%s\" for %s.%s (%s)", param.ParamType, ClassName, method.MethodName, param.ParamName)
			}

		}
	}

	if returnType == "" {
		w.Writeln("  procedure T%s%s.%s(%s);", NameSpace, ClassName, method.MethodName, parameters)
	} else {
		w.Writeln("  function T%s%s.%s(%s): %s;", NameSpace, ClassName, method.MethodName, parameters, returnType)
	}

	if len(defineCommands)+len(definitionLines) > 0 {
		w.Writeln("  var")
		w.Writelns("    ", defineCommands)
		w.Writelns("    ", definitionLines)
	}

	w.Writeln("  begin")
	w.Writelns("    ", initCommands)

	if doInitCall {
		w.Writeln("    %sCheckError(%s, %s%s(%s));", wrapperCallPrefix, errorInstanceHandle, wrapperCallPrefix, callFunctionName, initCallParameters)
	}

	w.Writelns("    ", postInitCommands)

	w.Writeln("    %sCheckError(%s, %s%s(%s));", wrapperCallPrefix, errorInstanceHandle, wrapperCallPrefix, callFunctionName, callFunctionParameters)

	if len(implementationLines) > 0 {
		w.Writelns("    ", implementationLines)
		w.Writeln("   ")
	}
	w.Writelns("  ", resultCommands)

	w.Writeln("  end;")
	w.Writeln("")

	return nil
}

func writePascalFunctionType(method ComponentDefinitionMethod, w LanguageWriter, NameSpace string, ClassName string, isGlobal bool, spacing string) error {

	PascalCallbackName := ""
	parameters := ""
	if isGlobal {
		PascalCallbackName = fmt.Sprintf("T%s%sFunc", NameSpace, method.MethodName)
	} else {
		PascalCallbackName = fmt.Sprintf("T%s%s_%sFunc", NameSpace, ClassName, method.MethodName)
		parameters = fmt.Sprintf("p%s: T%sHandle", ClassName, NameSpace)
	}

	w.Writeln(spacing + "(**")
	w.Writeln(spacing+"* %s", method.MethodDescription)
	w.Writeln(spacing + "*")
	if !isGlobal {
		w.Writeln(spacing+"* @param[in] p%s - %s instance.", ClassName, ClassName)
	}

	for k := 0; k < len(method.Params); k++ {
		param := method.Params[k]

		pascalParams, err := generatePlainPascalParameter(param, ClassName, method.MethodName, NameSpace)
		if err != nil {
			return err
		}

		for _, pascalParam := range pascalParams {
			w.Writeln(spacing + pascalParam.ParamComment)
			if parameters != "" {
				parameters = parameters + "; "
			}
			parameters = parameters + pascalParam.ParamConvention + pascalParam.ParamName + ": " + pascalParam.ParamType
		}

	}

	w.Writeln(spacing + "* @return error code or 0 (success)")
	w.Writeln(spacing + "*)")

	w.Writeln(spacing+"%s = function(%s): T%sResult; cdecl;", PascalCallbackName, parameters, NameSpace)
	w.Writeln(spacing + "")

	return nil
}

func generatePlainPascalParameters(method ComponentDefinitionMethod, className string, NameSpace string) ([]pascalParameter, error) {
	parameters := []pascalParameter{}
	for k := 0; k < len(method.Params); k++ {
		param := method.Params[k]

		cParam, err := generatePlainPascalParameter(param, className, method.MethodName, NameSpace)
		if err != nil {
			return nil, nil
		}
		parameters = append(parameters, cParam...)
	}

	return parameters, nil
}

func buildDynamicPascalExample(w LanguageWriter, component ComponentDefinition, outputFolder string) error {
	NameSpace := component.NameSpace
	BaseName := component.BaseName
	global := component.Global
	w.Writeln("program %sPascalTest;", NameSpace)
	w.Writeln("")
	w.Writeln("uses")
	w.Writeln("  {$IFDEF UNIX}{$IFDEF UseCThreads}")
	w.Writeln("  cthreads,")
	w.Writeln("  {$ENDIF}{$ENDIF}")
	w.Writeln("  Classes, SysUtils, CustApp,")
	if len(component.ImportedComponentDefinitions) > 0 {
		for _, subComponent := range component.ImportedComponentDefinitions {
			w.Writeln("  Unit_%s,", subComponent.NameSpace)
		}
	}
	w.Writeln("  Unit_%s", NameSpace)
	w.Writeln("  { you can add units after this };")
	w.Writeln("")
	w.Writeln("type")
	w.Writeln("")
	w.Writeln("T%s_Example = class(TCustomApplication)", NameSpace)
	w.Writeln("protected")
	w.Writeln("  procedure DoRun; override;")
	w.Writeln("  procedure Test%s();", NameSpace)
	w.Writeln("public")
	w.Writeln("  constructor Create(TheOwner: TComponent); override;")
	w.Writeln("  destructor Destroy; override;")
	w.Writeln("end;")
	w.Writeln("")
	w.Writeln("")
	w.Writeln("procedure T%s_Example.Test%s();", NameSpace, NameSpace)
	w.Writeln("var")
	w.Writeln("  A%sWrapper: T%sWrapper;", NameSpace, NameSpace)
	w.Writeln("  AMajor, AMinor, AMicro: Cardinal;")
	if len(global.PrereleaseMethod) > 0 {
		w.Writeln("  APreReleaseInfo, ABuildInfo: string;")
	}
	w.Writeln("  AVersionString: string;")
	w.Writeln("  ALibPath: string;")
	w.Writeln("begin")
	w.Writeln("  writeln('loading DLL');")
	w.Writeln("  ALibPath := ''; // TODO add the location of the shared library binary here")
	w.Writeln("  A%sWrapper := T%sWrapper.Create(ALibPath + '/' + '%s.'); // TODO add the extension of the shared library file here", NameSpace, NameSpace, BaseName)
	w.Writeln("  try")
	w.Writeln("    writeln('loading DLL Done');")
	w.Writeln("    A%sWrapper.%s(AMajor, AMinor, AMicro);", NameSpace, global.VersionMethod)
	w.Writeln("    AVersionString := Format('%s.version = %s', [AMajor, AMinor, AMicro]);", NameSpace, "%d.%d.%d")
	if len(global.PrereleaseMethod) > 0 {
		w.Writeln("    if (A%sWrapper.%s(APreReleaseInfo) then", NameSpace, global.PrereleaseMethod)
		w.Writeln("      AVersionString := AVersionString + '-' + APreReleaseInfo;")
	}
	if len(global.BuildinfoMethod) > 0 {
		w.Writeln("    if (A%sWrapper.%s(ABuildInfo) then", NameSpace, global.BuildinfoMethod)
		w.Writeln("      AVersionString := AVersionString + '-' + ABuildInfo;")
	}
	w.Writeln("    writeln(AVersionString);")
	w.Writeln("  finally")
	w.Writeln("    FreeAndNil(A%sWrapper);", NameSpace)
	w.Writeln("  end;")
	w.Writeln("end;")
	w.Writeln("")
	w.Writeln("procedure T%s_Example.DoRun;", NameSpace)
	w.Writeln("begin")
	w.Writeln("  try")
	w.Writeln("    Test%s();", NameSpace)
	w.Writeln("  except")
	w.Writeln("    On E: Exception do")
	w.Writeln("      writeln('Fatal error: ', E.Message);")
	w.Writeln("  end;")
	w.Writeln("  Terminate")
	w.Writeln("end;")

	w.Writeln("")

	w.Writeln("constructor T%s_Example.Create(TheOwner: TComponent);", NameSpace)
	w.Writeln("begin")
	w.Writeln("  inherited Create(TheOwner);")
	w.Writeln("  StopOnException:=True;")
	w.Writeln("end;")

	w.Writeln("")

	w.Writeln("destructor T%s_Example.Destroy;", NameSpace)
	w.Writeln("begin")
	w.Writeln("  inherited Destroy;")
	w.Writeln("end;")
	w.Writeln("")

	w.Writeln("")
	w.Writeln("var")
	w.Writeln("  Application: T%s_Example;", NameSpace)
	w.Writeln("begin")
	w.Writeln("  Application:=T%s_Example.Create(nil);", NameSpace)
	w.Writeln("  Application.Run;")
	w.Writeln("  Application.Free;")
	w.Writeln("end.")

	return nil
}

func buildDynamicPascalExampleLPI(component ComponentDefinition, w LanguageWriter, outputFolder string) error {
	NameSpace := component.NameSpace

	otherUnitFiles := ""
	for _, subComponent := range component.ImportedComponentDefinitions {
		otherUnitFiles = otherUnitFiles + fmt.Sprintf(";..\\..\\..\\%s_Component\\Bindings\\Pascal", subComponent.NameSpace)
	}

	w.Writeln("<?xml version=\"1.0\" encoding=\"UTF-8\"?>")
	w.Writeln("<CONFIG>")
	w.Writeln("  <ProjectOptions>")
	w.Writeln("    <Version Value=\"%d\"/>", 10)
	w.Writeln("    <PathDelim Value=\"\\\"/>")
	w.Writeln("    <General>")
	w.Writeln("      <Flags>")
	w.Writeln("        <MainUnitHasCreateFormStatements Value=\"False\"/>")
	w.Writeln("        <MainUnitHasTitleStatement Value=\"False\"/>")
	w.Writeln("        <MainUnitHasScaledStatement Value=\"False\"/>")
	w.Writeln("      </Flags>")
	w.Writeln("      <SessionStorage Value=\"InProjectDir\"/>")
	w.Writeln("      <MainUnit Value=\"%d\"/>", 0)
	w.Writeln("      <Title Value=\"%s_Example\"/>", NameSpace)
	w.Writeln("      <UseAppBundle Value=\"False\"/>")
	w.Writeln("      <ResourceType Value=\"res\"/>")
	w.Writeln("    </General>")
	w.Writeln("    <BuildModes Count=\"%d\">", 2)
	w.Writeln("      <Item1 Name=\"Release\" Default=\"True\"/>")
	w.Writeln("      <Item2 Name=\"Debug\">")
	w.Writeln("        <CompilerOptions>")
	w.Writeln("          <Version Value=\"11\"/>")
	w.Writeln("          <PathDelim Value=\"\\\"/>")
	w.Writeln("          <Target>")
	w.Writeln("            <Filename Value=\"bin\\$(TargetCPU)-$(TargetOS)\\Release\\%s_Example\"/>", NameSpace)
	w.Writeln("          </Target>")
	w.Writeln("          <SearchPaths>")
	w.Writeln("            <IncludeFiles Value=\"$(ProjOutDir)\"/>")
	w.Writeln("            <OtherUnitFiles Value=\"..\\..\\Bindings\\Pascal " + otherUnitFiles + "\"/>")
	w.Writeln("            <UnitOutputDirectory Value=\"lib\\$(TargetCPU)-$(TargetOS)\"/>")
	w.Writeln("          </SearchPaths>")
	w.Writeln("          <Parsing>")
	w.Writeln("            <SyntaxOptions>")
	w.Writeln("              <IncludeAssertionCode Value=\"True\"/>")
	w.Writeln("            </SyntaxOptions>")
	w.Writeln("          </Parsing>")
	w.Writeln("          <CodeGeneration>")
	w.Writeln("            <RelocatableUnit Value=\"True\"/>")
	w.Writeln("          </CodeGeneration>")
	w.Writeln("          <Linking>")
	w.Writeln("            <Debugging>")
	w.Writeln("              <UseExternalDbgSyms Value=\"True\"/>")
	w.Writeln("            </Debugging>")
	w.Writeln("            <Options>")
	w.Writeln("              <ExecutableType Value=\"Library\"/>")
	w.Writeln("            </Options>")
	w.Writeln("          </Linking>")
	w.Writeln("        </CompilerOptions>")
	w.Writeln("      </Item2>")
	w.Writeln("    </BuildModes>")
	w.Writeln("    <PublishOptions>")
	w.Writeln("      <Version Value=\"%d\"/>", 2)
	w.Writeln("    </PublishOptions>")

	w.Writeln("    <RunParams>")
	w.Writeln("      <local>")
	w.Writeln("        <FormatVersion Value=\"1\"/>")
	w.Writeln("      </local>")
	w.Writeln("    </RunParams>")

	w.Writeln("    <Units Count=\"%d\">", 2)
	w.Writeln("      <Unit0>")
	w.Writeln("        <Filename Value=\"%s_Example.lpr\"/>", NameSpace)
	w.Writeln("        <IsPartOfProject Value=\"True\"/>")
	w.Writeln("      </Unit0>")
	w.Writeln("      <Unit1>")
	w.Writeln("        <Filename Value=\"Unit_%s.pas\"/>", NameSpace)
	w.Writeln("        <IsPartOfProject Value=\"True\"/>")
	w.Writeln("      </Unit1>")
	w.Writeln("    </Units>")

	w.Writeln("  </ProjectOptions>")
	w.Writeln("  <CompilerOptions>")
	w.Writeln("    <Version Value=\"%d\"/>", 11)
	w.Writeln("    <PathDelim Value=\"\\\"/>")
	w.Writeln("    <Target>")
	w.Writeln("      <Filename Value=\"bin\\$(TargetCPU)-$(TargetOS)\\Release\\%s_Example\"/>", NameSpace)
	w.Writeln("    </Target>")
	w.Writeln("    <SearchPaths>")
	w.Writeln("      <IncludeFiles Value=\"$(ProjOutDir)\"/>")
	w.Writeln("      <OtherUnitFiles Value=\"..\\..\\Bindings\\Pascal " + otherUnitFiles + "\"/>")
	w.Writeln("      <UnitOutputDirectory Value=\"lib\\$(TargetCPU)-$(TargetOS)\"/>")
	w.Writeln("    </SearchPaths>")
	w.Writeln("    <Parsing>")
	w.Writeln("      <SyntaxOptions>")
	w.Writeln("        <IncludeAssertionCode Value=\"True\"/>")
	w.Writeln("      </SyntaxOptions>")
	w.Writeln("    </Parsing>")
	w.Writeln("    <CodeGeneration>")
	w.Writeln("      <RelocatableUnit Value=\"True\"/>")
	w.Writeln("    </CodeGeneration>")
	w.Writeln("    <Linking>")
	w.Writeln("      <Debugging>")
	w.Writeln("        <StripSymbols Value=\"True\"/>")
	w.Writeln("        <UseExternalDbgSyms Value=\"True\"/>")
	w.Writeln("      </Debugging>")
	w.Writeln("      <Options>")
	w.Writeln("        <ExecutableType Value=\"Library\"/>")
	w.Writeln("      </Options>")
	w.Writeln("    </Linking>")
	w.Writeln("  </CompilerOptions>")
	w.Writeln("  <Debugging>")
	w.Writeln("    <Exceptions Count=\"%d\">", 3)
	w.Writeln("      <Item1>")
	w.Writeln("        <Name Value=\"EAbort\"/>")
	w.Writeln("      </Item1>")
	w.Writeln("      <Item2>")
	w.Writeln("        <Name Value=\"ECodetoolError\"/>")
	w.Writeln("      </Item2>")
	w.Writeln("      <Item3>")
	w.Writeln("        <Name Value=\"EFOpenError\"/>")
	w.Writeln("      </Item3>")
	w.Writeln("    </Exceptions>")
	w.Writeln("  </Debugging>")
	w.Writeln("</CONFIG>")
	w.Writeln("")

	return nil

}
