#!/usr/bin/env perl
use strict;
use warnings;
use Test::More;

BEGIN {
    eval { require Test::LeakTrace };
    plan skip_all => 'Test::LeakTrace required' if $@;
}
use Test::LeakTrace;

use object;

# Define test classes once
object::define('LeakPerson', 'name:Str', 'age:Int');
object::define('LeakWithDefault', 'val:Str:default(hello)');
object::define('LeakReadonly', 'fixed:Str:readonly');
object::define('LeakTyped', 'num:Num', 'arr:ArrayRef', 'hash:HashRef');

# Warmup
for (1..10) {
    my $obj = new LeakPerson name => 'Warmup', age => 1;
    $obj->name;
    $obj->age;
}

subtest 'accessor get no leak' => sub {
    my $obj = new LeakPerson name => 'Test', age => 25;
    no_leaks_ok {
        for (1..1000) {
            my $n = $obj->name;
            my $a = $obj->age;
        }
    } 'accessor get no leak';
};

subtest 'accessor set no leak' => sub {
    my $obj = new LeakPerson name => 'Start', age => 1;
    no_leaks_ok {
        for (1..1000) {
            $obj->name('Updated');
            $obj->age(42);
        }
    } 'accessor set no leak';
};

subtest 'default value access no leak' => sub {
    my $obj = new LeakWithDefault;
    no_leaks_ok {
        for (1..1000) {
            my $v = $obj->val;
        }
    } 'default value no leak';
};

subtest 'readonly access no leak' => sub {
    my $obj = new LeakReadonly fixed => 'immutable';
    no_leaks_ok {
        for (1..1000) {
            my $v = $obj->fixed;
        }
    } 'readonly access no leak';
};

subtest 'typed properties no leak' => sub {
    my $obj = new LeakTyped num => 3.14, arr => [1, 2, 3], hash => { a => 1 };
    no_leaks_ok {
        for (1..1000) {
            my $n = $obj->num;
            my $a = $obj->arr;
            my $h = $obj->hash;
        }
    } 'typed get no leak';
};

subtest 'typed setter no leak' => sub {
    my $obj = new LeakTyped num => 0, arr => [], hash => {};
    my $arr = [1, 2, 3];
    my $hash = { x => 1 };
    no_leaks_ok {
        for (1..1000) {
            $obj->num(3.14);
            $obj->arr($arr);
            $obj->hash($hash);
        }
    } 'typed set no leak';
};

subtest 'prototype operations no leak' => sub {
    my $parent = new LeakPerson name => 'Parent', age => 50;
    my $child = new LeakPerson name => 'Child', age => 20;
    object::set_prototype($child, $parent);

    no_leaks_ok {
        for (1..1000) {
            my $proto = object::prototype($child);
        }
    } 'prototype get no leak';
};

subtest 'freeze check no leak' => sub {
    my $obj = new LeakPerson name => 'Freeze', age => 30;
    object::freeze($obj);
    no_leaks_ok {
        for (1..1000) {
            my $f = object::is_frozen($obj);
        }
    } 'is_frozen no leak';
};

subtest 'lock check no leak' => sub {
    my $obj = new LeakPerson name => 'Lock', age => 30;
    object::lock($obj);
    no_leaks_ok {
        for (1..1000) {
            my $l = object::is_locked($obj);
        }
    } 'is_locked no leak';
};

subtest 'type introspection no leak' => sub {
    no_leaks_ok {
        for (1..1000) {
            my $has = object::has_type('Str');
            my @types = object::list_types();
        }
    } 'type introspection no leak';
};

done_testing;
