% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/s3_funcs.R
\name{fit.emc}
\alias{fit.emc}
\alias{fit}
\title{Model Estimation in EMC2}
\usage{
\method{fit}{emc}(
  emc,
  stage = NULL,
  iter = 1000,
  stop_criteria = NULL,
  search_width = 1,
  step_size = 100,
  verbose = TRUE,
  fileName = NULL,
  particle_factor = 50,
  cores_per_chain = 1,
  cores_for_chains = length(emc),
  max_tries = 20,
  thin = FALSE,
  ...
)

fit(emc, ...)
}
\arguments{
\item{emc}{An emc object created with \code{make_emc},
or a path to where the emc object is stored.}

\item{stage}{A string. Indicates which stage to start the run from, either \code{preburn}, \code{burn}, \code{adapt} or \code{sample}.
If unspecified, it will run the subsequent stage (if there is one).}

\item{iter}{An integer. Indicates how many iterations to run in the sampling stage.}

\item{stop_criteria}{A list. Defines the stopping criteria and for which types
of parameters these should hold. See the details and examples section.}

\item{search_width}{A double. Tunes target acceptance probability of the MCMC process.
This fine-tunes the width of the search space to obtain the desired acceptance probability.
1 is the default width, increases lead to broader search.}

\item{step_size}{An integer. After each step, the stopping requirements as specified
by \code{stop_criteria} are checked and proposal distributions are updated. Defaults to 100.}

\item{verbose}{Logical. Whether to print messages between each step with the current status regarding the \code{stop_criteria}.}

\item{fileName}{A string. If specified, will auto-save emc object at this location on every iteration.}

\item{particle_factor}{An integer. \code{particle_factor} multiplied by the square
root of the number of sampled parameters determines the number of particles used.}

\item{cores_per_chain}{An integer. How many cores to use per chain. Parallelizes across
participant calculations. Only available on Linux or Mac OS. For Windows, only
parallelization across chains (\code{cores_for_chains}) is available.}

\item{cores_for_chains}{An integer. How many cores to use across chains.
Defaults to the number of chains. The total number of cores used is equal to \code{cores_per_chain} * \code{cores_for_chains}.}

\item{max_tries}{An integer. How many times should it try to meet the finish
conditions as specified by \code{stop_criteria}? Defaults to 20. \code{max_tries} is
ignored if the required number of iterations has not been reached yet.}

\item{thin}{A boolean. If \code{TRUE} will automatically thin the MCMC samples, closely matched to the ESS.
Can also be set to a double, in which case 1/thin of the chain will be removed (does not have to be an integer).}

\item{...}{Additional optional arguments}
}
\value{
An emc object
}
\description{
General purpose function to estimate models specified in EMC2.
}
\details{
\code{stop_criteria} is either a list of lists with names of the stages,
or a single list in which case its assumed to be for the sample \code{stage} (see examples).
The potential stop criteria to be set are:

\code{selection} (character vector): For which parameters the \code{stop_criteria} should hold

\code{mean_gd} (numeric): The mean Gelman-Rubin diagnostic across all parameters in the selection

\code{max_gd} (numeric): The max Gelman-Rubin diagnostic across all parameters in the selection

\code{min_unique} (integer): The minimum number of unique samples in the MCMC chains across all parameters in the selection

\code{min_es} (integer): The minimum number of effective samples across all parameters in the selection

\code{omit_mpsrf} (Boolean): Whether to include the multivariate point-scale reduction factor in the Gelman-Rubin diagnostic. Default is \code{FALSE}.

\code{iter} (integer): The number of MCMC samples to collect.

The estimation is performed using particle-metropolis within-Gibbs sampling.
For sampling details see:

Gunawan, D., Hawkins, G. E., Tran, M.-N., Kohn, R., & Brown, S. (2020).
New estimation approaches for the hierarchical linear ballistic accumulator model.
\emph{Journal of Mathematical Psychology} ,96, 102368. doi.org/10.1016/j.jmp.2020.102368

Stevenson, N., Donzallaz, M. C., Innes, R. J., Forstmann, B., Matzke, D., & Heathcote, A. (2024).
EMC2: An R Package for cognitive models of choice. doi.org/10.31234/osf.io/2e4dq
}
\examples{
\donttest{
# Define a design first
ADmat <- matrix(c(-1/2,1/2),ncol=1,dimnames=list(NULL,"d"))
# We also define a match function for lM
matchfun=function(d)d$S==d$lR

# Drop most subjects for speed
dat <- forstmann[forstmann$subjects \%in\% unique(forstmann$subjects)[1:2],]
dat$subjects <- droplevels(dat$subjects)

design_LNR <- design(data = dat,model=LNR,matchfun=matchfun,
                     formula=list(m~lM,s~1,t0~1),
                     contrasts=list(m=list(lM=ADmat)))
# Before fit can be called, we first need to make an emc object
LNR_s <- make_emc(dat, design_LNR, rt_resolution = 0.05, n_chains = 2)
# Run fit, here illustrating how to use stop_criteria (also for speed purposes)
LNR_s <- fit(LNR_s, cores_for_chains = 1, stop_criteria = list(
  preburn = list(iter = 10), burn = list(mean_gd = 2.5), adapt = list(min_unique = 20),
  sample = list(iter = 25, max_gd = 2)), verbose = FALSE, particle_factor = 30, step_size = 25)
}
}
