% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/seasonal_onset.R
\name{seasonal_onset}
\alias{seasonal_onset}
\title{Automated and Early Detection of Seasonal Epidemic Onset}
\usage{
seasonal_onset(
  tsd,
  k = 5,
  level = 0.95,
  disease_threshold = NA_real_,
  family = c("quasipoisson", "poisson"),
  na_fraction_allowed = 0.4,
  season_start = NULL,
  season_end = season_start - 1,
  only_current_season = NULL
)
}
\arguments{
\item{tsd}{A \code{tsd} object containing time series data}

\item{k}{An integer specifying the window size for modeling growth rates and average sum of cases.}

\item{level}{The confidence level for onset parameter estimates, a numeric value between 0 and 1.}

\item{disease_threshold}{A number specifying the threshold for considering a disease outbreak. Should be given as incidence if
\code{population} and \code{incidence_denominator} are in the \code{tsd} object else as cases. It defines the per time-step disease threshold that has to be surpassed to possibly trigger a seasonal
onset alarm. If the average observation count in a window of size k exceeds \code{disease_threshold}, a seasonal
onset alarm can be triggered.}

\item{family}{A character string specifying the family for modeling. Choose between 'poisson', or 'quasipoisson'.
Must be one of: character, family-generator, or family object.}

\item{na_fraction_allowed}{Numeric value between 0 and 1 specifying the fraction of observations in the window
of size k that are allowed to be NA or zero, i.e. without cases, in onset calculations.}

\item{season_start, season_end}{Integers giving the start and end weeks of the seasons to
stratify the observations by. If set to \code{NULL}, it means no stratification by season.}

\item{only_current_season}{Should the output only include results for the current season?}
}
\value{
A \code{tsd_onset} object containing:
\itemize{
\item 'reference_time': The time point for which the growth rate is estimated.
\item 'cases': The cases at reference time point.
\item 'population': The population at reference time point.
\item 'incidence': The incidence at reference time point.
\item 'season': The stratification of observables in corresponding seasons.
\item 'growth_rate': The estimated growth rate.
\item 'lower_growth_rate': The lower bound of the growth rate's confidence interval.
\item 'upper_growth_rate': The upper bound of the growth rate's confidence interval.
\item 'growth_warning': Logical. Is the growth rate significantly higher than zero?
\item 'average_observation_window': The average of cases or incidence within the time window.
\item 'average_observation_warning': Logical. Does the average observations exceed the disease threshold?
\item 'seasonal_onset_alarm': Logical. Is there a seasonal onset alarm?
\item 'skipped_window': Logical. Was the window skipped due to missing observations?
\item 'converged': Logical. Was the IWLS judged to have converged?
\item 'seasonal_onset': Logical. The first detected seasonal onset in the season.
\item Attributes: \code{time_interval} and \code{incidence_denominator}.
}
}
\description{
This function performs automated and early detection of seasonal epidemic onsets on a \code{tsd} object.
It estimates growth rates and calculates the average sum of cases in consecutive time intervals (\code{k}).
If the time series data includes \code{population} it will be used as offset to adjust the growth rate in the glm,
additionally the output will include incidence, population and average sum of incidence.
}
\examples{
# Create a tibble object from sample data
tsd_data <- to_time_series(
  cases = c(100, 120, 150, 180, 220, 270),
  time = seq(from = as.Date("2023-01-01"), by = "1 week", length.out = 6)
)

# Estimate seasonal onset with a 3-day window
seasonal_onset(
  tsd = tsd_data,
  k = 3,
  level = 0.975,
  disease_threshold = 5,
  na_fraction_allowed = 0.4,
  season_start = 21,
  season_end = 20,
  only_current_season = FALSE
)
}
