#' @title Run Weighted CIF Analysis
#' @description Given the list from \code{\link{prep_data_weighted_cif}}, run Weighted RMLT1 (recovery/discharge)
#'   and Weighted RMLT2 (death) at a user-specified time horizon \code{tau}.
#'
#' @param prepped A list returned by \code{prep_data_weighted_cif()}.
#' @param tau Numeric time horizon (e.g., 15 or 29).
#' @importFrom survival survfit Surv
#' @importFrom stats var cov pnorm
#' @details
#' - Weighted RMLT1 uses \code{eta=1} for recovery/discharge analysis.
#' - Weighted RMLT2 uses \code{eta=2} for death analysis.
#'
#' @return A list with formatted results for WRMLT1 and WRMLT2.
#' @export
do_weighted_cif_analysis <- function(prepped, tau){
  Treatment_death    <- prepped$Treatment.death
  Control_death      <- prepped$Control.death
  Treatment_discharge <- prepped$Treatment.discharge
  Control_discharge   <- prepped$Control.discharge

  # WRMLT1 (recovery/discharge analysis)
  WRMLT1 <- table_weighted(Treatment_discharge, Control_discharge, eta=1, tau)
  # WRMLT2 (death analysis)
  WRMLT2 <- table_weighted(Treatment_death, Control_death, eta=2, tau)

  # Format results function (internal)
  format_results <- function(mat, analysis_type) {
    # Calculate standard errors from confidence intervals
    se_group1 <- (mat[1,3] - mat[1,2]) / (2 * 1.96)
    se_group2 <- (mat[2,3] - mat[2,2]) / (2 * 1.96)

    # Create formatted results
    group_results <- data.frame(
      Group = c("Group1 (trt=1)", "Group2 (trt=0)"),
      Estimate = c(mat[1,1], mat[2,1]),
      se = c(se_group1, se_group2),
      Lower95 = c(mat[1,2], mat[2,2]),
      Upper95 = c(mat[1,3], mat[2,3]),
      stringsAsFactors = FALSE
    )

    # Add contrast results
    contrast_se <- (mat[3,3] - mat[3,2]) / (2 * 1.96)
    contrast_results <- data.frame(
      Contrast = "Difference",
      Estimate = mat[3,1],
      se = contrast_se,
      Lower95 = mat[3,2],
      Upper95 = mat[3,3],
      p_value = mat[3,4],
      stringsAsFactors = FALSE
    )

    # Return formatted list
    list(
      analysis_type = analysis_type,
      groups = group_results,
      contrast = contrast_results
    )
  }

  # Format both results
  formatted_wrmlt1 <- format_results(WRMLT1, "WRMLT1")
  formatted_wrmlt2 <- format_results(WRMLT2, "WRMLT2")

  return(list(
    WRMLT1 = formatted_wrmlt1,
    WRMLT2 = formatted_wrmlt2
  ))
}
