% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sc2interval.R
\name{sc2interval}
\alias{sc2interval}
\title{Statechange (sc) Timestamps to Intervals}
\usage{
sc2interval(
  dataset,
  Datetime.colname = Datetime,
  Statechange.colname = State,
  State.colname = State,
  Interval.colname = Interval,
  full = TRUE,
  starting.state = NA,
  output.dataset = TRUE,
  Datetime.keep = FALSE,
  length.restriction = 60 * 60 * 24
)
}
\arguments{
\item{dataset}{A light logger dataset. Expects a \code{dataframe}. If not imported
by \link{LightLogR}, take care to choose a sensible variable for the
\code{Datetime.colname}.}

\item{Datetime.colname}{column name that contains the datetime. Defaults to
\code{"Datetime"} which is automatically correct for data imported with
\link{LightLogR}. Expects a \code{symbol}. Needs to be part of the \code{dataset}. Must
be of type \code{POSIXct}.}

\item{Statechange.colname, Interval.colname, State.colname}{Column names that
do contain the name/description of the \verb{state change} and that will contain
the \code{Interval} and \code{State} (which are also the default). Expects a \code{symbol}. The
\code{Statechange} column needs do be part of the \code{dataset}.}

\item{full, starting.state}{These arguments handle the state on the first day
before the first state change and after the last state change on the last
day. If \code{full = TRUE}(the default, expects a \code{logical}), it will create an
interval on the first day from 00:00:00 up until the state change. This
interval will be given the state specified in \code{starting.state}, which is \code{NA}
by default, but can be any \code{character} scalar. It will further extend the
interval for the last state change until the end of the last given day
(more specifically until 00:00:00 the next day).}

\item{output.dataset}{should the output be a \code{data.frame} (Default \code{TRUE}) or
a vector with \code{hms} (\code{FALSE}) times? Expects a \code{logical} scalar.}

\item{Datetime.keep}{If \code{TRUE}, the original \code{Datetime} column will be kept.}

\item{length.restriction}{If the length between intervals is too great, the
interval state can be set to \code{NA}, which effectively produces a gap in the
data. This makes sense when intervals are implausibly wrong (e.g. someone
slept for 50 hours), because when this data is combined with light logger
data, e.g., through \code{\link[=interval2state]{interval2state()}}, metrics and visualizations will
remove the interval.}
}
\value{
One of
\itemize{
\item a \code{data.frame} object identical to \code{dataset} but with the interval instead of the datetime. The original \code{Statechange} column now indicates the \code{State} during the \code{Interval}.
\item a named \code{vector} with the intervals, where the names are the states
}
}
\description{
Takes an input of \code{datetimes} and \code{Statechanges} and creates a column with
\code{Intervals}. If \code{full = TRUE}, it will also create intervals for the day
prior to the first state change and after the last. If \code{output.dataset = FALSE} it will give a named vector, otherwise a \code{tibble}. The \verb{state change}
info requires a description or name of the state (like \code{"sleep"} or \code{"wake"},
or \code{"wear"}) that goes into effect at the given \code{Datetime}. Works for grouped
data so that it does not mix up intervals between participants. Missing data
should be explicit if at all possible. Also, the maximum allowed length of an
interval can be set, so that implicit missing timestamps after a set period
of times can be enforced.
}
\examples{
library(tibble)
library(lubridate)
library(dplyr)
sample <- tibble::tibble(Datetime = c("2023-08-15 6:00:00",
                                      "2023-08-15 23:00:00",
                                      "2023-08-16 6:00:00",
                                      "2023-08-16 22:00:00",
                                      "2023-08-17 6:30:00",
                                      "2023-08-18 1:00:00"),
                         State = rep(c("wake", "sleep"), 3),
                         Id = "Participant")
#intervals from sample
sc2interval(sample)

#compare sample (y) and intervals (x)
sc2interval(sample) \%>\%
 mutate(Datetime = int_start(Interval)) \%>\%
 dplyr::left_join(sample, by = c("Id", "State"),
                  relationship = "many-to-many") \%>\%
 head()

}
