% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/upset_vp.R
\name{upset_vp}
\alias{upset_vp}
\title{Visualization of VPA and HP Using UpSetVP Diagram}
\usage{
upset_vp(
  x,
  plot.hp = TRUE,
  order.part = "effect",
  decreasing.part = TRUE,
  order.var = TRUE,
  decreasing.var = TRUE,
  cutoff = -1,
  nVar = 30,
  col.width = 0.6,
  pch.size = 3,
  line.lwd = 0.5,
  show.effect = TRUE,
  effect.cex = 2.7,
  title.cex = 10,
  axis.cex = 8,
  height.ratio = c(2, 1),
  width.ratio = c(1, 3)
)
}
\arguments{
\item{x}{A \code{\link{rdacca.hp}} object, which contains the output of VPA and HP from canonical analysis.}

\item{plot.hp}{The default is \code{TRUE}, which plots the individual effect for each predictor on left column diagram. If \code{FALSE}, compute and plot the sum of unique effect and common effect for each predictor.}

\item{order.part}{How the VPA components in matrix layout should be ordered. Options include \code{"effect"} (order the intersections by their effects) or \code{"degree"} (sort by the number of predictors involved in the intersection), default is \code{"effect"}.}

\item{decreasing.part}{How the intersections in \code{order.part} should be ordered. Default is \code{TRUE}, \code{"effect"} is decreasing (from greatest to least) or \code{"degree"} is increasing (from least to greatest).}

\item{order.var}{The predictors in the matrix layout should be ordered by. Default is \code{TRUE}, which orders the predictors by their effects. IF \code{FALSE}, sort by the order of predictors in input data.}

\item{decreasing.var}{If \code{order.var=TRUE}, how the predictors should be ordered. Default is \code{TRUE}, from greatest to least.}

\item{cutoff}{Effects below \code{cutoff} will not be displayed, default is \code{-1}. Note: Negative effects due to adjustment of R-squared mean negligible contributions, but they are included in the computation of the total contribution of each predictor category.}

\item{nVar}{Number of components in VPA to plot, default is \code{30}.}

\item{col.width}{Width of bars in column diagram, default is \code{0.6}.}

\item{pch.size}{Size of points in matrix diagram, default is \code{3}.}

\item{line.lwd}{Width of lines in matrix diagram, default is \code{0.5}.}

\item{show.effect}{Show the relative importance of predictors (unique, common, or individual effects) above bars, default is \code{TRUE}.}

\item{effect.cex}{Font size of the effects, default is \code{2.7}.}

\item{title.cex}{Font size of axis titles, default is \code{10}.}

\item{axis.cex}{Font size of axis labels, default is \code{8}.}

\item{height.ratio}{Ratio between matrix and top column diagram, default is \code{c(2, 1)}.}

\item{width.ratio}{Ratio between matrix and left column diagram, default is \code{c(1, 3)}.}
}
\value{
\itemize{Returns a ggplot2.}
}
\description{
Visualization of variance partitioning analysis (VPA) and hierarchical partitioning (HP) with unlimited number of predictor variables (or matrices of predictors) using UpSet matrix layout.
}
\details{
UpSetVP diagram is an extension of UpSet technique (Conway et al. 2017; Lex et al. 2014) to canonical analysis and is used to visualize the object of \code{\link{rdacca.hp}} (Lai et al. 2022). The matrix layout enables the effective representation of relative importance of predictors, such as the unique effects and common effects in VPA, as well as additional summary statistics or individual effects in HP. UpSetVP diagram could, in principle, allow visualization of any number of predictor variables or groups of predictor variables. But considering the interpretability of data, we would like to recommend that the number of predictors (or groups of predictors) no more than 7.
}
\examples{
\donttest{
library(rdacca.hp)

## A simple example of partial dbRDA
data(baima.fun)
data(baima.env)

# Bray-Curtis index was used to calculate community composition dissimilarity
baima.fun.bray <- vegdist(baima.fun, method = "bray")

# The relative importance of individual soil properties on EcM fungal community compositionon
soil <- baima.env[c("pH", "TP", "TK", "AN", "AP", "AK")]
baima.soil.vp <- rdacca.hp(baima.fun.bray, soil,
  method = "dbRDA", var.part = TRUE, type = "adjR2")

# Plot unique, common, as well as individual effects
upset_vp(baima.soil.vp, plot.hp = TRUE)

## Example was referenced from Gong et al. (2022)
if(requireNamespace("adespatial", quietly = TRUE)) {

# Distance-based Moran's eigenvector maps (dbMEM) was used to extract spatial relationships
space.dbmem <- adespatial::dbmem(baima.env[c("latitude", "lontitude")])
  
  # The relative importance of groups of environmental factors on EcM fungal community composition
  env.list <- list(
    elevation = baima.env["altitude"],
    season = baima.env["season"],
    space = data.frame(space.dbmem)[1:2],
    host = baima.env[c("em.GR", "em.abun")],
    climate = baima.env["sea.MT"],
    soil = baima.env[c("pH", "TP", "TK", "AN", "AP", "AK")]
  )
  baima.env.vp <- rdacca.hp(baima.fun.bray, env.list,
    method = "dbRDA", var.part = TRUE, type = "adjR2")
  
  # Plot unique, common, as well as individual effects
  upset_vp(baima.env.vp, plot.hp = TRUE, order.part = "degree")
}
}
}
\references{
Conway J R, Lex A, Gehlenborg N. UpSetR: an R package for the visualization of intersecting sets and their properties. Bioinformatics, 2017, 33(18): 2938-2940.

Gong S, Feng B, Jian S P, et al. Elevation Matters More than Season in Shaping the Heterogeneity of Soil and Root Associated Ectomycorrhizal Fungal Community. Microbiology spectrum, 2022, 10(1): e01950-21.

Lai J, Zou Y, Zhang J, et al. Generalizing hierarchical and variation partitioning in multiple regression and canonical analyses using the rdacca.hp R package. Methods in Ecology and Evolution, 2022.

Lex A, Gehlenborg N, Strobelt H, et al. UpSet: visualization of intersecting sets. IEEE transactions on visualization and computer graphics, 2014, 20(12): 1983-1992.
}
