% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bsrr.R
\name{bsrr}
\alias{bsrr}
\title{Best subset ridge regression}
\usage{
bsrr(
  x,
  y,
  family = c("gaussian", "binomial", "poisson", "cox"),
  method = c("pgsection", "sequential", "psequential"),
  tune = c("gic", "ebic", "bic", "aic", "cv"),
  s.list,
  lambda.list = 0,
  s.min,
  s.max,
  lambda.min = 0.001,
  lambda.max = 100,
  nlambda = 100,
  always.include = NULL,
  screening.num = NULL,
  normalize = NULL,
  weight = NULL,
  max.iter = 20,
  warm.start = TRUE,
  nfolds = 5,
  group.index = NULL,
  seed = NULL
)
}
\arguments{
\item{x}{Input matrix, of dimension \eqn{n \times p}; each row is an observation
vector and each column is a predictor/feature/variable.}

\item{y}{The response variable, of \code{n} observations. For \code{family = "binomial"} should be
a factor with two levels. For \code{family="poisson"}, \code{y} should be a vector with positive integer.
 For \code{family = "cox"}, \code{y} should be a two-column matrix
with columns named \code{time} and \code{status}.}

\item{family}{One of the following models: \code{"gaussian"}, \code{"binomial"},
\code{"poisson"}, or \code{"cox"}. Depending on the response. Any unambiguous substring can be given.}

\item{method}{The method to be used to select the optimal model size and \eqn{L_2} shrinkage. For
\code{method = "sequential"}, we solve the best subset ridge regression
problem for each \code{s} in \code{1,2,...,s.max} and \eqn{\lambda} in \code{lambda.list}.
For \code{method = "pgsection"} and \code{"psequential"}, the Powell method is used to
solve the best subset ridge regression problem. Any unambiguous substring can be given.}

\item{tune}{The criterion for choosing the model size and \eqn{L_2} shrinkage
parameters. Available options are \code{"gic"}, \code{"ebic"}, \code{"bic"}, \code{"aic"} and \code{"cv"}.
Default is \code{"gic"}. \code{"cv"} is recommanded for BSRR.}

\item{s.list}{An increasing list of sequential values representing the model
sizes. Only used for \code{method = "sequential"}. Default is \code{1:min(p,
round(n/log(n)))}.}

\item{lambda.list}{A lambda sequence for \code{"bsrr"}. Only used for \code{method = "sequential"}. Default is
\code{exp(seq(log(100), log(0.01), length.out = 100))}.}

\item{s.min}{The minimum value of model sizes. Only used for \code{"psequential"} and \code{"pgsection"}. Default is 1.}

\item{s.max}{The maximum value of model sizes. Only used for \code{"psequential"} and \code{"pgsection"}.
Default is \code{min(p, round(n/log(n)))}.}

\item{lambda.min}{The minimum value of lambda. Only used for \code{method =
"psequential"} and \code{"pgsection"}. Default is \code{0.001}.}

\item{lambda.max}{The maximum value of lambda. Only used for \code{method =
"psequential"} and \code{"pgsection"}. Default is \code{100}.}

\item{nlambda}{The number of \eqn{\lambda}s for the Powell path with sequential line search method.
Only valid for \code{method = "psequential"}.}

\item{always.include}{An integer vector containing the indexes of variables that should always be included in the model.}

\item{screening.num}{Users can pre-exclude some irrelevant variables according to maximum marginal likelihood estimators before fitting a
model by passing an integer to \code{screening.num} and the sure independence screening will choose a set of variables of this size.
Then the active set updates are restricted on this subset.}

\item{normalize}{Options for normalization. \code{normalize = 0} for
no normalization. Setting \code{normalize = 1} will
only subtract the mean of columns of \code{x}.
\code{normalize = 2} for scaling the columns of \code{x} to have \eqn{\sqrt n} norm.
\code{normalize = 3} for subtracting the means of the columns of \code{x} and \code{y}, and also
normalizing the columns of \code{x} to have \eqn{\sqrt n} norm.
If \code{normalize = NULL}, by default, \code{normalize} will be set \code{1} for \code{"gaussian"},
\code{2} for \code{"binomial"} and \code{"poisson"}, \code{3} for \code{"cox"}.}

\item{weight}{Observation weights. Default is \code{1} for each observation.}

\item{max.iter}{The maximum number of iterations in the \code{bsrr} function.
In most of the case, only a few steps can guarantee the convergence. Default
is \code{20}.}

\item{warm.start}{Whether to use the last solution as a warm start. Default
is \code{TRUE}.}

\item{nfolds}{The number of folds in cross-validation. Default is \code{5}.}

\item{group.index}{A vector of integers indicating the which group each variable is in.
For variables in the same group, they should be located in adjacent columns of \code{x}
and their corresponding index in \code{group.index} should be the same.
Denote the first group as \code{1}, the second \code{2}, etc.
If you do not fit a model with a group structure,
please set \code{group.index = NULL}. Default is \code{NULL}.}

\item{seed}{Seed to be used to divide the sample into K cross-validation folds. Default is \code{NULL}.}
}
\value{
A list with class attribute 'bsrr' and named components:
\item{beta}{The best fitting coefficients.}
 \item{coef0}{The best fitting
intercept.}
\item{loss}{The training loss of the best fitting model.}
\item{ic}{The information criterion of the best fitting model when model
selection is based on a certain information criterion.} \item{cvm}{The mean
cross-validated error for the best fitting model when model selection is
based on the cross-validation.}

\item{lambda}{The lambda chosen for the best fitting model}
\item{beta.all}{For \code{bsrr} objects obtained by \code{gsection}, \code{pgsection}
and \code{psequential}, \code{beta.all} is a matrix with each column be the coefficients
of the model in each iterative step in the tuning path.
For \code{bsrr} objects obtained by \code{sequential} method,
A list of the best fitting coefficients of size
\code{s=0,1,...,p} and \eqn{\lambda} in \code{lambda.list} with the
smallest loss function. For \code{"bsrr"} objects of \code{"bsrr"} type, the fitting coefficients of the
\eqn{i^{th} \lambda} and the \eqn{j^{th}} \code{s} are at the \eqn{i^{th}}
list component's \eqn{j^{th}} column.}
\item{coef0.all}{For \code{bsrr} objects obtained from \code{gsection}, \code{pgsection} and \code{psequential},
\code{coef0.all} contains the intercept for the model in each iterative step in the tuning path.
For \code{bsrr} objects obtained from \code{sequential} path,
\code{coef0.all} contains the best fitting
intercepts of size \eqn{s=0,1,\dots,p} and \eqn{\lambda} in
\code{lambda.list} with the smallest loss function.}
\item{loss.all}{For \code{bsrr} objects obtained from \code{gsection}, \code{pgsection} and \code{psequential},
\code{loss.all} contains the training loss of the model in each iterative step in the tuning path.
For \code{bsrr} objects obtained from \code{sequential} path, this is a
list of the training loss of the best fitting intercepts of model size
\eqn{s=0,1,\dots,p} and \eqn{\lambda} in \code{lambda.list}. For \code{"bsrr"} object obtained by \code{"bsrr"},
the training loss of the \eqn{i^{th} \lambda} and the \eqn{j^{th}} \code{s}
is at the \eqn{i^{th}} list component's \eqn{j^{th}} entry.}
\item{ic.all}{For \code{bsrr} objects obtained from \code{gsection}, \code{pgsection} and \code{psequential},
\code{ic.all} contains the values of the chosen information criterion of the model in each iterative step in the tuning path.
For \code{bsrr} objects obtained from \code{sequential} path, this is a
matrix of the values of the chosen information criterion of model size \eqn{s=0,1,\dots,p}
and \eqn{\lambda} in \code{lambda.list} with the smallest loss function. For \code{"bsrr"} object obtained by \code{"bsrr"},
the training loss of the \eqn{i^{th} \lambda} and the \eqn{j^{th}}
\code{s} is at the \eqn{i^{th}} row \eqn{j^{th}} column. Only available when
model selection is based on a certain information criterion.}

\item{cvm.all}{For \code{bsrr} objects obtained from \code{gsection}, \code{pgsection} and \code{psequential},
\code{cvm.all} contains the mean cross-validation error of the model in each iterative step in the tuning path.
For \code{bsrr} objects obtained from \code{sequential} path, this is a
 matrix of the mean cross-validation error of model size
\eqn{s=0,1,\dots,p} and \eqn{\lambda} in \code{lambda.list} with the
smallest loss function. For \code{"bsrr"} object obtained by \code{"bsrr"}, the training loss of the \eqn{i^{th}
\lambda} and the \eqn{j^{th}} \code{s} is at the \eqn{i^{th}} row
\eqn{j^{th}} column. Only available when model selection is based on the
cross-validation.}
\item{lambda.all}{The lambda chosen for each step in \code{pgsection} and \code{psequential}.}
\item{family}{Type of the model.}
\item{s.list}{The input
\code{s.list}.} \item{nsample}{The sample size.}
\item{type}{Either \code{"bss"} or \code{"bsrr"}.}
\item{method}{Method used for tuning parameters selection.}
\item{ic.type}{The criterion of model selection.}
}
\description{
Best subset ridge regression for generalized linear model and Cox's proportional
model.
}
\details{
The best ridge regression problem with model size \eqn{s} and the shrinkage parameter \eqn{\lambda} is
\deqn{\min_\beta -2 \log L(\beta) + \lambda\Vert\beta\Vert_2^2 \;\;{\rm
s.t.}\;\; \|\beta\|_0 \leq s.} In the GLM case, \eqn{\log L(\beta)} is the
log likelihood function; In the Cox model, \eqn{\log L(\beta)} is the log
partial likelihood function.

The best subset selection problem is a special case of the best ridge regression problem
with the shrinkage \eqn{\lambda=0}.

For each candidate model size and \eqn{\lambda}, the best subset ridge regression
problems are solved by the \eqn{L_2} penalized primal-dual active set (PDAS) algorithm, see Wen et
al (2020) for details. This algorithm
utilizes an active set updating strategy via primal and dual variables and
fits the sub-model by exploiting the fact that their support sets are
non-overlap and complementary. For the case of \code{method = "sequential"}
if \code{warm.start = "TRUE"}, we run the PDAS algorithm for a list of
sequential model sizes and use the estimate from the last iteration as a
warm start. For the case of
\code{ method = "psequential"} and \code{method = "pgsection"}, the Powell method
using a sequential line search method or a golden section search technique is
used for parameters determination.
}
\examples{

#-------------------linear model----------------------#
# Generate simulated data
n <- 200
p <- 20
k <- 5
rho <- 0.4
seed <- 10
Tbeta <- rep(0, p)
Tbeta[1:k*floor(p/k):floor(p/k)] <- rep(1, k)
Data <- gen.data(n, p, k, rho, family = "gaussian", beta = Tbeta, seed = seed)
x <- Data$x[1:140, ]
y <- Data$y[1:140]
x_new <- Data$x[141:200, ]
y_new <- Data$y[141:200]
lambda.list <- exp(seq(log(5), log(0.1), length.out = 10))
lm.bsrr <- bsrr(x, y, method = "pgsection")
coef(lm.bsrr)
print(lm.bsrr)
summary(lm.bsrr)
pred.bsrr <- predict(lm.bsrr, newx = x_new)

# generate plots
plot(lm.bsrr)
#-------------------logistic model----------------------#
#Generate simulated data
Data <- gen.data(n, p, k, rho, family = "binomial", beta = Tbeta, seed = seed)

x <- Data$x[1:140, ]
y <- Data$y[1:140]
x_new <- Data$x[141:200, ]
y_new <- Data$y[141:200]
lambda.list <- exp(seq(log(5), log(0.1), length.out = 10))
logi.bsrr <- bsrr(x, y, family = "binomial", lambda.list = lambda.list)
coef(logi.bsrr)
print(logi.bsrr)
summary(logi.bsrr)
pred.bsrr <- predict(logi.bsrr, newx = x_new)

# generate plots
plot(logi.bsrr)
#-------------------poisson model----------------------#
Data <- gen.data(n, p, k, rho=0.3, family = "poisson", beta = Tbeta, seed = seed)

x <- Data$x[1:140, ]
y <- Data$y[1:140]
x_new <- Data$x[141:200, ]
y_new <- Data$y[141:200]
lambda.list <- exp(seq(log(5), log(0.1), length.out = 10))
poi.bsrr <- bsrr(x, y, family = "poisson", lambda.list = lambda.list)
coef(poi.bsrr)
print(poi.bsrr)
summary(poi.bsrr)
pred.bsrr <- predict(poi.bsrr, newx = x_new)

# generate plots
plot(poi.bsrr)
#-------------------coxph model----------------------#
#Generate simulated data
Data <- gen.data(n, p, k, rho, family = "cox", scal = 10, beta = Tbeta)

x <- Data$x[1:140, ]
y <- Data$y[1:140, ]
x_new <- Data$x[141:200, ]
y_new <- Data$y[141:200, ]
lambda.list <- exp(seq(log(5), log(0.1), length.out = 10))
cox.bsrr <- bsrr(x, y, family = "cox", lambda.list = lambda.list)
coef(cox.bsrr)
print(cox.bsrr)
summary(cox.bsrr)
pred.bsrr <- predict(cox.bsrr, newx = x_new)

# generate plots
plot(cox.bsrr)

#----------------------High dimensional linear models--------------------#
\dontrun{
data <- gen.data(n, p = 1000, k, family = "gaussian", seed = seed)

# Best subset selection with SIS screening
lm.high <- bsrr(data$x, data$y, screening.num = 100)
}

#-------------------group selection----------------------#
beta <- rep(c(rep(1,2),rep(0,3)), 4)
Data <- gen.data(200, 20, 5, rho=0.4, beta = beta, seed =10)
x <- Data$x
y <- Data$y

group.index <- c(rep(1, 2), rep(2, 3), rep(3, 2), rep(4, 3),
                rep(5, 2), rep(6, 3), rep(7, 2), rep(8, 3))
lm.groupbsrr <- bsrr(x, y, s.min = 1, s.max = 8, group.index = group.index)
coef(lm.groupbsrr)
print(lm.groupbsrr)
summary(lm.groupbsrr)
pred.groupl0l2 <- predict(lm.groupbsrr, newx = x_new)
#-------------------include specified variables----------------------#
Data <- gen.data(n, p, k, rho, family = "gaussian", beta = Tbeta, seed = seed)
lm.bsrr <- bsrr(Data$x, Data$y, always.include = 2)

}
\seealso{
\code{\link{plot.bsrr}}, \code{\link{summary.bsrr}},
\code{\link{coef.bsrr}}, \code{\link{predict.bsrr}}.
}
\author{
Liyuan Hu, Kangkang Jiang, Yanhang Zhang, Jin Zhu, Canhong Wen and Xueqin Wang.
}
