% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smap.R
\name{simap}
\alias{simap}
\alias{simap_vec}
\alias{simap_lgl}
\alias{simap_int}
\alias{simap_dbl}
\alias{simap_chr}
\alias{simap_structure}
\title{Map Functions Over Glycan Structure Vectors with Indices}
\usage{
simap(.x, .f, ...)

simap_vec(.x, .f, ..., .ptype = NULL)

simap_lgl(.x, .f, ...)

simap_int(.x, .f, ...)

simap_dbl(.x, .f, ...)

simap_chr(.x, .f, ...)

simap_structure(.x, .f, ...)
}
\arguments{
\item{.x}{A glycan structure vector (glyrepr_structure).}

\item{.f}{A function that takes an igraph object (from \code{.x}) and an index/name,
returning a result.
Can be a function, purrr-style lambda (\code{~ paste(.x, .y)}), or a character string naming a function.}

\item{...}{Additional arguments passed to \code{.f}.}

\item{.ptype}{A prototype for the return type (for \code{simap_vec}).}
}
\value{
\itemize{
\item \code{simap()}: A list
\item \code{simap_vec()}: An atomic vector of type specified by \code{.ptype}
\item \code{simap_lgl()}: Returns a logical vector
\item \code{simap_int()}: Returns an integer vector
\item \code{simap_dbl()}: Returns a double vector
\item \code{simap_chr()}: Returns a character vector
\item \code{simap_structure()}: A new glyrepr_structure object
}
}
\description{
These functions apply a function to each unique structure in a glycan structure vector
along with their corresponding indices,
taking advantage of hash-based deduplication to avoid redundant computation.
Similar to purrr imap functions, but optimized for glycan structure vectors.
}
\details{
These functions only compute \code{.f} once for each unique combination of structure and corresponding
index/name, then map the results back to the original vector positions. This is much more efficient
than applying \code{.f} to each element individually when there are duplicate structures.

\strong{IMPORTANT PERFORMANCE NOTE:}
Due to the inclusion of position indices, \code{simap} functions have \strong{O(total_structures)}
time complexity because each position creates a unique combination, even with identical structures.

\strong{Alternative:} Consider \code{smap()} functions if position information is not required.

The index passed to \code{.f} is the position in the original vector (1-based).
If the vector has names, the names are passed instead of indices.

\strong{Return Types:}
\itemize{
\item \code{simap()}: Returns a list with the same length as \code{.x}
\item \code{simap_vec()}: Returns an atomic vector with the same length as \code{.x}
\item \code{simap_lgl()}: Returns a logical vector
\item \code{simap_int()}: Returns an integer vector
\item \code{simap_dbl()}: Returns a double vector
\item \code{simap_chr()}: Returns a character vector
\item \code{simap_structure()}: Returns a new glycan structure vector (\code{.f} must return igraph objects)
}
}
\examples{
# Create structure vectors with duplicates
core1 <- o_glycan_core_1()
core2 <- n_glycan_core()
structures <- glycan_structure(core1, core2, core1)  # core1 appears twice

# Map a function that uses both structure and index
simap_chr(structures, function(g, i) paste0("Structure_", i, "_vcount_", igraph::vcount(g)))

# Use purrr-style lambda functions  
simap_chr(structures, ~ paste0("Pos", .y, "_vertices", igraph::vcount(.x)))

}
