/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Cmodule.h" 
#define H5F_FRIEND     

#include "H5private.h"   
#include "H5Cpkg.h"      
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5FLprivate.h" 
#include "H5MFprivate.h" 
#include "H5MMprivate.h" 
#include "H5SLprivate.h" 

#if H5C_DO_MEMORY_SANITY_CHECKS
#define H5C_IMAGE_EXTRA_SPACE  8
#define H5C_IMAGE_SANITY_VALUE "DeadBeef"
#else 
#define H5C_IMAGE_EXTRA_SPACE 0
#endif 

typedef H5C_cache_entry_t *H5C_cache_entry_ptr_t;

static herr_t H5C__pin_entry_from_client(H5C_t *cache_ptr, H5C_cache_entry_t *entry_ptr);
static herr_t H5C__unpin_entry_real(H5C_t *cache_ptr, H5C_cache_entry_t *entry_ptr, bool update_rp);
static herr_t H5C__unpin_entry_from_client(H5C_t *cache_ptr, H5C_cache_entry_t *entry_ptr, bool update_rp);
static herr_t H5C__generate_image(H5F_t *f, H5C_t *cache_ptr, H5C_cache_entry_t *entry_ptr);
static herr_t H5C__discard_single_entry(H5F_t *f, H5C_t *cache_ptr, H5C_cache_entry_t *entry_ptr,
                                        bool destroy_entry, bool free_file_space,
                                        bool suppress_image_entry_frees);
static herr_t H5C__verify_len_eoa(H5F_t *f, const H5C_class_t *type, haddr_t addr, size_t *len, bool actual);
static void  *H5C__load_entry(H5F_t *f,
#ifdef H5_HAVE_PARALLEL
                             bool coll_access,
#endif 
                             const H5C_class_t *type, haddr_t addr, void *udata);
static herr_t H5C__mark_flush_dep_dirty(H5C_cache_entry_t *entry);
static herr_t H5C__mark_flush_dep_clean(H5C_cache_entry_t *entry);
static herr_t H5C__mark_flush_dep_serialized(H5C_cache_entry_t *entry);
static herr_t H5C__mark_flush_dep_unserialized(H5C_cache_entry_t *entry);
#ifndef NDEBUG
static void H5C__assert_flush_dep_nocycle(const H5C_cache_entry_t *entry,
                                          const H5C_cache_entry_t *base_entry);
#endif
static herr_t H5C__destroy_pf_entry_child_flush_deps(H5C_t *cache_ptr, H5C_cache_entry_t *pf_entry_ptr,
                                                     H5C_cache_entry_t **fd_children);
static herr_t H5C__deserialize_prefetched_entry(H5F_t *f, H5C_t *cache_ptr, H5C_cache_entry_t **entry_ptr_ptr,
                                                const H5C_class_t *type, haddr_t addr, void *udata);

H5FL_SEQ_DEFINE_STATIC(H5C_cache_entry_ptr_t);

static herr_t
H5C__pin_entry_from_client(H5C_t
#if !H5C_COLLECT_CACHE_STATS
                               H5_ATTR_UNUSED
#endif
                                             *cache_ptr,
                           H5C_cache_entry_t *entry_ptr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(cache_ptr);
    assert(entry_ptr);
    assert(entry_ptr->is_protected);

    
    if (entry_ptr->is_pinned) {
        
        if (entry_ptr->pinned_from_client)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTPIN, FAIL, "entry is already pinned");
    } 
    else {
        entry_ptr->is_pinned = true;

        H5C__UPDATE_STATS_FOR_PIN(cache_ptr, entry_ptr);
    } 

    
    entry_ptr->pinned_from_client = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__unpin_entry_real(H5C_t *cache_ptr, H5C_cache_entry_t *entry_ptr, bool update_rp)
{
    herr_t ret_value = SUCCEED; 

#ifdef H5C_DO_SANITY_CHECKS
    FUNC_ENTER_PACKAGE
#else
    FUNC_ENTER_PACKAGE_NOERR
#endif

    
    assert(cache_ptr);
    assert(entry_ptr);
    assert(entry_ptr->is_pinned);

    
    if (update_rp && !entry_ptr->is_protected)
        H5C__UPDATE_RP_FOR_UNPIN(cache_ptr, entry_ptr, FAIL);

    
    entry_ptr->is_pinned = false;

    
    H5C__UPDATE_STATS_FOR_UNPIN(cache_ptr, entry_ptr);

#ifdef H5C_DO_SANITY_CHECKS
done:
#endif
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__unpin_entry_from_client(H5C_t *cache_ptr, H5C_cache_entry_t *entry_ptr, bool update_rp)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(cache_ptr);
    assert(entry_ptr);

    
    if (!entry_ptr->is_pinned)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPIN, FAIL, "entry isn't pinned");
    if (!entry_ptr->pinned_from_client)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPIN, FAIL, "entry wasn't pinned by cache client");

    
    if (!entry_ptr->pinned_from_cache)
        if (H5C__unpin_entry_real(cache_ptr, entry_ptr, update_rp) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPIN, FAIL, "can't unpin entry");

    
    entry_ptr->pinned_from_client = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__generate_image(H5F_t *f, H5C_t *cache_ptr, H5C_cache_entry_t *entry_ptr)
{
    haddr_t  new_addr        = HADDR_UNDEF;
    haddr_t  old_addr        = HADDR_UNDEF;
    size_t   new_len         = 0;
    unsigned serialize_flags = H5C__SERIALIZE_NO_FLAGS_SET;
    herr_t   ret_value       = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(cache_ptr);
    assert(entry_ptr);
    assert(!entry_ptr->image_up_to_date);
    assert(entry_ptr->is_dirty);
    assert(!entry_ptr->is_protected);
    assert(entry_ptr->type);

    
    old_addr = entry_ptr->addr;

    
    if ((entry_ptr->type->pre_serialize) &&
        ((entry_ptr->type->pre_serialize)(f, (void *)entry_ptr, entry_ptr->addr, entry_ptr->size, &new_addr,
                                          &new_len, &serialize_flags) < 0))
        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "unable to pre-serialize entry");

    
    if (serialize_flags != H5C__SERIALIZE_NO_FLAGS_SET) {
        
        if (serialize_flags & ~(H5C__SERIALIZE_RESIZED_FLAG | H5C__SERIALIZE_MOVED_FLAG))
            HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "unknown serialize flag(s)");

#ifdef H5_HAVE_PARALLEL
        
        if (cache_ptr->aux_ptr != NULL)
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "resize/move in serialize occurred in parallel case");
#endif

        
        if (serialize_flags & H5C__SERIALIZE_RESIZED_FLAG) {
            
            assert(new_len > 0);

            
            if (NULL ==
                (entry_ptr->image_ptr = H5MM_realloc(entry_ptr->image_ptr, new_len + H5C_IMAGE_EXTRA_SPACE)))
                HGOTO_ERROR(H5E_CACHE, H5E_CANTALLOC, FAIL,
                            "memory allocation failed for on disk image buffer");

#if H5C_DO_MEMORY_SANITY_CHECKS
            H5MM_memcpy(((uint8_t *)entry_ptr->image_ptr) + new_len, H5C_IMAGE_SANITY_VALUE,
                        H5C_IMAGE_EXTRA_SPACE);
#endif 

            
            H5C__UPDATE_STATS_FOR_ENTRY_SIZE_CHANGE(cache_ptr, entry_ptr, new_len);

            
            H5C__UPDATE_INDEX_FOR_SIZE_CHANGE(cache_ptr, entry_ptr->size, new_len, entry_ptr,
                                              !entry_ptr->is_dirty, FAIL);

            
            H5C__UPDATE_RP_FOR_SIZE_CHANGE(cache_ptr, entry_ptr, new_len, FAIL);

            
            assert(entry_ptr->is_dirty);
            assert((entry_ptr->in_slist) || (!cache_ptr->slist_enabled));

            H5C__UPDATE_SLIST_FOR_SIZE_CHANGE(cache_ptr, entry_ptr->size, new_len);

            
            entry_ptr->size = new_len;
        } 

        
        if (serialize_flags & H5C__SERIALIZE_MOVED_FLAG) {
            
            H5C__UPDATE_STATS_FOR_MOVE(cache_ptr, entry_ptr);

            
            if (entry_ptr->addr == old_addr) {
                
                H5C__DELETE_FROM_INDEX(cache_ptr, entry_ptr, FAIL);
                H5C__REMOVE_ENTRY_FROM_SLIST(cache_ptr, entry_ptr, false, FAIL);

                
                entry_ptr->addr = new_addr;

                
                H5C__INSERT_IN_INDEX(cache_ptr, entry_ptr, FAIL);
                H5C__INSERT_ENTRY_IN_SLIST(cache_ptr, entry_ptr, FAIL);
            }    
            else 
                assert(entry_ptr->addr == new_addr);
        } 
    }     

    
    if (entry_ptr->type->serialize(f, entry_ptr->image_ptr, entry_ptr->size, (void *)entry_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "unable to serialize entry");

#if H5C_DO_MEMORY_SANITY_CHECKS
    assert(0 == memcmp(((uint8_t *)entry_ptr->image_ptr) + entry_ptr->size, H5C_IMAGE_SANITY_VALUE,
                       H5C_IMAGE_EXTRA_SPACE));
#endif 

    entry_ptr->image_up_to_date = true;

    
    assert(entry_ptr->flush_dep_nunser_children == 0);

    if (entry_ptr->flush_dep_nparents > 0)
        if (H5C__mark_flush_dep_serialized(entry_ptr) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                        "Can't propagate serialization status to fd parents");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C__flush_single_entry(H5F_t *f, H5C_cache_entry_t *entry_ptr, unsigned flags)
{
    H5C_t  *cache_ptr;                 
    bool    destroy;                   
    bool    clear_only;                
    bool    free_file_space;           
    bool    take_ownership;            
    bool    del_from_slist_on_destroy; 
    bool    during_flush;              
    bool    write_entry;               
    bool    destroy_entry;             
    bool    generate_image;            
    bool    update_page_buffer;        
    bool    was_dirty;
    bool    suppress_image_entry_writes = false;
    bool    suppress_image_entry_frees  = false;
    haddr_t entry_addr                  = HADDR_UNDEF;
    herr_t  ret_value                   = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(f);
    cache_ptr = f->shared->cache;
    assert(cache_ptr);
    assert(entry_ptr);
    assert(entry_ptr->ring != H5C_RING_UNDEFINED);
    assert(entry_ptr->type);

    
    destroy                   = ((flags & H5C__FLUSH_INVALIDATE_FLAG) != 0);
    clear_only                = ((flags & H5C__FLUSH_CLEAR_ONLY_FLAG) != 0);
    free_file_space           = ((flags & H5C__FREE_FILE_SPACE_FLAG) != 0);
    take_ownership            = ((flags & H5C__TAKE_OWNERSHIP_FLAG) != 0);
    del_from_slist_on_destroy = ((flags & H5C__DEL_FROM_SLIST_ON_DESTROY_FLAG) != 0);
    during_flush              = ((flags & H5C__DURING_FLUSH_FLAG) != 0);
    generate_image            = ((flags & H5C__GENERATE_IMAGE_FLAG) != 0);
    update_page_buffer        = ((flags & H5C__UPDATE_PAGE_BUFFER_FLAG) != 0);

    
    if (take_ownership)
        destroy_entry = false;
    else
        destroy_entry = destroy;

    
    if (entry_ptr->is_dirty && !clear_only)
        write_entry = true;
    else
        write_entry = false;

    
    if (cache_ptr->close_warning_received && cache_ptr->image_ctl.generate_image &&
        cache_ptr->num_entries_in_image > 0 && cache_ptr->image_entries != NULL) {

        
        assert(entry_ptr->image_up_to_date || !(entry_ptr->include_in_image));
        assert(entry_ptr->image_ptr || !(entry_ptr->include_in_image));
        assert((!clear_only) || !(entry_ptr->include_in_image));
        assert((!take_ownership) || !(entry_ptr->include_in_image));
        assert((!free_file_space) || !(entry_ptr->include_in_image));

        suppress_image_entry_frees = true;

        if (cache_ptr->image_ctl.flags & H5C_CI__SUPRESS_ENTRY_WRITES)
            suppress_image_entry_writes = true;
    } 

    
#ifdef H5C_DO_SANITY_CHECKS
    if (cache_ptr->slist_enabled) {
        if (entry_ptr->in_slist) {
            assert(entry_ptr->is_dirty);
            if (!entry_ptr->is_dirty)
                HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "entry in slist failed sanity checks");
        } 
    }
    else 
        assert(!entry_ptr->in_slist);
#endif 

    if (entry_ptr->is_protected)
        
        HGOTO_ERROR(H5E_CACHE, H5E_PROTECT, FAIL, "Attempt to flush a protected entry");

    
    entry_ptr->flush_in_progress = true;

    
    was_dirty = entry_ptr->is_dirty;

    
    if (write_entry || generate_image) {
        assert(entry_ptr->is_dirty);
        if (NULL == entry_ptr->image_ptr) {
            if (NULL == (entry_ptr->image_ptr = H5MM_malloc(entry_ptr->size + H5C_IMAGE_EXTRA_SPACE)))
                HGOTO_ERROR(H5E_CACHE, H5E_CANTALLOC, FAIL,
                            "memory allocation failed for on disk image buffer");

#if H5C_DO_MEMORY_SANITY_CHECKS
            H5MM_memcpy(((uint8_t *)entry_ptr->image_ptr) + entry_ptr->size, H5C_IMAGE_SANITY_VALUE,
                        H5C_IMAGE_EXTRA_SPACE);
#endif 

        } 

        if (!entry_ptr->image_up_to_date) {
            
            assert(!entry_ptr->prefetched);

            
            if (H5C__generate_image(f, cache_ptr, entry_ptr) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTGET, FAIL, "can't generate entry's image");
        } 
    }     

    
    if (write_entry) {
        assert(entry_ptr->is_dirty);

#ifdef H5C_DO_SANITY_CHECKS
        if (cache_ptr->check_write_permitted && !cache_ptr->write_permitted)
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Write when writes are always forbidden!?!?!");
#endif 

        
        if ((!suppress_image_entry_writes || !entry_ptr->include_in_image) &&
            ((entry_ptr->type->flags & H5C__CLASS_SKIP_WRITES) == 0)) {
            H5FD_mem_t mem_type = H5FD_MEM_DEFAULT;

#ifdef H5_HAVE_PARALLEL
            if (cache_ptr->coll_write_list) {
                if (H5SL_insert(cache_ptr->coll_write_list, entry_ptr, &entry_ptr->addr) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTINSERT, FAIL, "unable to insert skip list item");
            } 
            else {
#endif 
                if (entry_ptr->prefetched) {
                    assert(entry_ptr->type->id == H5AC_PREFETCHED_ENTRY_ID);
                    mem_type = cache_ptr->class_table_ptr[entry_ptr->prefetch_type_id]->mem_type;
                } 
                else
                    mem_type = entry_ptr->type->mem_type;

                if (H5F_block_write(f, mem_type, entry_ptr->addr, entry_ptr->size, entry_ptr->image_ptr) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "Can't write image to file");
#ifdef H5_HAVE_PARALLEL
            }
#endif    
        } 

        
        if (entry_ptr->type->notify &&
            (entry_ptr->type->notify)(H5C_NOTIFY_ACTION_AFTER_FLUSH, entry_ptr) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL, "can't notify client of entry flush");
    } 

    

    
    if (clear_only) {
        
        if (was_dirty)
            H5C__UPDATE_STATS_FOR_CLEAR(cache_ptr, entry_ptr);
    }
    else if (write_entry) {
        assert(was_dirty);

        
        H5C__UPDATE_STATS_FOR_FLUSH(cache_ptr, entry_ptr);
    } 

    

    
    if (destroy) {
        
        if (take_ownership)
            assert(!destroy_entry);
        else
            assert(destroy_entry);

        assert(!entry_ptr->is_pinned);

        
        H5C__UPDATE_STATS_FOR_EVICTION(cache_ptr, entry_ptr, take_ownership);

        
        if (entry_ptr->type->notify &&
            (entry_ptr->type->notify)(H5C_NOTIFY_ACTION_BEFORE_EVICT, entry_ptr) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL, "can't notify client about entry to evict");

        
        H5C__DELETE_FROM_INDEX(cache_ptr, entry_ptr, FAIL);

        if (entry_ptr->in_slist && del_from_slist_on_destroy)
            H5C__REMOVE_ENTRY_FROM_SLIST(cache_ptr, entry_ptr, during_flush, FAIL);

#ifdef H5_HAVE_PARALLEL
        
        if (entry_ptr->coll_access) {
            entry_ptr->coll_access = false;
            H5C__REMOVE_FROM_COLL_LIST(cache_ptr, entry_ptr, FAIL);
        } 
#endif    

        H5C__UPDATE_RP_FOR_EVICTION(cache_ptr, entry_ptr, FAIL);

        
        if (H5C__untag_entry(cache_ptr, entry_ptr) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTREMOVE, FAIL, "can't remove entry from tag list");

        
        assert(entry_ptr->flush_dep_nparents == 0);
        assert(entry_ptr->flush_dep_nchildren == 0);
    } 
    else {
        assert(clear_only || write_entry);
        assert(entry_ptr->is_dirty);
        assert((!cache_ptr->slist_enabled) || (entry_ptr->in_slist));

        
        H5C__UPDATE_RP_FOR_FLUSH(cache_ptr, entry_ptr, FAIL);
        H5C__REMOVE_ENTRY_FROM_SLIST(cache_ptr, entry_ptr, during_flush, FAIL);

        
        entry_ptr->is_dirty = false;

        H5C__UPDATE_INDEX_FOR_ENTRY_CLEAN(cache_ptr, entry_ptr, FAIL);

        
        if (was_dirty) {
            
            if (entry_ptr->type->notify &&
                (entry_ptr->type->notify)(H5C_NOTIFY_ACTION_ENTRY_CLEANED, entry_ptr) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                            "can't notify client about entry dirty flag cleared");

            
            if (entry_ptr->flush_dep_ndirty_children != 0)
                assert(entry_ptr->flush_dep_ndirty_children == 0);
            if (entry_ptr->flush_dep_nparents > 0)
                if (H5C__mark_flush_dep_clean(entry_ptr) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKCLEAN, FAIL, "Can't propagate flush dep clean flag");
        } 
    }     

    
    entry_ptr->flush_in_progress = false;

    
    entry_addr = entry_ptr->addr;

    
    if (destroy) {
        
        assert(destroy_entry != take_ownership);

        if (H5C__discard_single_entry(f, cache_ptr, entry_ptr, destroy_entry, free_file_space,
                                      suppress_image_entry_frees) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTFREE, FAIL, "can't discard cache entry");
    }

    
    if (update_page_buffer) {
        
        assert(!destroy);
        assert(entry_ptr->image_ptr);

        if (f->shared->page_buf && (f->shared->page_buf->page_size >= entry_ptr->size))
            if (H5PB_update_entry(f->shared->page_buf, entry_ptr->addr, entry_ptr->size,
                                  entry_ptr->image_ptr) > 0)
                HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Failed to update PB with metadata cache");
    } 

    if (cache_ptr->log_flush)
        if ((cache_ptr->log_flush)(cache_ptr, entry_addr, was_dirty, flags) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "log_flush callback failed");

done:
    assert((ret_value != SUCCEED) || (destroy_entry) || (!entry_ptr->flush_in_progress));
    assert((ret_value != SUCCEED) || (destroy_entry) || (take_ownership) || (!entry_ptr->is_dirty));

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__discard_single_entry(H5F_t *f, H5C_t *cache_ptr, H5C_cache_entry_t *entry_ptr, bool destroy_entry,
                          bool free_file_space, bool suppress_image_entry_frees)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(cache_ptr);
    assert(entry_ptr);

    
    assert(0 == entry_ptr->flush_dep_nparents);

    
    if (suppress_image_entry_frees && entry_ptr->include_in_image)
        entry_ptr->image_ptr = NULL;
    else if (entry_ptr->image_ptr != NULL)
        entry_ptr->image_ptr = H5MM_xfree(entry_ptr->image_ptr);

    
    if (!entry_ptr->prefetched) {
        assert(0 == entry_ptr->fd_parent_count);
        assert(NULL == entry_ptr->fd_parent_addrs);
    } 

    
    if (free_file_space) {
        hsize_t fsf_size;

        
        assert(H5_addr_defined(entry_ptr->addr));
        assert(!H5F_IS_TMP_ADDR(f, entry_ptr->addr));
#ifndef NDEBUG
        {
            size_t curr_len;

            
            entry_ptr->type->image_len((void *)entry_ptr, &curr_len);
            assert(curr_len == entry_ptr->size);
        }
#endif

        
        if (entry_ptr->type->fsf_size) {
            if ((entry_ptr->type->fsf_size)((void *)entry_ptr, &fsf_size) < 0)
                
                HDONE_ERROR(H5E_CACHE, H5E_CANTFREE, FAIL, "unable to get file space free size");
        }    
        else 
            fsf_size = entry_ptr->size;

        
        if ((ret_value >= 0) && H5MF_xfree(f, entry_ptr->type->mem_type, entry_ptr->addr, fsf_size) < 0)
            
            HDONE_ERROR(H5E_CACHE, H5E_CANTFREE, FAIL, "unable to free file space for cache entry");
    } 

    
    entry_ptr->cache_ptr = NULL;

    
    cache_ptr->entries_removed_counter++;
    cache_ptr->last_entry_removed_ptr = entry_ptr;

    if (entry_ptr == cache_ptr->entry_watched_for_removal)
        cache_ptr->entry_watched_for_removal = NULL;

    
    
    if (destroy_entry) {
        if (entry_ptr->is_dirty) {
            
            entry_ptr->is_dirty = false;

            
            if (entry_ptr->type->notify &&
                (entry_ptr->type->notify)(H5C_NOTIFY_ACTION_ENTRY_CLEANED, entry_ptr) < 0)
                
                HDONE_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                            "can't notify client about entry dirty flag cleared");
        } 

        
        assert(entry_ptr->image_ptr == NULL);

        if (entry_ptr->type->free_icr((void *)entry_ptr) < 0)
            
            HDONE_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "free_icr callback failed");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__verify_len_eoa(H5F_t *f, const H5C_class_t *type, haddr_t addr, size_t *len, bool actual)
{
    H5FD_mem_t cooked_type;         
    haddr_t    eoa;                 
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    cooked_type = (type->mem_type == H5FD_MEM_GHEAP) ? H5FD_MEM_DRAW : type->mem_type;

    
    eoa = H5F_get_eoa(f, cooked_type);
    if (!H5_addr_defined(eoa))
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "invalid EOA address for file");

    
    if (H5_addr_gt(addr, eoa))
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "address of object past end of allocation");

    
    if (H5_addr_lt((addr + *len), addr) || H5_addr_gt((addr + *len), eoa)) {
        if (actual)
            HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "actual len exceeds EOA");
        else {
            
            *len = (size_t)(eoa - addr);

            if (*len <= 0)
                HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "len not positive after adjustment for EOA");
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5C__load_entry(H5F_t *f,
#ifdef H5_HAVE_PARALLEL
                bool coll_access,
#endif 
                const H5C_class_t *type, haddr_t addr, void *udata)
{
    bool               dirty = false; 
    uint8_t           *image = NULL;  
    void              *thing = NULL;  
    H5C_cache_entry_t *entry = NULL;  
    size_t             len;           
#ifdef H5_HAVE_PARALLEL
    int      mpi_rank = 0;             
    MPI_Comm comm     = MPI_COMM_NULL; 
    int      mpi_code;                 
#endif                                 
    void *ret_value = NULL;            

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(f->shared);
    assert(f->shared->cache);
    assert(type);
    assert(H5_addr_defined(addr));
    assert(type->get_initial_load_size);
    if (type->flags & H5C__CLASS_SPECULATIVE_LOAD_FLAG)
        assert(type->get_final_load_size);
    else
        assert(NULL == type->get_final_load_size);
    assert(type->deserialize);

    
    assert(!((type->flags & H5C__CLASS_SKIP_READS) && (type->flags & H5C__CLASS_SPECULATIVE_LOAD_FLAG)));

    
    if (type->get_initial_load_size(udata, &len) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTGET, NULL, "can't retrieve image size");
    assert(len > 0);

    
    if (type->flags & H5C__CLASS_SPECULATIVE_LOAD_FLAG)
        if (H5C__verify_len_eoa(f, type, addr, &len, false) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, NULL, "invalid len with respect to EOA");

    
    if (NULL == (image = (uint8_t *)H5MM_malloc(len + H5C_IMAGE_EXTRA_SPACE)))
        HGOTO_ERROR(H5E_CACHE, H5E_CANTALLOC, NULL, "memory allocation failed for on disk image buffer");
#if H5C_DO_MEMORY_SANITY_CHECKS
    H5MM_memcpy(image + len, H5C_IMAGE_SANITY_VALUE, H5C_IMAGE_EXTRA_SPACE);
#endif 

#ifdef H5_HAVE_PARALLEL
    if (H5F_HAS_FEATURE(f, H5FD_FEAT_HAS_MPI)) {
        if ((mpi_rank = H5F_mpi_get_rank(f)) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, NULL, "Can't get MPI rank");
        if ((comm = H5F_mpi_get_comm(f)) == MPI_COMM_NULL)
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, NULL, "get_comm request failed");
    }  
#endif 

    
    if (0 == (type->flags & H5C__CLASS_SKIP_READS)) {
        unsigned tries, max_tries;   
        unsigned retries;            
        htri_t   chk_ret;            
        size_t   actual_len = len;   
        uint64_t nanosec    = 1;     
        void    *new_image;          
        bool     len_changed = true; 

        
        max_tries = tries = H5F_GET_READ_ATTEMPTS(f);

        
        do {
            if (actual_len != len) {
                
                if (len == 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, NULL, "len is a bad value");

                if (NULL == (new_image = H5MM_realloc(image, len + H5C_IMAGE_EXTRA_SPACE)))
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTALLOC, NULL, "image null after H5MM_realloc()");
                image = (uint8_t *)new_image;

#if H5C_DO_MEMORY_SANITY_CHECKS
                H5MM_memcpy(image + len, H5C_IMAGE_SANITY_VALUE, H5C_IMAGE_EXTRA_SPACE);
#endif        
            } 

#ifdef H5_HAVE_PARALLEL
            if (!coll_access || 0 == mpi_rank) {
#endif 
                if (H5F_block_read(f, type->mem_type, addr, len, image) < 0) {
#ifdef H5_HAVE_PARALLEL
                    if (coll_access) {
                        
                        memset(image, 0, len);
                        HDONE_ERROR(H5E_CACHE, H5E_READERROR, NULL, "Can't read image*");
                    }
                    else
#endif
                        HGOTO_ERROR(H5E_CACHE, H5E_READERROR, NULL, "Can't read image*");
                }

#ifdef H5_HAVE_PARALLEL
            } 
            
            if (coll_access) {
                int buf_size;

                H5_CHECKED_ASSIGN(buf_size, int, len, size_t);
                if (MPI_SUCCESS != (mpi_code = MPI_Bcast(image, buf_size, MPI_BYTE, 0, comm)))
                    HMPI_GOTO_ERROR(NULL, "MPI_Bcast failed", mpi_code)
            } 
#endif        

            
            if ((type->flags & H5C__CLASS_SPECULATIVE_LOAD_FLAG) && len_changed) {
                
                actual_len = len;
                if (type->get_final_load_size(image, len, udata, &actual_len) < 0)
                    continue; 

                
                if (actual_len != len) {
                    
                    if (H5C__verify_len_eoa(f, type, addr, &actual_len, true) < 0)
                        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, NULL, "actual_len exceeds EOA");

                    
                    if (actual_len == 0)
                        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, NULL, "actual_len is a bad value");

                    
                    if (NULL == (new_image = H5MM_realloc(image, actual_len + H5C_IMAGE_EXTRA_SPACE)))
                        HGOTO_ERROR(H5E_CACHE, H5E_CANTALLOC, NULL, "image null after H5MM_realloc()");
                    image = (uint8_t *)new_image;

#if H5C_DO_MEMORY_SANITY_CHECKS
                    H5MM_memcpy(image + actual_len, H5C_IMAGE_SANITY_VALUE, H5C_IMAGE_EXTRA_SPACE);
#endif 

                    if (actual_len > len) {
#ifdef H5_HAVE_PARALLEL
                        if (!coll_access || 0 == mpi_rank) {
#endif 
                            
                            if (H5F_block_read(f, type->mem_type, addr + len, actual_len - len, image + len) <
                                0) {
#ifdef H5_HAVE_PARALLEL
                                if (coll_access) {
                                    
                                    memset(image + len, 0, actual_len - len);
                                    HDONE_ERROR(H5E_CACHE, H5E_CANTLOAD, NULL, "can't read image");
                                }
                                else
#endif
                                    HGOTO_ERROR(H5E_CACHE, H5E_CANTLOAD, NULL, "can't read image");
                            }

#ifdef H5_HAVE_PARALLEL
                        }
                        
                        if (coll_access) {
                            int buf_size;

                            H5_CHECKED_ASSIGN(buf_size, int, actual_len - len, size_t);
                            if (MPI_SUCCESS !=
                                (mpi_code = MPI_Bcast(image + len, buf_size, MPI_BYTE, 0, comm)))
                                HMPI_GOTO_ERROR(NULL, "MPI_Bcast failed", mpi_code)
                        } 
#endif                    
                    }     
                }         
                else {
                    
                    len_changed = false;

                    
                    len = actual_len;
                } 
            }     

            
            if (type->verify_chksum == NULL)
                break;

            
            if ((chk_ret = type->verify_chksum(image, actual_len, udata)) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTGET, NULL, "failure from verify_chksum callback");
            if (chk_ret == true)
                break;

            
            H5_nanosleep(nanosec);
            nanosec *= 2; 
        } while (--tries);

        
        if (tries == 0)
            HGOTO_ERROR(H5E_CACHE, H5E_READERROR, NULL,
                        "incorrect metadata checksum after all read attempts");

        
        retries = max_tries - tries;
        if (retries) 
            if (H5F_track_metadata_read_retries(f, (unsigned)type->mem_type, retries) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, NULL, "cannot track read tries = %u ", retries);

        
        len = actual_len;
    } 

    
    if (NULL == (thing = type->deserialize(image, len, udata, &dirty)))
        HGOTO_ERROR(H5E_CACHE, H5E_CANTLOAD, NULL, "Can't deserialize image");

    entry = (H5C_cache_entry_t *)thing;

    

    assert((dirty == false) || (type->id == 5 || type->id == 6));

    entry->cache_ptr        = f->shared->cache;
    entry->addr             = addr;
    entry->size             = len;
    entry->image_ptr        = image;
    entry->image_up_to_date = !dirty;
    entry->type             = type;
    entry->is_dirty         = dirty;
    entry->dirtied          = false;
    entry->is_protected     = false;
    entry->is_read_only     = false;
    entry->ro_ref_count     = 0;
    entry->is_pinned        = false;
    entry->in_slist         = false;
#ifdef H5_HAVE_PARALLEL
    entry->clear_on_unprotect = false;
    entry->flush_immediately  = false;
    entry->coll_access        = coll_access;
#endif 
    entry->flush_in_progress   = false;
    entry->destroy_in_progress = false;

    entry->ring = H5C_RING_UNDEFINED;

    
    entry->flush_dep_parent          = NULL;
    entry->flush_dep_nparents        = 0;
    entry->flush_dep_parent_nalloc   = 0;
    entry->flush_dep_nchildren       = 0;
    entry->flush_dep_ndirty_children = 0;
    entry->flush_dep_nunser_children = 0;
    entry->ht_next                   = NULL;
    entry->ht_prev                   = NULL;
    entry->il_next                   = NULL;
    entry->il_prev                   = NULL;

    entry->next = NULL;
    entry->prev = NULL;

#if H5C_MAINTAIN_CLEAN_AND_DIRTY_LRU_LISTS
    entry->aux_next = NULL;
    entry->aux_prev = NULL;
#endif 

#ifdef H5_HAVE_PARALLEL
    entry->coll_next = NULL;
    entry->coll_prev = NULL;
#endif 

    
    entry->include_in_image     = false;
    entry->lru_rank             = 0;
    entry->image_dirty          = false;
    entry->fd_parent_count      = 0;
    entry->fd_parent_addrs      = NULL;
    entry->fd_child_count       = 0;
    entry->fd_dirty_child_count = 0;
    entry->image_fd_height      = 0;
    entry->prefetched           = false;
    entry->prefetch_type_id     = 0;
    entry->age                  = 0;
    entry->prefetched_dirty     = false;
#ifndef NDEBUG 
    entry->serialization_count = 0;
#endif

    
    entry->tl_next  = NULL;
    entry->tl_prev  = NULL;
    entry->tag_info = NULL;

    H5C__RESET_CACHE_ENTRY_STATS(entry);

    ret_value = thing;

done:
    
    if (NULL == ret_value) {
        
        if (thing && type->free_icr(thing) < 0)
            HDONE_ERROR(H5E_CACHE, H5E_CANTFLUSH, NULL, "free_icr callback failed");
        if (image)
            image = (uint8_t *)H5MM_xfree(image);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__mark_flush_dep_dirty(H5C_cache_entry_t *entry)
{
    unsigned u;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(entry);

    
    for (u = 0; u < entry->flush_dep_nparents; u++) {
        
        assert(entry->flush_dep_parent[u]->flush_dep_ndirty_children <
               entry->flush_dep_parent[u]->flush_dep_nchildren);

        
        entry->flush_dep_parent[u]->flush_dep_ndirty_children++;

        
        if (entry->flush_dep_parent[u]->type->notify &&
            (entry->flush_dep_parent[u]->type->notify)(H5C_NOTIFY_ACTION_CHILD_DIRTIED,
                                                       entry->flush_dep_parent[u]) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                        "can't notify parent about child entry dirty flag set");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__mark_flush_dep_clean(H5C_cache_entry_t *entry)
{
    int    i;                   
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(entry);

    
    
    for (i = ((int)entry->flush_dep_nparents) - 1; i >= 0; i--) {
        
        assert(entry->flush_dep_parent[i]->flush_dep_ndirty_children > 0);

        
        entry->flush_dep_parent[i]->flush_dep_ndirty_children--;

        
        if (entry->flush_dep_parent[i]->type->notify &&
            (entry->flush_dep_parent[i]->type->notify)(H5C_NOTIFY_ACTION_CHILD_CLEANED,
                                                       entry->flush_dep_parent[i]) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                        "can't notify parent about child entry dirty flag reset");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__mark_flush_dep_serialized(H5C_cache_entry_t *entry_ptr)
{
    int    i;                   
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(entry_ptr);

    
    
    for (i = ((int)entry_ptr->flush_dep_nparents) - 1; i >= 0; i--) {
        
        assert(entry_ptr->flush_dep_parent);
        assert(entry_ptr->flush_dep_parent[i]->flush_dep_nunser_children > 0);

        
        entry_ptr->flush_dep_parent[i]->flush_dep_nunser_children--;

        
        if (entry_ptr->flush_dep_parent[i]->type->notify &&
            (entry_ptr->flush_dep_parent[i]->type->notify)(H5C_NOTIFY_ACTION_CHILD_SERIALIZED,
                                                           entry_ptr->flush_dep_parent[i]) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                        "can't notify parent about child entry serialized flag set");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__mark_flush_dep_unserialized(H5C_cache_entry_t *entry_ptr)
{
    unsigned u;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(entry_ptr);

    
    for (u = 0; u < entry_ptr->flush_dep_nparents; u++) {
        
        assert(entry_ptr->flush_dep_parent);
        assert(entry_ptr->flush_dep_parent[u]->flush_dep_nunser_children <
               entry_ptr->flush_dep_parent[u]->flush_dep_nchildren);

        
        entry_ptr->flush_dep_parent[u]->flush_dep_nunser_children++;

        
        if (entry_ptr->flush_dep_parent[u]->type->notify &&
            (entry_ptr->flush_dep_parent[u]->type->notify)(H5C_NOTIFY_ACTION_CHILD_UNSERIALIZED,
                                                           entry_ptr->flush_dep_parent[u]) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                        "can't notify parent about child entry serialized flag reset");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

#ifndef NDEBUG

static void
H5C__assert_flush_dep_nocycle(const H5C_cache_entry_t *entry, const H5C_cache_entry_t *base_entry)
{
    unsigned u; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(entry);
    assert(base_entry);

    
    assert(base_entry != entry);

    
    for (u = 0; u < entry->flush_dep_nparents; u++)
        H5C__assert_flush_dep_nocycle(entry->flush_dep_parent[u], base_entry);

    FUNC_LEAVE_NOAPI_VOID
} 
#endif

herr_t
H5C__serialize_single_entry(H5F_t *f, H5C_t *cache_ptr, H5C_cache_entry_t *entry_ptr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(cache_ptr);
    assert(entry_ptr);
    assert(!entry_ptr->prefetched);
    assert(!entry_ptr->image_up_to_date);
    assert(entry_ptr->is_dirty);
    assert(!entry_ptr->is_protected);
    assert(!entry_ptr->flush_in_progress);
    assert(entry_ptr->type);

    
    entry_ptr->flush_in_progress = true;

    
    if (NULL == entry_ptr->image_ptr) {
        assert(entry_ptr->size > 0);
        if (NULL == (entry_ptr->image_ptr = H5MM_malloc(entry_ptr->size + H5C_IMAGE_EXTRA_SPACE)))
            HGOTO_ERROR(H5E_CACHE, H5E_CANTALLOC, FAIL, "memory allocation failed for on disk image buffer");
#if H5C_DO_MEMORY_SANITY_CHECKS
        H5MM_memcpy(((uint8_t *)entry_ptr->image_ptr) + entry_ptr->size, H5C_IMAGE_SANITY_VALUE,
                    H5C_IMAGE_EXTRA_SPACE);
#endif 
    }  

    
    if (H5C__generate_image(f, cache_ptr, entry_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTSERIALIZE, FAIL, "Can't generate image for cache entry");

    
    entry_ptr->flush_in_progress = false;

done:
    assert((ret_value != SUCCEED) || (!entry_ptr->flush_in_progress));
    assert((ret_value != SUCCEED) || (entry_ptr->image_up_to_date));
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__destroy_pf_entry_child_flush_deps(H5C_t *cache_ptr, H5C_cache_entry_t *pf_entry_ptr,
                                       H5C_cache_entry_t **fd_children)
{
    H5C_cache_entry_t *entry_ptr;
#ifndef NDEBUG
    unsigned entries_visited = 0;
#endif
    int    fd_children_found = 0;
    bool   found;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(cache_ptr);
    assert(pf_entry_ptr);
    assert(pf_entry_ptr->type);
    assert(pf_entry_ptr->type->id == H5AC_PREFETCHED_ENTRY_ID);
    assert(pf_entry_ptr->prefetched);
    assert(pf_entry_ptr->fd_child_count > 0);
    assert(fd_children);

    
    entry_ptr = cache_ptr->il_head;
    while (entry_ptr != NULL) {
        
        if (entry_ptr->prefetched && (entry_ptr->flush_dep_nparents > 0)) {
            unsigned u; 

            
            u     = 0;
            found = false;

            
            assert(entry_ptr->type);
            assert(entry_ptr->type->id == H5AC_PREFETCHED_ENTRY_ID);
            assert(entry_ptr->fd_parent_count >= entry_ptr->flush_dep_nparents);
            assert(entry_ptr->fd_parent_addrs);
            assert(entry_ptr->flush_dep_parent);

            
            while (!found && (u < entry_ptr->fd_parent_count)) {
                
                assert(entry_ptr->flush_dep_parent[u]);

                
                if (pf_entry_ptr == entry_ptr->flush_dep_parent[u])
                    found = true;

                u++;
            } 

            if (found) {
                assert(NULL == fd_children[fd_children_found]);

                
                fd_children[fd_children_found] = entry_ptr;
                fd_children_found++;
                if (H5C_destroy_flush_dependency(pf_entry_ptr, entry_ptr) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTUNDEPEND, FAIL,
                                "can't destroy pf entry child flush dependency");

#ifndef NDEBUG
                
                found = false;
                u     = 0;
                while (!found && u < entry_ptr->fd_parent_count) {
                    if (pf_entry_ptr->addr == entry_ptr->fd_parent_addrs[u])
                        found = true;
                    u++;
                } 
                assert(found);
#endif
            } 
        }     

#ifndef NDEBUG
        entries_visited++;
#endif
        entry_ptr = entry_ptr->il_next;
    } 

    
    assert(NULL == fd_children[fd_children_found]);
    assert((unsigned)fd_children_found == pf_entry_ptr->fd_child_count);
    assert(entries_visited == cache_ptr->index_len);
    assert(!pf_entry_ptr->is_pinned);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__deserialize_prefetched_entry(H5F_t *f, H5C_t *cache_ptr, H5C_cache_entry_t **entry_ptr_ptr,
                                  const H5C_class_t *type, haddr_t addr, void *udata)
{
    bool dirty = false;                     
    size_t             len;                 
    void              *thing = NULL;        
    H5C_cache_entry_t *pf_entry_ptr;        
                                            
    H5C_cache_entry_t *ds_entry_ptr;        
    H5C_cache_entry_t **fd_children = NULL; 
                                            
                                            
                                            
                                            
    unsigned flush_flags = (H5C__FLUSH_INVALIDATE_FLAG | H5C__FLUSH_CLEAR_ONLY_FLAG);
    int      i;
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(f->shared);
    assert(f->shared->cache);
    assert(f->shared->cache == cache_ptr);
    assert(entry_ptr_ptr);
    assert(*entry_ptr_ptr);
    pf_entry_ptr = *entry_ptr_ptr;
    assert(pf_entry_ptr->type);
    assert(pf_entry_ptr->type->id == H5AC_PREFETCHED_ENTRY_ID);
    assert(pf_entry_ptr->prefetched);
    assert(pf_entry_ptr->image_up_to_date);
    assert(pf_entry_ptr->image_ptr);
    assert(pf_entry_ptr->size > 0);
    assert(pf_entry_ptr->addr == addr);
    assert(type);
    assert(type->id == pf_entry_ptr->prefetch_type_id);
    assert(type->mem_type == cache_ptr->class_table_ptr[type->id]->mem_type);

    
    assert(!(type->flags & H5C__CLASS_SKIP_READS));

    
    assert(!((type->flags & H5C__CLASS_SKIP_READS) && (type->flags & H5C__CLASS_SPECULATIVE_LOAD_FLAG)));
    assert(H5_addr_defined(addr));
    assert(type->get_initial_load_size);
    assert(type->deserialize);

    
    assert(pf_entry_ptr->fd_parent_count == pf_entry_ptr->flush_dep_nparents);
    for (i = (int)(pf_entry_ptr->fd_parent_count) - 1; i >= 0; i--) {
        assert(pf_entry_ptr->flush_dep_parent);
        assert(pf_entry_ptr->flush_dep_parent[i]);
        assert(pf_entry_ptr->flush_dep_parent[i]->flush_dep_nchildren > 0);
        assert(pf_entry_ptr->fd_parent_addrs);
        assert(pf_entry_ptr->flush_dep_parent[i]->addr == pf_entry_ptr->fd_parent_addrs[i]);

        if (H5C_destroy_flush_dependency(pf_entry_ptr->flush_dep_parent[i], pf_entry_ptr) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTUNDEPEND, FAIL, "can't destroy pf entry parent flush dependency");

        pf_entry_ptr->fd_parent_addrs[i] = HADDR_UNDEF;
    } 
    assert(pf_entry_ptr->flush_dep_nparents == 0);

    
    if (pf_entry_ptr->fd_child_count > 0) {
        if (NULL == (fd_children = (H5C_cache_entry_t **)H5MM_calloc(
                         sizeof(H5C_cache_entry_t **) * (size_t)(pf_entry_ptr->fd_child_count + 1))))
            HGOTO_ERROR(H5E_CACHE, H5E_CANTALLOC, FAIL, "memory allocation failed for fd child ptr array");

        if (H5C__destroy_pf_entry_child_flush_deps(cache_ptr, pf_entry_ptr, fd_children) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTUNDEPEND, FAIL,
                        "can't destroy pf entry child flush dependency(s).");
    } 

    
    len = pf_entry_ptr->size;

    
    if (NULL == (thing = type->deserialize(pf_entry_ptr->image_ptr, len, udata, &dirty)))
        HGOTO_ERROR(H5E_CACHE, H5E_CANTLOAD, FAIL, "Can't deserialize image");
    ds_entry_ptr = (H5C_cache_entry_t *)thing;

    

    assert((dirty == false) || (type->id == 5 || type->id == 6));

    ds_entry_ptr->cache_ptr = f->shared->cache;
    ds_entry_ptr->addr      = addr;
    ds_entry_ptr->size      = len;
    assert(ds_entry_ptr->size < H5C_MAX_ENTRY_SIZE);
    ds_entry_ptr->image_ptr        = pf_entry_ptr->image_ptr;
    ds_entry_ptr->image_up_to_date = !dirty;
    ds_entry_ptr->type             = type;
    ds_entry_ptr->is_dirty         = dirty | pf_entry_ptr->is_dirty;
    ds_entry_ptr->dirtied          = false;
    ds_entry_ptr->is_protected     = false;
    ds_entry_ptr->is_read_only     = false;
    ds_entry_ptr->ro_ref_count     = 0;
    ds_entry_ptr->is_pinned        = false;
    ds_entry_ptr->in_slist         = false;
#ifdef H5_HAVE_PARALLEL
    ds_entry_ptr->clear_on_unprotect = false;
    ds_entry_ptr->flush_immediately  = false;
    ds_entry_ptr->coll_access        = false;
#endif 
    ds_entry_ptr->flush_in_progress   = false;
    ds_entry_ptr->destroy_in_progress = false;

    ds_entry_ptr->ring = pf_entry_ptr->ring;

    
    ds_entry_ptr->flush_dep_parent          = NULL;
    ds_entry_ptr->flush_dep_nparents        = 0;
    ds_entry_ptr->flush_dep_parent_nalloc   = 0;
    ds_entry_ptr->flush_dep_nchildren       = 0;
    ds_entry_ptr->flush_dep_ndirty_children = 0;
    ds_entry_ptr->flush_dep_nunser_children = 0;

    
    ds_entry_ptr->ht_next = NULL;
    ds_entry_ptr->ht_prev = NULL;
    ds_entry_ptr->il_next = NULL;
    ds_entry_ptr->il_prev = NULL;

    
    ds_entry_ptr->next = NULL;
    ds_entry_ptr->prev = NULL;
#if H5C_MAINTAIN_CLEAN_AND_DIRTY_LRU_LISTS
    ds_entry_ptr->aux_next = NULL;
    ds_entry_ptr->aux_prev = NULL;
#endif 
#ifdef H5_HAVE_PARALLEL
    pf_entry_ptr->coll_next = NULL;
    pf_entry_ptr->coll_prev = NULL;
#endif 

    
    ds_entry_ptr->include_in_image     = false;
    ds_entry_ptr->lru_rank             = 0;
    ds_entry_ptr->image_dirty          = false;
    ds_entry_ptr->fd_parent_count      = 0;
    ds_entry_ptr->fd_parent_addrs      = NULL;
    ds_entry_ptr->fd_child_count       = pf_entry_ptr->fd_child_count;
    ds_entry_ptr->fd_dirty_child_count = 0;
    ds_entry_ptr->image_fd_height      = 0;
    ds_entry_ptr->prefetched           = false;
    ds_entry_ptr->prefetch_type_id     = 0;
    ds_entry_ptr->age                  = 0;
    ds_entry_ptr->prefetched_dirty     = pf_entry_ptr->prefetched_dirty;
#ifndef NDEBUG 
    ds_entry_ptr->serialization_count = 0;
#endif

    H5C__RESET_CACHE_ENTRY_STATS(ds_entry_ptr);

    
    if (H5C__tag_entry(cache_ptr, ds_entry_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTTAG, FAIL, "Cannot tag metadata entry");

    
    pf_entry_ptr->image_ptr = NULL;

    if (pf_entry_ptr->is_dirty) {
        assert(((cache_ptr->slist_enabled) && (pf_entry_ptr->in_slist)) ||
               ((!cache_ptr->slist_enabled) && (!pf_entry_ptr->in_slist)));

        flush_flags |= H5C__DEL_FROM_SLIST_ON_DESTROY_FLAG;
    } 

    if (H5C__flush_single_entry(f, pf_entry_ptr, flush_flags) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTEXPUNGE, FAIL, "can't expunge prefetched entry");

#ifndef NDEGUG 
    H5C__SEARCH_INDEX(cache_ptr, addr, pf_entry_ptr, FAIL);

    assert(NULL == pf_entry_ptr);
#endif

    
    H5C__INSERT_IN_INDEX(cache_ptr, ds_entry_ptr, FAIL);

    assert(!ds_entry_ptr->in_slist);
    if (ds_entry_ptr->is_dirty)
        H5C__INSERT_ENTRY_IN_SLIST(cache_ptr, ds_entry_ptr, FAIL);

    H5C__UPDATE_RP_FOR_INSERTION(cache_ptr, ds_entry_ptr, FAIL);

    
    if (ds_entry_ptr->type->notify &&
        (ds_entry_ptr->type->notify)(H5C_NOTIFY_ACTION_AFTER_LOAD, ds_entry_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL, "can't notify client about entry loaded into cache");

    
    i = 0;
    if (fd_children != NULL) {
        H5C__UPDATE_RP_FOR_PROTECT(cache_ptr, ds_entry_ptr, FAIL);
        ds_entry_ptr->is_protected = true;
        while (fd_children[i] != NULL) {
            
            assert((fd_children[i])->prefetched);
            assert((fd_children[i])->fd_parent_count > 0);
            assert((fd_children[i])->fd_parent_addrs);

#ifndef NDEBUG
            {
                int  j;
                bool found;

                j     = 0;
                found = false;
                while ((j < (int)((fd_children[i])->fd_parent_count)) && (!found)) {
                    if ((fd_children[i])->fd_parent_addrs[j] == ds_entry_ptr->addr)
                        found = true;

                    j++;
                } 
                assert(found);
            }
#endif

            if (H5C_create_flush_dependency(ds_entry_ptr, fd_children[i]) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTDEPEND, FAIL, "Can't restore child flush dependency");

            i++;
        } 

        H5C__UPDATE_RP_FOR_UNPROTECT(cache_ptr, ds_entry_ptr, FAIL);
        ds_entry_ptr->is_protected = false;
    } 
    assert((unsigned)i == ds_entry_ptr->fd_child_count);

    ds_entry_ptr->fd_child_count = 0;
    H5C__UPDATE_STATS_FOR_PREFETCH_HIT(cache_ptr);

    
    *entry_ptr_ptr = ds_entry_ptr;

done:
    if (fd_children)
        fd_children = (H5C_cache_entry_t **)H5MM_xfree((void *)fd_children);

    
    if (FAIL == ret_value)
        if (thing && type->free_icr(thing) < 0)
            HDONE_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "free_icr callback failed");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_insert_entry(H5F_t *f, const H5C_class_t *type, haddr_t addr, void *thing, unsigned int flags)
{
    H5C_t      *cache_ptr;
    H5AC_ring_t ring = H5C_RING_UNDEFINED;
    bool        insert_pinned;
    bool        flush_last;
#ifdef H5_HAVE_PARALLEL
    bool coll_access = false; 
#endif                        
    bool               write_permitted = true;
    size_t             empty_space;
    H5C_cache_entry_t *entry_ptr = NULL;
    H5C_cache_entry_t *test_entry_ptr;
    bool               entry_tagged = false;
    herr_t             ret_value    = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(f);
    assert(f->shared);

    cache_ptr = f->shared->cache;

    assert(cache_ptr);
    assert(type);
    assert(type->mem_type == cache_ptr->class_table_ptr[type->id]->mem_type);
    assert(type->image_len);
    assert(H5_addr_defined(addr));
    assert(thing);

#ifdef H5C_DO_EXTREME_SANITY_CHECKS
    
    
    if (H5C__validate_protected_entry_list(cache_ptr) < 0 || H5C__validate_pinned_entry_list(cache_ptr) < 0 ||
        H5C__validate_lru_list(cache_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "an extreme sanity check failed on entry");
#endif 

    insert_pinned = ((flags & H5C__PIN_ENTRY_FLAG) != 0);
    flush_last    = ((flags & H5C__FLUSH_LAST_FLAG) != 0);

    
    ring = H5CX_get_ring();

    entry_ptr = (H5C_cache_entry_t *)thing;

    

    H5C__SEARCH_INDEX(cache_ptr, addr, test_entry_ptr, FAIL);

    if (test_entry_ptr != NULL) {
        if (test_entry_ptr == entry_ptr)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTINS, FAIL, "entry already in cache");
        else
            HGOTO_ERROR(H5E_CACHE, H5E_CANTINS, FAIL, "duplicate entry in cache");
    } 

    entry_ptr->cache_ptr = cache_ptr;
    entry_ptr->addr      = addr;
    entry_ptr->type      = type;

    entry_ptr->image_ptr        = NULL;
    entry_ptr->image_up_to_date = false;

    entry_ptr->is_protected = false;
    entry_ptr->is_read_only = false;
    entry_ptr->ro_ref_count = 0;

    entry_ptr->is_pinned          = insert_pinned;
    entry_ptr->pinned_from_client = insert_pinned;
    entry_ptr->pinned_from_cache  = false;
    entry_ptr->flush_me_last      = flush_last;

    
    entry_ptr->is_dirty = true;

    
    entry_ptr->dirtied = false;

    
    if ((type->image_len)(thing, &(entry_ptr->size)) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTGETSIZE, FAIL, "can't get size of thing");
    assert(entry_ptr->size > 0 && entry_ptr->size < H5C_MAX_ENTRY_SIZE);

    entry_ptr->in_slist = false;

#ifdef H5_HAVE_PARALLEL
    entry_ptr->clear_on_unprotect = false;
    entry_ptr->flush_immediately  = false;
#endif 

    entry_ptr->flush_in_progress   = false;
    entry_ptr->destroy_in_progress = false;

    entry_ptr->ring = ring;

    
    entry_ptr->flush_dep_parent          = NULL;
    entry_ptr->flush_dep_nparents        = 0;
    entry_ptr->flush_dep_parent_nalloc   = 0;
    entry_ptr->flush_dep_nchildren       = 0;
    entry_ptr->flush_dep_ndirty_children = 0;
    entry_ptr->flush_dep_nunser_children = 0;

    entry_ptr->ht_next = NULL;
    entry_ptr->ht_prev = NULL;
    entry_ptr->il_next = NULL;
    entry_ptr->il_prev = NULL;

    entry_ptr->next = NULL;
    entry_ptr->prev = NULL;

#if H5C_MAINTAIN_CLEAN_AND_DIRTY_LRU_LISTS
    entry_ptr->aux_next = NULL;
    entry_ptr->aux_prev = NULL;
#endif 

#ifdef H5_HAVE_PARALLEL
    entry_ptr->coll_next = NULL;
    entry_ptr->coll_prev = NULL;
#endif 

    
    entry_ptr->include_in_image     = false;
    entry_ptr->lru_rank             = 0;
    entry_ptr->image_dirty          = false;
    entry_ptr->fd_parent_count      = 0;
    entry_ptr->fd_parent_addrs      = NULL;
    entry_ptr->fd_child_count       = 0;
    entry_ptr->fd_dirty_child_count = 0;
    entry_ptr->image_fd_height      = 0;
    entry_ptr->prefetched           = false;
    entry_ptr->prefetch_type_id     = 0;
    entry_ptr->age                  = 0;
    entry_ptr->prefetched_dirty     = false;
#ifndef NDEBUG 
    entry_ptr->serialization_count = 0;
#endif

    
    entry_ptr->tl_next  = NULL;
    entry_ptr->tl_prev  = NULL;
    entry_ptr->tag_info = NULL;

    
    if (H5C__tag_entry(cache_ptr, entry_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTTAG, FAIL, "Cannot tag metadata entry");
    entry_tagged = true;

    H5C__RESET_CACHE_ENTRY_STATS(entry_ptr);

    if (cache_ptr->flash_size_increase_possible &&
        (entry_ptr->size > cache_ptr->flash_size_increase_threshold))
        if (H5C__flash_increase_cache_size(cache_ptr, 0, entry_ptr->size) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTINS, FAIL, "H5C__flash_increase_cache_size failed");

    if (cache_ptr->index_size >= cache_ptr->max_cache_size)
        empty_space = 0;
    else
        empty_space = cache_ptr->max_cache_size - cache_ptr->index_size;

    if (cache_ptr->evictions_enabled &&
        (((cache_ptr->index_size + entry_ptr->size) > cache_ptr->max_cache_size) ||
         (((empty_space + cache_ptr->clean_index_size) < cache_ptr->min_clean_size)))) {
        size_t space_needed;

        if (empty_space <= entry_ptr->size)
            cache_ptr->cache_full = true;

        if (cache_ptr->check_write_permitted != NULL) {
            if ((cache_ptr->check_write_permitted)(f, &write_permitted) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTINS, FAIL, "Can't get write_permitted");
        } 
        else
            write_permitted = cache_ptr->write_permitted;

        assert(entry_ptr->size <= H5C_MAX_ENTRY_SIZE);
        space_needed = entry_ptr->size;
        if (space_needed > cache_ptr->max_cache_size)
            space_needed = cache_ptr->max_cache_size;

        

        if (H5C__make_space_in_cache(f, space_needed, write_permitted) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTINS, FAIL, "H5C__make_space_in_cache failed");
    } 

    H5C__INSERT_IN_INDEX(cache_ptr, entry_ptr, FAIL);

    
    assert(entry_ptr->is_dirty);
    H5C__INSERT_ENTRY_IN_SLIST(cache_ptr, entry_ptr, FAIL);
    H5C__UPDATE_RP_FOR_INSERTION(cache_ptr, entry_ptr, FAIL);

#ifdef H5C_DO_EXTREME_SANITY_CHECKS
    if (H5C__validate_protected_entry_list(cache_ptr) < 0 || H5C__validate_pinned_entry_list(cache_ptr) < 0 ||
        H5C__validate_lru_list(cache_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "an extreme sanity check failed just before done");
#endif 

    
    if (entry_ptr->type->notify && (entry_ptr->type->notify)(H5C_NOTIFY_ACTION_AFTER_INSERT, entry_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL, "can't notify client about entry inserted into cache");

    H5C__UPDATE_STATS_FOR_INSERTION(cache_ptr, entry_ptr);

#ifdef H5_HAVE_PARALLEL
    if (H5F_HAS_FEATURE(f, H5FD_FEAT_HAS_MPI))
        coll_access = H5F_get_coll_metadata_reads(f);

    entry_ptr->coll_access = coll_access;
    if (coll_access) {
        H5C__INSERT_IN_COLL_LIST(cache_ptr, entry_ptr, FAIL);

        
        if (H5P_USER_TRUE == H5F_COLL_MD_READ(f)) {
            if (cache_ptr->max_cache_size * 80 < cache_ptr->coll_list_size * 100) {
                if (H5C_clear_coll_entries(cache_ptr, true) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "can't clear collective metadata entries");
            } 
        }     
        else {
            if (cache_ptr->max_cache_size * 40 < cache_ptr->coll_list_size * 100) {
                if (H5C_clear_coll_entries(cache_ptr, true) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "can't clear collective metadata entries");
            } 
        }     
    }         
#endif

done:
#ifdef H5C_DO_EXTREME_SANITY_CHECKS
    if (H5C__validate_protected_entry_list(cache_ptr) < 0 || H5C__validate_pinned_entry_list(cache_ptr) < 0 ||
        H5C__validate_lru_list(cache_ptr) < 0)
        HDONE_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "an extreme sanity check failed on Rexit");
#endif 

    if (ret_value < 0 && entry_tagged)
        if (H5C__untag_entry(cache_ptr, entry_ptr) < 0)
            HDONE_ERROR(H5E_CACHE, H5E_CANTREMOVE, FAIL, "can't remove entry from tag list");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_mark_entry_dirty(void *thing)
{
    H5C_t             *cache_ptr;
    H5C_cache_entry_t *entry_ptr = (H5C_cache_entry_t *)thing;
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(entry_ptr);
    assert(H5_addr_defined(entry_ptr->addr));
    cache_ptr = entry_ptr->cache_ptr;
    assert(cache_ptr);

    if (entry_ptr->is_protected) {
        assert(!((entry_ptr)->is_read_only));

        
        entry_ptr->dirtied = true;

        
        if (entry_ptr->image_up_to_date) {
            entry_ptr->image_up_to_date = false;

            if (entry_ptr->flush_dep_nparents > 0)
                if (H5C__mark_flush_dep_unserialized(entry_ptr) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                                "Can't propagate serialization status to fd parents");
        } 
    }     
    else if (entry_ptr->is_pinned) {
        bool was_clean; 
        bool image_was_up_to_date;

        
        was_clean = !entry_ptr->is_dirty;

        
        image_was_up_to_date = entry_ptr->image_up_to_date;

        
        entry_ptr->is_dirty         = true;
        entry_ptr->image_up_to_date = false;

        
        if (was_clean)
            H5C__UPDATE_INDEX_FOR_ENTRY_DIRTY(cache_ptr, entry_ptr, FAIL);
        if (!entry_ptr->in_slist)
            H5C__INSERT_ENTRY_IN_SLIST(cache_ptr, entry_ptr, FAIL);

        
        H5C__UPDATE_STATS_FOR_DIRTY_PIN(cache_ptr, entry_ptr);

        
        if (was_clean) {
            
            if (entry_ptr->type->notify &&
                (entry_ptr->type->notify)(H5C_NOTIFY_ACTION_ENTRY_DIRTIED, entry_ptr) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                            "can't notify client about entry dirty flag set");

            
            if (entry_ptr->flush_dep_nparents > 0)
                if (H5C__mark_flush_dep_dirty(entry_ptr) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKDIRTY, FAIL, "Can't propagate flush dep dirty flag");
        } 
        if (image_was_up_to_date)
            if (entry_ptr->flush_dep_nparents > 0)
                if (H5C__mark_flush_dep_unserialized(entry_ptr) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                                "Can't propagate serialization status to fd parents");
    } 
    else
        HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKDIRTY, FAIL, "Entry is neither pinned nor protected??");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_mark_entry_clean(void *_thing)
{
    H5C_t             *cache_ptr;
    H5C_cache_entry_t *entry_ptr = (H5C_cache_entry_t *)_thing;
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(entry_ptr);
    assert(H5_addr_defined(entry_ptr->addr));
    cache_ptr = entry_ptr->cache_ptr;
    assert(cache_ptr);

    
    if (entry_ptr->is_protected)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKCLEAN, FAIL, "entry is protected");
    else if (entry_ptr->is_pinned) {
        bool was_dirty; 

        
        was_dirty = entry_ptr->is_dirty;

        
        entry_ptr->is_dirty = false;

        
        if (was_dirty)
            H5C__UPDATE_INDEX_FOR_ENTRY_CLEAN(cache_ptr, entry_ptr, FAIL);
        if (entry_ptr->in_slist)
            H5C__REMOVE_ENTRY_FROM_SLIST(cache_ptr, entry_ptr, false, FAIL);

        
        H5C__UPDATE_STATS_FOR_CLEAR(cache_ptr, entry_ptr);

        
        if (was_dirty) {
            
            if (entry_ptr->type->notify &&
                (entry_ptr->type->notify)(H5C_NOTIFY_ACTION_ENTRY_CLEANED, entry_ptr) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                            "can't notify client about entry dirty flag cleared");

            
            if (entry_ptr->flush_dep_nparents > 0)
                if (H5C__mark_flush_dep_clean(entry_ptr) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKCLEAN, FAIL, "Can't propagate flush dep clean");
        } 
    }     
    else
        HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKCLEAN, FAIL, "Entry is not pinned??");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_mark_entry_unserialized(void *thing)
{
    H5C_cache_entry_t *entry     = (H5C_cache_entry_t *)thing;
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(entry);
    assert(H5_addr_defined(entry->addr));

    if (entry->is_protected || entry->is_pinned) {
        assert(!entry->is_read_only);

        
        if (entry->image_up_to_date) {
            entry->image_up_to_date = false;

            if (entry->flush_dep_nparents > 0)
                if (H5C__mark_flush_dep_unserialized(entry) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTSET, FAIL,
                                "Can't propagate serialization status to fd parents");
        } 
    }     
    else
        HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKUNSERIALIZED, FAIL,
                    "Entry to unserialize is neither pinned nor protected??");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_mark_entry_serialized(void *_thing)
{
    H5C_cache_entry_t *entry     = (H5C_cache_entry_t *)_thing;
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(entry);
    assert(H5_addr_defined(entry->addr));

    
    if (entry->is_protected)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKSERIALIZED, FAIL, "entry is protected");
    else if (entry->is_pinned) {
        
        if (!entry->image_up_to_date) {
            
            entry->image_up_to_date = true;

            
            if (entry->flush_dep_nparents > 0)
                if (H5C__mark_flush_dep_serialized(entry) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKSERIALIZED, FAIL,
                                "Can't propagate flush dep serialize");
        } 
    }     
    else
        HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKSERIALIZED, FAIL, "Entry is not pinned??");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_move_entry(H5C_t *cache_ptr, const H5C_class_t *type, haddr_t old_addr, haddr_t new_addr)
{
    H5C_cache_entry_t *entry_ptr      = NULL;
    H5C_cache_entry_t *test_entry_ptr = NULL;
    herr_t             ret_value      = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(cache_ptr);
    assert(type);
    assert(H5_addr_defined(old_addr));
    assert(H5_addr_defined(new_addr));
    assert(H5_addr_ne(old_addr, new_addr));

#ifdef H5C_DO_EXTREME_SANITY_CHECKS
    if (H5C__validate_protected_entry_list(cache_ptr) < 0 || H5C__validate_pinned_entry_list(cache_ptr) < 0 ||
        H5C__validate_lru_list(cache_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "an extreme sanity check failed on entry");
#endif 

    H5C__SEARCH_INDEX(cache_ptr, old_addr, entry_ptr, FAIL);

    if (entry_ptr == NULL || entry_ptr->type != type)
        
        HGOTO_DONE(SUCCEED);

    assert(entry_ptr->addr == old_addr);
    assert(entry_ptr->type == type);

    
    
    if (entry_ptr->is_read_only)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTMOVE, FAIL, "can't move R/O entry");

    H5C__SEARCH_INDEX(cache_ptr, new_addr, test_entry_ptr, FAIL);

    if (test_entry_ptr != NULL) { 
        if (test_entry_ptr->type == type)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTMOVE, FAIL, "target already moved & reinserted???");
        else
            HGOTO_ERROR(H5E_CACHE, H5E_CANTMOVE, FAIL, "new address already in use?");
    } 

    
    if (!entry_ptr->destroy_in_progress) {
        H5C__DELETE_FROM_INDEX(cache_ptr, entry_ptr, FAIL);

        if (entry_ptr->in_slist) {
            assert(cache_ptr->slist_ptr);
            H5C__REMOVE_ENTRY_FROM_SLIST(cache_ptr, entry_ptr, false, FAIL);
        } 
    }     

    entry_ptr->addr = new_addr;

    if (!entry_ptr->destroy_in_progress) {
        bool was_dirty; 

        
        was_dirty = entry_ptr->is_dirty;

        
        entry_ptr->is_dirty = true;

        
        if (entry_ptr->image_up_to_date) {
            entry_ptr->image_up_to_date = false;
            if (entry_ptr->flush_dep_nparents > 0)
                if (H5C__mark_flush_dep_unserialized(entry_ptr) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                                "Can't propagate serialization status to fd parents");
        } 

        
        H5C__INSERT_IN_INDEX(cache_ptr, entry_ptr, FAIL);
        H5C__INSERT_ENTRY_IN_SLIST(cache_ptr, entry_ptr, FAIL);

        
        if (!entry_ptr->flush_in_progress) {
            
            H5C__UPDATE_RP_FOR_MOVE(cache_ptr, entry_ptr, was_dirty, FAIL);

            
            if (!was_dirty) {
                
                if (entry_ptr->type->notify &&
                    (entry_ptr->type->notify)(H5C_NOTIFY_ACTION_ENTRY_DIRTIED, entry_ptr) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                                "can't notify client about entry dirty flag set");

                
                if (entry_ptr->flush_dep_nparents > 0)
                    if (H5C__mark_flush_dep_dirty(entry_ptr) < 0)
                        HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKDIRTY, FAIL,
                                    "Can't propagate flush dep dirty flag");
            } 
        }     
    }         

    H5C__UPDATE_STATS_FOR_MOVE(cache_ptr, entry_ptr);

done:
#ifdef H5C_DO_EXTREME_SANITY_CHECKS
    if (H5C__validate_protected_entry_list(cache_ptr) < 0 || H5C__validate_pinned_entry_list(cache_ptr) < 0 ||
        H5C__validate_lru_list(cache_ptr) < 0)
        HDONE_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "an extreme sanity check failed on Rexit");
#endif 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_resize_entry(void *thing, size_t new_size)
{
    H5C_t             *cache_ptr;
    H5C_cache_entry_t *entry_ptr = (H5C_cache_entry_t *)thing;
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(entry_ptr);
    assert(H5_addr_defined(entry_ptr->addr));
    cache_ptr = entry_ptr->cache_ptr;
    assert(cache_ptr);

    
    if (new_size <= 0)
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "New size is non-positive");
    if (!(entry_ptr->is_pinned || entry_ptr->is_protected))
        HGOTO_ERROR(H5E_CACHE, H5E_BADTYPE, FAIL, "Entry isn't pinned or protected??");

#ifdef H5C_DO_EXTREME_SANITY_CHECKS
    if (H5C__validate_protected_entry_list(cache_ptr) < 0 || H5C__validate_pinned_entry_list(cache_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "an extreme sanity check failed on entry");
#endif 

    
    if (entry_ptr->size != new_size) {
        bool was_clean;

        
        was_clean = !entry_ptr->is_dirty;

        
        entry_ptr->is_dirty = true;

        
        if (entry_ptr->image_up_to_date) {
            entry_ptr->image_up_to_date = false;
            if (entry_ptr->flush_dep_nparents > 0)
                if (H5C__mark_flush_dep_unserialized(entry_ptr) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                                "Can't propagate serialization status to fd parents");
        } 

        
        if (entry_ptr->image_ptr)
            entry_ptr->image_ptr = H5MM_xfree(entry_ptr->image_ptr);

        
        if (cache_ptr->flash_size_increase_possible) {
            if (new_size > entry_ptr->size) {
                size_t size_increase;

                size_increase = new_size - entry_ptr->size;
                if (size_increase >= cache_ptr->flash_size_increase_threshold)
                    if (H5C__flash_increase_cache_size(cache_ptr, entry_ptr->size, new_size) < 0)
                        HGOTO_ERROR(H5E_CACHE, H5E_CANTRESIZE, FAIL, "flash cache increase failed");
            }
        }

        
        if (entry_ptr->is_pinned)
            H5C__DLL_UPDATE_FOR_SIZE_CHANGE(cache_ptr->pel_len, cache_ptr->pel_size, entry_ptr->size,
                                            new_size, FAIL)
        if (entry_ptr->is_protected)
            H5C__DLL_UPDATE_FOR_SIZE_CHANGE(cache_ptr->pl_len, cache_ptr->pl_size, entry_ptr->size, new_size,
                                            FAIL)

#ifdef H5_HAVE_PARALLEL
        if (entry_ptr->coll_access)
            H5C__DLL_UPDATE_FOR_SIZE_CHANGE(cache_ptr->coll_list_len, cache_ptr->coll_list_size,
                                            entry_ptr->size, new_size, FAIL)
#endif 

        
        H5C__UPDATE_STATS_FOR_ENTRY_SIZE_CHANGE(cache_ptr, entry_ptr, new_size);

        
        H5C__UPDATE_INDEX_FOR_SIZE_CHANGE(cache_ptr, entry_ptr->size, new_size, entry_ptr, was_clean, FAIL);

        
        if (entry_ptr->in_slist)
            H5C__UPDATE_SLIST_FOR_SIZE_CHANGE(cache_ptr, entry_ptr->size, new_size);

        
        entry_ptr->size = new_size;

        if (!entry_ptr->in_slist)
            H5C__INSERT_ENTRY_IN_SLIST(cache_ptr, entry_ptr, FAIL);

        if (entry_ptr->is_pinned)
            H5C__UPDATE_STATS_FOR_DIRTY_PIN(cache_ptr, entry_ptr);

        
        if (was_clean) {
            
            if (entry_ptr->type->notify &&
                (entry_ptr->type->notify)(H5C_NOTIFY_ACTION_ENTRY_DIRTIED, entry_ptr) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                            "can't notify client about entry dirty flag set");

            
            if (entry_ptr->flush_dep_nparents > 0)
                if (H5C__mark_flush_dep_dirty(entry_ptr) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKDIRTY, FAIL, "Can't propagate flush dep dirty flag");
        } 
    }     

done:
#ifdef H5C_DO_EXTREME_SANITY_CHECKS
    if (H5C__validate_protected_entry_list(cache_ptr) < 0 || H5C__validate_pinned_entry_list(cache_ptr) < 0)
        HDONE_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "an extreme sanity check failed on Rexit");
#endif 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_pin_protected_entry(void *thing)
{
    H5C_t             *cache_ptr;
    H5C_cache_entry_t *entry_ptr = (H5C_cache_entry_t *)thing; 
    herr_t             ret_value = SUCCEED;                    

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(entry_ptr);
    assert(H5_addr_defined(entry_ptr->addr));
    cache_ptr = entry_ptr->cache_ptr;
    assert(cache_ptr);

#ifdef H5C_DO_EXTREME_SANITY_CHECKS
    if (H5C__validate_protected_entry_list(cache_ptr) < 0 || H5C__validate_pinned_entry_list(cache_ptr) < 0 ||
        H5C__validate_lru_list(cache_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "an extreme sanity check failed on entry");
#endif 

    
    if (!entry_ptr->is_protected)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTPIN, FAIL, "Entry isn't protected");

    
    if (H5C__pin_entry_from_client(cache_ptr, entry_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTPIN, FAIL, "Can't pin entry by client");

done:
#ifdef H5C_DO_EXTREME_SANITY_CHECKS
    if (H5C__validate_protected_entry_list(cache_ptr) < 0 || H5C__validate_pinned_entry_list(cache_ptr) < 0 ||
        H5C__validate_lru_list(cache_ptr) < 0)
        HDONE_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "an extreme sanity check failed on Rexit");
#endif 

    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5C_protect(H5F_t *f, const H5C_class_t *type, haddr_t addr, void *udata, unsigned flags)
{
    H5C_t      *cache_ptr;
    H5AC_ring_t ring = H5C_RING_UNDEFINED;
    bool        hit;
    bool        have_write_permitted = false;
    bool        read_only            = false;
    bool        flush_last;
#ifdef H5_HAVE_PARALLEL
    bool coll_access = false; 
#endif                        
    bool               write_permitted = false;
    bool               was_loaded      = false; 
    size_t             empty_space;
    void              *thing;
    H5C_cache_entry_t *entry_ptr;
    void              *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(f);
    assert(f->shared);
    cache_ptr = f->shared->cache;
    assert(cache_ptr);
    assert(type);
    assert(type->mem_type == cache_ptr->class_table_ptr[type->id]->mem_type);
    assert(H5_addr_defined(addr));

#ifdef H5C_DO_EXTREME_SANITY_CHECKS
    if (H5C__validate_protected_entry_list(cache_ptr) < 0 || H5C__validate_pinned_entry_list(cache_ptr) < 0 ||
        H5C__validate_lru_list(cache_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, NULL, "an extreme sanity check failed on entry");
#endif 

    
    if (cache_ptr->load_image) {
        cache_ptr->load_image = false;
        if (H5C__load_cache_image(f) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTLOAD, NULL, "Can't load cache image");
    } 

    read_only  = ((flags & H5C__READ_ONLY_FLAG) != 0);
    flush_last = ((flags & H5C__FLUSH_LAST_FLAG) != 0);

    
    ring = H5CX_get_ring();

#ifdef H5_HAVE_PARALLEL
    if (H5F_HAS_FEATURE(f, H5FD_FEAT_HAS_MPI))
        coll_access = H5F_get_coll_metadata_reads(f);
#endif 

    
    H5C__SEARCH_INDEX(cache_ptr, addr, entry_ptr, NULL);

    if (entry_ptr != NULL) {
        if (entry_ptr->ring != ring)
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, NULL, "ring type mismatch occurred for cache entry");

        if (entry_ptr->prefetched) {
            
            if (H5C__deserialize_prefetched_entry(f, cache_ptr, &entry_ptr, type, addr, udata) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTLOAD, NULL, "can't deserialize prefetched entry");

            assert(!entry_ptr->prefetched);
            assert(entry_ptr->addr == addr);
        } 

        
        if (entry_ptr->type != type)
            HGOTO_ERROR(H5E_CACHE, H5E_BADTYPE, NULL, "incorrect cache entry type");

#ifdef H5_HAVE_PARALLEL
        
        if (coll_access) {
            if (!entry_ptr->is_dirty && !entry_ptr->coll_access) {
                MPI_Comm comm;     
                int      mpi_code; 
                int      buf_size;

                if (MPI_COMM_NULL == (comm = H5F_mpi_get_comm(f)))
                    HGOTO_ERROR(H5E_FILE, H5E_CANTGET, NULL, "get_comm request failed");

                if (entry_ptr->image_ptr == NULL) {
                    int mpi_rank;

                    if ((mpi_rank = H5F_mpi_get_rank(f)) < 0)
                        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, NULL, "Can't get MPI rank");

                    if (NULL == (entry_ptr->image_ptr = H5MM_malloc(entry_ptr->size + H5C_IMAGE_EXTRA_SPACE)))
                        HGOTO_ERROR(H5E_CACHE, H5E_CANTALLOC, NULL,
                                    "memory allocation failed for on disk image buffer");
#if H5C_DO_MEMORY_SANITY_CHECKS
                    H5MM_memcpy(((uint8_t *)entry_ptr->image_ptr) + entry_ptr->size, H5C_IMAGE_SANITY_VALUE,
                                H5C_IMAGE_EXTRA_SPACE);
#endif 
                    if (0 == mpi_rank && H5C__generate_image(f, cache_ptr, entry_ptr) < 0)
                        
                        HDONE_ERROR(H5E_CACHE, H5E_CANTGET, NULL, "can't generate entry's image");
                } 
                assert(entry_ptr->image_ptr);

                H5_CHECKED_ASSIGN(buf_size, int, entry_ptr->size, size_t);
                if (MPI_SUCCESS != (mpi_code = MPI_Bcast(entry_ptr->image_ptr, buf_size, MPI_BYTE, 0, comm)))
                    HMPI_GOTO_ERROR(NULL, "MPI_Bcast failed", mpi_code)

                
                entry_ptr->coll_access = true;
                H5C__INSERT_IN_COLL_LIST(cache_ptr, entry_ptr, NULL);
            } 
            else if (entry_ptr->coll_access)
                H5C__MOVE_TO_TOP_IN_COLL_LIST(cache_ptr, entry_ptr, NULL);
        } 
#endif    

#ifdef H5C_DO_TAGGING_SANITY_CHECKS
        {
            
            if (cache_ptr->ignore_tags != true) {
                haddr_t tag; 

                

                
                tag = H5CX_get_tag();

                if (H5C_verify_tag(entry_ptr->type->id, tag) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTGET, NULL, "tag verification failed");
            } 
        }
#endif

        hit   = true;
        thing = (void *)entry_ptr;
    }
    else {
        
        hit = false;
        if (NULL == (thing = H5C__load_entry(f,
#ifdef H5_HAVE_PARALLEL
                                             coll_access,
#endif 
                                             type, addr, udata)))
            HGOTO_ERROR(H5E_CACHE, H5E_CANTLOAD, NULL, "can't load entry");

        entry_ptr = (H5C_cache_entry_t *)thing;
        cache_ptr->entries_loaded_counter++;

        entry_ptr->ring = ring;
#ifdef H5_HAVE_PARALLEL
        if (H5F_HAS_FEATURE(f, H5FD_FEAT_HAS_MPI) && entry_ptr->coll_access)
            H5C__INSERT_IN_COLL_LIST(cache_ptr, entry_ptr, NULL);
#endif 

        
        if (H5C__tag_entry(cache_ptr, entry_ptr) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTTAG, NULL, "Cannot tag metadata entry");

        
        if (cache_ptr->flash_size_increase_possible &&
            (entry_ptr->size > cache_ptr->flash_size_increase_threshold))
            if (H5C__flash_increase_cache_size(cache_ptr, 0, entry_ptr->size) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTPROTECT, NULL, "H5C__flash_increase_cache_size failed");

        if (cache_ptr->index_size >= cache_ptr->max_cache_size)
            empty_space = 0;
        else
            empty_space = cache_ptr->max_cache_size - cache_ptr->index_size;

        
        if (cache_ptr->evictions_enabled &&
            (((cache_ptr->index_size + entry_ptr->size) > cache_ptr->max_cache_size) ||
             ((empty_space + cache_ptr->clean_index_size) < cache_ptr->min_clean_size))) {

            size_t space_needed;

            if (empty_space <= entry_ptr->size)
                cache_ptr->cache_full = true;

            if (cache_ptr->check_write_permitted != NULL) {
                if ((cache_ptr->check_write_permitted)(f, &write_permitted) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTPROTECT, NULL, "Can't get write_permitted 1");
                else
                    have_write_permitted = true;
            } 
            else {
                write_permitted      = cache_ptr->write_permitted;
                have_write_permitted = true;
            } 

            assert(entry_ptr->size <= H5C_MAX_ENTRY_SIZE);
            space_needed = entry_ptr->size;
            if (space_needed > cache_ptr->max_cache_size)
                space_needed = cache_ptr->max_cache_size;

            
            if (H5C__make_space_in_cache(f, space_needed, write_permitted) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTPROTECT, NULL, "H5C__make_space_in_cache failed");
        } 

        
        entry_ptr->flush_me_last = flush_last;

        H5C__INSERT_IN_INDEX(cache_ptr, entry_ptr, NULL);
        if (entry_ptr->is_dirty && !entry_ptr->in_slist)
            H5C__INSERT_ENTRY_IN_SLIST(cache_ptr, entry_ptr, NULL);

        
        H5C__UPDATE_RP_FOR_INSERTION(cache_ptr, entry_ptr, NULL);

        
        
        was_loaded = true;
    } 

    assert(entry_ptr->addr == addr);
    assert(entry_ptr->type == type);

    if (entry_ptr->is_protected) {
        if (read_only && entry_ptr->is_read_only) {
            assert(entry_ptr->ro_ref_count > 0);
            (entry_ptr->ro_ref_count)++;
        } 
        else
            HGOTO_ERROR(H5E_CACHE, H5E_CANTPROTECT, NULL, "Target already protected & not read only?!?");
    } 
    else {
        H5C__UPDATE_RP_FOR_PROTECT(cache_ptr, entry_ptr, NULL);

        entry_ptr->is_protected = true;
        if (read_only) {
            entry_ptr->is_read_only = true;
            entry_ptr->ro_ref_count = 1;
        } 
        entry_ptr->dirtied = false;
    } 

    H5C__UPDATE_CACHE_HIT_RATE_STATS(cache_ptr, hit);
    H5C__UPDATE_STATS_FOR_PROTECT(cache_ptr, entry_ptr, hit);

    ret_value = thing;

    if (cache_ptr->evictions_enabled &&
        (cache_ptr->size_decreased ||
         (cache_ptr->resize_enabled && (cache_ptr->cache_accesses >= cache_ptr->resize_ctl.epoch_length)))) {

        if (!have_write_permitted) {
            if (cache_ptr->check_write_permitted != NULL) {
                if ((cache_ptr->check_write_permitted)(f, &write_permitted) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTPROTECT, NULL, "Can't get write_permitted");
                else
                    have_write_permitted = true;
            }
            else {
                write_permitted      = cache_ptr->write_permitted;
                have_write_permitted = true;
            }
        }

        if (cache_ptr->resize_enabled && (cache_ptr->cache_accesses >= cache_ptr->resize_ctl.epoch_length))
            if (H5C__auto_adjust_cache_size(f, write_permitted) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTPROTECT, NULL, "Cache auto-resize failed");

        if (cache_ptr->size_decreased) {
            cache_ptr->size_decreased = false;

            
            if (cache_ptr->index_size >= cache_ptr->max_cache_size)
                empty_space = 0;
            else
                empty_space = cache_ptr->max_cache_size - cache_ptr->index_size;

            if ((cache_ptr->index_size > cache_ptr->max_cache_size) ||
                ((empty_space + cache_ptr->clean_index_size) < cache_ptr->min_clean_size)) {

                if (cache_ptr->index_size > cache_ptr->max_cache_size)
                    cache_ptr->cache_full = true;

                if (H5C__make_space_in_cache(f, (size_t)0, write_permitted) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTPROTECT, NULL, "H5C__make_space_in_cache failed");
            }
        } 
    }

    
    if (was_loaded)
        
        if (entry_ptr->type->notify && (entry_ptr->type->notify)(H5C_NOTIFY_ACTION_AFTER_LOAD, entry_ptr) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, NULL,
                        "can't notify client about entry inserted into cache");

#ifdef H5_HAVE_PARALLEL
    
    if (coll_access) {
        if (H5P_USER_TRUE == H5F_COLL_MD_READ(f)) {
            if (cache_ptr->max_cache_size * 80 < cache_ptr->coll_list_size * 100)
                if (H5C_clear_coll_entries(cache_ptr, true) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, NULL, "can't clear collective metadata entries");
        } 
        else {
            if (cache_ptr->max_cache_size * 40 < cache_ptr->coll_list_size * 100)
                if (H5C_clear_coll_entries(cache_ptr, true) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, NULL, "can't clear collective metadata entries");
        } 
    }     
#endif    

done:
#ifdef H5C_DO_EXTREME_SANITY_CHECKS
    if (H5C__validate_protected_entry_list(cache_ptr) < 0 || H5C__validate_pinned_entry_list(cache_ptr) < 0 ||
        H5C__validate_lru_list(cache_ptr) < 0)
        HDONE_ERROR(H5E_CACHE, H5E_SYSTEM, NULL, "an extreme sanity check failed on Rexit");
#endif 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_unpin_entry(void *_entry_ptr)
{
    H5C_t             *cache_ptr;
    H5C_cache_entry_t *entry_ptr = (H5C_cache_entry_t *)_entry_ptr; 
    herr_t             ret_value = SUCCEED;                         

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(entry_ptr);
    cache_ptr = entry_ptr->cache_ptr;
    assert(cache_ptr);

#ifdef H5C_DO_EXTREME_SANITY_CHECKS
    if (H5C__validate_protected_entry_list(cache_ptr) < 0 || H5C__validate_pinned_entry_list(cache_ptr) < 0 ||
        H5C__validate_lru_list(cache_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "an extreme sanity check failed on entry");
#endif 

    
    if (H5C__unpin_entry_from_client(cache_ptr, entry_ptr, true) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPIN, FAIL, "Can't unpin entry from client");

done:
#ifdef H5C_DO_EXTREME_SANITY_CHECKS
    if (H5C__validate_protected_entry_list(cache_ptr) < 0 || H5C__validate_pinned_entry_list(cache_ptr) < 0 ||
        H5C__validate_lru_list(cache_ptr) < 0)
        HDONE_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "an extreme sanity check failed on Rexit");
#endif 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_unprotect(H5F_t *f, haddr_t addr, void *thing, unsigned flags)
{
    H5C_t *cache_ptr;
    bool   deleted;
    bool   dirtied;
    bool   pin_entry;
    bool   unpin_entry;
    bool   free_file_space;
    bool   take_ownership;
    bool   was_clean;
#ifdef H5_HAVE_PARALLEL
    bool clear_entry = false;
#endif 
    H5C_cache_entry_t *entry_ptr;
    H5C_cache_entry_t *test_entry_ptr;
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    deleted         = ((flags & H5C__DELETED_FLAG) != 0);
    dirtied         = ((flags & H5C__DIRTIED_FLAG) != 0);
    pin_entry       = ((flags & H5C__PIN_ENTRY_FLAG) != 0);
    unpin_entry     = ((flags & H5C__UNPIN_ENTRY_FLAG) != 0);
    free_file_space = ((flags & H5C__FREE_FILE_SPACE_FLAG) != 0);
    take_ownership  = ((flags & H5C__TAKE_OWNERSHIP_FLAG) != 0);

    assert(f);
    assert(f->shared);

    cache_ptr = f->shared->cache;

    assert(cache_ptr);
    assert(H5_addr_defined(addr));
    assert(thing);
    assert(!(pin_entry && unpin_entry));

    
    assert((!free_file_space) || (deleted));

    
    assert((!take_ownership) || (deleted));

    
    assert(!(free_file_space && take_ownership));

    entry_ptr = (H5C_cache_entry_t *)thing;
    assert(entry_ptr->addr == addr);

    
    dirtied |= entry_ptr->dirtied;
    was_clean = !(entry_ptr->is_dirty);

#ifdef H5C_DO_EXTREME_SANITY_CHECKS
    if (H5C__validate_protected_entry_list(cache_ptr) < 0 || H5C__validate_pinned_entry_list(cache_ptr) < 0 ||
        H5C__validate_lru_list(cache_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "an extreme sanity check failed on entry");
#endif 

    
    if (entry_ptr->ro_ref_count > 1) {
        
        assert(entry_ptr->is_protected);
        assert(entry_ptr->is_read_only);

        if (dirtied)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPROTECT, FAIL, "Read only entry modified??");

        
        (entry_ptr->ro_ref_count)--;

        
        if (pin_entry) {
            
            if (H5C__pin_entry_from_client(cache_ptr, entry_ptr) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTPIN, FAIL, "Can't pin entry by client");
        }
        else if (unpin_entry) {
            
            if (H5C__unpin_entry_from_client(cache_ptr, entry_ptr, false) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPIN, FAIL, "Can't unpin entry by client");
        } 
    }
    else {
        if (entry_ptr->is_read_only) {
            
            assert(entry_ptr->ro_ref_count == 1);

            if (dirtied)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPROTECT, FAIL, "Read only entry modified??");

            entry_ptr->is_read_only = false;
            entry_ptr->ro_ref_count = 0;
        } 

#ifdef H5_HAVE_PARALLEL
        
        if (entry_ptr->clear_on_unprotect) {
            
            assert(entry_ptr->is_dirty);

            entry_ptr->clear_on_unprotect = false;
            if (!dirtied)
                clear_entry = true;
        } 
#endif    

        if (!entry_ptr->is_protected)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPROTECT, FAIL, "Entry already unprotected??");

        
        entry_ptr->is_dirty = (entry_ptr->is_dirty || dirtied);
        if (dirtied && entry_ptr->image_up_to_date) {
            entry_ptr->image_up_to_date = false;
            if (entry_ptr->flush_dep_nparents > 0)
                if (H5C__mark_flush_dep_unserialized(entry_ptr) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                                "Can't propagate serialization status to fd parents");
        } 

        
        if (was_clean && entry_ptr->is_dirty) {
            
            H5C__UPDATE_INDEX_FOR_ENTRY_DIRTY(cache_ptr, entry_ptr, FAIL);

            
            if (entry_ptr->type->notify &&
                (entry_ptr->type->notify)(H5C_NOTIFY_ACTION_ENTRY_DIRTIED, entry_ptr) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                            "can't notify client about entry dirty flag set");

            
            if (entry_ptr->flush_dep_nparents > 0)
                if (H5C__mark_flush_dep_dirty(entry_ptr) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKDIRTY, FAIL, "Can't propagate flush dep dirty flag");
        } 
        
        else if (!was_clean && !entry_ptr->is_dirty) {

            
            if (entry_ptr->type->notify &&
                (entry_ptr->type->notify)(H5C_NOTIFY_ACTION_ENTRY_CLEANED, entry_ptr) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                            "can't notify client about entry dirty flag cleared");

            
            if (entry_ptr->flush_dep_nparents > 0)
                if (H5C__mark_flush_dep_clean(entry_ptr) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKDIRTY, FAIL, "Can't propagate flush dep dirty flag");
        } 

        
        if (pin_entry) {
            
            if (H5C__pin_entry_from_client(cache_ptr, entry_ptr) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTPIN, FAIL, "Can't pin entry by client");
        }
        else if (unpin_entry) {
            
            if (H5C__unpin_entry_from_client(cache_ptr, entry_ptr, false) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPIN, FAIL, "Can't unpin entry by client");
        } 

        
        H5C__UPDATE_RP_FOR_UNPROTECT(cache_ptr, entry_ptr, FAIL);

        entry_ptr->is_protected = false;

        
        if (entry_ptr->is_dirty && !entry_ptr->in_slist)
            
            H5C__INSERT_ENTRY_IN_SLIST(cache_ptr, entry_ptr, FAIL);

        
        if (deleted) {
            unsigned flush_flags = (H5C__FLUSH_CLEAR_ONLY_FLAG | H5C__FLUSH_INVALIDATE_FLAG);

            
            H5C__SEARCH_INDEX(cache_ptr, addr, test_entry_ptr, FAIL);

            if (test_entry_ptr == NULL)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPROTECT, FAIL, "entry not in hash table?!?");
            else if (test_entry_ptr != entry_ptr)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPROTECT, FAIL,
                            "hash table contains multiple entries for addr?!?");

            
            if (free_file_space)
                flush_flags |= H5C__FREE_FILE_SPACE_FLAG;

            
            if (take_ownership)
                flush_flags |= H5C__TAKE_OWNERSHIP_FLAG;

            
            flush_flags |= H5C__DEL_FROM_SLIST_ON_DESTROY_FLAG;

            assert((!cache_ptr->slist_enabled) || (((!was_clean) || dirtied) == (entry_ptr->in_slist)));

            if (H5C__flush_single_entry(f, entry_ptr, flush_flags) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPROTECT, FAIL, "Can't flush entry");
        } 
#ifdef H5_HAVE_PARALLEL
        else if (clear_entry) {
            
            H5C__SEARCH_INDEX(cache_ptr, addr, test_entry_ptr, FAIL);

            if (test_entry_ptr == NULL)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPROTECT, FAIL, "entry not in hash table?!?");
            else if (test_entry_ptr != entry_ptr)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPROTECT, FAIL,
                            "hash table contains multiple entries for addr?!?");

            if (H5C__flush_single_entry(f, entry_ptr,
                                        H5C__FLUSH_CLEAR_ONLY_FLAG | H5C__DEL_FROM_SLIST_ON_DESTROY_FLAG) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPROTECT, FAIL, "Can't clear entry");
        } 
#endif    
    }

    H5C__UPDATE_STATS_FOR_UNPROTECT(cache_ptr);

done:
#ifdef H5C_DO_EXTREME_SANITY_CHECKS
    if (H5C__validate_protected_entry_list(cache_ptr) < 0 || H5C__validate_pinned_entry_list(cache_ptr) < 0 ||
        H5C__validate_lru_list(cache_ptr) < 0)
        HDONE_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "an extreme sanity check failed on Rexit");
#endif 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_unsettle_entry_ring(void *_entry)
{
    H5C_cache_entry_t *entry = (H5C_cache_entry_t *)_entry; 
    H5C_t             *cache;                               
    herr_t             ret_value = SUCCEED;                 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(entry);
    assert(entry->ring != H5C_RING_UNDEFINED);
    assert((H5C_RING_USER == entry->ring) || (H5C_RING_RDFSM == entry->ring) ||
           (H5C_RING_MDFSM == entry->ring));
    cache = entry->cache_ptr;
    assert(cache);

    switch (entry->ring) {
        case H5C_RING_USER:
            
            break;

        case H5C_RING_RDFSM:
            if (cache->rdfsm_settled) {
                if (cache->flush_in_progress || cache->close_warning_received)
                    HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "unexpected rdfsm ring unsettle");
                cache->rdfsm_settled = false;
            } 
            break;

        case H5C_RING_MDFSM:
            if (cache->mdfsm_settled) {
                if (cache->flush_in_progress || cache->close_warning_received)
                    HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "unexpected mdfsm ring unsettle");
                cache->mdfsm_settled = false;
            } 
            break;

        default:
            assert(false); 
            break;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_create_flush_dependency(void *parent_thing, void *child_thing)
{
    H5C_t             *cache_ptr;
    H5C_cache_entry_t *parent_entry = (H5C_cache_entry_t *)parent_thing; 
    H5C_cache_entry_t *child_entry  = (H5C_cache_entry_t *)child_thing;  
    herr_t             ret_value    = SUCCEED;                           

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(parent_entry);
    assert(H5_addr_defined(parent_entry->addr));
    assert(child_entry);
    assert(H5_addr_defined(child_entry->addr));
    cache_ptr = parent_entry->cache_ptr;
    assert(cache_ptr);
    assert(cache_ptr == child_entry->cache_ptr);
#ifndef NDEBUG
    
    {
        unsigned u;

        for (u = 0; u < child_entry->flush_dep_nparents; u++)
            assert(child_entry->flush_dep_parent[u] != parent_entry);
    } 
#endif

    
    if (child_entry == parent_entry)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTDEPEND, FAIL, "Child entry flush dependency parent can't be itself");
    if (!(parent_entry->is_protected || parent_entry->is_pinned))
        HGOTO_ERROR(H5E_CACHE, H5E_CANTDEPEND, FAIL, "Parent entry isn't pinned or protected");

    
    if (!parent_entry->is_pinned) {
        
        assert(parent_entry->flush_dep_nchildren == 0);
        assert(!parent_entry->pinned_from_client);
        assert(!parent_entry->pinned_from_cache);

        
        parent_entry->is_pinned = true;
        H5C__UPDATE_STATS_FOR_PIN(cache_ptr, parent_entry);
    } 

    
    parent_entry->pinned_from_cache = true;

    
    if (child_entry->flush_dep_nparents >= child_entry->flush_dep_parent_nalloc) {
        if (child_entry->flush_dep_parent_nalloc == 0) {
            
            assert(!child_entry->flush_dep_parent);

            if (NULL == (child_entry->flush_dep_parent =
                             H5FL_SEQ_MALLOC(H5C_cache_entry_ptr_t, H5C_FLUSH_DEP_PARENT_INIT)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                            "memory allocation failed for flush dependency parent list");
            child_entry->flush_dep_parent_nalloc = H5C_FLUSH_DEP_PARENT_INIT;
        } 
        else {
            
            assert(child_entry->flush_dep_parent);

            if (NULL == (child_entry->flush_dep_parent =
                             H5FL_SEQ_REALLOC(H5C_cache_entry_ptr_t, child_entry->flush_dep_parent,
                                              2 * child_entry->flush_dep_parent_nalloc)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                            "memory allocation failed for flush dependency parent list");
            child_entry->flush_dep_parent_nalloc *= 2;
        } 
        cache_ptr->entry_fd_height_change_counter++;
    } 

    
    child_entry->flush_dep_parent[child_entry->flush_dep_nparents] = parent_entry;
    child_entry->flush_dep_nparents++;

    
    parent_entry->flush_dep_nchildren++;

    
    if (child_entry->is_dirty) {
        
        assert(parent_entry->flush_dep_ndirty_children < parent_entry->flush_dep_nchildren);

        parent_entry->flush_dep_ndirty_children++;

        
        if (parent_entry->type->notify &&
            (parent_entry->type->notify)(H5C_NOTIFY_ACTION_CHILD_DIRTIED, parent_entry) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                        "can't notify parent about child entry dirty flag set");
    } 

    
    if (!child_entry->image_up_to_date) {
        assert(parent_entry->flush_dep_nunser_children < parent_entry->flush_dep_nchildren);

        parent_entry->flush_dep_nunser_children++;

        
        if (parent_entry->type->notify &&
            (parent_entry->type->notify)(H5C_NOTIFY_ACTION_CHILD_UNSERIALIZED, parent_entry) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                        "can't notify parent about child entry serialized flag reset");
    } 

    
    assert(parent_entry->is_pinned);
    assert(parent_entry->flush_dep_nchildren > 0);
    assert(child_entry->flush_dep_parent);
    assert(child_entry->flush_dep_nparents > 0);
    assert(child_entry->flush_dep_parent_nalloc > 0);
#ifndef NDEBUG
    H5C__assert_flush_dep_nocycle(parent_entry, child_entry);
#endif

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_destroy_flush_dependency(void *parent_thing, void *child_thing)
{
    H5C_t             *cache_ptr;
    H5C_cache_entry_t *parent_entry = (H5C_cache_entry_t *)parent_thing; 
    H5C_cache_entry_t *child_entry  = (H5C_cache_entry_t *)child_thing;  
    unsigned           u;                                                
    herr_t             ret_value = SUCCEED;                              

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(parent_entry);
    assert(H5_addr_defined(parent_entry->addr));
    assert(child_entry);
    assert(H5_addr_defined(child_entry->addr));
    cache_ptr = parent_entry->cache_ptr;
    assert(cache_ptr);
    assert(cache_ptr == child_entry->cache_ptr);

    
    if (!parent_entry->is_pinned)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTUNDEPEND, FAIL, "Parent entry isn't pinned");
    if (NULL == child_entry->flush_dep_parent)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTUNDEPEND, FAIL,
                    "Child entry doesn't have a flush dependency parent array");
    if (0 == parent_entry->flush_dep_nchildren)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTUNDEPEND, FAIL,
                    "Parent entry flush dependency ref. count has no child dependencies");

    
    for (u = 0; u < child_entry->flush_dep_nparents; u++)
        if (child_entry->flush_dep_parent[u] == parent_entry)
            break;
    if (u == child_entry->flush_dep_nparents)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTUNDEPEND, FAIL,
                    "Parent entry isn't a flush dependency parent for child entry");

    
    if (u < (child_entry->flush_dep_nparents - 1))
        memmove(&child_entry->flush_dep_parent[u], &child_entry->flush_dep_parent[u + 1],
                (child_entry->flush_dep_nparents - u - 1) * sizeof(child_entry->flush_dep_parent[0]));
    child_entry->flush_dep_nparents--;

    
    parent_entry->flush_dep_nchildren--;
    if (0 == parent_entry->flush_dep_nchildren) {
        
        assert(parent_entry->pinned_from_cache);

        
        if (!parent_entry->pinned_from_client)
            if (H5C__unpin_entry_real(cache_ptr, parent_entry, true) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPIN, FAIL, "Can't unpin entry");

        
        parent_entry->pinned_from_cache = false;
    } 

    
    if (child_entry->is_dirty) {
        
        assert(parent_entry->flush_dep_ndirty_children > 0);

        parent_entry->flush_dep_ndirty_children--;

        
        if (parent_entry->type->notify &&
            (parent_entry->type->notify)(H5C_NOTIFY_ACTION_CHILD_CLEANED, parent_entry) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                        "can't notify parent about child entry dirty flag reset");
    } 

    
    if (!child_entry->image_up_to_date) {
        assert(parent_entry->flush_dep_nunser_children > 0);

        parent_entry->flush_dep_nunser_children--;

        
        if (parent_entry->type->notify &&
            (parent_entry->type->notify)(H5C_NOTIFY_ACTION_CHILD_SERIALIZED, parent_entry) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL,
                        "can't notify parent about child entry serialized flag set");
    } 

    
    if (child_entry->flush_dep_nparents == 0) {
        child_entry->flush_dep_parent = H5FL_SEQ_FREE(H5C_cache_entry_ptr_t, child_entry->flush_dep_parent);
        child_entry->flush_dep_parent_nalloc = 0;
    } 
    else if (child_entry->flush_dep_parent_nalloc > H5C_FLUSH_DEP_PARENT_INIT &&
             child_entry->flush_dep_nparents <= (child_entry->flush_dep_parent_nalloc / 4)) {
        if (NULL == (child_entry->flush_dep_parent =
                         H5FL_SEQ_REALLOC(H5C_cache_entry_ptr_t, child_entry->flush_dep_parent,
                                          child_entry->flush_dep_parent_nalloc / 4)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                        "memory allocation failed for flush dependency parent list");
        child_entry->flush_dep_parent_nalloc /= 4;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_expunge_entry(H5F_t *f, const H5C_class_t *type, haddr_t addr, unsigned flags)
{
    H5C_t             *cache_ptr;
    H5C_cache_entry_t *entry_ptr   = NULL;
    unsigned           flush_flags = (H5C__FLUSH_INVALIDATE_FLAG | H5C__FLUSH_CLEAR_ONLY_FLAG);
    herr_t             ret_value   = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(f);
    assert(f->shared);
    cache_ptr = f->shared->cache;
    assert(cache_ptr);
    assert(type);
    assert(H5_addr_defined(addr));

#ifdef H5C_DO_EXTREME_SANITY_CHECKS
    if (H5C__validate_lru_list(cache_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "LRU extreme sanity check failed on entry");
#endif 

    
    H5C__SEARCH_INDEX(cache_ptr, addr, entry_ptr, FAIL);
    if ((entry_ptr == NULL) || (entry_ptr->type != type))
        
        HGOTO_DONE(SUCCEED);

    assert(entry_ptr->addr == addr);
    assert(entry_ptr->type == type);

    
    if (entry_ptr->is_protected)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTEXPUNGE, FAIL, "Target entry is protected");
    if (entry_ptr->is_pinned)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTEXPUNGE, FAIL, "Target entry is pinned");

    

    
    flush_flags |= (flags & H5C__FREE_FILE_SPACE_FLAG);

    
    flush_flags |= H5C__DEL_FROM_SLIST_ON_DESTROY_FLAG;

    if (H5C__flush_single_entry(f, entry_ptr, flush_flags) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTEXPUNGE, FAIL, "can't flush entry");

done:
#ifdef H5C_DO_EXTREME_SANITY_CHECKS
    if (H5C__validate_lru_list(cache_ptr) < 0)
        HDONE_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "LRU extreme sanity check failed on Rexit");
#endif 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_remove_entry(void *_entry)
{
    H5C_cache_entry_t *entry = (H5C_cache_entry_t *)_entry; 
    H5C_t             *cache;                               
    herr_t             ret_value = SUCCEED;                 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(entry);
    assert(entry->ring != H5C_RING_UNDEFINED);
    cache = entry->cache_ptr;
    assert(cache);

    
    if (entry->is_dirty)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTREMOVE, FAIL, "can't remove dirty entry from cache");
    if (entry->is_protected)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTREMOVE, FAIL, "can't remove protected entry from cache");
    if (entry->is_pinned)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTREMOVE, FAIL, "can't remove pinned entry from cache");
    
    if (entry->flush_dep_nparents > 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTREMOVE, FAIL,
                    "can't remove entry with flush dependency parents from cache");
    if (entry->flush_dep_nchildren > 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTREMOVE, FAIL,
                    "can't remove entry with flush dependency children from cache");

    
    assert(!entry->in_slist);
    assert(!entry->flush_in_progress);

    

    
    H5C__UPDATE_STATS_FOR_EVICTION(cache, entry, true);

    
    if (entry->type->notify && (entry->type->notify)(H5C_NOTIFY_ACTION_BEFORE_EVICT, entry) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTNOTIFY, FAIL, "can't notify client about entry to evict");

    

    H5C__DELETE_FROM_INDEX(cache, entry, FAIL);

#ifdef H5_HAVE_PARALLEL
    
    if (entry->coll_access) {
        entry->coll_access = false;
        H5C__REMOVE_FROM_COLL_LIST(cache, entry, FAIL);
    }  
#endif 

    H5C__UPDATE_RP_FOR_EVICTION(cache, entry, FAIL);

    
    if (H5C__untag_entry(cache, entry) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTREMOVE, FAIL, "can't remove entry from tag list");

    
    cache->entries_removed_counter++;
    cache->last_entry_removed_ptr = entry;
    if (entry == cache->entry_watched_for_removal)
        cache->entry_watched_for_removal = NULL;

    

    
    if (entry->image_ptr != NULL)
        entry->image_ptr = H5MM_xfree(entry->image_ptr);

    
    entry->cache_ptr = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
