/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Tmodule.h" 
#define H5R_FRIEND     

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5Rpkg.h"      
#include "H5Tconv.h"     
#include "H5Tconv_reference.h"

H5FL_BLK_DEFINE_STATIC(ref_seq);

herr_t
H5T__conv_ref(const H5T_t *src, const H5T_t *dst, H5T_cdata_t *cdata,
              const H5T_conv_ctx_t H5_ATTR_UNUSED *conv_ctx, size_t nelmts, size_t buf_stride,
              size_t bkg_stride, void *buf, void *bkg)
{
    uint8_t *s        = NULL;        
    uint8_t *d        = NULL;        
    uint8_t *b        = NULL;        
    ssize_t  s_stride = 0;           
    ssize_t  d_stride = 0;           
    ssize_t  b_stride;               
    size_t   safe          = 0;      
    void    *conv_buf      = NULL;   
    size_t   conv_buf_size = 0;      
    size_t   elmtno        = 0;      
    size_t   orig_d_stride = 0;      
    size_t   orig_nelmts   = nelmts; 
    bool     convert_forward =
        true; 
    bool conversions_made =
        false; 
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    switch (cdata->command) {
        case H5T_CONV_INIT:
            
            if (NULL == src || NULL == dst)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, FAIL, "not a datatype");
            if (H5T_REFERENCE != src->shared->type)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, FAIL, "not a H5T_REFERENCE datatype");
            if (H5T_REFERENCE != dst->shared->type)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, FAIL, "not a H5T_REFERENCE datatype");
            
            if (!dst->shared->u.atomic.u.r.opaque)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, FAIL, "not an H5T_STD_REF datatype");

            
            cdata->need_bkg = H5T_BKG_NO;
            break;

        case H5T_CONV_FREE:
            break;

        case H5T_CONV_CONV: {
            
            if (NULL == src || NULL == dst)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");

            assert(src->shared->u.atomic.u.r.cls);

            
            if (buf_stride) {
                assert(buf_stride >= src->shared->size);
                assert(buf_stride >= dst->shared->size);
                H5_CHECK_OVERFLOW(buf_stride, size_t, ssize_t);
                s_stride = d_stride = (ssize_t)buf_stride;
            } 
            else {
                H5_CHECK_OVERFLOW(src->shared->size, size_t, ssize_t);
                H5_CHECK_OVERFLOW(dst->shared->size, size_t, ssize_t);
                s_stride = (ssize_t)src->shared->size;
                d_stride = (ssize_t)dst->shared->size;
            } 
            if (bkg) {
                if (bkg_stride)
                    b_stride = (ssize_t)bkg_stride;
                else
                    b_stride = d_stride;
            } 
            else
                b_stride = 0;

            
            orig_d_stride   = (size_t)d_stride;
            convert_forward = !(d_stride > s_stride);

            
            
            while (nelmts > 0) {
                
                if (d_stride > s_stride) {
                    
                    assert(s_stride > 0);
                    assert(d_stride > 0);
                    assert(b_stride >= 0);

                    
                    
                    
                    safe =
                        nelmts - (((nelmts * (size_t)s_stride) + ((size_t)d_stride - 1)) / (size_t)d_stride);

                    
                    
                    if (safe < 2) {
                        s = (uint8_t *)buf + (nelmts - 1) * (size_t)s_stride;
                        d = (uint8_t *)buf + (nelmts - 1) * (size_t)d_stride;
                        if (bkg)
                            b = (uint8_t *)bkg + (nelmts - 1) * (size_t)b_stride;
                        s_stride = -s_stride;
                        d_stride = -d_stride;
                        b_stride = -b_stride;

                        safe = nelmts;
                    } 
                    else {
                        s = (uint8_t *)buf + (nelmts - safe) * (size_t)s_stride;
                        d = (uint8_t *)buf + (nelmts - safe) * (size_t)d_stride;
                        if (bkg)
                            b = (uint8_t *)bkg + (nelmts - safe) * (size_t)b_stride;
                    } 
                }     
                else {
                    
                    s = d = (uint8_t *)buf;
                    b     = (uint8_t *)bkg;
                    safe  = nelmts;
                } 

                for (elmtno = 0; elmtno < safe; elmtno++) {
                    size_t buf_size;
                    bool   dst_copy = false;
                    bool   is_nil; 

                    
                    if ((*(src->shared->u.atomic.u.r.cls->isnull))(src->shared->u.atomic.u.r.file, s,
                                                                   &is_nil) < 0)
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, FAIL,
                                    "can't check if reference data is 'nil'");

                    if (is_nil) {
                        
                        if ((*(dst->shared->u.atomic.u.r.cls->setnull))(dst->shared->u.atomic.u.r.file, d,
                                                                        b) < 0)
                            HGOTO_ERROR(H5E_DATATYPE, H5E_WRITEERROR, FAIL,
                                        "can't set reference data to 'nil'");
                    } 
                    else {
                        
                        if (0 == (buf_size = src->shared->u.atomic.u.r.cls->getsize(
                                      src->shared->u.atomic.u.r.file, s, src->shared->size,
                                      dst->shared->u.atomic.u.r.file, &dst_copy)))
                            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "unable to obtain size of reference");

                        
                        if (conv_buf_size < buf_size) {
                            conv_buf_size = buf_size;
                            if (NULL == (conv_buf = H5FL_BLK_REALLOC(ref_seq, conv_buf, conv_buf_size)))
                                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                                            "memory allocation failed for type conversion");
                            memset(conv_buf, 0, conv_buf_size);
                        } 

                        if (dst_copy && (src->shared->u.atomic.u.r.loc == H5T_LOC_DISK))
                            H5MM_memcpy(conv_buf, s, buf_size);
                        else {
                            
                            if (src->shared->u.atomic.u.r.cls->read(
                                    src->shared->u.atomic.u.r.file, s, src->shared->size,
                                    dst->shared->u.atomic.u.r.file, conv_buf, buf_size) < 0)
                                HGOTO_ERROR(H5E_DATATYPE, H5E_READERROR, FAIL, "can't read reference data");
                        } 

                        if (dst_copy && (dst->shared->u.atomic.u.r.loc == H5T_LOC_DISK))
                            H5MM_memcpy(d, conv_buf, buf_size);
                        else {
                            
                            if (dst->shared->u.atomic.u.r.cls->write(
                                    src->shared->u.atomic.u.r.file, conv_buf, buf_size,
                                    src->shared->u.atomic.u.r.rtype, dst->shared->u.atomic.u.r.file, d,
                                    dst->shared->size, b) < 0)
                                HGOTO_ERROR(H5E_DATATYPE, H5E_WRITEERROR, FAIL, "can't write reference data");
                        } 
                    }     

                    
                    conversions_made = true;

                    
                    s += s_stride;
                    d += d_stride;

                    if (b)
                        b += b_stride;
                } 

                
                nelmts -= safe;
            } 
        }     
        break;

        default: 
            HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unknown conversion command");
    } 

done:
    
    if (ret_value < 0 && conversions_made) {
        H5R_ref_priv_t ref_priv;
        size_t         dest_count;

        
        if (nelmts < orig_nelmts || (convert_forward && elmtno < safe)) {
            dest_count = orig_nelmts - nelmts;

            
            if (convert_forward) {
                d = (uint8_t *)buf;
                dest_count += elmtno; 
            }
            else
                d = (uint8_t *)buf + (nelmts * orig_d_stride);

            
            while (dest_count > 0) {
                memcpy(&ref_priv, d, sizeof(H5R_ref_priv_t));
                H5R__destroy(&ref_priv); 
                d += orig_d_stride;
                dest_count--;
            }
        }

        
        if (!convert_forward && elmtno < safe) {
            dest_count = elmtno;

            
            if (d_stride > 0)
                d = (uint8_t *)buf + ((nelmts - safe) * orig_d_stride);
            else
                d = (uint8_t *)buf + ((nelmts - elmtno) * orig_d_stride);

            
            while (dest_count > 0) {
                memcpy(&ref_priv, d, sizeof(H5R_ref_priv_t));
                H5R__destroy(&ref_priv); 
                d += orig_d_stride;
                dest_count--;
            }
        }
    }

    
    if (conv_buf)
        conv_buf = H5FL_BLK_FREE(ref_seq, conv_buf);

    FUNC_LEAVE_NOAPI(ret_value)
} 
