/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5PLmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5MMprivate.h" 
#include "H5PLpkg.h"     

#define H5PL_INITIAL_PATH_CAPACITY 16

#define H5PL_PATH_CAPACITY_ADD 16

static herr_t H5PL__insert_at(const char *path, unsigned int idx);
static herr_t H5PL__make_space_at(unsigned int idx);
static herr_t H5PL__replace_at(const char *path, unsigned int idx);
static herr_t H5PL__expand_path_table(void);
static herr_t H5PL__path_table_iterate_process_path(const char *plugin_path, H5PL_iterate_type_t iter_type,
                                                    H5PL_iterate_t iter_op, void *op_data);
static herr_t H5PL__find_plugin_in_path(const H5PL_search_params_t *search_params, bool *found,
                                        const char *dir, const void **plugin_info);

static char **H5PL_paths_g = NULL;

static unsigned H5PL_num_paths_g = 0;

static unsigned H5PL_path_capacity_g = H5PL_INITIAL_PATH_CAPACITY;

static herr_t
H5PL__insert_at(const char *path, unsigned int idx)
{
    char  *path_copy = NULL;    
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(path);
    assert(strlen(path));

    
    if (H5PL_num_paths_g == H5PL_path_capacity_g)
        if (H5PL__expand_path_table() < 0)
            HGOTO_ERROR(H5E_PLUGIN, H5E_CANTALLOC, FAIL, "can't expand path table");

    
    if (NULL == (path_copy = H5MM_strdup(path)))
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTALLOC, FAIL, "can't make internal copy of path");

#ifdef H5_HAVE_WIN32_API
    
    if (H5_expand_windows_env_vars(&path_copy))
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTCONVERT, FAIL, "can't expand environment variable string");
#endif 

    
    if (H5PL_paths_g[idx])
        if (H5PL__make_space_at(idx) < 0)
            HGOTO_ERROR(H5E_PLUGIN, H5E_NOSPACE, FAIL, "unable to make space in the table for the new entry");

    
    H5PL_paths_g[idx] = path_copy;
    H5PL_num_paths_g++;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5PL__make_space_at(unsigned int idx)
{
    unsigned u;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(idx < H5PL_path_capacity_g);

    
    for (u = H5PL_num_paths_g; u > idx; u--)
        H5PL_paths_g[u] = H5PL_paths_g[u - 1];

    H5PL_paths_g[idx] = NULL;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5PL__replace_at(const char *path, unsigned int idx)
{
    char  *path_copy = NULL;    
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(path);
    assert(strlen(path));

    
    if (!H5PL_paths_g[idx])
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTFREE, FAIL, "path entry at index %u in the table is NULL", idx);

    
    if (NULL == (path_copy = H5MM_strdup(path)))
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTALLOC, FAIL, "can't make internal copy of path");

#ifdef H5_HAVE_WIN32_API
    
    if (H5_expand_windows_env_vars(&path_copy))
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTCONVERT, FAIL, "can't expand environment variable string");
#endif 

    
    H5PL_paths_g[idx] = (char *)H5MM_xfree(H5PL_paths_g[idx]);

    
    H5PL_paths_g[idx] = path_copy;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5PL__create_path_table(void)
{
    char *env_var = NULL;       
    char *paths   = NULL;       
    char  *next_path = NULL;    
    char  *lasts     = NULL;    
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    H5PL_num_paths_g     = 0;
    H5PL_path_capacity_g = H5PL_INITIAL_PATH_CAPACITY;
    if (NULL == (H5PL_paths_g = (char **)H5MM_calloc((size_t)H5PL_path_capacity_g * sizeof(char *))))
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTALLOC, FAIL, "can't allocate memory for path table");

    
    env_var = getenv(HDF5_PLUGIN_PATH);
    if (NULL == env_var)
        paths = H5MM_strdup(H5PL_DEFAULT_PATH);
    else
        paths = H5MM_strdup(env_var);

    if (NULL == paths)
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTALLOC, FAIL, "can't allocate memory for path copy");

    
    next_path = HDstrtok_r(paths, H5PL_PATH_SEPARATOR, &lasts);
    while (next_path) {

        
        if (H5PL__append_path(next_path) < 0)
            HGOTO_ERROR(H5E_PLUGIN, H5E_CANTALLOC, FAIL, "can't insert path: %s", next_path);

        
        next_path = HDstrtok_r(NULL, H5PL_PATH_SEPARATOR, &lasts);
    } 

done:
    if (paths)
        paths = (char *)H5MM_xfree(paths);

    
    if (FAIL == ret_value) {
        if (H5PL_paths_g)
            H5PL_paths_g = (char **)H5MM_xfree(H5PL_paths_g);
        H5PL_path_capacity_g = 0;
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5PL__close_path_table(void)
{
    unsigned u;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_NOERR

    
    for (u = 0; u < H5PL_num_paths_g; u++)
        if (H5PL_paths_g[u])
            H5PL_paths_g[u] = (char *)H5MM_xfree(H5PL_paths_g[u]);

    
    H5PL_paths_g = (char **)H5MM_xfree(H5PL_paths_g);

    
    H5PL_num_paths_g = 0;

    FUNC_LEAVE_NOAPI(ret_value)

} 

unsigned
H5PL__get_num_paths(void)
{
    FUNC_ENTER_PACKAGE_NOERR

    FUNC_LEAVE_NOAPI(H5PL_num_paths_g)

} 

static herr_t
H5PL__expand_path_table(void)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    H5PL_path_capacity_g += H5PL_PATH_CAPACITY_ADD;

    
    if (NULL ==
        (H5PL_paths_g = (char **)H5MM_realloc(H5PL_paths_g, (size_t)H5PL_path_capacity_g * sizeof(char *))))
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTALLOC, FAIL, "allocating additional memory for path table failed");

    
    memset(H5PL_paths_g + H5PL_num_paths_g, 0, (size_t)H5PL_PATH_CAPACITY_ADD * sizeof(char *));

done:
    
    if (FAIL == ret_value)
        H5PL_path_capacity_g -= H5PL_PATH_CAPACITY_ADD;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5PL__append_path(const char *path)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(path);
    assert(strlen(path));

    
    if (H5PL__insert_at(path, H5PL_num_paths_g) < 0)
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTINSERT, FAIL, "unable to append search path");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5PL__prepend_path(const char *path)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(path);
    assert(strlen(path));

    
    if (H5PL__insert_at(path, 0) < 0)
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTINSERT, FAIL, "unable to prepend search path");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5PL__replace_path(const char *path, unsigned int idx)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(path);
    assert(strlen(path));
    assert(idx < H5PL_path_capacity_g);

    
    if (H5PL__replace_at(path, idx) < 0)
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTINSERT, FAIL, "unable to replace search path");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5PL__insert_path(const char *path, unsigned int idx)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(path);
    assert(strlen(path));
    assert(idx < H5PL_path_capacity_g);

    
    if (H5PL__insert_at(path, idx) < 0)
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTINSERT, FAIL, "unable to insert search path");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5PL__remove_path(unsigned int idx)
{
    unsigned u;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(idx < H5PL_path_capacity_g);

    
    if (!H5PL_paths_g[idx])
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTDELETE, FAIL, "search path at index %u is NULL", idx);

    
    H5PL_num_paths_g--;
    H5PL_paths_g[idx] = (char *)H5MM_xfree(H5PL_paths_g[idx]);

    
    for (u = idx; u < H5PL_num_paths_g; u++)
        H5PL_paths_g[u] = H5PL_paths_g[u + 1];

    
    H5PL_paths_g[H5PL_num_paths_g] = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

const char *
H5PL__get_path(unsigned int idx)
{
    char *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    if (idx >= H5PL_num_paths_g)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, NULL, "path index %u is out of range in table", idx);

    return H5PL_paths_g[idx];
done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5PL__path_table_iterate(H5PL_iterate_type_t iter_type, H5PL_iterate_t iter_op, void *op_data)
{
    unsigned int u;
    herr_t       ret_value = H5_ITER_CONT;

    FUNC_ENTER_PACKAGE

    for (u = 0; (u < H5PL_num_paths_g) && (ret_value == H5_ITER_CONT); u++) {
        if ((ret_value =
                 H5PL__path_table_iterate_process_path(H5PL_paths_g[u], iter_type, iter_op, op_data)) < 0)
            HGOTO_ERROR(H5E_PLUGIN, H5E_BADITER, H5_ITER_ERROR,
                        "can't iterate over plugins in plugin path '%s'", H5PL_paths_g[u]);
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

#ifndef H5_HAVE_WIN32_API
static herr_t
H5PL__path_table_iterate_process_path(const char *plugin_path, H5PL_iterate_type_t iter_type,
                                      H5PL_iterate_t iter_op, void *op_data)
{
    H5PL_type_t    plugin_type;
    const void    *plugin_info = NULL;
    bool           plugin_loaded;
    char          *path      = NULL;
    DIR           *dirp      = NULL; 
    struct dirent *dp        = NULL; 
    herr_t         ret_value = H5_ITER_CONT;

    FUNC_ENTER_PACKAGE

    assert(plugin_path);
    assert(iter_op);

    
    if (!(dirp = HDopendir(plugin_path)))
        HGOTO_DONE(H5_ITER_CONT);

    
    while (NULL != (dp = HDreaddir(dirp))) {
        
#ifndef __CYGWIN__
        if (!strncmp(dp->d_name, "lib", (size_t)3) &&
            (strstr(dp->d_name, ".so") || strstr(dp->d_name, ".dylib"))) {
#else
        if (!strncmp(dp->d_name, "cyg", (size_t)3) && strstr(dp->d_name, ".dll")) {
#endif

            bool      plugin_matches;
            h5_stat_t my_stat;
            size_t    len;

            
            len = strlen(plugin_path) + strlen(H5PL_PATH_SEPARATOR) + strlen(dp->d_name) + 1  +
                  4; 

            if (NULL == (path = (char *)H5MM_calloc(len)))
                HGOTO_ERROR(H5E_PLUGIN, H5E_CANTALLOC, H5_ITER_ERROR, "can't allocate memory for path");

            snprintf(path, len, "%s/%s", plugin_path, dp->d_name);

            
            memset(&my_stat, 0, sizeof(h5_stat_t));
            if (HDstat(path, &my_stat) == -1)
                HGOTO_ERROR(H5E_FILE, H5E_CANTGET, H5_ITER_ERROR, "can't stat file %s -- error was: %s", path,
                            strerror(errno));

            
            if (S_ISDIR(my_stat.st_mode))
                continue;

            
            plugin_type   = H5PL_TYPE_ERROR;
            plugin_info   = NULL;
            plugin_loaded = false;
            if (H5PL__open(path, H5PL_TYPE_NONE, NULL, &plugin_loaded, &plugin_type, &plugin_info) < 0)
                HGOTO_ERROR(H5E_PLUGIN, H5E_CANTGET, H5_ITER_ERROR, "failed to open plugin '%s'", path);

            
            plugin_matches = (iter_type == H5PL_ITER_TYPE_ALL) ||
                             ((iter_type == H5PL_ITER_TYPE_FILTER) && (plugin_type == H5PL_TYPE_FILTER)) ||
                             ((iter_type == H5PL_ITER_TYPE_VOL) && (plugin_type == H5PL_TYPE_VOL)) ||
                             ((iter_type == H5PL_ITER_TYPE_VFD) && (plugin_type == H5PL_TYPE_VFD));

            
            if (plugin_loaded && plugin_matches && (ret_value = iter_op(plugin_type, plugin_info, op_data)))
                break;

            path = (char *)H5MM_xfree(path);
        } 
    }     

    if (ret_value < 0)
        HERROR(H5E_PLUGIN, H5E_CALLBACK, "callback operator function returned failure");

done:
    if (dirp)
        if (HDclosedir(dirp) < 0)
            HDONE_ERROR(H5E_FILE, H5E_CLOSEERROR, H5_ITER_ERROR, "can't close directory: %s",
                        strerror(errno));

    path = (char *)H5MM_xfree(path);

    FUNC_LEAVE_NOAPI(ret_value)
} 
#else  
static herr_t
H5PL__path_table_iterate_process_path(const char *plugin_path, H5PL_iterate_type_t iter_type,
                                      H5PL_iterate_t iter_op, void *op_data)
{
    WIN32_FIND_DATAA fdFile;
    HANDLE           hFind = INVALID_HANDLE_VALUE;
    H5PL_type_t      plugin_type;
    const void      *plugin_info = NULL;
    bool             plugin_loaded;
    char            *path = NULL;
    char             service[2048];
    herr_t           ret_value = H5_ITER_CONT;

    FUNC_ENTER_PACKAGE

    
    assert(plugin_path);
    assert(iter_op);

    
    snprintf(service, sizeof(service), "%s\\*.dll", plugin_path);
    if ((hFind = FindFirstFileA(service, &fdFile)) == INVALID_HANDLE_VALUE)
        HGOTO_DONE(H5_ITER_CONT);

    
    do {
        
        if (strcmp(fdFile.cFileName, ".") != 0 && strcmp(fdFile.cFileName, "..") != 0) {
            bool   plugin_matches;
            size_t len;

            
            len = strlen(plugin_path) + strlen(H5PL_PATH_SEPARATOR) + strlen(fdFile.cFileName) + 1;

            if (NULL == (path = (char *)H5MM_calloc(len)))
                HGOTO_ERROR(H5E_PLUGIN, H5E_CANTALLOC, H5_ITER_ERROR, "can't allocate memory for path");

            snprintf(path, len, "%s\\%s", plugin_path, fdFile.cFileName);

            
            if (fdFile.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
                continue;

            
            plugin_type   = H5PL_TYPE_ERROR;
            plugin_info   = NULL;
            plugin_loaded = false;
            if (H5PL__open(path, H5PL_TYPE_NONE, NULL, &plugin_loaded, &plugin_type, &plugin_info) < 0)
                HGOTO_ERROR(H5E_PLUGIN, H5E_CANTGET, H5_ITER_ERROR, "failed to open plugin '%s'", path);

            
            plugin_matches = (iter_type == H5PL_ITER_TYPE_ALL) ||
                             ((iter_type == H5PL_ITER_TYPE_FILTER) && (plugin_type == H5PL_TYPE_FILTER)) ||
                             ((iter_type == H5PL_ITER_TYPE_VOL) && (plugin_type == H5PL_TYPE_VOL)) ||
                             ((iter_type == H5PL_ITER_TYPE_VFD) && (plugin_type == H5PL_TYPE_VFD));

            
            if (plugin_loaded && plugin_matches && (ret_value = iter_op(plugin_type, plugin_info, op_data)))
                break;

            path = (char *)H5MM_xfree(path);
        }
    } while (FindNextFileA(hFind, &fdFile));

    if (ret_value < 0)
        HERROR(H5E_PLUGIN, H5E_CALLBACK, "callback operator function returned failure");

done:
    if (hFind != INVALID_HANDLE_VALUE)
        FindClose(hFind);

    path = (char *)H5MM_xfree(path);

    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

herr_t
H5PL__find_plugin_in_path_table(const H5PL_search_params_t *search_params, bool *found,
                                const void **plugin_info)
{
    unsigned int u; 
    herr_t       ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(search_params);
    assert(found);
    assert(plugin_info);

    
    *found       = false;
    *plugin_info = NULL;

    
    for (u = 0; u < H5PL_num_paths_g; u++) {

        
        if (H5PL__find_plugin_in_path(search_params, found, H5PL_paths_g[u], plugin_info) < 0)
            HERROR(H5E_PLUGIN, H5E_CANTGET, "search in path %s encountered an error", H5PL_paths_g[u]);

        
        if (*found) {
            if (!plugin_info)
                HGOTO_ERROR(H5E_PLUGIN, H5E_BADVALUE, FAIL, "plugin info should not be NULL");
            break;
        }
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

#ifndef H5_HAVE_WIN32_API
static herr_t
H5PL__find_plugin_in_path(const H5PL_search_params_t *search_params, bool *found, const char *dir,
                          const void **plugin_info)
{
    char          *path      = NULL;
    DIR           *dirp      = NULL; 
    struct dirent *dp        = NULL; 
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(search_params);
    assert(found);
    assert(dir);
    assert(plugin_info);

    
    *found = false;

    
    if (!(dirp = HDopendir(dir)))
        HGOTO_ERROR(H5E_PLUGIN, H5E_OPENERROR, FAIL, "can't open directory (%s). Please verify its existence",
                    dir);

    
    while (NULL != (dp = HDreaddir(dirp))) {

        
#ifndef __CYGWIN__
        if (!strncmp(dp->d_name, "lib", (size_t)3) &&
            (strstr(dp->d_name, ".so") || strstr(dp->d_name, ".dylib"))) {
#else
        if (!strncmp(dp->d_name, "cyg", (size_t)3) && strstr(dp->d_name, ".dll")) {
#endif

            h5_stat_t my_stat;
            size_t    len;

            
            len = strlen(dir) + strlen(H5PL_PATH_SEPARATOR) + strlen(dp->d_name) + 1  +
                  4; 

            if (NULL == (path = (char *)H5MM_calloc(len)))
                HGOTO_ERROR(H5E_PLUGIN, H5E_CANTALLOC, FAIL, "can't allocate memory for path");

            snprintf(path, len, "%s/%s", dir, dp->d_name);

            
            memset(&my_stat, 0, sizeof(h5_stat_t));
            if (HDstat(path, &my_stat) == -1)
                HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "can't stat file %s -- error was: %s", path,
                            strerror(errno));

            
            if (S_ISDIR(my_stat.st_mode)) {
                path = (char *)H5MM_xfree(path);
                continue;
            }

            
            if (H5PL__open(path, search_params->type, search_params->key, found, NULL, plugin_info) < 0)
                HGOTO_ERROR(H5E_PLUGIN, H5E_CANTGET, FAIL, "search in directory failed");
            if (*found)
                HGOTO_DONE(SUCCEED);

            path = (char *)H5MM_xfree(path);
        } 
    }     

done:
    if (dirp)
        if (HDclosedir(dirp) < 0)
            HDONE_ERROR(H5E_FILE, H5E_CLOSEERROR, FAIL, "can't close directory: %s", strerror(errno));

    path = (char *)H5MM_xfree(path);

    FUNC_LEAVE_NOAPI(ret_value)
} 
#else  
static herr_t
H5PL__find_plugin_in_path(const H5PL_search_params_t *search_params, bool *found, const char *dir,
                          const void **plugin_info)
{
    WIN32_FIND_DATAA fdFile;
    HANDLE           hFind = INVALID_HANDLE_VALUE;
    char            *path  = NULL;
    char             service[2048];
    herr_t           ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(search_params);
    assert(found);
    assert(dir);
    assert(plugin_info);

    
    *found = false;

    
    snprintf(service, sizeof(service), "%s\\*.dll", dir);
    if ((hFind = FindFirstFileA(service, &fdFile)) == INVALID_HANDLE_VALUE)
        HGOTO_ERROR(H5E_PLUGIN, H5E_OPENERROR, FAIL, "can't open directory");

    
    do {
        
        if (strcmp(fdFile.cFileName, ".") != 0 && strcmp(fdFile.cFileName, "..") != 0) {

            

            size_t len;

            
            len = strlen(dir) + strlen(H5PL_PATH_SEPARATOR) + strlen(fdFile.cFileName) + 1;

            if (NULL == (path = (char *)H5MM_calloc(len)))
                HGOTO_ERROR(H5E_PLUGIN, H5E_CANTALLOC, FAIL, "can't allocate memory for path");

            snprintf(path, len, "%s\\%s", dir, fdFile.cFileName);

            
            if (fdFile.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
                continue;

            
            if (H5PL__open(path, search_params->type, search_params->key, found, NULL, plugin_info) < 0)
                HGOTO_ERROR(H5E_PLUGIN, H5E_CANTGET, FAIL, "search in directory failed");
            if (*found)
                HGOTO_DONE(SUCCEED);

            path = (char *)H5MM_xfree(path);
        }
    } while (FindNextFileA(hFind, &fdFile));

done:
    if (hFind != INVALID_HANDLE_VALUE)
        FindClose(hFind);
    if (path)
        path = (char *)H5MM_xfree(path);

    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 
