% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/multilevel.cfa.R
\name{multilevel.cfa}
\alias{multilevel.cfa}
\title{Multilevel Confirmatory Factor Analysis}
\usage{
multilevel.cfa(data, ..., cluster, model = NULL, rescov = NULL,
               model.w = NULL, model.b = NULL, rescov.w = NULL, rescov.b = NULL,
               const = c("within", "shared", "config", "shareconf"),
               fix.resid = NULL, ident = c("marker", "var", "effect"),
               ls.fit = FALSE, estimator = c("ML", "MLR"),
               optim.method = c("nlminb", "em"), missing = c("listwise", "fiml"),
               print = c("all", "summary", "coverage", "descript", "fit", "est",
                         "modind", "resid"),
               mod.minval = 6.63, resid.minval = 0.1, digits = 3, p.digits = 3,
               as.na = NULL, write = NULL, append = TRUE, check = TRUE, output = TRUE)
}
\arguments{
\item{data}{a data frame. If \code{model}, \code{model.w}, and \code{model.b}
are \code{NULL}, multilevel confirmatory factor analysis
based on a measurement model with one factor labeled \code{wf}
at the Within level and one factor labeled \code{bf} at the
Between level comprising all variables in the data frame
is conducted. Note that the cluster variable specified in
\code{cluster} is excluded from \code{data} when specifying
the argument \code{cluster} using the variable name of the
cluster variable. If \code{model} or \code{mode.w}
and \code{model.b} is specified, the data frame needs to
contain all variables used in the \code{model} argument(s).}

\item{...}{an expression indicating the variable names in \code{data}.
Note that the operators \code{+}, \code{-},
\code{~}, \code{:}, \code{::}, and \code{!} can also be
used to select variables, see 'Details' in the
\code{\link{df.subset}} function.}

\item{cluster}{either a character string indicating the variable name of
the cluster variable in \code{data} or \code{data}, or a
vector representing the nested grouping structure (i.e.,
group or cluster variable).}

\item{model}{a character vector for specifying the same factor structure
with one factor at the Within and Between Level, or a list
of character vectors for specifying the same measurement
model with more than one factor at the Within and Between
Level, e.g.,\code{model = c("x1", "x2", "x3", "x4")} for
specifying a measurement model with one factor labeled \code{wf}
at the Within level and a measurement model with one factor
labeled \code{bf} at the Between level each comprising four
indicators, or \code{model = list(factor1 = c("x1", "x2", "x3", "x4"),
factor2 = c("x5", "x6", "x7", "x8"))} for specifying a
measurement model with two latent factors labeled \code{wfactor1}
and \code{wfactor2} at the Within level and a measurement
model with two latent factors labeled \code{bfactor1} and
\code{bfactor2} at the Between level each comprising four
indicators. Note that the name of each list element is used
to label factors, where prefixes \code{w} and \code{b} are
added the labels to distinguish factor labels at the Within
and Between level, i.e., all list elements need to be named,
otherwise factors are labeled with \code{"wf1", "wf2", "wf3"}
for labels at the Within level and \code{"bf1", "bf2", "bf3"}
for labels at the Between level and so on.}

\item{rescov}{a character vector or a list of character vectors for specifying
residual covariances at the Within level, e.g. \code{rescov = c("x1", "x2")}
for specifying a residual covariance between indicators \code{x1}
and \code{x2} at the Within level or \code{rescov = list(c("x1", "x2"), c("x3", "x4"))}
for specifying residual covariances between indicators \code{x1}
and \code{x2}, and indicators \code{x3} and \code{x4} at
the Within level. Note that residual covariances at the
Between level can only be specified by using the arguments
\code{model.w}, \code{model.b}, and \code{model.b}.}

\item{model.w}{a character vector specifying a measurement model with one
factor at the Within level, or a list of character vectors
for specifying a measurement model with more than one factor
at the Within level.}

\item{model.b}{a character vector specifying a measurement model with one
factor at the Between level, or a list of character vectors
for specifying a measurement model with more than one factor
at the Between level.}

\item{rescov.w}{a character vector or a list of character vectors for
specifying residual covariances at the Within level.}

\item{rescov.b}{a character vector or a list of character vectors for
specifying residual covariances at the Between level.}

\item{const}{a character string indicating the type of construct(s), i.e.,
\code{"within"} for within-cluster constructs, \code{"shared"}
for shared cluster-level constructs, \code{"config"} (default)
for configural cluster constructs, and \code{"shareconf"}
for simultaneous shared and configural cluster constructs.}

\item{fix.resid}{a character vector for specifying residual variances to be
fixed at 0 at the Between level, e.g., \code{fix.resid = c("x1", "x3")}
to fix residual variances of indicators \code{x1} and \code{x2}
at the Between level at 0. Note that it is also possible
to specify \code{fix.resid = "all"} which fixes all residual
variances at the Between level at 0 in line with the strong
factorial measurement invariance assumption across cluster.}

\item{ident}{a character string indicating the method used for identifying
and scaling latent variables, i.e., \code{"marker"} for the
marker variable method fixing the first factor loading of
each latent variable to 1, \code{"var"} for the fixed variance
method fixing the variance of each latent variable to 1,
or \code{"effect"} for the effects-coding method using equality
constraints so that the average of the factor loading for
each latent variable equals 1.}

\item{ls.fit}{logical: if \code{TRUE} (default) level-specific fit indices
are computed when specifying a model using the arguments
\code{model.w} and \code{model.b} given the model does not
contain any cross-level equality constraints.}

\item{estimator}{a character string indicating the estimator to be used:
\code{"ML"} for maximum likelihood with conventional standard
errors and \code{"MLR"} (default) for maximum likelihood
with Huber-White robust standard errors and a scaled test
statistic that is asymptotically equal to the Yuan-Bentler
test statistic. Note that by default, full information maximum
likelihood (FIML) method is used to deal with missing data
when using \code{"ML"} (\code{missing = "fiml"}), whereas
incomplete cases are removed listwise (i.e., \code{missing = "listwise"})
when using \code{"MLR"}.}

\item{optim.method}{a character string indicating the optimizer, i.e., \code{"nlminb"}
(default) for the unconstrained and bounds-constrained
quasi-Newton method optimizer and \code{"em"} for the
Expectation Maximization (EM) algorithm.}

\item{missing}{a character string indicating how to deal with missing data,
i.e., \code{"listwise"} (default) for listwise deletion or
\code{"fiml"} for full information maximum likelihood (FIML)
method. Note that FIML method is only available when \code{estimator = "ML"},
that it takes longer to estimate the model  using FIML, and
that FIML is prone to convergence issues which might be
resolved by switching to listwise deletion.}

\item{print}{a character string or character vector indicating which
results to show on the console, i.e. \code{"all"} for all
results, \code{"summary"} for a summary of the specification
of the estimation method and missing data handling in lavaan,
\code{"coverage"} for the variance-covariance coverage of
the data, \code{"descript"} for descriptive statistics,
\code{"fit"} for model fit,  \code{"est"} for parameter
estimates, and \code{"modind"} for modification indices.
By default, a summary of the specification, descriptive
statistics, model fit, and parameter estimates are printed.}

\item{mod.minval}{numeric value to filter modification indices and only
show modifications with a modification index value equal
or higher than this minimum value. By default, modification
indices equal or higher 6.63 are printed. Note that a
modification index value of 6.63 is equivalent to a
significance level of \eqn{\alpha = .01}.}

\item{resid.minval}{numeric value indicating the minimum absolute residual
correlation coefficients and standardized means to
highlight in boldface. By default, absolute residual
correlation coefficients and standardized means equal
or higher 0.1 are highlighted. Note that highlighting
can be disabled by setting the minimum value to 1.}

\item{digits}{an integer value indicating the number of decimal places
to be used for displaying results. Note that loglikelihood,
information criteria and chi-square test statistic is
printed with \code{digits} minus 1 decimal places.}

\item{p.digits}{an integer value indicating the number of decimal places
to be used for displaying the \emph{p}-value.}

\item{as.na}{a numeric vector indicating user-defined missing values,
i.e. these values are converted to \code{NA} before conducting
the analysis. Note that \code{as.na()} function is only
applied to \code{data} but not to \code{cluster}.}

\item{write}{a character string naming a file for writing the output into
either a text file with file extension \code{".txt"} (e.g.,
\code{"Output.txt"}) or Excel file with file extension
\code{".xlsx"}  (e.g., \code{"Output.xlsx"}). If the file
name does not contain any file extension, an Excel file will
be written.}

\item{append}{logical: if \code{TRUE} (default), output will be appended
to an existing text file with extension \code{.txt} specified
in \code{write}, if \code{FALSE} existing text file will be
overwritten.}

\item{check}{logical: if \code{TRUE} (default), argument specification, convergence
and model identification is checked.}

\item{output}{logical: if \code{TRUE} (default), output is shown.}
}
\value{
Returns an object of class \code{misty.object}, which is a list with following
entries:
\item{\code{call}}{function call}
\item{\code{type}}{type of analysis}
\item{\code{data}}{data frame used for the current analysis}
\item{\code{args}}{specification of function arguments}
\item{\code{model}}{specified model}
\item{\code{model.fit}}{fitted lavaan object (\code{mod.fit})}
\item{\code{check}}{results of the convergence and model identification check}
\item{\code{result}}{list with result tables, i.e., \code{summary} for the
                     summary of the specification of the estimation method
                     and missing data handling in lavaan, \code{coverage} for
                     the variance-covariance coverage of the data, \code{descript}
                     for descriptive statistics, \code{fit} for model fit,
                     \code{est} for parameter estimates, and \code{modind}
                     for modification indices.}
}
\description{
This function conducts multilevel confirmatory factor analysis to investigate
four types of constructs, i.e., within-cluster constructs, shared cluster-level
constructs, configural cluster constructs, and simultaneous shared and configural
cluster constructs by calling the \code{cfa} function in the R package \pkg{lavaan}.
By default, the function specifies and estimates a configural cluster construct
and provides a table with univariate sample statistics, model fit information,
and parameter estimates. Additionally, variance-covariance coverage of the data,
modification indices, and residual correlation matrix can be requested by specifying
the argument \code{print}.
}
\note{
The function uses the functions \code{cfa}, \code{lavInspect}, \code{lavTech},
\code{modindices}, \code{parameterEstimates}, and \code{standardizedsolution}
provided in the R package \pkg{lavaan} by Yves Rosseel (2012).
}
\examples{
\dontrun{

# Load data set "Demo.twolevel" in the lavaan package
data("Demo.twolevel", package = "lavaan")

#----------------------------------------------------------------------------
# Model specification using 'data' for a one-factor model
# with the same factor structure with one factor at the Within and Between Level

#..........
# Cluster variable specification

# Example 1a: Specification using the argument '...'
multilevel.cfa(Demo.twolevel, y1:y4, cluster = "cluster")

# Example 1b: Alternative specification with cluster variable 'cluster' in 'data'
multilevel.cfa(Demo.twolevel[, c("y1", "y2", "y3", "y4", "cluster")], cluster = "cluster")

# Example 1c: Alternative specification with cluster variable 'cluster' not in 'data'
multilevel.cfa(Demo.twolevel[, c("y1", "y2", "y3", "y4")], cluster = Demo.twolevel$cluster)

#..........
# Type of construct

# Example 2a: Within-cluster constructs
multilevel.cfa(Demo.twolevel, y1:y4, cluster = "cluster", const = "within")

# Example 2b: Shared cluster-level construct
multilevel.cfa(Demo.twolevel, y1:y4, cluster = "cluster", const = "shared")

# Example 2c: Configural cluster construct (default)
multilevel.cfa(Demo.twolevel, y1:y4, cluster = "cluster", const = "config")

# Example 2d: Simultaneous shared and configural cluster construct
multilevel.cfa(Demo.twolevel, y1:y4, cluster = "cluster", const = "shareconf")

#..........
# Residual covariances at the Within level

# Example 3a: Residual covariance between 'y1' and 'y3'
multilevel.cfa(Demo.twolevel, y1:y4, cluster = "cluster", rescov = c("y1", "y3"))

# Example 3b: Residual covariance between 'y1' and 'y3', and 'y2' and 'y4'
multilevel.cfa(Demo.twolevel, y1:y4, cluster = "cluster",
               rescov = list(c("y1", "y3"), c("y2", "y4")))

#..........
# Residual variances at the Between level fixed at 0

# Example 4a: All residual variances fixed at 0
# i.e., strong factorial invariance across clusters
multilevel.cfa(Demo.twolevel, y1:y4, cluster = "cluster", fix.resid = "all")

# Example 4b: Fesidual variances of 'y1', 'y2', and 'y4' fixed at 0
# i.e., partial strong factorial invariance across clusters
multilevel.cfa(Demo.twolevel, y1:y4, cluster = "cluster", fix.resid = c("y1", "y2", "y4"))

#..........
# Print all results

# Example 5: Set minimum value for modification indices to 1
multilevel.cfa(Demo.twolevel, y1:y4, cluster = "cluster", print = "all",
               mod.minval = 1)

#..........
# Example 6: lavaan model and summary of the estimated model

mod <- multilevel.cfa(Demo.twolevel, y1:y4, cluster = "cluster", output = FALSE)

# lavaan model syntax
cat(mod$model)

# Fitted lavaan object
lavaan::summary(mod$model.fit, standardized = TRUE, fit.measures = TRUE)

#..........
# Write results

# Example 7a: Assign results into an object and write results into an Excel file
mod <- multilevel.cfa(Demo.twolevel, y1:y4, cluster = "cluster", print = "all",
                      write = "Multilevel_CFA.txt", output = FALSE)

# Example 7b: Assign results into an object and write results into an Excel file
mod <- multilevel.cfa(Demo.twolevel, y1:y4, cluster = "cluster", print = "all",
                      output = FALSE)

# Write results into an Excel file
write.result(mod, "Multilevel_CFA.xlsx")

# Estimate model and write results into an Excel file
multilevel.cfa(Demo.twolevel, y1:y4, cluster = "cluster", print = "all",
               write = "Multilevel_CFA.xlsx")

#----------------------------------------------------------------------------
# Model specification using 'model' for one or multiple factor model
# with the same factor structure at the Within and Between Level

# Example 8a: One-factor model
multilevel.cfa(Demo.twolevel, cluster = "cluster", model = c("y1", "y2", "y3", "y4"))

# Example 8b: Two-factor model
multilevel.cfa(Demo.twolevel, cluster = "cluster",
               model = list(c("y1", "y2", "y3"), c("y4", "y5", "y6")))

# Example 8c: Two-factor model with user-specified labels for the factors
multilevel.cfa(Demo.twolevel, cluster = "cluster",
               model = list(factor1 = c("y1", "y2", "y3"), factor2 = c("y4", "y5", "y6")))

#..........
# Type of construct

# Example 9a: Within-cluster constructs
multilevel.cfa(Demo.twolevel, cluster = "cluster", const = "within",
               model = list(c("y1", "y2", "y3"), c("y4", "y5", "y6")))

# Example 9b: Shared cluster-level construct
multilevel.cfa(Demo.twolevel, cluster = "cluster", const = "shared",
               model = list(c("y1", "y2", "y3"), c("y4", "y5", "y6")))

# Example 9c: Configural cluster construct (default)
multilevel.cfa(Demo.twolevel, cluster = "cluster", const = "config",
               model = list(c("y1", "y2", "y3"), c("y4", "y5", "y6")))

# Example 9d: Simultaneous shared and configural cluster construct
multilevel.cfa(Demo.twolevel, cluster = "cluster", const = "shareconf",
               model = list(c("y1", "y2", "y3"), c("y4", "y5", "y6")))

#..........
# Residual covariances at the Within level

# Example 10a: Residual covariance between 'y1' and 'y4' at the Within level
multilevel.cfa(Demo.twolevel, cluster = "cluster",
               model = list(c("y1", "y2", "y3"), c("y4", "y5", "y6")),
               rescov = c("y1", "y4"))

# Example 10b: Fix all residual variances at 0
# i.e., strong factorial invariance across clusters
multilevel.cfa(Demo.twolevel, cluster = "cluster",
               model = list(c("y1", "y2", "y3"), c("y4", "y5", "y6")),
               fix.resid = "all")

#----------------------------------------------------------------------------
# Model specification using 'model.w' and 'model.b' for one or multiple factor model
# with different factor structure at the Within and Between Level

# Example 11a: Two-factor model at the Within level and one-factor model at the Between level
multilevel.cfa(Demo.twolevel, cluster = "cluster",
               model.w = list(c("y1", "y2", "y3"), c("y4", "y5", "y6")),
               model.b = c("y1", "y2", "y3", "y4", "y5", "y6"))

# Example 11b: Residual covariance between 'y1' and 'y4' at the Within level
# Residual covariance between 'y5' and 'y6' at the Between level
multilevel.cfa(Demo.twolevel, cluster = "cluster",
               model.w = list(c("y1", "y2", "y3"), c("y4", "y5", "y6")),
               model.b = c("y1", "y2", "y3", "y4", "y5", "y6"),
               rescov.w = c("y1", "y4"),
               rescov.b = c("y5", "y6"))
}
}
\references{
Rosseel, Y. (2012). lavaan: An R Package for Structural Equation Modeling.
\emph{Journal of Statistical Software, 48}, 1-36. https://doi.org/10.18637/jss.v048.i02
}
\seealso{
\code{\link{item.cfa}}, \code{\link{multilevel.fit}}, \code{\link{multilevel.invar}},
\code{\link{multilevel.omega}}, \code{\link{multilevel.cor}}, \code{\link{multilevel.descript}}
}
\author{
Takuya Yanagida \email{takuya.yanagida@univie.ac.at}
}
