# PoissonInvNHE.R
#' Poisson Inverse NHE Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Poisson Inverse NHE distribution.
#'
#' The Poisson Inverse NHE distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Poisson Inverse NHE distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \quad \frac{1-\exp \left[-\lambda \exp \left\{1-(1+\alpha / x)
#' ^\beta\right\}\right]}{1-\exp (-\lambda)} \quad ;\;x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dpois.inv.NHE()} — Density function
#'   \item \code{ppois.inv.NHE()} — Distribution function
#'   \item \code{qpois.inv.NHE()} — Quantile function
#'   \item \code{rpois.inv.NHE()} — Random generation
#'   \item \code{hpois.inv.NHE()} — Hazard function
#' }
#'
#' @usage
#' dpois.inv.NHE(x, alpha, beta, lambda, log = FALSE)
#' ppois.inv.NHE(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qpois.inv.NHE(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rpois.inv.NHE(n, alpha, beta, lambda)
#' hpois.inv.NHE(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dpois.inv.NHE}: numeric vector of (log-)densities
#'   \item \code{ppois.inv.NHE}: numeric vector of probabilities
#'   \item \code{qpois.inv.NHE}: numeric vector of quantiles
#'   \item \code{rpois.inv.NHE}: numeric vector of random variates
#'   \item \code{hpois.inv.NHE}: numeric vector of hazard values
#' }
#'
#' @references
#' Chaudhary,A.K.& Kumar, V.(2020). 
#' Poisson Inverse NHE Distribution. 
#' \emph{International Journal of Science and Research(IJSR)}, 
#' \bold{9(12)}, 1603--1610.   
#'
#' @examples
#' x <- seq(0.1, 10, 0.2)
#' dpois.inv.NHE(x, 2.0, 0.5, 0.2)
#' ppois.inv.NHE(x, 2.0, 0.5, 0.2)
#' qpois.inv.NHE(0.5, 2.0, 0.5, 0.2)
#' rpois.inv.NHE(10, 2.0, 0.5, 0.2)
#' hpois.inv.NHE(x, 2.0, 0.5, 0.2)
#' 
#' # Data
#' x <- fibers63
#' # ML estimates
#' params = list(alpha=1.0174, beta=5.1414, lambda=23.3476)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = ppois.inv.NHE, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qpois.inv.NHE, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dpois.inv.NHE, pfun=ppois.inv.NHE, plot=FALSE)
#' print.gofic(out)
#' 
#' @name PoissonInvNHE
#' @aliases dpois.inv.NHE ppois.inv.NHE qpois.inv.NHE rpois.inv.NHE hpois.inv.NHE
#' @export
NULL

#' @rdname PoissonInvNHE
#' @usage NULL
#' @export  
dpois.inv.NHE <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
	tmp <- 1/(1 - exp(-lambda))
	u <- 1+(alpha/x[valid])
	z <- exp(1-(u^beta))
    gx <- alpha * beta * lambda *(x[valid]^-2)*(u ^(beta-1)) * z * exp(-lambda*z)	
    pdf[valid] <- tmp * gx
	}
    if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname PoissonInvNHE
#' @usage NULL
#' @export 
ppois.inv.NHE <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {	
	tmp <- 1/(1 - exp(-lambda))
	u <- 1 + (alpha/q[valid])
	z <- exp(1-(u^beta))
    cdf[valid] <- tmp * (1-exp(-lambda * z))
	}
    if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname PoissonInvNHE
#' @usage NULL
#' @export
qpois.inv.NHE <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta' and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
	z <- -(1/lambda) * log(1-(1-exp(-lambda))*p[valid])		
    qtl[valid] <- alpha / ((1-log(z)) ^ (1/beta) -1)
	}
    return(qtl)   
 } 

#' @rdname PoissonInvNHE
#' @usage NULL
#' @export
rpois.inv.NHE <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qpois.inv.NHE(u, alpha, beta, lambda)
    return(q)
}

#' @rdname PoissonInvNHE
#' @usage NULL
#' @export
hpois.inv.NHE <- function (x, alpha, beta, lambda)
{
 	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
    
    nume <- dpois.inv.NHE(x, alpha, beta, lambda)
	surv <- 1.0 - ppois.inv.NHE(x, alpha, beta, lambda)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
} 
