#' Get an overview of the given root scans
#'
#' \code{combineStatsWithDL} - This function goes through a set of image
#' directories and combines all existing statistics_withDepthLayers.csv-files
#' into one data frame, that is optionally saved as a csv.
#' @param data_dir (Optional, default = NULL) String specifying the name
#' (full path) of the directory containing all image directories of
#' interest.
#' @param data_dirs (Optional, default = NULL) Character vector specifying all
#' of the individual root scan directories of interest. This is only used if
#' \code{data_dir} is set to NULL.
#' @param out_file (Optional, default = NA) Full path of the output-csv-file.
#' If set to NULL, no output file is created. If its is set to NA and
#' \code{data_dir} is not NULL, then the output file
#' all_statistics_withDepthLayers.csv is created in \code{data_dir}.
#'
#' @return \code{combineStatsWithDL} A data frame containing the combined
#' depth layer information of the images.
#'
#' @export
#' @rdname combineStatsWithDL
#'
#' @examples
#' combineStatsWithDL(data_dir = NULL, out_file = NULL)
combineStatsWithDL <- function(data_dir, data_dirs = NULL,
                               out_file = NA){
  # Check the given data directory.
  if(!is.null(data_dir)){
    # List all subdirectories in the directory.
    dirs <- list.dirs(data_dir, full.names = TRUE, recursive = FALSE)
  } else if(!is.null(data_dirs)){
    dirs <- data_dirs
  } else {
    message("No directories specified. Returning NULL.\n")
    return(NULL)
  }
  dirs <- standardizePaths(dirs)
  # Check whether the directories contain the data file and combine them. ------
  combined_df <- data.frame()
  have_csv <- rep(FALSE, length(dirs))
  for(i in 1:length(dirs)){
    if(file.exists(paste0(dirs[i], "/statistics_withDepthLayers.csv"))){
      have_csv[i] <- TRUE
      temp_df <- utils::read.csv(paste0(dirs[i],
                                        "/statistics_withDepthLayers.csv"))
      combined_df <- rbind(combined_df, temp_df)
    }
  }
  message(paste(sum(have_csv),"of the",length(dirs),
                " folders contained a statistics_withDepthLayers.csv-file."))
  # The minus signs get changed into "." sometimes. Reverse this: --------------
  colnames(combined_df) <- gsub(".", "-", colnames(combined_df), fixed = TRUE)

  # Add new columns containing the sums over all depth layers
  combined_df$pixels_root_total <- rowSums(combined_df[,c(20,26,32,38,44,50,
                                                          56,62)])
  combined_df$skel_pixels_total <- rowSums(combined_df[,c(21,27,33,39,45,51,
                                                          57,63)])
  combined_df$skel_pixels_low3_total <- rowSums(combined_df[,c(22,28,34,40,46,
                                                               52,58,64)])
  combined_df$`skel_pixels_3-7_total` <- rowSums(combined_df[,c(23,29,35,41,47,
                                                                53,59,65)])
  combined_df$skel_pixels_larg7_total <- rowSums(combined_df[,c(24,30,36,42,48,
                                                                54,60,66)])
  combined_df$kimura_length6_total <- rowSums(combined_df[,c(25,31,37,43,49,55,
                                                             61,67)])
  # Save and return the data frame. --------------------------------------------
  if(sum(have_csv) > 0){
    if(!is.null(out_file)){
      if(is.na(out_file)){
        out_file <- paste0(data_dir,"//all_statistics_withDepthLayers.csv")
      }
      utils::write.csv(combined_df, file = out_file)
    }
  }
  return(combined_df)
}

