# Test utility functions
# These tests verify ACAT_combine, binarize_expression, and other utilities

test_that("ACAT_combine returns valid p-value", {
    pvals <- c(0.01, 0.05, 0.3, 0.8)
    combined <- ACAT_combine(pvals)

    expect_true(combined >= 0 && combined <= 1)
})

test_that("ACAT_combine handles edge cases", {
    # All significant
    expect_lt(ACAT_combine(c(0.001, 0.002, 0.003)), 0.01)

    # All non-significant
    expect_gt(ACAT_combine(c(0.5, 0.6, 0.7)), 0.1)

    # Single p-value
    expect_equal(ACAT_combine(0.05), 0.05, tolerance = 0.01)
})

test_that("ACAT_combine rejects invalid inputs", {
    expect_error(ACAT_combine(c(0.1, NA, 0.3)))
    expect_error(ACAT_combine(c(-0.1, 0.5)))
    expect_error(ACAT_combine(c(0.5, 1.5)))
})

test_that("ACAT_combine handles p = 0", {
    result <- ACAT_combine(c(0, 0.1, 0.2))
    expect_equal(result, 0)
})

test_that("binarize_expression returns correct dimensions", {
    set.seed(42)
    expr <- matrix(rpois(100, lambda = 10), nrow = 10, ncol = 10)
    rownames(expr) <- paste0("gene_", 1:10)
    colnames(expr) <- paste0("spot_", 1:10)

    bin <- binarize_expression(expr, method = "kmeans")

    expect_equal(dim(bin), dim(expr))
    expect_equal(rownames(bin), rownames(expr))
    expect_equal(colnames(bin), colnames(expr))
})

test_that("binarize_expression returns only 0 and 1", {
    set.seed(42)
    expr <- matrix(rpois(100, lambda = 10), nrow = 10, ncol = 10)

    for (method in c("kmeans", "rank", "median", "mean")) {
        bin <- binarize_expression(expr, method = method)
        expect_true(all(bin %in% c(0, 1)))
    }
})

test_that("binarize_expression rank method respects percentage", {
    set.seed(42)
    n_spots <- 100
    expr <- matrix(rpois(10 * n_spots, lambda = 10), nrow = 10, ncol = n_spots)

    rank_percent <- 30
    bin <- binarize_expression(expr, method = "rank", rank_percent = rank_percent)

    # Average percentage should be close to rank_percent
    avg_pct_high <- mean(rowSums(bin)) / n_spots * 100
    expect_true(abs(avg_pct_high - rank_percent) < 10)  # Within 10% tolerance
})

test_that("binarize_expression handles constant expression", {
    expr <- matrix(5, nrow = 3, ncol = 10)

    # Should not error, should return all zeros or all ones
    bin <- binarize_expression(expr, method = "kmeans")
    expect_true(all(bin %in% c(0, 1)))
})

test_that("row_standardize normalizes rows correctly", {
    W <- matrix(c(0, 1, 2, 1, 0, 3, 2, 3, 0), nrow = 3, byrow = TRUE)

    W_std <- row_standardize(W)

    # Each non-zero row should sum to 1
    row_sums <- rowSums(W_std)
    for (i in seq_len(nrow(W))) {
        if (sum(W[i, ]) > 0) {
            expect_equal(row_sums[i], 1, tolerance = 1e-10)
        }
    }
})

test_that("row_standardize handles zero rows", {
    # Matrix with first row all zeros
    W <- matrix(c(0, 0, 0, 1, 0, 3, 2, 3, 0), nrow = 3, byrow = TRUE)

    # Should not error
    W_std <- row_standardize(W)
    
    # Zero row stays zero (divided by 1, which means 0/1 = 0)
    expect_equal(W_std[1, ], c(0, 0, 0))
    
    # Non-zero rows sum to 1
    expect_equal(sum(W_std[2, ]), 1, tolerance = 1e-10)
    expect_equal(sum(W_std[3, ]), 1, tolerance = 1e-10)
})
