% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dlmtree.R
\name{dlmtree}
\alias{dlmtree}
\title{Fit tree structured distributed lag models}
\usage{
dlmtree(
  formula,
  data,
  exposure.data,
  dlm.type = "linear",
  family = "gaussian",
  mixture = FALSE,
  het = FALSE,
  control.mcmc = list(),
  control.hyper = list(),
  control.family = list(),
  control.tdlnm = list(),
  control.het = list(),
  control.mix = list(),
  control.monotone = list(),
  control.diagnose = list()
)
}
\arguments{
\item{formula}{object of class formula, a symbolic description of the fixed
effect model to be fitted, e.g. y ~ a + b.}

\item{data}{data frame containing variables used in the formula.}

\item{exposure.data}{numerical matrix of exposure data with same length as data, for a mixture setting (tdlmm, hdlmm):
named list containing equally sized numerical matrices of exposure data having same length as data.}

\item{dlm.type}{dlm model specification: "linear" (default), "nonlinear", "monotone".}

\item{family}{'gaussian' for continuous response, 'logit' for binomial, 'zinb' for zero-inflated negative binomial.}

\item{mixture}{flag for mixture, set to TRUE for tdlmm and hdlmm (default: FALSE).}

\item{het}{flag for heterogeneity, set to TRUE for hdlm and hdlmm (default: FALSE).}

\item{control.mcmc}{list of MCMC control parameters. This is passed to \link{dlmtree.control.mcmc}.}

\item{control.hyper}{list of hyperparameter control parameters. This is passed to \link{dlmtree.control.hyper}}

\item{control.family}{list of family control parameters. This is passed to \link{dlmtree.control.family}}

\item{control.tdlnm}{list of TDLNM control parameters. This is passed to \link{dlmtree.control.tdlnm}}

\item{control.het}{list of control parameters for heterogeneous models. This is passed to \link{dlmtree.control.het}}

\item{control.mix}{list of mixture control parameters. This is passed to \link{dlmtree.control.mix}}

\item{control.monotone}{list of control parameters for monotone model. This is passed to \link{dlmtree.control.monotone}}

\item{control.diagnose}{list of control parameters for diagnostics. This is passed to \link{dlmtree.control.diagnose}}
}
\value{
object of one of the classes: tdlm, tdlmm, tdlnm, hdlm, hdlmm, monotone
}
\description{
The 'dlmtree' function accommodates various response variable types, including continuous, binary, and zero-inflated count values.
The function is designed to handle both single exposure and exposure mixtures. For a single exposure, users are offered options to model non-linear effects (tdlnm),
linear effects (tdlm), or heterogeneous subgroup/individualized effects (hdlm). In the case of exposure mixtures, the function supports
lagged interactions (tdlmm), and heterogeneous subgroup/individualized effects (hdlmm) allowing for a comprehensive exploration of mixture exposure heterogeneity.
Additionally, users can fine-tune parameters to impose effect shrinkage and perform exposure selection, enhancing the adaptability and precision of the modeling process.
For more detailed documentation, visit: \href{https://danielmork.github.io/dlmtree/}{dlmtree website}.
}
\details{
dlmtree

Model is recommended to be run for at minimum 5000 burn-in
iterations followed by 15000 sampling iterations with a thinning factor of 5.
Convergence can be checked by re-running the model and validating consistency
of results. Examples are provided below for the syntax for running different types of models.
For more examples, visit: \href{https://danielmork.github.io/dlmtree/}{dlmtree website}.
}
\examples{
\donttest{

  # The first three examples are for one lagged exposure


  # treed distributed lag model (TDLM)
  # binary outcome with logit link

  D <- sim.tdlmm(sim = "A", mean.p = 0.5, n = 1000)
  tdlm.fit <- dlmtree(y ~ .,
                      data = D$dat,
                      exposure.data = D$exposures[[1]],
                      dlm.type = "linear",
                      family = "logit",
                      control.family = list(binomial.size = 1))

  # summarize results
  tdlm.sum <- summary(tdlm.fit)
  tdlm.sum

  # plot results
  plot(tdlm.sum)



  # Treed distributed lag nonlinear model (TDLNM)
  # Gaussian regression model
  D <- sim.tdlnm(sim = "A", error.to.signal = 1)
  tdlnm.fit <- dlmtree(formula = y ~ .,
                       data = D$dat,
                       exposure.data = D$exposures,
                       dlm.type = "nonlinear",
                       family = "gaussian")

  # summarize results
  tdlnm.sum <- summary(tdlnm.fit)
  tdlnm.sum

  # plot results
  plot(tdlnm.sum)



  # Heterogeneous TDLM (HDLM), similar to first example but with heterogeneous exposure response
  D <- sim.hdlmm(sim = "B", n = 1000)
  hdlm.fit <- dlmtree(y ~ .,
                      data = D$dat,
                      exposure.data = D$exposures,
                      dlm.type = "linear",
                      family = "gaussian",
                      het = TRUE)

  # summarize results
  hdlm.sum <- summary(hdlm.fit)
  hdlm.sum

  # shiny app for HDLM
  if (interactive()) {
    shiny(hdlm.fit)
  }



  # The next two examples are for a mixture (or multivariate) exposure


  # Treed distributed lag mixture model (TDLMM)
  # Model for mixutre (or multivariate) lagged exposures
  # with a homogenious exposure-time-response function
  D <- sim.tdlmm(sim = "B", error = 25, n = 1000)
  tdlmm.fit <- dlmtree(y ~ .,
                       data = D$dat, exposure.data = D$exposures,
                       control.mix = list(interactions = "noself"),
                       dlm.type = "linear", family = "gaussian",
                       mixture = TRUE)

  # summarize results
  tdlmm.sum <- summary(tdlmm.fit)

  # plot the marginal exposure-response for one exposure
  plot(tdlmm.sum, exposure1 = "e1")

  # plot exposure-response surface
  plot(tdlmm.sum, exposure1 = "e1", exposure2 = "e2")



  # heterogeneous version of TDLMM
  D <- sim.hdlmm(sim = "D", n = 1000)
  hdlmm.fit <- dlmtree(y ~ .,
                       data = D$dat,
                       exposure.data = D$exposures,
                       dlm.type = "linear",
                       family = "gaussian",
                       mixture = TRUE,
                       het = TRUE)

  # summarize results
  hdlmm.sum <- summary(hdlmm.fit)
  hdlmm.sum

  # summarize results
  if (interactive()) {
    shiny(hdlmm.fit)
  }


}
}
