% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_cox_pairwise_df.R
\name{get_cox_pairwise_df}
\alias{get_cox_pairwise_df}
\title{Generate Table of Pairwise Cox-PH and Log-Rank Results}
\usage{
get_cox_pairwise_df(model_formula, data, arm, ref_group = NULL)
}
\arguments{
\item{model_formula}{(\code{formula})\cr
A \code{formula} object specifying the survival model, typically in the form \code{Surv(time, status) ~ arm + covariates}.}

\item{data}{(\code{data.frame})\cr
A \code{data.frame} containing the survival data, including time, status, and the arm variable.}

\item{arm}{(\code{character})\cr
A single character string specifying the name of the column in \code{data} that contains the grouping/treatment
\strong{arm variable}. This column \strong{must be a factor} for correct stratification and comparison.}

\item{ref_group}{(\code{character} or \code{NULL})\cr
A single character string specifying the level of the \code{arm} variable to be used as the \strong{reference group} for
all pairwise comparisons. If \code{NULL} (the default), the \strong{first unique level} of the \code{arm} column is automatically
selected as the reference group.}
}
\value{
A \code{data.frame} with the results of the pairwise comparisons. The columns include:
\itemize{
\item \code{arm}: (rownames of the \code{data.frame}) The comparison arm (group) being tested against the reference group.
\item \code{hr}: The Hazard Ratio (HR) for the comparison arm vs. the reference arm, formatted to two decimal places.
\item \code{ci}: The 95\% confidence interval for the HR, presented as a string in the format "(lower, upper)", with
values formatted to two decimal places.
\item \code{pval}: The log-rank p-value for the comparison.
}
}
\description{
This function performs pairwise comparisons of treatment arms using the \strong{Cox Proportional Hazards model} and
calculates the corresponding \strong{log-rank p-value}. Each comparison tests a non-reference group against a specified
reference group.
}
\details{
The function iterates through each unique arm (excluding the reference group). For each iteration, it
filters the data to include only the current comparison arm and the reference arm, and then:
\itemize{
\item Fits a Cox model using \code{survival::coxph}.
\item Performs a log-rank test using \code{survival::survdiff}.
}
The Hazard Ratio and its 95\% confidence interval are extracted from the Cox model summary, and the p-value is
extracted from the log-rank test.
}
\examples{
# Example data setup (assuming 'time' is event time, 'status' is event indicator (1=event),
# and 'arm' is the treatment group)
library(dplyr) # For better data handling

# Prepare data in a modern dplyr-friendly way
surv_data <- survival::lung |>
  mutate(
    arm = factor(sample(c("A", "B", "C"), n(), replace = TRUE)),
    status = status - 1 # Convert status to 0/1
  ) |>
  filter(if_all(everything(), ~ !is.na(.)))

formula <- survival::Surv(time, status) ~ arm
results_tbl <- get_cox_pairwise_df(
  model_formula = formula,
  data = surv_data,
  arm = "arm",
  ref_group = "A"
)
print(results_tbl)

}
\seealso{
\code{annotate_gg_km()}, \code{gg_km()}, and the \code{survival} package functions \code{survival::coxph} and
\code{survival::survdiff}.
}
