% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/VizMostLikelyQuantileMap.R
\name{VizMostLikelyQuantileMap}
\alias{VizMostLikelyQuantileMap}
\title{Plot Maps of Most Likely Quantiles}
\usage{
VizMostLikelyQuantileMap(
  probs,
  lon,
  lat,
  cat_dim = "bin",
  bar_titles = NULL,
  col_unknown_cat = "white",
  drawleg = TRUE,
  margin_scale = NULL,
  bar_extra_margin = c(1.5, 0, 1.5, 0),
  ...
)
}
\arguments{
\item{probs}{A list of bi-dimensional arrays with the named dimensions 
'latitude' (or 'lat') and 'longitude' (or 'lon'), with equal size and in the 
same order, or a single tri-dimensional array with an additional dimension 
(e.g. 'bin') for the different categories. The arrays must contain 
probability values between 0 and 1, and the probabilities for all categories 
of a grid cell should not exceed 1 when added.}

\item{lon}{A numeric vector with the longitudes of the map grid, in the same 
order as the values along the corresponding dimension in \code{probs}.}

\item{lat}{A numeric vector with the latitudes of the map grid, in the same 
order as the values along the corresponding dimension in \code{probs}.}

\item{cat_dim}{The name of the dimension along which the different categories 
are stored in \code{probs}. This only applies if \code{probs} is provided in 
the form of 3-dimensional array. The default expected name is 'bin'.}

\item{bar_titles}{Vector of character strings with the names to be drawn on 
top of the color bar for each of the categories. As many titles as 
categories provided in \code{probs} must be provided.}

\item{col_unknown_cat}{Character string with a colour representation of the 
colour to be used to paint the cells for which no category can be clearly 
assigned. Takes the value 'white' by default.}

\item{drawleg}{Where to draw the common colour bar. Can take values TRUE, 
FALSE or:\cr
'up', 'u', 'U', 'top', 't', 'T', 'north', 'n', 'N'\cr
'down', 'd', 'D', 'bottom', 'b', 'B', 'south', 's', 'S' (default)\cr
'right', 'r', 'R', 'east', 'e', 'E'\cr
'left', 'l', 'L', 'west', 'w', 'W'}

\item{margin_scale}{Numeric vector of length 4 specifying the plot margins 
(excluding the color bar), in the order: bottom, left, top, and right. If 
not specified (NULL), the default margins from \code{par("mar")}, which are 
c(5.1, 4.1, 4.1, 2.1), are used. Default is NULL.}

\item{bar_extra_margin}{Numeric vector of length 4 specifying the margins to 
be added around the color bar, in the order: bottom, left, top, and right. 
The units are margin lines. The default values are c(1.5, 0, 1.5, 0).}

\item{...}{Additional parameters to be sent to \code{VizCombinedMap} and 
\code{VizEquiMap}.}
}
\value{
Invisibly returns \code{NULL} after producing the plot.
}
\description{
This function receives as main input (via the parameter 
\code{probs}) a collection of longitude-latitude maps, each containing the 
probabilities (from 0 to 1) of the different grid cells of belonging to a 
category. As many categories as maps provided as inputs are understood to 
exist. The maps of probabilities must be provided on a common rectangular 
regular grid, and a vector with the longitudes and a vector with the latitudes 
of the grid must be provided. The input maps can be provided in two forms, 
either as a list of multiple two-dimensional arrays (one for each category) or 
as a three-dimensional array, where one of the dimensions corresponds to the 
different categories.
}
\examples{

# Simple example:

x <- array(1:(20 * 10), dim = c(lat = 10, lon = 20)) / 200
a <- x * 0.6
b <- (1 - x) * 0.6
c <- 1 - (a + b)
lons <- seq(0, 359.5, length = 20)
lats <- seq(-89.5, 89.5, length = 10)

VizMostLikelyQuantileMap(list(a, b, c), lons, lats, 
                         toptitle = 'Most likely tercile map',
                         bar_titles = paste('\% of belonging to', c('a', 'b', 'c')), 
                         brks = 20, width = 10, height = 8,
                         bar_extra_margin = c(1, 0, 1.5, 0))


# More complex example:

# 1. Generation of sample data
lons <- seq(0, 359.5, length = 40)
lats <- seq(-89.5, 89.5, length = 20)

# Generate sample data (with dimensions time, lat, lon)
sample_data <- sample(1:5, 50 * 20 * 40, replace = TRUE)
dim(sample_data) <- c(time = 50, lat = 20, lon = 40)

# 2. Binning sample data
n_bins <- 4
prob_thresholds <- 1:n_bins / n_bins
prob_thresholds <- prob_thresholds[1:(n_bins - 1)]
thresholds <- quantile(sample_data, prob_thresholds)

binning <- function(x, thresholds) {
  n_samples <- length(x)
  n_bins <- length(thresholds) + 1
  thresholds <- c(thresholds, max(x))
  result <- 1:n_bins
  lower_threshold <- min(x) - 1
  for (i in 1:n_bins) {
    result[i] <- sum(x > lower_threshold & x <= thresholds[i]) / n_samples
    lower_threshold <- thresholds[i]
  }
  dim(result) <- c(bin = n_bins)
  result
}
bins <- apply(sample_data, 2:3, binning, thresholds)
names(dim(bins))[1] <- "bin"

VizMostLikelyQuantileMap(bins, lons, lats, 
                         toptitle = 'Most likely quantile map',
                         bar_titles = paste('\% of belonging to', letters[1:n_bins]),
                         brks = 20, width = 10, height = 10,
                         bar_extra_margin = c(1, 0, 1.5, 0))

}
\seealso{
\code{VizCombinedMap} and \code{VizEquiMap}
}
\author{
Veronica Torralba, \email{veronica.torralba@bsc.es}, Nicolau Manubens, 
\email{nicolau.manubens@bsc.es}
}
