#' Guard for Authenticating with the Google OpenID Connect server
#'
#' This guard requests you to log in with google and authenticates you
#' through their service. Your server must be registered and have a valid client
#' ID and client secret for this to work. Read more about registering an
#' application at <https://developers.google.com/identity/protocols/oauth2>. If
#' you want to limit access to only select users you should make sure to provide
#' a `validate` function that checks the userinfo against a whitelist.
#'
#' # User information
#' `guard_google()` automatically adds user information according to the
#' description in [guard_oidc()]. It sets the `provider` field to `"google"`.
#'
#' @inheritParams guard_oidc
#' @inheritDotParams guard_oidc -service_url -service_name
#'
#' @return A [GuardOIDC] object
#'
#' @references [Documentation for Googles OpenID Connect flow](https://developers.google.com/identity/openid-connect/openid-connect)
#'
#' @export
#'
#' @examples
#' google <- guard_google(
#'   redirect_url = "https://example.com/auth",
#'   client_id = "MY_APP_ID",
#'   client_secret = "SUCHASECRET"
#' )
#'
#' # Add it to a fireproof plugin
#' fp <- Fireproof$new()
#' fp$add_guard(google, "google_auth")
#'
#' # Use it in an endpoint
#' fp$add_auth("get", "/*", google_auth)
#'
guard_google <- function(
  redirect_url,
  client_id,
  client_secret,
  oauth_scopes = "profile",
  service_params = list(
    access_type = "offline",
    include_granted_scopes = "true"
  ),
  ...,
  name = "google"
) {
  guard_oidc(
    service_url = "https://accounts.google.com/",
    redirect_url = redirect_url,
    client_id = client_id,
    client_secret = client_secret,
    oauth_scopes = oauth_scopes,
    service_name = "google",
    service_params = service_params,
    ...,
    name = name
  )
}
