% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/image_collection.R
\name{create_image_collection}
\alias{create_image_collection}
\title{Create an image collection from a set of GDAL datasets or files}
\usage{
create_image_collection(
  files,
  format = NULL,
  out_file = tempfile(fileext = ".sqlite"),
  date_time = NULL,
  band_names = NULL,
  use_subdatasets = FALSE,
  unroll_archives = TRUE,
  quiet = FALSE,
  one_band_per_file = NULL
)
}
\arguments{
\item{files}{character vector with paths to image files on disk or any GDAL dataset identifiers (including virtual file systems and higher level drivers or GDAL subdatasets)}

\item{format}{collection format, can be either a name to use predefined formats (as output from \code{\link{collection_formats}}) or a path to a custom JSON format description file}

\item{out_file}{optional name of the output SQLite database file, defaults to a temporary file}

\item{date_time}{vector with date/ time for files; can be of class character, Date, or POSIXct (argument is only applicable for image collections without collection format)}

\item{band_names}{character vector with band names, length must match the number of bands in provided files (argument is only applicable for image collections without collection format)}

\item{use_subdatasets}{logical; use GDAL subdatasets of provided files (argument is only applicable for image collections without collection format)}

\item{unroll_archives}{automatically convert .zip, .tar archives and .gz compressed files to GDAL virtual file system dataset identifiers (e.g. by prepending /vsizip/) and add contained files to the list of considered files}

\item{quiet}{logical; if TRUE, do not print resulting image collection if return value is not assigned to a variable}

\item{one_band_per_file}{logical; if TRUE, assume that band_names are given for all files (argument is only applicable for image collections without collection format, see Details)}
}
\value{
image collection proxy object, which can be used to create a data cube using \code{\link{raster_cube}}
}
\description{
This function iterates over files or GDAL dataset identifiers and extracts datetime, image identifiers, and band information according to a given
collection format.
}
\details{
An image collection is a simple index (a SQLite database) containing references to existing image files / GDAL dataset identifiers.

Collections can be created in two different ways: First, if a collection format is specified (argument \code{format}), date/time, bands, 
and metadata are automatically extracted from provided files. This is the most general approach but requires a collection format for
the specific dataset. 

Second, image collections can sometimes be created without collection format by manually specifying date/time of images 
(argument \code{date_time}) and names of bands (argument \code{band_names}). This is possible if either each image file contains \emph{all} 
bands of the collection or only a single band. In the former case \code{band_names} simply contains the names of the bands or can be NULL 
to use default names. In the latter case (image files contain a single band only), the lengths of \code{band_names} and \code{date_time} must be identical.
By default, the function assumes one band per file if \code{length(band_names) == length(files)}. In the unlikely situation that this is 
not desired, it can be explicitly set using \code{one_band_per_file}.
}
\examples{
# 1. create image collection using a collection format 
L8_files <- list.files(system.file("L8NY18", package = "gdalcubes"),
                       ".TIF", recursive = TRUE, full.names = TRUE)
x = create_image_collection(L8_files, "L8_L1TP")
x 


# 2. create image collection without format for a single band
L8_files_B4 <- list.files(system.file("L8NY18", package = "gdalcubes"),
                       "_B4.TIF", recursive = TRUE, full.names = TRUE)
d = as.Date(substr(basename(L8_files_B4), 18, 25), "\%Y\%m\%d")
y = create_image_collection(L8_files_B4, date_time = d, band_names = "B4")
y


# 3. create image collection without format for all bands
d = as.Date(substr(basename(L8_files), 18, 25), "\%Y\%m\%d")
fname = basename(tools::file_path_sans_ext(L8_files))
b = substr(fname, 27, nchar(fname))
z = create_image_collection(L8_files, date_time = d, band_names = b)
z

}
