/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Fmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5FLprivate.h" 
#include "H5FDprivate.h" 
#include "H5MMprivate.h" 
#include "H5VMprivate.h" 

#define H5F_ACCUM_THROTTLE  8
#define H5F_ACCUM_THRESHOLD 2048
#define H5F_ACCUM_MAX_SIZE  (1024 * 1024) 

typedef enum {
    H5F_ACCUM_PREPEND, 
    H5F_ACCUM_APPEND   
} H5F_accum_adjust_t;

H5FL_BLK_DEFINE_STATIC(meta_accum);

herr_t
H5F__accum_read(H5F_shared_t *f_sh, H5FD_mem_t map_type, haddr_t addr, size_t size, void *buf )
{
    H5FD_t *file;                
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f_sh);
    assert(buf);

    
    file = f_sh->lf;

    
    if ((f_sh->feature_flags & H5FD_FEAT_ACCUMULATE_METADATA) && map_type != H5FD_MEM_DRAW) {
        H5F_meta_accum_t *accum; 

        
        accum = &f_sh->accum;

        if (size < H5F_ACCUM_MAX_SIZE) {
            
            assert(!accum->buf || (accum->alloc_size >= accum->size));

            
            if (H5_addr_defined(accum->loc) &&
                (H5_addr_overlap(addr, size, accum->loc, accum->size) || ((addr + size) == accum->loc) ||
                 (accum->loc + accum->size) == addr)) {
                size_t  amount_before; 
                haddr_t new_addr;      
                size_t  new_size;      

                
                new_addr = MIN(addr, accum->loc);
                new_size = (size_t)(MAX((addr + size), (accum->loc + accum->size)) - new_addr);

                
                if (new_size > accum->alloc_size) {
                    size_t new_alloc_size; 

                    
                    new_alloc_size = (size_t)1 << (1 + H5VM_log2_gen((uint64_t)(new_size - 1)));

                    
                    if (NULL == (accum->buf = H5FL_BLK_REALLOC(meta_accum, accum->buf, new_alloc_size)))
                        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                                    "unable to allocate metadata accumulator buffer");

                    
                    accum->alloc_size = new_alloc_size;

                    
                    memset(accum->buf + accum->size, 0, (accum->alloc_size - accum->size));
                } 

                
                if (addr < accum->loc) {
                    
                    H5_CHECKED_ASSIGN(amount_before, size_t, (accum->loc - addr), hsize_t);

                    
                    memmove(accum->buf + amount_before, accum->buf, accum->size);

                    
                    if (accum->dirty)
                        accum->dirty_off += amount_before;

                    
                    if (H5FD_read(file, map_type, addr, amount_before, accum->buf) < 0)
                        HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL, "driver read request failed");
                } 
                else
                    amount_before = 0;

                
                if ((addr + size) > (accum->loc + accum->size)) {
                    size_t amount_after; 

                    
                    H5_CHECKED_ASSIGN(amount_after, size_t, ((addr + size) - (accum->loc + accum->size)),
                                      hsize_t);

                    
                    if (H5FD_read(file, map_type, (accum->loc + accum->size), amount_after,
                                  (accum->buf + accum->size + amount_before)) < 0)
                        HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL, "driver read request failed");
                } 

                
                H5MM_memcpy(buf, accum->buf + (addr - new_addr), size);

                
                accum->loc  = new_addr;
                accum->size = new_size;
            } 
            
            else {
                
                if (H5FD_read(file, map_type, addr, size, buf) < 0)
                    HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL, "driver read request failed");
            } 
        }     
        else {
            
            if (H5FD_read(file, map_type, addr, size, buf) < 0)
                HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL, "driver read request failed");

            
            
            if (accum->dirty &&
                H5_addr_overlap(addr, size, accum->loc + accum->dirty_off, accum->dirty_len)) {
                haddr_t dirty_loc = accum->loc + accum->dirty_off; 
                size_t  buf_off;                                   
                size_t  dirty_off;                                 
                size_t  overlap_size;                              

                
                if (H5_addr_le(addr, dirty_loc)) {
                    
                    buf_off = (size_t)(dirty_loc - addr);

                    
                    dirty_off = 0;

                    
                    if (H5_addr_lt(addr + size, dirty_loc + accum->dirty_len))
                        overlap_size = (size_t)((addr + size) - buf_off);
                    else 
                        overlap_size = accum->dirty_len;
                }      
                else { 
                    
                    buf_off      = 0;
                    dirty_off    = (size_t)(addr - dirty_loc);
                    overlap_size = (size_t)((dirty_loc + accum->dirty_len) - addr);
                } 

                
                H5MM_memcpy((unsigned char *)buf + buf_off,
                            (unsigned char *)accum->buf + accum->dirty_off + dirty_off, overlap_size);
            } 
        }     
    }         
    else {
        
        if (H5FD_read(file, map_type, addr, size, buf) < 0)
            HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL, "driver read request failed");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5F__accum_adjust(H5F_meta_accum_t *accum, H5FD_t *file, H5F_accum_adjust_t adjust, size_t size)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(accum);
    assert(file);
    assert(H5F_ACCUM_APPEND == adjust || H5F_ACCUM_PREPEND == adjust);
    assert(size > 0);
    assert(size <= H5F_ACCUM_MAX_SIZE);

    
    if ((size + accum->size) > accum->alloc_size) {
        size_t new_size; 

        
        new_size = (size_t)1 << (1 + H5VM_log2_gen((uint64_t)((size + accum->size) - 1)));

        
        if (new_size > H5F_ACCUM_MAX_SIZE) {
            size_t shrink_size;  
            size_t remnant_size; 

            

            
            if (size > (H5F_ACCUM_MAX_SIZE / 2)) {
                new_size     = H5F_ACCUM_MAX_SIZE;
                shrink_size  = accum->size;
                remnant_size = 0;
            } 
            else {
                if (H5F_ACCUM_PREPEND == adjust) {
                    new_size     = (H5F_ACCUM_MAX_SIZE / 2);
                    shrink_size  = (H5F_ACCUM_MAX_SIZE / 2);
                    remnant_size = accum->size - shrink_size;
                } 
                else {
                    size_t adjust_size = size + accum->dirty_len;

                    
                    if (accum->dirty && (adjust_size <= H5F_ACCUM_MAX_SIZE)) {
                        if ((ssize_t)(H5F_ACCUM_MAX_SIZE - (accum->dirty_off + adjust_size)) >=
                            (ssize_t)(2 * size))
                            shrink_size = accum->dirty_off / 2;
                        else
                            shrink_size = accum->dirty_off;
                        remnant_size = accum->size - shrink_size;
                        new_size     = remnant_size + size;
                    } 
                    else {
                        new_size     = (H5F_ACCUM_MAX_SIZE / 2);
                        shrink_size  = (H5F_ACCUM_MAX_SIZE / 2);
                        remnant_size = accum->size - shrink_size;
                    } 
                }     
            }         

            
            if (accum->dirty) {
                
                if (H5F_ACCUM_PREPEND == adjust) {
                    
                    if ((accum->size - shrink_size) < (accum->dirty_off + accum->dirty_len)) {
                        
                        if (H5FD_write(file, H5FD_MEM_DEFAULT, (accum->loc + accum->dirty_off),
                                       accum->dirty_len, (accum->buf + accum->dirty_off)) < 0)
                            HGOTO_ERROR(H5E_FILE, H5E_WRITEERROR, FAIL, "file write failed");

                        
                        accum->dirty = false;
                    } 
                }     
                else {
                    
                    if (shrink_size > accum->dirty_off) {
                        
                        if (H5FD_write(file, H5FD_MEM_DEFAULT, (accum->loc + accum->dirty_off),
                                       accum->dirty_len, (accum->buf + accum->dirty_off)) < 0)
                            HGOTO_ERROR(H5E_FILE, H5E_WRITEERROR, FAIL, "file write failed");

                        
                        accum->dirty = false;
                    } 

                    
                    accum->dirty_off -= shrink_size;
                } 
            }     

            
            accum->size = remnant_size;

            
            if (H5F_ACCUM_APPEND == adjust) {
                
                memmove(accum->buf, (accum->buf + shrink_size), remnant_size);

                
                accum->loc += shrink_size;
            } 
        }     

        
        if (new_size > accum->alloc_size) {
            unsigned char *new_buf; 

            
            if (NULL == (new_buf = H5FL_BLK_REALLOC(meta_accum, accum->buf, new_size)))
                HGOTO_ERROR(H5E_FILE, H5E_CANTALLOC, FAIL, "unable to allocate metadata accumulator buffer");

            
            accum->buf        = new_buf;
            accum->alloc_size = new_size;

            
            memset(accum->buf + accum->size, 0, (accum->alloc_size - (accum->size + size)));
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F__accum_write(H5F_shared_t *f_sh, H5FD_mem_t map_type, haddr_t addr, size_t size, const void *buf)
{
    H5FD_t *file;                
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f_sh);
    assert(H5F_SHARED_INTENT(f_sh) & H5F_ACC_RDWR);
    assert(buf);

    
    file = f_sh->lf;

    
    if ((f_sh->feature_flags & H5FD_FEAT_ACCUMULATE_METADATA) && map_type != H5FD_MEM_DRAW) {
        H5F_meta_accum_t *accum; 

        
        accum = &f_sh->accum;

        if (size < H5F_ACCUM_MAX_SIZE) {
            
            assert(!accum->buf || (accum->alloc_size >= accum->size));

            
            if (accum->size > 0) {
                
                if (H5_addr_defined(accum->loc) && (addr + size) == accum->loc) {
                    
                    if (H5F__accum_adjust(accum, file, H5F_ACCUM_PREPEND, size) < 0)
                        HGOTO_ERROR(H5E_IO, H5E_CANTRESIZE, FAIL, "can't adjust metadata accumulator");

                    
                    memmove(accum->buf + size, accum->buf, accum->size);

                    
                    H5MM_memcpy(accum->buf, buf, size);

                    
                    accum->loc = addr;
                    accum->size += size;

                    
                    if (accum->dirty)
                        accum->dirty_len = size + accum->dirty_off + accum->dirty_len;
                    else {
                        accum->dirty_len = size;
                        accum->dirty     = true;
                    } 
                    accum->dirty_off = 0;
                } 
                
                else if (H5_addr_defined(accum->loc) && addr == (accum->loc + accum->size)) {
                    
                    if (H5F__accum_adjust(accum, file, H5F_ACCUM_APPEND, size) < 0)
                        HGOTO_ERROR(H5E_IO, H5E_CANTRESIZE, FAIL, "can't adjust metadata accumulator");

                    
                    H5MM_memcpy(accum->buf + accum->size, buf, size);

                    
                    if (accum->dirty)
                        accum->dirty_len = size + (accum->size - accum->dirty_off);
                    else {
                        accum->dirty_off = accum->size;
                        accum->dirty_len = size;
                        accum->dirty     = true;
                    } 

                    
                    accum->size += size;
                } 
                
                else if (H5_addr_defined(accum->loc) &&
                         H5_addr_overlap(addr, size, accum->loc, accum->size)) {
                    size_t add_size; 

                    
                    if (addr >= accum->loc && (addr + size) <= (accum->loc + accum->size)) {
                        size_t dirty_off = (size_t)(addr - accum->loc);

                        
                        H5MM_memcpy(accum->buf + dirty_off, buf, size);

                        
                        if (accum->dirty) {
                            
                            if (dirty_off <= accum->dirty_off) {
                                if ((dirty_off + size) <= (accum->dirty_off + accum->dirty_len))
                                    accum->dirty_len = (accum->dirty_off + accum->dirty_len) - dirty_off;
                                else
                                    accum->dirty_len = size;
                                accum->dirty_off = dirty_off;
                            } 
                            else {
                                if ((dirty_off + size) <= (accum->dirty_off + accum->dirty_len))
                                    ; 
                                else
                                    accum->dirty_len = (dirty_off + size) - accum->dirty_off;
                            } 
                        }     
                        else {
                            accum->dirty_off = dirty_off;
                            accum->dirty_len = size;
                            accum->dirty     = true;
                        } 
                    }     
                    
                    else if (addr < accum->loc && (addr + size) <= (accum->loc + accum->size)) {
                        size_t old_offset; 

                        
                        H5_CHECKED_ASSIGN(add_size, size_t, (accum->loc - addr), hsize_t);

                        
                        if (H5F__accum_adjust(accum, file, H5F_ACCUM_PREPEND, add_size) < 0)
                            HGOTO_ERROR(H5E_IO, H5E_CANTRESIZE, FAIL, "can't adjust metadata accumulator");

                        
                        H5_CHECKED_ASSIGN(old_offset, size_t, (addr + size) - accum->loc, hsize_t);

                        
                        memmove(accum->buf + size, accum->buf + old_offset, (accum->size - old_offset));

                        
                        H5MM_memcpy(accum->buf, buf, size);

                        
                        accum->loc = addr;
                        accum->size += add_size;

                        
                        if (accum->dirty) {
                            size_t curr_dirty_end = add_size + accum->dirty_off + accum->dirty_len;

                            accum->dirty_off = 0;
                            if (size <= curr_dirty_end)
                                accum->dirty_len = curr_dirty_end;
                            else
                                accum->dirty_len = size;
                        } 
                        else {
                            accum->dirty_off = 0;
                            accum->dirty_len = size;
                            accum->dirty     = true;
                        } 
                    }     
                    
                    else if (addr >= accum->loc && (addr + size) > (accum->loc + accum->size)) {
                        size_t dirty_off; 

                        
                        H5_CHECKED_ASSIGN(add_size, size_t, (addr + size) - (accum->loc + accum->size),
                                          hsize_t);

                        
                        if (H5F__accum_adjust(accum, file, H5F_ACCUM_APPEND, add_size) < 0)
                            HGOTO_ERROR(H5E_IO, H5E_CANTRESIZE, FAIL, "can't adjust metadata accumulator");

                        
                        dirty_off = (size_t)(addr - accum->loc);

                        
                        H5MM_memcpy(accum->buf + dirty_off, buf, size);

                        
                        accum->size += add_size;

                        
                        if (accum->dirty) {
                            
                            if (dirty_off <= accum->dirty_off) {
                                accum->dirty_off = dirty_off;
                                accum->dirty_len = size;
                            } 
                            else {
                                accum->dirty_len = (dirty_off + size) - accum->dirty_off;
                            } 
                        }     
                        else {
                            accum->dirty_off = dirty_off;
                            accum->dirty_len = size;
                            accum->dirty     = true;
                        } 
                    }     
                    
                    else {
                        
                        if (size > accum->alloc_size) {
                            size_t new_alloc_size; 

                            
                            new_alloc_size = (size_t)1 << (1 + H5VM_log2_gen((uint64_t)(size - 1)));

                            
                            if (NULL ==
                                (accum->buf = H5FL_BLK_REALLOC(meta_accum, accum->buf, new_alloc_size)))
                                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                                            "unable to allocate metadata accumulator buffer");

                            
                            accum->alloc_size = new_alloc_size;

                            
                            memset(accum->buf + size, 0, (accum->alloc_size - size));
                        } 

                        
                        H5MM_memcpy(accum->buf, buf, size);

                        
                        accum->loc  = addr;
                        accum->size = size;

                        
                        accum->dirty_off = 0;
                        accum->dirty_len = size;
                        accum->dirty     = true;
                    } 
                }     
                
                else {
                    
                    if (accum->dirty) {
                        if (H5FD_write(file, H5FD_MEM_DEFAULT, accum->loc + accum->dirty_off,
                                       accum->dirty_len, accum->buf + accum->dirty_off) < 0)
                            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "file write failed");

                        
                        accum->dirty = false;
                    } 

                    
                    
                    if (size > accum->alloc_size) {
                        size_t new_size;   
                        size_t clear_size; 

                        
                        new_size = (size_t)1 << (1 + H5VM_log2_gen((uint64_t)(size - 1)));

                        
                        if (NULL == (accum->buf = H5FL_BLK_REALLOC(meta_accum, accum->buf, new_size)))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                                        "unable to allocate metadata accumulator buffer");

                        
                        accum->alloc_size = new_size;

                        
                        clear_size = MAX(accum->size, size);
                        memset(accum->buf + clear_size, 0, (accum->alloc_size - clear_size));
                    } 
                    else {
                        
                        if (size < (accum->alloc_size / H5F_ACCUM_THROTTLE) &&
                            accum->alloc_size > H5F_ACCUM_THRESHOLD) {
                            size_t tmp_size =
                                (accum->alloc_size / H5F_ACCUM_THROTTLE); 

                            
                            if (NULL == (accum->buf = H5FL_BLK_REALLOC(meta_accum, accum->buf, tmp_size)))
                                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                                            "unable to allocate metadata accumulator buffer");

                            
                            accum->alloc_size = tmp_size;
                        } 
                    }     

                    
                    accum->loc  = addr;
                    accum->size = size;

                    
                    H5MM_memcpy(accum->buf, buf, size);

                    
                    accum->dirty_off = 0;
                    accum->dirty_len = size;
                    accum->dirty     = true;
                } 
            }     
            
            else {
                
                if (size > accum->alloc_size) {
                    size_t new_size; 

                    
                    new_size = (size_t)1 << (1 + H5VM_log2_gen((uint64_t)(size - 1)));

                    
                    if (NULL == (accum->buf = H5FL_BLK_REALLOC(meta_accum, accum->buf, new_size)))
                        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                                    "unable to allocate metadata accumulator buffer");

                    
                    accum->alloc_size = new_size;

                    
                    memset(accum->buf + size, 0, (accum->alloc_size - size));
                } 

                
                accum->loc  = addr;
                accum->size = size;

                
                H5MM_memcpy(accum->buf, buf, size);

                
                accum->dirty_off = 0;
                accum->dirty_len = size;
                accum->dirty     = true;
            } 
        }     
        else {
            
            if ((H5F_SHARED_INTENT(f_sh) & H5F_ACC_SWMR_WRITE) > 0)
                
                if (H5F__accum_reset(f_sh, true, false) < 0)
                    HGOTO_ERROR(H5E_IO, H5E_CANTRESET, FAIL, "can't reset accumulator");

            
            if (H5FD_write(file, map_type, addr, size, buf) < 0)
                HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "file write failed");

            
            
            if (H5_addr_defined(accum->loc) && H5_addr_overlap(addr, size, accum->loc, accum->size)) {
                
                if (H5_addr_le(addr, accum->loc)) {
                    
                    if (H5_addr_le(addr + size, accum->loc + accum->size)) {
                        size_t overlap_size; 

                        
                        overlap_size = (size_t)((addr + size) - accum->loc);

                        
                        if (accum->dirty) {
                            haddr_t dirty_start =
                                accum->loc + accum->dirty_off; 
                            haddr_t dirty_end =
                                dirty_start + accum->dirty_len; 

                            
                            if (H5_addr_le(dirty_end, addr + size)) {
                                accum->dirty     = false;
                                accum->dirty_len = 0;
                            } 
                            else {
                                
                                if (H5_addr_le(addr + size, dirty_start))
                                    accum->dirty_off = overlap_size;
                                else { 
                                    accum->dirty_off = 0;
                                    accum->dirty_len -= (size_t)((addr + size) - dirty_start);
                                } 
                            }     
                        }         

                        
                        accum->loc += overlap_size;
                        accum->size -= overlap_size;
                        memmove(accum->buf, accum->buf + overlap_size, accum->size);
                    }      
                    else { 
                        
                        if (H5F__accum_reset(f_sh, false, false) < 0)
                            HGOTO_ERROR(H5E_IO, H5E_CANTRESET, FAIL, "can't reset accumulator");
                    }                    
                }                        
                else {                   
                    size_t overlap_size; 

                    
                    assert(H5_addr_gt(addr + size, accum->loc + accum->size));

                    
                    overlap_size = (size_t)((accum->loc + accum->size) - addr);

                    
                    if (accum->dirty) {
                        haddr_t dirty_start =
                            accum->loc + accum->dirty_off; 
                        haddr_t dirty_end =
                            dirty_start + accum->dirty_len; 

                        
                        if (H5_addr_ge(dirty_start, addr)) {
                            accum->dirty     = false;
                            accum->dirty_len = 0;
                        } 
                        else {
                            
                            if (H5_addr_le(dirty_end, addr))
                                ; 
                            else  
                                accum->dirty_len = (size_t)(addr - dirty_start);
                        } 
                    }     

                    
                    accum->size -= overlap_size;
                } 
            }     
        }         
    }             
    else {
        
        if (H5FD_write(file, map_type, addr, size, buf) < 0)
            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "file write failed");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F__accum_free(H5F_shared_t *f_sh, H5FD_mem_t H5_ATTR_UNUSED type, haddr_t addr, hsize_t size)
{
    H5F_meta_accum_t *accum;               
    H5FD_t           *file;                
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f_sh);

    
    accum = &f_sh->accum;

    
    file = f_sh->lf;

    
    if ((f_sh->feature_flags & H5FD_FEAT_ACCUMULATE_METADATA) && H5_addr_defined(accum->loc) &&
        H5_addr_overlap(addr, size, accum->loc, accum->size)) {
        size_t overlap_size; 

        
        
        assert(H5FD_MEM_DRAW != type);
        assert(H5FD_MEM_GHEAP != type); 

        
        if (H5_addr_le(addr, accum->loc)) {
            
            if (H5_addr_ge(addr + size, accum->loc + accum->size)) {
                
                accum->loc   = HADDR_UNDEF;
                accum->size  = 0;
                accum->dirty = false;
            } 
            
            else {
                size_t new_accum_size; 

                
                H5_CHECKED_ASSIGN(overlap_size, size_t, (addr + size) - accum->loc, haddr_t);
                
                
                assert(overlap_size < accum->size);
                new_accum_size = accum->size - overlap_size;

                
                memmove(accum->buf, accum->buf + overlap_size, new_accum_size);

                
                accum->loc += overlap_size;
                accum->size = new_accum_size;

                
                if (accum->dirty) {
                    
                    if (overlap_size < accum->dirty_off)
                        accum->dirty_off -= overlap_size;
                    else {
                        
                        if (overlap_size < (accum->dirty_off + accum->dirty_len)) {
                            accum->dirty_len = (accum->dirty_off + accum->dirty_len) - overlap_size;
                            accum->dirty_off = 0;
                        } 
                        
                        else
                            accum->dirty = false;
                    } 
                }     
            }         
        }             
        
        else {
            haddr_t dirty_end   = accum->loc + accum->dirty_off + accum->dirty_len;
            haddr_t dirty_start = accum->loc + accum->dirty_off;

            
            H5_CHECKED_ASSIGN(overlap_size, size_t, (accum->loc + accum->size) - addr, haddr_t);

            
            if (accum->dirty && H5_addr_lt(addr, dirty_end)) {
                haddr_t tail_addr;

                
                tail_addr = addr + size;

                
                if (H5_addr_lt(addr, dirty_start)) {
                    
                    if (H5_addr_le(tail_addr, dirty_start)) {
                        
                        if (H5FD_write(file, H5FD_MEM_DEFAULT, dirty_start, accum->dirty_len,
                                       accum->buf + accum->dirty_off) < 0)
                            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "file write failed");
                    } 
                    
                    
                    else if (H5_addr_lt(tail_addr, dirty_end)) {
                        size_t write_size;
                        size_t dirty_delta;

                        write_size  = (size_t)(dirty_end - tail_addr);
                        dirty_delta = accum->dirty_len - write_size;

                        assert(write_size > 0);

                        
                        if (H5FD_write(file, H5FD_MEM_DEFAULT, dirty_start + dirty_delta, write_size,
                                       accum->buf + accum->dirty_off + dirty_delta) < 0)
                            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "file write failed");
                    } 

                    
                    accum->dirty = false;
                } 
                
                else {
                    
                    if (H5_addr_lt(tail_addr, dirty_end)) {
                        size_t write_size;
                        size_t dirty_delta;

                        write_size  = (size_t)(dirty_end - tail_addr);
                        dirty_delta = accum->dirty_len - write_size;

                        assert(write_size > 0);

                        
                        if (H5FD_write(file, H5FD_MEM_DEFAULT, dirty_start + dirty_delta, write_size,
                                       accum->buf + accum->dirty_off + dirty_delta) < 0)
                            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "file write failed");
                    } 

                    
                    if (H5_addr_eq(addr, dirty_start)) {
                        
                        accum->dirty = false;
                    } 
                    
                    else {
                        accum->dirty_len = (size_t)(addr - dirty_start);
                    } 
                }     

            } 

            
            accum->size = accum->size - overlap_size;
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F__accum_flush(H5F_shared_t *f_sh)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f_sh);

    
    if ((f_sh->feature_flags & H5FD_FEAT_ACCUMULATE_METADATA) && f_sh->accum.dirty) {
        H5FD_t *file; 

        
        file = f_sh->lf;

        
        if (H5FD_write(file, H5FD_MEM_DEFAULT, f_sh->accum.loc + f_sh->accum.dirty_off, f_sh->accum.dirty_len,
                       f_sh->accum.buf + f_sh->accum.dirty_off) < 0)
            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "file write failed");

        
        f_sh->accum.dirty = false;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F__accum_reset(H5F_shared_t *f_sh, bool flush, bool force)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f_sh);

    
    if (flush)
        if (H5F__accum_flush(f_sh) < 0) {
            HDONE_ERROR(H5E_FILE, H5E_CANTFLUSH, FAIL, "can't flush metadata accumulator");
            if (!force)
                HGOTO_DONE(FAIL);
        }

    
    if (f_sh->feature_flags & H5FD_FEAT_ACCUMULATE_METADATA) {
        
        if (f_sh->accum.buf)
            f_sh->accum.buf = H5FL_BLK_FREE(meta_accum, f_sh->accum.buf);

        
        f_sh->accum.alloc_size = f_sh->accum.size = 0;
        f_sh->accum.loc                           = HADDR_UNDEF;
        f_sh->accum.dirty                         = false;
        f_sh->accum.dirty_len                     = 0;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
