/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Fmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5Gprivate.h"  
#include "H5MMprivate.h" 
#include "H5Pprivate.h"  

static void H5F__mount_count_ids_recurse(H5F_t *f, unsigned *nopen_files, unsigned *nopen_objs);

herr_t
H5F__close_mounts(H5F_t *f)
{
    unsigned u;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(f);

    
    for (u = f->shared->mtab.nmounts - 1; u < f->shared->mtab.nmounts; u--) {
        
        if (f->shared->mtab.child[u].file->parent == f) {
            
            f->shared->mtab.child[u].file->parent = NULL;

            
            if (H5G_close(f->shared->mtab.child[u].group) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTCLOSEOBJ, FAIL, "can't close child group");

            
            if (H5F_try_close(f->shared->mtab.child[u].file, NULL) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTCLOSEFILE, FAIL, "can't close child file");

            
            memmove(f->shared->mtab.child + u, f->shared->mtab.child + u + 1,
                    (f->shared->mtab.nmounts - u - 1) * sizeof(f->shared->mtab.child[0]));
            f->shared->mtab.nmounts--;
            f->nmounts--;
        }
    }

    assert(f->nmounts == 0);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F_mount(const H5G_loc_t *loc, const char *name, H5F_t *child, hid_t H5_ATTR_UNUSED plist_id)
{
    H5G_t     *mount_point = NULL;  
    H5F_t     *ancestor    = NULL;  
    H5F_t     *parent      = NULL;  
    unsigned   lt, rt, md;          
    int        cmp;                 
    H5G_loc_t  mp_loc;              
    H5G_name_t mp_path;             
    H5O_loc_t  mp_oloc;             
    H5G_loc_t  root_loc;            
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(loc);
    assert(name && *name);
    assert(child);
    assert(true == H5P_isa_class(plist_id, H5P_FILE_MOUNT));

    
    mp_loc.oloc = &mp_oloc;
    mp_loc.path = &mp_path;
    H5G_loc_reset(&mp_loc);

    
    if (child->parent)
        HGOTO_ERROR(H5E_FILE, H5E_MOUNT, FAIL, "file is already mounted");
    if (H5G_loc_find(loc, name, &mp_loc) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "group not found");
    
    if (mp_loc.oloc->holding_file != false)
        HGOTO_ERROR(H5E_FILE, H5E_MOUNT, FAIL, "mount path cannot contain links to external files");

    
    if (NULL == (mount_point = H5G_open(&mp_loc)))
        HGOTO_ERROR(H5E_FILE, H5E_MOUNT, FAIL, "mount point not found");

    
    if (H5G_MOUNTED(mount_point))
        HGOTO_ERROR(H5E_FILE, H5E_MOUNT, FAIL, "mount point is already in use");

    
    
    parent = H5G_fileof(mount_point);
    assert(parent);
    mp_loc.oloc = H5G_oloc(mount_point);
    assert(mp_loc.oloc);
    mp_loc.path = H5G_nameof(mount_point);
    assert(mp_loc.path);
    for (ancestor = parent; ancestor; ancestor = ancestor->parent)
        if (ancestor->shared == child->shared)
            HGOTO_ERROR(H5E_FILE, H5E_MOUNT, FAIL, "mount would introduce a cycle");

    
    if (parent->shared->fc_degree != child->shared->fc_degree)
        HGOTO_ERROR(H5E_FILE, H5E_MOUNT, FAIL, "mounted file has different file close degree than parent");

    
    lt = md = 0;
    rt      = parent->shared->mtab.nmounts;
    cmp     = -1;
    while (lt < rt && cmp) {
        H5O_loc_t *oloc; 

        md   = (lt + rt) / 2;
        oloc = H5G_oloc(parent->shared->mtab.child[md].group);
        cmp  = H5_addr_cmp(mp_loc.oloc->addr, oloc->addr);
        if (cmp < 0)
            rt = md;
        else if (cmp > 0)
            lt = md + 1;
    }
    if (cmp > 0)
        md++;
    if (!cmp)
        HGOTO_ERROR(H5E_FILE, H5E_MOUNT, FAIL, "mount point is already in use");

    
    if (parent->shared->mtab.nmounts >= parent->shared->mtab.nalloc) {
        unsigned     n = MAX(16, 2 * parent->shared->mtab.nalloc);
        H5F_mount_t *x = (H5F_mount_t *)H5MM_realloc(parent->shared->mtab.child,
                                                     n * sizeof(parent->shared->mtab.child[0]));

        if (!x)
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed for mount table");
        parent->shared->mtab.child  = x;
        parent->shared->mtab.nalloc = n;
    }

    
    memmove(parent->shared->mtab.child + md + 1, parent->shared->mtab.child + md,
            (parent->shared->mtab.nmounts - md) * sizeof(parent->shared->mtab.child[0]));
    parent->shared->mtab.nmounts++;
    parent->nmounts++;
    parent->shared->mtab.child[md].group = mount_point;
    parent->shared->mtab.child[md].file  = child;
    child->parent                        = parent;

    
    if (H5G_mount(parent->shared->mtab.child[md].group) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTCLOSEOBJ, FAIL, "unable to set group mounted flag");

    
    if (NULL == (root_loc.oloc = H5G_oloc(child->shared->root_grp)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get object location for root group");
    if (NULL == (root_loc.path = H5G_nameof(child->shared->root_grp)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get path for root group");

    
    
    if (H5G_name_replace(NULL, H5G_NAME_MOUNT, mp_loc.oloc->file, mp_loc.path->full_path_r,
                         root_loc.oloc->file, root_loc.path->full_path_r) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_MOUNT, FAIL, "unable to replace name");

done:
    if (ret_value < 0) {
        if (mount_point) {
            if (H5G_close(mount_point) < 0)
                HDONE_ERROR(H5E_FILE, H5E_CANTCLOSEOBJ, FAIL, "unable to close mounted group");
        }
        else {
            if (H5G_loc_free(&mp_loc) < 0)
                HDONE_ERROR(H5E_SYM, H5E_CANTRELEASE, FAIL, "unable to free mount location");
        }
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F_unmount(const H5G_loc_t *loc, const char *name)
{
    H5G_t     *child_group = NULL;   
    H5F_t     *child       = NULL;   
    H5F_t     *parent      = NULL;   
    H5O_loc_t *mnt_oloc;             
    H5G_name_t mp_path;              
    H5O_loc_t  mp_oloc;              
    H5G_loc_t  mp_loc;               
    bool       mp_loc_setup = false; 
    H5G_loc_t  root_loc;             
    int        child_idx;            
    herr_t     ret_value = SUCCEED;  

    FUNC_ENTER_NOAPI(FAIL)

    assert(loc);
    assert(name && *name);

    
    mp_loc.oloc = &mp_oloc;
    mp_loc.path = &mp_path;
    H5G_loc_reset(&mp_loc);

    
    if (H5G_loc_find(loc, name, &mp_loc ) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_NOTFOUND, FAIL, "group not found");
    mp_loc_setup = true;
    child        = mp_loc.oloc->file;
    mnt_oloc     = H5G_oloc(child->shared->root_grp);
    child_idx    = -1;

    if (child->parent && H5_addr_eq(mp_oloc.addr, mnt_oloc->addr)) {
        unsigned u; 

        
        parent = child->parent;
        for (u = 0; u < parent->shared->mtab.nmounts; u++) {
            if (parent->shared->mtab.child[u].file->shared == child->shared) {
                
                child_idx = (int)u;
                break;
            }
        }
    }
    else {
        unsigned lt, rt, md = 0; 
        int      cmp;            

        
        parent = child; 
        lt     = 0;
        rt     = parent->shared->mtab.nmounts;
        cmp    = -1;

        while (lt < rt && cmp) {
            md       = (lt + rt) / 2;
            mnt_oloc = H5G_oloc(parent->shared->mtab.child[md].group);
            cmp      = H5_addr_cmp(mp_oloc.addr, mnt_oloc->addr);
            if (cmp < 0)
                rt = md;
            else
                lt = md + 1;
        }

        if (cmp)
            HGOTO_ERROR(H5E_FILE, H5E_MOUNT, FAIL, "not a mount point");

        
        child_idx = (int)md;
        H5G_loc_free(&mp_loc);
        mp_loc_setup = false;
        mp_loc.oloc  = mnt_oloc;
        mp_loc.path  = H5G_nameof(parent->shared->mtab.child[md].group);
        child        = parent->shared->mtab.child[child_idx].file;

        
        parent = child->parent;
    } 
    assert(child_idx >= 0);

    
    child_group = parent->shared->mtab.child[child_idx].group;

    
    if (NULL == (root_loc.oloc = H5G_oloc(child->shared->root_grp)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get object location for root group");
    if (NULL == (root_loc.path = H5G_nameof(child->shared->root_grp)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get path for root group");

    
    if (H5G_name_replace(NULL, H5G_NAME_UNMOUNT, mp_loc.oloc->file, mp_loc.path->full_path_r,
                         root_loc.oloc->file, root_loc.path->full_path_r) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "unable to replace name");

    
    memmove(parent->shared->mtab.child + (unsigned)child_idx,
            (parent->shared->mtab.child + (unsigned)child_idx) + 1,
            ((parent->shared->mtab.nmounts - (unsigned)child_idx) - 1) *
                sizeof(parent->shared->mtab.child[0]));
    parent->shared->mtab.nmounts -= 1;
    parent->nmounts -= 1;

    
    if (H5G_unmount(child_group) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTRELEASE, FAIL, "unable to reset group mounted flag");
    if (H5G_close(child_group) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTCLOSEOBJ, FAIL, "unable to close unmounted group");

    
    child->parent = NULL;
    if (H5F_try_close(child, NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTCLOSEFILE, FAIL, "unable to close unmounted file");

done:
    
    if (mp_loc_setup)
        H5G_loc_free(&mp_loc);

    FUNC_LEAVE_NOAPI(ret_value)
} 

bool
H5F_is_mount(const H5F_t *file)
{
    bool ret_value = false; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(file);

    if (file->parent != NULL)
        ret_value = true;
    else
        ret_value = false;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static void
H5F__mount_count_ids_recurse(H5F_t *f, unsigned *nopen_files, unsigned *nopen_objs)
{
    unsigned u; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(nopen_files);
    assert(nopen_objs);

    
    if (H5F_ID_EXISTS(f))
        *nopen_files += 1;

    
    *nopen_objs += (f->nopen_objs - f->nmounts);

    
    for (u = 0; u < f->shared->mtab.nmounts; u++) {
        
        if (f->shared->mtab.child[u].file->parent == f) {
            
            if (H5G_get_shared_count(f->shared->mtab.child[u].group) > 1)
                *nopen_objs += 1;

            H5F__mount_count_ids_recurse(f->shared->mtab.child[u].file, nopen_files, nopen_objs);
        }
    }

    FUNC_LEAVE_NOAPI_VOID
} 

herr_t
H5F__mount_count_ids(H5F_t *f, unsigned *nopen_files, unsigned *nopen_objs)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(nopen_files);
    assert(nopen_objs);

    
    while (f->parent)
        f = f->parent;

    
    H5F__mount_count_ids_recurse(f, nopen_files, nopen_objs);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5F__flush_mounts_recurse(H5F_t *f)
{
    unsigned nerrors = 0;         
    unsigned u;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);

    
    for (u = 0; u < f->shared->mtab.nmounts; u++)
        if (H5F__flush_mounts_recurse(f->shared->mtab.child[u].file) < 0)
            nerrors++;

    
    if (H5F__flush(f) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTFLUSH, FAIL, "unable to flush file's cached information");

    
    if (nerrors)
        HGOTO_ERROR(H5E_FILE, H5E_CANTFLUSH, FAIL, "unable to flush file's child mounts");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F_flush_mounts(H5F_t *f)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);

    
    while (f->parent)
        f = f->parent;

    
    if (H5F__flush_mounts_recurse(f) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTFLUSH, FAIL, "unable to flush mounted file hierarchy");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F_traverse_mount(H5O_loc_t *oloc )
{
    H5F_t *parent = oloc->file, 
        *child    = NULL;       
    unsigned   lt, rt, md = 0;  
    int        cmp;
    H5O_loc_t *mnt_oloc  = NULL;    
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(oloc);

    
    do {
        
        lt  = 0;
        rt  = parent->shared->mtab.nmounts;
        cmp = -1;
        while (lt < rt && cmp) {
            md       = (lt + rt) / 2;
            mnt_oloc = H5G_oloc(parent->shared->mtab.child[md].group);
            cmp      = H5_addr_cmp(oloc->addr, mnt_oloc->addr);
            if (cmp < 0)
                rt = md;
            else
                lt = md + 1;
        }

        
        if (0 == cmp) {
            
            child = parent->shared->mtab.child[md].file;

            
            mnt_oloc = H5G_oloc(child->shared->root_grp);

            
            if (H5O_loc_free(oloc) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTFREE, FAIL, "unable to free object location");

            
            if (H5O_loc_copy_deep(oloc, mnt_oloc) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTCOPY, FAIL, "unable to copy object location");

            
            oloc->file = child;

            
            parent = child;
        } 
    } while (!cmp);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
