/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5Opkg.h"      
#include "H5SMprivate.h" 

typedef struct {
    H5F_t         *f;        
    int            sequence; 
    unsigned       nfailed;  
    H5O_operator_t op;       
    void          *op_data;  
    bool           adj_link; 
} H5O_iter_rm_t;

static herr_t H5O__msg_reset_real(const H5O_msg_class_t *type, void *native);
static herr_t H5O__msg_remove_cb(H5O_t *oh, H5O_mesg_t *mesg , unsigned sequence,
                                 void *_udata );
static herr_t H5O__copy_mesg(H5F_t *f, H5O_t *oh, size_t idx, const H5O_msg_class_t *type, const void *mesg,
                             unsigned mesg_flags, unsigned update_flags);

herr_t
H5O_msg_create(const H5O_loc_t *loc, unsigned type_id, unsigned mesg_flags, unsigned update_flags, void *mesg)
{
    H5O_t *oh        = NULL;    
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(loc);
    assert(type_id < NELMTS(H5O_msg_class_g));
    assert(0 == (mesg_flags & ~H5O_MSG_FLAG_BITS));
    assert(mesg);

    
    if (NULL == (oh = H5O_pin(loc)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPIN, FAIL, "unable to pin object header");

    
    if (H5O_msg_append_oh(loc->file, oh, type_id, mesg_flags, update_flags, mesg) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_WRITEERROR, FAIL, "unable to append to object header");

done:
    if (oh && H5O_unpin(oh) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPIN, FAIL, "unable to unpin object header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O_msg_append_oh(H5F_t *f, H5O_t *oh, unsigned type_id, unsigned mesg_flags, unsigned update_flags,
                  void *mesg)
{
    const H5O_msg_class_t *type;                
    herr_t                 ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(oh);
    assert(H5O_ATTR_ID != type_id); 
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);
    assert(0 == (mesg_flags & ~H5O_MSG_FLAG_BITS));
    assert(mesg);

    
    if (H5O__msg_append_real(f, oh, type, mesg_flags, update_flags, mesg) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTINSERT, FAIL, "unable to create new message in header");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__msg_append_real(H5F_t *f, H5O_t *oh, const H5O_msg_class_t *type, unsigned mesg_flags,
                     unsigned update_flags, void *mesg)
{
    size_t idx;                 
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(oh);
    assert(type);
    assert(0 == (mesg_flags & ~H5O_MSG_FLAG_BITS));
    assert(mesg);

    
    if (H5O__msg_alloc(f, oh, type, &mesg_flags, mesg, &idx) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_NOSPACE, FAIL, "unable to create new message");

    
    if (H5O__copy_mesg(f, oh, idx, type, mesg, mesg_flags, update_flags) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, FAIL, "unable to write message");
#ifdef H5O_DEBUG
    H5O__assert(oh);
#endif 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O_msg_write(const H5O_loc_t *loc, unsigned type_id, unsigned mesg_flags, unsigned update_flags, void *mesg)
{
    H5O_t                 *oh = NULL;           
    const H5O_msg_class_t *type;                
    herr_t                 ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(loc);
    assert(loc->file);
    assert(H5_addr_defined(loc->addr));
    assert(H5O_ATTR_ID != type_id); 
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);
    assert(mesg);
    assert(0 == (mesg_flags & ~H5O_MSG_FLAG_BITS));

    
    if (NULL == (oh = H5O_pin(loc)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPIN, FAIL, "unable to pin object header");

    
    if (H5O__msg_write_real(loc->file, oh, type, mesg_flags, update_flags, mesg) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_WRITEERROR, FAIL, "unable to write object header message");

done:
    if (oh && H5O_unpin(oh) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPIN, FAIL, "unable to unpin object header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O_msg_write_oh(H5F_t *f, H5O_t *oh, unsigned type_id, unsigned mesg_flags, unsigned update_flags,
                 void *mesg)
{
    const H5O_msg_class_t *type;                
    herr_t                 ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_TAG(oh->cache_info.addr, FAIL)

    
    assert(f);
    assert(oh);
    assert(H5O_ATTR_ID != type_id); 
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);
    assert(mesg);
    assert(0 == (mesg_flags & ~H5O_MSG_FLAG_BITS));

    
    if (H5O__msg_write_real(f, oh, type, mesg_flags, update_flags, mesg) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_WRITEERROR, FAIL, "unable to write object header message");

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5O__msg_write_real(H5F_t *f, H5O_t *oh, const H5O_msg_class_t *type, unsigned mesg_flags,
                    unsigned update_flags, void *mesg)
{
    H5O_mesg_t *idx_msg;             
    size_t      idx;                 
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(oh);
    assert(type);
    assert(type != H5O_MSG_ATTR);
    assert(mesg);
    assert(0 == (mesg_flags & ~H5O_MSG_FLAG_BITS));

    
    for (idx = 0, idx_msg = &oh->mesg[0]; idx < oh->nmesgs; idx++, idx_msg++)
        if (type == idx_msg->type)
            break;
    if (idx == oh->nmesgs)
        HGOTO_ERROR(H5E_OHDR, H5E_NOTFOUND, FAIL, "message type not found");

    
    if (!(update_flags & H5O_UPDATE_FORCE) && (idx_msg->flags & H5O_MSG_FLAG_CONSTANT))
        HGOTO_ERROR(H5E_OHDR, H5E_WRITEERROR, FAIL, "unable to modify constant message");
    
    else if ((idx_msg->flags & H5O_MSG_FLAG_SHARED) || (idx_msg->flags & H5O_MSG_FLAG_SHAREABLE)) {
        htri_t status; 

        
        assert(((H5O_shared_t *)idx_msg->native)->type != H5O_SHARE_TYPE_COMMITTED);

        
        assert(!(mesg_flags & H5O_MSG_FLAG_DONTSHARE));

        
        assert(type->share_flags & H5O_SHARE_IS_SHARABLE);

        
        
        if (H5SM_delete(f, oh, (H5O_shared_t *)idx_msg->native) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTDELETE, FAIL, "unable to delete message from SOHM index");

        
        if ((status = H5SM_try_share(f, ((mesg_flags & H5O_MSG_FLAG_SHARED) ? NULL : oh), 0,
                                     idx_msg->type->id, mesg, &mesg_flags)) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_BADMESG, FAIL, "error while trying to share message");
        if (status == false && (mesg_flags & H5O_MSG_FLAG_SHARED))
            HGOTO_ERROR(H5E_OHDR, H5E_BADMESG, FAIL, "message changed sharing status");
    } 

    
    if (H5O__copy_mesg(f, oh, idx, type, mesg, mesg_flags, update_flags) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL, "unable to write message");
#ifdef H5O_DEBUG
    H5O__assert(oh);
#endif 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5O_msg_read(const H5O_loc_t *loc, unsigned type_id, void *mesg)
{
    H5O_t *oh        = NULL; 
    void  *ret_value = NULL; 

    FUNC_ENTER_NOAPI_TAG(loc->addr, NULL)

    
    assert(loc);
    assert(loc->file);
    assert(H5_addr_defined(loc->addr));
    assert(type_id < NELMTS(H5O_msg_class_g));

    
    if (NULL == (oh = H5O_protect(loc, H5AC__READ_ONLY_FLAG, false)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, NULL, "unable to protect object header");

    
    if (NULL == (ret_value = H5O_msg_read_oh(loc->file, oh, type_id, mesg)))
        HGOTO_ERROR(H5E_OHDR, H5E_READERROR, NULL, "unable to read object header message");

done:
    if (oh && H5O_unprotect(loc, oh, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, NULL, "unable to release object header");

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

void *
H5O_msg_read_oh(H5F_t *f, H5O_t *oh, unsigned type_id, void *mesg)
{
    const H5O_msg_class_t *type; 
    unsigned               idx;  
    void                  *ret_value = NULL;

    FUNC_ENTER_NOAPI_NOINIT

    
    assert(f);
    assert(oh);
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);

    
    for (idx = 0; idx < oh->nmesgs; idx++)
        if (type == oh->mesg[idx].type)
            break;
    if (idx == oh->nmesgs)
        HGOTO_ERROR(H5E_OHDR, H5E_NOTFOUND, NULL, "message type not found");

    
    H5O_LOAD_NATIVE(f, 0, oh, &(oh->mesg[idx]), NULL)

    
    if (NULL == (ret_value = (type->copy)(oh->mesg[idx].native, mesg)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, NULL, "unable to copy message to user space");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O_msg_reset(unsigned type_id, void *native)
{
    const H5O_msg_class_t *type;                
    herr_t                 ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);

    
    if (H5O__msg_reset_real(type, native) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTRESET, FAIL, "unable to reset object header");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__msg_reset_real(const H5O_msg_class_t *type, void *native)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(type);

    if (native) {
        if (type->reset) {
            if ((type->reset)(native) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTRELEASE, FAIL, "reset method failed");
        } 
        else
            memset(native, 0, type->native_size);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5O_msg_free(unsigned type_id, void *mesg)
{
    const H5O_msg_class_t *type;             
    void                  *ret_value = NULL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);

    
    ret_value = H5O_msg_free_real(type, mesg);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__msg_free_mesg(H5O_mesg_t *mesg)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(mesg);

    
    mesg->native = H5O_msg_free_real(mesg->type, mesg->native);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

void *
H5O_msg_free_real(const H5O_msg_class_t *type, void *msg_native)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    

    if (msg_native) {
        H5O__msg_reset_real(type, msg_native);
        if (type && type->free)
            (type->free)(msg_native);
        else
            H5MM_xfree(msg_native);
    }

    FUNC_LEAVE_NOAPI(NULL)
} 

void *
H5O_msg_copy(unsigned type_id, const void *mesg, void *dst)
{
    const H5O_msg_class_t *type;             
    void                  *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(mesg);
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);

    
    if (NULL == (ret_value = (type->copy)(mesg, dst)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, NULL, "unable to copy object header message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5O_msg_count(const H5O_loc_t *loc, unsigned type_id)
{
    H5O_t                 *oh = NULL;        
    const H5O_msg_class_t *type;             
    unsigned               msg_count;        
    int                    ret_value = FAIL; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(loc);
    assert(loc->file);
    assert(H5_addr_defined(loc->addr));
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);

    
    if (NULL == (oh = H5O_protect(loc, H5AC__READ_ONLY_FLAG, false)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to protect object header");

    
    msg_count = H5O__msg_count_real(oh, type);
    H5_CHECKED_ASSIGN(ret_value, int, msg_count, unsigned);

done:
    if (oh && H5O_unprotect(loc, oh, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to release object header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

unsigned
H5O__msg_count_real(const H5O_t *oh, const H5O_msg_class_t *type)
{
    unsigned u;             
    unsigned ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(oh);
    assert(type);

    
    for (u = ret_value = 0; u < oh->nmesgs; u++)
        if (oh->mesg[u].type == type)
            ret_value++;

    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5O_msg_exists(const H5O_loc_t *loc, unsigned type_id)
{
    H5O_t *oh        = NULL; 
    htri_t ret_value = FAIL; 

    FUNC_ENTER_NOAPI_TAG(loc->addr, FAIL)

    assert(loc);
    assert(loc->file);
    assert(type_id < NELMTS(H5O_msg_class_g));

    
    if (NULL == (oh = H5O_protect(loc, H5AC__READ_ONLY_FLAG, false)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to protect object header");

    
    if ((ret_value = H5O_msg_exists_oh(oh, type_id)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_READERROR, FAIL, "unable to verify object header message");

done:
    if (oh && H5O_unprotect(loc, oh, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to release object header");

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

htri_t
H5O_msg_exists_oh(const H5O_t *oh, unsigned type_id)
{
    const H5O_msg_class_t *type;              
    unsigned               u;                 
    htri_t                 ret_value = false; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(oh);
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);

    
    for (u = 0; u < oh->nmesgs; u++)
        if (type == oh->mesg[u].type)
            HGOTO_DONE(true);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O_msg_remove(const H5O_loc_t *loc, unsigned type_id, int sequence, bool adj_link)
{
    H5O_t                 *oh = NULL;        
    const H5O_msg_class_t *type;             
    herr_t                 ret_value = FAIL; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(loc);
    assert(loc->file);
    assert(H5_addr_defined(loc->addr));
    assert(H5O_ATTR_ID != type_id); 
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);

    
    if (NULL == (oh = H5O_pin(loc)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPIN, FAIL, "unable to pin object header");

    
    if ((ret_value = H5O__msg_remove_real(loc->file, oh, type, sequence, NULL, NULL, adj_link)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTDELETE, FAIL, "unable to remove object header message");

done:
    if (oh && H5O_unpin(oh) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPIN, FAIL, "unable to unpin object header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O_msg_remove_op(const H5O_loc_t *loc, unsigned type_id, int sequence, H5O_operator_t op, void *op_data,
                  bool adj_link)
{
    H5O_t                 *oh = NULL;        
    const H5O_msg_class_t *type;             
    herr_t                 ret_value = FAIL; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(loc);
    assert(loc->file);
    assert(H5_addr_defined(loc->addr));
    assert(H5O_ATTR_ID != type_id); 
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);

    
    if (NULL == (oh = H5O_pin(loc)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPIN, FAIL, "unable to pin object header");

    
    if ((ret_value = H5O__msg_remove_real(loc->file, oh, type, sequence, op, op_data, adj_link)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTDELETE, FAIL, "unable to remove object header message");

done:
    if (oh && H5O_unpin(oh) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPIN, FAIL, "unable to unpin object header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__msg_remove_cb(H5O_t *oh, H5O_mesg_t *mesg , unsigned sequence, void *_udata )
{
    H5O_iter_rm_t *udata      = (H5O_iter_rm_t *)_udata; 
    htri_t         try_remove = false;                   
    herr_t         ret_value  = H5_ITER_CONT;            

    FUNC_ENTER_PACKAGE

    
    assert(mesg);

    
    if (udata->op) {
        
        if ((try_remove = (udata->op)(mesg->native, sequence, udata->op_data)) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTDELETE, H5_ITER_ERROR,
                        "object header message deletion callback failed");
    } 
    else {
        
        if ((int)sequence == udata->sequence || H5O_ALL == udata->sequence)
            try_remove = true;
    } 

    
    if (try_remove) {
        
        
        if (H5O__release_mesg(udata->f, oh, mesg, udata->adj_link) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTDELETE, H5_ITER_ERROR, "unable to release message");

        
        oh->mesgs_modified = H5O_MODIFY_CONDENSE;

        
        if (udata->sequence == H5O_FIRST || udata->sequence != H5O_ALL)
            HGOTO_DONE(H5_ITER_STOP);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__msg_remove_real(H5F_t *f, H5O_t *oh, const H5O_msg_class_t *type, int sequence, H5O_operator_t app_op,
                     void *op_data, bool adj_link)
{
    H5O_iter_rm_t       udata;               
    H5O_mesg_operator_t op;                  
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(oh);
    assert(type);

    
    if (0 == (H5F_INTENT(f) & H5F_ACC_RDWR))
        HGOTO_ERROR(H5E_OHDR, H5E_WRITEERROR, FAIL, "no write intent on file");

    
    udata.f        = f;
    udata.sequence = sequence;
    udata.nfailed  = 0;
    udata.op       = app_op;
    udata.op_data  = op_data;
    udata.adj_link = adj_link;

    
    op.op_type  = H5O_MESG_OP_LIB;
    op.u.lib_op = H5O__msg_remove_cb;
    if (H5O__msg_iterate_real(f, oh, type, &op, &udata) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_NOTFOUND, FAIL, "error iterating over messages");

    
    if (udata.nfailed)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL, "unable to remove constant message(s)");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O_msg_iterate(const H5O_loc_t *loc, unsigned type_id, const H5O_mesg_operator_t *op, void *op_data)
{
    H5O_t                 *oh = NULL;        
    const H5O_msg_class_t *type;             
    herr_t                 ret_value = FAIL; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(loc);
    assert(loc->file);
    assert(H5_addr_defined(loc->addr));
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);
    assert(op);

    
    if (NULL == (oh = H5O_protect(loc, H5AC__READ_ONLY_FLAG, false)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to protect object header");

    
    if ((ret_value = H5O__msg_iterate_real(loc->file, oh, type, op, op_data)) < 0)
        HERROR(H5E_OHDR, H5E_BADITER, "unable to iterate over object header messages");

done:
    if (oh && H5O_unprotect(loc, oh, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to release object header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__msg_iterate_real(H5F_t *f, H5O_t *oh, const H5O_msg_class_t *type, const H5O_mesg_operator_t *op,
                      void *op_data)
{
    H5O_mesg_t *idx_msg;                  
    unsigned    idx;                      
    unsigned    sequence;                 
    herr_t      ret_value = H5_ITER_CONT; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(oh);
    assert(type);
    assert(op);
    assert(op->u.app_op);

    
    oh->recursion_level++;

    
    for (sequence = 0, idx = 0, idx_msg = &oh->mesg[0]; idx < oh->nmesgs && !ret_value; idx++, idx_msg++) {
        if (type == idx_msg->type) {
            
            H5O_LOAD_NATIVE(f, 0, oh, idx_msg, FAIL)

            
            if (op->op_type == H5O_MESG_OP_LIB)
                ret_value = (op->u.lib_op)(oh, idx_msg, sequence, op_data);
            else
                ret_value = (op->u.app_op)(idx_msg->native, sequence, op_data);

            
            if (ret_value != 0)
                break;

            
            sequence++;
        } 
    }     

    
    if (ret_value < 0)
        HERROR(H5E_OHDR, H5E_CANTLIST, "iterator function failed");

done:
    
    oh->recursion_level--;

    
    if (oh->recursion_level == 0 && oh->mesgs_modified) {
        
        
        if (oh->mesgs_modified & H5O_MODIFY_CONDENSE)
            if (H5O__condense_header(f, oh) < 0)
                HDONE_ERROR(H5E_OHDR, H5E_CANTPACK, FAIL, "can't pack object header");

        
        if (H5O_touch_oh(f, oh, false) < 0)
            HDONE_ERROR(H5E_OHDR, H5E_CANTUPDATE, FAIL, "unable to update time on object");

        
        if (H5AC_mark_entry_dirty(oh) < 0)
            HDONE_ERROR(H5E_OHDR, H5E_CANTMARKDIRTY, FAIL, "unable to mark object header as dirty");

        
        oh->mesgs_modified = 0;
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

size_t
H5O_msg_raw_size(const H5F_t *f, unsigned type_id, bool disable_shared, const void *mesg)
{
    const H5O_msg_class_t *type;          
    size_t                 ret_value = 0; 

    FUNC_ENTER_NOAPI(0)

    
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);
    assert(type->raw_size);
    assert(f);
    assert(mesg);

    
    if (0 == (ret_value = (type->raw_size)(f, disable_shared, mesg)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCOUNT, 0, "unable to determine size of message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

size_t
H5O_msg_size_f(const H5F_t *f, hid_t ocpl_id, unsigned type_id, const void *mesg, size_t extra_raw)
{
    const H5O_msg_class_t *type;          
    H5P_genplist_t        *ocpl;          
    uint8_t                oh_flags;      
    size_t                 ret_value = 0; 

    FUNC_ENTER_NOAPI(0)

    
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);
    assert(type->raw_size);
    assert(f);
    assert(mesg);

    
    if (NULL == (ocpl = (H5P_genplist_t *)H5I_object(ocpl_id)))
        HGOTO_ERROR(H5E_PLIST, H5E_BADTYPE, 0, "not a property list");

    
    if (H5P_get(ocpl, H5O_CRT_OHDR_FLAGS_NAME, &oh_flags) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, 0, "can't get object header flags");

    
    if ((ret_value = (type->raw_size)(f, false, mesg)) == 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCOUNT, 0, "unable to determine size of message");

    
    ret_value += extra_raw;

    
    ret_value = (size_t)H5O_ALIGN_F(f, ret_value);

    
    ret_value += (size_t)H5O_SIZEOF_MSGHDR_F(
        f, (H5F_STORE_MSG_CRT_IDX(f) || oh_flags & H5O_HDR_ATTR_CRT_ORDER_TRACKED));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

size_t
H5O_msg_size_oh(const H5F_t *f, const H5O_t *oh, unsigned type_id, const void *mesg, size_t extra_raw)
{
    const H5O_msg_class_t *type;          
    size_t                 ret_value = 0; 

    FUNC_ENTER_NOAPI(0)

    
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);
    assert(type->raw_size);
    assert(f);
    assert(mesg);

    
    if ((ret_value = (type->raw_size)(f, false, mesg)) == 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCOUNT, 0, "unable to determine size of message");

    
    ret_value += extra_raw;

    
    ret_value = (size_t)H5O_ALIGN_OH(oh, ret_value);

    
    ret_value += (size_t)H5O_SIZEOF_MSGHDR_OH(oh);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5O_msg_can_share(unsigned type_id, const void *mesg)
{
    const H5O_msg_class_t *type;             
    htri_t                 ret_value = FAIL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);
    assert(mesg);

    
    if (type->can_share)
        ret_value = (type->can_share)(mesg);
    else {
        
        ret_value = (type->share_flags & H5O_SHARE_IS_SHARABLE) ? true : false;
    } 

    
    assert((type->post_copy_file && type->copy_file) || ret_value == false);

    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5O_msg_can_share_in_ohdr(unsigned type_id)
{
    const H5O_msg_class_t *type;             
    htri_t                 ret_value = FAIL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);

    
    ret_value = (type->share_flags & H5O_SHARE_IN_OHDR) ? true : false;

    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5O_msg_is_shared(unsigned type_id, const void *mesg)
{
    const H5O_msg_class_t *type;             
    htri_t                 ret_value = FAIL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
#ifdef H5O_ENABLE_BOGUS
    if (type_id >= NELMTS(H5O_msg_class_g))
        ret_value = false;
    else
#endif 
    {
        assert(type_id < NELMTS(H5O_msg_class_g));
        type = H5O_msg_class_g[type_id]; 
        assert(type);
        assert(mesg);

        
        if (type->share_flags & H5O_SHARE_IS_SHARABLE)
            ret_value = H5O_IS_STORED_SHARED(((const H5O_shared_t *)mesg)->type);
        else
            ret_value = false;
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O_msg_set_share(unsigned type_id, const H5O_shared_t *share, void *mesg)
{
    const H5O_msg_class_t *type;                
    herr_t                 ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);
    assert(type->share_flags & H5O_SHARE_IS_SHARABLE);
    assert(mesg);
    assert(share);
    assert(share->type != H5O_SHARE_TYPE_UNSHARED);

    
    if (type->set_share) {
        if ((type->set_share)(mesg, share) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "unable to set shared message information");
    } 
    else {
        
        if (H5O_set_shared((H5O_shared_t *)mesg, share) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "unable to set shared message information");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O_msg_reset_share(unsigned H5_ATTR_NDEBUG_UNUSED type_id, void *mesg)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(type_id < NELMTS(H5O_msg_class_g));
    assert(H5O_msg_class_g[type_id]); 
    assert(H5O_msg_class_g[type_id]->share_flags & H5O_SHARE_IS_SHARABLE);
    assert(mesg);

    
    memset((H5O_shared_t *)mesg, 0, sizeof(H5O_shared_t));

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5O_msg_get_crt_index(unsigned type_id, const void *mesg, H5O_msg_crt_idx_t *crt_idx)
{
    const H5O_msg_class_t *type; 
    herr_t                 ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);
    assert(mesg);
    assert(crt_idx);

    
    if (type->get_crt_index) {
        
        if ((type->get_crt_index)(mesg, crt_idx) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "unable to retrieve creation index");
    } 
    else
        *crt_idx = 0;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O_msg_encode(H5F_t *f, unsigned type_id, bool disable_shared, unsigned char *buf, const void *mesg)
{
    const H5O_msg_class_t *type;                
    herr_t                 ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);

    
    if ((type->encode)(f, disable_shared, SIZE_MAX, buf, mesg) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTENCODE, FAIL, "unable to encode message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5O_msg_decode(H5F_t *f, H5O_t *open_oh, unsigned type_id, size_t buf_size, const unsigned char *buf)
{
    const H5O_msg_class_t *type;             
    unsigned               ioflags   = 0;    
    void                  *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(f);
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);

    
    if (NULL == (ret_value = (type->decode)(f, open_oh, 0, &ioflags, buf_size, buf)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTDECODE, NULL, "unable to decode message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5O__msg_copy_file(const H5O_msg_class_t *type, H5F_t *file_src, void *native_src, H5F_t *file_dst,
                   bool *recompute_size, unsigned *mesg_flags, H5O_copy_t *cpy_info, void *udata)
{
    void *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(type);
    assert(type->copy_file);
    assert(file_src);
    assert(native_src);
    assert(file_dst);
    assert(recompute_size);
    assert(cpy_info);

    
    if (NULL == (ret_value = (type->copy_file)(file_src, native_src, file_dst, recompute_size, mesg_flags,
                                               cpy_info, udata)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, NULL, "unable to copy object header message to file");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__msg_alloc(H5F_t *f, H5O_t *oh, const H5O_msg_class_t *type, unsigned *mesg_flags, void *native,
               size_t *mesg_idx)
{
    size_t new_idx;             
    htri_t shared_mesg;         
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(oh);
    assert(mesg_flags);
    assert(!(*mesg_flags & H5O_MSG_FLAG_SHARED));
    assert(type);
    assert(native);
    assert(mesg_idx);

    
    if ((shared_mesg = H5O_msg_is_shared(type->id, native)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "error determining if message is shared");
    else if (shared_mesg > 0) {
        
        if (type->link && (type->link)(f, oh, native) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_LINKCOUNT, FAIL, "unable to adjust shared message ref count");
        *mesg_flags |= H5O_MSG_FLAG_SHARED;
    } 
    else {
        
        if (H5SM_try_share(f, oh, 0, type->id, native, mesg_flags) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_WRITEERROR, FAIL, "error determining if message should be shared");
    } 

    
    if (H5O__alloc(f, oh, type, native, &new_idx) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL, "unable to allocate space for message");

    
    if (type->get_crt_index) {
        
        if ((type->get_crt_index)(native, &oh->mesg[new_idx].crt_idx) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "unable to retrieve creation index");
    } 

    
    *mesg_idx = new_idx;

done:
    
    if (oh->recursion_level == 0) {
        for (unsigned u = 0; oh->num_deleted_mesgs > 0 && u < oh->nmesgs;)
            if (oh->mesg[u].type->id == H5O_DELETED_ID) {
                
                assert(u != *mesg_idx);
                if (*mesg_idx > u)
                    (*mesg_idx)--;

                
                if (u < (oh->nmesgs - 1))
                    memmove(&oh->mesg[u], &oh->mesg[u + 1], ((oh->nmesgs - 1) - u) * sizeof(H5O_mesg_t));
                oh->nmesgs--;
                oh->num_deleted_mesgs--;
            }
            else
                u++;
        assert(oh->num_deleted_mesgs == 0);
    }
    else
        
        oh->mesgs_modified = H5O_MODIFY_CONDENSE;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__copy_mesg(H5F_t *f, H5O_t *oh, size_t idx, const H5O_msg_class_t *type, const void *mesg,
               unsigned mesg_flags, unsigned update_flags)
{
    H5O_chunk_proxy_t *chk_proxy   = NULL;           
    H5O_mesg_t        *idx_msg     = &oh->mesg[idx]; 
    bool               chk_dirtied = false;          
    herr_t             ret_value   = SUCCEED;        

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(oh);
    assert(type);
    assert(type->copy);
    assert(mesg);

    
    if (NULL == (chk_proxy = H5O__chunk_protect(f, oh, idx_msg->chunkno)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to protect object header chunk");

    
    H5O__msg_reset_real(type, idx_msg->native);

    
    if (NULL == (idx_msg->native = (type->copy)(mesg, idx_msg->native)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL, "unable to copy message to object header");

    
    idx_msg->flags = (uint8_t)mesg_flags;

    
    idx_msg->dirty = true;
    chk_dirtied    = true;

    
    if (H5O__chunk_unprotect(f, chk_proxy, chk_dirtied) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to release object header chunk");
    chk_proxy = NULL;

    
    if (update_flags & H5O_UPDATE_TIME)
        if (H5O_touch_oh(f, oh, false) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTUPDATE, FAIL, "unable to update time on object");

done:
    
    if (chk_proxy && H5O__chunk_unprotect(f, chk_proxy, chk_dirtied) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to release object header chunk");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O_msg_delete(H5F_t *f, H5O_t *open_oh, unsigned type_id, void *mesg)
{
    const H5O_msg_class_t *type;                
    herr_t                 ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);

    
    if ((type->del) && (type->del)(f, open_oh, mesg) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTDELETE, FAIL, "unable to delete file space for object header message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__delete_mesg(H5F_t *f, H5O_t *oh, H5O_mesg_t *mesg)
{
    const H5O_msg_class_t *type      = mesg->type; 
    herr_t                 ret_value = SUCCEED;    

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(mesg);
    assert(oh);

    
    if (type->del) {
        
        H5O_LOAD_NATIVE(f, H5O_DECODEIO_NOCHANGE, oh, mesg, FAIL)

        if ((type->del)(f, oh, mesg->native) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTDELETE, FAIL,
                        "unable to delete file space for object header message");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O_msg_flush(H5F_t *f, H5O_t *oh, H5O_mesg_t *mesg)
{
    uint8_t *p;                   
    unsigned msg_id;              
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(oh);

    
    p = mesg->raw - H5O_SIZEOF_MSGHDR_OH(oh);

    
    if (mesg->type == H5O_MSG_UNKNOWN)
        msg_id = *(H5O_unknown_t *)(mesg->native);
    else
        msg_id = (uint8_t)mesg->type->id;

    
    if (oh->version == H5O_VERSION_1)
        UINT16ENCODE(p, msg_id);
    else
        *p++ = (uint8_t)msg_id;
    assert(mesg->raw_size < H5O_MESG_MAX_SIZE);
    UINT16ENCODE(p, mesg->raw_size);
    *p++ = mesg->flags;

    
    if (oh->version == H5O_VERSION_1) {
        *p++ = 0; 
        *p++ = 0; 
        *p++ = 0; 
    }             
    
    else {
        
        if (oh->flags & H5O_HDR_ATTR_CRT_ORDER_TRACKED)
            UINT16ENCODE(p, mesg->crt_idx);
    } 
    assert(p == mesg->raw);

#ifndef NDEBUG
    
    if (H5O_NULL_ID == msg_id)
        assert(oh->chunk[mesg->chunkno].gap == 0);
    else {
        
        assert(H5O_DELETED_ID != msg_id);

        
        assert(mesg->native);
    }
#endif 

    
    if (mesg->native && mesg->type != H5O_MSG_UNKNOWN) {
        
        assert(mesg->raw >= oh->chunk[mesg->chunkno].image);
        assert(mesg->raw_size == H5O_ALIGN_OH(oh, mesg->raw_size));
        assert(mesg->raw + mesg->raw_size <=
               oh->chunk[mesg->chunkno].image + (oh->chunk[mesg->chunkno].size - H5O_SIZEOF_CHKSUM_OH(oh)));
        assert(msg_id != H5O_DELETED_ID);

#ifndef NDEBUG
        
        {
            size_t msg_size;

            msg_size = mesg->type->raw_size(f, false, mesg->native);
            msg_size = H5O_ALIGN_OH(oh, msg_size);
            assert(msg_size <= mesg->raw_size);
        }
#endif 

        assert(mesg->type->encode);
        if ((mesg->type->encode)(f, false, mesg->raw_size, mesg->raw, mesg->native) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTENCODE, FAIL, "unable to encode object header message");
    } 

    
    mesg->dirty = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__flush_msgs(H5F_t *f, H5O_t *oh)
{
    H5O_mesg_t *curr_msg;            
    unsigned    u;                   
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(oh);

    
    for (u = 0, curr_msg = &oh->mesg[0]; u < oh->nmesgs; u++, curr_msg++)
        if (curr_msg->dirty)
            if (H5O_msg_flush(f, oh, curr_msg) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTENCODE, FAIL, "unable to encode object header message");

    
    if (oh->nmesgs != u)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTFLUSH, FAIL, "corrupt object header - too few messages");

#ifndef NDEBUG
    
    oh->ndecode_dirtied = 0;
#endif 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O_msg_get_flags(const H5O_loc_t *loc, unsigned type_id, uint8_t *flags)
{
    H5O_t                 *oh = NULL;           
    const H5O_msg_class_t *type;                
    H5O_mesg_t            *idx_msg;             
    unsigned               idx;                 
    herr_t                 ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(loc);
    assert(loc->file);
    assert(H5_addr_defined(loc->addr));
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);

    
    if (NULL == (oh = H5O_protect(loc, H5AC__READ_ONLY_FLAG, false)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to protect object header");

    
    for (idx = 0, idx_msg = &oh->mesg[0]; idx < oh->nmesgs; idx++, idx_msg++)
        if (type == idx_msg->type)
            break;

    if (idx == oh->nmesgs)
        HGOTO_ERROR(H5E_OHDR, H5E_NOTFOUND, FAIL, "message type not found");

    
    *flags = idx_msg->flags;

done:
    if (oh && H5O_unprotect(loc, oh, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to release object header");

    FUNC_LEAVE_NOAPI(ret_value)
} 
