/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Pmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5Iprivate.h"  
#include "H5Ppkg.h"      

typedef struct {
    H5P_iterate_t iter_func; 
    hid_t         id;        
    void         *iter_data; 
} H5P_iter_ud_t;

bool H5_PKG_INIT_VAR = false;

hid_t
H5Pcopy(hid_t id)
{
    void *obj;                         
    hid_t ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    if (H5P_DEFAULT == id)
        HGOTO_DONE(H5P_DEFAULT);

    
    if (H5I_GENPROP_LST != H5I_get_type(id) && H5I_GENPROP_CLS != H5I_get_type(id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not property object");
    if (NULL == (obj = H5I_object(id)))
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, H5I_INVALID_HID, "property object doesn't exist");

    
    if (H5I_GENPROP_LST == H5I_get_type(id)) {
        if ((ret_value = H5P_copy_plist((H5P_genplist_t *)obj, true)) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, H5I_INVALID_HID, "can't copy property list");
    } 
    
    else {
        H5P_genclass_t *copy_class; 

        
        if ((copy_class = H5P__copy_pclass((H5P_genclass_t *)obj)) == NULL)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, H5I_INVALID_HID, "can't copy property class");

        
        if ((ret_value = H5I_register(H5I_GENPROP_CLS, copy_class, true)) < 0) {
            H5P__close_class(copy_class);
            HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, H5I_INVALID_HID,
                        "unable to register property list class");
        } 
    }     

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Pcreate_class(hid_t parent, const char *name, H5P_cls_create_func_t cls_create, void *create_data,
                H5P_cls_copy_func_t cls_copy, void *copy_data, H5P_cls_close_func_t cls_close,
                void *close_data)
{
    H5P_genclass_t *par_class = NULL;            
    H5P_genclass_t *pclass    = NULL;            
    hid_t           ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (H5P_DEFAULT != parent && (H5I_GENPROP_CLS != H5I_get_type(parent)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not a property list class");
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "invalid class name");
    if ((create_data != NULL && cls_create == NULL) || (copy_data != NULL && cls_copy == NULL) ||
        (close_data != NULL && cls_close == NULL))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "data specified, but no callback provided");

    
    if (parent == H5P_DEFAULT)
        par_class = NULL;
    else if (NULL == (par_class = (H5P_genclass_t *)H5I_object(parent)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "can't retrieve parent class");

    
    if (NULL == (pclass = H5P__create_class(par_class, name, H5P_TYPE_USER, cls_create, create_data, cls_copy,
                                            copy_data, cls_close, close_data)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCREATE, H5I_INVALID_HID, "unable to create property list class");

    
    if ((ret_value = H5I_register(H5I_GENPROP_CLS, pclass, true)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register property list class");

done:
    if (H5I_INVALID_HID == ret_value && pclass)
        H5P__close_class(pclass);

    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Pcreate(hid_t cls_id)
{
    H5P_genclass_t *pclass;                      
    hid_t           ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (pclass = (H5P_genclass_t *)H5I_object_verify(cls_id, H5I_GENPROP_CLS)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not a property list class");

    
    if ((ret_value = H5P_create_id(pclass, true)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCREATE, H5I_INVALID_HID, "unable to create property list");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pregister2(hid_t cls_id, const char *name, size_t size, void *def_value, H5P_prp_create_func_t prp_create,
             H5P_prp_set_func_t prp_set, H5P_prp_get_func_t prp_get, H5P_prp_delete_func_t prp_delete,
             H5P_prp_copy_func_t prp_copy, H5P_prp_compare_func_t prp_cmp, H5P_prp_close_func_t prp_close)
{
    H5P_genclass_t *pclass;      
    H5P_genclass_t *orig_pclass; 
    herr_t          ret_value;   

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (pclass = (H5P_genclass_t *)H5I_object_verify(cls_id, H5I_GENPROP_CLS)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list class");
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid class name");
    if (size > 0 && def_value == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "properties >0 size must have default");

    
    orig_pclass = pclass;
    if ((ret_value = H5P__register(&pclass, name, size, def_value, prp_create, prp_set, prp_get, NULL, NULL,
                                   prp_delete, prp_copy, prp_cmp, prp_close)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL, "unable to register property in class");

    
    if (pclass != orig_pclass) {
        H5P_genclass_t *old_pclass; 

        
        if (NULL == (old_pclass = (H5P_genclass_t *)H5I_subst(cls_id, pclass)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to substitute property class in ID");
        assert(old_pclass == orig_pclass);

        
        if (H5P__close_class(old_pclass) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTCLOSEOBJ, FAIL,
                        "unable to close original property class after substitution");
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pinsert2(hid_t plist_id, const char *name, size_t size, void *value, H5P_prp_set_func_t prp_set,
           H5P_prp_get_func_t prp_get, H5P_prp_delete_func_t prp_delete, H5P_prp_copy_func_t prp_copy,
           H5P_prp_compare_func_t prp_cmp, H5P_prp_close_func_t prp_close)
{
    H5P_genplist_t *plist;     
    herr_t          ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object_verify(plist_id, H5I_GENPROP_LST)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list");
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid property name");
    if (size > 0 && value == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "properties >0 size must have default");

    
    if ((ret_value = H5P_insert(plist, name, size, value, prp_set, prp_get, NULL, NULL, prp_delete, prp_copy,
                                prp_cmp, prp_close)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL, "unable to register property in plist");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset(hid_t plist_id, const char *name, const void *value)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object_verify(plist_id, H5I_GENPROP_LST)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list");
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid property name");
    if (value == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid property value");

    
    if (H5P_set(plist, name, value) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL, "unable to set value in plist");

done:
    FUNC_LEAVE_API(ret_value)
} 

htri_t
H5Pexist(hid_t id, const char *name)
{
    H5P_genplist_t *plist;     
    H5P_genclass_t *pclass;    
    htri_t          ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (H5I_GENPROP_LST != H5I_get_type(id) && H5I_GENPROP_CLS != H5I_get_type(id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property object");
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid property name");

    
    if (H5I_GENPROP_LST == H5I_get_type(id)) {
        if (NULL == (plist = (H5P_genplist_t *)H5I_object(id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list");
        if ((ret_value = H5P_exist_plist(plist, name)) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL, "property does not exist in list");
    } 
    else if (H5I_GENPROP_CLS == H5I_get_type(id)) {
        if (NULL == (pclass = (H5P_genclass_t *)H5I_object(id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property class");
        if ((ret_value = H5P__exist_pclass(pclass, name)) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL, "property does not exist in class");
    } 
    else
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property object");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_size(hid_t id, const char *name, size_t *size )
{
    H5P_genclass_t *pclass;    
    H5P_genplist_t *plist;     
    herr_t          ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (H5I_GENPROP_LST != H5I_get_type(id) && H5I_GENPROP_CLS != H5I_get_type(id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property object");
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid property name");
    if (size == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid property size");

    if (H5I_GENPROP_LST == H5I_get_type(id)) {
        if (NULL == (plist = (H5P_genplist_t *)H5I_object(id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list");

        
        if ((ret_value = H5P__get_size_plist(plist, name, size)) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL, "unable to query size in plist");
    } 
    else if (H5I_GENPROP_CLS == H5I_get_type(id)) {
        if (NULL == (pclass = (H5P_genclass_t *)H5I_object(id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list");

        
        if ((ret_value = H5P__get_size_pclass(pclass, name, size)) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL, "unable to query size in plist");
    } 
    else
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property object");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pencode2(hid_t plist_id, void *buf, size_t *nalloc, hid_t fapl_id)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object_verify(plist_id, H5I_GENPROP_LST)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list");

    
    if (H5CX_set_apl(&fapl_id, H5P_CLS_FACC, H5I_INVALID_HID, true) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, H5I_INVALID_HID, "can't set access property list info");

    
    if ((ret_value = H5P__encode(plist, true, buf, nalloc)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTENCODE, FAIL, "unable to encode property list");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Pdecode(const void *buf)
{
    hid_t ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if ((ret_value = H5P__decode(buf)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTDECODE, H5I_INVALID_HID, "unable to decode property list");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Pget_class(hid_t plist_id)
{
    H5P_genplist_t *plist;                       
    H5P_genclass_t *pclass    = NULL;            
    hid_t           ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object_verify(plist_id, H5I_GENPROP_LST)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not a property list");

    
    if ((pclass = H5P_get_class(plist)) == NULL)
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, H5I_INVALID_HID, "unable to query class of property list");

    
    if (H5P__access_class(pclass, H5P_MOD_INC_REF) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, H5I_INVALID_HID, "Can't increment class ID ref count");

    
    if ((ret_value = H5I_register(H5I_GENPROP_CLS, pclass, true)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register property list class");

done:
    if (H5I_INVALID_HID == ret_value && pclass)
        H5P__close_class(pclass);

    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_nprops(hid_t id, size_t *nprops )
{
    H5P_genplist_t *plist;               
    H5P_genclass_t *pclass;              
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5I_GENPROP_LST != H5I_get_type(id) && H5I_GENPROP_CLS != H5I_get_type(id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property object");
    if (nprops == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid property nprops pointer");

    if (H5I_GENPROP_LST == H5I_get_type(id)) {
        if (NULL == (plist = (H5P_genplist_t *)H5I_object(id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list");
        if (H5P__get_nprops_plist(plist, nprops) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL, "unable to query # of properties in plist");
    } 
    else if (H5I_GENPROP_CLS == H5I_get_type(id)) {
        if (NULL == (pclass = (H5P_genclass_t *)H5I_object(id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property class");
        if (H5P_get_nprops_pclass(pclass, nprops, false) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL, "unable to query # of properties in pclass");
    } 
    else
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property object");

done:
    FUNC_LEAVE_API(ret_value)
} 

htri_t
H5Pequal(hid_t id1, hid_t id2)
{
    void  *obj1, *obj2;       
    htri_t ret_value = false; 

    FUNC_ENTER_API(FAIL)

    
    if ((H5I_GENPROP_LST != H5I_get_type(id1) && H5I_GENPROP_CLS != H5I_get_type(id1)) ||
        (H5I_GENPROP_LST != H5I_get_type(id2) && H5I_GENPROP_CLS != H5I_get_type(id2)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not property objects");
    if (H5I_get_type(id1) != H5I_get_type(id2))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not the same kind of property objects");
    if (NULL == (obj1 = H5I_object(id1)) || NULL == (obj2 = H5I_object(id2)))
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, FAIL, "property object doesn't exist");

    
    if (H5I_GENPROP_LST == H5I_get_type(id1)) {
        int cmp_ret = 0;

        if (H5P__cmp_plist((const H5P_genplist_t *)obj1, (const H5P_genplist_t *)obj2, &cmp_ret) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTCOMPARE, FAIL, "can't compare property lists");

        
        ret_value = cmp_ret == 0 ? true : false;
    } 
    
    else {
        if (H5P__cmp_class((const H5P_genclass_t *)obj1, (const H5P_genclass_t *)obj2) == 0)
            ret_value = true;
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

htri_t
H5Pisa_class(hid_t plist_id, hid_t pclass_id)
{
    htri_t ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (H5I_GENPROP_LST != H5I_get_type(plist_id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list");
    if (H5I_GENPROP_CLS != H5I_get_type(pclass_id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property class");

    
    if ((ret_value = H5P_isa_class(plist_id, pclass_id)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL, "unable to compare property list classes");

done:
    FUNC_LEAVE_API(ret_value)
} 

static int
H5P__iterate_cb(H5P_genprop_t *prop, void *_udata)
{
    H5P_iter_ud_t *udata     = (H5P_iter_ud_t *)_udata; 
    int            ret_value = 0;                       

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(prop);
    assert(udata);

    
    ret_value = (*udata->iter_func)(udata->id, prop->name, udata->iter_data);

    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5Piterate(hid_t id, int *idx, H5P_iterate_t iter_func, void *iter_data)
{
    H5P_iter_ud_t udata;        
    int           fake_idx = 0; 
    void         *obj;          
    int           ret_value;    

    FUNC_ENTER_API(FAIL)

    
    if (H5I_GENPROP_LST != H5I_get_type(id) && H5I_GENPROP_CLS != H5I_get_type(id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property object");
    if (NULL == (obj = H5I_object(id)))
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, FAIL, "property object doesn't exist");
    if (iter_func == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid iteration callback");

    
    udata.iter_func = iter_func;
    udata.id        = id;
    udata.iter_data = iter_data;

    if (H5I_GENPROP_LST == H5I_get_type(id)) {
        
        if ((ret_value = H5P__iterate_plist((H5P_genplist_t *)obj, true, (idx ? idx : &fake_idx),
                                            H5P__iterate_cb, &udata)) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL, "unable to iterate over list");
    } 
    else if (H5I_GENPROP_CLS == H5I_get_type(id)) {
        
        if ((ret_value = H5P__iterate_pclass((H5P_genclass_t *)obj, (idx ? idx : &fake_idx), H5P__iterate_cb,
                                             &udata)) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL, "unable to iterate over class");
    } 
    else
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property object");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget(hid_t plist_id, const char *name, void *value )
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object_verify(plist_id, H5I_GENPROP_LST)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list");
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid property name");
    if (value == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid property value");

    
    if (H5P_get(plist, name, value) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "unable to query property value");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Premove(hid_t plist_id, const char *name)
{
    H5P_genplist_t *plist;     
    herr_t          ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object_verify(plist_id, H5I_GENPROP_LST)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list");
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid property name");

    
    if ((ret_value = H5P_remove(plist, name)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTDELETE, FAIL, "unable to remove property");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pcopy_prop(hid_t dst_id, hid_t src_id, const char *name)
{
    H5I_type_t src_id_type, dst_id_type; 
    herr_t     ret_value = SUCCEED;      

    FUNC_ENTER_API(FAIL)

    
    if ((src_id_type = H5I_get_type(src_id)) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid source ID");
    if ((dst_id_type = H5I_get_type(dst_id)) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid destination ID");
    if ((H5I_GENPROP_LST != src_id_type && H5I_GENPROP_CLS != src_id_type) ||
        (H5I_GENPROP_LST != dst_id_type && H5I_GENPROP_CLS != dst_id_type))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not property objects");
    if (src_id_type != dst_id_type)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not the same kind of property objects");
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no name given");

    
    if (H5I_GENPROP_LST == src_id_type) {
        if (H5P__copy_prop_plist(dst_id, src_id, name) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "can't copy property between lists");
    } 
    
    else {
        if (H5P__copy_prop_pclass(dst_id, src_id, name) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "can't copy property between classes");
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Punregister(hid_t pclass_id, const char *name)
{
    H5P_genclass_t *pclass;    
    herr_t          ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (pclass = (H5P_genclass_t *)H5I_object_verify(pclass_id, H5I_GENPROP_CLS)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list class");
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid property name");

    
    if ((ret_value = H5P__unregister(pclass, name)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL, "unable to remove property from class");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pclose(hid_t plist_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5P_DEFAULT != plist_id) {
        
        if (H5I_GENPROP_LST != H5I_get_type(plist_id))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list");

        
        if (H5I_dec_app_ref(plist_id) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTFREE, FAIL, "can't close");
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

H5_ATTR_MALLOC char *
H5Pget_class_name(hid_t pclass_id)
{
    H5P_genclass_t *pclass;    
    char           *ret_value; 

    FUNC_ENTER_API(NULL)

    
    if (NULL == (pclass = (H5P_genclass_t *)H5I_object_verify(pclass_id, H5I_GENPROP_CLS)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a property class");

    
    if ((ret_value = H5P_get_class_name(pclass)) == NULL)
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, NULL, "unable to query name of class");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Pget_class_parent(hid_t pclass_id)
{
    H5P_genclass_t *pclass;                      
    H5P_genclass_t *parent    = NULL;            
    hid_t           ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (pclass = (H5P_genclass_t *)H5I_object_verify(pclass_id, H5I_GENPROP_CLS)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not a property class");

    
    if (NULL == (parent = H5P__get_class_parent(pclass)))
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, H5I_INVALID_HID, "unable to query class of property list");

    
    if (H5P__access_class(parent, H5P_MOD_INC_REF) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, H5I_INVALID_HID, "Can't increment class ID ref count");

    
    if ((ret_value = H5I_register(H5I_GENPROP_CLS, parent, true)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register property list class");

done:
    if (H5I_INVALID_HID == ret_value && parent)
        H5P__close_class(parent);

    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pclose_class(hid_t cls_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5I_GENPROP_CLS != H5I_get_type(cls_id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list class");

    
    if (H5I_dec_app_ref(cls_id) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTFREE, FAIL, "can't close");

done:
    FUNC_LEAVE_API(ret_value)
} 
