% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/intervals-annotation.R
\name{gintervals.annotate}
\alias{gintervals.annotate}
\title{Annotates 1D intervals using nearest neighbors}
\usage{
gintervals.annotate(
  intervals,
  annotation_intervals,
  annotation_columns = NULL,
  column_names = NULL,
  dist_column = "dist",
  max_dist = Inf,
  na_value = NA,
  maxneighbors = 1,
  tie_method = c("first", "min.start", "min.end"),
  overwrite = FALSE,
  keep_order = TRUE,
  intervals.set.out = NULL,
  ...
)
}
\arguments{
\item{intervals}{Intervals to annotate (1D).}

\item{annotation_intervals}{Source intervals containing annotation data (1D).}

\item{annotation_columns}{Character vector of column names to copy from
\code{annotation_intervals}. If \code{NULL} (default), all non-basic
columns are used, i.e. everything beyond the coordinate/strand columns
among: chrom, start, end, chrom1, start1, end1, chrom2, start2, end2, strand.}

\item{column_names}{Optional custom names for the annotation columns. If
provided, must have the same length as \code{annotation_columns}. Defaults
to using the original names.}

\item{dist_column}{Name of the distance column to include. Use \code{NULL} to
omit the distance column. Defaults to "dist".}

\item{max_dist}{Maximum absolute distance. When finite, neighbors with
\code{|dist| > max_dist} result in annotation columns being set to
\code{na_value} for those rows, while the row itself is retained.}

\item{na_value}{Value(s) to use for annotations when beyond \code{max_dist}
or when no neighbor is found. Can be a single scalar recycled for all
columns, or a named list/vector supplying per-column values matching
\code{column_names}.}

\item{maxneighbors}{Maximum number of neighbors per interval (duplicates
intervals as needed). Defaults to 1.}

\item{tie_method}{Tie-breaking when distances are equal: one of
"first" (arbitrary but stable), "min.start" (smaller neighbor start first),
or "min.end" (smaller neighbor end first). Applies when
\code{maxneighbors > 1}.}

\item{overwrite}{When \code{FALSE} (default), errors if selected annotation
columns would overwrite existing columns in \code{intervals}. When
\code{TRUE}, conflicting base columns are replaced by the annotation
columns.}

\item{keep_order}{If \code{TRUE} (default), preserves the original order of
\code{intervals} rows in the output.}

\item{intervals.set.out}{intervals set name where the function result is
optionally outputted}

\item{...}{Additional arguments forwarded to \code{gintervals.neighbors}
(e.g., \code{mindist}, \code{maxdist}).}
}
\value{
A data frame containing the original intervals plus the requested
  annotation columns (and optional distance column). If
  \code{maxneighbors > 1}, rows may be duplicated per input interval to
  accommodate multiple neighbors.
}
\description{
Annotates one-dimensional intervals by finding nearest neighbors in another
set of intervals and adding selected columns from the neighbors to the
original intervals.
}
\details{
The function wraps and extends \code{gintervals.neighbors} to provide
convenient column selection/renaming, optional distance inclusion, distance
thresholding with custom NA values, multiple neighbors per interval, and
deterministic tie-breaking. Currently supports 1D intervals only.


- When \code{annotation_columns = NULL}, all non-basic columns present in
  \code{annotation_intervals} are included.
- Setting \code{dist_column = NULL} omits the distance column.
- If no neighbor is found for an interval, annotation columns are filled with
  \code{na_value} and the distance (when present) is \code{NA_real_}.
- Column name collisions are handled as follows: when \code{overwrite=FALSE}
  a clear error is emitted; when \code{overwrite=TRUE}, base columns with the
  same names are replaced by annotation columns.
}
\examples{
# Prepare toy data
intervs <- gintervals(1, c(1000, 5000), c(1100, 5050))
ann <- gintervals(1, c(900, 5400), c(950, 5500))
ann$remark <- c("a", "b")
ann$score <- c(10, 20)

# Basic usage with default columns (all non-basic columns)
gintervals.annotate(intervs, ann)

# Select specific columns, with custom names and distance column name
gintervals.annotate(
    intervs, ann,
    annotation_columns = c("remark"),
    column_names = c("ann_remark"),
    dist_column = "ann_dist"
)

# Distance threshold with scalar NA replacement
gintervals.annotate(
    intervs, ann,
    annotation_columns = c("remark"),
    max_dist = 200,
    na_value = "no_ann"
)

# Multiple neighbors with deterministic tie-breaking
nbrs <- gintervals.annotate(
    gintervals(1, 1000, 1100),
    {
        x <- gintervals(1, c(800, 1200), c(900, 1300))
        x$label <- c("left", "right")
        x
    },
    annotation_columns = "label",
    maxneighbors = 2,
    tie_method = "min.start"
)
nbrs

# Overwrite existing columns in the base intervals
intervs2 <- intervs
intervs2$remark <- c("orig1", "orig2")
gintervals.annotate(intervs2, ann, annotation_columns = "remark", overwrite = TRUE)
}
