rand.test.aov2 <- 
  function(x, g, b,
           var.equal = FALSE, median.test = FALSE,
           R = 9999, parallel = FALSE, cl = NULL,
           perm.dist = TRUE){
    # Repeated-Measures ANOVA Randomization Tests (Mean/Median)
    # Nathaniel E. Helwig (helwig@umn.edu)
    # last updated: 2026-01-14
    
    
    #########   INITIAL CHECKS   #########
    
    ### check x and g
    x <- as.matrix(x)
    g <- as.factor(g)
    b <- as.factor(b)
    N <- nrow(x)
    nvar <- ncol(x)
    if(length(g) != N) stop("Inputs 'x' and 'g' must satisfy:  nrow(x) == length(g)")
    if(length(b) != N) stop("Inputs 'x' and 'b' must satisfy:  nrow(x) == length(b)")
    
    ### define ngrp, nobs, and y
    ngrp <- nlevels(g)
    nobs <- nlevels(b)
    y <- x
    
    ### check var.equal
    var.equal <- as.logical(var.equal[1])
    
    ### check median.test
    median.test <- as.logical(median.test[1])
    
    ### check R
    R <- as.integer(R)
    if(R < 1) stop("Input 'R' must be a positive integer.")
    
    ### check parallel
    parallel <- as.logical(parallel[1])
    
    ### check 'cl'
    make.cl <- FALSE
    if(parallel){
      if(is.null(cl)){
        make.cl <- TRUE
        cl <- parallel::makeCluster(2L)
      } else {
        if(!any(class(cl) == "cluster")) stop("Input 'cl' must be an object of class 'cluster'.")
      }
    }
    
    ### build design
    if(median.test){
      p <- m <- mtm <- mtmi <- minv <- NULL
    } else {
      y <- scale(y, scale = FALSE)
      x <- model.matrix(~ g, contrasts.arg = list(g = "contr.sum"))[,-1]
      z <- model.matrix(~ b, contrasts.arg = list(b = "contr.sum"))[,-1]
      p <- ncol(x)
      m <- scale(cbind(x, z), scale = FALSE)
      mtm <- crossprod(m)
      mtmi <- psdinv(mtm)
      minv <- tcrossprod(mtmi, m)
    } # if(median.test)
    
    
    #########   ANOVA TEST   #########
    
    ### univariate or multivariate
    if(nvar == 1L){
      
      ## UNIVARIATE TEST
      
      ## vectorize y
      y <- as.numeric(y)
      
      ## observed test statistic
      Tstat <- Tstat.aov2(y = y, g = g, ngrp = ngrp, nobs = nobs, 
                          var.equal = var.equal, median.test = median.test, 
                          p = p, m = m, mtm = mtm, mtmi = mtmi, minv = minv)
      
      ## approximate permutation test (given input R)
      nperm <- R + 1L
      permdist <- rep(0, nperm)
      permdist[1] <- Tstat
      
      ## parallel or sequential computation?
      if(parallel){
        permdist[2:nperm] <- parallel::parSapply(cl = cl, X = integer(R), 
                                                 FUN = Tperm.aov2, 
                                                 y = y, g = g, ngrp = ngrp, nobs = nobs, 
                                                 var.equal = var.equal, median.test = median.test, 
                                                 p = p, m = m, mtm = mtm, mtmi = mtmi, minv = minv)
      } else {
        permdist[2:nperm] <- sapply(X = integer(R),
                                    FUN = Tperm.aov2, 
                                    y = y, g = g, ngrp = ngrp, nobs = nobs, 
                                    var.equal = var.equal, median.test = median.test, 
                                    p = p, m = m, mtm = mtm, mtmi = mtmi, minv = minv)
      } # end if(parallel)
      
      ## permutation p-value (greater than alternative = only option)
      p.value <- mean(permdist >= Tstat)
      
      
    } else {
      
      ## MULTIVARIATE TEST
      
      ## observed test statistic
      Tuni <- Tstat.aov2.mv(y = y, g = g, ngrp = ngrp, nobs = nobs, 
                            var.equal = var.equal, median.test = median.test, 
                            p = p, m = m, mtm = mtm, mtmi = mtmi, minv = minv,
                            combine = FALSE)
      Tstat <- max(Tuni)
      
      ## approximate permutation test (given input R)
      nperm <- R + 1L
      permdist <- rep(0, nperm)
      permdist[1] <- Tstat
      
      ## parallel or sequential computation?
      if(parallel){
        permdist[2:nperm] <- parallel::parSapply(cl = cl, X = integer(R), 
                                                 FUN = Tperm.aov2.mv, 
                                                 y = y, g = g, ngrp = ngrp, 
                                                 nobs = nobs, 
                                                 var.equal = var.equal, 
                                                 median.test = median.test, 
                                                 p = p, m = m, mtm = mtm, 
                                                 mtmi = mtmi, minv = minv)
      } else {
        permdist[2:nperm] <- sapply(X = integer(R),
                                    FUN = Tperm.aov2.mv, 
                                    y = y, g = g, ngrp = ngrp, nobs = nobs, 
                                    var.equal = var.equal, median.test = median.test, 
                                    p = p, m = m, mtm = mtm, mtmi = mtmi, minv = minv)
      } # end if(parallel)
      
      ## permutation p-value
      uni.p.value <- rep(NA, nvar)
      p.value <- mean(permdist >= Tstat)
      for(v in 1:nvar) uni.p.value[v] <- mean(permdist >= Tuni[v])
      
    } # end if(nvar == 1L)
    
    
    #########   RESULTS   #########
    
    ### return results
    if(make.cl) parallel::stopCluster(cl)
    if(!perm.dist) permdist <- NULL
    res <- list(statistic = Tstat, p.value = p.value,
                perm.dist = permdist, repeated = FALSE,
                var.equal = var.equal, median.test = median.test, R = R)
    if(nvar > 1L) {
      res$univariate <- Tuni
      res$adj.p.values <- uni.p.value
    }
    class(res) <- "rand.test.aov2"
    return(res)
    
  } # end rand.test.aov2.R


### permutation replication (univariate)
Tperm.aov2 <-
  function(i, y, g, ngrp, nobs, var.equal = FALSE, median.test = FALSE, 
           p = 1, m = NULL, mtm = NULL, mtmi = NULL, minv = NULL){
    ymat <- matrix(y, nrow = nobs, ncol = ngrp)
    ymat <- t(apply(ymat, 1, sample))
    Tstat <- Tstat.aov2(y = c(ymat), g = g, ngrp = ngrp, nobs = nobs, 
                        var.equal = var.equal, median.test = median.test, 
                        p = p, m = m, mtm = mtm, mtmi = mtmi, minv = minv)
    return(Tstat)
  } # end Tperm.aov2.R

### permutation replication (univariate)
Tperm.aov2.mv <-
  function(i, y, g, b, blev, ngrp, nobs, var.equal = FALSE, median.test = FALSE, 
           p = 1, m = NULL, mtm = NULL, mtmi = NULL, minv = NULL, combine = TRUE){
    # note: y is already a matrix that is ngrp * nobs x nvar
    for(i in 1:nobs){
      idx <- which(b == blev[i]) 
      ysub <- y[idx,,drop=FALSE]       # ngrp x nvar (for i-th subject)
      y[idx,,drop=FALSE] <- ysub[sample.int(ngrp),,drop=FALSE]   # shuffle g
    }
    Tstat <- Tstat.aov2.mv(y = y, g = g, ngrp = ngrp, nobs = nobs, 
                           var.equal = var.equal, median.test = median.test, 
                           p = p, m = m, mtm = mtm, mtmi = mtmi, minv = minv,
                           combine = combine)
    return(Tstat)
  } # end Tperm.aov2.R

### test statistic (univariate)
Tstat.aov2 <-
  function(y, g, ngrp, nobs, var.equal = FALSE, median.test = FALSE, 
           p = 1, m = NULL, mtm = NULL, mtmi = NULL, minv = NULL){
    
    if(median.test){
      if(var.equal){
        # Friedman ANOVA
        ymat <- matrix(y, nrow = nobs, ncol = ngrp)
        ry <- apply(ymat, 1, rank)
        ryg <- rowMeans(ry)
        top <- sum((ryg - (ngrp + 1)/2)^2)
        bot <- (ngrp * (ngrp + 1)) / (12 * nobs)
        Tstat <- top / bot
      } else {
        # studentized Friedman ANOVA
        ymat <- matrix(y, nrow = nobs, ncol = ngrp)
        ry <- apply(ymat, 1, rank)
        ryg <- rowMeans(ry)
        top <- sum((ryg - (ngrp + 1)/2)^2)
        rygv <- apply(ry, 1, function(x) var(x) / length(x))
        bot <- sqrt(sum(nobs^2 * rygv))
        Tstat <- top / bot
      }
    } else {
      Tstat <- Tstat.lm2(m = m, y = y, p = p, homosced = var.equal,
                         mtm = mtm, mtmi = mtmi, minv = minv)
      
    } # end if(median.test)
    Tstat
  } # end Tstat.aov2.R

### test statistic (multivariate)
Tstat.aov2.mv <-
  function(y, g, ngrp, nobs, var.equal = FALSE, median.test = FALSE, 
           p = 1, m = NULL, mtm = NULL, mtmi = NULL, minv = NULL, combine = TRUE){
    nvar <- ncol(y)
    Tstat <- rep(0.0, nvar)
    for(j in 1:nvar) {
      Tstat[j] <- Tstat.aov2(y = y[,j], g = g, ngrp = ngrp, nobs = nobs,
                             var.equal = var.equal, median.test = median.test,
                             p = p, m = m, mtm = mtm, mtmi = mtmi, minv = minv)
    }
    if(combine) Tstat <- max(Tstat)
    Tstat
  } # end Tstat.aov2.mv.R
