#' @title Michaelis-Menten Kinetics (Linear Form) with Inhibition Comparison
#' @name michaelis_menten
#' @description
#' Performs Michaelis-Menten kinetic analysis using linearized transformations
#' (Lineweaver-Burk by default) to compare multiple datasets and infer inhibition type.
#'
#' The function fits linear regressions to transformed data and compares slopes
#' and intercepts across conditions.
#'
#' Interpretation of inhibition patterns (Lineweaver-Burk):
#' \itemize{
#'   \item Competitive inhibition:
#'     Same y-intercept (1/Vmax), increased slope -> apparent Km increases
#'   \item Noncompetitive inhibition:
#'     Same x-intercept (-1/Km), increased y-intercept -> Vmax decreases
#'   \item Uncompetitive inhibition:
#'     Parallel lines -> both Km and Vmax decrease proportionally
#' }
#'
#' @param data A data.frame containing concentration and rate data.
#' @param conc_col Column name for substrate or drug concentration.
#' @param rate_col Column name for reaction rate or elimination rate.
#' @param group_col Column indicating different conditions (e.g., inhibitor levels).
#' @param transform Linearization method: "Lineweaver-Burk" (default),
#'   "Eadie-Hofstee", or "Hanes-Woolf".
#' @param inhibition_type Expected inhibition type:
#'   "competitive", "noncompetitive", "uncompetitive", "multi-inhibition" or "none".
#' @param plot Logical; if TRUE, generates linearized comparison plot.
#'
#' @import scales
#' @import stats
#' @import ggplot2
#' @importFrom scales hue_pal
#' @importFrom stats lm na.omit
#' @importFrom ggplot2 ggplot aes geom_point geom_smooth labs theme theme_bw
#' element_text element_blank
#'
#' @return A list containing:
#' \describe{
#'   \item{\code{fitted_parameters}}{A data frame summarizing linear regression
#'         results for each group or condition, including the estimated slope,
#'         intercept, coefficient of determination (\eqn{R^2}), and derived
#'         Michaelis-Menten parameters (\code{Km} and \code{Vmax}) computed according
#'         to the selected linear transformation.}
#'   \item{\code{transformed_data}}{A data frame containing the processed and
#'         linearized concentration and rate data (\code{x} and \code{y}) used for
#'         model fitting and visualization, along with the original group labels.}
#'   \item{\code{interpretation}}{A character string describing the expected
#'         inhibition pattern based on the specified \code{inhibition_type} and
#'         the comparison of slopes and intercepts across groups.}
#' }
#' @examples
#' # Example I: Single Michaelis-Menten dataset (no inhibition)
#' df1 <- data.frame(
#'   concentration = c(0.5, 1, 2, 4, 6, 8, 10),
#'   rate = c(0.48, 0.85, 1.45, 2.20, 2.70, 3.05, 3.25),
#'   group = "No Inhibitor"
#' )
#' # Lineweaver-Burk
#' michaelis_menten(
#'   data = df1,
#'   conc_col = "concentration",
#'   rate_col = "rate",
#'   group_col = "group",
#'   transform = "Lineweaver-Burk",
#'   inhibition_type = "none",
#'   plot = TRUE
#' )
#' # Eadie-Hofstee
#' michaelis_menten(
#'   data = df1,
#'   conc_col = "concentration",
#'   rate_col = "rate",
#'   group_col = "group",
#'   transform = "Eadie-Hofstee",
#'   inhibition_type = "none",
#'   plot = TRUE
#' )
#' # Hanes-Woolf
#' michaelis_menten(
#'   data = df1,
#'   conc_col = "concentration",
#'   rate_col = "rate",
#'   group_col = "group",
#'   transform = "Hanes-Woolf",
#'   inhibition_type = "none",
#'   plot = TRUE
#' )
#'
#' # Example II: Two datasets compared (inhibition analysis)
#' df2 <- data.frame(
#'   concentration = rep(c(0.5, 1, 2, 4, 6, 8, 10), 2),
#'   rate = c(
#'     # Reference (no inhibitor)
#'     0.50, 0.90, 1.50, 2.30, 2.80, 3.10, 3.30,
#'     # Condition B (possible inhibitor)
#'     0.35, 0.65, 1.10, 1.70, 2.10, 2.40, 2.55
#'   ),
#'   group = rep(c("No Inhibitor", "Inhibitor"), each = 7)
#' )
#' # Lineweaver-Burk
#' michaelis_menten(
#'   data = df2,
#'   conc_col = "concentration",
#'   rate_col = "rate",
#'   group_col = "group",
#'   transform = "Lineweaver-Burk",
#'   inhibition_type = "uncompetitive",
#'   plot = TRUE
#' )
#' # Eadie-Hofstee
#' michaelis_menten(
#'   data = df2,
#'   conc_col = "concentration",
#'   rate_col = "rate",
#'   group_col = "group",
#'   transform = "Eadie-Hofstee",
#'   inhibition_type = "competitive",
#'   plot = TRUE
#' )
#' # Hanes-Woolf
#' michaelis_menten(
#'   data = df2,
#'   conc_col = "concentration",
#'   rate_col = "rate",
#'   group_col = "group",
#'   transform = "Hanes-Woolf",
#'   inhibition_type = "competitive",
#'   plot = TRUE
#' )
#'
#' # Example III: Six datasets compared (one reference, five test conditions)
#' df3 <- data.frame(
#'   concentration = rep(c(0.5, 1, 2, 4, 6, 8, 10), 6),
#'   rate = c(
#'     # Reference
#'     0.50, 0.90, 1.50, 2.30, 2.80, 3.10, 3.30,
#'     # Mixed Noncompetitive inhibitor A
#'     0.35, 0.65, 1.10, 1.70, 2.10, 2.40, 2.55,
#'     # Uncompetitive inhibitor
#'     0.40, 0.70, 1.15, 1.75, 2.15, 2.45, 2.60,
#'     # Mixed Noncompetitive inhibitor B
#'     0.30, 0.55, 0.95, 1.45, 1.85, 2.10, 2.20,
#'     # Mixed Noncompetitive + higher dose
#'     0.28, 0.50, 0.85, 1.30, 1.65, 1.85, 1.95,
#'     # Uncompetitive + higher dose
#'     0.38, 0.65, 1.10, 1.65, 2.05, 2.30, 2.40
#'   ),
#'   group = rep(c(
#'     "Reference",
#'     "Noncompetitive (Mixed) A",
#'     "Uncompetitive",
#'     "Noncompetitive (Mixed) B",
#'     "Noncompetitive (Mixed) High",
#'     "Uncompetitive High"
#'   ), each = 7)
#' )
#' # Lineweaver-Burk
#' michaelis_menten(
#'   data = df3,
#'   conc_col = "concentration",
#'   rate_col = "rate",
#'   group_col = "group",
#'   transform = "Lineweaver-Burk",
#'   inhibition_type = "multi-inhibition",
#'   plot = TRUE
#' )
#' # Eadie-Hofstee
#' michaelis_menten(
#'   data = df3,
#'   conc_col = "concentration",
#'   rate_col = "rate",
#'   group_col = "group",
#'   transform = "Eadie-Hofstee",
#'   inhibition_type = "multi-inhibition",
#'   plot = TRUE
#' )
#' # Hanes-Woolf
#' michaelis_menten(
#'   data = df3,
#'   conc_col = "concentration",
#'   rate_col = "rate",
#'   group_col = "group",
#'   transform = "Hanes-Woolf",
#'   inhibition_type = "multi-inhibition",
#'   plot = TRUE
#' )
#' @references Michaelis, L. and Menten, M. (1913) Die kinetik der invertinwirkung.
#' Biochemistry Zeitung, 79, 333-369.
#' @author Paul Angelo C. Manlapaz
#' @export

utils::globalVariables(c("conc", "rate", "group", "x", "y", "slope", "intercept",
                         "Km", "Vmax", "R2"))

michaelis_menten <- function(data,
                             conc_col = "concentration",
                             rate_col = "rate",
                             group_col,
                             transform = c("Lineweaver-Burk",
                                           "Eadie-Hofstee",
                                           "Hanes-Woolf"),
                             inhibition_type = c("none",
                                                 "competitive",
                                                 "noncompetitive",
                                                 "uncompetitive",
                                                 "multi-inhibition"),
                             plot = TRUE) {

  transform <- match.arg(transform)
  inhibition_type <- match.arg(inhibition_type)

  if (!requireNamespace("ggplot2", quietly = TRUE))
    stop("Package 'ggplot2' is required.")
  if (!requireNamespace("scales", quietly = TRUE))
    stop("Package 'scales' is required.")

  df <- data[, c(conc_col, rate_col, group_col)]
  colnames(df) <- c("conc", "rate", "group")
  df <- stats::na.omit(df)
  df <- df[df$conc > 0 & df$rate > 0, ]
  df$group <- as.factor(df$group)

  # Linearization
  if (transform == "Lineweaver-Burk") {
    df$x <- 1 / df$conc
    df$y <- 1 / df$rate
    xlab <- "1 / Concentration"
    ylab <- "1 / Rate"

    # Vmax = 1 / y-intercept; Km = slope * Vmax
    compute_pk <- function(slope, intercept) {
      Vmax <- 1 / intercept
      Km <- slope * Vmax
      c(Km = Km, Vmax = Vmax)
    }


  } else if (transform == "Eadie-Hofstee") {
    df$x <- df$rate / df$conc
    df$y <- df$rate
    xlab <- "Rate / Concentration"
    ylab <- "Rate"

    # Eadie-Hofstee: slope = -Km, intercept = Vmax
    compute_pk <- function(slope, intercept) {
      Vmax <- intercept
      Km <- -slope
      c(Km = Km, Vmax = Vmax)
    }

  } else if (transform == "Hanes-Woolf") {
    df$x <- df$conc
    df$y <- df$conc / df$rate
    xlab <- "Concentration"
    ylab <- "Concentration / Rate"

    # Hanes-Woolf: slope = 1/Vmax, intercept = Km/Vmax
    compute_pk <- function(slope, intercept) {
      Vmax <- 1 / slope
      Km <- intercept / slope
      c(Km = Km, Vmax = Vmax)
    }
  }

  # Fit per group
  fits <- lapply(split(df, df$group), function(d) {
    fit <- stats::lm(y ~ x, data = d)
    coef_vals <- as.numeric(coef(fit))  # ensure numeric
    pk <- compute_pk(slope = coef_vals[2], intercept = coef_vals[1])
    data.frame(
      group = unique(d$group),
      slope = coef_vals[2],
      intercept = coef_vals[1],
      Km = pk["Km"],
      Vmax = pk["Vmax"],
      R2 = summary(fit)$r.squared
    )
  })

  fit_results <- do.call(rbind, fits)

  # Plot
  if (plot) {
    cols <- scales::hue_pal()(nlevels(df$group))

    p <- ggplot2::ggplot(df, ggplot2::aes(x = x, y = y, color = group)) +
      ggplot2::geom_point(size = 3) +
      ggplot2::geom_smooth(method = "lm", se = FALSE, linewidth = 1) +
      ggplot2::labs(
        title = "Michaelis-Menten Linear Plot",
        subtitle = paste(
          "Transform:", transform,
          "| Inhibition:", inhibition_type
        ),
        x = xlab,
        y = ylab,
        color = "Condition"
      ) +
      ggplot2::theme_bw(base_size = 14) +
      ggplot2::theme(
        plot.title = ggplot2::element_text(face = "bold", hjust = 0.5),
        plot.subtitle = ggplot2::element_text(hjust = 0.5),
        panel.grid.major = ggplot2::element_blank(),  # remove major grid
        panel.grid.minor = ggplot2::element_blank()   # remove minor grid
      )

    print(p)
  }

  # Interpretation helper
  interpretation <- switch(
    inhibition_type,
    competitive =
      "Competitive inhibition: slopes differ, y-intercepts approximately equal -> Km increases, Vmax unchanged.",
    noncompetitive =
      "Noncompetitive inhibition: y-intercepts differ, x-intercepts similar -> Vmax decreases, Km unchanged.",
    uncompetitive =
      "Uncompetitive inhibition: parallel lines -> both Km and Vmax decrease.",
    none =
      "No inhibition: baseline kinetic parameters.",
    "multi-inhibition" =
      "Multiple inhibition types: inspect slopes, intercepts, Km, and Vmax to classify."
  )

  list(
    fitted_parameters = fit_results,
    transformed_data = df,
    interpretation = interpretation
  )
}
