% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pplot.R
\name{as.pplot}
\alias{as.pplot}
\title{Coerce into a '\code{pplot}' object}
\usage{
as.pplot(
  ...,
  names_k = NULL,
  names_s = NULL,
  names_g = NULL,
  color_g = NULL,
  shape_g = NULL,
  n.rows = NULL,
  scales = "free_y",
  Latex = FALSE
)
}
\arguments{
\item{...}{A single ggplot or list(s) of ggplots to be transformed.}

\item{names_k}{Vector. Names of the variables \eqn{k=1,\ldots,K}. 
If \code{NULL} (the default), the names of the first plot are reused. 
For LaTeX exports, use e.g. \code{paste0("y_{ ", 1:dim_K, " }")}.}

\item{names_s}{Vector. Names of the shocks \eqn{s=1,\ldots,S}. If \code{NULL} 
(the default), the names of the first plot are reused. 
For LaTeX exports, use e.g. \code{paste0("\\\epsilon_{ ", 1:dim_S, " }")}.}

\item{names_g}{Vector. Names of the layered plots \eqn{g=1,\ldots,G}. If \code{NULL} 
(the default), the names of the plots given in \code{...} are reused.}

\item{color_g}{Vector. Colors of the layered plots \eqn{g=1,\ldots,G}.}

\item{shape_g}{Vector. Shapes of the layered plots \eqn{g=1,\ldots,G}, 
see \strong{ggplot2}'s \code{\link[ggplot2]{geom_point}} for shapes. 
If \code{NULL} (the default), no points will be set on the IRF-lines.}

\item{n.rows}{Integer. Number of rows in \code{\link[ggplot2]{facet_wrap}}. 
If \code{NULL} (the default), the dimensions of the facet plots given 
in \code{...} are reused.}

\item{scales}{Character. Should scales be fixed (\code{"fixed"}), 
free (\code{"free"}), or free in one dimension (\code{"free_x"}, 
\code{"free_y"}, the default)? See \code{\link[ggplot2]{facet_wrap}}.}

\item{Latex}{Logical. If \code{TRUE}, the arrows of the facet labels are 
written in LaTeX math mode.}
}
\value{
A list of class '\code{pplot}'. Objects of this class contain the elements:
  \item{F.plot}{'\code{ggplot}' object for the merged plot.}
  \item{L.plot}{List of '\code{ggplot}' objects containing all \eqn{G} plots.}
  \item{args_pplot}{List of characters and integers indicating the
                    specifications used for creating \code{F.plot}.}
}
\description{
Coerce into a '\code{pplot}' object. This function allows (1) to
  overlay and colorize multiple plots of IRF and confidence bands in a single 
  '\code{ggplot}' object and (2) to overwrite shock and variable names in
  verbatim LaTeX math mode suitable for the export via \strong{tikzDevice}.
}
\details{
\code{\link{as.pplot}} is used as an intermediary in the '\code{pplot}' 
  functions to achieve compatibility with different classes. Equivalently to 
  \code{\link{as.pvarx}} for panels of \eqn{N} VAR objects, \code{\link{as.pplot}} 
  summarizes panels of \eqn{G} plot objects that have been returned 
  from the '\code{plot}' method for class '\code{svarirf}' or '\code{sboot}'. 
  If the user wishes to extend the compatibility of the '\code{pplot}' 
  functions with further classes, she may simply specify accordant 
  \code{\link{as.pplot}}-methods instead of altering the original 
  '\code{pplot}' functions.
}
\examples{
### gallery of merged IRF plots ###
\donttest{
library("ggplot2")
data("PCAP")
names_k = c("g", "k", "l", "y")  # variable names
names_i = levels(PCAP$id_i)      # country names 
L.data  = sapply(names_i, FUN=function(i) 
  ts(PCAP[PCAP$id_i==i, names_k], start=1960, end=2019, frequency=1), 
  simplify=FALSE)
L.vars = lapply(L.data, FUN=function(x) vars::VAR(x, p=2, type="both"))
L.chol = lapply(L.vars, FUN=function(x) svars::id.chol(x))

# overlay all IRF to get an overview on the stability #
L.irf = lapply(L.chol, FUN=function(x) plot(irf(x, n.ahead=30)))
summary(as.pvarx(L.vars))
as.pplot(L.irf)

# overlay IRF of selected countries and quantiles of all countries #
F.mg  = plot(sboot.mg(L.chol, n.ahead=30), lowerq=0.05, upperq=0.95)
R.irf = as.pplot(MG=F.mg, L.irf[c("DEU", "FRA", "ITA", "JPN")])
plot(R.irf)  # emphasize MG-IRF in next step
R.irf = as.pplot(R.irf, color_g="black", shape_g=c(20, rep(NA, 4)))
R.irf$F.plot + guides(fill="none") + labs(color="Country", shape="Country")

# compare two mean-groups and their quantiles #
idx_nord = c(5, 6, 10, 17, 20)  # Nordic countries
R.irf = as.pplot(color_g=c("black", "blue"), 
  Other  = plot(sboot.mg(L.chol[-idx_nord])), 
  Nordic = plot(sboot.mg(L.chol[ idx_nord])))
plot(R.irf)
 
# compare different shock ordering for MG-IRF #
R.pid1 = pid.chol(L.vars)
R.pid2 = pid.chol(L.vars, order_k=4:1)
R.pid3 = pid.chol(L.vars, order_k=c(1,4,2,3))

R.pal = RColorBrewer::brewer.pal(n=8, name="Spectral")[c(8, 1, 4)]
R.irf = as.pplot(color_g=R.pal, shape_g=c(2, 3, 20), 
  GKLY = plot(irf(R.pid1, n.ahead=25)), 
  YLKG = plot(irf(R.pid2, n.ahead=25)), 
  GYKL = plot(irf(R.pid3, n.ahead=25)))
R.mg = as.pplot(color_g=R.pal, shape_g=c(2, 3, 20), 
  GKLY = plot(sboot.mg(R.pid1, n.ahead=25), lowerq=0.05, upperq=0.95), 
  YLKG = plot(sboot.mg(R.pid2, n.ahead=25), lowerq=0.05, upperq=0.95), 
  GYKL = plot(sboot.mg(R.pid3, n.ahead=25), lowerq=0.05, upperq=0.95))
         
# colorize and export a single sub-plot to Latex #
library("tikzDevice")
textwidth = 15.5/2.54  # LaTeX textwidth from "cm" into "inch"
file_fig  = file.path(tempdir(), "Fig_irf.tex")

R.irf = as.pplot(
  DEU = plot(irf(L.chol[["DEU"]], n.ahead=50), selection=list(4, 1)), 
  FRA = plot(irf(L.chol[["FRA"]], n.ahead=50), selection=list(4, 1)),
  color_g = c("black", "blue"),
  names_g = c("Germany", "France"),
  names_k = "y", 
  names_s = "\\\\epsilon_{ g }", 
  Latex   = TRUE)

tikz(file=file_fig, width=1.2*textwidth, height=0.8*textwidth)
  R.irf$F.plot + labs(color="Country") + theme_minimal()
dev.off()
}

}
\seealso{
\code{\link{PP}}, \code{\link{irf.pvarx}}, \code{\link{pid.dc}}, and 
  \code{\link{id.iv}} for further examples of edited plots, in particular for 
  subset and reordered facet plots with reshaped facet dimensions.
}
