% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/import_ghcn_hourly.R
\name{import_ghcn_hourly}
\alias{import_ghcn_hourly}
\title{Import data from the Global Historical Climatology hourly (GHCNh) database}
\usage{
import_ghcn_hourly(
  station = "UKI0000EGLL",
  year = NULL,
  source = c("psv", "parquet"),
  hourly = TRUE,
  extra = FALSE,
  abbr_names = TRUE,
  append_codes = FALSE,
  codes = c("measurement_code", "quality_code", "report_type", "source_code",
    "source_id"),
  progress = rlang::is_interactive()
)
}
\arguments{
\item{station}{One or more site codes for the station(s) of interest,
obtained using \code{\link[=import_ghcn_stations]{import_ghcn_stations()}}.}

\item{year}{One or more years of interest. If \code{NULL}, the default, all years
of data available for the chosen \code{station}s will be imported. Note that, in
the GHCNd and GHCNm, files are split by station but not year, so setting a
\code{year} will not speed up the download. Specifying fewer years will improve
the speed of a GHCNh download, however.}

\item{source}{There are two identical data formats to read from - \code{"psv"}
(flat, pipe-delimited files) and \code{"parquet"} (a newer, faster, columnar
format). The latter is typically faster, but requires the \code{arrow} package
as an additional dependency. Note that this only applies when \code{year} is not
\code{NULL}; all \code{by-site} files are \code{psv} files.}

\item{hourly}{Should hourly means be calculated? The default is \code{TRUE}. If
\code{FALSE} then the raw data are returned, which can be sub-hourly.}

\item{extra}{Should additional columns be returned? The default, \code{FALSE},
returns an opinionated selection of elements that'll be of most interest to
most users. \code{TRUE} will return everything available.}

\item{abbr_names}{Should column names be abbreviated? When \code{TRUE}, the
default, columns like \code{"wind_direction"} are shortened to \code{"wd"}. When
\code{FALSE}, names will match the raw data, albeit in lower case.}

\item{append_codes}{Logical. Should various codes and flags be appended to
the output dataframe?}

\item{codes}{When \code{append_codes} is \code{TRUE}, which codes should be appended to
the dataframe? Any combination of \code{"measurement_code"}, \code{"quality_code"},
\code{"report_type"}, \code{"source_code"}, and/or \code{"source_id"}.}

\item{progress}{Show a progress bar when importing many stations/years?
Defaults to \code{TRUE} in interactive R sessions. Passed to \code{.progress} in
\code{\link[purrr:map]{purrr::map()}} and/or \code{\link[purrr:pmap]{purrr::pmap()}}.}
}
\value{
a \link[tibble:tibble-package]{tibble}
}
\description{
This function flexibly accesses meteorological data from the GHCNh database.
Users can provide any number of years and stations, and fully control the
sorts of data flag codes that are returned with the data. By default, column
names are shortened for easier use in R, but longer, more descriptive names
can be requested.
}
\section{Data Definition}{


The following core elements are in the GHCNh:
\itemize{
\item \strong{wind_direction:} (\code{wd}) Wind direction from true north (degrees). 360 = north, 180 = south, 270 = west; 000 indicates calm winds.
\item \strong{wind_speed:} (\code{ws}) Average wind speed (m/s).
\item \strong{temperature:} (\code{air_temp}) Air (dry bulb) temperature at approximately 2 meters above ground level, in degrees Celsius (to tenths).
\item \strong{station_level_pressure:} (\code{atmos_pres}) Pressure observed at station elevation; true barometric pressure at the location (hPa).
\item \strong{visibility:} (\code{visibility}) Horizontal visibility distance (km).
\item \strong{dew_point_temperature:} (\code{dew_point}) Dew point temperature in degrees Celsius (to tenths).
\item \strong{relative_humidity:} (\code{rh}) Relative humidity in whole percent, measured or calculated from temperature and dew point.
\item \strong{sky_cover:} (\code{cl}) Maximum of all \strong{sky_cover_X} elements (oktas).
\item \strong{sky_cover_baseht:} (\code{cl_baseht}) The height above ground level (AGL) of the lowest cloud or obscuring phenomena layer aloft with 5/8 or more summation total sky cover, which may be predominantly opaque, or the vertical visibility into a surface-based obstruction.
\item \strong{sky_cover_X:} (\code{cl_X}) Fraction of sky covered by clouds (oktas). Up to 3 layers reported.
\item \strong{sky_cover_baseht_X:} (\code{cl_baseht_X}) Cloud base height for lowest layer (m). Up to 3 layers reported.
\item \strong{precipitation:} (\code{precip}) Total liquid precipitation (rain or melted snow) for the hour (mm). "T" indicates trace amounts.
\item \strong{precipitation_XX_hour:} (\code{precip_XX}) 3-hour total liquid precipitation (mm). "T" indicates trace amounts.
}

When \code{extra = TRUE}, the following additional columns are included:
\itemize{
\item \strong{sea_level_pressure:} (\code{sea_pres}) Estimated pressure at sea level directly below the station using actual temperature profile (hPa).
\item \strong{wind_gust:} (\code{wg}) Peak short-duration wind speed (m/s) exceeding the average wind speed.
\item \strong{wet_bulb_temperature:} (\code{wet_bulb}) Wet bulb temperature in degrees Celsius (to tenths), measured or calculated from temperature, dew point, and station pressure.
\item \strong{snow_depth:} (\code{snow_depth}) Depth of snowpack on the ground (mm).
\item \strong{altimeter:} (\code{altimeter}) Pressure reduced to mean sea level using standard atmosphere profile (hPa).
\item \strong{pressure_3hr_change:} (\code{pres_03}) Change in atmospheric pressure over a 3-hour period (hPa), with tendency code.
}

If \code{hourly = FALSE}, the following character columns may also be present.
\itemize{
\item \strong{pres_wx_MWX:} (\code{wx_mwX}) Present weather observation from manual reports (code). Up to 3 observations per report.
\item \strong{pres_wx_AUX:} (\code{wx_auX}) Present weather observation from automated ASOS/AWOS sensors (code). Up to 3 observations per report.
\item \strong{pres_wx_AWX:} (\code{wx_aqX}) Present weather observation from automated sensors (code). Up to 3 observations per report.
\item \strong{remarks:} (\code{remarks}) Raw observation remarks encoded in ICAO-standard METAR/SYNOP format.
}
}

\section{Parallel Processing}{


If you are importing a lot of meteorological data, this can take a long
while. This is because each combination of year and station requires
downloading a separate data file from NOAA's online data directory, and the
time each download takes can quickly add up. Many data import functions in
\code{{worldmet}} can use parallel processing to speed downloading up, powered
by the capable \code{{mirai}} package. If users have any \code{{mirai}} "daemons"
set, these functions will download files in parallel. The greatest benefits
will be seen if you spawn as many daemons as you have cores on your
machine, although one fewer than the available cores is often a good rule
of thumb. Your mileage may vary, however, and naturally spawning more
daemons than station-year combinations will lead to diminishing returns.

\if{html}{\out{<div class="sourceCode">}}\preformatted{# set workers - once per session
mirai::daemons(4)

# import lots of data - NB: no change in the import function!
big_met <- import_ghcn_hourly(code = "UKI0000EGLL", year = 2010:2025)
}\if{html}{\out{</div>}}
}

\seealso{
Other GHCN functions: 
\code{\link{import_ghcn_countries}()},
\code{\link{import_ghcn_daily}()},
\code{\link{import_ghcn_inventory}()},
\code{\link{import_ghcn_monthly_temp}()},
\code{\link{import_ghcn_stations}()}
}
\author{
Jack Davison
}
\concept{GHCN functions}
