#' Print method for xtvfreg objects
#'
#' @param x An object of class "xtvfreg"
#' @param ... Additional arguments (currently unused)
#' @return Invisibly returns the input object
#' @export
#' 
#' @examples
#' set.seed(456)
#' n_id <- 30
#' n_time <- 4
#' panel_data <- data.frame(
#'   id = rep(1:n_id, each = n_time),
#'   group = rep(c("A", "B"), length.out = n_id * n_time),
#'   x = rnorm(n_id * n_time)
#' )
#' panel_data$m_x <- ave(panel_data$x, panel_data$id, FUN = mean)
#' panel_data$d_x <- panel_data$x - panel_data$m_x
#' panel_data$y <- 1 + panel_data$m_x + rnorm(n_id * n_time)
#' 
#' result <- xtvfreg(
#'   formula = y ~ 1,
#'   data = panel_data,
#'   group = "group",
#'   panel_id = "id",
#'   mean_vars = c("m_x", "d_x"),
#'   var_vars = "m_x",
#'   verbose = FALSE
#' )
#' 
#' print(result)
print.xtvfreg <- function(x, ...) {
  cat("\nVarying Fixed Effects Panel Regression\n")
  cat("\nCall:\n")
  print(x$call)
  cat("\nDependent variable:", x$depvar, "\n")
  cat("Panel ID:", x$panel_id, "\n")
  cat("Group variable:", x$group_var, "\n")
  cat("Number of groups:", length(x$groups), "\n")
  
  cat("\nConvergence Summary:\n")
  for (g in names(x$convergence)) {
    conv <- x$convergence[[g]]
    status <- if (conv$converged) "converged" else "NOT converged"
    cat(sprintf("  Group %s: %s (%d iterations, LL=%.4f)\n",
                g, status, conv$n_iter, conv$ll_final))
  }
  
  cat("\nVariance Decomposition Summary:\n")
  for (g in names(x$variance_decomp)) {
    vd <- x$variance_decomp[[g]]
    cat(sprintf("  Group %s:\n", g))
    cat(sprintf("    Mean model: %.1f%%\n", vd$prop_mean * 100))
    cat(sprintf("    Variance model: %.1f%%\n", vd$prop_var * 100))
    cat(sprintf("    Unexplained: %.1f%%\n", vd$prop_unexplained * 100))
  }
  
  cat("\nUse summary() for detailed results\n")
  invisible(x)
}


#' Summary method for xtvfreg objects
#'
#' @param object An object of class "xtvfreg"
#' @param ... Additional arguments (currently unused)
#' @return Invisibly returns the input object
#' @export
#' 
#' @examples
#' set.seed(456)
#' n_id <- 30
#' n_time <- 4
#' panel_data <- data.frame(
#'   id = rep(1:n_id, each = n_time),
#'   group = rep(c("A", "B"), length.out = n_id * n_time),
#'   x = rnorm(n_id * n_time)
#' )
#' panel_data$m_x <- ave(panel_data$x, panel_data$id, FUN = mean)
#' panel_data$d_x <- panel_data$x - panel_data$m_x
#' panel_data$y <- 1 + panel_data$m_x + rnorm(n_id * n_time)
#' 
#' result <- xtvfreg(
#'   formula = y ~ 1,
#'   data = panel_data,
#'   group = "group",
#'   panel_id = "id",
#'   mean_vars = c("m_x", "d_x"),
#'   var_vars = "m_x",
#'   verbose = FALSE
#' )
#' 
#' summary(result)
summary.xtvfreg <- function(object, ...) {
  cat("\nVarying Fixed Effects Panel Regression\n")
  cat(strrep("=", 78), "\n\n")
  
  for (g in names(object$results)) {
    cat(sprintf("Group: %s\n", g))
    cat(strrep("-", 78), "\n")
    
    # Convergence info
    conv <- object$convergence[[g]]
    cat(sprintf("Convergence: %s (%d iterations)\n",
                if (conv$converged) "Yes" else "No",
                conv$n_iter))
    cat(sprintf("Log-likelihood: %.4f\n", conv$ll_final))
    
    # Variance decomposition
    vd <- object$variance_decomp[[g]]
    cat("\nVariance Decomposition:\n")
    cat(sprintf("  Total variance: %.6f\n", vd$var_total))
    cat(sprintf("  Mean model: %.6f (%.1f%%)\n", 
                vd$var_fitted, vd$prop_mean * 100))
    cat(sprintf("  Variance model: %.6f (%.1f%%)\n",
                vd$var_heterosced, vd$prop_var * 100))
    cat(sprintf("  Unexplained: %.6f (%.1f%%)\n",
                vd$var_total - vd$var_fitted - vd$var_heterosced,
                vd$prop_unexplained * 100))
    
    # Mean equation
    cat("\nMean Equation:\n")
    print(summary(object$results[[g]]$mean_model))
    
    # Variance equation
    cat("\nVariance Equation:\n")
    print(summary(object$results[[g]]$var_model))
    
    cat("\n")
    cat(strrep("=", 78), "\n\n")
  }
  
  invisible(object)
}


#' Coefficient extraction method for xtvfreg objects
#'
#' @param object An object of class "xtvfreg"
#' @param equation Character; "mean" or "variance"
#' @param group Optional; specific group to extract. If NULL, returns all groups
#' @param ... Additional arguments (currently unused)
#' @return Named vector or list of coefficient vectors
#' @importFrom stats coef
#' @export
#' 
#' @examples
#' set.seed(456)
#' n_id <- 30
#' n_time <- 4
#' panel_data <- data.frame(
#' id = factor(rep(1:n_id, each = n_time)),  # Convert to factor here
#' group = rep(c("A", "B"), length.out = n_id * n_time),
#' x = rnorm(n_id * n_time)
#' )
#' panel_data$m_x <- ave(panel_data$x, panel_data$id, FUN = mean)
#' panel_data$d_x <- panel_data$x - panel_data$m_x
#' panel_data$y <- 1 + panel_data$m_x + rnorm(n_id * n_time)
#' 
#' result <- xtvfreg(
#'   formula = y ~ 1,
#'   data = panel_data,
#'   group = "group",
#'   panel_id = "id",
#'   mean_vars = c("m_x", "d_x"),
#'   var_vars = "m_x",
#'   verbose = FALSE
#' )
#' 
#' # Extract coefficients
#' coef(result, equation = "mean")
#' coef(result, equation = "mean", group = "A")
coef.xtvfreg <- function(object, equation = "mean", group = NULL, ...) {
  
  if (!equation %in% c("mean", "variance")) {
    stop("equation must be either 'mean' or 'variance'")
  }
  
  model_name <- if (equation == "mean") "mean_model" else "var_model"
  
  if (!is.null(group)) {
    if (!group %in% names(object$results)) {
      stop("Group not found in results")
    }
    return(coef(object$results[[group]][[model_name]]))
  } else {
    coef_list <- lapply(object$results, function(x) coef(x[[model_name]]))
    return(coef_list)
  }
}

#' Variance-covariance matrix extraction method
#'
#' @param object An object of class "xtvfreg"
#' @param equation Character; "mean" or "variance"
#' @param group Optional; specific group to extract. If NULL, returns all groups
#' @param ... Additional arguments (currently unused)
#' @return Matrix or list of matrices
#' @importFrom stats vcov
#' @export
#' 
#' @examples
#' set.seed(456)  
#' n_id <- 30     
#' n_time <- 4    
#' panel_data <- data.frame(
#'   id = rep(1:n_id, each = n_time),
#'   group = rep(c("A", "B"), length.out = n_id * n_time),
#'   x = rnorm(n_id * n_time)
#' )
#' panel_data$m_x <- ave(panel_data$x, panel_data$id, FUN = mean)
#' panel_data$d_x <- panel_data$x - panel_data$m_x
#' panel_data$y <- 1 + panel_data$m_x + rnorm(n_id * n_time)
#' 
#' result <- xtvfreg(
#'   formula = y ~ 1,
#'   data = panel_data,
#'   group = "group",
#'   panel_id = "id",
#'   mean_vars = c("m_x", "d_x"),
#'   var_vars = "m_x",
#'   verbose = FALSE
#' )
#' 
#' # Extract variance-covariance matrix
#' vcov(result, equation = "mean", group = "A")
vcov.xtvfreg <- function(object, equation = "mean", group = NULL, ...) {
  
  if (!equation %in% c("mean", "variance")) {
    stop("equation must be either 'mean' or 'variance'")
  }
  
  model_name <- if (equation == "mean") "mean_model" else "var_model"
  
  if (!is.null(group)) {
    if (!group %in% names(object$results)) {
      stop("Group not found in results")
    }
    return(vcov(object$results[[group]][[model_name]]))
  } else {
    vcov_list <- lapply(object$results, function(x) vcov(x[[model_name]]))
    return(vcov_list)
  }
}