/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import {
  AccessToken,
  BasicOrTokenAuth,
  FacadeCredentials,
  HttpStatusCode,
  OperationFail,
  OperationOk,
  PaytoString,
  Paytos,
  TalerError,
  TalerMerchantApi,
  TalerRevenueHttpClient,
  opFixedSuccess,
} from "@gnu-taler/taler-util";
import { readUnexpectedResponseDetails, type HttpRequestLibrary } from "@gnu-taler/taler-util/http";
import {
  BrowserFetchHttpLib,
  LocalNotificationBannerBulma,
  useChallengeHandler,
  useLocalNotificationBetter,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";
import { useState } from "preact/hooks";
import { NotificationCard } from "../../../../components/menu/index.js";
import { SolveMFAChallenges } from "../../../../components/SolveMFA.js";
import { useSessionContext } from "../../../../context/session.js";
import { Notification } from "../../../../utils/types.js";
import { CreatePage } from "./CreatePage.js";
import { Session } from "inspector";

export type Entity = TalerMerchantApi.AccountAddDetails;
interface Props {
  onBack?: () => void;
  onConfirm: () => void;
}

export default function CreateValidator({ onConfirm, onBack }: Props): VNode {

  return (
    <>
      <CreatePage onBack={onBack} onCreated={onConfirm} />
    </>
  );
}

export enum TestRevenueErrorType {
  CANT_VALIDATE,
}

export async function testRevenueAPI(
  revenueAPI: URL,
  creds: FacadeCredentials | undefined,
) {
  const httpLib: HttpRequestLibrary = new BrowserFetchHttpLib();
  const api = new TalerRevenueHttpClient(revenueAPI.href, httpLib);
  const auth: BasicOrTokenAuth | undefined =
    creds === undefined
      ? undefined
      : creds.type === "none"
        ? undefined
        : creds.type === "basic"
          ? {
              type: "basic",
              username: creds.username,
              password: creds.password,
            }
          : creds.type === "bearer"
            ? {
                type: "bearer",
                token: creds.token as AccessToken,
              }
            : undefined;

  try {
    const config = await api.getConfig(auth);

    if (config.type === "fail") {
      return config;
    }

    const resp = await api.getHistory(auth);
    if (resp.type === "fail") {
      return resp;
    }

    if (!resp.body.credit_account) {
      return {
        type: "fail" as const,
        case: TestRevenueErrorType.CANT_VALIDATE,
        detail: undefined,
      };
    }
    const str = resp.body.credit_account as Paytos.FullPaytoString
    const uri = Paytos.fromString(str)
    if (uri.type === "fail") return uri
    return opFixedSuccess(uri.body);
  } catch (err) {
    // FIXME: should we return some other error code here?
    throw err;
  }
}
