/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { AbsoluteTime, TalerMerchantApi } from "@gnu-taler/taler-util";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { format } from "date-fns";
import { Fragment, VNode, h } from "preact";
import { useState } from "preact/hooks";
import { DatePicker } from "../../../../components/picker/DatePicker.js";
import {
  dateFormatForSettings,
  usePreference,
} from "../../../../hooks/preference.js";
import { CardTable } from "./Table.js";
import { WithId } from "../../../../declaration.js";
import { OrderListSection } from "./index.js";

const TALER_SCREEN_ID = 47;

export interface ListPageProps {
  onChangeSection: (s: OrderListSection) => void;
  section: OrderListSection;

  jumpToDate?: AbsoluteTime;
  onSelectDate: (date?: AbsoluteTime) => void;

  orders: (TalerMerchantApi.OrderHistoryEntry & WithId)[];
  onLoadMoreBefore?: () => void;
  onLoadMoreAfter?: () => void;

  onSelectOrder: (o: TalerMerchantApi.OrderHistoryEntry & WithId) => void;
  onRefundOrder: (o: TalerMerchantApi.OrderHistoryEntry & WithId) => void;
  onCreate: () => void;
}

export function ListPage({
  onLoadMoreAfter,
  onLoadMoreBefore,
  orders,
  onSelectOrder,
  onRefundOrder,
  jumpToDate,
  onChangeSection,
  onSelectDate,
  section,
  onCreate,
}: ListPageProps): VNode {
  const { i18n } = useTranslationContext();
  const dateTooltip = i18n.str`Select date to show nearby orders`;
  const [pickDate, setPickDate] = useState(false);
  const [settings] = usePreference();

  return (
    <Fragment>
      <div class="columns">
        <div class="column is-two-thirds">
          <div class="tabs" style={{ overflow: "inherit" }}>
            <ul>
              <li
                class={
                  section === OrderListSection.NEW ? "is-active" : undefined
                }
              >
                <div
                  class="has-tooltip-right"
                  data-tooltip={i18n.str`Only show unpaid orders`}
                >
                  <a onClick={() => onChangeSection(OrderListSection.NEW)}>
                    <i18n.Translate>New</i18n.Translate>
                    <i class="mdi mdi-information ml-2" />
                  </a>
                </div>
              </li>
              <li
                class={
                  section === OrderListSection.PAID ? "is-active" : undefined
                }
              >
                <div
                  class="has-tooltip-right"
                  data-tooltip={i18n.str`Only show paid orders`}
                >
                  <a onClick={() => onChangeSection(OrderListSection.PAID)}>
                    <i18n.Translate>Paid</i18n.Translate>
                    <i class="mdi mdi-information ml-2" />
                  </a>
                </div>
              </li>
              <li
                class={
                  section === OrderListSection.REFUNDED
                    ? "is-active"
                    : undefined
                }
              >
                <div
                  class="has-tooltip-right"
                  data-tooltip={i18n.str`Only show orders with refunds`}
                >
                  <a onClick={() => onChangeSection(OrderListSection.REFUNDED)}>
                    <i18n.Translate>Refunded</i18n.Translate>
                    <i class="mdi mdi-information ml-2" />
                  </a>
                </div>
              </li>
              <li
                class={
                  section === OrderListSection.PENDING ? "is-active" : undefined
                }
              >
                <div
                  class="has-tooltip-left"
                  data-tooltip={i18n.str`Show only paid orders for which the wire transfer is still pending.`}
                >
                  <a onClick={() => onChangeSection(OrderListSection.PENDING)}>
                    <i18n.Translate>Not wired</i18n.Translate>
                    <i class="mdi mdi-information ml-2" />
                  </a>
                </div>
              </li>
              <li
                class={
                  section === OrderListSection.INCOMING
                    ? "is-active"
                    : undefined
                }
              >
                <div
                  class="has-tooltip-left"
                  data-tooltip={i18n.str`Only display orders that have already been transferred by the payment service provider`}
                >
                  <a onClick={() => onChangeSection(OrderListSection.INCOMING)}>
                    <i18n.Translate>Completed</i18n.Translate>
                    <i class="mdi mdi-information ml-2" />
                  </a>
                </div>
              </li>
              <li class={section === undefined ? "is-active" : undefined}>
                <div
                  class="has-tooltip-right"
                  data-tooltip={i18n.str`Remove all filters`}
                >
                  <a onClick={() => onChangeSection(OrderListSection.ALL)}>
                    <i18n.Translate>All</i18n.Translate>
                    <i class="mdi mdi-information ml-2" />
                  </a>
                </div>
              </li>
            </ul>
          </div>
        </div>
        <div class="column ">
          <div class="buttons is-right">
            <div class="field has-addons">
              {jumpToDate && (
                <div class="control">
                  <a
                    class="button is-fullwidth"
                    onClick={() => onSelectDate(undefined)}
                  >
                    <span
                      class="icon"
                      data-tooltip={i18n.str`Clear date filter`}
                    >
                      <i class="mdi mdi-close" />
                    </span>
                  </a>
                </div>
              )}
              <div class="control">
                <span class="has-tooltip-top" data-tooltip={dateTooltip}>
                  <input
                    class="input"
                    type="text"
                    readonly
                    value={
                      !jumpToDate || jumpToDate.t_ms === "never"
                        ? ""
                        : format(
                            jumpToDate.t_ms,
                            dateFormatForSettings(settings),
                          )
                    }
                    placeholder={i18n.str`Jump to date (${dateFormatForSettings(
                      settings,
                    )})`}
                    onClick={() => {
                      setPickDate(true);
                    }}
                  />
                </span>
              </div>
              <div class="control">
                <span class="has-tooltip-left" data-tooltip={dateTooltip}>
                  <a
                    class="button  is-fullwidth"
                    onClick={() => {
                      setPickDate(true);
                    }}
                  >
                    <span class="icon">
                      <i class="mdi mdi-calendar" />
                    </span>
                  </a>
                </span>
              </div>
            </div>
          </div>
        </div>
      </div>

      <DatePicker
        opened={pickDate}
        closeFunction={() => setPickDate(false)}
        dateReceiver={(d) => {
          onSelectDate(AbsoluteTime.fromMilliseconds(d.getTime()));
        }}
      />

      <CardTable
        orders={orders}
        onCreate={onCreate}
        section={section}
        onSelect={onSelectOrder}
        onRefund={onRefundOrder}
        onLoadMoreAfter={onLoadMoreAfter}
        onLoadMoreBefore={onLoadMoreBefore}
      />
    </Fragment>
  );
}
