/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AbsoluteTime,
  AmountString,
  Duration,
  PreparePayResultType,
  TalerMerchantApi,
  TalerMerchantInstanceHttpClient,
  j2s,
  succeedOrThrow,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import {
  applyTimeTravelV2,
  createSimpleTestkudosEnvironmentV3,
  withdrawViaBankV3,
} from "../harness/environments.js";
import { GlobalTestState } from "../harness/harness.js";

/**
 * Test APIs related to merchant wire transfers.
 */
export async function runMerchantWireTest(t: GlobalTestState) {
  // Set up test environment

  const {
    walletClient,
    bankClient,
    exchange,
    merchant,
    merchantAdminAccessToken,
  } = await createSimpleTestkudosEnvironmentV3(t);

  // Withdraw digital cash into the wallet.

  await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  // Order that can only be paid within five minutes.
  const order: TalerMerchantApi.Order = {
    summary: "Buy me!",
    amount: "TESTKUDOS:5",
    fulfillment_url: "taler://fulfillment-success/thx",
    pay_deadline: AbsoluteTime.toProtocolTimestamp(
      AbsoluteTime.addDuration(
        AbsoluteTime.now(),
        Duration.fromSpec({ minutes: 5 }),
      ),
    ),
    refund_deadline: AbsoluteTime.toProtocolTimestamp(AbsoluteTime.zero()),
    wire_transfer_deadline: AbsoluteTime.toProtocolTimestamp(
      AbsoluteTime.addDuration(
        AbsoluteTime.now(),
        Duration.fromSpec({ minutes: 10 }),
      ),
    ),
  };

  const merchantClient = new TalerMerchantInstanceHttpClient(
    merchant.makeInstanceBaseUrl(),
  );

  const orderResp = succeedOrThrow(
    await merchantClient.createOrder(merchantAdminAccessToken, {
      order,
    }),
  );

  let orderStatus = succeedOrThrow(
    await merchantClient.getOrderDetails(
      merchantAdminAccessToken,
      orderResp.order_id,
    ),
  );

  t.assertTrue(orderStatus.order_status === "unpaid");

  const preparePayResult = await walletClient.call(
    WalletApiOperation.PreparePayForUri,
    {
      talerPayUri: orderStatus.taler_pay_uri,
    },
  );

  t.assertDeepEqual(
    preparePayResult.status,
    PreparePayResultType.PaymentPossible,
  );

  const confirmPayResult = await walletClient.call(
    WalletApiOperation.ConfirmPay,
    { transactionId: preparePayResult.transactionId },
  );
  console.log("confirm pay result:");
  console.log(j2s(confirmPayResult));
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  await applyTimeTravelV2(
    Duration.toMilliseconds(Duration.fromSpec({ hours: 2 })),
    { walletClient, exchange, merchant },
  );

  await exchange.runAggregatorOnce();
  await merchant.runDepositcheckOnce();
  {
    const resp = succeedOrThrow(
      await merchantClient.listIncomingWireTransfers(merchantAdminAccessToken),
    );
    t.assertDeepEqual(resp.incoming.length, 1);
    // No reconciliation happened yet.
    t.assertTrue(resp.incoming[0].expected_credit_amount == null);
    console.log(j2s(resp));
  }

  await merchant.runReconciliationOnce();

  let incomingAmount: AmountString;
  let incomingWtid: string;
  let incomingPayto: string;
  {
    const resp = succeedOrThrow(
      await merchantClient.listIncomingWireTransfers(merchantAdminAccessToken),
    );
    console.log(j2s(resp));
    t.assertDeepEqual(resp.incoming.length, 1);
    const det = resp.incoming[0];
    t.assertTrue(det.expected_credit_amount != null);
    t.assertAmountEquals(det.expected_credit_amount, "TESTKUDOS:4.79");
    incomingAmount = det.expected_credit_amount;
    incomingWtid = det.wtid;
    incomingPayto = det.payto_uri;
  }

  {
    const resp = succeedOrThrow(
      await merchantClient.listConfirmedWireTransfers(merchantAdminAccessToken),
    );
    console.log(j2s(resp));
    t.assertDeepEqual(resp.transfers.length, 0);
  }

  {
    const resp = succeedOrThrow(
      await merchantClient.listConfirmedWireTransfers(merchantAdminAccessToken),
    );
    console.log(j2s(resp));
    t.assertDeepEqual(resp.transfers.length, 0);
  }

  succeedOrThrow(
    await merchantClient.informWireTransfer(merchantAdminAccessToken, {
      credit_amount: incomingAmount,
      exchange_url: exchange.baseUrl,
      payto_uri: incomingPayto,
      wtid: incomingWtid,
    }),
  );

  {
    const resp = succeedOrThrow(
      await merchantClient.listConfirmedWireTransfers(merchantAdminAccessToken),
    );
    console.log(j2s(resp));
    t.assertDeepEqual(resp.transfers.length, 1);
  }
}

runMerchantWireTest.suites = ["wallet"];
