/*
 This file is part of GNU Taler
 (C) 2022-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  urlPattern,
  useCurrentLocation,
  useNavigationContext,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";

import { assertUnreachable } from "@gnu-taler/taler-util";
import { useErrorBoundary } from "preact/hooks";
import { CheckChallengeIsUpToDate } from "./components/CheckChallengeIsUpToDate.js";
import { SessionId } from "./hooks/session.js";
import { AnswerChallenge } from "./pages/AnswerChallenge.js";
import { AskChallenge } from "./pages/AskChallenge.js";
import { CallengeCompleted } from "./pages/CallengeCompleted.js";
import { Frame } from "./pages/Frame.js";
import { Setup } from "./pages/Setup.js";

export function Routing(): VNode {
  // check session and defined if this is
  // public routing or private
  return (
    <Frame>
      <PublicRounting />
    </Frame>
  );
}

const publicPages = {
  noinfo: urlPattern(/\/noinfo/, () => `#/noinfo`),
  authorize: urlPattern(/\/authorize/, () => `#/authorize`),
  ask: urlPattern(/\/ask/, () => `#/ask`),
  answer: urlPattern(/\/answer/, () => `#/answer`),
  completed: urlPattern(/\/completed/, () => `#/completed`),
  setup: urlPattern<{ client: string }>(
    /\/setup\/(?<client>[0-9]+)/,
    ({ client }) => `#/setup/${client}`,
  ),
};

function safeGetParam(
  ps: Record<string, string[]>,
  n: string,
): string | undefined {
  if (!ps[n] || ps[n].length == 0) return undefined;
  return ps[n][0];
}

export function safeToURL(s: string | undefined): URL | undefined {
  if (s === undefined) return undefined;
  try {
    return new URL(s);
  } catch (e) {
    return undefined;
  }
}

function getSession(params: Record<string, string[]>) {
  const clientId = safeGetParam(params, "client_id");
  const redirectURL = safeToURL(safeGetParam(params, "redirect_uri"));
  const state = safeGetParam(params, "state");
  const nonce = safeGetParam(params, "nonce");

  const sessionId: SessionId | undefined =
    !clientId || !redirectURL || !state || !nonce
      ? undefined
      : {
          clientId,
          nonce: nonce,
          redirectURL: redirectURL.href,
          state,
        };
  return sessionId;
}

function PublicRounting(): VNode {
  const loc = useCurrentLocation(publicPages);
  const { i18n } = useTranslationContext();
  const { navigateTo } = useNavigationContext();
  useErrorBoundary((e) => {
    console.log("error", e);
  });

  const location: typeof loc =
    loc.name === undefined
      ? {
          ...loc,
          name: "authorize",
        }
      : loc;

  switch (location.name) {
    case "noinfo": {
      return <div>no info</div>;
    }
    case "setup": {
      const secret = safeGetParam(location.params, "secret");
      const redirectURL = safeToURL(
        safeGetParam(location.params, "redirect_uri"),
      );

      return (
        <Setup
          clientId={location.values.client}
          secret={secret}
          redirectURL={redirectURL}
          onCreated={() => {
            navigateTo(publicPages.ask.url({}));
          }}
        />
      );
    }
    case "authorize": {
      const sessionId = getSession(location.params);

      if (!sessionId) {
        return (
          <div>
            <i18n.Translate>
              The application needs to be loaded with 4 request parameters. One
              or more are missing:
            </i18n.Translate>
            {JSON.stringify({ params: location.params }, undefined, 2)}
          </div>
        );
      }
      return (
        <CheckChallengeIsUpToDate
          session={sessionId}
          onCompleted={() => {
            navigateTo(publicPages.completed.url({}));
          }}
          onChangeLeft={() => {
            navigateTo(publicPages.ask.url({}));
          }}
          onNoMoreChanges={() => {
            navigateTo(publicPages.ask.url({}));
          }}
        >
          <i18n.Translate>No nonce has been found</i18n.Translate>
        </CheckChallengeIsUpToDate>
      );
    }
    case "ask": {
      const sessionId = getSession(location.params);

      if (!sessionId) {
        return (
          <div>
            <i18n.Translate>
              The application needs to be loaded with 4 request parameters. One
              or more are missing:
            </i18n.Translate>
            {JSON.stringify({ params: location.params }, undefined, 2)}
          </div>
        );
      }
      return (
        <AskChallenge
          session={sessionId}
          focus
          routeSolveChallenge={publicPages.answer}
          onSendSuccesful={() => {
            navigateTo(publicPages.answer.url({}));
          }}
        />
      );
    }
    case "answer": {
      const sessionId = getSession(location.params);

      if (!sessionId) {
        return (
          <div>
            <i18n.Translate>
              The application needs to be loaded with 4 request parameters. One
              or more are missing:
            </i18n.Translate>
            {JSON.stringify({ params: location.params }, undefined, 2)}
          </div>
        );
      }

      return (
        <AnswerChallenge
          focus
          session={sessionId}
          routeAsk={publicPages.ask}
          onComplete={() => {
            navigateTo(publicPages.completed.url({}));
          }}
        />
      );
    }
    case "completed": {
      return <CallengeCompleted />;
    }
    default:
      assertUnreachable(location);
  }
}
