/*
 This file is part of GNU Taler
 (C) 2022-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  AbsoluteTime,
  Amounts,
  Paytos,
  TalerError
} from "@gnu-taler/taler-util";
import { useTransactions } from "../../hooks/account.js";
import { Props, State, Transaction } from "./index.js";

export function useComponentState({
  account,
  routeCreateWireTransfer,
}: Props): State {
  const result = useTransactions(account);
  if (!result) {
    return {
      status: "loading",
      error: undefined,
    };
  }
  if (result instanceof TalerError) {
    return {
      status: "loading-error",
      error: result,
    };
  }
  if (result.type === "fail") {
    return {
      status: "loading",
      error: undefined,
    };
  }

  const transactions = result.body
    .map((tx) => {
      const negative = tx.direction === "debit";
      const cp = Paytos.fromString(
        negative ? tx.creditor_payto_uri : tx.debtor_payto_uri,
      );
      const counterpart =
        cp === undefined || cp.type === "fail"
          ? undefined
          : cp.body.displayName;

      const when = AbsoluteTime.fromProtocolTimestamp(tx.date);
      const amount = Amounts.parse(tx.amount);
      const subject = tx.subject;
      return {
        negative,
        counterpart,
        when,
        amount,
        subject,
      };
    })
    .filter((x): x is Transaction => x !== undefined);

  return {
    status: "ready",
    error: undefined,
    routeCreateWireTransfer,
    transactions,
    onGoNext: result.loadNext,
    onGoStart: result.loadFirst,
  };
}
