/*
 * Copyright (C) 2014-2025 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#pragma once

#include <QLabel>
#include <QTimer>

#include "src/datovka_shared/identifiers/account_id.h" /* AcntId is used in slots. Q_MOC_INCLUDE */
#include "src/json/message_id.h" /* Json::MsgId2List is used in slots. Q_MOC_INCLUDE */

class DbContainer; /* Forward declaration. */
class QAction; /* Forward declaration. */
class QPushButton; /* Forward declaration. */
class QTextEdit; /* Forward declaration. */
class TimestampDb; /* Forward declaration. */
class TstValidityHash; /* Forward declaration. */

/*!
 * @brief Timestamp control label.
 */
class TimestampTrackingControl : public QLabel {
	Q_OBJECT

public:
	/*!
	 * @brief Constructor.
	 *
	 * @param[in] timestampDb Timestamp database.
	 * @param[in] msgDbs Message databases.
	 * @param[in] actionTimeStampRenewal Timestamp renewal action.
	 * @param[in] parent Parent widget.
	 */
	explicit TimestampTrackingControl(TimestampDb *timestampDb,
	    DbContainer *msgDbs, QAction *actionTimeStampRenewal,
	    QWidget *parent = Q_NULLPTR);

	/*!
	 * @brief Destructor.
	 */
	~TimestampTrackingControl(void);

protected:
	/*!
	 * @brief Raises status list widget.
	 *
	 * @param[in] event Mouse press event.
	 */
	virtual
	void mousePressEvent(QMouseEvent *event) Q_DECL_OVERRIDE;

private Q_SLOTS:
	/*!
	 * @brief Calls generateOverviewAndPopupOnWarning() and resets timer.
	 */
	void periodicStatusCheck(void);

	/*!
	 * @brief Trigger timestamp renewal action.
	 *
	 * @note Triggers m_actionEditAvailableTags. Closes the widget.
	 */
	void callActionTimeStampRenewal(void);

	/*!
	 * @brief Update data when timestamp tracking enabled.
	 *
	 * @param[in] msgIds Message identifiers.
	 * @param[in] acntId Account identifier.
	 */
	void watchIdentificationsInserted(const Json::MsgId2List &msgIds,
	    const AcntId &acntId);

	/*!
	 * @brief Update data when timestamp tracking disabled.
	 *
	 * @param[in] msgIds Message identifiers.
	 */
	void watchIdentificationsDeleted(const Json::MsgId2List &msgIds);

	/*!
	 * @brief Update data when timestamp data updated.
	 *
	 * @param[in] values Updated values.
	 */
	void watchMsgTimestampDataUpdated(const TstValidityHash &values);

private:
	/*!
	 * @brief Generated overview text and raise pop-up on warnings.
	 */
	void generateOverviewAndPopupOnWarning(void);

	/*!
	 * @brief Raise the pop-up widget.
	 */
	void raisePopup(void);

	TimestampDb *m_timestampDb; /*!< Timestamp database. */
	DbContainer *m_msgDbs; /*!< Message databases. */

	QWidget *m_tstPopup; /*!< Timestamp tracking pop-up. */
	QTextEdit *m_textEdit; /*!< Displays status overview. */
	QPushButton *m_timestampRenewalButton; /*!< Button to call timestamp renewal action. */
	QAction *m_actionTimeStampRenewal; /*!< Timestamp renewal action. */

	QTimer m_checkTimer; /*!< Check for expiring tracked timestamps. */
	bool m_popupRaisedOnWarning; /*< Set to true if pop-up already raised on warning. */
};
