{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GNetworkMonitor@ provides an easy-to-use cross-platform API
-- for monitoring network connectivity. On Linux, the available
-- implementations are based on the kernel\'s netlink interface and
-- on NetworkManager.
-- 
-- There is also an implementation for use inside Flatpak sandboxes.
-- 
-- /Since: 2.32/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Interfaces.NetworkMonitor
    ( 

-- * Exported types
    NetworkMonitor(..)                      ,
    IsNetworkMonitor                        ,
    toNetworkMonitor                        ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [canReach]("GI.Gio.Interfaces.NetworkMonitor#g:method:canReach"), [canReachAsync]("GI.Gio.Interfaces.NetworkMonitor#g:method:canReachAsync"), [canReachFinish]("GI.Gio.Interfaces.NetworkMonitor#g:method:canReachFinish"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [init]("GI.Gio.Interfaces.Initable#g:method:init"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getConnectivity]("GI.Gio.Interfaces.NetworkMonitor#g:method:getConnectivity"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getNetworkAvailable]("GI.Gio.Interfaces.NetworkMonitor#g:method:getNetworkAvailable"), [getNetworkMetered]("GI.Gio.Interfaces.NetworkMonitor#g:method:getNetworkMetered"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveNetworkMonitorMethod             ,
#endif

-- ** canReach #method:canReach#

#if defined(ENABLE_OVERLOADING)
    NetworkMonitorCanReachMethodInfo        ,
#endif
    networkMonitorCanReach                  ,


-- ** canReachAsync #method:canReachAsync#

#if defined(ENABLE_OVERLOADING)
    NetworkMonitorCanReachAsyncMethodInfo   ,
#endif
    networkMonitorCanReachAsync             ,


-- ** canReachFinish #method:canReachFinish#

#if defined(ENABLE_OVERLOADING)
    NetworkMonitorCanReachFinishMethodInfo  ,
#endif
    networkMonitorCanReachFinish            ,


-- ** getConnectivity #method:getConnectivity#

#if defined(ENABLE_OVERLOADING)
    NetworkMonitorGetConnectivityMethodInfo ,
#endif
    networkMonitorGetConnectivity           ,


-- ** getDefault #method:getDefault#

    networkMonitorGetDefault                ,


-- ** getNetworkAvailable #method:getNetworkAvailable#

#if defined(ENABLE_OVERLOADING)
    NetworkMonitorGetNetworkAvailableMethodInfo,
#endif
    networkMonitorGetNetworkAvailable       ,


-- ** getNetworkMetered #method:getNetworkMetered#

#if defined(ENABLE_OVERLOADING)
    NetworkMonitorGetNetworkMeteredMethodInfo,
#endif
    networkMonitorGetNetworkMetered         ,




 -- * Properties


-- ** connectivity #attr:connectivity#
-- | More detailed information about the host\'s network connectivity.
-- See 'GI.Gio.Interfaces.NetworkMonitor.networkMonitorGetConnectivity' and
-- t'GI.Gio.Enums.NetworkConnectivity' for more details.
-- 
-- /Since: 2.44/

#if defined(ENABLE_OVERLOADING)
    NetworkMonitorConnectivityPropertyInfo  ,
#endif
    getNetworkMonitorConnectivity           ,
#if defined(ENABLE_OVERLOADING)
    networkMonitorConnectivity              ,
#endif


-- ** networkAvailable #attr:networkAvailable#
-- | Whether the network is considered available. That is, whether the
-- system has a default route for at least one of IPv4 or IPv6.
-- 
-- Real-world networks are of course much more complicated than
-- this; the machine may be connected to a wifi hotspot that
-- requires payment before allowing traffic through, or may be
-- connected to a functioning router that has lost its own upstream
-- connectivity. Some hosts might only be accessible when a VPN is
-- active. Other hosts might only be accessible when the VPN is
-- not active. Thus, it is best to use 'GI.Gio.Interfaces.NetworkMonitor.networkMonitorCanReach'
-- or 'GI.Gio.Interfaces.NetworkMonitor.networkMonitorCanReachAsync' to test for reachability
-- on a host-by-host basis. (On the other hand, when the property is
-- 'P.False', the application can reasonably expect that no remote
-- hosts at all are reachable, and should indicate this to the user
-- in its UI.)
-- 
-- See also [NetworkMonitor::networkChanged]("GI.Gio.Interfaces.NetworkMonitor#g:signal:networkChanged").
-- 
-- /Since: 2.32/

#if defined(ENABLE_OVERLOADING)
    NetworkMonitorNetworkAvailablePropertyInfo,
#endif
    getNetworkMonitorNetworkAvailable       ,
#if defined(ENABLE_OVERLOADING)
    networkMonitorNetworkAvailable          ,
#endif


-- ** networkMetered #attr:networkMetered#
-- | Whether the network is considered metered.
-- 
-- That is, whether the
-- system has traffic flowing through the default connection that is
-- subject to limitations set by service providers. For example, traffic
-- might be billed by the amount of data transmitted, or there might be a
-- quota on the amount of traffic per month. This is typical with tethered
-- connections (3G and 4G) and in such situations, bandwidth intensive
-- applications may wish to avoid network activity where possible if it will
-- cost the user money or use up their limited quota. Anything more than a
-- few hundreds of kilobytes of data usage per hour should be avoided without
-- asking permission from the user.
-- 
-- If more information is required about specific devices then the
-- system network management API should be used instead (for example,
-- NetworkManager or ConnMan).
-- 
-- If this information is not available then no networks will be
-- marked as metered.
-- 
-- See also [NetworkMonitor:networkAvailable]("GI.Gio.Interfaces.NetworkMonitor#g:attr:networkAvailable").
-- 
-- /Since: 2.46/

#if defined(ENABLE_OVERLOADING)
    NetworkMonitorNetworkMeteredPropertyInfo,
#endif
    getNetworkMonitorNetworkMetered         ,
#if defined(ENABLE_OVERLOADING)
    networkMonitorNetworkMetered            ,
#endif




 -- * Signals


-- ** networkChanged #signal:networkChanged#

    NetworkMonitorNetworkChangedCallback    ,
#if defined(ENABLE_OVERLOADING)
    NetworkMonitorNetworkChangedSignalInfo  ,
#endif
    afterNetworkMonitorNetworkChanged       ,
    onNetworkMonitorNetworkChanged          ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.PollFD as GLib.PollFD
import qualified GI.GLib.Structs.Source as GLib.Source
import qualified GI.GObject.Callbacks as GObject.Callbacks
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.GObject.Structs.Parameter as GObject.Parameter
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Initable as Gio.Initable
import {-# SOURCE #-} qualified GI.Gio.Interfaces.SocketConnectable as Gio.SocketConnectable
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.SocketAddress as Gio.SocketAddress
import {-# SOURCE #-} qualified GI.Gio.Objects.SocketAddressEnumerator as Gio.SocketAddressEnumerator

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Initable as Gio.Initable
import {-# SOURCE #-} qualified GI.Gio.Interfaces.SocketConnectable as Gio.SocketConnectable
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable

#endif

-- interface NetworkMonitor 
-- | Memory-managed wrapper type.
newtype NetworkMonitor = NetworkMonitor (SP.ManagedPtr NetworkMonitor)
    deriving (Eq)

instance SP.ManagedPtrNewtype NetworkMonitor where
    toManagedPtr (NetworkMonitor p) = p

foreign import ccall "g_network_monitor_get_type"
    c_g_network_monitor_get_type :: IO B.Types.GType

instance B.Types.TypedObject NetworkMonitor where
    glibType = c_g_network_monitor_get_type

instance B.Types.GObject NetworkMonitor

-- | Type class for types which can be safely cast to t'NetworkMonitor', for instance with `toNetworkMonitor`.
class (SP.GObject o, O.IsDescendantOf NetworkMonitor o) => IsNetworkMonitor o
instance (SP.GObject o, O.IsDescendantOf NetworkMonitor o) => IsNetworkMonitor o

instance O.HasParentTypes NetworkMonitor
type instance O.ParentTypes NetworkMonitor = '[Gio.Initable.Initable, GObject.Object.Object]

-- | Cast to t'NetworkMonitor', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toNetworkMonitor :: (MIO.MonadIO m, IsNetworkMonitor o) => o -> m NetworkMonitor
toNetworkMonitor = MIO.liftIO . B.ManagedPtr.unsafeCastTo NetworkMonitor

-- | Convert t'NetworkMonitor' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe NetworkMonitor) where
    gvalueGType_ = c_g_network_monitor_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr NetworkMonitor)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr NetworkMonitor)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject NetworkMonitor ptr
        else return P.Nothing
        
    

-- VVV Prop "connectivity"
   -- Type: TInterface (Name {namespace = "Gio", name = "NetworkConnectivity"})
   -- Flags: [PropertyReadable]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@connectivity@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' networkMonitor #connectivity
-- @
getNetworkMonitorConnectivity :: (MonadIO m, IsNetworkMonitor o) => o -> m Gio.Enums.NetworkConnectivity
getNetworkMonitorConnectivity obj = MIO.liftIO $ B.Properties.getObjectPropertyEnum obj "connectivity"

#if defined(ENABLE_OVERLOADING)
data NetworkMonitorConnectivityPropertyInfo
instance AttrInfo NetworkMonitorConnectivityPropertyInfo where
    type AttrAllowedOps NetworkMonitorConnectivityPropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint NetworkMonitorConnectivityPropertyInfo = IsNetworkMonitor
    type AttrSetTypeConstraint NetworkMonitorConnectivityPropertyInfo = (~) ()
    type AttrTransferTypeConstraint NetworkMonitorConnectivityPropertyInfo = (~) ()
    type AttrTransferType NetworkMonitorConnectivityPropertyInfo = ()
    type AttrGetType NetworkMonitorConnectivityPropertyInfo = Gio.Enums.NetworkConnectivity
    type AttrLabel NetworkMonitorConnectivityPropertyInfo = "connectivity"
    type AttrOrigin NetworkMonitorConnectivityPropertyInfo = NetworkMonitor
    attrGet = getNetworkMonitorConnectivity
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.NetworkMonitor.connectivity"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-NetworkMonitor.html#g:attr:connectivity"
        })
#endif

-- VVV Prop "network-available"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@network-available@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' networkMonitor #networkAvailable
-- @
getNetworkMonitorNetworkAvailable :: (MonadIO m, IsNetworkMonitor o) => o -> m Bool
getNetworkMonitorNetworkAvailable obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "network-available"

#if defined(ENABLE_OVERLOADING)
data NetworkMonitorNetworkAvailablePropertyInfo
instance AttrInfo NetworkMonitorNetworkAvailablePropertyInfo where
    type AttrAllowedOps NetworkMonitorNetworkAvailablePropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint NetworkMonitorNetworkAvailablePropertyInfo = IsNetworkMonitor
    type AttrSetTypeConstraint NetworkMonitorNetworkAvailablePropertyInfo = (~) ()
    type AttrTransferTypeConstraint NetworkMonitorNetworkAvailablePropertyInfo = (~) ()
    type AttrTransferType NetworkMonitorNetworkAvailablePropertyInfo = ()
    type AttrGetType NetworkMonitorNetworkAvailablePropertyInfo = Bool
    type AttrLabel NetworkMonitorNetworkAvailablePropertyInfo = "network-available"
    type AttrOrigin NetworkMonitorNetworkAvailablePropertyInfo = NetworkMonitor
    attrGet = getNetworkMonitorNetworkAvailable
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.NetworkMonitor.networkAvailable"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-NetworkMonitor.html#g:attr:networkAvailable"
        })
#endif

-- VVV Prop "network-metered"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@network-metered@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' networkMonitor #networkMetered
-- @
getNetworkMonitorNetworkMetered :: (MonadIO m, IsNetworkMonitor o) => o -> m Bool
getNetworkMonitorNetworkMetered obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "network-metered"

#if defined(ENABLE_OVERLOADING)
data NetworkMonitorNetworkMeteredPropertyInfo
instance AttrInfo NetworkMonitorNetworkMeteredPropertyInfo where
    type AttrAllowedOps NetworkMonitorNetworkMeteredPropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint NetworkMonitorNetworkMeteredPropertyInfo = IsNetworkMonitor
    type AttrSetTypeConstraint NetworkMonitorNetworkMeteredPropertyInfo = (~) ()
    type AttrTransferTypeConstraint NetworkMonitorNetworkMeteredPropertyInfo = (~) ()
    type AttrTransferType NetworkMonitorNetworkMeteredPropertyInfo = ()
    type AttrGetType NetworkMonitorNetworkMeteredPropertyInfo = Bool
    type AttrLabel NetworkMonitorNetworkMeteredPropertyInfo = "network-metered"
    type AttrOrigin NetworkMonitorNetworkMeteredPropertyInfo = NetworkMonitor
    attrGet = getNetworkMonitorNetworkMetered
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.NetworkMonitor.networkMetered"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-NetworkMonitor.html#g:attr:networkMetered"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList NetworkMonitor
type instance O.AttributeList NetworkMonitor = NetworkMonitorAttributeList
type NetworkMonitorAttributeList = ('[ '("connectivity", NetworkMonitorConnectivityPropertyInfo), '("networkAvailable", NetworkMonitorNetworkAvailablePropertyInfo), '("networkMetered", NetworkMonitorNetworkMeteredPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
networkMonitorConnectivity :: AttrLabelProxy "connectivity"
networkMonitorConnectivity = AttrLabelProxy

networkMonitorNetworkAvailable :: AttrLabelProxy "networkAvailable"
networkMonitorNetworkAvailable = AttrLabelProxy

networkMonitorNetworkMetered :: AttrLabelProxy "networkMetered"
networkMonitorNetworkMetered = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveNetworkMonitorMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveNetworkMonitorMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveNetworkMonitorMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveNetworkMonitorMethod "canReach" o = NetworkMonitorCanReachMethodInfo
    ResolveNetworkMonitorMethod "canReachAsync" o = NetworkMonitorCanReachAsyncMethodInfo
    ResolveNetworkMonitorMethod "canReachFinish" o = NetworkMonitorCanReachFinishMethodInfo
    ResolveNetworkMonitorMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveNetworkMonitorMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveNetworkMonitorMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveNetworkMonitorMethod "init" o = Gio.Initable.InitableInitMethodInfo
    ResolveNetworkMonitorMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveNetworkMonitorMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveNetworkMonitorMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveNetworkMonitorMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveNetworkMonitorMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveNetworkMonitorMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveNetworkMonitorMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveNetworkMonitorMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveNetworkMonitorMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveNetworkMonitorMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveNetworkMonitorMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveNetworkMonitorMethod "getConnectivity" o = NetworkMonitorGetConnectivityMethodInfo
    ResolveNetworkMonitorMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveNetworkMonitorMethod "getNetworkAvailable" o = NetworkMonitorGetNetworkAvailableMethodInfo
    ResolveNetworkMonitorMethod "getNetworkMetered" o = NetworkMonitorGetNetworkMeteredMethodInfo
    ResolveNetworkMonitorMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveNetworkMonitorMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveNetworkMonitorMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveNetworkMonitorMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveNetworkMonitorMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveNetworkMonitorMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveNetworkMonitorMethod t NetworkMonitor, O.OverloadedMethod info NetworkMonitor p) => OL.IsLabel t (NetworkMonitor -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveNetworkMonitorMethod t NetworkMonitor, O.OverloadedMethod info NetworkMonitor p, R.HasField t NetworkMonitor p) => R.HasField t NetworkMonitor p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveNetworkMonitorMethod t NetworkMonitor, O.OverloadedMethodInfo info NetworkMonitor) => OL.IsLabel t (O.MethodProxy info NetworkMonitor) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- method NetworkMonitor::can_reach
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "monitor"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "NetworkMonitor" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GNetworkMonitor" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "connectable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SocketConnectable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSocketConnectable"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_network_monitor_can_reach" g_network_monitor_can_reach :: 
    Ptr NetworkMonitor ->                   -- monitor : TInterface (Name {namespace = "Gio", name = "NetworkMonitor"})
    Ptr Gio.SocketConnectable.SocketConnectable -> -- connectable : TInterface (Name {namespace = "Gio", name = "SocketConnectable"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Attempts to determine whether or not the host pointed to by
-- /@connectable@/ can be reached, without actually trying to connect to
-- it.
-- 
-- This may return 'P.True' even when [NetworkMonitor:networkAvailable]("GI.Gio.Interfaces.NetworkMonitor#g:attr:networkAvailable")
-- is 'P.False', if, for example, /@monitor@/ can determine that
-- /@connectable@/ refers to a host on a local network.
-- 
-- If /@monitor@/ believes that an attempt to connect to /@connectable@/
-- will succeed, it will return 'P.True'. Otherwise, it will return
-- 'P.False' and set /@error@/ to an appropriate error (such as
-- 'GI.Gio.Enums.IOErrorEnumHostUnreachable').
-- 
-- Note that although this does not attempt to connect to
-- /@connectable@/, it may still block for a brief period of time (eg,
-- trying to do multicast DNS on the local network), so if you do not
-- want to block, you should use 'GI.Gio.Interfaces.NetworkMonitor.networkMonitorCanReachAsync'.
-- 
-- /Since: 2.32/
networkMonitorCanReach ::
    (B.CallStack.HasCallStack, MonadIO m, IsNetworkMonitor a, Gio.SocketConnectable.IsSocketConnectable b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@monitor@/: a t'GI.Gio.Interfaces.NetworkMonitor.NetworkMonitor'
    -> b
    -- ^ /@connectable@/: a t'GI.Gio.Interfaces.SocketConnectable.SocketConnectable'
    -> Maybe (c)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable', or 'P.Nothing'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
networkMonitorCanReach monitor connectable cancellable = liftIO $ do
    monitor' <- unsafeManagedPtrCastPtr monitor
    connectable' <- unsafeManagedPtrCastPtr connectable
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_network_monitor_can_reach monitor' connectable' maybeCancellable
        touchManagedPtr monitor
        touchManagedPtr connectable
        whenJust cancellable touchManagedPtr
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data NetworkMonitorCanReachMethodInfo
instance (signature ~ (b -> Maybe (c) -> m ()), MonadIO m, IsNetworkMonitor a, Gio.SocketConnectable.IsSocketConnectable b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod NetworkMonitorCanReachMethodInfo a signature where
    overloadedMethod = networkMonitorCanReach

instance O.OverloadedMethodInfo NetworkMonitorCanReachMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.NetworkMonitor.networkMonitorCanReach",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-NetworkMonitor.html#v:networkMonitorCanReach"
        })


#endif

-- method NetworkMonitor::can_reach_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "monitor"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "NetworkMonitor" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GNetworkMonitor" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "connectable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SocketConnectable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSocketConnectable"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n    to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_network_monitor_can_reach_async" g_network_monitor_can_reach_async :: 
    Ptr NetworkMonitor ->                   -- monitor : TInterface (Name {namespace = "Gio", name = "NetworkMonitor"})
    Ptr Gio.SocketConnectable.SocketConnectable -> -- connectable : TInterface (Name {namespace = "Gio", name = "SocketConnectable"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously attempts to determine whether or not the host
-- pointed to by /@connectable@/ can be reached, without actually
-- trying to connect to it.
-- 
-- For more details, see 'GI.Gio.Interfaces.NetworkMonitor.networkMonitorCanReach'.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.NetworkMonitor.networkMonitorCanReachFinish'
-- to get the result of the operation.
networkMonitorCanReachAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsNetworkMonitor a, Gio.SocketConnectable.IsSocketConnectable b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@monitor@/: a t'GI.Gio.Interfaces.NetworkMonitor.NetworkMonitor'
    -> b
    -- ^ /@connectable@/: a t'GI.Gio.Interfaces.SocketConnectable.SocketConnectable'
    -> Maybe (c)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable', or 'P.Nothing'
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --     to call when the request is satisfied
    -> m ()
networkMonitorCanReachAsync monitor connectable cancellable callback = liftIO $ do
    monitor' <- unsafeManagedPtrCastPtr monitor
    connectable' <- unsafeManagedPtrCastPtr connectable
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_network_monitor_can_reach_async monitor' connectable' maybeCancellable maybeCallback userData
    touchManagedPtr monitor
    touchManagedPtr connectable
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data NetworkMonitorCanReachAsyncMethodInfo
instance (signature ~ (b -> Maybe (c) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsNetworkMonitor a, Gio.SocketConnectable.IsSocketConnectable b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod NetworkMonitorCanReachAsyncMethodInfo a signature where
    overloadedMethod = networkMonitorCanReachAsync

instance O.OverloadedMethodInfo NetworkMonitorCanReachAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.NetworkMonitor.networkMonitorCanReachAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-NetworkMonitor.html#v:networkMonitorCanReachAsync"
        })


#endif

-- method NetworkMonitor::can_reach_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "monitor"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "NetworkMonitor" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GNetworkMonitor" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_network_monitor_can_reach_finish" g_network_monitor_can_reach_finish :: 
    Ptr NetworkMonitor ->                   -- monitor : TInterface (Name {namespace = "Gio", name = "NetworkMonitor"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes an async network connectivity test.
-- See 'GI.Gio.Interfaces.NetworkMonitor.networkMonitorCanReachAsync'.
networkMonitorCanReachFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsNetworkMonitor a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@monitor@/: a t'GI.Gio.Interfaces.NetworkMonitor.NetworkMonitor'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
networkMonitorCanReachFinish monitor result_ = liftIO $ do
    monitor' <- unsafeManagedPtrCastPtr monitor
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_network_monitor_can_reach_finish monitor' result_'
        touchManagedPtr monitor
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data NetworkMonitorCanReachFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsNetworkMonitor a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod NetworkMonitorCanReachFinishMethodInfo a signature where
    overloadedMethod = networkMonitorCanReachFinish

instance O.OverloadedMethodInfo NetworkMonitorCanReachFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.NetworkMonitor.networkMonitorCanReachFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-NetworkMonitor.html#v:networkMonitorCanReachFinish"
        })


#endif

-- method NetworkMonitor::get_connectivity
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "monitor"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "NetworkMonitor" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the #GNetworkMonitor"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gio" , name = "NetworkConnectivity" })
-- throws : False
-- Skip return : False

foreign import ccall "g_network_monitor_get_connectivity" g_network_monitor_get_connectivity :: 
    Ptr NetworkMonitor ->                   -- monitor : TInterface (Name {namespace = "Gio", name = "NetworkMonitor"})
    IO CUInt

-- | Gets a more detailed networking state than
-- 'GI.Gio.Interfaces.NetworkMonitor.networkMonitorGetNetworkAvailable'.
-- 
-- If [NetworkMonitor:networkAvailable]("GI.Gio.Interfaces.NetworkMonitor#g:attr:networkAvailable") is 'P.False', then the
-- connectivity state will be 'GI.Gio.Enums.NetworkConnectivityLocal'.
-- 
-- If [NetworkMonitor:networkAvailable]("GI.Gio.Interfaces.NetworkMonitor#g:attr:networkAvailable") is 'P.True', then the
-- connectivity state will be 'GI.Gio.Enums.NetworkConnectivityFull' (if there
-- is full Internet connectivity), 'GI.Gio.Enums.NetworkConnectivityLimited' (if
-- the host has a default route, but appears to be unable to actually
-- reach the full Internet), or 'GI.Gio.Enums.NetworkConnectivityPortal' (if the
-- host is trapped behind a \"captive portal\" that requires some sort
-- of login or acknowledgement before allowing full Internet access).
-- 
-- Note that in the case of 'GI.Gio.Enums.NetworkConnectivityLimited' and
-- 'GI.Gio.Enums.NetworkConnectivityPortal', it is possible that some sites are
-- reachable but others are not. In this case, applications can
-- attempt to connect to remote servers, but should gracefully fall
-- back to their \"offline\" behavior if the connection attempt fails.
-- 
-- /Since: 2.44/
networkMonitorGetConnectivity ::
    (B.CallStack.HasCallStack, MonadIO m, IsNetworkMonitor a) =>
    a
    -- ^ /@monitor@/: the t'GI.Gio.Interfaces.NetworkMonitor.NetworkMonitor'
    -> m Gio.Enums.NetworkConnectivity
    -- ^ __Returns:__ the network connectivity state
networkMonitorGetConnectivity monitor = liftIO $ do
    monitor' <- unsafeManagedPtrCastPtr monitor
    result <- g_network_monitor_get_connectivity monitor'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr monitor
    return result'

#if defined(ENABLE_OVERLOADING)
data NetworkMonitorGetConnectivityMethodInfo
instance (signature ~ (m Gio.Enums.NetworkConnectivity), MonadIO m, IsNetworkMonitor a) => O.OverloadedMethod NetworkMonitorGetConnectivityMethodInfo a signature where
    overloadedMethod = networkMonitorGetConnectivity

instance O.OverloadedMethodInfo NetworkMonitorGetConnectivityMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.NetworkMonitor.networkMonitorGetConnectivity",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-NetworkMonitor.html#v:networkMonitorGetConnectivity"
        })


#endif

-- method NetworkMonitor::get_network_available
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "monitor"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "NetworkMonitor" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the #GNetworkMonitor"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_network_monitor_get_network_available" g_network_monitor_get_network_available :: 
    Ptr NetworkMonitor ->                   -- monitor : TInterface (Name {namespace = "Gio", name = "NetworkMonitor"})
    IO CInt

-- | Checks if the network is available. \"Available\" here means that the
-- system has a default route available for at least one of IPv4 or
-- IPv6. It does not necessarily imply that the public Internet is
-- reachable. See [NetworkMonitor:networkAvailable]("GI.Gio.Interfaces.NetworkMonitor#g:attr:networkAvailable") for more details.
-- 
-- /Since: 2.32/
networkMonitorGetNetworkAvailable ::
    (B.CallStack.HasCallStack, MonadIO m, IsNetworkMonitor a) =>
    a
    -- ^ /@monitor@/: the t'GI.Gio.Interfaces.NetworkMonitor.NetworkMonitor'
    -> m Bool
    -- ^ __Returns:__ whether the network is available
networkMonitorGetNetworkAvailable monitor = liftIO $ do
    monitor' <- unsafeManagedPtrCastPtr monitor
    result <- g_network_monitor_get_network_available monitor'
    let result' = (/= 0) result
    touchManagedPtr monitor
    return result'

#if defined(ENABLE_OVERLOADING)
data NetworkMonitorGetNetworkAvailableMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsNetworkMonitor a) => O.OverloadedMethod NetworkMonitorGetNetworkAvailableMethodInfo a signature where
    overloadedMethod = networkMonitorGetNetworkAvailable

instance O.OverloadedMethodInfo NetworkMonitorGetNetworkAvailableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.NetworkMonitor.networkMonitorGetNetworkAvailable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-NetworkMonitor.html#v:networkMonitorGetNetworkAvailable"
        })


#endif

-- method NetworkMonitor::get_network_metered
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "monitor"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "NetworkMonitor" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the #GNetworkMonitor"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_network_monitor_get_network_metered" g_network_monitor_get_network_metered :: 
    Ptr NetworkMonitor ->                   -- monitor : TInterface (Name {namespace = "Gio", name = "NetworkMonitor"})
    IO CInt

-- | Checks if the network is metered.
-- See [NetworkMonitor:networkMetered]("GI.Gio.Interfaces.NetworkMonitor#g:attr:networkMetered") for more details.
-- 
-- /Since: 2.46/
networkMonitorGetNetworkMetered ::
    (B.CallStack.HasCallStack, MonadIO m, IsNetworkMonitor a) =>
    a
    -- ^ /@monitor@/: the t'GI.Gio.Interfaces.NetworkMonitor.NetworkMonitor'
    -> m Bool
    -- ^ __Returns:__ whether the connection is metered
networkMonitorGetNetworkMetered monitor = liftIO $ do
    monitor' <- unsafeManagedPtrCastPtr monitor
    result <- g_network_monitor_get_network_metered monitor'
    let result' = (/= 0) result
    touchManagedPtr monitor
    return result'

#if defined(ENABLE_OVERLOADING)
data NetworkMonitorGetNetworkMeteredMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsNetworkMonitor a) => O.OverloadedMethod NetworkMonitorGetNetworkMeteredMethodInfo a signature where
    overloadedMethod = networkMonitorGetNetworkMetered

instance O.OverloadedMethodInfo NetworkMonitorGetNetworkMeteredMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.NetworkMonitor.networkMonitorGetNetworkMetered",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-NetworkMonitor.html#v:networkMonitorGetNetworkMetered"
        })


#endif

-- method NetworkMonitor::get_default
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "NetworkMonitor" })
-- throws : False
-- Skip return : False

foreign import ccall "g_network_monitor_get_default" g_network_monitor_get_default :: 
    IO (Ptr NetworkMonitor)

-- | Gets the default t'GI.Gio.Interfaces.NetworkMonitor.NetworkMonitor' for the system.
-- 
-- /Since: 2.32/
networkMonitorGetDefault ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m NetworkMonitor
    -- ^ __Returns:__ a t'GI.Gio.Interfaces.NetworkMonitor.NetworkMonitor', which will be
    --     a dummy object if no network monitor is available
networkMonitorGetDefault  = liftIO $ do
    result <- g_network_monitor_get_default
    checkUnexpectedReturnNULL "networkMonitorGetDefault" result
    result' <- (newObject NetworkMonitor) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- signal NetworkMonitor::network-changed
-- | Emitted when the network configuration changes.
-- 
-- /Since: 2.32/
type NetworkMonitorNetworkChangedCallback =
    Bool
    -- ^ /@networkAvailable@/: the current value of [NetworkMonitor:networkAvailable]("GI.Gio.Interfaces.NetworkMonitor#g:attr:networkAvailable")
    -> IO ()

type C_NetworkMonitorNetworkChangedCallback =
    Ptr NetworkMonitor ->                   -- object
    CInt ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_NetworkMonitorNetworkChangedCallback`.
foreign import ccall "wrapper"
    mk_NetworkMonitorNetworkChangedCallback :: C_NetworkMonitorNetworkChangedCallback -> IO (FunPtr C_NetworkMonitorNetworkChangedCallback)

wrap_NetworkMonitorNetworkChangedCallback :: 
    GObject a => (a -> NetworkMonitorNetworkChangedCallback) ->
    C_NetworkMonitorNetworkChangedCallback
wrap_NetworkMonitorNetworkChangedCallback gi'cb gi'selfPtr networkAvailable _ = do
    let networkAvailable' = (/= 0) networkAvailable
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  networkAvailable'


-- | Connect a signal handler for the [networkChanged](#signal:networkChanged) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' networkMonitor #networkChanged callback
-- @
-- 
-- 
onNetworkMonitorNetworkChanged :: (IsNetworkMonitor a, MonadIO m) => a -> ((?self :: a) => NetworkMonitorNetworkChangedCallback) -> m SignalHandlerId
onNetworkMonitorNetworkChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_NetworkMonitorNetworkChangedCallback wrapped
    wrapped'' <- mk_NetworkMonitorNetworkChangedCallback wrapped'
    connectSignalFunPtr obj "network-changed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [networkChanged](#signal:networkChanged) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' networkMonitor #networkChanged callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterNetworkMonitorNetworkChanged :: (IsNetworkMonitor a, MonadIO m) => a -> ((?self :: a) => NetworkMonitorNetworkChangedCallback) -> m SignalHandlerId
afterNetworkMonitorNetworkChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_NetworkMonitorNetworkChangedCallback wrapped
    wrapped'' <- mk_NetworkMonitorNetworkChangedCallback wrapped'
    connectSignalFunPtr obj "network-changed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data NetworkMonitorNetworkChangedSignalInfo
instance SignalInfo NetworkMonitorNetworkChangedSignalInfo where
    type HaskellCallbackType NetworkMonitorNetworkChangedSignalInfo = NetworkMonitorNetworkChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_NetworkMonitorNetworkChangedCallback cb
        cb'' <- mk_NetworkMonitorNetworkChangedCallback cb'
        connectSignalFunPtr obj "network-changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.NetworkMonitor::network-changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-NetworkMonitor.html#g:signal:networkChanged"})

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList NetworkMonitor = NetworkMonitorSignalList
type NetworkMonitorSignalList = ('[ '("networkChanged", NetworkMonitorNetworkChangedSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif


