// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_fuse.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_fuse__A0__F62

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 64 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_64_3_T_T_0___gfx950)xyzw" };
#else
{ R"xyzw(amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_64_3_T_T_0___gfx950)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp16@16_64_3_T_T_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x7787e787u, 0x15f69baeu, 0, 109 }, // 7787e78715f69bae = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_64_3_T_T_0___gfx950__P__16_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x127478afu, 0xf23c179du, 0, 6 }, // 127478aff23c179d = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_64_3_T_T_0___gfx950__P__16_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xe708665au, 0x39043db3u, 0, 143 }, // e708665a39043db3 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_64_3_T_T_0___gfx950__P__16_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0xcb92cd0au, 0xeda8afe0u, 0, 86 }, // cb92cd0aeda8afe0 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_64_3_T_T_0___gfx950__P__16_16__CO__wave4_warp2_stg1--Arch_gfx950
    { 0x71c6bef7u, 0x049e014eu, 23, 109 }, // 71c6bef7049e014e = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_64_3_T_T_0___gfx950__P__32_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x30b800a2u, 0x6d0ead9cu, 23, 6 }, // 30b800a26d0ead9c = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_64_3_T_T_0___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xe4cbeedau, 0xf79bfab5u, 23, 143 }, // e4cbeedaf79bfab5 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_64_3_T_T_0___gfx950__P__32_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0x9ee73ce4u, 0x31e9f7e9u, 23, 86 }, // 9ee73ce431e9f7e9 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_64_3_T_T_0___gfx950__P__32_16__CO__wave4_warp2_stg1--Arch_gfx950
    { 0xef30e514u, 0x0034d10bu, 103, 109 }, // ef30e5140034d10b = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_64_3_T_T_0___gfx950__P__64_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x47ba893au, 0x76f46868u, 103, 126 }, // 47ba893a76f46868 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_64_3_T_T_0___gfx950__P__64_16__CO__wave1_warp4_stg1--Arch_gfx950
    { 0x27f34b81u, 0x559e233bu, 103, 6 }, // 27f34b81559e233b = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_64_3_T_T_0___gfx950__P__64_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x3ad5f088u, 0xcca501beu, 160, 6 }, // 3ad5f088cca501be = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_64_3_T_T_0___gfx950__P__64_32__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xa0eb46bcu, 0x035a49f9u, 166, 109 }, // a0eb46bc035a49f9 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp16@16_64_3_T_T_0___gfx950__P__64_64__CO__wave1_warp2_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_fuse_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_fuse_packed_string);

static int8_t lut[1][7][7] =
{
// GPU gfx950_mod0
{{ 2, 3, 1, 3, 3, 1, 0},
 { 5, 6, 5, 6, 3, 1, 1},
 { 9,10, 5, 5, 3, 5, 0},
 {10,10, 4, 6, 5, 5, 5},
 { 5, 5, 7, 5, 5, 5, 5},
 { 5, 5,10, 5, 5, 5, 5},
 { 7, 5, 8, 5,11,11,11}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelFuseContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelFuseContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_fuse__A0__F62 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 5;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_fuse__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 5;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = "i32"
// BLOCK_DMODEL = 64
// CAUSAL_TYPE = 3
// ENABLE_DROPOUT = True
// PADDED_HEAD = True
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_fuse WHERE gpu IN ('gfx950_mod0') AND inputs$Q_dtype = 'torch.float16' AND inputs$BLOCK_DMODEL = 64 AND inputs$CAUSAL_TYPE = 3 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

