// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A7__F197

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_16_0_T_F_1___gfx11xx)xyzw" };
#else
{ R"xyzw(amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_16_0_T_F_1___gfx11xx)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_16_0_T_F_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1152)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x98623a1cu, 0x017b0ed5u, 0, 46 }, // 98623a1c017b0ed5 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_16_0_T_F_1___gfx11xx__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1152
    { 0x3840452cu, 0x9c515639u, 40, 46 }, // 3840452c9c515639 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_16_0_T_F_1___gfx11xx__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1152
    { 0x30314ae0u, 0x1265da85u, 40, 63 }, // 30314ae01265da85 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_16_0_T_F_1___gfx11xx__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1152
    { 0xce505da2u, 0x93a00e3du, 40, 6 }, // ce505da293a00e3d = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_16_0_T_F_1___gfx11xx__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1152
    { 0xe667a886u, 0xefd2ff79u, 40, 23 }, // e667a886efd2ff79 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_16_0_T_F_1___gfx11xx__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1152
    { 0x4803e347u, 0x128a68d9u, 97, 138 }, // 4803e347128a68d9 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_16_0_T_F_1___gfx11xx__P__32_32__CO__wave1_warp4_stg1--Arch_gfx1152
    { 0x8b59fc6au, 0x9bec4e69u, 97, 80 }, // 8b59fc6a9bec4e69 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_16_0_T_F_1___gfx11xx__P__32_32__CO__wave2_warp4_stg1--Arch_gfx1152
    { 0x33821fe1u, 0x21e4d2a8u, 97, 155 }, // 33821fe121e4d2a8 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_16_0_T_F_1___gfx11xx__P__32_32__CO__wave3_warp4_stg1--Arch_gfx1152
    { 0x43471a79u, 0xdbe8302cu, 97, 103 }, // 43471a79dbe8302c = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_16_0_T_F_1___gfx11xx__P__32_32__CO__wave4_warp4_stg1--Arch_gfx1152
    { 0x9d43bc64u, 0x5f7dc7ceu, 120, 138 }, // 9d43bc645f7dc7ce = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_16_0_T_F_1___gfx11xx__P__64_16__CO__wave1_warp4_stg1--Arch_gfx1152
    { 0xab0b53bbu, 0xa57c8181u, 120, 80 }, // ab0b53bba57c8181 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_16_0_T_F_1___gfx11xx__P__64_16__CO__wave2_warp4_stg1--Arch_gfx1152
    { 0xd3aedfb9u, 0x814c9a62u, 120, 155 }, // d3aedfb9814c9a62 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_16_0_T_F_1___gfx11xx__P__64_16__CO__wave3_warp4_stg1--Arch_gfx1152
    { 0x81a197dau, 0xbf2e4d1eu, 120, 103 }, // 81a197dabf2e4d1e = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_16_0_T_F_1___gfx11xx__P__64_16__CO__wave4_warp4_stg1--Arch_gfx1152
    { 0xe252db3au, 0x99af4531u, 126, 103 }, // e252db3a99af4531 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_16_0_T_F_1___gfx11xx__P__64_32__CO__wave4_warp4_stg1--Arch_gfx1152
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1152_mod0
{{ 1, 6, 7, 8, 7, 7, 7, 5, 6, 6},
 {12, 3, 0, 3, 7, 2, 2, 2, 1, 1},
 { 3, 3, 3, 3, 2, 3, 7, 2, 3, 4},
 { 2, 3, 2,11, 9, 6, 5, 5, 6, 0},
 { 1, 2, 4,12,10, 7, 8, 7, 0,13},
 {11, 3, 1,11,12, 7, 7, 8, 3, 4},
 {11, 2, 1,12,11, 0, 6, 4, 3, 2},
 {11, 4, 1,10, 9, 3,13, 4, 4, 3},
 {11,12, 2,12,11, 2, 1, 1, 8, 6},
 {12,11, 2,12, 2, 4, 4, 7, 3, 1}}
// End of GPU gfx1152_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A7__F197 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 3;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 3;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = 0
// BLOCK_DMODEL = 16
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = True
// PADDED_HEAD = False
// BIAS_TYPE = 1

// SELECT * FROM FLASH$bwd_kernel_dq WHERE gpu IN ('gfx1100_mod0') AND inputs$Q_dtype = 'torch.bfloat16' AND inputs$BLOCK_DMODEL = 16 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 1

// vim: set fileencoding=utf-8

