﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/panorama/PanoramaRequest.h>
#include <aws/panorama/Panorama_EXPORTS.h>
#include <aws/panorama/model/NetworkPayload.h>

#include <utility>

namespace Aws {
namespace Panorama {
namespace Model {

/**
 */
class ProvisionDeviceRequest : public PanoramaRequest {
 public:
  AWS_PANORAMA_API ProvisionDeviceRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ProvisionDevice"; }

  AWS_PANORAMA_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>A description for the device.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  ProvisionDeviceRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A name for the device.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  ProvisionDeviceRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A networking configuration for the device.</p>
   */
  inline const NetworkPayload& GetNetworkingConfiguration() const { return m_networkingConfiguration; }
  inline bool NetworkingConfigurationHasBeenSet() const { return m_networkingConfigurationHasBeenSet; }
  template <typename NetworkingConfigurationT = NetworkPayload>
  void SetNetworkingConfiguration(NetworkingConfigurationT&& value) {
    m_networkingConfigurationHasBeenSet = true;
    m_networkingConfiguration = std::forward<NetworkingConfigurationT>(value);
  }
  template <typename NetworkingConfigurationT = NetworkPayload>
  ProvisionDeviceRequest& WithNetworkingConfiguration(NetworkingConfigurationT&& value) {
    SetNetworkingConfiguration(std::forward<NetworkingConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags for the device.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  ProvisionDeviceRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  ProvisionDeviceRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_description;

  Aws::String m_name;

  NetworkPayload m_networkingConfiguration;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_descriptionHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_networkingConfigurationHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Panorama
}  // namespace Aws
