{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- Contains the parameters that define a colorstate with cicp parameters.
-- 
-- Cicp parameters are specified in the ITU-T H.273
-- <https://www.itu.int/rec/T-REC-H.273/en specification>.
-- 
-- See the documentation of individual properties for supported values.
-- 
-- The \'unspecified\' value (2) is not treated in any special way, and
-- must be replaced by a different value before creating a color state.
-- 
-- @GdkCicpParams@ can be used as a builder object to construct a color
-- state from Cicp data with 'GI.Gdk.Objects.CicpParams.cicpParamsBuildColorState'.
-- The function will return an error if the given parameters are not
-- supported.
-- 
-- You can obtain a @GdkCicpParams@ object from a color state with
-- 'GI.Gdk.Structs.ColorState.colorStateCreateCicpParams'. This can be used to
-- create a variant of a color state, by changing just one of the cicp
-- parameters, or just to obtain information about the color state.
-- 
-- /Since: 4.16/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gdk.Objects.CicpParams
    ( 

-- * Exported types
    CicpParams(..)                          ,
    IsCicpParams                            ,
    toCicpParams                            ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [buildColorState]("GI.Gdk.Objects.CicpParams#g:method:buildColorState"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getColorPrimaries]("GI.Gdk.Objects.CicpParams#g:method:getColorPrimaries"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getMatrixCoefficients]("GI.Gdk.Objects.CicpParams#g:method:getMatrixCoefficients"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getRange]("GI.Gdk.Objects.CicpParams#g:method:getRange"), [getTransferFunction]("GI.Gdk.Objects.CicpParams#g:method:getTransferFunction").
-- 
-- ==== Setters
-- [setColorPrimaries]("GI.Gdk.Objects.CicpParams#g:method:setColorPrimaries"), [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setMatrixCoefficients]("GI.Gdk.Objects.CicpParams#g:method:setMatrixCoefficients"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty"), [setRange]("GI.Gdk.Objects.CicpParams#g:method:setRange"), [setTransferFunction]("GI.Gdk.Objects.CicpParams#g:method:setTransferFunction").

#if defined(ENABLE_OVERLOADING)
    ResolveCicpParamsMethod                 ,
#endif

-- ** buildColorState #method:buildColorState#

#if defined(ENABLE_OVERLOADING)
    CicpParamsBuildColorStateMethodInfo     ,
#endif
    cicpParamsBuildColorState               ,


-- ** getColorPrimaries #method:getColorPrimaries#

#if defined(ENABLE_OVERLOADING)
    CicpParamsGetColorPrimariesMethodInfo   ,
#endif
    cicpParamsGetColorPrimaries             ,


-- ** getMatrixCoefficients #method:getMatrixCoefficients#

#if defined(ENABLE_OVERLOADING)
    CicpParamsGetMatrixCoefficientsMethodInfo,
#endif
    cicpParamsGetMatrixCoefficients         ,


-- ** getRange #method:getRange#

#if defined(ENABLE_OVERLOADING)
    CicpParamsGetRangeMethodInfo            ,
#endif
    cicpParamsGetRange                      ,


-- ** getTransferFunction #method:getTransferFunction#

#if defined(ENABLE_OVERLOADING)
    CicpParamsGetTransferFunctionMethodInfo ,
#endif
    cicpParamsGetTransferFunction           ,


-- ** new #method:new#

    cicpParamsNew                           ,


-- ** setColorPrimaries #method:setColorPrimaries#

#if defined(ENABLE_OVERLOADING)
    CicpParamsSetColorPrimariesMethodInfo   ,
#endif
    cicpParamsSetColorPrimaries             ,


-- ** setMatrixCoefficients #method:setMatrixCoefficients#

#if defined(ENABLE_OVERLOADING)
    CicpParamsSetMatrixCoefficientsMethodInfo,
#endif
    cicpParamsSetMatrixCoefficients         ,


-- ** setRange #method:setRange#

#if defined(ENABLE_OVERLOADING)
    CicpParamsSetRangeMethodInfo            ,
#endif
    cicpParamsSetRange                      ,


-- ** setTransferFunction #method:setTransferFunction#

#if defined(ENABLE_OVERLOADING)
    CicpParamsSetTransferFunctionMethodInfo ,
#endif
    cicpParamsSetTransferFunction           ,




 -- * Properties


-- ** colorPrimaries #attr:colorPrimaries#
-- | The color primaries to use.
-- 
-- Supported values:
-- 
-- * 1: BT.709 \/ sRGB
-- * 2: unspecified
-- * 5: PAL
-- * 6,7: BT.601 \/ NTSC
-- * 9: BT.2020
-- * 12: Display P3
-- 
-- 
-- /Since: 4.16/

#if defined(ENABLE_OVERLOADING)
    CicpParamsColorPrimariesPropertyInfo    ,
#endif
#if defined(ENABLE_OVERLOADING)
    cicpParamsColorPrimaries                ,
#endif
    constructCicpParamsColorPrimaries       ,
    getCicpParamsColorPrimaries             ,
    setCicpParamsColorPrimaries             ,


-- ** matrixCoefficients #attr:matrixCoefficients#
-- | The matrix coefficients (for YUV to RGB conversion).
-- 
-- Supported values:
-- 
-- * 0: RGB
-- * 1: BT.709
-- * 2: unspecified
-- * 5,6: BT.601
-- * 9: BT.2020
-- 
-- 
-- /Since: 4.16/

#if defined(ENABLE_OVERLOADING)
    CicpParamsMatrixCoefficientsPropertyInfo,
#endif
#if defined(ENABLE_OVERLOADING)
    cicpParamsMatrixCoefficients            ,
#endif
    constructCicpParamsMatrixCoefficients   ,
    getCicpParamsMatrixCoefficients         ,
    setCicpParamsMatrixCoefficients         ,


-- ** range #attr:range#
-- | Whether the data is using the full range of values.
-- 
-- The range of the data.
-- 
-- /Since: 4.16/

#if defined(ENABLE_OVERLOADING)
    CicpParamsRangePropertyInfo             ,
#endif
#if defined(ENABLE_OVERLOADING)
    cicpParamsRange                         ,
#endif
    constructCicpParamsRange                ,
    getCicpParamsRange                      ,
    setCicpParamsRange                      ,


-- ** transferFunction #attr:transferFunction#
-- | The transfer function to use.
-- 
-- Supported values:
-- 
-- * 1,6,14,15: BT.709, BT.601, BT.2020
-- * 2: unspecified
-- * 4: gamma 2.2
-- * 5: gamma 2.8
-- * 8: linear
-- * 13: sRGB
-- * 16: BT.2100 PQ
-- * 18: BT.2100 HLG
-- 
-- 
-- /Since: 4.16/

#if defined(ENABLE_OVERLOADING)
    CicpParamsTransferFunctionPropertyInfo  ,
#endif
#if defined(ENABLE_OVERLOADING)
    cicpParamsTransferFunction              ,
#endif
    constructCicpParamsTransferFunction     ,
    getCicpParamsTransferFunction           ,
    setCicpParamsTransferFunction           ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gdk.Enums as Gdk.Enums
import {-# SOURCE #-} qualified GI.Gdk.Structs.ColorState as Gdk.ColorState

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gdk.Enums as Gdk.Enums
import {-# SOURCE #-} qualified GI.Gdk.Structs.ColorState as Gdk.ColorState

#endif

-- | Memory-managed wrapper type.
newtype CicpParams = CicpParams (SP.ManagedPtr CicpParams)
    deriving (Eq)

instance SP.ManagedPtrNewtype CicpParams where
    toManagedPtr (CicpParams p) = p

foreign import ccall "gdk_cicp_params_get_type"
    c_gdk_cicp_params_get_type :: IO B.Types.GType

instance B.Types.TypedObject CicpParams where
    glibType = c_gdk_cicp_params_get_type

instance B.Types.GObject CicpParams

-- | Type class for types which can be safely cast to t'CicpParams', for instance with `toCicpParams`.
class (SP.GObject o, O.IsDescendantOf CicpParams o) => IsCicpParams o
instance (SP.GObject o, O.IsDescendantOf CicpParams o) => IsCicpParams o

instance O.HasParentTypes CicpParams
type instance O.ParentTypes CicpParams = '[GObject.Object.Object]

-- | Cast to t'CicpParams', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toCicpParams :: (MIO.MonadIO m, IsCicpParams o) => o -> m CicpParams
toCicpParams = MIO.liftIO . B.ManagedPtr.unsafeCastTo CicpParams

-- | Convert t'CicpParams' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe CicpParams) where
    gvalueGType_ = c_gdk_cicp_params_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr CicpParams)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr CicpParams)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject CicpParams ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveCicpParamsMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveCicpParamsMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveCicpParamsMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveCicpParamsMethod "buildColorState" o = CicpParamsBuildColorStateMethodInfo
    ResolveCicpParamsMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveCicpParamsMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveCicpParamsMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveCicpParamsMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveCicpParamsMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveCicpParamsMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveCicpParamsMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveCicpParamsMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveCicpParamsMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveCicpParamsMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveCicpParamsMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveCicpParamsMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveCicpParamsMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveCicpParamsMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveCicpParamsMethod "getColorPrimaries" o = CicpParamsGetColorPrimariesMethodInfo
    ResolveCicpParamsMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveCicpParamsMethod "getMatrixCoefficients" o = CicpParamsGetMatrixCoefficientsMethodInfo
    ResolveCicpParamsMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveCicpParamsMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveCicpParamsMethod "getRange" o = CicpParamsGetRangeMethodInfo
    ResolveCicpParamsMethod "getTransferFunction" o = CicpParamsGetTransferFunctionMethodInfo
    ResolveCicpParamsMethod "setColorPrimaries" o = CicpParamsSetColorPrimariesMethodInfo
    ResolveCicpParamsMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveCicpParamsMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveCicpParamsMethod "setMatrixCoefficients" o = CicpParamsSetMatrixCoefficientsMethodInfo
    ResolveCicpParamsMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveCicpParamsMethod "setRange" o = CicpParamsSetRangeMethodInfo
    ResolveCicpParamsMethod "setTransferFunction" o = CicpParamsSetTransferFunctionMethodInfo
    ResolveCicpParamsMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveCicpParamsMethod t CicpParams, O.OverloadedMethod info CicpParams p) => OL.IsLabel t (CicpParams -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveCicpParamsMethod t CicpParams, O.OverloadedMethod info CicpParams p, R.HasField t CicpParams p) => R.HasField t CicpParams p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveCicpParamsMethod t CicpParams, O.OverloadedMethodInfo info CicpParams) => OL.IsLabel t (O.MethodProxy info CicpParams) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- VVV Prop "color-primaries"
   -- Type: TBasicType TUInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@color-primaries@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' cicpParams #colorPrimaries
-- @
getCicpParamsColorPrimaries :: (MonadIO m, IsCicpParams o) => o -> m Word32
getCicpParamsColorPrimaries obj = MIO.liftIO $ B.Properties.getObjectPropertyUInt32 obj "color-primaries"

-- | Set the value of the “@color-primaries@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' cicpParams [ #colorPrimaries 'Data.GI.Base.Attributes.:=' value ]
-- @
setCicpParamsColorPrimaries :: (MonadIO m, IsCicpParams o) => o -> Word32 -> m ()
setCicpParamsColorPrimaries obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyUInt32 obj "color-primaries" val

-- | Construct a t'GValueConstruct' with valid value for the “@color-primaries@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructCicpParamsColorPrimaries :: (IsCicpParams o, MIO.MonadIO m) => Word32 -> m (GValueConstruct o)
constructCicpParamsColorPrimaries val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyUInt32 "color-primaries" val

#if defined(ENABLE_OVERLOADING)
data CicpParamsColorPrimariesPropertyInfo
instance AttrInfo CicpParamsColorPrimariesPropertyInfo where
    type AttrAllowedOps CicpParamsColorPrimariesPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint CicpParamsColorPrimariesPropertyInfo = IsCicpParams
    type AttrSetTypeConstraint CicpParamsColorPrimariesPropertyInfo = (~) Word32
    type AttrTransferTypeConstraint CicpParamsColorPrimariesPropertyInfo = (~) Word32
    type AttrTransferType CicpParamsColorPrimariesPropertyInfo = Word32
    type AttrGetType CicpParamsColorPrimariesPropertyInfo = Word32
    type AttrLabel CicpParamsColorPrimariesPropertyInfo = "color-primaries"
    type AttrOrigin CicpParamsColorPrimariesPropertyInfo = CicpParams
    attrGet = getCicpParamsColorPrimaries
    attrSet = setCicpParamsColorPrimaries
    attrTransfer _ v = do
        return v
    attrConstruct = constructCicpParamsColorPrimaries
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Objects.CicpParams.colorPrimaries"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Objects-CicpParams.html#g:attr:colorPrimaries"
        })
#endif

-- VVV Prop "matrix-coefficients"
   -- Type: TBasicType TUInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@matrix-coefficients@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' cicpParams #matrixCoefficients
-- @
getCicpParamsMatrixCoefficients :: (MonadIO m, IsCicpParams o) => o -> m Word32
getCicpParamsMatrixCoefficients obj = MIO.liftIO $ B.Properties.getObjectPropertyUInt32 obj "matrix-coefficients"

-- | Set the value of the “@matrix-coefficients@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' cicpParams [ #matrixCoefficients 'Data.GI.Base.Attributes.:=' value ]
-- @
setCicpParamsMatrixCoefficients :: (MonadIO m, IsCicpParams o) => o -> Word32 -> m ()
setCicpParamsMatrixCoefficients obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyUInt32 obj "matrix-coefficients" val

-- | Construct a t'GValueConstruct' with valid value for the “@matrix-coefficients@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructCicpParamsMatrixCoefficients :: (IsCicpParams o, MIO.MonadIO m) => Word32 -> m (GValueConstruct o)
constructCicpParamsMatrixCoefficients val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyUInt32 "matrix-coefficients" val

#if defined(ENABLE_OVERLOADING)
data CicpParamsMatrixCoefficientsPropertyInfo
instance AttrInfo CicpParamsMatrixCoefficientsPropertyInfo where
    type AttrAllowedOps CicpParamsMatrixCoefficientsPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint CicpParamsMatrixCoefficientsPropertyInfo = IsCicpParams
    type AttrSetTypeConstraint CicpParamsMatrixCoefficientsPropertyInfo = (~) Word32
    type AttrTransferTypeConstraint CicpParamsMatrixCoefficientsPropertyInfo = (~) Word32
    type AttrTransferType CicpParamsMatrixCoefficientsPropertyInfo = Word32
    type AttrGetType CicpParamsMatrixCoefficientsPropertyInfo = Word32
    type AttrLabel CicpParamsMatrixCoefficientsPropertyInfo = "matrix-coefficients"
    type AttrOrigin CicpParamsMatrixCoefficientsPropertyInfo = CicpParams
    attrGet = getCicpParamsMatrixCoefficients
    attrSet = setCicpParamsMatrixCoefficients
    attrTransfer _ v = do
        return v
    attrConstruct = constructCicpParamsMatrixCoefficients
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Objects.CicpParams.matrixCoefficients"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Objects-CicpParams.html#g:attr:matrixCoefficients"
        })
#endif

-- VVV Prop "range"
   -- Type: TInterface (Name {namespace = "Gdk", name = "CicpRange"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@range@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' cicpParams #range
-- @
getCicpParamsRange :: (MonadIO m, IsCicpParams o) => o -> m Gdk.Enums.CicpRange
getCicpParamsRange obj = MIO.liftIO $ B.Properties.getObjectPropertyEnum obj "range"

-- | Set the value of the “@range@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' cicpParams [ #range 'Data.GI.Base.Attributes.:=' value ]
-- @
setCicpParamsRange :: (MonadIO m, IsCicpParams o) => o -> Gdk.Enums.CicpRange -> m ()
setCicpParamsRange obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyEnum obj "range" val

-- | Construct a t'GValueConstruct' with valid value for the “@range@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructCicpParamsRange :: (IsCicpParams o, MIO.MonadIO m) => Gdk.Enums.CicpRange -> m (GValueConstruct o)
constructCicpParamsRange val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyEnum "range" val

#if defined(ENABLE_OVERLOADING)
data CicpParamsRangePropertyInfo
instance AttrInfo CicpParamsRangePropertyInfo where
    type AttrAllowedOps CicpParamsRangePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint CicpParamsRangePropertyInfo = IsCicpParams
    type AttrSetTypeConstraint CicpParamsRangePropertyInfo = (~) Gdk.Enums.CicpRange
    type AttrTransferTypeConstraint CicpParamsRangePropertyInfo = (~) Gdk.Enums.CicpRange
    type AttrTransferType CicpParamsRangePropertyInfo = Gdk.Enums.CicpRange
    type AttrGetType CicpParamsRangePropertyInfo = Gdk.Enums.CicpRange
    type AttrLabel CicpParamsRangePropertyInfo = "range"
    type AttrOrigin CicpParamsRangePropertyInfo = CicpParams
    attrGet = getCicpParamsRange
    attrSet = setCicpParamsRange
    attrTransfer _ v = do
        return v
    attrConstruct = constructCicpParamsRange
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Objects.CicpParams.range"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Objects-CicpParams.html#g:attr:range"
        })
#endif

-- VVV Prop "transfer-function"
   -- Type: TBasicType TUInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@transfer-function@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' cicpParams #transferFunction
-- @
getCicpParamsTransferFunction :: (MonadIO m, IsCicpParams o) => o -> m Word32
getCicpParamsTransferFunction obj = MIO.liftIO $ B.Properties.getObjectPropertyUInt32 obj "transfer-function"

-- | Set the value of the “@transfer-function@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' cicpParams [ #transferFunction 'Data.GI.Base.Attributes.:=' value ]
-- @
setCicpParamsTransferFunction :: (MonadIO m, IsCicpParams o) => o -> Word32 -> m ()
setCicpParamsTransferFunction obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyUInt32 obj "transfer-function" val

-- | Construct a t'GValueConstruct' with valid value for the “@transfer-function@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructCicpParamsTransferFunction :: (IsCicpParams o, MIO.MonadIO m) => Word32 -> m (GValueConstruct o)
constructCicpParamsTransferFunction val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyUInt32 "transfer-function" val

#if defined(ENABLE_OVERLOADING)
data CicpParamsTransferFunctionPropertyInfo
instance AttrInfo CicpParamsTransferFunctionPropertyInfo where
    type AttrAllowedOps CicpParamsTransferFunctionPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint CicpParamsTransferFunctionPropertyInfo = IsCicpParams
    type AttrSetTypeConstraint CicpParamsTransferFunctionPropertyInfo = (~) Word32
    type AttrTransferTypeConstraint CicpParamsTransferFunctionPropertyInfo = (~) Word32
    type AttrTransferType CicpParamsTransferFunctionPropertyInfo = Word32
    type AttrGetType CicpParamsTransferFunctionPropertyInfo = Word32
    type AttrLabel CicpParamsTransferFunctionPropertyInfo = "transfer-function"
    type AttrOrigin CicpParamsTransferFunctionPropertyInfo = CicpParams
    attrGet = getCicpParamsTransferFunction
    attrSet = setCicpParamsTransferFunction
    attrTransfer _ v = do
        return v
    attrConstruct = constructCicpParamsTransferFunction
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Objects.CicpParams.transferFunction"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Objects-CicpParams.html#g:attr:transferFunction"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList CicpParams
type instance O.AttributeList CicpParams = CicpParamsAttributeList
type CicpParamsAttributeList = ('[ '("colorPrimaries", CicpParamsColorPrimariesPropertyInfo), '("matrixCoefficients", CicpParamsMatrixCoefficientsPropertyInfo), '("range", CicpParamsRangePropertyInfo), '("transferFunction", CicpParamsTransferFunctionPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
cicpParamsColorPrimaries :: AttrLabelProxy "colorPrimaries"
cicpParamsColorPrimaries = AttrLabelProxy

cicpParamsMatrixCoefficients :: AttrLabelProxy "matrixCoefficients"
cicpParamsMatrixCoefficients = AttrLabelProxy

cicpParamsRange :: AttrLabelProxy "range"
cicpParamsRange = AttrLabelProxy

cicpParamsTransferFunction :: AttrLabelProxy "transferFunction"
cicpParamsTransferFunction = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList CicpParams = CicpParamsSignalList
type CicpParamsSignalList = ('[ '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method CicpParams::new
-- method type : Constructor
-- Args: []
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "CicpParams" })
-- throws : False
-- Skip return : False

foreign import ccall "gdk_cicp_params_new" gdk_cicp_params_new :: 
    IO (Ptr CicpParams)

-- | Creates a new @GdkCicpParams@ object.
-- 
-- The initial values of the properties are the values for \"undefined\"
-- and need to be set before a color state object can be built.
-- 
-- /Since: 4.16/
cicpParamsNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m CicpParams
    -- ^ __Returns:__ a new @GdkCicpParams@
cicpParamsNew  = liftIO $ do
    result <- gdk_cicp_params_new
    checkUnexpectedReturnNULL "cicpParamsNew" result
    result' <- (wrapObject CicpParams) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method CicpParams::build_color_state
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "CicpParams" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkCicpParams`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "ColorState" })
-- throws : True
-- Skip return : False

foreign import ccall "gdk_cicp_params_build_color_state" gdk_cicp_params_build_color_state :: 
    Ptr CicpParams ->                       -- self : TInterface (Name {namespace = "Gdk", name = "CicpParams"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gdk.ColorState.ColorState)

-- | Creates a new @GdkColorState@ object for the cicp parameters in /@self@/.
-- 
-- Note that this may fail if the cicp parameters in /@self@/ are not
-- supported by GTK. In that case, @NULL@ is returned, and /@error@/ is set
-- with an error message that can be presented to the user.
-- 
-- /Since: 4.16/
cicpParamsBuildColorState ::
    (B.CallStack.HasCallStack, MonadIO m, IsCicpParams a) =>
    a
    -- ^ /@self@/: a @GdkCicpParams@
    -> m Gdk.ColorState.ColorState
    -- ^ __Returns:__ A newly allocated @GdkColorState@ /(Can throw 'Data.GI.Base.GError.GError')/
cicpParamsBuildColorState self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    onException (do
        result <- propagateGError $ gdk_cicp_params_build_color_state self'
        checkUnexpectedReturnNULL "cicpParamsBuildColorState" result
        result' <- (wrapBoxed Gdk.ColorState.ColorState) result
        touchManagedPtr self
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data CicpParamsBuildColorStateMethodInfo
instance (signature ~ (m Gdk.ColorState.ColorState), MonadIO m, IsCicpParams a) => O.OverloadedMethod CicpParamsBuildColorStateMethodInfo a signature where
    overloadedMethod = cicpParamsBuildColorState

instance O.OverloadedMethodInfo CicpParamsBuildColorStateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Objects.CicpParams.cicpParamsBuildColorState",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Objects-CicpParams.html#v:cicpParamsBuildColorState"
        })


#endif

-- method CicpParams::get_color_primaries
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "CicpParams" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkCicpParams`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "gdk_cicp_params_get_color_primaries" gdk_cicp_params_get_color_primaries :: 
    Ptr CicpParams ->                       -- self : TInterface (Name {namespace = "Gdk", name = "CicpParams"})
    IO Word32

-- | Returns the value of the color-primaries property
-- of /@self@/.
-- 
-- /Since: 4.16/
cicpParamsGetColorPrimaries ::
    (B.CallStack.HasCallStack, MonadIO m, IsCicpParams a) =>
    a
    -- ^ /@self@/: a @GdkCicpParams@
    -> m Word32
    -- ^ __Returns:__ the color-primaries value
cicpParamsGetColorPrimaries self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gdk_cicp_params_get_color_primaries self'
    touchManagedPtr self
    return result

#if defined(ENABLE_OVERLOADING)
data CicpParamsGetColorPrimariesMethodInfo
instance (signature ~ (m Word32), MonadIO m, IsCicpParams a) => O.OverloadedMethod CicpParamsGetColorPrimariesMethodInfo a signature where
    overloadedMethod = cicpParamsGetColorPrimaries

instance O.OverloadedMethodInfo CicpParamsGetColorPrimariesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Objects.CicpParams.cicpParamsGetColorPrimaries",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Objects-CicpParams.html#v:cicpParamsGetColorPrimaries"
        })


#endif

-- method CicpParams::get_matrix_coefficients
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "CicpParams" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkCicpParams`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "gdk_cicp_params_get_matrix_coefficients" gdk_cicp_params_get_matrix_coefficients :: 
    Ptr CicpParams ->                       -- self : TInterface (Name {namespace = "Gdk", name = "CicpParams"})
    IO Word32

-- | Gets the matrix-coefficients property of /@self@/.
-- 
-- /Since: 4.16/
cicpParamsGetMatrixCoefficients ::
    (B.CallStack.HasCallStack, MonadIO m, IsCicpParams a) =>
    a
    -- ^ /@self@/: a @GdkCicpParams@
    -> m Word32
    -- ^ __Returns:__ the matrix-coefficients value
cicpParamsGetMatrixCoefficients self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gdk_cicp_params_get_matrix_coefficients self'
    touchManagedPtr self
    return result

#if defined(ENABLE_OVERLOADING)
data CicpParamsGetMatrixCoefficientsMethodInfo
instance (signature ~ (m Word32), MonadIO m, IsCicpParams a) => O.OverloadedMethod CicpParamsGetMatrixCoefficientsMethodInfo a signature where
    overloadedMethod = cicpParamsGetMatrixCoefficients

instance O.OverloadedMethodInfo CicpParamsGetMatrixCoefficientsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Objects.CicpParams.cicpParamsGetMatrixCoefficients",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Objects-CicpParams.html#v:cicpParamsGetMatrixCoefficients"
        })


#endif

-- method CicpParams::get_range
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "CicpParams" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkCicpParams`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "CicpRange" })
-- throws : False
-- Skip return : False

foreign import ccall "gdk_cicp_params_get_range" gdk_cicp_params_get_range :: 
    Ptr CicpParams ->                       -- self : TInterface (Name {namespace = "Gdk", name = "CicpParams"})
    IO CUInt

-- | Gets the range property of /@self@/.
-- 
-- /Since: 4.16/
cicpParamsGetRange ::
    (B.CallStack.HasCallStack, MonadIO m, IsCicpParams a) =>
    a
    -- ^ /@self@/: a @GdkCicpParams@
    -> m Gdk.Enums.CicpRange
    -- ^ __Returns:__ the range value
cicpParamsGetRange self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gdk_cicp_params_get_range self'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data CicpParamsGetRangeMethodInfo
instance (signature ~ (m Gdk.Enums.CicpRange), MonadIO m, IsCicpParams a) => O.OverloadedMethod CicpParamsGetRangeMethodInfo a signature where
    overloadedMethod = cicpParamsGetRange

instance O.OverloadedMethodInfo CicpParamsGetRangeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Objects.CicpParams.cicpParamsGetRange",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Objects-CicpParams.html#v:cicpParamsGetRange"
        })


#endif

-- method CicpParams::get_transfer_function
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "CicpParams" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkCicpParams`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "gdk_cicp_params_get_transfer_function" gdk_cicp_params_get_transfer_function :: 
    Ptr CicpParams ->                       -- self : TInterface (Name {namespace = "Gdk", name = "CicpParams"})
    IO Word32

-- | Gets the transfer-function property of /@self@/.
-- 
-- /Since: 4.16/
cicpParamsGetTransferFunction ::
    (B.CallStack.HasCallStack, MonadIO m, IsCicpParams a) =>
    a
    -- ^ /@self@/: a @GdkCicpParams@
    -> m Word32
    -- ^ __Returns:__ the transfer-function value
cicpParamsGetTransferFunction self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gdk_cicp_params_get_transfer_function self'
    touchManagedPtr self
    return result

#if defined(ENABLE_OVERLOADING)
data CicpParamsGetTransferFunctionMethodInfo
instance (signature ~ (m Word32), MonadIO m, IsCicpParams a) => O.OverloadedMethod CicpParamsGetTransferFunctionMethodInfo a signature where
    overloadedMethod = cicpParamsGetTransferFunction

instance O.OverloadedMethodInfo CicpParamsGetTransferFunctionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Objects.CicpParams.cicpParamsGetTransferFunction",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Objects-CicpParams.html#v:cicpParamsGetTransferFunction"
        })


#endif

-- method CicpParams::set_color_primaries
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "CicpParams" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkCicpParams`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "color_primaries"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new color primaries value"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gdk_cicp_params_set_color_primaries" gdk_cicp_params_set_color_primaries :: 
    Ptr CicpParams ->                       -- self : TInterface (Name {namespace = "Gdk", name = "CicpParams"})
    Word32 ->                               -- color_primaries : TBasicType TUInt
    IO ()

-- | Sets the color-primaries property of /@self@/.
-- 
-- /Since: 4.16/
cicpParamsSetColorPrimaries ::
    (B.CallStack.HasCallStack, MonadIO m, IsCicpParams a) =>
    a
    -- ^ /@self@/: a @GdkCicpParams@
    -> Word32
    -- ^ /@colorPrimaries@/: the new color primaries value
    -> m ()
cicpParamsSetColorPrimaries self colorPrimaries = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    gdk_cicp_params_set_color_primaries self' colorPrimaries
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data CicpParamsSetColorPrimariesMethodInfo
instance (signature ~ (Word32 -> m ()), MonadIO m, IsCicpParams a) => O.OverloadedMethod CicpParamsSetColorPrimariesMethodInfo a signature where
    overloadedMethod = cicpParamsSetColorPrimaries

instance O.OverloadedMethodInfo CicpParamsSetColorPrimariesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Objects.CicpParams.cicpParamsSetColorPrimaries",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Objects-CicpParams.html#v:cicpParamsSetColorPrimaries"
        })


#endif

-- method CicpParams::set_matrix_coefficients
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "CicpParams" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation { rawDocText = Nothing , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "matrix_coefficients"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new matrix-coefficients value"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gdk_cicp_params_set_matrix_coefficients" gdk_cicp_params_set_matrix_coefficients :: 
    Ptr CicpParams ->                       -- self : TInterface (Name {namespace = "Gdk", name = "CicpParams"})
    Word32 ->                               -- matrix_coefficients : TBasicType TUInt
    IO ()

-- | /@self@/ a @GdkCicpParams@
-- Sets the matrix-coefficients property of /@self@/.
-- 
-- /Since: 4.16/
cicpParamsSetMatrixCoefficients ::
    (B.CallStack.HasCallStack, MonadIO m, IsCicpParams a) =>
    a
    -> Word32
    -- ^ /@matrixCoefficients@/: the new matrix-coefficients value
    -> m ()
cicpParamsSetMatrixCoefficients self matrixCoefficients = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    gdk_cicp_params_set_matrix_coefficients self' matrixCoefficients
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data CicpParamsSetMatrixCoefficientsMethodInfo
instance (signature ~ (Word32 -> m ()), MonadIO m, IsCicpParams a) => O.OverloadedMethod CicpParamsSetMatrixCoefficientsMethodInfo a signature where
    overloadedMethod = cicpParamsSetMatrixCoefficients

instance O.OverloadedMethodInfo CicpParamsSetMatrixCoefficientsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Objects.CicpParams.cicpParamsSetMatrixCoefficients",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Objects-CicpParams.html#v:cicpParamsSetMatrixCoefficients"
        })


#endif

-- method CicpParams::set_range
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "CicpParams" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkCipParams`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "range"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "CicpRange" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the range value" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gdk_cicp_params_set_range" gdk_cicp_params_set_range :: 
    Ptr CicpParams ->                       -- self : TInterface (Name {namespace = "Gdk", name = "CicpParams"})
    CUInt ->                                -- range : TInterface (Name {namespace = "Gdk", name = "CicpRange"})
    IO ()

-- | Sets the range property of /@self@/
-- 
-- /Since: 4.16/
cicpParamsSetRange ::
    (B.CallStack.HasCallStack, MonadIO m, IsCicpParams a) =>
    a
    -- ^ /@self@/: a @GdkCipParams@
    -> Gdk.Enums.CicpRange
    -- ^ /@range@/: the range value
    -> m ()
cicpParamsSetRange self range = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let range' = (fromIntegral . fromEnum) range
    gdk_cicp_params_set_range self' range'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data CicpParamsSetRangeMethodInfo
instance (signature ~ (Gdk.Enums.CicpRange -> m ()), MonadIO m, IsCicpParams a) => O.OverloadedMethod CicpParamsSetRangeMethodInfo a signature where
    overloadedMethod = cicpParamsSetRange

instance O.OverloadedMethodInfo CicpParamsSetRangeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Objects.CicpParams.cicpParamsSetRange",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Objects-CicpParams.html#v:cicpParamsSetRange"
        })


#endif

-- method CicpParams::set_transfer_function
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "CicpParams" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkCicpParams`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "transfer_function"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new transfer-function value"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gdk_cicp_params_set_transfer_function" gdk_cicp_params_set_transfer_function :: 
    Ptr CicpParams ->                       -- self : TInterface (Name {namespace = "Gdk", name = "CicpParams"})
    Word32 ->                               -- transfer_function : TBasicType TUInt
    IO ()

-- | Sets the transfer-function property of /@self@/.
-- 
-- /Since: 4.16/
cicpParamsSetTransferFunction ::
    (B.CallStack.HasCallStack, MonadIO m, IsCicpParams a) =>
    a
    -- ^ /@self@/: a @GdkCicpParams@
    -> Word32
    -- ^ /@transferFunction@/: the new transfer-function value
    -> m ()
cicpParamsSetTransferFunction self transferFunction = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    gdk_cicp_params_set_transfer_function self' transferFunction
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data CicpParamsSetTransferFunctionMethodInfo
instance (signature ~ (Word32 -> m ()), MonadIO m, IsCicpParams a) => O.OverloadedMethod CicpParamsSetTransferFunctionMethodInfo a signature where
    overloadedMethod = cicpParamsSetTransferFunction

instance O.OverloadedMethodInfo CicpParamsSetTransferFunctionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Objects.CicpParams.cicpParamsSetTransferFunction",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Objects-CicpParams.html#v:cicpParamsSetTransferFunction"
        })


#endif


