

-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Pango.Functions
    ( 

 -- * Methods


-- ** attrAllowBreaksNew #method:attrAllowBreaksNew#

    attrAllowBreaksNew                      ,


-- ** attrBackgroundAlphaNew #method:attrBackgroundAlphaNew#

    attrBackgroundAlphaNew                  ,


-- ** attrBackgroundNew #method:attrBackgroundNew#

    attrBackgroundNew                       ,


-- ** attrBaselineShiftNew #method:attrBaselineShiftNew#

    attrBaselineShiftNew                    ,


-- ** attrBreak #method:attrBreak#

    attrBreak                               ,


-- ** attrFallbackNew #method:attrFallbackNew#

    attrFallbackNew                         ,


-- ** attrFamilyNew #method:attrFamilyNew#

    attrFamilyNew                           ,


-- ** attrFontScaleNew #method:attrFontScaleNew#

    attrFontScaleNew                        ,


-- ** attrForegroundAlphaNew #method:attrForegroundAlphaNew#

    attrForegroundAlphaNew                  ,


-- ** attrForegroundNew #method:attrForegroundNew#

    attrForegroundNew                       ,


-- ** attrGravityHintNew #method:attrGravityHintNew#

    attrGravityHintNew                      ,


-- ** attrGravityNew #method:attrGravityNew#

    attrGravityNew                          ,


-- ** attrInsertHyphensNew #method:attrInsertHyphensNew#

    attrInsertHyphensNew                    ,


-- ** attrLetterSpacingNew #method:attrLetterSpacingNew#

    attrLetterSpacingNew                    ,


-- ** attrLineHeightNew #method:attrLineHeightNew#

    attrLineHeightNew                       ,


-- ** attrLineHeightNewAbsolute #method:attrLineHeightNewAbsolute#

    attrLineHeightNewAbsolute               ,


-- ** attrOverlineColorNew #method:attrOverlineColorNew#

    attrOverlineColorNew                    ,


-- ** attrOverlineNew #method:attrOverlineNew#

    attrOverlineNew                         ,


-- ** attrRiseNew #method:attrRiseNew#

    attrRiseNew                             ,


-- ** attrScaleNew #method:attrScaleNew#

    attrScaleNew                            ,


-- ** attrSentenceNew #method:attrSentenceNew#

    attrSentenceNew                         ,


-- ** attrShowNew #method:attrShowNew#

    attrShowNew                             ,


-- ** attrStretchNew #method:attrStretchNew#

    attrStretchNew                          ,


-- ** attrStrikethroughColorNew #method:attrStrikethroughColorNew#

    attrStrikethroughColorNew               ,


-- ** attrStrikethroughNew #method:attrStrikethroughNew#

    attrStrikethroughNew                    ,


-- ** attrStyleNew #method:attrStyleNew#

    attrStyleNew                            ,


-- ** attrTextTransformNew #method:attrTextTransformNew#

    attrTextTransformNew                    ,


-- ** attrUnderlineColorNew #method:attrUnderlineColorNew#

    attrUnderlineColorNew                   ,


-- ** attrUnderlineNew #method:attrUnderlineNew#

    attrUnderlineNew                        ,


-- ** attrVariantNew #method:attrVariantNew#

    attrVariantNew                          ,


-- ** attrWeightNew #method:attrWeightNew#

    attrWeightNew                           ,


-- ** attrWordNew #method:attrWordNew#

    attrWordNew                             ,


-- ** break #method:break#

    break                                   ,


-- ** defaultBreak #method:defaultBreak#

    defaultBreak                            ,


-- ** findBaseDir #method:findBaseDir#

    findBaseDir                             ,


-- ** findParagraphBoundary #method:findParagraphBoundary#

    findParagraphBoundary                   ,


-- ** getLogAttrs #method:getLogAttrs#

    getLogAttrs                             ,


-- ** getMirrorChar #method:getMirrorChar#

    getMirrorChar                           ,


-- ** isZeroWidth #method:isZeroWidth#

    isZeroWidth                             ,


-- ** itemize #method:itemize#

    itemize                                 ,


-- ** itemizeWithBaseDir #method:itemizeWithBaseDir#

    itemizeWithBaseDir                      ,


-- ** markupParserFinish #method:markupParserFinish#

    markupParserFinish                      ,


-- ** markupParserNew #method:markupParserNew#

    markupParserNew                         ,


-- ** parseEnum #method:parseEnum#

    parseEnum                               ,


-- ** parseMarkup #method:parseMarkup#

    parseMarkup                             ,


-- ** parseStretch #method:parseStretch#

    parseStretch                            ,


-- ** parseStyle #method:parseStyle#

    parseStyle                              ,


-- ** parseVariant #method:parseVariant#

    parseVariant                            ,


-- ** parseWeight #method:parseWeight#

    parseWeight                             ,


-- ** quantizeLineGeometry #method:quantizeLineGeometry#

    quantizeLineGeometry                    ,


-- ** readLine #method:readLine#

    readLine                                ,


-- ** reorderItems #method:reorderItems#

    reorderItems                            ,


-- ** scanInt #method:scanInt#

    scanInt                                 ,


-- ** scanString #method:scanString#

    scanString                              ,


-- ** scanWord #method:scanWord#

    scanWord                                ,


-- ** shape #method:shape#

    shape                                   ,


-- ** shapeFull #method:shapeFull#

    shapeFull                               ,


-- ** shapeItem #method:shapeItem#

    shapeItem                               ,


-- ** shapeWithFlags #method:shapeWithFlags#

    shapeWithFlags                          ,


-- ** skipSpace #method:skipSpace#

    skipSpace                               ,


-- ** splitFileList #method:splitFileList#

    splitFileList                           ,


-- ** tailorBreak #method:tailorBreak#

    tailorBreak                             ,


-- ** trimString #method:trimString#

    trimString                              ,


-- ** unicharDirection #method:unicharDirection#

    unicharDirection                        ,


-- ** unitsFromDouble #method:unitsFromDouble#

    unitsFromDouble                         ,


-- ** unitsToDouble #method:unitsToDouble#

    unitsToDouble                           ,


-- ** version #method:version#

    version                                 ,


-- ** versionCheck #method:versionCheck#

    versionCheck                            ,


-- ** versionString #method:versionString#

    versionString                           ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GLib.Structs.MarkupParseContext as GLib.MarkupParseContext
import qualified GI.GLib.Structs.String as GLib.String
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Interfaces.ListModel as Gio.ListModel
import qualified GI.HarfBuzz.Structs.FeatureT as HarfBuzz.FeatureT
import qualified GI.Pango.Callbacks as Pango.Callbacks
import {-# SOURCE #-} qualified GI.Pango.Enums as Pango.Enums
import {-# SOURCE #-} qualified GI.Pango.Flags as Pango.Flags
import {-# SOURCE #-} qualified GI.Pango.Objects.Context as Pango.Context
import {-# SOURCE #-} qualified GI.Pango.Objects.Coverage as Pango.Coverage
import {-# SOURCE #-} qualified GI.Pango.Objects.Font as Pango.Font
import {-# SOURCE #-} qualified GI.Pango.Objects.FontFace as Pango.FontFace
import {-# SOURCE #-} qualified GI.Pango.Objects.FontFamily as Pango.FontFamily
import {-# SOURCE #-} qualified GI.Pango.Objects.FontMap as Pango.FontMap
import {-# SOURCE #-} qualified GI.Pango.Objects.Fontset as Pango.Fontset
import {-# SOURCE #-} qualified GI.Pango.Structs.Analysis as Pango.Analysis
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrClass as Pango.AttrClass
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrColor as Pango.AttrColor
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrFloat as Pango.AttrFloat
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrFontDesc as Pango.AttrFontDesc
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrFontFeatures as Pango.AttrFontFeatures
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrInt as Pango.AttrInt
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrIterator as Pango.AttrIterator
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrLanguage as Pango.AttrLanguage
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrList as Pango.AttrList
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrShape as Pango.AttrShape
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrSize as Pango.AttrSize
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrString as Pango.AttrString
import {-# SOURCE #-} qualified GI.Pango.Structs.Attribute as Pango.Attribute
import {-# SOURCE #-} qualified GI.Pango.Structs.Color as Pango.Color
import {-# SOURCE #-} qualified GI.Pango.Structs.FontDescription as Pango.FontDescription
import {-# SOURCE #-} qualified GI.Pango.Structs.FontMetrics as Pango.FontMetrics
import {-# SOURCE #-} qualified GI.Pango.Structs.GlyphString as Pango.GlyphString
import {-# SOURCE #-} qualified GI.Pango.Structs.Item as Pango.Item
import {-# SOURCE #-} qualified GI.Pango.Structs.Language as Pango.Language
import {-# SOURCE #-} qualified GI.Pango.Structs.LogAttr as Pango.LogAttr
import {-# SOURCE #-} qualified GI.Pango.Structs.Matrix as Pango.Matrix
import {-# SOURCE #-} qualified GI.Pango.Structs.Rectangle as Pango.Rectangle

#else
import qualified GI.GLib.Structs.MarkupParseContext as GLib.MarkupParseContext
import qualified GI.GLib.Structs.String as GLib.String
import {-# SOURCE #-} qualified GI.Pango.Enums as Pango.Enums
import {-# SOURCE #-} qualified GI.Pango.Flags as Pango.Flags
import {-# SOURCE #-} qualified GI.Pango.Objects.Context as Pango.Context
import {-# SOURCE #-} qualified GI.Pango.Structs.Analysis as Pango.Analysis
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrIterator as Pango.AttrIterator
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrList as Pango.AttrList
import {-# SOURCE #-} qualified GI.Pango.Structs.Attribute as Pango.Attribute
import {-# SOURCE #-} qualified GI.Pango.Structs.GlyphString as Pango.GlyphString
import {-# SOURCE #-} qualified GI.Pango.Structs.Item as Pango.Item
import {-# SOURCE #-} qualified GI.Pango.Structs.Language as Pango.Language
import {-# SOURCE #-} qualified GI.Pango.Structs.LogAttr as Pango.LogAttr

#endif

-- function version_string
-- Args: []
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "pango_version_string" pango_version_string :: 
    IO CString

-- | Returns the version of Pango available at run-time.
-- 
-- This is similar to the macro 'GI.Pango.Constants.VERSION_STRING' except that the
-- macro returns the version available at compile-time.
-- 
-- /Since: 1.16/
versionString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m T.Text
    -- ^ __Returns:__ A string containing the version of Pango library available
    --   at run time. The returned string is owned by Pango and should not
    --   be modified or freed.
versionString  = liftIO $ do
    result <- pango_version_string
    checkUnexpectedReturnNULL "versionString" result
    result' <- cstringToText result
    return result'


-- function version_check
-- Args: [ Arg
--           { argCName = "required_major"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the required major version"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "required_minor"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the required minor version"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "required_micro"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the required major version"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "pango_version_check" pango_version_check :: 
    Int32 ->                                -- required_major : TBasicType TInt
    Int32 ->                                -- required_minor : TBasicType TInt
    Int32 ->                                -- required_micro : TBasicType TInt
    IO CString

-- | Checks that the Pango library in use is compatible with the
-- given version.
-- 
-- Generally you would pass in the constants 'GI.Pango.Constants.VERSION_MAJOR',
-- 'GI.Pango.Constants.VERSION_MINOR', 'GI.Pango.Constants.VERSION_MICRO' as the three arguments
-- to this function; that produces a check that the library in use at
-- run-time is compatible with the version of Pango the application or
-- module was compiled against.
-- 
-- Compatibility is defined by two things: first the version
-- of the running library is newer than the version
-- /@requiredMajor@/.required_minor./@requiredMicro@/. Second
-- the running library must be binary compatible with the
-- version /@requiredMajor@/.required_minor./@requiredMicro@/
-- (same major version.)
-- 
-- For compile-time version checking use @/PANGO_VERSION_CHECK()/@.
-- 
-- /Since: 1.16/
versionCheck ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Int32
    -- ^ /@requiredMajor@/: the required major version
    -> Int32
    -- ^ /@requiredMinor@/: the required minor version
    -> Int32
    -- ^ /@requiredMicro@/: the required major version
    -> m (Maybe T.Text)
    -- ^ __Returns:__ 'P.Nothing' if the Pango library is compatible
    --   with the given version, or a string describing the version
    --   mismatch.  The returned string is owned by Pango and should not
    --   be modified or freed.
versionCheck requiredMajor requiredMinor requiredMicro = liftIO $ do
    result <- pango_version_check requiredMajor requiredMinor requiredMicro
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    return maybeResult


-- function version
-- Args: []
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "pango_version" pango_version :: 
    IO Int32

-- | Returns the encoded version of Pango available at run-time.
-- 
-- This is similar to the macro @/PANGO_VERSION/@ except that the macro
-- returns the encoded version available at compile-time. A version
-- number can be encoded into an integer using @/PANGO_VERSION_ENCODE()/@.
-- 
-- /Since: 1.16/
version ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m Int32
    -- ^ __Returns:__ The encoded version of Pango library available at run time.
version  = liftIO $ do
    result <- pango_version
    return result


-- function units_to_double
-- Args: [ Arg
--           { argCName = "i"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "value in Pango units"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TDouble)
-- throws : False
-- Skip return : False

foreign import ccall "pango_units_to_double" pango_units_to_double :: 
    Int32 ->                                -- i : TBasicType TInt
    IO CDouble

-- | Converts a number in Pango units to floating-point.
-- 
-- The conversion is done by dividing /@i@/ by 'GI.Pango.Constants.SCALE'.
-- 
-- /Since: 1.16/
unitsToDouble ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Int32
    -- ^ /@i@/: value in Pango units
    -> m Double
    -- ^ __Returns:__ the double value.
unitsToDouble i = liftIO $ do
    result <- pango_units_to_double i
    let result' = realToFrac result
    return result'


-- function units_from_double
-- Args: [ Arg
--           { argCName = "d"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "double floating-point value"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "pango_units_from_double" pango_units_from_double :: 
    CDouble ->                              -- d : TBasicType TDouble
    IO Int32

-- | Converts a floating-point number to Pango units.
-- 
-- The conversion is done by multiplying /@d@/ by 'GI.Pango.Constants.SCALE' and
-- rounding the result to nearest integer.
-- 
-- /Since: 1.16/
unitsFromDouble ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Double
    -- ^ /@d@/: double floating-point value
    -> m Int32
    -- ^ __Returns:__ the value in Pango units.
unitsFromDouble d = liftIO $ do
    let d' = realToFrac d
    result <- pango_units_from_double d'
    return result


-- function unichar_direction
-- Args: [ Arg
--           { argCName = "ch"
--           , argType = TBasicType TUniChar
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a Unicode character"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Direction" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_unichar_direction" pango_unichar_direction :: 
    CInt ->                                 -- ch : TBasicType TUniChar
    IO CUInt

-- | Determines the inherent direction of a character.
-- 
-- The inherent direction is either @PANGO_DIRECTION_LTR@, @PANGO_DIRECTION_RTL@,
-- or @PANGO_DIRECTION_NEUTRAL@.
-- 
-- This function is useful to categorize characters into left-to-right
-- letters, right-to-left letters, and everything else. If full Unicode
-- bidirectional type of a character is needed, t'GI.Pango.Enums.BidiType'.@/for_unichar/@()
-- can be used instead.
unicharDirection ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Char
    -- ^ /@ch@/: a Unicode character
    -> m Pango.Enums.Direction
    -- ^ __Returns:__ the direction of the character.
unicharDirection ch = liftIO $ do
    let ch' = (P.fromIntegral . SP.ord) ch
    result <- pango_unichar_direction ch'
    let result' = (toEnum . fromIntegral) result
    return result'


-- function trim_string
-- Args: [ Arg
--           { argCName = "str"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "pango_trim_string" pango_trim_string :: 
    CString ->                              -- str : TBasicType TUTF8
    IO CString

{-# DEPRECATED trimString ["(Since version 1.38)"] #-}
-- | Trims leading and trailing whitespace from a string.
trimString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@str@/: a string
    -> m T.Text
    -- ^ __Returns:__ A newly-allocated string that must be freed with 'GI.GLib.Functions.free'
trimString str = liftIO $ do
    str' <- textToCString str
    result <- pango_trim_string str'
    checkUnexpectedReturnNULL "trimString" result
    result' <- cstringToText result
    freeMem result
    freeMem str'
    return result'


-- function tailor_break
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "text to process. Must be valid UTF-8"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "length in bytes of @text"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "analysis"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Analysis" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "`PangoAnalysis` for @text"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "Byte offset of @text from the beginning of the\n  paragraph, or -1 to ignore attributes from @analysis"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attrs"
--           , argType =
--               TCArray
--                 False
--                 (-1)
--                 5
--                 (TInterface Name { namespace = "Pango" , name = "LogAttr" })
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "array with one\n  `PangoLogAttr` per character in @text, plus one extra, to be filled in"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attrs_len"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "length of @attrs array"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "attrs_len"
--              , argType = TBasicType TInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "length of @attrs array"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_tailor_break" pango_tailor_break :: 
    CString ->                              -- text : TBasicType TUTF8
    Int32 ->                                -- length : TBasicType TInt
    Ptr Pango.Analysis.Analysis ->          -- analysis : TInterface (Name {namespace = "Pango", name = "Analysis"})
    Int32 ->                                -- offset : TBasicType TInt
    Ptr Pango.LogAttr.LogAttr ->            -- attrs : TCArray False (-1) 5 (TInterface (Name {namespace = "Pango", name = "LogAttr"}))
    Int32 ->                                -- attrs_len : TBasicType TInt
    IO ()

-- | Apply language-specific tailoring to the breaks in /@attrs@/.
-- 
-- The line breaks are assumed to have been produced by 'GI.Pango.Functions.defaultBreak'.
-- 
-- If /@offset@/ is not -1, it is used to apply attributes from /@analysis@/ that are
-- relevant to line breaking.
-- 
-- Note that it is better to pass -1 for /@offset@/ and use 'GI.Pango.Functions.attrBreak'
-- to apply attributes to the whole paragraph.
-- 
-- /Since: 1.44/
tailorBreak ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@text@/: text to process. Must be valid UTF-8
    -> Int32
    -- ^ /@length@/: length in bytes of /@text@/
    -> Pango.Analysis.Analysis
    -- ^ /@analysis@/: @PangoAnalysis@ for /@text@/
    -> Int32
    -- ^ /@offset@/: Byte offset of /@text@/ from the beginning of the
    --   paragraph, or -1 to ignore attributes from /@analysis@/
    -> [Pango.LogAttr.LogAttr]
    -- ^ /@attrs@/: array with one
    --   @PangoLogAttr@ per character in /@text@/, plus one extra, to be filled in
    -> m ([Pango.LogAttr.LogAttr])
tailorBreak text length_ analysis offset attrs = liftIO $ do
    let attrsLen = fromIntegral $ P.length attrs
    text' <- textToCString text
    analysis' <- unsafeManagedPtrGetPtr analysis
    attrs' <- mapM unsafeManagedPtrGetPtr attrs
    attrs'' <- packBlockArray 64 attrs'
    pango_tailor_break text' length_ analysis' offset attrs'' attrsLen
    attrs''' <- (unpackBlockArrayWithLength 64 attrsLen) attrs''
    attrs'''' <- mapM (wrapPtr Pango.LogAttr.LogAttr) attrs'''
    freeMem attrs''
    touchManagedPtr analysis
    mapM_ touchManagedPtr attrs
    freeMem text'
    return attrs''''


-- function split_file_list
-- Args: [ Arg
--           { argCName = "str"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a %G_SEARCHPATH_SEPARATOR separated list of filenames"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TCArray True (-1) (-1) (TBasicType TUTF8))
-- throws : False
-- Skip return : False

foreign import ccall "pango_split_file_list" pango_split_file_list :: 
    CString ->                              -- str : TBasicType TUTF8
    IO (Ptr CString)

{-# DEPRECATED splitFileList ["(Since version 1.38)"] #-}
-- | Splits a 'GI.GLib.Constants.SEARCHPATH_SEPARATOR'-separated list of files, stripping
-- white space and substituting ~\/ with $HOME\/.
splitFileList ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@str@/: a 'GI.GLib.Constants.SEARCHPATH_SEPARATOR' separated list of filenames
    -> m [T.Text]
    -- ^ __Returns:__ a list of
    --   strings to be freed with 'GI.GLib.Functions.strfreev'
splitFileList str = liftIO $ do
    str' <- textToCString str
    result <- pango_split_file_list str'
    checkUnexpectedReturnNULL "splitFileList" result
    result' <- unpackZeroTerminatedUTF8CArray result
    mapZeroTerminatedCArray freeMem result
    freeMem result
    freeMem str'
    return result'


-- function skip_space
-- Args: [ Arg
--           { argCName = "pos"
--           , argType = TBasicType TUTF8
--           , direction = DirectionInout
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "in/out string position"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "pango_skip_space" pango_skip_space :: 
    Ptr CString ->                          -- pos : TBasicType TUTF8
    IO CInt

{-# DEPRECATED skipSpace ["(Since version 1.38)"] #-}
-- | Skips 0 or more characters of white space.
skipSpace ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@pos@/: in\/out string position
    -> m ((Bool, T.Text))
    -- ^ __Returns:__ 'P.False' if skipping the white space leaves
    --   the position at a \'\\0\' character.
skipSpace pos = liftIO $ do
    pos' <- textToCString pos
    pos'' <- allocMem :: IO (Ptr CString)
    poke pos'' pos'
    result <- pango_skip_space pos''
    let result' = (/= 0) result
    pos''' <- peek pos''
    pos'''' <- cstringToText pos'''
    freeMem pos''
    return (result', pos'''')


-- function shape_with_flags
-- Args: [ Arg
--           { argCName = "item_text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "valid UTF-8 text to shape"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "item_length"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the length (in bytes) of @item_text.\n    -1 means nul-terminated text."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "paragraph_text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "text of the paragraph (see details)."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "paragraph_length"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the length (in bytes) of @paragraph_text.\n    -1 means nul-terminated text."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "analysis"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Analysis" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "`PangoAnalysis` structure from [func@Pango.itemize]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "glyphs"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "GlyphString" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "glyph string in which to store results"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "ShapeFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags influencing the shaping process"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_shape_with_flags" pango_shape_with_flags :: 
    CString ->                              -- item_text : TBasicType TUTF8
    Int32 ->                                -- item_length : TBasicType TInt
    CString ->                              -- paragraph_text : TBasicType TUTF8
    Int32 ->                                -- paragraph_length : TBasicType TInt
    Ptr Pango.Analysis.Analysis ->          -- analysis : TInterface (Name {namespace = "Pango", name = "Analysis"})
    Ptr Pango.GlyphString.GlyphString ->    -- glyphs : TInterface (Name {namespace = "Pango", name = "GlyphString"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Pango", name = "ShapeFlags"})
    IO ()

-- | Convert the characters in /@text@/ into glyphs.
-- 
-- Given a segment of text and the corresponding @PangoAnalysis@ structure
-- returned from 'GI.Pango.Functions.itemize', convert the characters into glyphs.
-- You may also pass in only a substring of the item from 'GI.Pango.Functions.itemize'.
-- 
-- This is similar to 'GI.Pango.Functions.shapeFull', except it also takes flags
-- that can influence the shaping process.
-- 
-- Some aspects of hyphen insertion and text transformation (in particular,
-- capitalization) require log attrs, and thus can only be handled by
-- 'GI.Pango.Functions.shapeItem'.
-- 
-- Note that the extra attributes in the /@analyis@/ that is returned from
-- 'GI.Pango.Functions.itemize' have indices that are relative to the entire paragraph,
-- so you do not pass the full paragraph text as /@paragraphText@/, you need
-- to subtract the item offset from their indices before calling
-- 'GI.Pango.Functions.shapeWithFlags'.
-- 
-- /Since: 1.44/
shapeWithFlags ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@itemText@/: valid UTF-8 text to shape
    -> Int32
    -- ^ /@itemLength@/: the length (in bytes) of /@itemText@/.
    --     -1 means nul-terminated text.
    -> Maybe (T.Text)
    -- ^ /@paragraphText@/: text of the paragraph (see details).
    -> Int32
    -- ^ /@paragraphLength@/: the length (in bytes) of /@paragraphText@/.
    --     -1 means nul-terminated text.
    -> Pango.Analysis.Analysis
    -- ^ /@analysis@/: @PangoAnalysis@ structure from 'GI.Pango.Functions.itemize'
    -> [Pango.Flags.ShapeFlags]
    -- ^ /@flags@/: flags influencing the shaping process
    -> m (Pango.GlyphString.GlyphString)
shapeWithFlags itemText itemLength paragraphText paragraphLength analysis flags = liftIO $ do
    itemText' <- textToCString itemText
    maybeParagraphText <- case paragraphText of
        Nothing -> return FP.nullPtr
        Just jParagraphText -> do
            jParagraphText' <- textToCString jParagraphText
            return jParagraphText'
    analysis' <- unsafeManagedPtrGetPtr analysis
    glyphs <- SP.callocBoxedBytes 32 :: IO (Ptr Pango.GlyphString.GlyphString)
    let flags' = gflagsToWord flags
    pango_shape_with_flags itemText' itemLength maybeParagraphText paragraphLength analysis' glyphs flags'
    glyphs' <- (wrapBoxed Pango.GlyphString.GlyphString) glyphs
    touchManagedPtr analysis
    freeMem itemText'
    freeMem maybeParagraphText
    return glyphs'


-- function shape_item
-- Args: [ Arg
--           { argCName = "item"
--           , argType = TInterface Name { namespace = "Pango" , name = "Item" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "`PangoItem` to shape"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "paragraph_text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "text of the paragraph (see details)."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "paragraph_length"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the length (in bytes) of @paragraph_text.\n    -1 means nul-terminated text."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "log_attrs"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "LogAttr" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "array of `PangoLogAttr` for @item"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "glyphs"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "GlyphString" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "glyph string in which to store results"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "ShapeFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags influencing the shaping process"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_shape_item" pango_shape_item :: 
    Ptr Pango.Item.Item ->                  -- item : TInterface (Name {namespace = "Pango", name = "Item"})
    CString ->                              -- paragraph_text : TBasicType TUTF8
    Int32 ->                                -- paragraph_length : TBasicType TInt
    Ptr Pango.LogAttr.LogAttr ->            -- log_attrs : TInterface (Name {namespace = "Pango", name = "LogAttr"})
    Ptr Pango.GlyphString.GlyphString ->    -- glyphs : TInterface (Name {namespace = "Pango", name = "GlyphString"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Pango", name = "ShapeFlags"})
    IO ()

-- | Convert the characters in /@item@/ into glyphs.
-- 
-- This is similar to 'GI.Pango.Functions.shapeWithFlags', except it takes a
-- @PangoItem@ instead of separate /@itemText@/ and /@analysis@/ arguments.
-- 
-- It also takes /@logAttrs@/, which are needed for implementing some aspects
-- of hyphen insertion and text transforms (in particular, capitalization).
-- 
-- Note that the extra attributes in the /@analyis@/ that is returned from
-- 'GI.Pango.Functions.itemize' have indices that are relative to the entire paragraph,
-- so you do not pass the full paragraph text as /@paragraphText@/, you need
-- to subtract the item offset from their indices before calling
-- 'GI.Pango.Functions.shapeWithFlags'.
-- 
-- /Since: 1.50/
shapeItem ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Pango.Item.Item
    -- ^ /@item@/: @PangoItem@ to shape
    -> Maybe (T.Text)
    -- ^ /@paragraphText@/: text of the paragraph (see details).
    -> Int32
    -- ^ /@paragraphLength@/: the length (in bytes) of /@paragraphText@/.
    --     -1 means nul-terminated text.
    -> Maybe (Pango.LogAttr.LogAttr)
    -- ^ /@logAttrs@/: array of @PangoLogAttr@ for /@item@/
    -> [Pango.Flags.ShapeFlags]
    -- ^ /@flags@/: flags influencing the shaping process
    -> m (Pango.GlyphString.GlyphString)
shapeItem item paragraphText paragraphLength logAttrs flags = liftIO $ do
    item' <- unsafeManagedPtrGetPtr item
    maybeParagraphText <- case paragraphText of
        Nothing -> return FP.nullPtr
        Just jParagraphText -> do
            jParagraphText' <- textToCString jParagraphText
            return jParagraphText'
    maybeLogAttrs <- case logAttrs of
        Nothing -> return FP.nullPtr
        Just jLogAttrs -> do
            jLogAttrs' <- unsafeManagedPtrGetPtr jLogAttrs
            return jLogAttrs'
    glyphs <- SP.callocBoxedBytes 32 :: IO (Ptr Pango.GlyphString.GlyphString)
    let flags' = gflagsToWord flags
    pango_shape_item item' maybeParagraphText paragraphLength maybeLogAttrs glyphs flags'
    glyphs' <- (wrapBoxed Pango.GlyphString.GlyphString) glyphs
    touchManagedPtr item
    whenJust logAttrs touchManagedPtr
    freeMem maybeParagraphText
    return glyphs'


-- function shape_full
-- Args: [ Arg
--           { argCName = "item_text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "valid UTF-8 text to shape."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "item_length"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the length (in bytes) of @item_text. -1 means nul-terminated text."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "paragraph_text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "text of the paragraph (see details)."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "paragraph_length"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the length (in bytes) of @paragraph_text. -1 means nul-terminated text."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "analysis"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Analysis" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "`PangoAnalysis` structure from [func@Pango.itemize]."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "glyphs"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "GlyphString" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "glyph string in which to store results."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_shape_full" pango_shape_full :: 
    CString ->                              -- item_text : TBasicType TUTF8
    Int32 ->                                -- item_length : TBasicType TInt
    CString ->                              -- paragraph_text : TBasicType TUTF8
    Int32 ->                                -- paragraph_length : TBasicType TInt
    Ptr Pango.Analysis.Analysis ->          -- analysis : TInterface (Name {namespace = "Pango", name = "Analysis"})
    Ptr Pango.GlyphString.GlyphString ->    -- glyphs : TInterface (Name {namespace = "Pango", name = "GlyphString"})
    IO ()

-- | Convert the characters in /@text@/ into glyphs.
-- 
-- Given a segment of text and the corresponding @PangoAnalysis@ structure
-- returned from 'GI.Pango.Functions.itemize', convert the characters into glyphs.
-- You may also pass in only a substring of the item from 'GI.Pango.Functions.itemize'.
-- 
-- This is similar to 'GI.Pango.Functions.shape', except it also can optionally take
-- the full paragraph text as input, which will then be used to perform
-- certain cross-item shaping interactions. If you have access to the broader
-- text of which /@itemText@/ is part of, provide the broader text as
-- /@paragraphText@/. If /@paragraphText@/ is 'P.Nothing', item text is used instead.
-- 
-- Some aspects of hyphen insertion and text transformation (in particular,
-- capitalization) require log attrs, and thus can only be handled by
-- 'GI.Pango.Functions.shapeItem'.
-- 
-- Note that the extra attributes in the /@analyis@/ that is returned from
-- 'GI.Pango.Functions.itemize' have indices that are relative to the entire paragraph,
-- so you do not pass the full paragraph text as /@paragraphText@/, you need
-- to subtract the item offset from their indices before calling
-- 'GI.Pango.Functions.shapeFull'.
-- 
-- /Since: 1.32/
shapeFull ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@itemText@/: valid UTF-8 text to shape.
    -> Int32
    -- ^ /@itemLength@/: the length (in bytes) of /@itemText@/. -1 means nul-terminated text.
    -> Maybe (T.Text)
    -- ^ /@paragraphText@/: text of the paragraph (see details).
    -> Int32
    -- ^ /@paragraphLength@/: the length (in bytes) of /@paragraphText@/. -1 means nul-terminated text.
    -> Pango.Analysis.Analysis
    -- ^ /@analysis@/: @PangoAnalysis@ structure from 'GI.Pango.Functions.itemize'.
    -> m (Pango.GlyphString.GlyphString)
shapeFull itemText itemLength paragraphText paragraphLength analysis = liftIO $ do
    itemText' <- textToCString itemText
    maybeParagraphText <- case paragraphText of
        Nothing -> return FP.nullPtr
        Just jParagraphText -> do
            jParagraphText' <- textToCString jParagraphText
            return jParagraphText'
    analysis' <- unsafeManagedPtrGetPtr analysis
    glyphs <- SP.callocBoxedBytes 32 :: IO (Ptr Pango.GlyphString.GlyphString)
    pango_shape_full itemText' itemLength maybeParagraphText paragraphLength analysis' glyphs
    glyphs' <- (wrapBoxed Pango.GlyphString.GlyphString) glyphs
    touchManagedPtr analysis
    freeMem itemText'
    freeMem maybeParagraphText
    return glyphs'


-- function shape
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the text to process"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the length (in bytes) of @text"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "analysis"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Analysis" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "`PangoAnalysis` structure from [func@Pango.itemize]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "glyphs"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "GlyphString" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "glyph string in which to store results"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_shape" pango_shape :: 
    CString ->                              -- text : TBasicType TUTF8
    Int32 ->                                -- length : TBasicType TInt
    Ptr Pango.Analysis.Analysis ->          -- analysis : TInterface (Name {namespace = "Pango", name = "Analysis"})
    Ptr Pango.GlyphString.GlyphString ->    -- glyphs : TInterface (Name {namespace = "Pango", name = "GlyphString"})
    IO ()

-- | Convert the characters in /@text@/ into glyphs.
-- 
-- Given a segment of text and the corresponding @PangoAnalysis@ structure
-- returned from 'GI.Pango.Functions.itemize', convert the characters into glyphs. You
-- may also pass in only a substring of the item from 'GI.Pango.Functions.itemize'.
-- 
-- It is recommended that you use 'GI.Pango.Functions.shapeFull' instead, since
-- that API allows for shaping interaction happening across text item
-- boundaries.
-- 
-- Some aspects of hyphen insertion and text transformation (in particular,
-- capitalization) require log attrs, and thus can only be handled by
-- 'GI.Pango.Functions.shapeItem'.
-- 
-- Note that the extra attributes in the /@analyis@/ that is returned from
-- 'GI.Pango.Functions.itemize' have indices that are relative to the entire paragraph,
-- so you need to subtract the item offset from their indices before
-- calling 'GI.Pango.Functions.shape'.
shape ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@text@/: the text to process
    -> Int32
    -- ^ /@length@/: the length (in bytes) of /@text@/
    -> Pango.Analysis.Analysis
    -- ^ /@analysis@/: @PangoAnalysis@ structure from 'GI.Pango.Functions.itemize'
    -> m (Pango.GlyphString.GlyphString)
shape text length_ analysis = liftIO $ do
    text' <- textToCString text
    analysis' <- unsafeManagedPtrGetPtr analysis
    glyphs <- SP.callocBoxedBytes 32 :: IO (Ptr Pango.GlyphString.GlyphString)
    pango_shape text' length_ analysis' glyphs
    glyphs' <- (wrapBoxed Pango.GlyphString.GlyphString) glyphs
    touchManagedPtr analysis
    freeMem text'
    return glyphs'


-- function scan_word
-- Args: [ Arg
--           { argCName = "pos"
--           , argType = TBasicType TUTF8
--           , direction = DirectionInout
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "in/out string position"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "out"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "String" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GString` into which to write the result"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "pango_scan_word" pango_scan_word :: 
    Ptr CString ->                          -- pos : TBasicType TUTF8
    Ptr GLib.String.String ->               -- out : TInterface (Name {namespace = "GLib", name = "String"})
    IO CInt

{-# DEPRECATED scanWord ["(Since version 1.38)"] #-}
-- | Scans a word into a @GString@ buffer.
-- 
-- A word consists of [A-Za-z_] followed by zero or more
-- [A-Za-z_0-9]. Leading white space is skipped.
scanWord ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@pos@/: in\/out string position
    -> GLib.String.String
    -- ^ /@out@/: a @GString@ into which to write the result
    -> m ((Bool, T.Text))
    -- ^ __Returns:__ 'P.False' if a parse error occurred
scanWord pos out = liftIO $ do
    pos' <- textToCString pos
    pos'' <- allocMem :: IO (Ptr CString)
    poke pos'' pos'
    out' <- unsafeManagedPtrGetPtr out
    result <- pango_scan_word pos'' out'
    let result' = (/= 0) result
    pos''' <- peek pos''
    pos'''' <- cstringToText pos'''
    touchManagedPtr out
    freeMem pos''
    return (result', pos'''')


-- function scan_string
-- Args: [ Arg
--           { argCName = "pos"
--           , argType = TBasicType TUTF8
--           , direction = DirectionInout
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "in/out string position"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "out"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "String" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GString` into which to write the result"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "pango_scan_string" pango_scan_string :: 
    Ptr CString ->                          -- pos : TBasicType TUTF8
    Ptr GLib.String.String ->               -- out : TInterface (Name {namespace = "GLib", name = "String"})
    IO CInt

{-# DEPRECATED scanString ["(Since version 1.38)"] #-}
-- | Scans a string into a @GString@ buffer.
-- 
-- The string may either be a sequence of non-white-space characters,
-- or a quoted string with \'\"\'. Instead a quoted string, \'\\\"\' represents
-- a literal quote. Leading white space outside of quotes is skipped.
scanString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@pos@/: in\/out string position
    -> GLib.String.String
    -- ^ /@out@/: a @GString@ into which to write the result
    -> m ((Bool, T.Text))
    -- ^ __Returns:__ 'P.False' if a parse error occurred
scanString pos out = liftIO $ do
    pos' <- textToCString pos
    pos'' <- allocMem :: IO (Ptr CString)
    poke pos'' pos'
    out' <- unsafeManagedPtrGetPtr out
    result <- pango_scan_string pos'' out'
    let result' = (/= 0) result
    pos''' <- peek pos''
    pos'''' <- cstringToText pos'''
    touchManagedPtr out
    freeMem pos''
    return (result', pos'''')


-- function scan_int
-- Args: [ Arg
--           { argCName = "pos"
--           , argType = TBasicType TUTF8
--           , direction = DirectionInout
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "in/out string position"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "out"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an int into which to write the result"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "pango_scan_int" pango_scan_int :: 
    Ptr CString ->                          -- pos : TBasicType TUTF8
    Ptr Int32 ->                            -- out : TBasicType TInt
    IO CInt

{-# DEPRECATED scanInt ["(Since version 1.38)"] #-}
-- | Scans an integer.
-- 
-- Leading white space is skipped.
scanInt ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@pos@/: in\/out string position
    -> m ((Bool, T.Text, Int32))
    -- ^ __Returns:__ 'P.False' if a parse error occurred
scanInt pos = liftIO $ do
    pos' <- textToCString pos
    pos'' <- allocMem :: IO (Ptr CString)
    poke pos'' pos'
    out <- allocMem :: IO (Ptr Int32)
    result <- pango_scan_int pos'' out
    let result' = (/= 0) result
    pos''' <- peek pos''
    pos'''' <- cstringToText pos'''
    out' <- peek out
    freeMem pos''
    freeMem out
    return (result', pos'''', out')


-- function reorder_items
-- Args: [ Arg
--           { argCName = "items"
--           , argType =
--               TGList (TInterface Name { namespace = "Pango" , name = "Item" })
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GList` of `PangoItem`\n  in logical order."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TGList (TInterface Name { namespace = "Pango" , name = "Item" }))
-- throws : False
-- Skip return : False

foreign import ccall "pango_reorder_items" pango_reorder_items :: 
    Ptr (GList (Ptr Pango.Item.Item)) ->    -- items : TGList (TInterface (Name {namespace = "Pango", name = "Item"}))
    IO (Ptr (GList (Ptr Pango.Item.Item)))

-- | Reorder items from logical order to visual order.
-- 
-- The visual order is determined from the associated directional
-- levels of the items. The original list is unmodified.
-- 
-- (Please open a bug if you use this function.
--  It is not a particularly convenient interface, and the code
--  is duplicated elsewhere in Pango for that reason.)
reorderItems ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    [Pango.Item.Item]
    -- ^ /@items@/: a @GList@ of @PangoItem@
    --   in logical order.
    -> m [Pango.Item.Item]
    -- ^ __Returns:__ a @GList@
    --   of @PangoItem@ structures in visual order.
reorderItems items = liftIO $ do
    items' <- mapM unsafeManagedPtrGetPtr items
    items'' <- packGList items'
    result <- pango_reorder_items items''
    result' <- unpackGList result
    result'' <- mapM (wrapBoxed Pango.Item.Item) result'
    g_list_free result
    mapM_ touchManagedPtr items
    g_list_free items''
    return result''


-- function read_line
-- Args: [ Arg
--           { argCName = "stream"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a stdio stream" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "str"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "String" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "`GString` buffer into which to write the result"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "pango_read_line" pango_read_line :: 
    Ptr () ->                               -- stream : TBasicType TPtr
    Ptr GLib.String.String ->               -- str : TInterface (Name {namespace = "GLib", name = "String"})
    IO Int32

{-# DEPRECATED readLine ["(Since version 1.38)"] #-}
-- | Reads an entire line from a file into a buffer.
-- 
-- Lines may be delimited with \'\\n\', \'\\r\', \'\\n\\r\', or \'\\r\\n\'. The delimiter
-- is not written into the buffer. Text after a \'#\' character is treated as
-- a comment and skipped. \'\\\' can be used to escape a # character.
-- \'\\\' proceeding a line delimiter combines adjacent lines. A \'\\\' proceeding
-- any other character is ignored and written into the output buffer
-- unmodified.
readLine ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Ptr ()
    -- ^ /@stream@/: a stdio stream
    -> GLib.String.String
    -- ^ /@str@/: @GString@ buffer into which to write the result
    -> m Int32
    -- ^ __Returns:__ 0 if the stream was already at an @/EOF/@ character,
    --   otherwise the number of lines read (this is useful for maintaining
    --   a line number counter which doesn\'t combine lines with \'\\\')
readLine stream str = liftIO $ do
    str' <- unsafeManagedPtrGetPtr str
    result <- pango_read_line stream str'
    touchManagedPtr str
    return result


-- function quantize_line_geometry
-- Args: [ Arg
--           { argCName = "thickness"
--           , argType = TBasicType TInt
--           , direction = DirectionInout
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "pointer to the thickness of a line, in Pango units"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "position"
--           , argType = TBasicType TInt
--           , direction = DirectionInout
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "corresponding position"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_quantize_line_geometry" pango_quantize_line_geometry :: 
    Ptr Int32 ->                            -- thickness : TBasicType TInt
    Ptr Int32 ->                            -- position : TBasicType TInt
    IO ()

-- | Quantizes the thickness and position of a line to whole device pixels.
-- 
-- This is typically used for underline or strikethrough. The purpose of
-- this function is to avoid such lines looking blurry.
-- 
-- Care is taken to make sure /@thickness@/ is at least one pixel when this
-- function returns, but returned /@position@/ may become zero as a result
-- of rounding.
-- 
-- /Since: 1.12/
quantizeLineGeometry ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Int32
    -- ^ /@thickness@/: pointer to the thickness of a line, in Pango units
    -> Int32
    -- ^ /@position@/: corresponding position
    -> m ((Int32, Int32))
quantizeLineGeometry thickness position = liftIO $ do
    thickness' <- allocMem :: IO (Ptr Int32)
    poke thickness' thickness
    position' <- allocMem :: IO (Ptr Int32)
    poke position' position
    pango_quantize_line_geometry thickness' position'
    thickness'' <- peek thickness'
    position'' <- peek position'
    freeMem thickness'
    freeMem position'
    return (thickness'', position'')


-- function parse_weight
-- Args: [ Arg
--           { argCName = "str"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string to parse." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "weight"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Weight" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoWeight` to store the result in."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "warn"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "if %TRUE, issue a g_warning() on bad input."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "pango_parse_weight" pango_parse_weight :: 
    CString ->                              -- str : TBasicType TUTF8
    Ptr CUInt ->                            -- weight : TInterface (Name {namespace = "Pango", name = "Weight"})
    CInt ->                                 -- warn : TBasicType TBoolean
    IO CInt

-- | Parses a font weight.
-- 
-- The allowed values are \"heavy\",
-- \"ultrabold\", \"bold\", \"normal\", \"light\", \"ultraleight\"
-- and integers. Case variations are ignored.
parseWeight ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@str@/: a string to parse.
    -> Bool
    -- ^ /@warn@/: if 'P.True', issue a @/g_warning()/@ on bad input.
    -> m ((Bool, Pango.Enums.Weight))
    -- ^ __Returns:__ 'P.True' if /@str@/ was successfully parsed.
parseWeight str warn = liftIO $ do
    str' <- textToCString str
    weight <- allocMem :: IO (Ptr CUInt)
    let warn' = (P.fromIntegral . P.fromEnum) warn
    result <- pango_parse_weight str' weight warn'
    let result' = (/= 0) result
    weight' <- peek weight
    let weight'' = (toEnum . fromIntegral) weight'
    freeMem str'
    freeMem weight
    return (result', weight'')


-- function parse_variant
-- Args: [ Arg
--           { argCName = "str"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string to parse." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "variant"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Variant" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoVariant` to store the result in."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "warn"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "if %TRUE, issue a g_warning() on bad input."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "pango_parse_variant" pango_parse_variant :: 
    CString ->                              -- str : TBasicType TUTF8
    Ptr CUInt ->                            -- variant : TInterface (Name {namespace = "Pango", name = "Variant"})
    CInt ->                                 -- warn : TBasicType TBoolean
    IO CInt

-- | Parses a font variant.
-- 
-- The allowed values are \"normal\", \"small-caps\", \"all-small-caps\",
-- \"petite-caps\", \"all-petite-caps\", \"unicase\" and \"title-caps\",
-- case variations being ignored.
parseVariant ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@str@/: a string to parse.
    -> Bool
    -- ^ /@warn@/: if 'P.True', issue a @/g_warning()/@ on bad input.
    -> m ((Bool, Pango.Enums.Variant))
    -- ^ __Returns:__ 'P.True' if /@str@/ was successfully parsed.
parseVariant str warn = liftIO $ do
    str' <- textToCString str
    variant <- allocMem :: IO (Ptr CUInt)
    let warn' = (P.fromIntegral . P.fromEnum) warn
    result <- pango_parse_variant str' variant warn'
    let result' = (/= 0) result
    variant' <- peek variant
    let variant'' = (toEnum . fromIntegral) variant'
    freeMem str'
    freeMem variant
    return (result', variant'')


-- function parse_style
-- Args: [ Arg
--           { argCName = "str"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string to parse." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "style"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Style" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoStyle` to store the result in."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "warn"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "if %TRUE, issue a g_warning() on bad input."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "pango_parse_style" pango_parse_style :: 
    CString ->                              -- str : TBasicType TUTF8
    Ptr CUInt ->                            -- style : TInterface (Name {namespace = "Pango", name = "Style"})
    CInt ->                                 -- warn : TBasicType TBoolean
    IO CInt

-- | Parses a font style.
-- 
-- The allowed values are \"normal\", \"italic\" and \"oblique\", case
-- variations being
-- ignored.
parseStyle ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@str@/: a string to parse.
    -> Bool
    -- ^ /@warn@/: if 'P.True', issue a @/g_warning()/@ on bad input.
    -> m ((Bool, Pango.Enums.Style))
    -- ^ __Returns:__ 'P.True' if /@str@/ was successfully parsed.
parseStyle str warn = liftIO $ do
    str' <- textToCString str
    style <- allocMem :: IO (Ptr CUInt)
    let warn' = (P.fromIntegral . P.fromEnum) warn
    result <- pango_parse_style str' style warn'
    let result' = (/= 0) result
    style' <- peek style
    let style'' = (toEnum . fromIntegral) style'
    freeMem str'
    freeMem style
    return (result', style'')


-- function parse_stretch
-- Args: [ Arg
--           { argCName = "str"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string to parse." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "stretch"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Stretch" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoStretch` to store the result in."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "warn"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "if %TRUE, issue a g_warning() on bad input."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "pango_parse_stretch" pango_parse_stretch :: 
    CString ->                              -- str : TBasicType TUTF8
    Ptr CUInt ->                            -- stretch : TInterface (Name {namespace = "Pango", name = "Stretch"})
    CInt ->                                 -- warn : TBasicType TBoolean
    IO CInt

-- | Parses a font stretch.
-- 
-- The allowed values are
-- \"ultra_condensed\", \"extra_condensed\", \"condensed\",
-- \"semi_condensed\", \"normal\", \"semi_expanded\", \"expanded\",
-- \"extra_expanded\" and \"ultra_expanded\". Case variations are
-- ignored and the \'_\' characters may be omitted.
parseStretch ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@str@/: a string to parse.
    -> Bool
    -- ^ /@warn@/: if 'P.True', issue a @/g_warning()/@ on bad input.
    -> m ((Bool, Pango.Enums.Stretch))
    -- ^ __Returns:__ 'P.True' if /@str@/ was successfully parsed.
parseStretch str warn = liftIO $ do
    str' <- textToCString str
    stretch <- allocMem :: IO (Ptr CUInt)
    let warn' = (P.fromIntegral . P.fromEnum) warn
    result <- pango_parse_stretch str' stretch warn'
    let result' = (/= 0) result
    stretch' <- peek stretch
    let stretch'' = (toEnum . fromIntegral) stretch'
    freeMem str'
    freeMem stretch
    return (result', stretch'')


-- function parse_markup
-- Args: [ Arg
--           { argCName = "markup_text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "markup to parse (see the [Pango Markup](pango_markup.html) docs)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "length of @markup_text, or -1 if nul-terminated"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "accel_marker"
--           , argType = TBasicType TUniChar
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "character that precedes an accelerator, or 0 for none"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attr_list"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "address of return location for a `PangoAttrList`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "address of return location for text with tags stripped"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "accel_char"
--           , argType = TBasicType TUniChar
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "address of return location for accelerator char"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "pango_parse_markup" pango_parse_markup :: 
    CString ->                              -- markup_text : TBasicType TUTF8
    Int32 ->                                -- length : TBasicType TInt
    CInt ->                                 -- accel_marker : TBasicType TUniChar
    Ptr (Ptr Pango.AttrList.AttrList) ->    -- attr_list : TInterface (Name {namespace = "Pango", name = "AttrList"})
    Ptr CString ->                          -- text : TBasicType TUTF8
    Ptr CInt ->                             -- accel_char : TBasicType TUniChar
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Parses marked-up text to create a plain-text string and an attribute list.
-- 
-- See the <http://developer.gnome.org/pango/stable/pango_markup.html Pango Markup> docs for details about the
-- supported markup.
-- 
-- If /@accelMarker@/ is nonzero, the given character will mark the
-- character following it as an accelerator. For example, /@accelMarker@/
-- might be an ampersand or underscore. All characters marked
-- as an accelerator will receive a 'GI.Pango.Enums.UnderlineLow' attribute,
-- and the first character so marked will be returned in /@accelChar@/.
-- Two /@accelMarker@/ characters following each other produce a single
-- literal /@accelMarker@/ character.
-- 
-- To parse a stream of pango markup incrementally, use [func/@markupParserNew@/].
-- 
-- If any error happens, none of the output arguments are touched except
-- for /@error@/.
parseMarkup ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@markupText@/: markup to parse (see the <http://developer.gnome.org/pango/stable/pango_markup.html Pango Markup> docs)
    -> Int32
    -- ^ /@length@/: length of /@markupText@/, or -1 if nul-terminated
    -> Char
    -- ^ /@accelMarker@/: character that precedes an accelerator, or 0 for none
    -> m ((Pango.AttrList.AttrList, T.Text, Char))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
parseMarkup markupText length_ accelMarker = liftIO $ do
    markupText' <- textToCString markupText
    let accelMarker' = (P.fromIntegral . SP.ord) accelMarker
    attrList <- callocMem :: IO (Ptr (Ptr Pango.AttrList.AttrList))
    text <- callocMem :: IO (Ptr CString)
    accelChar <- allocMem :: IO (Ptr CInt)
    onException (do
        _ <- propagateGError $ pango_parse_markup markupText' length_ accelMarker' attrList text accelChar
        attrList' <- peek attrList
        attrList'' <- (wrapBoxed Pango.AttrList.AttrList) attrList'
        text' <- peek text
        text'' <- cstringToText text'
        freeMem text'
        accelChar' <- peek accelChar
        let accelChar'' = (chr . fromIntegral) accelChar'
        freeMem markupText'
        freeMem attrList
        freeMem text
        freeMem accelChar
        return (attrList'', text'', accelChar'')
     ) (do
        freeMem markupText'
        freeMem attrList
        freeMem text
        freeMem accelChar
     )


-- function parse_enum
-- Args: [ Arg
--           { argCName = "type"
--           , argType = TBasicType TGType
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "enum type to parse, eg. %PANGO_TYPE_ELLIPSIZE_MODE"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "str"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "string to parse" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "integer to store the result in"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "warn"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "if %TRUE, issue a g_warning() on bad input"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "possible_values"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "place to store list of possible\n  values on failure"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "pango_parse_enum" pango_parse_enum :: 
    CGType ->                               -- type : TBasicType TGType
    CString ->                              -- str : TBasicType TUTF8
    Ptr Int32 ->                            -- value : TBasicType TInt
    CInt ->                                 -- warn : TBasicType TBoolean
    Ptr CString ->                          -- possible_values : TBasicType TUTF8
    IO CInt

{-# DEPRECATED parseEnum ["(Since version 1.38)"] #-}
-- | Parses an enum type and stores the result in /@value@/.
-- 
-- If /@str@/ does not match the nick name of any of the possible values
-- for the enum and is not an integer, 'P.False' is returned, a warning
-- is issued if /@warn@/ is 'P.True', and a string representing the list of
-- possible values is stored in /@possibleValues@/. The list is
-- slash-separated, eg. \"none\/start\/middle\/end\".
-- 
-- If failed and /@possibleValues@/ is not 'P.Nothing', returned string should
-- be freed using 'GI.GLib.Functions.free'.
-- 
-- /Since: 1.16/
parseEnum ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    GType
    -- ^ /@type@/: enum type to parse, eg. @/PANGO_TYPE_ELLIPSIZE_MODE/@
    -> Maybe (T.Text)
    -- ^ /@str@/: string to parse
    -> Bool
    -- ^ /@warn@/: if 'P.True', issue a @/g_warning()/@ on bad input
    -> m ((Bool, Int32, T.Text))
    -- ^ __Returns:__ 'P.True' if /@str@/ was successfully parsed
parseEnum type_ str warn = liftIO $ do
    let type_' = gtypeToCGType type_
    maybeStr <- case str of
        Nothing -> return FP.nullPtr
        Just jStr -> do
            jStr' <- textToCString jStr
            return jStr'
    value <- allocMem :: IO (Ptr Int32)
    let warn' = (P.fromIntegral . P.fromEnum) warn
    possibleValues <- callocMem :: IO (Ptr CString)
    result <- pango_parse_enum type_' maybeStr value warn' possibleValues
    let result' = (/= 0) result
    value' <- peek value
    possibleValues' <- peek possibleValues
    possibleValues'' <- cstringToText possibleValues'
    freeMem possibleValues'
    freeMem maybeStr
    freeMem value
    freeMem possibleValues
    return (result', value', possibleValues'')


-- function markup_parser_new
-- Args: [ Arg
--           { argCName = "accel_marker"
--           , argType = TBasicType TUniChar
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "character that precedes an accelerator, or 0 for none"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "GLib" , name = "MarkupParseContext" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_markup_parser_new" pango_markup_parser_new :: 
    CInt ->                                 -- accel_marker : TBasicType TUniChar
    IO (Ptr GLib.MarkupParseContext.MarkupParseContext)

-- | Incrementally parses marked-up text to create a plain-text string
-- and an attribute list.
-- 
-- See the <http://developer.gnome.org/pango/stable/pango_markup.html Pango Markup> docs for details about the
-- supported markup.
-- 
-- If /@accelMarker@/ is nonzero, the given character will mark the
-- character following it as an accelerator. For example, /@accelMarker@/
-- might be an ampersand or underscore. All characters marked
-- as an accelerator will receive a 'GI.Pango.Enums.UnderlineLow' attribute,
-- and the first character so marked will be returned in /@accelChar@/,
-- when calling [func/@markupParserFinish@/]. Two /@accelMarker@/ characters
-- following each other produce a single literal /@accelMarker@/ character.
-- 
-- To feed markup to the parser, use 'GI.GLib.Structs.MarkupParseContext.markupParseContextParse'
-- on the returned t'GI.GLib.Structs.MarkupParseContext.MarkupParseContext'. When done with feeding markup
-- to the parser, use [func/@markupParserFinish@/] to get the data out
-- of it, and then use 'GI.GLib.Structs.MarkupParseContext.markupParseContextFree' to free it.
-- 
-- This function is designed for applications that read Pango markup
-- from streams. To simply parse a string containing Pango markup,
-- the 'GI.Pango.Functions.parseMarkup' API is recommended instead.
-- 
-- /Since: 1.31.0/
markupParserNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Char
    -- ^ /@accelMarker@/: character that precedes an accelerator, or 0 for none
    -> m GLib.MarkupParseContext.MarkupParseContext
    -- ^ __Returns:__ a @GMarkupParseContext@ that should be
    -- destroyed with 'GI.GLib.Structs.MarkupParseContext.markupParseContextFree'.
markupParserNew accelMarker = liftIO $ do
    let accelMarker' = (P.fromIntegral . SP.ord) accelMarker
    result <- pango_markup_parser_new accelMarker'
    checkUnexpectedReturnNULL "markupParserNew" result
    result' <- (newBoxed GLib.MarkupParseContext.MarkupParseContext) result
    return result'


-- function markup_parser_finish
-- Args: [ Arg
--           { argCName = "context"
--           , argType =
--               TInterface
--                 Name { namespace = "GLib" , name = "MarkupParseContext" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A valid parse context that was returned from [func@markup_parser_new]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attr_list"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "address of return location for a `PangoAttrList`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "address of return location for text with tags stripped"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "accel_char"
--           , argType = TBasicType TUniChar
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "address of return location for accelerator char"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "pango_markup_parser_finish" pango_markup_parser_finish :: 
    Ptr GLib.MarkupParseContext.MarkupParseContext -> -- context : TInterface (Name {namespace = "GLib", name = "MarkupParseContext"})
    Ptr (Ptr Pango.AttrList.AttrList) ->    -- attr_list : TInterface (Name {namespace = "Pango", name = "AttrList"})
    Ptr CString ->                          -- text : TBasicType TUTF8
    Ptr CInt ->                             -- accel_char : TBasicType TUniChar
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes parsing markup.
-- 
-- After feeding a Pango markup parser some data with 'GI.GLib.Structs.MarkupParseContext.markupParseContextParse',
-- use this function to get the list of attributes and text out of the
-- markup. This function will not free /@context@/, use 'GI.GLib.Structs.MarkupParseContext.markupParseContextFree'
-- to do so.
-- 
-- /Since: 1.31.0/
markupParserFinish ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    GLib.MarkupParseContext.MarkupParseContext
    -- ^ /@context@/: A valid parse context that was returned from [func/@markupParserNew@/]
    -> m ((Pango.AttrList.AttrList, T.Text, Char))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
markupParserFinish context = liftIO $ do
    context' <- unsafeManagedPtrGetPtr context
    attrList <- callocMem :: IO (Ptr (Ptr Pango.AttrList.AttrList))
    text <- callocMem :: IO (Ptr CString)
    accelChar <- allocMem :: IO (Ptr CInt)
    onException (do
        _ <- propagateGError $ pango_markup_parser_finish context' attrList text accelChar
        attrList' <- peek attrList
        attrList'' <- (wrapBoxed Pango.AttrList.AttrList) attrList'
        text' <- peek text
        text'' <- cstringToText text'
        freeMem text'
        accelChar' <- peek accelChar
        let accelChar'' = (chr . fromIntegral) accelChar'
        touchManagedPtr context
        freeMem attrList
        freeMem text
        freeMem accelChar
        return (attrList'', text'', accelChar'')
     ) (do
        freeMem attrList
        freeMem text
        freeMem accelChar
     )


-- function log2vis_get_embedding_levels
-- XXX Could not generate function log2vis_get_embedding_levels

{-  Bad introspection data: `TCArray False (-1) (-1) (TBasicType TUInt8)' is an array type, but contains no length information,
    so it cannot be unpacked.
-}

-- function itemize_with_base_dir
-- Args: [ Arg
--           { argCName = "context"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Context" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a structure holding information that affects\n  the itemization process."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "base_dir"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Direction" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "base direction to use for bidirectional processing"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the text to itemize."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_index"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "first byte in @text to process"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the number of bytes (not characters) to process\n  after @start_index. This must be >= 0."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attrs"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the set of attributes that apply to @text."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cached_iter"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrIterator" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Cached attribute iterator"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TGList (TInterface Name { namespace = "Pango" , name = "Item" }))
-- throws : False
-- Skip return : False

foreign import ccall "pango_itemize_with_base_dir" pango_itemize_with_base_dir :: 
    Ptr Pango.Context.Context ->            -- context : TInterface (Name {namespace = "Pango", name = "Context"})
    CUInt ->                                -- base_dir : TInterface (Name {namespace = "Pango", name = "Direction"})
    CString ->                              -- text : TBasicType TUTF8
    Int32 ->                                -- start_index : TBasicType TInt
    Int32 ->                                -- length : TBasicType TInt
    Ptr Pango.AttrList.AttrList ->          -- attrs : TInterface (Name {namespace = "Pango", name = "AttrList"})
    Ptr Pango.AttrIterator.AttrIterator ->  -- cached_iter : TInterface (Name {namespace = "Pango", name = "AttrIterator"})
    IO (Ptr (GList (Ptr Pango.Item.Item)))

-- | Like @pango_itemize()@, but with an explicitly specified base direction.
-- 
-- The base direction is used when computing bidirectional levels.
-- [func/@itemize@/] gets the base direction from the @PangoContext@
-- (see 'GI.Pango.Objects.Context.contextSetBaseDir').
-- 
-- /Since: 1.4/
itemizeWithBaseDir ::
    (B.CallStack.HasCallStack, MonadIO m, Pango.Context.IsContext a) =>
    a
    -- ^ /@context@/: a structure holding information that affects
    --   the itemization process.
    -> Pango.Enums.Direction
    -- ^ /@baseDir@/: base direction to use for bidirectional processing
    -> T.Text
    -- ^ /@text@/: the text to itemize.
    -> Int32
    -- ^ /@startIndex@/: first byte in /@text@/ to process
    -> Int32
    -- ^ /@length@/: the number of bytes (not characters) to process
    --   after /@startIndex@/. This must be >= 0.
    -> Pango.AttrList.AttrList
    -- ^ /@attrs@/: the set of attributes that apply to /@text@/.
    -> Maybe (Pango.AttrIterator.AttrIterator)
    -- ^ /@cachedIter@/: Cached attribute iterator
    -> m [Pango.Item.Item]
    -- ^ __Returns:__ a @GList@ of
    --   t'GI.Pango.Structs.Item.Item' structures. The items should be freed using
    --   'GI.Pango.Structs.Item.itemFree' probably in combination with t'GI.GLib.Structs.List.List'.@/free_full/@().
itemizeWithBaseDir context baseDir text startIndex length_ attrs cachedIter = liftIO $ do
    context' <- unsafeManagedPtrCastPtr context
    let baseDir' = (fromIntegral . fromEnum) baseDir
    text' <- textToCString text
    attrs' <- unsafeManagedPtrGetPtr attrs
    maybeCachedIter <- case cachedIter of
        Nothing -> return FP.nullPtr
        Just jCachedIter -> do
            jCachedIter' <- unsafeManagedPtrGetPtr jCachedIter
            return jCachedIter'
    result <- pango_itemize_with_base_dir context' baseDir' text' startIndex length_ attrs' maybeCachedIter
    result' <- unpackGList result
    result'' <- mapM (wrapBoxed Pango.Item.Item) result'
    g_list_free result
    touchManagedPtr context
    touchManagedPtr attrs
    whenJust cachedIter touchManagedPtr
    freeMem text'
    return result''


-- function itemize
-- Args: [ Arg
--           { argCName = "context"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Context" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a structure holding information that affects\n  the itemization process."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the text to itemize. Must be valid UTF-8"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_index"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "first byte in @text to process"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the number of bytes (not characters) to process\n  after @start_index. This must be >= 0."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attrs"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the set of attributes that apply to @text."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cached_iter"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrIterator" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Cached attribute iterator"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TGList (TInterface Name { namespace = "Pango" , name = "Item" }))
-- throws : False
-- Skip return : False

foreign import ccall "pango_itemize" pango_itemize :: 
    Ptr Pango.Context.Context ->            -- context : TInterface (Name {namespace = "Pango", name = "Context"})
    CString ->                              -- text : TBasicType TUTF8
    Int32 ->                                -- start_index : TBasicType TInt
    Int32 ->                                -- length : TBasicType TInt
    Ptr Pango.AttrList.AttrList ->          -- attrs : TInterface (Name {namespace = "Pango", name = "AttrList"})
    Ptr Pango.AttrIterator.AttrIterator ->  -- cached_iter : TInterface (Name {namespace = "Pango", name = "AttrIterator"})
    IO (Ptr (GList (Ptr Pango.Item.Item)))

-- | Breaks a piece of text into segments with consistent directional
-- level and font.
-- 
-- Each byte of /@text@/ will be contained in exactly one of the items in the
-- returned list; the generated list of items will be in logical order (the
-- start offsets of the items are ascending).
-- 
-- /@cachedIter@/ should be an iterator over /@attrs@/ currently positioned
-- at a range before or containing /@startIndex@/; /@cachedIter@/ will be
-- advanced to the range covering the position just after
-- /@startIndex@/ + /@length@/. (i.e. if itemizing in a loop, just keep passing
-- in the same /@cachedIter@/).
itemize ::
    (B.CallStack.HasCallStack, MonadIO m, Pango.Context.IsContext a) =>
    a
    -- ^ /@context@/: a structure holding information that affects
    --   the itemization process.
    -> T.Text
    -- ^ /@text@/: the text to itemize. Must be valid UTF-8
    -> Int32
    -- ^ /@startIndex@/: first byte in /@text@/ to process
    -> Int32
    -- ^ /@length@/: the number of bytes (not characters) to process
    --   after /@startIndex@/. This must be >= 0.
    -> Pango.AttrList.AttrList
    -- ^ /@attrs@/: the set of attributes that apply to /@text@/.
    -> Maybe (Pango.AttrIterator.AttrIterator)
    -- ^ /@cachedIter@/: Cached attribute iterator
    -> m [Pango.Item.Item]
    -- ^ __Returns:__ a @GList@ of
    --   t'GI.Pango.Structs.Item.Item' structures. The items should be freed using
    --   'GI.Pango.Structs.Item.itemFree' in combination with t'GI.GLib.Structs.List.List'.@/free_full/@().
itemize context text startIndex length_ attrs cachedIter = liftIO $ do
    context' <- unsafeManagedPtrCastPtr context
    text' <- textToCString text
    attrs' <- unsafeManagedPtrGetPtr attrs
    maybeCachedIter <- case cachedIter of
        Nothing -> return FP.nullPtr
        Just jCachedIter -> do
            jCachedIter' <- unsafeManagedPtrGetPtr jCachedIter
            return jCachedIter'
    result <- pango_itemize context' text' startIndex length_ attrs' maybeCachedIter
    result' <- unpackGList result
    result'' <- mapM (wrapBoxed Pango.Item.Item) result'
    g_list_free result
    touchManagedPtr context
    touchManagedPtr attrs
    whenJust cachedIter touchManagedPtr
    freeMem text'
    return result''


-- function is_zero_width
-- Args: [ Arg
--           { argCName = "ch"
--           , argType = TBasicType TUniChar
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a Unicode character"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "pango_is_zero_width" pango_is_zero_width :: 
    CInt ->                                 -- ch : TBasicType TUniChar
    IO CInt

-- | Checks if a character that should not be normally rendered.
-- 
-- This includes all Unicode characters with \"ZERO WIDTH\" in their name,
-- as well as *bidi* formatting characters, and a few other ones.
-- 
-- This is totally different from 'GI.GLib.Functions.unicharIszerowidth' and is at best misnamed.
-- 
-- /Since: 1.10/
isZeroWidth ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Char
    -- ^ /@ch@/: a Unicode character
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@ch@/ is a zero-width character, 'P.False' otherwise
isZeroWidth ch = liftIO $ do
    let ch' = (P.fromIntegral . SP.ord) ch
    result <- pango_is_zero_width ch'
    let result' = (/= 0) result
    return result'


-- function get_mirror_char
-- Args: [ Arg
--           { argCName = "ch"
--           , argType = TBasicType TUniChar
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a Unicode character"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "mirrored_ch"
--           , argType = TBasicType TUniChar
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "location to store the mirrored character"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "pango_get_mirror_char" pango_get_mirror_char :: 
    CInt ->                                 -- ch : TBasicType TUniChar
    Ptr CInt ->                             -- mirrored_ch : TBasicType TUniChar
    IO CInt

{-# DEPRECATED getMirrorChar ["(Since version 1.30)","Use 'GI.GLib.Functions.unicharGetMirrorChar' instead;","  the docs for that function provide full details."] #-}
-- | Returns the mirrored character of a Unicode character.
-- 
-- Mirror characters are determined by the Unicode mirrored property.
getMirrorChar ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Char
    -- ^ /@ch@/: a Unicode character
    -> m ((Bool, Char))
    -- ^ __Returns:__ 'P.True' if /@ch@/ has a mirrored character and /@mirroredCh@/ is
    -- filled in, 'P.False' otherwise
getMirrorChar ch = liftIO $ do
    let ch' = (P.fromIntegral . SP.ord) ch
    mirroredCh <- allocMem :: IO (Ptr CInt)
    result <- pango_get_mirror_char ch' mirroredCh
    let result' = (/= 0) result
    mirroredCh' <- peek mirroredCh
    let mirroredCh'' = (chr . fromIntegral) mirroredCh'
    freeMem mirroredCh
    return (result', mirroredCh'')


-- function get_log_attrs
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "text to process. Must be valid UTF-8"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "length in bytes of @text"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "level"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "embedding level, or -1 if unknown"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "language"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Language" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "language tag" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attrs"
--           , argType =
--               TCArray
--                 False
--                 (-1)
--                 5
--                 (TInterface Name { namespace = "Pango" , name = "LogAttr" })
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "array with one\n  `PangoLogAttr` per character in @text, plus one extra, to be filled in"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attrs_len"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "length of @attrs array"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "attrs_len"
--              , argType = TBasicType TInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "length of @attrs array"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_get_log_attrs" pango_get_log_attrs :: 
    CString ->                              -- text : TBasicType TUTF8
    Int32 ->                                -- length : TBasicType TInt
    Int32 ->                                -- level : TBasicType TInt
    Ptr Pango.Language.Language ->          -- language : TInterface (Name {namespace = "Pango", name = "Language"})
    Ptr Pango.LogAttr.LogAttr ->            -- attrs : TCArray False (-1) 5 (TInterface (Name {namespace = "Pango", name = "LogAttr"}))
    Int32 ->                                -- attrs_len : TBasicType TInt
    IO ()

-- | Computes a @PangoLogAttr@ for each character in /@text@/.
-- 
-- The /@attrs@/ array must have one @PangoLogAttr@ for
-- each position in /@text@/; if /@text@/ contains N characters,
-- it has N+1 positions, including the last position at the
-- end of the text. /@text@/ should be an entire paragraph;
-- logical attributes can\'t be computed without context
-- (for example you need to see spaces on either side of
-- a word to know the word is a word).
getLogAttrs ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@text@/: text to process. Must be valid UTF-8
    -> Int32
    -- ^ /@length@/: length in bytes of /@text@/
    -> Int32
    -- ^ /@level@/: embedding level, or -1 if unknown
    -> Pango.Language.Language
    -- ^ /@language@/: language tag
    -> [Pango.LogAttr.LogAttr]
    -- ^ /@attrs@/: array with one
    --   @PangoLogAttr@ per character in /@text@/, plus one extra, to be filled in
    -> m ([Pango.LogAttr.LogAttr])
getLogAttrs text length_ level language attrs = liftIO $ do
    let attrsLen = fromIntegral $ P.length attrs
    text' <- textToCString text
    language' <- unsafeManagedPtrGetPtr language
    attrs' <- mapM unsafeManagedPtrGetPtr attrs
    attrs'' <- packBlockArray 64 attrs'
    pango_get_log_attrs text' length_ level language' attrs'' attrsLen
    attrs''' <- (unpackBlockArrayWithLength 64 attrsLen) attrs''
    attrs'''' <- mapM (wrapPtr Pango.LogAttr.LogAttr) attrs'''
    freeMem attrs''
    touchManagedPtr language
    mapM_ touchManagedPtr attrs
    freeMem text'
    return attrs''''


-- function find_paragraph_boundary
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "UTF-8 text" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "length of @text in bytes, or -1 if nul-terminated"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "paragraph_delimiter_index"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location\n for index of delimiter"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "next_paragraph_start"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for\n start of next paragraph"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_find_paragraph_boundary" pango_find_paragraph_boundary :: 
    CString ->                              -- text : TBasicType TUTF8
    Int32 ->                                -- length : TBasicType TInt
    Ptr Int32 ->                            -- paragraph_delimiter_index : TBasicType TInt
    Ptr Int32 ->                            -- next_paragraph_start : TBasicType TInt
    IO ()

-- | Locates a paragraph boundary in /@text@/.
-- 
-- A boundary is caused by delimiter characters, such as
-- a newline, carriage return, carriage return-newline pair,
-- or Unicode paragraph separator character.
-- 
-- The index of the run of delimiters is returned in
-- /@paragraphDelimiterIndex@/. The index of the start of the
-- next paragraph (index after all delimiters) is stored n
-- /@nextParagraphStart@/.
-- 
-- If no delimiters are found, both /@paragraphDelimiterIndex@/
-- and /@nextParagraphStart@/ are filled with the length of /@text@/
-- (an index one off the end).
findParagraphBoundary ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@text@/: UTF-8 text
    -> Int32
    -- ^ /@length@/: length of /@text@/ in bytes, or -1 if nul-terminated
    -> m ((Int32, Int32))
findParagraphBoundary text length_ = liftIO $ do
    text' <- textToCString text
    paragraphDelimiterIndex <- allocMem :: IO (Ptr Int32)
    nextParagraphStart <- allocMem :: IO (Ptr Int32)
    pango_find_paragraph_boundary text' length_ paragraphDelimiterIndex nextParagraphStart
    paragraphDelimiterIndex' <- peek paragraphDelimiterIndex
    nextParagraphStart' <- peek nextParagraphStart
    freeMem text'
    freeMem paragraphDelimiterIndex
    freeMem nextParagraphStart
    return (paragraphDelimiterIndex', nextParagraphStart')


-- function find_base_dir
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the text to process. Must be valid UTF-8"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "length of @text in bytes (may be -1 if @text is nul-terminated)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Direction" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_find_base_dir" pango_find_base_dir :: 
    CString ->                              -- text : TBasicType TUTF8
    Int32 ->                                -- length : TBasicType TInt
    IO CUInt

-- | Searches a string the first character that has a strong
-- direction, according to the Unicode bidirectional algorithm.
-- 
-- /Since: 1.4/
findBaseDir ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@text@/: the text to process. Must be valid UTF-8
    -> Int32
    -- ^ /@length@/: length of /@text@/ in bytes (may be -1 if /@text@/ is nul-terminated)
    -> m Pango.Enums.Direction
    -- ^ __Returns:__ The direction corresponding to the first strong character.
    --   If no such character is found, then 'GI.Pango.Enums.DirectionNeutral' is returned.
findBaseDir text length_ = liftIO $ do
    text' <- textToCString text
    result <- pango_find_base_dir text' length_
    let result' = (toEnum . fromIntegral) result
    freeMem text'
    return result'


-- function extents_to_pixels
-- XXX Could not generate function extents_to_pixels

-- Not implemented: Nullable inout structs not supported

-- function default_break
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "text to break. Must be valid UTF-8"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "length of text in bytes (may be -1 if @text is nul-terminated)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "analysis"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Analysis" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoAnalysis` structure for the @text"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attrs"
--           , argType =
--               TCArray
--                 False
--                 (-1)
--                 4
--                 (TInterface Name { namespace = "Pango" , name = "LogAttr" })
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "logical\n  attributes to fill in"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attrs_len"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "size of the array passed as @attrs"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "attrs_len"
--              , argType = TBasicType TInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "size of the array passed as @attrs"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_default_break" pango_default_break :: 
    CString ->                              -- text : TBasicType TUTF8
    Int32 ->                                -- length : TBasicType TInt
    Ptr Pango.Analysis.Analysis ->          -- analysis : TInterface (Name {namespace = "Pango", name = "Analysis"})
    Ptr Pango.LogAttr.LogAttr ->            -- attrs : TCArray False (-1) 4 (TInterface (Name {namespace = "Pango", name = "LogAttr"}))
    Int32 ->                                -- attrs_len : TBasicType TInt
    IO ()

-- | This is the default break algorithm.
-- 
-- It applies rules from the <http://www.unicode.org/unicode/reports/tr14/ Unicode Line Breaking Algorithm>
-- without language-specific tailoring, therefore the /@analyis@/ argument is unused
-- and can be 'P.Nothing'.
-- 
-- See 'GI.Pango.Functions.tailorBreak' for language-specific breaks.
-- 
-- See 'GI.Pango.Functions.attrBreak' for attribute-based customization.
defaultBreak ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@text@/: text to break. Must be valid UTF-8
    -> Int32
    -- ^ /@length@/: length of text in bytes (may be -1 if /@text@/ is nul-terminated)
    -> Maybe (Pango.Analysis.Analysis)
    -- ^ /@analysis@/: a @PangoAnalysis@ structure for the /@text@/
    -> [Pango.LogAttr.LogAttr]
    -- ^ /@attrs@/: logical
    --   attributes to fill in
    -> m ([Pango.LogAttr.LogAttr])
defaultBreak text length_ analysis attrs = liftIO $ do
    let attrsLen = fromIntegral $ P.length attrs
    text' <- textToCString text
    maybeAnalysis <- case analysis of
        Nothing -> return FP.nullPtr
        Just jAnalysis -> do
            jAnalysis' <- unsafeManagedPtrGetPtr jAnalysis
            return jAnalysis'
    attrs' <- mapM unsafeManagedPtrGetPtr attrs
    attrs'' <- packBlockArray 64 attrs'
    pango_default_break text' length_ maybeAnalysis attrs'' attrsLen
    attrs''' <- (unpackBlockArrayWithLength 64 attrsLen) attrs''
    attrs'''' <- mapM (wrapPtr Pango.LogAttr.LogAttr) attrs'''
    freeMem attrs''
    whenJust analysis touchManagedPtr
    mapM_ touchManagedPtr attrs
    freeMem text'
    return attrs''''


-- function break
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the text to process. Must be valid UTF-8"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "length of @text in bytes (may be -1 if @text is nul-terminated)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "analysis"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Analysis" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "`PangoAnalysis` structure for @text"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attrs"
--           , argType =
--               TCArray
--                 False
--                 (-1)
--                 4
--                 (TInterface Name { namespace = "Pango" , name = "LogAttr" })
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an array to\n  store character information in"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attrs_len"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "size of the array passed as @attrs"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "attrs_len"
--              , argType = TBasicType TInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "size of the array passed as @attrs"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_break" pango_break :: 
    CString ->                              -- text : TBasicType TUTF8
    Int32 ->                                -- length : TBasicType TInt
    Ptr Pango.Analysis.Analysis ->          -- analysis : TInterface (Name {namespace = "Pango", name = "Analysis"})
    Ptr Pango.LogAttr.LogAttr ->            -- attrs : TCArray False (-1) 4 (TInterface (Name {namespace = "Pango", name = "LogAttr"}))
    Int32 ->                                -- attrs_len : TBasicType TInt
    IO ()

{-# DEPRECATED break ["(Since version 1.44)","Use 'GI.Pango.Functions.defaultBreak',","  'GI.Pango.Functions.tailorBreak' and 'GI.Pango.Functions.attrBreak'."] #-}
-- | Determines possible line, word, and character breaks
-- for a string of Unicode text with a single analysis.
-- 
-- For most purposes you may want to use 'GI.Pango.Functions.getLogAttrs'.
break ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@text@/: the text to process. Must be valid UTF-8
    -> Int32
    -- ^ /@length@/: length of /@text@/ in bytes (may be -1 if /@text@/ is nul-terminated)
    -> Pango.Analysis.Analysis
    -- ^ /@analysis@/: @PangoAnalysis@ structure for /@text@/
    -> [Pango.LogAttr.LogAttr]
    -- ^ /@attrs@/: an array to
    --   store character information in
    -> m ([Pango.LogAttr.LogAttr])
break text length_ analysis attrs = liftIO $ do
    let attrsLen = fromIntegral $ P.length attrs
    text' <- textToCString text
    analysis' <- unsafeManagedPtrGetPtr analysis
    attrs' <- mapM unsafeManagedPtrGetPtr attrs
    attrs'' <- packBlockArray 64 attrs'
    pango_break text' length_ analysis' attrs'' attrsLen
    attrs''' <- (unpackBlockArrayWithLength 64 attrsLen) attrs''
    attrs'''' <- mapM (wrapPtr Pango.LogAttr.LogAttr) attrs'''
    freeMem attrs''
    touchManagedPtr analysis
    mapM_ touchManagedPtr attrs
    freeMem text'
    return attrs''''


-- function attr_word_new
-- Args: []
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_word_new" pango_attr_word_new :: 
    IO (Ptr Pango.Attribute.Attribute)

-- | Marks the range of the attribute as a single word.
-- 
-- Note that this may require adjustments to word and
-- sentence classification around the range.
-- 
-- /Since: 1.50/
attrWordNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrWordNew  = liftIO $ do
    result <- pango_attr_word_new
    checkUnexpectedReturnNULL "attrWordNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_weight_new
-- Args: [ Arg
--           { argCName = "weight"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Weight" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the weight" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_weight_new" pango_attr_weight_new :: 
    CUInt ->                                -- weight : TInterface (Name {namespace = "Pango", name = "Weight"})
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new font weight attribute.
attrWeightNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Pango.Enums.Weight
    -- ^ /@weight@/: the weight
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrWeightNew weight = liftIO $ do
    let weight' = (fromIntegral . fromEnum) weight
    result <- pango_attr_weight_new weight'
    checkUnexpectedReturnNULL "attrWeightNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_variant_new
-- Args: [ Arg
--           { argCName = "variant"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Variant" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the variant" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_variant_new" pango_attr_variant_new :: 
    CUInt ->                                -- variant : TInterface (Name {namespace = "Pango", name = "Variant"})
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new font variant attribute (normal or small caps).
attrVariantNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Pango.Enums.Variant
    -- ^ /@variant@/: the variant
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated @PangoAttribute@,
    --   which should be freed with 'GI.Pango.Structs.Attribute.attributeDestroy'.
attrVariantNew variant = liftIO $ do
    let variant' = (fromIntegral . fromEnum) variant
    result <- pango_attr_variant_new variant'
    checkUnexpectedReturnNULL "attrVariantNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_underline_new
-- Args: [ Arg
--           { argCName = "underline"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Underline" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the underline style"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_underline_new" pango_attr_underline_new :: 
    CUInt ->                                -- underline : TInterface (Name {namespace = "Pango", name = "Underline"})
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new underline-style attribute.
attrUnderlineNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Pango.Enums.Underline
    -- ^ /@underline@/: the underline style
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrUnderlineNew underline = liftIO $ do
    let underline' = (fromIntegral . fromEnum) underline
    result <- pango_attr_underline_new underline'
    checkUnexpectedReturnNULL "attrUnderlineNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_underline_color_new
-- Args: [ Arg
--           { argCName = "red"
--           , argType = TBasicType TUInt16
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the red value (ranging from 0 to 65535)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "green"
--           , argType = TBasicType TUInt16
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the green value" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "blue"
--           , argType = TBasicType TUInt16
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the blue value" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_underline_color_new" pango_attr_underline_color_new :: 
    Word16 ->                               -- red : TBasicType TUInt16
    Word16 ->                               -- green : TBasicType TUInt16
    Word16 ->                               -- blue : TBasicType TUInt16
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new underline color attribute.
-- 
-- This attribute modifies the color of underlines.
-- If not set, underlines will use the foreground color.
-- 
-- /Since: 1.8/
attrUnderlineColorNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Word16
    -- ^ /@red@/: the red value (ranging from 0 to 65535)
    -> Word16
    -- ^ /@green@/: the green value
    -> Word16
    -- ^ /@blue@/: the blue value
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrUnderlineColorNew red green blue = liftIO $ do
    result <- pango_attr_underline_color_new red green blue
    checkUnexpectedReturnNULL "attrUnderlineColorNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_text_transform_new
-- Args: [ Arg
--           { argCName = "transform"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "TextTransform" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "`PangoTextTransform` to apply"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_text_transform_new" pango_attr_text_transform_new :: 
    CUInt ->                                -- transform : TInterface (Name {namespace = "Pango", name = "TextTransform"})
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new attribute that influences how characters
-- are transformed during shaping.
-- 
-- /Since: 1.50/
attrTextTransformNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Pango.Enums.TextTransform
    -- ^ /@transform@/: @PangoTextTransform@ to apply
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrTextTransformNew transform = liftIO $ do
    let transform' = (fromIntegral . fromEnum) transform
    result <- pango_attr_text_transform_new transform'
    checkUnexpectedReturnNULL "attrTextTransformNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_style_new
-- Args: [ Arg
--           { argCName = "style"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Style" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the slant style" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_style_new" pango_attr_style_new :: 
    CUInt ->                                -- style : TInterface (Name {namespace = "Pango", name = "Style"})
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new font slant style attribute.
attrStyleNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Pango.Enums.Style
    -- ^ /@style@/: the slant style
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrStyleNew style = liftIO $ do
    let style' = (fromIntegral . fromEnum) style
    result <- pango_attr_style_new style'
    checkUnexpectedReturnNULL "attrStyleNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_strikethrough_new
-- Args: [ Arg
--           { argCName = "strikethrough"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "%TRUE if the text should be struck-through"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_strikethrough_new" pango_attr_strikethrough_new :: 
    CInt ->                                 -- strikethrough : TBasicType TBoolean
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new strike-through attribute.
attrStrikethroughNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Bool
    -- ^ /@strikethrough@/: 'P.True' if the text should be struck-through
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrStrikethroughNew strikethrough = liftIO $ do
    let strikethrough' = (P.fromIntegral . P.fromEnum) strikethrough
    result <- pango_attr_strikethrough_new strikethrough'
    checkUnexpectedReturnNULL "attrStrikethroughNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_strikethrough_color_new
-- Args: [ Arg
--           { argCName = "red"
--           , argType = TBasicType TUInt16
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the red value (ranging from 0 to 65535)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "green"
--           , argType = TBasicType TUInt16
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the green value" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "blue"
--           , argType = TBasicType TUInt16
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the blue value" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_strikethrough_color_new" pango_attr_strikethrough_color_new :: 
    Word16 ->                               -- red : TBasicType TUInt16
    Word16 ->                               -- green : TBasicType TUInt16
    Word16 ->                               -- blue : TBasicType TUInt16
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new strikethrough color attribute.
-- 
-- This attribute modifies the color of strikethrough lines.
-- If not set, strikethrough lines will use the foreground color.
-- 
-- /Since: 1.8/
attrStrikethroughColorNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Word16
    -- ^ /@red@/: the red value (ranging from 0 to 65535)
    -> Word16
    -- ^ /@green@/: the green value
    -> Word16
    -- ^ /@blue@/: the blue value
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrStrikethroughColorNew red green blue = liftIO $ do
    result <- pango_attr_strikethrough_color_new red green blue
    checkUnexpectedReturnNULL "attrStrikethroughColorNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_stretch_new
-- Args: [ Arg
--           { argCName = "stretch"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Stretch" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the stretch" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_stretch_new" pango_attr_stretch_new :: 
    CUInt ->                                -- stretch : TInterface (Name {namespace = "Pango", name = "Stretch"})
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new font stretch attribute.
attrStretchNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Pango.Enums.Stretch
    -- ^ /@stretch@/: the stretch
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrStretchNew stretch = liftIO $ do
    let stretch' = (fromIntegral . fromEnum) stretch
    result <- pango_attr_stretch_new stretch'
    checkUnexpectedReturnNULL "attrStretchNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_show_new
-- Args: [ Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "ShowFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "`PangoShowFlags` to apply"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_show_new" pango_attr_show_new :: 
    CUInt ->                                -- flags : TInterface (Name {namespace = "Pango", name = "ShowFlags"})
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new attribute that influences how invisible
-- characters are rendered.
-- 
-- /Since: 1.44/
attrShowNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    [Pango.Flags.ShowFlags]
    -- ^ /@flags@/: @PangoShowFlags@ to apply
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrShowNew flags = liftIO $ do
    let flags' = gflagsToWord flags
    result <- pango_attr_show_new flags'
    checkUnexpectedReturnNULL "attrShowNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_sentence_new
-- Args: []
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_sentence_new" pango_attr_sentence_new :: 
    IO (Ptr Pango.Attribute.Attribute)

-- | Marks the range of the attribute as a single sentence.
-- 
-- Note that this may require adjustments to word and
-- sentence classification around the range.
-- 
-- /Since: 1.50/
attrSentenceNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrSentenceNew  = liftIO $ do
    result <- pango_attr_sentence_new
    checkUnexpectedReturnNULL "attrSentenceNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_scale_new
-- Args: [ Arg
--           { argCName = "scale_factor"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "factor to scale the font"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_scale_new" pango_attr_scale_new :: 
    CDouble ->                              -- scale_factor : TBasicType TDouble
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new font size scale attribute.
-- 
-- The base font for the affected text will have
-- its size multiplied by /@scaleFactor@/.
attrScaleNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Double
    -- ^ /@scaleFactor@/: factor to scale the font
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrScaleNew scaleFactor = liftIO $ do
    let scaleFactor' = realToFrac scaleFactor
    result <- pango_attr_scale_new scaleFactor'
    checkUnexpectedReturnNULL "attrScaleNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_rise_new
-- Args: [ Arg
--           { argCName = "rise"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the amount that the text should be displaced vertically,\n  in Pango units. Positive values displace the text upwards."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_rise_new" pango_attr_rise_new :: 
    Int32 ->                                -- rise : TBasicType TInt
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new baseline displacement attribute.
attrRiseNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Int32
    -- ^ /@rise@/: the amount that the text should be displaced vertically,
    --   in Pango units. Positive values displace the text upwards.
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrRiseNew rise = liftIO $ do
    result <- pango_attr_rise_new rise
    checkUnexpectedReturnNULL "attrRiseNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_overline_new
-- Args: [ Arg
--           { argCName = "overline"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Overline" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the overline style" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_overline_new" pango_attr_overline_new :: 
    CUInt ->                                -- overline : TInterface (Name {namespace = "Pango", name = "Overline"})
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new overline-style attribute.
-- 
-- /Since: 1.46/
attrOverlineNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Pango.Enums.Overline
    -- ^ /@overline@/: the overline style
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrOverlineNew overline = liftIO $ do
    let overline' = (fromIntegral . fromEnum) overline
    result <- pango_attr_overline_new overline'
    checkUnexpectedReturnNULL "attrOverlineNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_overline_color_new
-- Args: [ Arg
--           { argCName = "red"
--           , argType = TBasicType TUInt16
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the red value (ranging from 0 to 65535)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "green"
--           , argType = TBasicType TUInt16
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the green value" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "blue"
--           , argType = TBasicType TUInt16
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the blue value" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_overline_color_new" pango_attr_overline_color_new :: 
    Word16 ->                               -- red : TBasicType TUInt16
    Word16 ->                               -- green : TBasicType TUInt16
    Word16 ->                               -- blue : TBasicType TUInt16
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new overline color attribute.
-- 
-- This attribute modifies the color of overlines.
-- If not set, overlines will use the foreground color.
-- 
-- /Since: 1.46/
attrOverlineColorNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Word16
    -- ^ /@red@/: the red value (ranging from 0 to 65535)
    -> Word16
    -- ^ /@green@/: the green value
    -> Word16
    -- ^ /@blue@/: the blue value
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrOverlineColorNew red green blue = liftIO $ do
    result <- pango_attr_overline_color_new red green blue
    checkUnexpectedReturnNULL "attrOverlineColorNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_line_height_new_absolute
-- Args: [ Arg
--           { argCName = "height"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the line height, in %PANGO_SCALE-ths of a point"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_line_height_new_absolute" pango_attr_line_height_new_absolute :: 
    Int32 ->                                -- height : TBasicType TInt
    IO (Ptr Pango.Attribute.Attribute)

-- | Override the height of logical line extents to be /@height@/.
-- 
-- This affects the values returned by
-- 'GI.Pango.Structs.LayoutLine.layoutLineGetExtents',
-- 'GI.Pango.Structs.LayoutLine.layoutLineGetPixelExtents' and
-- 'GI.Pango.Structs.LayoutIter.layoutIterGetLineExtents'.
-- 
-- /Since: 1.50/
attrLineHeightNewAbsolute ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Int32
    -- ^ /@height@/: the line height, in 'GI.Pango.Constants.SCALE'-ths of a point
    -> m Pango.Attribute.Attribute
attrLineHeightNewAbsolute height = liftIO $ do
    result <- pango_attr_line_height_new_absolute height
    checkUnexpectedReturnNULL "attrLineHeightNewAbsolute" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_line_height_new
-- Args: [ Arg
--           { argCName = "factor"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the scaling factor to apply to the logical height"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_line_height_new" pango_attr_line_height_new :: 
    CDouble ->                              -- factor : TBasicType TDouble
    IO (Ptr Pango.Attribute.Attribute)

-- | Modify the height of logical line extents by a factor.
-- 
-- This affects the values returned by
-- 'GI.Pango.Structs.LayoutLine.layoutLineGetExtents',
-- 'GI.Pango.Structs.LayoutLine.layoutLineGetPixelExtents' and
-- 'GI.Pango.Structs.LayoutIter.layoutIterGetLineExtents'.
-- 
-- /Since: 1.50/
attrLineHeightNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Double
    -- ^ /@factor@/: the scaling factor to apply to the logical height
    -> m Pango.Attribute.Attribute
attrLineHeightNew factor = liftIO $ do
    let factor' = realToFrac factor
    result <- pango_attr_line_height_new factor'
    checkUnexpectedReturnNULL "attrLineHeightNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_letter_spacing_new
-- Args: [ Arg
--           { argCName = "letter_spacing"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "amount of extra space to add between\n  graphemes of the text, in Pango units"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_letter_spacing_new" pango_attr_letter_spacing_new :: 
    Int32 ->                                -- letter_spacing : TBasicType TInt
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new letter-spacing attribute.
-- 
-- /Since: 1.6/
attrLetterSpacingNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Int32
    -- ^ /@letterSpacing@/: amount of extra space to add between
    --   graphemes of the text, in Pango units
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrLetterSpacingNew letterSpacing = liftIO $ do
    result <- pango_attr_letter_spacing_new letterSpacing
    checkUnexpectedReturnNULL "attrLetterSpacingNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_insert_hyphens_new
-- Args: [ Arg
--           { argCName = "insert_hyphens"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "%TRUE if hyphens should be inserted"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_insert_hyphens_new" pango_attr_insert_hyphens_new :: 
    CInt ->                                 -- insert_hyphens : TBasicType TBoolean
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new insert-hyphens attribute.
-- 
-- Pango will insert hyphens when breaking lines in
-- the middle of a word. This attribute can be used
-- to suppress the hyphen.
-- 
-- /Since: 1.44/
attrInsertHyphensNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Bool
    -- ^ /@insertHyphens@/: 'P.True' if hyphens should be inserted
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrInsertHyphensNew insertHyphens = liftIO $ do
    let insertHyphens' = (P.fromIntegral . P.fromEnum) insertHyphens
    result <- pango_attr_insert_hyphens_new insertHyphens'
    checkUnexpectedReturnNULL "attrInsertHyphensNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_gravity_new
-- Args: [ Arg
--           { argCName = "gravity"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Gravity" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the gravity value; should not be %PANGO_GRAVITY_AUTO"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_gravity_new" pango_attr_gravity_new :: 
    CUInt ->                                -- gravity : TInterface (Name {namespace = "Pango", name = "Gravity"})
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new gravity attribute.
-- 
-- /Since: 1.16/
attrGravityNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Pango.Enums.Gravity
    -- ^ /@gravity@/: the gravity value; should not be 'GI.Pango.Enums.GravityAuto'
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrGravityNew gravity = liftIO $ do
    let gravity' = (fromIntegral . fromEnum) gravity
    result <- pango_attr_gravity_new gravity'
    checkUnexpectedReturnNULL "attrGravityNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_gravity_hint_new
-- Args: [ Arg
--           { argCName = "hint"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "GravityHint" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the gravity hint value"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_gravity_hint_new" pango_attr_gravity_hint_new :: 
    CUInt ->                                -- hint : TInterface (Name {namespace = "Pango", name = "GravityHint"})
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new gravity hint attribute.
-- 
-- /Since: 1.16/
attrGravityHintNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Pango.Enums.GravityHint
    -- ^ /@hint@/: the gravity hint value
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrGravityHintNew hint = liftIO $ do
    let hint' = (fromIntegral . fromEnum) hint
    result <- pango_attr_gravity_hint_new hint'
    checkUnexpectedReturnNULL "attrGravityHintNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_foreground_new
-- Args: [ Arg
--           { argCName = "red"
--           , argType = TBasicType TUInt16
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the red value (ranging from 0 to 65535)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "green"
--           , argType = TBasicType TUInt16
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the green value" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "blue"
--           , argType = TBasicType TUInt16
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the blue value" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_foreground_new" pango_attr_foreground_new :: 
    Word16 ->                               -- red : TBasicType TUInt16
    Word16 ->                               -- green : TBasicType TUInt16
    Word16 ->                               -- blue : TBasicType TUInt16
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new foreground color attribute.
attrForegroundNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Word16
    -- ^ /@red@/: the red value (ranging from 0 to 65535)
    -> Word16
    -- ^ /@green@/: the green value
    -> Word16
    -- ^ /@blue@/: the blue value
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrForegroundNew red green blue = liftIO $ do
    result <- pango_attr_foreground_new red green blue
    checkUnexpectedReturnNULL "attrForegroundNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_foreground_alpha_new
-- Args: [ Arg
--           { argCName = "alpha"
--           , argType = TBasicType TUInt16
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the alpha value, between 1 and 65536"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_foreground_alpha_new" pango_attr_foreground_alpha_new :: 
    Word16 ->                               -- alpha : TBasicType TUInt16
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new foreground alpha attribute.
-- 
-- /Since: 1.38/
attrForegroundAlphaNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Word16
    -- ^ /@alpha@/: the alpha value, between 1 and 65536
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrForegroundAlphaNew alpha = liftIO $ do
    result <- pango_attr_foreground_alpha_new alpha
    checkUnexpectedReturnNULL "attrForegroundAlphaNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_font_scale_new
-- Args: [ Arg
--           { argCName = "scale"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "FontScale" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a `PangoFontScale` value, which indicates font size change relative\n  to the size of the previous run."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_font_scale_new" pango_attr_font_scale_new :: 
    CUInt ->                                -- scale : TInterface (Name {namespace = "Pango", name = "FontScale"})
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new font scale attribute.
-- 
-- The effect of this attribute is to change the font size of a run,
-- relative to the size of preceding run.
-- 
-- /Since: 1.50/
attrFontScaleNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Pango.Enums.FontScale
    -- ^ /@scale@/: a @PangoFontScale@ value, which indicates font size change relative
    --   to the size of the previous run.
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrFontScaleNew scale = liftIO $ do
    let scale' = (fromIntegral . fromEnum) scale
    result <- pango_attr_font_scale_new scale'
    checkUnexpectedReturnNULL "attrFontScaleNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_family_new
-- Args: [ Arg
--           { argCName = "family"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the family or comma-separated list of families"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_family_new" pango_attr_family_new :: 
    CString ->                              -- family : TBasicType TUTF8
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new font family attribute.
attrFamilyNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@family@/: the family or comma-separated list of families
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrFamilyNew family = liftIO $ do
    family' <- textToCString family
    result <- pango_attr_family_new family'
    checkUnexpectedReturnNULL "attrFamilyNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    freeMem family'
    return result'


-- function attr_fallback_new
-- Args: [ Arg
--           { argCName = "enable_fallback"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "%TRUE if we should fall back on other fonts\n  for characters the active font is missing"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_fallback_new" pango_attr_fallback_new :: 
    CInt ->                                 -- enable_fallback : TBasicType TBoolean
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new font fallback attribute.
-- 
-- If fallback is disabled, characters will only be
-- used from the closest matching font on the system.
-- No fallback will be done to other fonts on the system
-- that might contain the characters in the text.
-- 
-- /Since: 1.4/
attrFallbackNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Bool
    -- ^ /@enableFallback@/: 'P.True' if we should fall back on other fonts
    --   for characters the active font is missing
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrFallbackNew enableFallback = liftIO $ do
    let enableFallback' = (P.fromIntegral . P.fromEnum) enableFallback
    result <- pango_attr_fallback_new enableFallback'
    checkUnexpectedReturnNULL "attrFallbackNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_break
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "text to break. Must be valid UTF-8"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "length of text in bytes (may be -1 if @text is nul-terminated)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attr_list"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "`PangoAttrList` to apply"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "Byte offset of @text from the beginning of the paragraph"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attrs"
--           , argType =
--               TCArray
--                 False
--                 (-1)
--                 5
--                 (TInterface Name { namespace = "Pango" , name = "LogAttr" })
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "array with one\n  `PangoLogAttr` per character in @text, plus one extra, to be filled in"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attrs_len"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "length of @attrs array"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "attrs_len"
--              , argType = TBasicType TInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "length of @attrs array"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_break" pango_attr_break :: 
    CString ->                              -- text : TBasicType TUTF8
    Int32 ->                                -- length : TBasicType TInt
    Ptr Pango.AttrList.AttrList ->          -- attr_list : TInterface (Name {namespace = "Pango", name = "AttrList"})
    Int32 ->                                -- offset : TBasicType TInt
    Ptr Pango.LogAttr.LogAttr ->            -- attrs : TCArray False (-1) 5 (TInterface (Name {namespace = "Pango", name = "LogAttr"}))
    Int32 ->                                -- attrs_len : TBasicType TInt
    IO ()

-- | Apply customization from attributes to the breaks in /@attrs@/.
-- 
-- The line breaks are assumed to have been produced
-- by 'GI.Pango.Functions.defaultBreak' and 'GI.Pango.Functions.tailorBreak'.
-- 
-- /Since: 1.50/
attrBreak ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@text@/: text to break. Must be valid UTF-8
    -> Int32
    -- ^ /@length@/: length of text in bytes (may be -1 if /@text@/ is nul-terminated)
    -> Pango.AttrList.AttrList
    -- ^ /@attrList@/: @PangoAttrList@ to apply
    -> Int32
    -- ^ /@offset@/: Byte offset of /@text@/ from the beginning of the paragraph
    -> [Pango.LogAttr.LogAttr]
    -- ^ /@attrs@/: array with one
    --   @PangoLogAttr@ per character in /@text@/, plus one extra, to be filled in
    -> m ([Pango.LogAttr.LogAttr])
attrBreak text length_ attrList offset attrs = liftIO $ do
    let attrsLen = fromIntegral $ P.length attrs
    text' <- textToCString text
    attrList' <- unsafeManagedPtrGetPtr attrList
    attrs' <- mapM unsafeManagedPtrGetPtr attrs
    attrs'' <- packBlockArray 64 attrs'
    pango_attr_break text' length_ attrList' offset attrs'' attrsLen
    attrs''' <- (unpackBlockArrayWithLength 64 attrsLen) attrs''
    attrs'''' <- mapM (wrapPtr Pango.LogAttr.LogAttr) attrs'''
    freeMem attrs''
    touchManagedPtr attrList
    mapM_ touchManagedPtr attrs
    freeMem text'
    return attrs''''


-- function attr_baseline_shift_new
-- Args: [ Arg
--           { argCName = "shift"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "either a `PangoBaselineShift` enumeration value or an absolute value (> 1024)\n  in Pango units, relative to the baseline of the previous run.\n  Positive values displace the text upwards."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_baseline_shift_new" pango_attr_baseline_shift_new :: 
    Int32 ->                                -- shift : TBasicType TInt
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new baseline displacement attribute.
-- 
-- The effect of this attribute is to shift the baseline of a run,
-- relative to the run of preceding run.
-- 
-- \<picture>
--   \<source srcset=\"baseline-shift-dark.png\" media=\"(prefers-color-scheme: dark)\">
--   \<img alt=\"Baseline Shift\" src=\"baseline-shift-light.png\">
-- \<\/picture>
-- 
-- /Since: 1.50/
attrBaselineShiftNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Int32
    -- ^ /@shift@/: either a @PangoBaselineShift@ enumeration value or an absolute value (> 1024)
    --   in Pango units, relative to the baseline of the previous run.
    --   Positive values displace the text upwards.
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrBaselineShiftNew shift = liftIO $ do
    result <- pango_attr_baseline_shift_new shift
    checkUnexpectedReturnNULL "attrBaselineShiftNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_background_new
-- Args: [ Arg
--           { argCName = "red"
--           , argType = TBasicType TUInt16
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the red value (ranging from 0 to 65535)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "green"
--           , argType = TBasicType TUInt16
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the green value" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "blue"
--           , argType = TBasicType TUInt16
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the blue value" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_background_new" pango_attr_background_new :: 
    Word16 ->                               -- red : TBasicType TUInt16
    Word16 ->                               -- green : TBasicType TUInt16
    Word16 ->                               -- blue : TBasicType TUInt16
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new background color attribute.
attrBackgroundNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Word16
    -- ^ /@red@/: the red value (ranging from 0 to 65535)
    -> Word16
    -- ^ /@green@/: the green value
    -> Word16
    -- ^ /@blue@/: the blue value
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrBackgroundNew red green blue = liftIO $ do
    result <- pango_attr_background_new red green blue
    checkUnexpectedReturnNULL "attrBackgroundNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_background_alpha_new
-- Args: [ Arg
--           { argCName = "alpha"
--           , argType = TBasicType TUInt16
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the alpha value, between 1 and 65536"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_background_alpha_new" pango_attr_background_alpha_new :: 
    Word16 ->                               -- alpha : TBasicType TUInt16
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new background alpha attribute.
-- 
-- /Since: 1.38/
attrBackgroundAlphaNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Word16
    -- ^ /@alpha@/: the alpha value, between 1 and 65536
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrBackgroundAlphaNew alpha = liftIO $ do
    result <- pango_attr_background_alpha_new alpha
    checkUnexpectedReturnNULL "attrBackgroundAlphaNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'


-- function attr_allow_breaks_new
-- Args: [ Arg
--           { argCName = "allow_breaks"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "%TRUE if we line breaks are allowed"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Attribute" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_allow_breaks_new" pango_attr_allow_breaks_new :: 
    CInt ->                                 -- allow_breaks : TBasicType TBoolean
    IO (Ptr Pango.Attribute.Attribute)

-- | Create a new allow-breaks attribute.
-- 
-- If breaks are disabled, the range will be kept in a
-- single run, as far as possible.
-- 
-- /Since: 1.44/
attrAllowBreaksNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Bool
    -- ^ /@allowBreaks@/: 'P.True' if we line breaks are allowed
    -> m Pango.Attribute.Attribute
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttribute@, which should be freed with
    --   'GI.Pango.Structs.Attribute.attributeDestroy'
attrAllowBreaksNew allowBreaks = liftIO $ do
    let allowBreaks' = (P.fromIntegral . P.fromEnum) allowBreaks
    result <- pango_attr_allow_breaks_new allowBreaks'
    checkUnexpectedReturnNULL "attrAllowBreaksNew" result
    result' <- (wrapBoxed Pango.Attribute.Attribute) result
    return result'



