﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/iotfleetwise/IoTFleetWiseRequest.h>
#include <aws/iotfleetwise/IoTFleetWise_EXPORTS.h>
#include <aws/iotfleetwise/model/Tag.h>

#include <utility>

namespace Aws {
namespace IoTFleetWise {
namespace Model {

/**
 */
class CreateFleetRequest : public IoTFleetWiseRequest {
 public:
  AWS_IOTFLEETWISE_API CreateFleetRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateFleet"; }

  AWS_IOTFLEETWISE_API Aws::String SerializePayload() const override;

  AWS_IOTFLEETWISE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p> The unique ID of the fleet to create. </p>
   */
  inline const Aws::String& GetFleetId() const { return m_fleetId; }
  inline bool FleetIdHasBeenSet() const { return m_fleetIdHasBeenSet; }
  template <typename FleetIdT = Aws::String>
  void SetFleetId(FleetIdT&& value) {
    m_fleetIdHasBeenSet = true;
    m_fleetId = std::forward<FleetIdT>(value);
  }
  template <typename FleetIdT = Aws::String>
  CreateFleetRequest& WithFleetId(FleetIdT&& value) {
    SetFleetId(std::forward<FleetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A brief description of the fleet to create. </p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateFleetRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The Amazon Resource Name (ARN) of a signal catalog. </p>
   */
  inline const Aws::String& GetSignalCatalogArn() const { return m_signalCatalogArn; }
  inline bool SignalCatalogArnHasBeenSet() const { return m_signalCatalogArnHasBeenSet; }
  template <typename SignalCatalogArnT = Aws::String>
  void SetSignalCatalogArn(SignalCatalogArnT&& value) {
    m_signalCatalogArnHasBeenSet = true;
    m_signalCatalogArn = std::forward<SignalCatalogArnT>(value);
  }
  template <typename SignalCatalogArnT = Aws::String>
  CreateFleetRequest& WithSignalCatalogArn(SignalCatalogArnT&& value) {
    SetSignalCatalogArn(std::forward<SignalCatalogArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Metadata that can be used to manage the fleet.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateFleetRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateFleetRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_fleetId;

  Aws::String m_description;

  Aws::String m_signalCatalogArn;

  Aws::Vector<Tag> m_tags;
  bool m_fleetIdHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_signalCatalogArnHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoTFleetWise
}  // namespace Aws
