﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/textract/TextractRequest.h>
#include <aws/textract/Textract_EXPORTS.h>
#include <aws/textract/model/AdaptersConfig.h>
#include <aws/textract/model/DocumentLocation.h>
#include <aws/textract/model/FeatureType.h>
#include <aws/textract/model/NotificationChannel.h>
#include <aws/textract/model/OutputConfig.h>
#include <aws/textract/model/QueriesConfig.h>

#include <utility>

namespace Aws {
namespace Textract {
namespace Model {

/**
 */
class StartDocumentAnalysisRequest : public TextractRequest {
 public:
  AWS_TEXTRACT_API StartDocumentAnalysisRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartDocumentAnalysis"; }

  AWS_TEXTRACT_API Aws::String SerializePayload() const override;

  AWS_TEXTRACT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The location of the document to be processed.</p>
   */
  inline const DocumentLocation& GetDocumentLocation() const { return m_documentLocation; }
  inline bool DocumentLocationHasBeenSet() const { return m_documentLocationHasBeenSet; }
  template <typename DocumentLocationT = DocumentLocation>
  void SetDocumentLocation(DocumentLocationT&& value) {
    m_documentLocationHasBeenSet = true;
    m_documentLocation = std::forward<DocumentLocationT>(value);
  }
  template <typename DocumentLocationT = DocumentLocation>
  StartDocumentAnalysisRequest& WithDocumentLocation(DocumentLocationT&& value) {
    SetDocumentLocation(std::forward<DocumentLocationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of the types of analysis to perform. Add TABLES to the list to return
   * information about the tables that are detected in the input document. Add FORMS
   * to return detected form data. To perform both types of analysis, add TABLES and
   * FORMS to <code>FeatureTypes</code>. All lines and words detected in the document
   * are included in the response (including text that isn't related to the value of
   * <code>FeatureTypes</code>). </p>
   */
  inline const Aws::Vector<FeatureType>& GetFeatureTypes() const { return m_featureTypes; }
  inline bool FeatureTypesHasBeenSet() const { return m_featureTypesHasBeenSet; }
  template <typename FeatureTypesT = Aws::Vector<FeatureType>>
  void SetFeatureTypes(FeatureTypesT&& value) {
    m_featureTypesHasBeenSet = true;
    m_featureTypes = std::forward<FeatureTypesT>(value);
  }
  template <typename FeatureTypesT = Aws::Vector<FeatureType>>
  StartDocumentAnalysisRequest& WithFeatureTypes(FeatureTypesT&& value) {
    SetFeatureTypes(std::forward<FeatureTypesT>(value));
    return *this;
  }
  inline StartDocumentAnalysisRequest& AddFeatureTypes(FeatureType value) {
    m_featureTypesHasBeenSet = true;
    m_featureTypes.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The idempotent token that you use to identify the start request. If you use
   * the same token with multiple <code>StartDocumentAnalysis</code> requests, the
   * same <code>JobId</code> is returned. Use <code>ClientRequestToken</code> to
   * prevent the same job from being accidentally started more than once. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/textract/latest/dg/api-async.html">Calling
   * Amazon Textract Asynchronous Operations</a>.</p>
   */
  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  StartDocumentAnalysisRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An identifier that you specify that's included in the completion notification
   * published to the Amazon SNS topic. For example, you can use <code>JobTag</code>
   * to identify the type of document that the completion notification corresponds to
   * (such as a tax form or a receipt).</p>
   */
  inline const Aws::String& GetJobTag() const { return m_jobTag; }
  inline bool JobTagHasBeenSet() const { return m_jobTagHasBeenSet; }
  template <typename JobTagT = Aws::String>
  void SetJobTag(JobTagT&& value) {
    m_jobTagHasBeenSet = true;
    m_jobTag = std::forward<JobTagT>(value);
  }
  template <typename JobTagT = Aws::String>
  StartDocumentAnalysisRequest& WithJobTag(JobTagT&& value) {
    SetJobTag(std::forward<JobTagT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon SNS topic ARN that you want Amazon Textract to publish the
   * completion status of the operation to. </p>
   */
  inline const NotificationChannel& GetNotificationChannel() const { return m_notificationChannel; }
  inline bool NotificationChannelHasBeenSet() const { return m_notificationChannelHasBeenSet; }
  template <typename NotificationChannelT = NotificationChannel>
  void SetNotificationChannel(NotificationChannelT&& value) {
    m_notificationChannelHasBeenSet = true;
    m_notificationChannel = std::forward<NotificationChannelT>(value);
  }
  template <typename NotificationChannelT = NotificationChannel>
  StartDocumentAnalysisRequest& WithNotificationChannel(NotificationChannelT&& value) {
    SetNotificationChannel(std::forward<NotificationChannelT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Sets if the output will go to a customer defined bucket. By default, Amazon
   * Textract will save the results internally to be accessed by the
   * GetDocumentAnalysis operation.</p>
   */
  inline const OutputConfig& GetOutputConfig() const { return m_outputConfig; }
  inline bool OutputConfigHasBeenSet() const { return m_outputConfigHasBeenSet; }
  template <typename OutputConfigT = OutputConfig>
  void SetOutputConfig(OutputConfigT&& value) {
    m_outputConfigHasBeenSet = true;
    m_outputConfig = std::forward<OutputConfigT>(value);
  }
  template <typename OutputConfigT = OutputConfig>
  StartDocumentAnalysisRequest& WithOutputConfig(OutputConfigT&& value) {
    SetOutputConfig(std::forward<OutputConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The KMS key used to encrypt the inference results. This can be in either Key
   * ID or Key Alias format. When a KMS key is provided, the KMS key will be used for
   * server-side encryption of the objects in the customer bucket. When this
   * parameter is not enabled, the result will be encrypted server side,using
   * SSE-S3.</p>
   */
  inline const Aws::String& GetKMSKeyId() const { return m_kMSKeyId; }
  inline bool KMSKeyIdHasBeenSet() const { return m_kMSKeyIdHasBeenSet; }
  template <typename KMSKeyIdT = Aws::String>
  void SetKMSKeyId(KMSKeyIdT&& value) {
    m_kMSKeyIdHasBeenSet = true;
    m_kMSKeyId = std::forward<KMSKeyIdT>(value);
  }
  template <typename KMSKeyIdT = Aws::String>
  StartDocumentAnalysisRequest& WithKMSKeyId(KMSKeyIdT&& value) {
    SetKMSKeyId(std::forward<KMSKeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const QueriesConfig& GetQueriesConfig() const { return m_queriesConfig; }
  inline bool QueriesConfigHasBeenSet() const { return m_queriesConfigHasBeenSet; }
  template <typename QueriesConfigT = QueriesConfig>
  void SetQueriesConfig(QueriesConfigT&& value) {
    m_queriesConfigHasBeenSet = true;
    m_queriesConfig = std::forward<QueriesConfigT>(value);
  }
  template <typename QueriesConfigT = QueriesConfig>
  StartDocumentAnalysisRequest& WithQueriesConfig(QueriesConfigT&& value) {
    SetQueriesConfig(std::forward<QueriesConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the adapter to be used when analyzing a document.</p>
   */
  inline const AdaptersConfig& GetAdaptersConfig() const { return m_adaptersConfig; }
  inline bool AdaptersConfigHasBeenSet() const { return m_adaptersConfigHasBeenSet; }
  template <typename AdaptersConfigT = AdaptersConfig>
  void SetAdaptersConfig(AdaptersConfigT&& value) {
    m_adaptersConfigHasBeenSet = true;
    m_adaptersConfig = std::forward<AdaptersConfigT>(value);
  }
  template <typename AdaptersConfigT = AdaptersConfig>
  StartDocumentAnalysisRequest& WithAdaptersConfig(AdaptersConfigT&& value) {
    SetAdaptersConfig(std::forward<AdaptersConfigT>(value));
    return *this;
  }
  ///@}
 private:
  DocumentLocation m_documentLocation;

  Aws::Vector<FeatureType> m_featureTypes;

  Aws::String m_clientRequestToken;

  Aws::String m_jobTag;

  NotificationChannel m_notificationChannel;

  OutputConfig m_outputConfig;

  Aws::String m_kMSKeyId;

  QueriesConfig m_queriesConfig;

  AdaptersConfig m_adaptersConfig;
  bool m_documentLocationHasBeenSet = false;
  bool m_featureTypesHasBeenSet = false;
  bool m_clientRequestTokenHasBeenSet = false;
  bool m_jobTagHasBeenSet = false;
  bool m_notificationChannelHasBeenSet = false;
  bool m_outputConfigHasBeenSet = false;
  bool m_kMSKeyIdHasBeenSet = false;
  bool m_queriesConfigHasBeenSet = false;
  bool m_adaptersConfigHasBeenSet = false;
};

}  // namespace Model
}  // namespace Textract
}  // namespace Aws
