/*  This file is part of SAIL (https://github.com/HappySeaFox/sail)

    Copyright (c) 2020 Dmitry Baryshev

    The MIT License

    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in all
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
    SOFTWARE.
*/

#include <stdbool.h>

#include <sail-common/sail-common.h>

#include "common/bmp/bmp.h"

/*
 * Codec-specific state.
 */
struct bmp_state
{
    struct sail_io* io;
    const struct sail_load_options* load_options;
    const struct sail_save_options* save_options;

    bool frame_processed;
    void* common_bmp_state;
};

static sail_status_t alloc_bmp_state(struct sail_io* io,
                                     const struct sail_load_options* load_options,
                                     const struct sail_save_options* save_options,
                                     struct bmp_state** bmp_state)
{
    void* ptr;
    SAIL_TRY(sail_malloc(sizeof(struct bmp_state), &ptr));
    *bmp_state = ptr;

    **bmp_state = (struct bmp_state){
        .io           = io,
        .load_options = load_options,
        .save_options = save_options,

        .frame_processed  = false,
        .common_bmp_state = NULL,
    };

    return SAIL_OK;
}

static void destroy_bmp_state(struct bmp_state* bmp_state)
{
    if (bmp_state == NULL)
    {
        return;
    }

    sail_free(bmp_state);
}

/*
 * Decoding functions.
 */

SAIL_EXPORT sail_status_t sail_codec_load_init_v8_bmp(struct sail_io* io,
                                                      const struct sail_load_options* load_options,
                                                      void** state)
{
    *state = NULL;

    /* Allocate a new state. */
    struct bmp_state* bmp_state;
    SAIL_TRY(alloc_bmp_state(io, load_options, NULL, &bmp_state));
    *state = bmp_state;

    SAIL_TRY(
        bmp_private_read_init(io, bmp_state->load_options, &bmp_state->common_bmp_state, SAIL_READ_BMP_FILE_HEADER));

    return SAIL_OK;
}

SAIL_EXPORT sail_status_t sail_codec_load_seek_next_frame_v8_bmp(void* state, struct sail_image** image)
{
    struct bmp_state* bmp_state = state;

    if (bmp_state->frame_processed)
    {
        return SAIL_ERROR_NO_MORE_FRAMES;
    }

    bmp_state->frame_processed = true;

    SAIL_TRY(bmp_private_read_seek_next_frame(bmp_state->common_bmp_state, bmp_state->io, image));

    return SAIL_OK;
}

SAIL_EXPORT sail_status_t sail_codec_load_frame_v8_bmp(void* state, struct sail_image* image)
{
    struct bmp_state* bmp_state = state;

    SAIL_TRY(bmp_private_read_frame(bmp_state->common_bmp_state, bmp_state->io, image));

    return SAIL_OK;
}

SAIL_EXPORT sail_status_t sail_codec_load_finish_v8_bmp(void** state)
{
    struct bmp_state* bmp_state = *state;

    *state = NULL;

    if (bmp_state->common_bmp_state != NULL)
    {
        SAIL_TRY_OR_CLEANUP(bmp_private_read_finish(&bmp_state->common_bmp_state, bmp_state->io),
                            /* cleanup */ destroy_bmp_state(bmp_state));
    }

    destroy_bmp_state(bmp_state);

    return SAIL_OK;
}

/*
 * Encoding functions.
 */

SAIL_EXPORT sail_status_t sail_codec_save_init_v8_bmp(struct sail_io* io,
                                                      const struct sail_save_options* save_options,
                                                      void** state)
{
    *state = NULL;

    /* Allocate a new state. */
    struct bmp_state* bmp_state;
    SAIL_TRY(alloc_bmp_state(io, NULL, save_options, &bmp_state));
    *state = bmp_state;

    SAIL_TRY(
        bmp_private_write_init(io, bmp_state->save_options, &bmp_state->common_bmp_state, SAIL_WRITE_BMP_FILE_HEADER));

    return SAIL_OK;
}

SAIL_EXPORT sail_status_t sail_codec_save_seek_next_frame_v8_bmp(void* state, const struct sail_image* image)
{
    struct bmp_state* bmp_state = state;

    if (bmp_state->frame_processed)
    {
        return SAIL_ERROR_NO_MORE_FRAMES;
    }

    bmp_state->frame_processed = true;

    SAIL_TRY(bmp_private_write_seek_next_frame(bmp_state->common_bmp_state, bmp_state->io, image));

    return SAIL_OK;
}

SAIL_EXPORT sail_status_t sail_codec_save_frame_v8_bmp(void* state, const struct sail_image* image)
{
    struct bmp_state* bmp_state = state;

    SAIL_TRY(bmp_private_write_frame(bmp_state->common_bmp_state, bmp_state->io, image));

    return SAIL_OK;
}

SAIL_EXPORT sail_status_t sail_codec_save_finish_v8_bmp(void** state)
{
    struct bmp_state* bmp_state = *state;

    *state = NULL;

    if (bmp_state->common_bmp_state != NULL)
    {
        SAIL_TRY_OR_CLEANUP(bmp_private_write_finish(&bmp_state->common_bmp_state, bmp_state->io),
                            /* cleanup */ destroy_bmp_state(bmp_state));
    }

    destroy_bmp_state(bmp_state);

    return SAIL_OK;
}
