// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief Cross-sections for light hadrons at 3.773, 3.650 GeV
  class BESII_2007_I750713 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BESII_2007_I750713);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      // Initialise and register projections
      declare(FinalState(), "FS");
      declare(UnstableParticles(), "UFS");

      // Book histograms
      _sqs = -1;
      if (isCompatibleWithSqrtS(3.65))  _sqs = 100;
      else if (isCompatibleWithSqrtS(3.773))  _sqs = 0;
      raiseBeamErrorIf(_sqs < 0);

      for (size_t ix=3; ix<19; ++ix) {
        if (ix==5 || ix==10) continue;
        book(_nMeson[ix], "TMP/n"s + toString(ix));
        book(_nMeson[ix+100], "TMP/n"s + toString(ix+100));
      }
    }


    void findChildren(const Particle& p,map<long,int>& nRes, int& ncount) const {
      for (const Particle &child : p.children()) {
        if (child.children().empty()) {
          nRes[child.pid()]-=1;
          --ncount;
        }
        else {
          findChildren(child,nRes,ncount);
        }
      }
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const FinalState& fs = apply<FinalState>(event, "FS");

      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
        nCount[p.pid()] += 1;
        ++ntotal;
      }
      const FinalState& ufs = apply<FinalState>(event, "UFS");

      for (const Particle& p : ufs.particles()) {
        if (p.children().empty()) continue;
        if (p.pid()!=221 && p.pid()!=333) continue;
        map<long,int> nRes = nCount;
        int ncount = ntotal;
        findChildren(p,nRes,ncount);
        // eta
        if (p.pid()==221) {
          if (ncount==4) {
            bool matched = true;
            for (const auto& val : nRes) {
              if (abs(val.first)==211) {
                if (val.second!=2) {
                  matched = false;
                  break;
                }
              }
              else if (val.second!=0) {
                matched = false;
                break;
              }
            }
            if (matched) _nMeson[12+_sqs]->fill();
          }
        }
        else if (p.pid()==333) {
          if (ncount!=1) continue;
          bool matched = true;
          for (const auto& val : nRes) {
            if (abs(val.first)==321) {
              if (val.second!=1) {
                matched = false;
                break;
              }
            }
            else if (val.second!=0) {
              matched = false;
              break;
            }
          }
          if (matched)  _nMeson[7+_sqs]->fill();
        }
      }
      if (ntotal==3 &&  nCount[111]==1 &&
          nCount[-2212] == 1 && nCount[ 2212]==1) _nMeson[16+_sqs]->fill();
      else if (ntotal==4) {
        if (nCount[-211] == 2 && nCount[ 211]==2) _nMeson[3+_sqs]->fill();
        else if (nCount[-211] == 1 && nCount[ 211]==1 &&
                 nCount[-321] == 1 && nCount[ 321]==1) _nMeson[4+_sqs]->fill();
        else if (nCount[-321] == 2 && nCount[ 321]==2) _nMeson[6+_sqs]->fill();
        else if (nCount[-211 ] == 1 && nCount[ 211 ]==1 &&
                 nCount[-2212] == 1 && nCount[ 2212]==1) _nMeson[8+_sqs]->fill();
        else if (nCount[-321 ] == 1 && nCount[ 321 ]==1 &&
                 nCount[-2212] == 1 && nCount[ 2212]==1) _nMeson[9+_sqs]->fill();
      }
      else if (ntotal==5 && nCount[111]==1) {
        if (nCount[-211] == 2 && nCount[ 211]==2) _nMeson[13+_sqs]->fill();
        else if (nCount[-211] == 1 && nCount[ 211]==1 &&
                 nCount[-321] == 1 && nCount[ 321]==1) _nMeson[14+_sqs]->fill();
        else if (nCount[-321] == 2 && nCount[ 321]==2)  _nMeson[15+_sqs]->fill();
        else if (nCount[-211 ] == 1 && nCount[ 211 ]==1 &&
                 nCount[-2212] == 1 && nCount[ 2212]==1) _nMeson[17+_sqs]->fill();
      }
      else if (ntotal==6 && nCount[211]==3 && nCount[-211]==3) _nMeson[11+_sqs]->fill();
      else if (ntotal==7 && nCount[111]==1 &&
               nCount[211]==3 && nCount[-211]==3) _nMeson[18+_sqs]->fill();
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_nMeson, crossSection()/ sumOfWeights() /picobarn);
      BinnedEstimatePtr<string>  mult;
      for (unsigned int ix=3; ix<19; ++ix) {
        if (ix==5 || ix==10) continue;
        book(mult, 1, 1, ix);
        mult->bin(1).set(_nMeson[ix]->val(), _nMeson[ix]->err());
        if (ix!=15) mult->bin(2).set(_nMeson[ix+100]->val(), _nMeson[ix+100]->err());
      }
    }
    /// @}


    /// @name Histograms
    /// @{
    map<int,CounterPtr> _nMeson;
    int _sqs;
    /// @}

  };


  RIVET_DECLARE_PLUGIN(BESII_2007_I750713);

}
